/*
 * drivers/usb/dwc_otg/dwc_otg_os_dep.c
 *
 * USB dwc otg linux porting dependency
 *
 *	created by gaowei  2013-3-20
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * OS-Level Implementations */
/* This is the Linux kernel implementation of the DWC platform library. */
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/kthread.h>
#include <linux/moduleparam.h>
#include <linux/ctype.h>
#include <linux/crypto.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/dma-mapping.h>
#include <linux/cdev.h>
#include <linux/errno.h>
#include <linux/interrupt.h>
#include <linux/jiffies.h>
#include <linux/list.h>
#include <linux/pci.h>
#include <linux/random.h>
#include <linux/scatterlist.h>
#include <linux/slab.h>
#include <linux/stat.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/usb.h>

#include <linux/version.h>
#include <linux/usb/gadget.h>
#include <asm/io.h>
#include <asm/page.h>
#include <asm/uaccess.h>
#include <asm/unaligned.h>

#include "dwc_otg_os_dep.h"
#include "dwc_otg_dbg.h"
#include "dwc_list.h"


/* Byte Ordering Conversions */

uint32_t DWC_CPU_TO_LE32(uint32_t *p)
{
#ifdef __LITTLE_ENDIAN
	return *p;
#else
	uint8_t *u_p = (uint8_t *)p;

	return (u_p[3] | (u_p[2] << 8) | (u_p[1] << 16) | (u_p[0] << 24));
#endif
}

char *DWC_STRDUP(char const *str)
{
	int len = strlen(str) + 1;
	char *new = DWC_ALLOC_ATOMIC(len);

	if (!new) {
		return NULL;
	}

	memcpy(new, str, len);
	return new;
}

/* Locking */

spinlock_t *DWC_SPINLOCK_ALLOC(void)
{
	spinlock_t *sl = (spinlock_t *)1;

#if defined(CONFIG_PREEMPT) || defined(CONFIG_SMP)
	sl = DWC_ALLOC(sizeof(*sl));
	if (!sl) {
		DWC_ERROR("Cannot allocate memory for spinlock\n");
		return NULL;
	}

	spin_lock_init(sl);
#endif
	return (spinlock_t *)sl;
}

uint32_t DWC_TIME(void)
{
	return jiffies_to_msecs(jiffies);
}

static void timer_callback(unsigned long data)
{
	dwc_timer_t *timer = (dwc_timer_t *)data;
	unsigned long flags;

	DWC_SPINLOCK_IRQSAVE(timer->lock, flags);
	timer->scheduled = 0;
	DWC_SPINUNLOCK_IRQRESTORE(timer->lock, flags);
	DWC_DEBUG("Timer %s callback", timer->name);
	timer->cb(timer->data);
}

dwc_timer_t *DWC_TIMER_ALLOC(char *name, dwc_timer_callback_t cb, void *data)
{
	dwc_timer_t *t = DWC_ALLOC(sizeof(*t));

	if (!t) {
		DWC_ERROR("Cannot allocate memory for timer");
		return NULL;
	}

	t->t = DWC_ALLOC(sizeof(*t->t));
	if (!t->t) {
		DWC_ERROR("Cannot allocate memory for timer->t");
		goto no_timer;
	}

	t->name = DWC_STRDUP(name);
	if (!t->name) {
		DWC_ERROR("Cannot allocate memory for timer->name");
		goto no_name;
	}

	t->lock = DWC_SPINLOCK_ALLOC();
	if (!t->lock) {
		DWC_ERROR("Cannot allocate memory for lock");
		goto no_lock;
	}

	t->scheduled = 0;
	t->t->base = &boot_tvec_bases;
	t->t->expires = jiffies;
	setup_timer(t->t, timer_callback, (unsigned long)t);

	t->cb = cb;
	t->data = data;

	return t;

 no_lock:
	DWC_FREE(t->name);
 no_name:
	DWC_FREE(t->t);
 no_timer:
	DWC_FREE(t);
	return NULL;
}

void DWC_TIMER_FREE(dwc_timer_t *timer)
{
	unsigned long  flags;

	DWC_SPINLOCK_IRQSAVE(timer->lock, flags);

	if (timer->scheduled) {
		del_timer(timer->t);
		timer->scheduled = 0;
	}

	DWC_SPINUNLOCK_IRQRESTORE(timer->lock, flags);
	DWC_SPINLOCK_FREE(timer->lock);
	DWC_FREE(timer->t);
	DWC_FREE(timer->name);
	DWC_FREE(timer);
}

void DWC_TIMER_SCHEDULE(dwc_timer_t *timer, uint32_t time)
{
	unsigned long  flags;

	DWC_SPINLOCK_IRQSAVE(timer->lock, flags);

	if (!timer->scheduled) {
		timer->scheduled = 1;
	//	DWC_DEBUG("Scheduling timer %s to expire in +%d msec", timer->name, time);
		timer->t->expires = jiffies + msecs_to_jiffies(time);
		add_timer(timer->t);
	} else {
	//	DWC_DEBUG("Modifying timer %s to expire in +%d msec", timer->name, time);
		mod_timer(timer->t, jiffies + msecs_to_jiffies(time));
	}

	DWC_SPINUNLOCK_IRQRESTORE(timer->lock, flags);
}

dwc_waitq_t *DWC_WAITQ_ALLOC(void)
{
	dwc_waitq_t *wq = DWC_ALLOC(sizeof(*wq));

	if (!wq) {
		DWC_ERROR("Cannot allocate memory for waitqueue\n");
		return NULL;
	}

	init_waitqueue_head(&wq->queue);
	wq->abort = 0;
	return wq;
}

int32_t DWC_WAITQ_WAIT_TIMEOUT(dwc_waitq_t *wq, dwc_waitq_condition_t cond,
			       void *data, int32_t msecs)
{
	int32_t tmsecs;
	int result = wait_event_interruptible_timeout(wq->queue,
						      cond(data) || wq->abort,
						      msecs_to_jiffies(msecs));
	if (result == -ERESTARTSYS) {
		wq->abort = 0;
		return -DWC_E_RESTART;
	}

	if (wq->abort == 1) {
		wq->abort = 0;
		return -DWC_E_ABORT;
	}

	wq->abort = 0;

	if (result > 0) {
		tmsecs = jiffies_to_msecs(result);
		if (!tmsecs) {
			return 1;
		}

		return tmsecs;
	}

	if (result == 0) {
		return -DWC_E_TIMEOUT;
	}

	return -DWC_E_UNKNOWN;
}

void DWC_WAITQ_TRIGGER(dwc_waitq_t *wq)
{
	wq->abort = 0;
	wake_up_interruptible(&wq->queue);
}

void DWC_WAITQ_ABORT(dwc_waitq_t *wq)
{
	wq->abort = 1;
	wake_up_interruptible(&wq->queue);
}


/* Threading */

dwc_thread_t *DWC_THREAD_RUN(dwc_thread_function_t func, char *name, void *data)
{
	struct task_struct *thread = kthread_run(func, data, name);

	if (thread == ERR_PTR(-ENOMEM)) {
		return NULL;
	}

	return (dwc_thread_t *)thread;
}

int DWC_THREAD_STOP(dwc_thread_t *thread)
{
	return kthread_stop((struct task_struct *)thread);
}

dwc_bool_t DWC_THREAD_SHOULD_STOP(void)
{
	return kthread_should_stop();
}

/* tasklets
 - run in interrupt context (cannot sleep)
 - each tasklet runs on a single CPU
 - different tasklets can be running simultaneously on different CPUs
 */
struct dwc_tasklet {
	struct tasklet_struct t;
	dwc_tasklet_callback_t cb;
	void *data;
};

static void tasklet_callback(unsigned long data)
{
	dwc_tasklet_t *t = (dwc_tasklet_t *)data;
	t->cb(t->data);
}

dwc_tasklet_t *DWC_TASK_ALLOC(char *name, dwc_tasklet_callback_t cb, void *data)
{
	dwc_tasklet_t *t = DWC_ALLOC(sizeof(*t));

	if (t) {
		t->cb = cb;
		t->data = data;
		tasklet_init(&t->t, tasklet_callback, (unsigned long)t);
	} else {
		DWC_ERROR("Cannot allocate memory for tasklet\n");
	}

	return t;
}

void DWC_TASK_FREE(dwc_tasklet_t *task)
{
	DWC_FREE(task);
}

void DWC_TASK_SCHEDULE(dwc_tasklet_t *task)
{
	tasklet_schedule(&task->t);
}

static void do_work(struct work_struct *work)
{
	unsigned long flags;
	struct delayed_work *dw = container_of(work, struct delayed_work, work);
	work_container_t *container = container_of(dw, struct work_container, work);
	dwc_workq_t *wq = container->wq;

	container->cb(container->data);

#ifdef DEBUG
	DWC_CIRCLEQ_REMOVE(&wq->entries, container, entry);
#endif
	DWC_DEBUG("Work done: %s, container=%p", container->name, container);
	if (container->name) {
		DWC_FREE(container->name);
	}
	DWC_FREE(container);

	DWC_SPINLOCK_IRQSAVE(wq->lock, flags);
	wq->pending--;
	DWC_SPINUNLOCK_IRQRESTORE(wq->lock, flags);
	DWC_WAITQ_TRIGGER(wq->waitq);
}

static int work_done(void *data)
{
	dwc_workq_t *workq = (dwc_workq_t *)data;
	return workq->pending == 0;
}

int DWC_WORKQ_WAIT_WORK_DONE(dwc_workq_t *workq, int timeout)
{
	return DWC_WAITQ_WAIT_TIMEOUT(workq->waitq, work_done, workq, timeout);
}

dwc_workq_t *DWC_WORKQ_ALLOC(char *name)
{
	dwc_workq_t *wq = DWC_ALLOC(sizeof(*wq));

	if (!wq) {
		return NULL;
	}

	wq->wq = create_singlethread_workqueue(name);
	if (!wq->wq) {
		goto no_wq;
	}

	wq->pending = 0;

	wq->lock = DWC_SPINLOCK_ALLOC();
	if (!wq->lock) {
		goto no_lock;
	}

	wq->waitq = DWC_WAITQ_ALLOC();
	if (!wq->waitq) {
		goto no_waitq;
	}

#ifdef DEBUG
	DWC_CIRCLEQ_INIT(&wq->entries);
#endif
	return wq;

 no_waitq:
	DWC_SPINLOCK_FREE(wq->lock);
 no_lock:
	destroy_workqueue(wq->wq);
 no_wq:
	DWC_FREE(wq);

	return NULL;
}

void DWC_WAITQ_FREE(dwc_waitq_t *wq)
{
	DWC_FREE(wq);
}

int32_t DWC_WAITQ_WAIT(dwc_waitq_t *wq, dwc_waitq_condition_t cond, void *data)
{
	int result = wait_event_interruptible(wq->queue,
					      cond(data) || wq->abort);
	if (result == -ERESTARTSYS) {
		wq->abort = 0;
		return -DWC_E_RESTART;
	}

	if (wq->abort == 1) {
		wq->abort = 0;
		return -DWC_E_ABORT;
	}

	wq->abort = 0;

	if (result == 0) {
		return 0;
	}

	return -DWC_E_UNKNOWN;
}

void DWC_WORKQ_FREE(dwc_workq_t *wq)
{
#ifdef DEBUG
	if (wq->pending != 0) {
		struct work_container *wc;
		DWC_ERROR("Destroying work queue with pending work");
		DWC_CIRCLEQ_FOREACH(wc, &wq->entries, entry) {
			DWC_ERROR("Work %s still pending", wc->name);
		}
	}
#endif
	destroy_workqueue(wq->wq);
	DWC_SPINLOCK_FREE(wq->lock);
	DWC_WAITQ_FREE(wq->waitq);
	DWC_FREE(wq);
}

void DWC_WORKQ_SCHEDULE(dwc_workq_t *wq, dwc_work_callback_t cb, void *data,
			char *format, ...)
{
	unsigned long flags;
	work_container_t *container;
	static char name[128];
	va_list args;

	va_start(args, format);
	DWC_VSNPRINTF(name, 128, format, args);
	va_end(args);

	DWC_SPINLOCK_IRQSAVE(wq->lock, flags);
	wq->pending++;
	DWC_SPINUNLOCK_IRQRESTORE(wq->lock, flags);
	DWC_WAITQ_TRIGGER(wq->waitq);

	container = DWC_ALLOC_ATOMIC(sizeof(*container));
	if (!container) {
		DWC_ERROR("Cannot allocate memory for container\n");
		return;
	}

	container->name = DWC_STRDUP(name);
	if (!container->name) {
		DWC_ERROR("Cannot allocate memory for container->name\n");
		DWC_FREE(container);
		return;
	}

	container->cb = cb;
	container->data = data;
	container->wq = wq;
	DWC_DEBUG("Queueing work: %s, container=%p", container->name, container);
	INIT_WORK(&container->work.work, do_work);

#ifdef DEBUG
	DWC_CIRCLEQ_INSERT_TAIL(&wq->entries, container, entry);
#endif
	queue_work(wq->wq, &container->work.work);
}

void DWC_WORKQ_SCHEDULE_DELAYED(dwc_workq_t *wq, dwc_work_callback_t cb,
				void *data, uint32_t time, char *format, ...)
{
	unsigned long flags;
	work_container_t *container;
	static char name[128];
	va_list args;

	va_start(args, format);
	DWC_VSNPRINTF(name, 128, format, args);
	va_end(args);

	DWC_SPINLOCK_IRQSAVE(wq->lock, flags);
	wq->pending++;
	DWC_SPINUNLOCK_IRQRESTORE(wq->lock, flags);
	DWC_WAITQ_TRIGGER(wq->waitq);

	container = DWC_ALLOC_ATOMIC(sizeof(*container));
	if (!container) {
		DWC_ERROR("Cannot allocate memory for container\n");
		return;
	}

	container->name = DWC_STRDUP(name);
	if (!container->name) {
		DWC_ERROR("Cannot allocate memory for container->name\n");
		DWC_FREE(container);
		return;
	}

	container->cb = cb;
	container->data = data;
	container->wq = wq;
	DWC_DEBUG("Queueing work: %s, container=%p", container->name, container);
	INIT_DELAYED_WORK(&container->work, do_work);

#ifdef DEBUG
	DWC_CIRCLEQ_INSERT_TAIL(&wq->entries, container, entry);
#endif
	queue_delayed_work(wq->wq, &container->work, msecs_to_jiffies(time));
}

int DWC_WORKQ_PENDING(dwc_workq_t *wq)
{
	return wq->pending;
}



EXPORT_SYMBOL(DWC_STRDUP);
EXPORT_SYMBOL(DWC_SPINLOCK_ALLOC);
EXPORT_SYMBOL(DWC_TIME);
EXPORT_SYMBOL(DWC_TIMER_ALLOC);
EXPORT_SYMBOL(DWC_TIMER_FREE);
EXPORT_SYMBOL(DWC_TIMER_SCHEDULE);
EXPORT_SYMBOL(DWC_WAITQ_ALLOC);
EXPORT_SYMBOL(DWC_WAITQ_FREE);
EXPORT_SYMBOL(DWC_WAITQ_WAIT);
EXPORT_SYMBOL(DWC_WAITQ_WAIT_TIMEOUT);
EXPORT_SYMBOL(DWC_WAITQ_TRIGGER);
EXPORT_SYMBOL(DWC_WAITQ_ABORT);
EXPORT_SYMBOL(DWC_THREAD_RUN);
EXPORT_SYMBOL(DWC_THREAD_STOP);
EXPORT_SYMBOL(DWC_THREAD_SHOULD_STOP);
EXPORT_SYMBOL(DWC_TASK_ALLOC);
EXPORT_SYMBOL(DWC_TASK_FREE);
EXPORT_SYMBOL(DWC_TASK_SCHEDULE);
EXPORT_SYMBOL(DWC_WORKQ_WAIT_WORK_DONE);
EXPORT_SYMBOL(DWC_WORKQ_ALLOC);
EXPORT_SYMBOL(DWC_WORKQ_FREE);
EXPORT_SYMBOL(DWC_WORKQ_SCHEDULE);
EXPORT_SYMBOL(DWC_WORKQ_SCHEDULE_DELAYED);
EXPORT_SYMBOL(DWC_WORKQ_PENDING);


