/*
 * Gadget Driver for Android
 *
 * Copyright (C) 2008 Google, Inc.
 * Author: Mike Lockwood <lockwood@android.com>
 *         Benoit Goby <benoit@android.com>
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/kernel.h>
#include <linux/utsname.h>
#include <linux/platform_device.h>

#include <linux/usb/ch9.h>
#include <linux/usb/composite.h>
#include <linux/usb/gadget.h>
#include <linux/android_notify.h>
#include <mach/highspeed_debug.h>

#include "gadget_chips.h"
#include <mach/iomap.h>

/*
 * Kbuild is not very cooperative with respect to linking separately
 * compiled library objects into one module.  So for now we won't use
 * separate compilation ... ensuring init/exit sections work to shrink
 * the runtime footprint, and giving us at least some parts of what
 * a "gcc --combine ... part1.c part2.c part3.c ... " build would.
 */
#include "usbstring.c"
#include "config.c"
#include "epautoconf.c"
#include "composite.c"

//#include "f_fs.c"
//#include "f_audio_source.c"
#include "f_mass_storage.c"
#include "u_serial.c"
#include "f_acm.c"
#include "f_serial.c"
#include "f_adb.c"
#ifdef _USE_VEHICLE_DC
#include "adb_server.c"
#include "usb_rpmsg_server.c"

#endif
//#include "f_mtp.c"
//#include "f_accessory.c"
#define USB_ETH_RNDIS y
#include "f_ecm.c"
#include "f_rndis.c"
#include "rndis.c"
#include "f_mbim.c"
#include "u_ether.c"
#include "u_diag.c"
#ifdef _USE_MBIM//CONFIG_USB_F_DIAG_ACM
#include "f_diag_acm.c"
#else
#include "f_diag.c"
#endif

MODULE_AUTHOR("Mike Lockwood");
MODULE_DESCRIPTION("Android Composite USB Driver");
MODULE_LICENSE("GPL");
MODULE_VERSION("1.0");

#ifndef CONFIG_SYSTEM_RECOVERY
int zDrvNand_WriteBootflag( int flag );
unsigned int zDrvNand_ReadBootflag(void);
#endif

#define pr_err		USB_DEBUG

#ifdef _USE_VEHICLE_DC
void usb_check_plugin_msg_sent(void);
#endif

static const char longname[] = "Gadget Android";

/* Default vendor and product IDs, overridden by userspace */
#define VENDOR_ID		0x19D2
#define PRODUCT_ID		0x0193

/* USB net card name */
#define USB_VNIC_NAME	"usblan"

/* USB NET CARD MAX NUM */
#define MAX_ECM_INSTANCES 3


#define CFG_LUN_NUM_TWO	1

#define MBIM_PACKET_MAX_NUM  10
#define RNDIS_PACKET_MAX_NUM  10


#define OS_string_descriptor_id	0xEE
#define bMS_Code_Original		0x04
#define bMS_Code_Change		0x08
struct android_usb_function {
	char *name;
	void *config;

	struct device *dev;
	struct usb_config_descriptor *cof;
	char *dev_name;
	struct device_attribute **attributes;

	/* for android_dev.enabled_functions */
	struct list_head enabled_list;

	/* Optional: initialization during gadget bind */
	int (*init)(struct android_usb_function *, struct usb_composite_dev *);
	/* Optional: cleanup during gadget unbind */
	void (*cleanup)(struct android_usb_function *);
	/* Optional: called when the function is added the list of
	 *		enabled functions */
	void (*enable)(struct android_usb_function *);
	/* Optional: called when it is removed */
	void (*disable)(struct android_usb_function *);

	int (*bind_config)(struct android_usb_function *,
			   struct usb_configuration *);

	/* Optional: called when the configuration is removed */
	void (*unbind_config)(struct android_usb_function *,
			      struct usb_configuration *);
	/* Optional: handle ctrl requests before the device is configured */
	int (*ctrlrequest)(struct android_usb_function *,
					struct usb_composite_dev *,
					const struct usb_ctrlrequest *);
};

struct android_dev {
	struct android_usb_function **functions;
	struct list_head enabled_functions;
	struct usb_composite_dev *cdev;
	struct device *dev;

	bool cdrom_only;
	bool enabled;
	int disable_depth;
	//int bRequest;
	//int wValue;
	struct mutex mutex;
	bool connected;
	bool sw_connected;
	struct work_struct work;
	struct work_struct usbmode;
	char ffs_aliases[256];

	struct usb_ctrlrequest vendor_req;
};

static struct class *android_class;
static struct android_dev *_android_dev;
static int  multi_packet_num = 10;
static int  ether_skb_num = 32;

static int android_bind_config(struct usb_configuration *c);
static void android_unbind_config(struct usb_configuration *c);
void usb_mods_init(void);
void usb_mods_exit(void);
void usb_mods_activate(void);
void usb_set_ms_auto_reject(int flag);
extern void dwc_otg_clk_enable(int isOn);
extern void usb_record_dbginfo(usb_dbginfo_type inf_type, int status, int info_value);
extern void usb_set_switch_mode(int cur_mode);


typedef enum usb_enum_mode_type{
	USB_ENUM_MODE_DEBUG = 0,
	USB_ENUM_MODE_USER,
	USB_ENUM_MODE_FACTORY,
	USB_ENUM_MODE_AMT,
	USB_ENUM_MODE_EYE_DIAGRAM,		
	USB_ENUM_MODE_MAX,
}enum_mode_type;

enum_mode_type usb_cur_enum_mode = USB_ENUM_MODE_DEBUG;



/* string IDs are assigned dynamically */
#define STRING_MANUFACTURER_IDX		0
#define STRING_PRODUCT_IDX		1
#define STRING_SERIAL_IDX		2
#define STRING_CONFIGURATION_IDX		3

static char manufacturer_string[256];
static char product_string[256];
static char serial_string[256];
static char configuration_string[256];

/* String Table */
static struct usb_string strings_dev[] = {
	[STRING_MANUFACTURER_IDX].s = manufacturer_string,
	[STRING_PRODUCT_IDX].s = product_string,
	[STRING_SERIAL_IDX].s = serial_string,
	[STRING_CONFIGURATION_IDX].s = configuration_string,
	{  }			/* end of list */
};

static struct usb_gadget_strings stringtab_dev = {
	.language	= 0x0409,	/* en-us */
	.strings	= strings_dev,
};

static struct usb_gadget_strings *dev_strings[] = {
	&stringtab_dev,
	NULL,
};

static struct usb_device_descriptor device_desc = {
	.bLength              = sizeof(device_desc),
	.bDescriptorType      = USB_DT_DEVICE,
	.bcdUSB               = __constant_cpu_to_le16(0x0200),
	.bDeviceClass         = USB_CLASS_PER_INTERFACE,
	.idVendor             = __constant_cpu_to_le16(VENDOR_ID),
	.idProduct            = __constant_cpu_to_le16(PRODUCT_ID),
	.bcdDevice            = __constant_cpu_to_le16(0xffff),
	.bNumConfigurations   = 1,
};

static struct usb_configuration android_config_driver = {
	.label		= "android",
	.unbind		= android_unbind_config,
	.bConfigurationValue = 1,
	.iConfiguration            =	1,
#if 0	
	.bmAttributes	= USB_CONFIG_ATT_ONE | USB_CONFIG_ATT_SELFPOWER | USB_CONFIG_ATT_WAKEUP,
#else
	.bmAttributes	= USB_CONFIG_ATT_ONE | USB_CONFIG_ATT_SELFPOWER,
#endif	
	.bMaxPower	= 0xFA, /* 500ma */
};

int get_usb_enum_mode(void)
{
	return usb_cur_enum_mode;
}
EXPORT_SYMBOL_GPL(get_usb_enum_mode);

static void usb_mode_work(struct work_struct *data)
{
	struct android_dev *dev = container_of(data, struct android_dev, usbmode);
	 
	switch(dev->vendor_req.bRequest)
	{
		case USB_SWITCH_to_DEBUG:
			if(usb_cur_enum_mode == USB_ENUM_MODE_DEBUG){
				printk("usb_mode_work already debug mode\n");
				break;
			}
			if(dev->vendor_req.wValue == 0x0002)
			{
#ifndef CONFIG_SYSTEM_RECOVERY
				zDrvNand_WriteBootflag(1);
#endif
				usb_notify_up(USB_SWITCH_DEBUG,NULL);
				usb_record_dbginfo(USB_SWITCH_TO_DEBUG, 0, 0);
			}
			break;
			
		case USB_SWITCH_to_DEBUG_AT:
			if(dev->vendor_req.wValue == 0x0101)
			{
#ifndef CONFIG_SYSTEM_RECOVERY
				zDrvNand_WriteBootflag(0);
#endif
				usb_notify_up(USB_SWITCH_DEBUG,NULL);
			}
			break;

		case USB_SWITCH_to_USER:
			if(usb_cur_enum_mode == USB_ENUM_MODE_USER){
				printk("usb_mode_work already user mode\n");
				break;
			}
#ifndef CONFIG_SYSTEM_RECOVERY
			zDrvNand_WriteBootflag(1);
#endif
			usb_notify_up(USB_SWITCH_USER, NULL);
				usb_record_dbginfo(USB_SWITCH_TO_USER, 0, 0);
			break;

		case USB_SWITCH_to_FACTORY:
			usb_notify_up(USB_SWITCH_FACTORY, NULL);
				usb_record_dbginfo(USB_SWITCH_TO_FACTORY, 0, 0);
			break;

		case USB_SWITCH_to_AMT:
			usb_notify_up(USB_SWITCH_AMT, NULL);
				usb_record_dbginfo(USB_SWITCH_TO_AMT, 0, 0);
			break;
		case USB_SWITCH_to_EYE_DIAGRAM:
			usb_notify_up(USB_SWITCH_EYE_DIAGRAM, NULL);
			break;
	}
}

int get_vnic_multi_packet_num(void)
{
	return multi_packet_num;
}

unsigned int gether_ether_skb_num(void)
{
	return ether_skb_num;
}

static void android_work(struct work_struct *data)
{
	struct android_dev *dev = container_of(data, struct android_dev, work);
	struct usb_composite_dev *cdev = dev->cdev;
	char *disconnected[2] = { "USB_STATE=DISCONNECTED", NULL };
	char *connected[2]    = { "USB_STATE=CONNECTED", NULL };
	char *configured[2]   = { "USB_STATE=CONFIGURED", NULL };
	char **uevent_envp = NULL;
	unsigned long flags;
	USBSTACK_DBG("%s, %u", __func__, __LINE__);
	spin_lock_irqsave(&cdev->lock, flags);
	if (cdev->config)
		uevent_envp = configured;
	else if (dev->connected != dev->sw_connected)
		uevent_envp = dev->connected ? connected : disconnected;
	dev->sw_connected = dev->connected;
	spin_unlock_irqrestore(&cdev->lock, flags);

	if (uevent_envp) {
		kobject_uevent_env(&dev->dev->kobj, KOBJ_CHANGE, uevent_envp);
		//pr_info("%s: sent uevent %s\n", __func__, uevent_envp[0]);
	} else {
		//pr_info("%s: did not send uevent (%d %d %p)\n", __func__,
		//	 dev->connected, dev->sw_connected, cdev->config);
	}
}

static void android_enable(struct android_dev *dev)
{
	int ret;
	struct usb_composite_dev *cdev = dev->cdev;

	//USB_DEBUG("disable_depth:%d", dev->disable_depth);
	USBSTACK_DBG("%s, disable_depth:%d", __func__, dev->disable_depth);
	if (WARN_ON(!dev->disable_depth))
		return;
	printk("----android_enable, bmattr:%x\n", android_config_driver.bmAttributes);
	if (--dev->disable_depth == 0) {
		ret =usb_add_config(cdev, &android_config_driver,
					android_bind_config);
		if(ret)
			USBSTACK_DBG("usb_add_config, ret:%d", ret);
		usb_gadget_connect(cdev->gadget);
	}
}

static void android_disable(struct android_dev *dev)
{
	struct usb_composite_dev *cdev = dev->cdev;
	USBSTACK_DBG("%s, disable_depth:%d", __func__, dev->disable_depth);
	if (dev->disable_depth++ == 0) {
		usb_gadget_disconnect(cdev->gadget);
		/* Cancel pending control requests */
		usb_ep_dequeue(cdev->gadget->ep0, cdev->req);
		usb_remove_config(cdev, &android_config_driver);
	}
}

/*-------------------------------------------------------------------------*/
/* Supported functions initialization */

#if 0
struct functionfs_config {
	bool opened;
	bool enabled;
	struct ffs_data *data;
};

static int ffs_function_init(struct android_usb_function *f,
			     struct usb_composite_dev *cdev)
{
	f->config = kzalloc(sizeof(struct functionfs_config), GFP_KERNEL);
	if (!f->config)
		return -ENOMEM;

	return functionfs_init();
}

static void ffs_function_cleanup(struct android_usb_function *f)
{
	functionfs_cleanup();
	kfree(f->config);
}

static void ffs_function_enable(struct android_usb_function *f)
{
	struct android_dev *dev = _android_dev;
	struct functionfs_config *config = f->config;

	config->enabled = true;

	/* Disable the gadget until the function is ready */
	if (!config->opened)
		android_disable(dev);
}

static void ffs_function_disable(struct android_usb_function *f)
{
	struct android_dev *dev = _android_dev;
	struct functionfs_config *config = f->config;

	config->enabled = false;

	/* Balance the disable that was called in closed_callback */
	if (!config->opened)
		android_enable(dev);
}

static int ffs_function_bind_config(struct android_usb_function *f,
				    struct usb_configuration *c)
{
	struct functionfs_config *config = f->config;
	return functionfs_bind_config(c->cdev, c, config->data);
}

static ssize_t
ffs_aliases_show(struct device *pdev, struct device_attribute *attr, char *buf)
{
	struct android_dev *dev = _android_dev;
	int ret;

	mutex_lock(&dev->mutex);
	ret = sprintf(buf, "%s\n", dev->ffs_aliases);
	mutex_unlock(&dev->mutex);

	return ret;
}

static ssize_t
ffs_aliases_store(struct device *pdev, struct device_attribute *attr,
					const char *buf, size_t size)
{
	struct android_dev *dev = _android_dev;
	char buff[256];

	mutex_lock(&dev->mutex);

	if (dev->enabled) {
		mutex_unlock(&dev->mutex);
		return -EBUSY;
	}

	strlcpy(buff, buf, sizeof(buff));
	strlcpy(dev->ffs_aliases, strim(buff), sizeof(dev->ffs_aliases));

	mutex_unlock(&dev->mutex);

	return size;
}

static DEVICE_ATTR(aliases, S_IRUGO | S_IWUSR, ffs_aliases_show,
					       ffs_aliases_store);
static struct device_attribute *ffs_function_attributes[] = {
	&dev_attr_aliases,
	NULL
};

static struct android_usb_function ffs_function = {
	.name		= "ffs",
	.init		= ffs_function_init,
	.enable		= ffs_function_enable,
	.disable	= ffs_function_disable,
	.cleanup	= ffs_function_cleanup,
	.bind_config	= ffs_function_bind_config,
	.attributes	= ffs_function_attributes,
};

static int functionfs_ready_callback(struct ffs_data *ffs)
{
	struct android_dev *dev = _android_dev;
	struct functionfs_config *config = ffs_function.config;
	int ret = 0;

	mutex_lock(&dev->mutex);

	ret = functionfs_bind(ffs, dev->cdev);
	if (ret)
		goto err;

	config->data = ffs;
	config->opened = true;

	if (config->enabled)
		android_enable(dev);

err:
	mutex_unlock(&dev->mutex);
	return ret;
}

static void functionfs_closed_callback(struct ffs_data *ffs)
{
	struct android_dev *dev = _android_dev;
	struct functionfs_config *config = ffs_function.config;

	mutex_lock(&dev->mutex);

	if (config->enabled)
		android_disable(dev);

	config->opened = false;
	config->data = NULL;

	functionfs_unbind(ffs);

	mutex_unlock(&dev->mutex);
}

static int functionfs_check_dev_callback(const char *dev_name)
{
	return 0;
}
#endif

#define ECM_TIMEOUT	6000

struct ecm_function_config {
	u8	ethaddr[MAX_ECM_INSTANCES][ETH_ALEN];;
	int	instances;
	int	receive_setIfac;
	int	receive_setICfg;
	struct delayed_work work;	
};

extern void dwc_otg_wakelock(int lock_flag,int phase);
static void ecm_function_work(struct work_struct *data)
{
	USBSTACK_DBG("ecm work");
	dwc_otg_wakelock(1,0);
	usb_notify_up(USB_DEVICE_PLUGOUT, NULL);
	usb_notify_up(USB_DEVICE_PLUGIN, NULL);
	dwc_otg_wakelock(0,0);
}
static int ecm_function_ctrlrequest(struct android_usb_function * f,
					struct usb_composite_dev *dev,
					const struct usb_ctrlrequest * ctrl)
{
	int	value = -1;
	struct ecm_function_config *ecm = f->config;
#if 1	
	switch (ctrl->bRequestType & USB_TYPE_MASK){
	case USB_TYPE_STANDARD:
		if(ctrl->bRequest == USB_REQ_SET_CONFIGURATION){
			if((!ecm->receive_setIfac)&&(!ecm->receive_setICfg)){
				printk("ecm-receive_setICfg\n");
				//schedule_delayed_work(&ecm->work, msecs_to_jiffies(ECM_TIMEOUT));
			}			
			if((ecm->receive_setIfac)&&(!ecm->receive_setICfg))
				gether_ecm_uevent(ecm->instances, 1);
			ecm->receive_setICfg = 1;
		}else if(ctrl->bRequest == USB_REQ_SET_INTERFACE){
			u8	intf = (le16_to_cpu(ctrl->wIndex)) & 0xFF;
			struct usb_function	*f_intf = dev->config->interface[intf];
			struct f_ecm		*f_ecm = func_to_ecm(f_intf);
				printk("ecm-receive_setinterface\n");
			if (intf != f_ecm->data_id)
				return value;
			if(ecm->receive_setICfg){
				//USBSTACK_DBG("ecm-delayed-work cancel");
				//cancel_delayed_work_sync(&ecm->work);
				gether_ecm_uevent(ecm->instances, 1);
			}
			ecm->receive_setIfac = 1;
		}
		break;
	case USB_TYPE_VENDOR:
		if((ctrl->bRequest == bMS_Code_Original )||
			(ctrl->bRequest == bMS_Code_Change )){
				printk("ecm-sys-id-err");
				schedule_delayed_work(&ecm->work, msecs_to_jiffies(50));
		}
		break;		
	}
#endif
	return value;
}

static int
ecm_function_init(struct android_usb_function *f,
		struct usb_composite_dev *cdev)
{
	struct ecm_function_config *ecm;
	
	USBSTACK_DBG("%s", __func__);
	
	ecm = kzalloc(sizeof(struct ecm_function_config), GFP_KERNEL);

	if (!ecm)
		return -ENOMEM;

	ecm->instances = 1;
	INIT_DELAYED_WORK(&ecm->work, ecm_function_work);
	f->config = ecm;
	
	return 0;
}

static void ecm_function_cleanup(struct android_usb_function *f)
{
	USBSTACK_DBG("%s", __func__);
	kfree(f->config);
	f->config = NULL;
}

static int
ecm_function_bind_config(struct android_usb_function *f,
		struct usb_configuration *c)
{
	int ret, i;
    ret =0;
	struct ecm_function_config *ecm = f->config;

	if (!ecm) {
		pr_err("%s: ecm_pdata\n", __func__);
		return -1;
	}
	USBSTACK_DBG("%s, instances:%d", __func__, ecm->instances);
	ecm->receive_setIfac = 0;
	ecm->receive_setICfg = 0;
	
	for (i = 0; ((i < ecm->instances) && (i< MAX_ECM_INSTANCES)); i++) {
		ret = gether_setup_name_num(c->cdev->gadget, 
							ecm->ethaddr[i], USB_VNIC_NAME, i);
		if (ret) {
			pr_err("%s: gether_setup failed\n", __func__);
			return ret;
		}

		ret = ecm_bind_config_num(c, ecm->ethaddr[i], i);
		if (ret) {
			pr_err("Could not bind ecm%u config\n", i);
			break;
		}
	}

	return ret;	
}

static void ecm_function_unbind_config(struct android_usb_function *f,
						struct usb_configuration *c)
{
	int i;
	struct ecm_function_config *ecm = f->config;
	
	USBSTACK_DBG("%s", __func__);
	cancel_delayed_work_sync(&ecm->work);
	
	for(i=0; i<ecm->instances; i++){
		gether_cleanup_num(i);
	}
}

static ssize_t ecm_ethaddr_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct android_usb_function *f = dev_get_drvdata(dev);
    if(NULL == f)
    {
        return -EINVAL;
    }
    struct ecm_function_config *ecm = f->config;
    if(NULL == ecm)
    {
        return -EINVAL;
    }
    USBSTACK_DBG("%s", __func__);
	return sprintf(buf, "%02x:%02x:%02x:%02x:%02x:%02x\n",
		ecm->ethaddr[0][0], ecm->ethaddr[0][1], ecm->ethaddr[0][2],
		ecm->ethaddr[0][3], ecm->ethaddr[0][4], ecm->ethaddr[0][5]);
}

static ssize_t ecm_ethaddr_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t size)
{
	struct android_usb_function *f = dev_get_drvdata(dev);
    if(NULL == f)
    {
        return -EINVAL;
    }
    struct ecm_function_config *ecm = f->config;
    if(NULL == ecm)
    {
        return -EINVAL;
    }
    USBSTACK_DBG("%s", __func__);
	if (sscanf(buf, "%02x:%02x:%02x:%02x:%02x:%02x\n",
		    (unsigned int *)&ecm->ethaddr[0][0], (unsigned int *)&ecm->ethaddr[0][1],
		    (unsigned int *)&ecm->ethaddr[0][2], (unsigned int *)&ecm->ethaddr[0][3],
		    (unsigned int *)&ecm->ethaddr[0][4], (unsigned int *)&ecm->ethaddr[0][5]) == 6)
		return size;
	return -EINVAL;
}

static DEVICE_ATTR(ethaddr_ecm, S_IRUGO | S_IWUSR, ecm_ethaddr_show,
					       ecm_ethaddr_store);

static ssize_t ecm_instances_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct android_usb_function *f = dev_get_drvdata(dev);
    if(NULL == f)
    {
        return -EINVAL;
    }
	struct ecm_function_config *config = f->config;
	    if(NULL == config)
    {
        return -EINVAL;
    }
    USBSTACK_DBG("%s", __func__);
	return sprintf(buf, "%d\n", config->instances);
}

static ssize_t ecm_instances_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t size)
{
	struct android_usb_function *f = dev_get_drvdata(dev);
    if(NULL == f)
    {
        return -EINVAL;
    }
    struct ecm_function_config *config = f->config;
	if(NULL == config)
    {
        return -EINVAL;
    }
	int value;
	USBSTACK_DBG("%s", __func__);
	sscanf(buf, "%d", &value);
	if ((value > MAX_ECM_INSTANCES)||(value < 1)){
		USB_DEBUG("WARNNING:SET INVALID ECM NUM:%d!!!", value);
		value = 1;
	}
	config->instances = value;
	return size;
}

static DEVICE_ATTR(ecm_instances, S_IRUGO | S_IWUSR, ecm_instances_show,
						 ecm_instances_store);

static struct device_attribute *ecm_function_attributes[] = {
	&dev_attr_ethaddr_ecm,
	&dev_attr_ecm_instances,
	NULL
};


static struct android_usb_function  ecm_function = {
	.name		= "ecm",
	.init		= ecm_function_init,
	.cleanup	= ecm_function_cleanup,
	.bind_config	= ecm_function_bind_config,
	.unbind_config	= ecm_function_unbind_config,
	.attributes	= ecm_function_attributes,
	.ctrlrequest	= ecm_function_ctrlrequest,
};


#define MAX_DIAG_INSTANCES 1
struct diag_function_config {
	int instances;
};

static int
diag_function_init(struct android_usb_function *f,
		struct usb_composite_dev *cdev)
{
	USB_DEBUG("DIAG INIT");
	USBSTACK_DBG("%s", __func__);
	f->config = kzalloc(sizeof(struct diag_function_config), GFP_KERNEL);
	if (!f->config)
		return -ENOMEM;
	((struct diag_function_config *)(f->config))->instances = 1;
	return diag_setup(cdev->gadget, MAX_DIAG_INSTANCES);
}


static void diag_function_cleanup(struct android_usb_function *f)
{
	USBSTACK_DBG("%s", __func__);
	diag_cleanup();
	kfree(f->config);
	f->config = NULL;
}

static int
diag_function_bind_config(struct android_usb_function *f,
		struct usb_configuration *c)
{
	int i;
	int ret = 0;
	struct diag_function_config *config = f->config;
	USB_DEBUG("diag config, instances:%d",config->instances);
	USBSTACK_DBG("%s", __func__);
	for (i = 0; i < config->instances; i++) {
		//ret = acm_bind_config(c, i);
		USB_DEBUG("instance :%d", i);

#ifdef _USE_MBIM//CONFIG_USB_F_DIAG_ACM	
		ret = diag_acm_bind_config(c, i);
#else
		ret = diag_bind_config(c, i);
#endif
		USB_DEBUG("ret:%d",ret);
		if (ret) {
			pr_err("Could not bind diag%u config\n", i);
			break;
		}
	}

	return ret;
}

static struct android_usb_function diag_function = {
	.name		= "diag",
	//.enable		= adb_android_function_enable,
	//.disable	= adb_android_function_disable,
	.init		= diag_function_init,
	.cleanup	= diag_function_cleanup,
	.bind_config	= diag_function_bind_config,
};

struct adb_data {
	bool opened;
	bool enabled;
};

static int
adb_function_init(struct android_usb_function *f,
		struct usb_composite_dev *cdev)
{
	USBSTACK_DBG("%s", __func__);
	f->config = kzalloc(sizeof(struct adb_data), GFP_KERNEL);
	if (!f->config)
		return -ENOMEM;

	return adb_setup();
}

static void adb_function_cleanup(struct android_usb_function *f)
{
	USBSTACK_DBG("%s", __func__);
	adb_cleanup();
	kfree(f->config);
}

static int
adb_function_bind_config(struct android_usb_function *f,
		struct usb_configuration *c)
{
	USBSTACK_DBG("%s", __func__);
	return adb_bind_config(c);
}

static void adb_android_function_enable(struct android_usb_function *f)
{
	struct android_dev *dev = _android_dev;
	struct adb_data *data = f->config;
	USBSTACK_DBG("%s, open:%d", __func__, data->opened);
	data->enabled = true;

	/* Disable the gadget until adbd is ready */
	if (!data->opened)
		android_disable(dev);
}

static void adb_android_function_disable(struct android_usb_function *f)
{
	struct android_dev *dev = _android_dev;
	struct adb_data *data = f->config;
	USBSTACK_DBG("%s data opend:%d", __func__, data->opened);
	data->enabled = false;

	/* Balance the disable that was called in closed_callback */
	if (!data->opened)
		android_enable(dev);
}

static struct android_usb_function adb_function = {
	.name		= "adb",
	//.enable		= adb_android_function_enable,
	//.disable	= adb_android_function_disable,
	.init		= adb_function_init,
	.cleanup	= adb_function_cleanup,
	.bind_config	= adb_function_bind_config,
};

static void adb_ready_callback(void)
{
	struct android_dev *dev = _android_dev;
	struct adb_data *data = adb_function.config;
	USBSTACK_DBG("%s, data enable:%d", __func__, data->enabled);
	mutex_lock(&dev->mutex);

	data->opened = true;

	if (data->enabled){
		usb_gadget_set_selfpowered(dev->cdev->gadget);
		android_enable(dev);
	}

	mutex_unlock(&dev->mutex);
	USBSTACK_DBG("%s,%u", __func__, __LINE__);
}

static void adb_closed_callback(void)
{
	struct android_dev *dev = _android_dev;
	struct adb_data *data = adb_function.config;
	USBSTACK_DBG("%s data enabled:%d", __func__, data->enabled);
	mutex_lock(&dev->mutex);

	data->opened = false;

	if (data->enabled){
		dwc_otg_clk_enable(1);
		android_disable(dev);
		usb_gadget_clear_selfpowered(dev->cdev->gadget);
	}

	mutex_unlock(&dev->mutex);
	USBSTACK_DBG("%s %u", __func__, __LINE__);
}


#define MAX_ACM_INSTANCES 3
#define MAX_SERIAL_INSTANCES 4
struct acm_function_config {
	int instances;
};

static int
acm_function_init(struct android_usb_function *f,
		struct usb_composite_dev *cdev)
{
	USB_DEBUG("ACM INIT");
	USBSTACK_DBG("%s", __func__);
	f->config = kzalloc(sizeof(struct acm_function_config), GFP_KERNEL);
	if (!f->config)
		return -ENOMEM;
	((struct acm_function_config *)(f->config))->instances =1;
	return gserial_setup(cdev->gadget, MAX_SERIAL_INSTANCES);
}

static void acm_function_cleanup(struct android_usb_function *f)
{
	USBSTACK_DBG("%s", __func__);
	gserial_cleanup();
	kfree(f->config);
	f->config = NULL;
}

static int
acm_function_bind_config(struct android_usb_function *f,
		struct usb_configuration *c)
{
	int i;
	int ret = 0;
	struct acm_function_config *config = f->config;
	USB_DEBUG("acm config, instances:%d",config->instances);
	USBSTACK_DBG("%s", __func__);
	for (i = 0; i < config->instances; i++) {
		USB_DEBUG("instance :%d", i);
		ret = acm_bind_config(c, i);
		//ret = gser_bind_config(c, i);
		USB_DEBUG("ret:%d",ret);
		if (ret) {
			pr_err("Could not bind acm%u config\n", i);
			break;
		}
	}

	return ret;
}

static int
serial_function_bind_config(struct android_usb_function *f,
		struct usb_configuration *c)
{
	int i;
	int ret = 0;
	struct acm_function_config *config = f->config;
	USB_DEBUG("acm config, instances:%d",config->instances);
	USBSTACK_DBG("%s", __func__);
	for (i = 0; i < config->instances; i++) {
		USB_DEBUG("instance :%d", i);
		ret = gser_bind_config(c, i);
		USB_DEBUG("ret:%d",ret);
		if (ret) {
			pr_err("Could not bind acm%u config\n", i);
			break;
		}
	}

	return ret;
}

static ssize_t acm_instances_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct android_usb_function *f = dev_get_drvdata(dev);
    if(NULL == f)
    {
        return -EINVAL;
    }
	struct acm_function_config *config = f->config;
    if(NULL == config)
    {
        return -EINVAL;
    }

	USBSTACK_DBG("%s", __func__);
	return sprintf(buf, "%d\n", config->instances);
}

static ssize_t acm_instances_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t size)
{
	struct android_usb_function *f = dev_get_drvdata(dev);
    if(NULL == f)
    {
        return -EINVAL;
    }
    struct acm_function_config *config = f->config;
	if(NULL == config)
    {
        return -EINVAL;
    }
    int value;
	USBSTACK_DBG("%s", __func__);
	sscanf(buf, "%d", &value);
	if ((value > MAX_ACM_INSTANCES)||(value<1)){
		USB_DEBUG("WARNNING:SET INVALID ACM NUM:%d!!!", value);
		value = MAX_ACM_INSTANCES;
	}
	config->instances = value;
	return size;
}

static DEVICE_ATTR(instances, S_IRUGO | S_IWUSR, acm_instances_show,
						 acm_instances_store);
static struct device_attribute *acm_function_attributes[] = {
	&dev_attr_instances,
	NULL
};

static struct android_usb_function acm_function = {
	.name		= "acm",
	.init		= acm_function_init,
	.cleanup	= acm_function_cleanup,
	.bind_config	= acm_function_bind_config,
	.attributes	= acm_function_attributes,
};


static int
serial_function_init(struct android_usb_function *f,
		struct usb_composite_dev *cdev)
{
	USB_DEBUG("ACM INIT");
	USBSTACK_DBG("%s", __func__);
	f->config = kzalloc(sizeof(struct acm_function_config), GFP_KERNEL);
	if (!f->config)
		return -ENOMEM;
	((struct acm_function_config *)(f->config))->instances =1;
	return gserial_setup(cdev->gadget, MAX_SERIAL_INSTANCES);
}


static ssize_t serial_instances_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct android_usb_function *f = dev_get_drvdata(dev);
    if(NULL == f)
    {
        return -EINVAL;
    }
    struct acm_function_config *config = f->config;
    if(NULL == config)
    {
        return -EINVAL;
    }
    USBSTACK_DBG("%s", __func__);
	return sprintf(buf, "%d\n", config->instances);
}

static ssize_t serial_instances_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t size)
{
	struct android_usb_function *f = dev_get_drvdata(dev);
    if(NULL == f)
    {
        return -EINVAL;
    }
    struct acm_function_config *config = f->config;
    if(NULL == config)
    {
        return -EINVAL;
    }
    int value;
	USBSTACK_DBG("%s", __func__);
	sscanf(buf, "%d", &value);
	if ((value > MAX_ACM_INSTANCES)||(value<1)){
		USB_DEBUG("WARNNING:SET INVALID ACM NUM:%d!!!", value);
		value = MAX_ACM_INSTANCES;
	}
	config->instances = value;
	return size;
}

static DEVICE_ATTR(instances_serial, S_IRUGO | S_IWUSR, serial_instances_show,
						 serial_instances_store);
static struct device_attribute *serial_function_attributes[] = {
	&dev_attr_instances_serial,
	NULL
};


static struct android_usb_function serial_function = {
	.name		= "serial",
	.init		= serial_function_init,
	.cleanup	= acm_function_cleanup,
	.bind_config	= serial_function_bind_config,
	.attributes	= serial_function_attributes,
};


#if 0
static int
mtp_function_init(struct android_usb_function *f,
		struct usb_composite_dev *cdev)
{
	return mtp_setup();
}

static void mtp_function_cleanup(struct android_usb_function *f)
{
	mtp_cleanup();
}

static int
mtp_function_bind_config(struct android_usb_function *f,
		struct usb_configuration *c)
{
	return mtp_bind_config(c, false);
}

static int
ptp_function_init(struct android_usb_function *f,
		struct usb_composite_dev *cdev)
{
	/* nothing to do - initialization is handled by mtp_function_init */
	return 0;
}

static void ptp_function_cleanup(struct android_usb_function *f)
{
	/* nothing to do - cleanup is handled by mtp_function_cleanup */
}

static int
ptp_function_bind_config(struct android_usb_function *f,
		struct usb_configuration *c)
{
	return mtp_bind_config(c, true);
}

static int mtp_function_ctrlrequest(struct android_usb_function *f,
					struct usb_composite_dev *cdev,
					const struct usb_ctrlrequest *c)
{
	return mtp_ctrlrequest(cdev, c);
}

static struct android_usb_function mtp_function = {
	.name		= "mtp",
	.init		= mtp_function_init,
	.cleanup	= mtp_function_cleanup,
	.bind_config	= mtp_function_bind_config,
	.ctrlrequest	= mtp_function_ctrlrequest,
};

/* PTP function is same as MTP with slightly different interface descriptor */
static struct android_usb_function ptp_function = {
	.name		= "ptp",
	.init		= ptp_function_init,
	.cleanup	= ptp_function_cleanup,
	.bind_config	= ptp_function_bind_config,
};
#endif

struct rndis_function_config {
	u8      ethaddr[ETH_ALEN];
	u32     vendorID;
	char	manufacturer[256];
	/* "Wireless" RNDIS; auto-detected by Windows */
	bool	wceis;
};

static int
rndis_function_init(struct android_usb_function *f,
		struct usb_composite_dev *cdev)
{
	USBSTACK_DBG("%s", __func__);
	f->config = kzalloc(sizeof(struct rndis_function_config), GFP_KERNEL);
	if (!f->config)
		return -ENOMEM;
	return 0;
}

static void rndis_function_cleanup(struct android_usb_function *f)
{
	USBSTACK_DBG("%s", __func__);
	kfree(f->config);
	f->config = NULL;
}

extern unsigned int ecm_setup_work_time;
static int
rndis_function_bind_config(struct android_usb_function *f,
		struct usb_configuration *c)
{
	int ret;
	struct rndis_function_config *rndis = f->config;

	if(ecm_setup_work_time == 1){
		ecm_setup_work_time = 0;
		ecm_work_run_cnt = 0;
	}
	if (!rndis) {
		pr_err("%s: rndis_pdata\n", __func__);
		return -1;
	}
	USBSTACK_DBG("%s", __func__);

	pr_info("%s MAC: %02X:%02X:%02X:%02X:%02X:%02X\n", __func__,
		rndis->ethaddr[0], rndis->ethaddr[1], rndis->ethaddr[2],
		rndis->ethaddr[3], rndis->ethaddr[4], rndis->ethaddr[5]);

    
	ret =  gether_setup_name(c->cdev->gadget, rndis->ethaddr, USB_VNIC_NAME);
	if (ret) {
		printk("%s: gether_setup_name failed, ret:%d", __func__, ret);
		return ret;
	}

    //Ӧòδʹ
	if (rndis->wceis) {
		/* "Wireless" RNDIS; auto-detected by Windows */
		rndis_iad_descriptor.bFunctionClass =
						USB_CLASS_WIRELESS_CONTROLLER;
		rndis_iad_descriptor.bFunctionSubClass = 0x01;
		rndis_iad_descriptor.bFunctionProtocol = 0x03;
		rndis_control_intf.bInterfaceClass =
						USB_CLASS_WIRELESS_CONTROLLER;
		rndis_control_intf.bInterfaceSubClass =	 0x01;
		rndis_control_intf.bInterfaceProtocol =	 0x03;
	}
    return  rndis_bind_config_vendor(c, rndis->ethaddr, rndis->vendorID,rndis->manufacturer); 
    
}

static void rndis_function_unbind_config(struct android_usb_function *f,
						struct usb_configuration *c)
{
	USBSTACK_DBG("%s", __func__);
	gether_cleanup();
}

static ssize_t rndis_manufacturer_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct android_usb_function *f = dev_get_drvdata(dev);
    if(NULL == f)
    {
        return -EINVAL;
    }
	struct rndis_function_config *config = f->config;
    if(NULL == config)
    {
        return -EINVAL;
    }    
	return sprintf(buf, "%s\n", config->manufacturer);
}

static ssize_t rndis_manufacturer_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t size)
{
	struct android_usb_function *f = dev_get_drvdata(dev);
    if(NULL == f)
    {
        return -EINVAL;
    }
	struct rndis_function_config *config = f->config;
    if(NULL == config)
    {
        return -EINVAL;
    }
	if (size >= sizeof(config->manufacturer))
		return -EINVAL;
	if (sscanf(buf, "%s", config->manufacturer) == 1)
		return size;
	return -1;
}

static DEVICE_ATTR(manufacturer, S_IRUGO | S_IWUSR, rndis_manufacturer_show,
						    rndis_manufacturer_store);

static ssize_t rndis_wceis_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct android_usb_function *f = dev_get_drvdata(dev);
    if(NULL == f)
    {
        return -EINVAL;
    }
	struct rndis_function_config *config = f->config;
    if(NULL == config)
    {
        return -EINVAL;
    }

	return sprintf(buf, "%d\n", config->wceis);
}

static ssize_t rndis_wceis_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t size)
{
	struct android_usb_function *f = dev_get_drvdata(dev);
    if(NULL == f)
    {
        return -EINVAL;
    }
	struct rndis_function_config *config = f->config;
    if(NULL == config)
    {
        return -EINVAL;
    }
	int value;

	if (sscanf(buf, "%d", &value) == 1) {
		config->wceis = value;
		return size;
	}
	return -EINVAL;
}

static DEVICE_ATTR(wceis, S_IRUGO | S_IWUSR, rndis_wceis_show,
					     rndis_wceis_store);

static ssize_t rndis_ethaddr_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct android_usb_function *f = dev_get_drvdata(dev);
    if(NULL == f)
    {
        return -EINVAL;
    }
	struct rndis_function_config *rndis = f->config;
    if(NULL == rndis)
    {
        return -EINVAL;
    }

	return sprintf(buf, "%02x:%02x:%02x:%02x:%02x:%02x\n",
		rndis->ethaddr[0], rndis->ethaddr[1], rndis->ethaddr[2],
		rndis->ethaddr[3], rndis->ethaddr[4], rndis->ethaddr[5]);
}

static ssize_t rndis_ethaddr_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t size)
{
	struct android_usb_function *f = dev_get_drvdata(dev);
    if(NULL == f)
    {
        return -EINVAL;
    }
	struct rndis_function_config *rndis = f->config;
    if(NULL == rndis)
    {
        return -EINVAL;
    }

	if (sscanf(buf, "%02x:%02x:%02x:%02x:%02x:%02x\n",
		    (unsigned int *)&rndis->ethaddr[0], (unsigned int *)&rndis->ethaddr[1],
		    (unsigned int *)&rndis->ethaddr[2], (unsigned int *)&rndis->ethaddr[3],
		    (unsigned int *)&rndis->ethaddr[4], (unsigned int *)&rndis->ethaddr[5]) == 6)
		return size;
	return -EINVAL;
}

static DEVICE_ATTR(ethaddr, S_IRUGO | S_IWUSR, rndis_ethaddr_show,
					       rndis_ethaddr_store);

static ssize_t rndis_vendorID_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct android_usb_function *f = dev_get_drvdata(dev);
    if(NULL == f)
    {
        return -EINVAL;
    }
	struct rndis_function_config *config = f->config;
    if(NULL == config)
    {
        return -EINVAL;
    }
	return sprintf(buf, "%04x\n", config->vendorID);
}

static ssize_t rndis_vendorID_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t size)
{
	struct android_usb_function *f = dev_get_drvdata(dev);
    if(NULL == f)
    {
        return -EINVAL;
    }

	struct rndis_function_config *config = f->config;
    if(NULL == config)
    {
        return -EINVAL;
    }
	int value;

	if (sscanf(buf, "%04x", (unsigned int *)(&value)) == 1) {
		config->vendorID = value;
		return size;
	}
	return -EINVAL;
}

static DEVICE_ATTR(vendorID, S_IRUGO | S_IWUSR, rndis_vendorID_show,
						rndis_vendorID_store);


extern unsigned long long g_test_xmit_pktnum;
extern unsigned long long g_test_xmit_pkterr1;
extern unsigned long long g_test_xmit_pkterr2;
extern unsigned long long g_test_rx_pkt ;
extern unsigned long long g_test_rx_complt_pkt;

static ssize_t rndis_pktNum_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	return sprintf(buf, "pktnun:%04x, tx_total:%lld, tx_err1:%lld, tx_err2:%lld,rx:%lld, rx_cpl:%lld\n", multi_packet_num, 
		g_test_xmit_pktnum, g_test_xmit_pkterr1, g_test_xmit_pkterr2,
		g_test_rx_pkt,g_test_rx_complt_pkt);
}

static ssize_t rndis_pktNum_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t size)
{
	int value;

	sscanf(buf, "%d", &value);
	if ((value > RNDIS_PACKET_MAX_NUM )||(value<1)){
		USB_DEBUG("WARNNING:SET INVALID PACKET NUM:%d!!!", value);
		value = RNDIS_PACKET_MAX_NUM;
	}
	multi_packet_num = value;
	return size;
}
static DEVICE_ATTR(pktNum, S_IRUGO | S_IWUSR, rndis_pktNum_show,
						rndis_pktNum_store);

static ssize_t ether_skbNum_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	return sprintf(buf, "skbnun:%04x\n", ether_skb_num);
}

static ssize_t ether_skbNum_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t size)
{
	int value;

	sscanf(buf, "%d", &value);
	if ((value > 512)||(value<5)){
		USB_DEBUG("WARNNING:SET INVALID PACKET NUM:%d!!!", value);
		value = 512;
	}
	ether_skb_num = value;
	return size;
}

static DEVICE_ATTR(skbNum, S_IRUGO | S_IWUSR, ether_skbNum_show,
						ether_skbNum_store);

static struct device_attribute *rndis_function_attributes[] = {
	&dev_attr_manufacturer,
	&dev_attr_wceis,
	&dev_attr_ethaddr,
	&dev_attr_vendorID,
	&dev_attr_pktNum,
	&dev_attr_skbNum,
	NULL
};

static struct android_usb_function rndis_function = {
	.name		= "rndis",
	.init		= rndis_function_init,
	.cleanup	= rndis_function_cleanup,
	.bind_config	= rndis_function_bind_config,
	.unbind_config	= rndis_function_unbind_config,
	.attributes	= rndis_function_attributes,
};



struct mbim_function_config {
	u8      ethaddr[ETH_ALEN];
	u32     vendorID;
	char	manufacturer[256];
	bool	wceis;
};

static int mbim_function_init(struct android_usb_function *f,
		struct usb_composite_dev *cdev)
{
	printk("mbim_function_init\n");
	USBSTACK_DBG("%s", __func__);
    int ret = 0 ;
	f->config = kzalloc(sizeof(struct mbim_function_config), GFP_KERNEL);
	if (!f->config)
		return -ENOMEM;
#if 1
	ret = mbim_conn_chanel_init() ;

    if(ret)
    {
        kfree(f->config) ;
        f->config = NULL ;
        return ret ;
    }
#endif
	return 0;
}

static void mbim_function_cleanup(struct android_usb_function *f)
{
	USBSTACK_DBG("%s", __func__);
	kfree(f->config);
	f->config = NULL;
}

//extern unsigned int ecm_setup_work_time;
static int mbim_function_bind_config(struct android_usb_function *f,
		struct usb_configuration *c)
{
	printk("mbim_function_bind_config\n");
	int ret;
	struct mbim_function_config *mbim = f->config;

	
	if (!mbim) {
		pr_err("%s: mbim_pdata\n", __func__);
		return -1;
	}
	USBSTACK_DBG("%s", __func__);

	pr_info("%s MAC: %02X:%02X:%02X:%02X:%02X:%02X\n", __func__,
		mbim->ethaddr[0], mbim->ethaddr[1], mbim->ethaddr[2],
		mbim->ethaddr[3], mbim->ethaddr[4], mbim->ethaddr[5]);

	ret = mbim_bind_config(c, mbim->ethaddr); 
	if (ret) {
		printk("%s: mbim_bind_config failed, ret:%d", __func__, ret);
		return ret;
	}
	return gether_setup_name(c->cdev->gadget, mbim->ethaddr, USB_VNIC_NAME);
}

static void mbim_function_unbind_config(struct android_usb_function *f,
						struct usb_configuration *c)
{
	USBSTACK_DBG("%s", __func__);
	gether_cleanup();
}

#if 0
static int mbim_function_ctrlrequest(struct android_usb_function * f,
                    struct usb_composite_dev *dev,
                    const struct usb_ctrlrequest * ctrl)
{
    int value = -1;
    struct mbim_function_config *mbim = f->config;
#if 1	
    switch (ctrl->bRequestType & USB_TYPE_MASK){
    case USB_TYPE_STANDARD:
        if(ctrl->bRequest == USB_REQ_SET_CONFIGURATION){
            if((!mbim->receive_setIfac)&&(!mbim->receive_setICfg)){
                printk("mbim-receive_setICfg\n");
                //schedule_delayed_work(&ecm->work, msecs_to_jiffies(ECM_TIMEOUT));
            }           
            if((mbim->receive_setIfac)&&(!mbim->receive_setICfg))
                gether_mbim_uevent(mbim->instances, 1);
            mbim->receive_setICfg = 1;
        }else if(ctrl->bRequest == USB_REQ_SET_INTERFACE){
            u8  intf = (le16_to_cpu(ctrl->wIndex)) & 0xFF;
            struct usb_function *f_intf = dev->config->interface[intf];
            struct f_mbim       *f_mbim = func_to_mbim(f_intf);
                printk("ecm-receive_setinterface\n");
            if (intf != f_mbim->data_id)
                return value;
            if(mbim->receive_setICfg){
                //USBSTACK_DBG("ecm-delayed-work cancel");
                //cancel_delayed_work_sync(&ecm->work);
                gether_mbim_uevent(mbim->instances, 1);
            }
            mbim->receive_setIfac = 1;
        }
        break;
    case USB_TYPE_VENDOR:
        if((ctrl->bRequest == bMS_Code_Original )||
            (ctrl->bRequest == bMS_Code_Change )){
                printk("mbim-sys-id-err");
                schedule_delayed_work(&mbim->work, msecs_to_jiffies(50));
        }
        break;  
    case USB_TYPE_CLASS:
        if(ctrl->bRequest == USB_CDC_GET_NTB_PARAMETERS){
        value = w_length > sizeof mbim_ntb_parameters ?
            sizeof mbim_ntb_parameters : w_length;
        memcpy(req->buf, &mbim_ntb_parameters, value);
        //USBSTACK_DBG(cdev, "Host asked NTB parameters\n");
        break;
        }
    }
#endif
    return value;
}
#endif


#if 0
static ssize_t mbim_manufacturer_show(struct device *dev,
        struct device_attribute *attr, char *buf)
{
    struct android_usb_function *f = dev_get_drvdata(dev);
    struct rndis_function_config *config = f->config;
    return sprintf(buf, "%s\n", config->manufacturer);
}

static ssize_t mbim_manufacturer_store(struct device *dev,
        struct device_attribute *attr, const char *buf, size_t size)
{
    struct android_usb_function *f = dev_get_drvdata(dev);
    struct mbim_function_config *config = f->config;

    if (size >= sizeof(config->manufacturer))
        return -EINVAL;
    if (sscanf(buf, "%s", config->manufacturer) == 1)
        return size;
    return -1;
}

static DEVICE_ATTR(manufacturer, S_IRUGO | S_IWUSR, mbim_manufacturer_show,
                            mbim_manufacturer_store);

static ssize_t mbim_wceis_show(struct device *dev,
        struct device_attribute *attr, char *buf)
{
    struct android_usb_function *f = dev_get_drvdata(dev);
    struct mbim_function_config *config = f->config;
    return sprintf(buf, "%d\n", config->wceis);
}
static ssize_t mbim_wceis_store(struct device *dev,
        struct device_attribute *attr, const char *buf, size_t size)
{
    struct android_usb_function *f = dev_get_drvdata(dev);
    struct mbim_function_config *config = f->config;
    int value;

    if (sscanf(buf, "%d", &value) == 1) {
        config->wceis = value;
        return size;
    }
    return -EINVAL;
}

static DEVICE_ATTR(wceis, S_IRUGO | S_IWUSR, mbim_wceis_show,
                         mbim_wceis_store);

static ssize_t mbim_ethaddr_show(struct device *dev,
        struct device_attribute *attr, char *buf)
{
    struct android_usb_function *f = dev_get_drvdata(dev);
    struct mbim_function_config *mbim = f->config;
    return sprintf(buf, "%02x:%02x:%02x:%02x:%02x:%02x\n",
        mbim->ethaddr[0], mbim->ethaddr[1], mbim->ethaddr[2],
        mbim->ethaddr[3], mbim->ethaddr[4], mbim->ethaddr[5]);
}

static ssize_t mbim_ethaddr_store(struct device *dev,
        struct device_attribute *attr, const char *buf, size_t size)
{
    struct android_usb_function *f = dev_get_drvdata(dev);
    struct mbim_function_config *mbim = f->config;

    if (sscanf(buf, "%02x:%02x:%02x:%02x:%02x:%02x\n",
            (unsigned int *)&mbim->ethaddr[0], (unsigned int *)&mbim->ethaddr[1],
            (unsigned int *)&mbim->ethaddr[2], (unsigned int *)&mbim->ethaddr[3],
            (unsigned int *)&mbim->ethaddr[4], (unsigned int *)&mbim->ethaddr[5]) == 6)
        return size;
    return -EINVAL;
}

static DEVICE_ATTR(ethaddr, S_IRUGO | S_IWUSR, mbim_ethaddr_show,
                           mbim_ethaddr_store);

static ssize_t mbim_vendorID_show(struct device *dev,
        struct device_attribute *attr, char *buf)
{
    struct android_usb_function *f = dev_get_drvdata(dev);
    struct mbim_function_config *config = f->config;
    return sprintf(buf, "%04x\n", config->vendorID);
}

static ssize_t mbim_vendorID_store(struct device *dev,
        struct device_attribute *attr, const char *buf, size_t size)
{
    struct android_usb_function *f = dev_get_drvdata(dev);
    struct mbim_function_config *config = f->config;
    int value;

    if (sscanf(buf, "%04x", (unsigned int *)(&value)) == 1) {
        config->vendorID = value;
        return size;
    }
    return -EINVAL;
}

static DEVICE_ATTR(vendorID, S_IRUGO | S_IWUSR, mbim_vendorID_show,
                        mbim_vendorID_store);


extern unsigned long long g_test_xmit_pktnum;
extern unsigned long long g_test_xmit_pkterr1;
extern unsigned long long g_test_xmit_pkterr2;
extern unsigned long long g_test_rx_pkt ;
extern unsigned long long g_test_rx_complt_pkt;

static DEVICE_ATTR(skbNum, S_IRUGO | S_IWUSR, ether_skbNum_show,
                        ether_skbNum_store);
#endif

static ssize_t mbim_pktNum_show(struct device *dev,
        struct device_attribute *attr, char *buf)
{
    return sprintf(buf, "pktnun:%04x, tx_total:%lld, tx_err1:%lld, tx_err2:%lld,rx:%lld, rx_cpl:%lld\n", multi_packet_num, 
        g_test_xmit_pktnum, g_test_xmit_pkterr1, g_test_xmit_pkterr2,
        g_test_rx_pkt,g_test_rx_complt_pkt);
}

static ssize_t mbim_pktNum_store(struct device *dev,
        struct device_attribute *attr, const char *buf, size_t size)
{
    int value;

    sscanf(buf, "%d", &value);
    if ((value > MBIM_PACKET_MAX_NUM) || (value<1)){
        USB_DEBUG("WARNNING:SET INVALID PACKET NUM:%d!!!", value);
        value = MBIM_PACKET_MAX_NUM ;
    }
    multi_packet_num = value;
    return size;
}
static DEVICE_ATTR(mbimPktNum, S_IRUGO | S_IWUSR, mbim_pktNum_show,
                                mbim_pktNum_store);

static struct device_attribute *mbim_function_attributes[] = {
    //&dev_attr_mbim,
    //&dev_attr_wceis,
    //&dev_attr_ethaddr,
    //&dev_attr_vendorID,
    &dev_attr_mbimPktNum,
    //&dev_attr_skbNum,
    NULL,
} ;





 static struct android_usb_function mbim_function = {
      .name       = "mbim",
      .init       = mbim_function_init,
      .cleanup    = mbim_function_cleanup,
      .bind_config    = mbim_function_bind_config,
      .unbind_config  = mbim_function_unbind_config,
      .attributes   = mbim_function_attributes,
       //.ctrlrequest  = mbim_function_ctrlrequest,
     };



struct mass_storage_function_config {
	struct fsg_config fsg;
	struct fsg_common *common;
	int 	fsg_mods_init;
};



static char OS_str_des[] = 
{	0x12, 0x03, 0x4D, 0x00, 0x53, 0x00, 0x46, 0x00, 
	0x54, 0x00, 0x31, 0x00, 0x30, 0x00, 0x30, 0x00, 
	0x04, 0x00 
};

static char  OS_ext_cfg_des_null[] =
{
	0x28,0x00,0x00,0x00,0x00,0x01,0x04,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
};

static int mass_storage_function_ctrlrequest(struct android_usb_function * f,
					struct usb_composite_dev *dev,
					const struct usb_ctrlrequest * ctrl)
{
	int	value = -1;
	u16	w_value = le16_to_cpu(ctrl->wValue);
	u16	w_index = le16_to_cpu(ctrl->wIndex);
	u16	w_length = le16_to_cpu(ctrl->wLength);
	
	struct mass_storage_function_config *config = f->config;
	struct usb_request	*req = dev->req;

	/* ǵãʶϵͳ */
	if(!config->fsg_mods_init){
		/*modsйأ δmodsֱӷ */
		return value;
	}

	switch (ctrl->bRequestType & USB_TYPE_MASK){

	case USB_TYPE_STANDARD:
		if(ctrl->bRequest == USB_REQ_GET_DESCRIPTOR){
			if(((w_value>>8) == USB_DT_STRING)&&((w_value & 0xFF) == OS_string_descriptor_id)){
				USBSTACK_DBG("Get Sys des");
				printk("Get Sys des\n");
				value = min(w_length, sizeof(OS_str_des));
				memcpy(req->buf, OS_str_des, value);
				req->length = value;
				req->zero = value < w_length;
				
				value = usb_ep_queue(dev->gadget->ep0, req, GFP_ATOMIC);
				if (value < 0) {
					USBSTACK_DBG("ep_queue --> %d", value);
					req->status = 0;
					composite_setup_complete(dev->gadget->ep0, req);
				}
			}
		}else if(ctrl->bRequest == USB_REQ_SET_CONFIGURATION){
		//	usb_mods_activate();
		}
		break;

	case USB_TYPE_CLASS:
		if(ctrl->bRequest == US_BULK_GET_MAX_LUN){
			usb_mods_activate();
		}
		break;
	case USB_TYPE_VENDOR:
		if(ctrl->bRequest == bMS_Code_Original ){
			USBSTACK_DBG("os string bms org code");
			printk("os string bms org code\n");
#if ((defined CONFIG_ARCH_ZX297520V3_MIFI)||(defined CONFIG_ARCH_ZX297520V3_UFI)) && (defined CONFIG_MIN_VERSION)
			usb_set_ms_auto_reject(1);
#else
			usb_set_ms_auto_reject(0);
#endif
		}else if(ctrl->bRequest == bMS_Code_Change){
			USBSTACK_DBG("os string bms changed code");
			printk("os string bms changed code\n");
			usb_set_ms_auto_reject(1);
		}
		if((ctrl->bRequest == bMS_Code_Original )||
			(ctrl->bRequest == bMS_Code_Change )){
			usb_set_sys_id(0);
			value = min(w_length, sizeof(OS_ext_cfg_des_null));
			memcpy(req->buf, &OS_ext_cfg_des_null[0], value);
			req->length = value;
			req->zero = value < w_length;
			value = usb_ep_queue(dev->gadget->ep0, req, GFP_ATOMIC);
			if (value < 0) {
				USBSTACK_DBG("ep_queue --> %d", value);
				req->status = 0;
				composite_setup_complete(dev->gadget->ep0, req);
			}
		}
		break;
	}
		
	return value;
}


static int mass_storage_function_init(struct android_usb_function *f,
					struct usb_composite_dev *cdev)
{
	struct mass_storage_function_config *config;
	struct fsg_common *common;
	int err;
	USBSTACK_DBG("%s", __func__);
	config = kzalloc(sizeof(struct mass_storage_function_config),
								GFP_KERNEL);
	if (!config)
		return -ENOMEM;

	/* config cdrom */
	config->fsg.nluns = 1;
	config->fsg.luns[0].removable = 1;
	config->fsg.luns[0].cdrom = 1;
	config->fsg.luns[0].ro = 1;

#if CFG_LUN_NUM_TWO
	/* config Udisk */
	config->fsg.nluns = 2;
	config->fsg.luns[1].removable = 1;
	config->fsg.luns[1].cdrom = 0;
	config->fsg.luns[1].ro = 0;
#endif

	common = fsg_common_init(NULL, cdev, &config->fsg);
	if (IS_ERR(common)) {
		kfree(config);
		return PTR_ERR(common);
	}

	err = sysfs_create_link(&f->dev->kobj,
				&common->luns[0].dev.kobj,
				"lun");
	if (err) {
		kfree(config);
        kfree(common);
		return err;
	}

	config->common = common;
	f->config = config;

	return 0;
}

static void mass_storage_function_cleanup(struct android_usb_function *f)
{
    struct mass_storage_function_config *config;
	USBSTACK_DBG("%s", __func__);
	//add by gsn, must kill process file-storage 
	config = (struct mass_storage_function_config *)f->config;
	if(config != NULL){
		fsg_common_release(&(config->common->ref));
	}
	kfree(f->config);
	f->config = NULL;
}

static int mass_storage_function_bind_config(struct android_usb_function *f,
						struct usb_configuration *c)
{
	struct mass_storage_function_config *config = f->config;
	USBSTACK_DBG("%s", __func__);
	//usb_set_ms_auto_eject(1);//  auto-reject cdrom 
	if(_android_dev->cdrom_only){
		usb_mods_init();
		config->fsg_mods_init = 1;
	}
	return fsg_bind_config(c->cdev, c, config->common);
}
static void mass_storage_function_unbind_config(struct android_usb_function *f,
						struct usb_configuration *c)
{
	struct mass_storage_function_config *config = f->config;
	USBSTACK_DBG("%s", __func__);
	if(config->fsg_mods_init){
		usb_mods_exit();
		config->fsg_mods_init = 0;		
	}
}

#if CFG_LUN_NUM_TWO

static ssize_t mass_storage_nluns_show(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	struct android_usb_function *f = dev_get_drvdata(dev);
    if(NULL == f)
    {
        return -EINVAL;
    }
	struct mass_storage_function_config *config = f->config;
    if(NULL == config)
    {
        return -EINVAL;
    }

	USBSTACK_DBG("fsg nluns_show: %d", config->common->nluns);
	
	return sprintf(buf, "%d\n", config->common->nluns);
}

static ssize_t mass_storage_nluns_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t size)
{
	struct android_usb_function *f = dev_get_drvdata(dev);
    if(NULL == f)
    {
        return -EINVAL;
    }
	struct mass_storage_function_config *config = f->config;
    if(NULL == config)
    {
        return -EINVAL;
    }

	int value;							
	if (sscanf(buf, "%d", &value) == 1) {
		if((value>2)||(value<1)){
			USB_DEBUG("WARNNING:SET INVALID LUN NUM:%d!!!", value);
			value = 2;
		}
		
		config->common->nluns = value;	

		USBSTACK_DBG("fsg nluns_store: %d", value);
		return size;						
	}								
	return -1;							
}

static DEVICE_ATTR(nluns, S_IRUGO | S_IWUSR,
					mass_storage_nluns_show,
					mass_storage_nluns_store);

#endif

static ssize_t mass_storage_inquiry_show(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	struct android_usb_function *f = dev_get_drvdata(dev);
    if(NULL == f)
    {
        return -EINVAL;
    }
	struct mass_storage_function_config *config = f->config;
    if(NULL == config)
    {
        return -EINVAL;
    }

	return sprintf(buf, "%s\n", config->common->inquiry_string);
}

static ssize_t mass_storage_inquiry_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t size)
{
	struct android_usb_function *f = dev_get_drvdata(dev);
    if(NULL == f)
    {
        return -EINVAL;
    }	
    struct mass_storage_function_config *config = f->config;
    if(NULL == config)
    {
        return -EINVAL;
    }
    if (size >= sizeof(config->common->inquiry_string))
		return -EINVAL;

	//sscanfոֹͣ, ޸Ϊmemcpy
#if 1
	memset(config->common->inquiry_string, 0, sizeof(config->common->inquiry_string));
	memcpy(config->common->inquiry_string, buf, size);
#else
	if (sscanf(buf, "%s", config->common->inquiry_string) != 1)
		return -EINVAL;
#endif
	return size;
}

static DEVICE_ATTR(inquiry_string, S_IRUGO | S_IWUSR,
					mass_storage_inquiry_show,
					mass_storage_inquiry_store);

static struct device_attribute *mass_storage_function_attributes[] = {
	//&dev_attr_inquiry_string,
#if CFG_LUN_NUM_TWO
	&dev_attr_nluns,
#endif
	NULL
};

static struct android_usb_function mass_storage_function = {
	.name		= "mass_storage",
	.init		= mass_storage_function_init,
	.cleanup	= mass_storage_function_cleanup,
	.bind_config	= mass_storage_function_bind_config,
	.unbind_config	 = mass_storage_function_unbind_config,
	.attributes	= mass_storage_function_attributes,
	.ctrlrequest	= mass_storage_function_ctrlrequest,
};


#if 0
static int accessory_function_init(struct android_usb_function *f,
					struct usb_composite_dev *cdev)
{
	return acc_setup();
}

static void accessory_function_cleanup(struct android_usb_function *f)
{
	acc_cleanup();
}

static int accessory_function_bind_config(struct android_usb_function *f,
						struct usb_configuration *c)
{
	return acc_bind_config(c);
}

static int accessory_function_ctrlrequest(struct android_usb_function *f,
						struct usb_composite_dev *cdev,
						const struct usb_ctrlrequest *c)
{
	return acc_ctrlrequest(cdev, c);
}


static struct android_usb_function accessory_function = {
	.name		= "accessory",
	.init		= accessory_function_init,
	.cleanup	= accessory_function_cleanup,
	.bind_config	= accessory_function_bind_config,
	.ctrlrequest	= accessory_function_ctrlrequest,
};

static int audio_source_function_init(struct android_usb_function *f,
			struct usb_composite_dev *cdev)
{
	struct audio_source_config *config;

	config = kzalloc(sizeof(struct audio_source_config), GFP_KERNEL);
	if (!config)
		return -ENOMEM;
	config->card = -1;
	config->device = -1;
	f->config = config;
	return 0;
}

static void audio_source_function_cleanup(struct android_usb_function *f)
{
	kfree(f->config);
}

static int audio_source_function_bind_config(struct android_usb_function *f,
						struct usb_configuration *c)
{
	struct audio_source_config *config = f->config;

	return audio_source_bind_config(c, config);
}

static void audio_source_function_unbind_config(struct android_usb_function *f,
						struct usb_configuration *c)
{
	struct audio_source_config *config = f->config;

	config->card = -1;
	config->device = -1;
}

static ssize_t audio_source_pcm_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct android_usb_function *f = dev_get_drvdata(dev);
	struct audio_source_config *config = f->config;

	/* print PCM card and device numbers */
	return sprintf(buf, "%d %d\n", config->card, config->device);
}

static DEVICE_ATTR(pcm, S_IRUGO | S_IWUSR, audio_source_pcm_show, NULL);

static struct device_attribute *audio_source_function_attributes[] = {
	&dev_attr_pcm,
	NULL
};

static struct android_usb_function audio_source_function = {
	.name		= "audio_source",
	.init		= audio_source_function_init,
	.cleanup	= audio_source_function_cleanup,
	.bind_config	= audio_source_function_bind_config,
	.unbind_config	= audio_source_function_unbind_config,
	.attributes	= audio_source_function_attributes,
};
#endif

static struct android_usb_function *supported_functions[] = {
	//&ffs_function,
	&ecm_function,
	&diag_function,
	&adb_function,
	&acm_function,
	&serial_function,
//	&mtp_function,
//	&ptp_function,
	&rndis_function,
	&mass_storage_function,
    &mbim_function,
//	&accessory_function,
//	&audio_source_function,
	NULL
};


static int android_init_functions(struct android_usb_function **functions,
				  struct usb_composite_dev *cdev)
{
	struct android_dev *dev = _android_dev;
	struct android_usb_function *f;
	struct device_attribute **attrs;
	struct device_attribute *attr;
	int err =0;
	int index = 0;
	USBSTACK_DBG("%s", __func__);
	for (; (f = *functions++); index++) {
		f->dev_name = kasprintf(GFP_KERNEL, "f_%s", f->name);
		if(f->dev_name == NULL){
			err = -ENOMEM;
			goto err_out;
		}		
		f->dev = device_create(android_class, dev->dev,
				MKDEV(0, index), f, f->dev_name);
		if (IS_ERR(f->dev)) {
			pr_err("%s: Failed to create dev %s", __func__,
							f->dev_name);
			err = PTR_ERR(f->dev);
			goto err_create;
		}

		if (f->init) {
			err = f->init(f, cdev);
			if (err) {
				pr_err("%s: Failed to init %s", __func__,
								f->name);
				goto err_out;
			}
		}

		attrs = f->attributes;
		if (attrs) {
			while ((attr = *attrs++) && !err)
				err = device_create_file(f->dev, attr);
		}
		if (err) {
			pr_err("%s: Failed to create function %s attributes",
					__func__, f->name);
			goto err_out;
		}
	}
	return 0;

err_out:
	device_destroy(android_class, f->dev->devt);
err_create:
	kfree(f->dev_name);
	return err;
}

static void android_cleanup_functions(struct android_usb_function **functions)
{
	struct android_usb_function *f;
	USBSTACK_DBG("%s", __func__);
	while (*functions) {
		f = *functions++;

		if (f->dev) {
			device_destroy(android_class, f->dev->devt);
			kfree(f->dev_name);
		}

		if (f->cleanup)
			f->cleanup(f);
	}
}

static int
android_bind_enabled_functions(struct android_dev *dev,
			       struct usb_configuration *c)
{
	struct android_usb_function *f;
	int ret;
	USBSTACK_DBG("%s", __func__);
	list_for_each_entry(f, &dev->enabled_functions, enabled_list) {
		USBSTACK_DBG("%s, %u %s", __func__, __LINE__, f->name);
		ret = f->bind_config(f, c);
		if (ret) {
			USBSTACK_DBG("%s: %s failed", __func__, f->name);
			return ret;
		}
	}
	return 0;
}

static void
android_unbind_enabled_functions(struct android_dev *dev,
			       struct usb_configuration *c)
{
	struct android_usb_function *f;
	USBSTACK_DBG("%s", __func__);
	list_for_each_entry(f, &dev->enabled_functions, enabled_list) {
		if (f->unbind_config)
			f->unbind_config(f, c);
	}
}

static int android_enable_function(struct android_dev *dev, char *name)
{
	struct android_usb_function **functions = dev->functions;
	struct android_usb_function *f;

	while ((f = *functions++)) {
		if (!strcmp(name, f->name)) {
			USBSTACK_DBG("%s--%s", __func__ ,  f->name );
			list_add_tail(&f->enabled_list,
						&dev->enabled_functions);
			return 0;
		}
	}
	return -EINVAL;
}

/*-------------------------------------------------------------------------*/
/* /sys/class/android_usb/android%d/ interface */

static ssize_t
functions_show(struct device *pdev, struct device_attribute *attr, char *buf)
{
	struct android_dev *dev = dev_get_drvdata(pdev);
    if(NULL == dev)
    {
        return -EINVAL;
    }
    struct android_usb_function *f;
	char *buff = buf;
	USBSTACK_DBG("%s", __func__);
	mutex_lock(&dev->mutex);

	list_for_each_entry(f, &dev->enabled_functions, enabled_list)
		buff += sprintf(buff, "%s,", f->name);

	mutex_unlock(&dev->mutex);
    USBSTACK_DBG("%s, %u", __func__, __LINE__);
	if (buff != buf)
		*(buff-1) = '\n';
	return buff - buf;
}

static ssize_t
functions_store(struct device *pdev, struct device_attribute *attr,
			       const char *buff, size_t size)
{
	struct android_dev *dev = dev_get_drvdata(pdev);
    if(NULL == dev)
    {
        return -EINVAL;
    }
    char *name;
	char buf[256], *b;
	char aliases[256], *a;
	int err;
	int is_ffs;
	int ffs_enabled = 0;
	int count = 0;
	USBSTACK_DBG("%s", __func__);
	mutex_lock(&dev->mutex);

	if (dev->enabled) {
		mutex_unlock(&dev->mutex);
		USBSTACK_DBG("%s, %u", __func__, __LINE__);
		return -EBUSY;
	}

	INIT_LIST_HEAD(&dev->enabled_functions);

	strlcpy(buf, buff, sizeof(buf));
	b = strim(buf);

	dev->cdrom_only = false;
	while (b) {
		name = strsep(&b, ",");
		if (!name)
			continue;

		is_ffs = 0;
		strlcpy(aliases, dev->ffs_aliases, sizeof(aliases));
		a = aliases;

		while (a) {
			char *alias = strsep(&a, ",");
			if (alias && !strcmp(name, alias)) {
				is_ffs = 1;
				break;
			}
		}

		if (is_ffs) {
			if (ffs_enabled)
				continue;
			err = android_enable_function(dev, "ffs");
			if (err)
				pr_err("android_usb: Cannot enable ffs (%d)",
									err);
			else
				ffs_enabled = 1;
			continue;
		}

		if((++count == 1)&&(strcmp(name, "mass_storage")==0)){
			dev->cdrom_only = true;
		}else{
			dev->cdrom_only = false;
		}
	
		err = android_enable_function(dev, name);
		if (err){
			pr_err("android_usb: Cannot enable '%s' (%d)",
							   name, err);
			printk(KERN_WARNING "android_usb: Cannot enable %s \n", name);
		}
	}

	mutex_unlock(&dev->mutex);
    USBSTACK_DBG("%s, %u", __func__, __LINE__);
	return size;
}

static ssize_t enable_show(struct device *pdev, struct device_attribute *attr,
			   char *buf)
{
	struct android_dev *dev = dev_get_drvdata(pdev);
    if(NULL == dev)
    {
        return -EINVAL;
    }
    USBSTACK_DBG("%s", __func__);
	return sprintf(buf, "%d\n", dev->enabled);
}

static ssize_t enable_store(struct device *pdev, struct device_attribute *attr,
			    const char *buff, size_t size)
{
	struct android_dev *dev = dev_get_drvdata(pdev);
    if(NULL == dev)
    {
        return -EINVAL;
    }

	struct usb_composite_dev *cdev = dev->cdev;
    if(NULL == cdev)
    {
        return -EINVAL;
    }

	//struct usb_configuration *c_desc = cdev->config;
	struct android_usb_function *f;
	int enabled = 0;

	if (!cdev)
		return -ENODEV;

    USBSTACK_DBG("%s, %u", __func__, __LINE__);
	mutex_lock(&dev->mutex);

	sscanf(buff, "%d", &enabled);
	USBSTACK_DBG("enable_store enable:%d, dev->enabled:%d", enabled, (size_t)dev->enabled);
	
	if (enabled && !dev->enabled) {
		USB_DEBUG("USB ENABLE");
		USBSTACK_DBG("USB ENABLE");

		//close usb power
		usb_gadget_set_selfpowered(cdev->gadget);		
		
		/*
		 * Update values in composite driver's copy of
		 * device descriptor.
		 */
		cdev->desc.idVendor = device_desc.idVendor;
		cdev->desc.idProduct = device_desc.idProduct;
		cdev->desc.bcdDevice = device_desc.bcdDevice;
		cdev->desc.bDeviceClass = device_desc.bDeviceClass;
		cdev->desc.bDeviceSubClass = device_desc.bDeviceSubClass;
		cdev->desc.bDeviceProtocol = device_desc.bDeviceProtocol;
		//c_desc->iConfiguration= android_config_driver.iConfiguration;
		list_for_each_entry(f, &dev->enabled_functions, enabled_list) {
			USB_DEBUG("function name:%s", f->name);
			USBSTACK_DBG("function name:%s", f->name);
			if (f->enable)
				f->enable(f);
#if 0
			if(!strcmp(f->name,"ecm"))
			{
				cdev->desc.bDeviceClass = USB_CLASS_COMM;
			}
#endif
		}
		android_enable(dev);
		dev->enabled = true;
#ifdef _USE_VEHICLE_DC		
		usb_check_plugin_msg_sent();
#endif
		usb_record_dbginfo(USB_DEV_ENABLE, 0, 0);
	} else if (!enabled && dev->enabled) {
		USB_DEBUG("USB DISENABLE");
		USBSTACK_DBG("USB DISENABLE");
		android_disable(dev);
		list_for_each_entry(f, &dev->enabled_functions, enabled_list) {
			if (f->disable)
				f->disable(f);
		}
		dev->enabled = false;
		//close usb power
		usb_gadget_clear_selfpowered(cdev->gadget);	
		usb_record_dbginfo(USB_DEV_DISABLE, 0, 0);	
	} else {
		pr_err("android_usb: already %s\n",
				dev->enabled ? "enabled" : "disabled");
		USBSTACK_DBG("android_usb: already %s\n",
				dev->enabled ? "enabled" : "disabled");
	}

	mutex_unlock(&dev->mutex);
	USBSTACK_DBG("%s, %u", __func__, __LINE__);
	return size;
}

void gadget_disable(void)
{
	struct android_dev *dev = _android_dev;
	struct android_usb_function *f;
	struct usb_composite_dev *cdev = dev->cdev;
	
	USB_DEBUG("USB DISENABLE");
	USBSTACK_DBG("USB DISENABLE");
	android_disable(dev);
	list_for_each_entry(f, &dev->enabled_functions, enabled_list) {
		if (f->disable)
			f->disable(f);
	}
	dev->enabled = false;
	//close usb power
	usb_gadget_clear_selfpowered(cdev->gadget); 

}
EXPORT_SYMBOL(gadget_disable);

static ssize_t state_show(struct device *pdev, struct device_attribute *attr,
			   char *buf)
{
	struct android_dev *dev = dev_get_drvdata(pdev);
    if(NULL == dev)
    {
        return -EINVAL;
    }
	struct usb_composite_dev *cdev = dev->cdev;
    if(NULL == cdev)
    {
        return -EINVAL;
    }
    char *state = "DISCONNECTED";
	unsigned long flags;

	if (!cdev)
		goto out;

	spin_lock_irqsave(&cdev->lock, flags);
	if (cdev->config)
		state = "CONFIGURED";
	else if (dev->connected)
		state = "CONNECTED";
	spin_unlock_irqrestore(&cdev->lock, flags);
out:
	return sprintf(buf, "%s\n", state);
}
static ssize_t log_show(struct device *pdev, struct device_attribute *attr,
			   char *buf)
{
	usb_dbg_showLog();
	return sprintf(buf, "%s\n", "OK");		\
}


static ssize_t iSerial_enable_show(struct device *pdev, struct device_attribute *attr,
			   char *buf)
{
	return sprintf(buf, "%d\n", strings_dev[STRING_SERIAL_IDX].id);		\
}

static u8 snID = 0;
static ssize_t iSerial_enable_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t size)
{
	int enabled = 0;
	struct android_dev *pdev = dev_get_drvdata(dev);
    if(NULL == pdev)
    {
        return -EINVAL ;
    }
	struct usb_composite_dev *cdev = pdev->cdev;
    if(NULL == cdev)
    {
        return -EINVAL;
    }

	sscanf(buf, "%d", &enabled);

	if(enabled == 0){
		strings_dev[STRING_SERIAL_IDX].id = 0;
		device_desc.iSerialNumber = 0;
		cdev->desc.iSerialNumber = 0;
	}else{
		strings_dev[STRING_SERIAL_IDX].id = snID;
		device_desc.iSerialNumber = snID;
		cdev->desc.iSerialNumber = snID;
	}
	return size;
}


#define DESCRIPTOR_ATTR(field, format_string)				\
static ssize_t								\
field ## _show(struct device *dev, struct device_attribute *attr,	\
		char *buf)						\
{									\
	return sprintf(buf, format_string, device_desc.field);		\
}									\
static ssize_t								\
field ## _store(struct device *dev, struct device_attribute *attr,	\
		const char *buf, size_t size)				\
{									\
	int value;							\
	if (sscanf(buf, format_string, &value) == 1) {			\
		device_desc.field = value;				\
		return size;						\
	}								\
	return -1;							\
}									\
static DEVICE_ATTR(field, S_IRUGO | S_IWUSR, field ## _show, field ## _store);

#define DESCRIPTOR_STRING_ATTR(field, buffer)				\
static ssize_t								\
field ## _show(struct device *dev, struct device_attribute *attr,	\
		char *buf)						\
{									\
	return sprintf(buf, "%s", buffer);				\
}									\
static ssize_t								\
field ## _store(struct device *dev, struct device_attribute *attr,	\
		const char *buf, size_t size)				\
{									\
	if (size >= sizeof(buffer))					\
		return -EINVAL;						\
	return strlcpy(buffer, buf, sizeof(buffer));			\
}									\
static DEVICE_ATTR(field, S_IRUGO | S_IWUSR, field ## _show, field ## _store);

#define CONFIGURATION_ATTR(field, format_string)				\
static ssize_t								\
field ## _show(struct device *dev, struct device_attribute *attr, \
			char *buf)						\
{												\
	return sprintf(buf, format_string, android_config_driver.field);		\
}									\
static ssize_t								\
field ## _store(struct device *dev, struct device_attribute *attr,	\
		const char *buf, size_t size)				\
{									\
	int value;							\
	if (sscanf(buf, format_string, &value) == 1) {			\
		android_config_driver.field = value;				\
		return size;						\
	}								\
	return -1;							\
}									\
static DEVICE_ATTR(field, S_IRUGO | S_IWUSR, field## _show, field ## _store);

#define CONFIGURATION_STRING_ATTR(field, buffer)				\
static ssize_t								\
field ## _show(struct device *dev, struct device_attribute *attr,	\
		char *buf)						\
{									\
	return sprintf(buf, "%s\n", buffer);				\
}									\
static ssize_t								\
field ## _store(struct device *dev, struct device_attribute *attr,	\
		const char *buf, size_t size)				\
{									\
	if (size >= sizeof(buffer))					\
		return -EINVAL;						\
	return strlcpy(buffer, buf, sizeof(buffer));			\
}									\
static DEVICE_ATTR(field, S_IRUGO | S_IWUSR, field## _show, field ## _store);



DESCRIPTOR_ATTR(idVendor, "%04x\n")
DESCRIPTOR_ATTR(idProduct, "%04x\n")
DESCRIPTOR_ATTR(bcdDevice, "%04x\n")
DESCRIPTOR_ATTR(bDeviceClass, "%d\n")
DESCRIPTOR_ATTR(bDeviceSubClass, "%d\n")
DESCRIPTOR_ATTR(bDeviceProtocol, "%d\n")
DESCRIPTOR_STRING_ATTR(iManufacturer, manufacturer_string)
DESCRIPTOR_STRING_ATTR(iProduct, product_string)
DESCRIPTOR_STRING_ATTR(iSerial, serial_string)

CONFIGURATION_STRING_ATTR(iConfiguration, configuration_string)
static DEVICE_ATTR(functions, S_IRUGO | S_IWUSR, functions_show,
						 functions_store);
static DEVICE_ATTR(enable, S_IRUGO | S_IWUSR, enable_show, enable_store);
static DEVICE_ATTR(state, S_IRUGO, state_show, NULL);

//DESCRIPTOR_ATTR(bMaxPacketSize0, "%d\n")

CONFIGURATION_ATTR(bmAttributes, "%d\n")
	
static DEVICE_ATTR(log, S_IRUGO, log_show, NULL);
static DEVICE_ATTR(iSerial_enable, S_IRUGO | S_IWUSR, iSerial_enable_show, iSerial_enable_store);

/*
 *this function show is suspend of device,1:suspended; 0:running
 */
static ssize_t suspend_state_show(struct device *dev, struct device_attribute *attr,
			  char *buf)
{	struct android_dev *pdev = dev_get_drvdata(dev);
    if(NULL == pdev)
    {
        return -EINVAL;
    }
	struct usb_composite_dev *cdev = pdev->cdev;
    if(NULL == cdev)
    {
        return -EINVAL;
    }
    
	return sprintf(buf, "%d\n", cdev->suspended);
}
static DEVICE_ATTR(suspend_state, S_IRUGO, suspend_state_show, NULL);

/*
 * this function used by app to wakeup usb host and device
 * to use this function, usb device should enable attribute remotewakeup 
 * in android_config_driver
*/
static ssize_t usb_wakeup_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t n)
{
	struct android_dev *pdev = dev_get_drvdata(dev);
	struct usb_composite_dev *cdev = NULL;

	if(NULL!=pdev)
		cdev= pdev->cdev;
	else
		return n;
	
	if (sysfs_streq(buf, "1"))
		usb_gadget_wakeup(cdev->gadget);

	return n;
}
static DEVICE_ATTR(wakeup, S_IWUSR, NULL, usb_wakeup_store);


static ssize_t usb_enum_mode_show(struct device *pdev, struct device_attribute *attr,
			   char *buf)
{
	return sprintf(buf, "%d\n", usb_cur_enum_mode);		
}
static ssize_t usb_enum_mode_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t size)
{
	unsigned long enum_mode = 0;
	enum_mode = simple_strtoul(buf, NULL, 16);

	if(enum_mode < 5){		
		usb_cur_enum_mode = enum_mode;
		usb_set_switch_mode(usb_cur_enum_mode);
	}else
		printk("set mode fail, default mode is %d\n", usb_cur_enum_mode);
	return size;
}
static DEVICE_ATTR(usb_enum_mode, S_IRUGO | S_IWUSR, usb_enum_mode_show, usb_enum_mode_store);


static struct device_attribute *android_usb_attributes[] = {
	&dev_attr_idVendor,
	&dev_attr_idProduct,
	&dev_attr_bcdDevice,
	&dev_attr_bDeviceClass,
	&dev_attr_bDeviceSubClass,
	&dev_attr_bDeviceProtocol,
	&dev_attr_iManufacturer,
	&dev_attr_iProduct,
	&dev_attr_iSerial,
	&dev_attr_functions,
	&dev_attr_enable,
	&dev_attr_state,
	&dev_attr_log,
	&dev_attr_iConfiguration,
	&dev_attr_iSerial_enable,
	&dev_attr_suspend_state,
	&dev_attr_wakeup,
	&dev_attr_usb_enum_mode,
	&dev_attr_bmAttributes,
	NULL
};

/*-------------------------------------------------------------------------*/
/* Composite driver */

static int android_bind_config(struct usb_configuration *c)
{
	struct android_dev *dev = _android_dev;
	int ret = 0;
	USBSTACK_DBG("%s", __func__);
	ret = android_bind_enabled_functions(dev, c);
	if (ret)
		return ret;

	return 0;
}

static void android_unbind_config(struct usb_configuration *c)
{
	struct android_dev *dev = _android_dev;
	USBSTACK_DBG("%s", __func__);
	android_unbind_enabled_functions(dev, c);
}

static int android_bind(struct usb_composite_dev *cdev)
{
	struct android_dev *dev = _android_dev;
	struct usb_gadget	*gadget = cdev->gadget;
	int			gcnum, id, ret;
	USBSTACK_DBG("%s", __func__);
	/*
	 * Start disconnected. Userspace will connect the gadget once
	 * it is done configuring the functions.
	 */
	usb_gadget_disconnect(gadget);

	ret = android_init_functions(dev->functions, cdev);
	if (ret)
		return ret;

	/* Allocate string descriptor numbers ... note that string
	 * contents can be overridden by the composite_dev glue.
	 */
	id = usb_string_id(cdev);
	if (id < 0)
		return id;
	strings_dev[STRING_MANUFACTURER_IDX].id = id;
	device_desc.iManufacturer = id;

	id = usb_string_id(cdev);
	if (id < 0)
		return id;
	strings_dev[STRING_CONFIGURATION_IDX].id = id;
	android_config_driver.iConfiguration= id;
	id = usb_string_id(cdev);
	if (id < 0)
		return id;
	strings_dev[STRING_PRODUCT_IDX].id = id;
	device_desc.iProduct = id;

	/* Default strings - should be updated by userspace */
	strncpy(manufacturer_string, "DEMO,Incorporated", sizeof(manufacturer_string)-1);
	strncpy(product_string, "DEMO Mobile Boardband", sizeof(product_string) - 1);
	strncpy(serial_string, "0123456789ABCDEF", sizeof(serial_string) - 1);
	strncpy(configuration_string, "DEMO Configuration", sizeof(configuration_string) - 1);

	id = usb_string_id(cdev);
	if (id < 0)
		return id;
	strings_dev[STRING_SERIAL_IDX].id = id;
	device_desc.iSerialNumber = id;
	snID = id;

	gcnum = usb_gadget_controller_number(gadget);
	if (gcnum >= 0)
		device_desc.bcdDevice = cpu_to_le16(0x0200 + gcnum);
	else {
		pr_warning("%s: controller '%s' not recognized\n",
			longname, gadget->name);
		device_desc.bcdDevice = __constant_cpu_to_le16(0x9999);
	}

	dev->cdev = cdev;

	return 0;
}

static int android_usb_unbind(struct usb_composite_dev *cdev)
{
	struct android_dev *dev = _android_dev;
	USBSTACK_DBG("%s", __func__);
	cancel_work_sync(&dev->work);
	android_cleanup_functions(dev->functions);
	return 0;
}

static struct usb_composite_driver android_usb_driver = {
	.name		= "android_usb",
	.dev		= &device_desc,
	.strings	= dev_strings,
	.unbind		= android_usb_unbind,
	.max_speed	= USB_SPEED_HIGH,
};

static int
android_setup(struct usb_gadget *gadget, const struct usb_ctrlrequest *c)
{
	struct android_dev		*dev = _android_dev;
    if(NULL == dev)
    {
        return -EINVAL;
    }
	struct usb_composite_dev	*cdev = get_gadget_data(gadget);
    if(NULL == cdev)
    {
        return -EINVAL;
    }
    struct usb_request		*req = cdev->req;
	struct android_usb_function	*f;
	int value = -EOPNOTSUPP;
	unsigned long flags;

#if 1
	USBSTACK_DBG("SETUP 0x%x,%x,v0x%x,w0x%x,l0x%x", 
		c->bRequestType, 
		c->bRequest,
		c->wValue,
		c->wIndex,
		c->wLength);
    USBSTACK_DBG("%s, %u", __func__, __LINE__);
//    usb_dbg_ep0reg();
#endif	
	req->zero = 0;
	req->complete = composite_setup_complete;
	req->length = 0;
	gadget->ep0->driver_data = cdev;

	list_for_each_entry(f, &dev->enabled_functions, enabled_list) {
		if (f->ctrlrequest) {
			value = f->ctrlrequest(f, cdev, c);
			if (value >= 0)
				break;
		}
	}

	spin_lock_irqsave(&cdev->lock, flags);
	if((c->bRequestType & USB_TYPE_MASK) == USB_TYPE_VENDOR){
	  	//USBSTACK_DBG("schedule_work vendor");
		dev->vendor_req.bRequest = c->bRequest;
		dev->vendor_req.wValue = c->wValue; 
		if(dev->vendor_req.bRequest == 0xA2){
			panic("now panic by user\n");
		}
		//if(usb_cur_enum_mode != USB_ENUM_MODE_USER)
			schedule_work(&dev->usbmode);
		//else
		//	printk("user mode can not do mode switch\n");
 	}
	spin_unlock_irqrestore(&cdev->lock, flags);

	if(((c->bRequestType & USB_TYPE_MASK) == USB_TYPE_VENDOR)&&(value==-1))
		return value=-1;
	 
	/* Special case the accessory function.
	 * It needs to handle control requests before it is enabled.
	 */
#if 0
	if (value < 0)
		value = acc_ctrlrequest(cdev, c);
#endif
	if (value < 0)
		value = composite_setup(gadget, c);

	spin_lock_irqsave(&cdev->lock, flags);
	if (!dev->connected) {
		dev->connected = 1;
		schedule_work(&dev->work);
	} else if (c->bRequest == USB_REQ_SET_CONFIGURATION &&
						cdev->config) {
		schedule_work(&dev->work);
	}
	spin_unlock_irqrestore(&cdev->lock, flags);

	return value;
}

static void android_disconnect(struct usb_gadget *gadget)
{
	struct android_dev *dev = _android_dev;
	struct usb_composite_dev *cdev = get_gadget_data(gadget);
	unsigned long flags;
	USBSTACK_DBG("%s", __func__);
	composite_disconnect(gadget);
	/* accessory HID support can be active while the
	   accessory function is not actually enabled,
	   so we need to inform it when we are disconnected.
	 */
#if 0
	acc_disconnect();
#endif
	spin_lock_irqsave(&cdev->lock, flags);
	dev->connected = 0;
	schedule_work(&dev->work);
	spin_unlock_irqrestore(&cdev->lock, flags);
}

static int android_create_device(struct android_dev *dev)
{
	struct device_attribute **attrs = android_usb_attributes;
	struct device_attribute *attr;
	int err;

	dev->dev = device_create(android_class, NULL,
					MKDEV(0, 0), NULL, "android0");
	if (IS_ERR(dev->dev))
		return PTR_ERR(dev->dev);

	dev_set_drvdata(dev->dev, dev);

	while ((attr = *attrs++)) {
		err = device_create_file(dev->dev, attr);
		if (err) {
			device_destroy(android_class, dev->dev->devt);
			return err;
		}
	}
	return 0;
}

#ifdef _USE_VEHICLE_DC
extern const char *usb_speed_string(enum usb_device_speed speed);

void android_set_rpmsg_resp(int type, char*resp)
{
	usb_rpmsg_cmd *t_resp = (usb_rpmsg_cmd *)resp;
	unsigned long flags;
	struct usb_composite_dev *cdev; 
	struct usb_gadget		*gadget;
	struct android_usb_function *func_t;
	struct android_usb_function *f;	
	char *buff = t_resp->param;
	int n = 0;
	int max_len = 252;
	int data_len = 0;	
	
	if(!_android_dev){
		printk("android_set_rpmsg_resp, _android_dev is NULL\n");
		t_resp->cmd = type;
		sprintf(t_resp->param, "%s\n", "DISCONNECTED");	
		return ;
	}
	
	mutex_lock(&_android_dev->mutex);
	cdev = _android_dev->cdev;
	if(NULL == cdev)
	{
		printk("android_set_rpmsg_resp, cdev is NULL\n");
		t_resp->cmd = type;
		sprintf(t_resp->param, "%s\n", "DISCONNECTED");	
		mutex_unlock(&_android_dev->mutex);	
		
	    return ;
	}
	printk("android_set_rpmsg_resp cmd type:%d\n", type);
	switch(type){
		case USB_RPMSG_GET_NET_TYPE:
			t_resp->cmd = USB_RPMSG_GET_NET_TYPE;
			
			if(_android_dev->enabled){
				func_t = list_first_entry(&_android_dev->enabled_functions, struct android_usb_function, enabled_list);
				if(func_t){
					if(strcmp(func_t->name, "rndis") == 0 ||
						strcmp(func_t->name, "ecm") == 0){
						sprintf(t_resp->param, "%s\n", func_t->name);
					}else{
						sprintf(t_resp->param, "%s\n", "none net mode");				
					}
				}else{
						sprintf(t_resp->param, "%s\n", "none net mode");				
				}
			}else{
				sprintf(t_resp->param, "%s\n", "DISCONNECTED");				
			}					
			
			break;
		case USB_RPMSG_GET_USB_SPEED:
			t_resp->cmd = USB_RPMSG_GET_USB_SPEED;
			spin_lock_irqsave(&cdev->lock, flags);
			gadget = cdev->gadget;
			if(NULL == gadget)
			{
				printk("android_set_rpmsg_resp, gadget is NULL\n");
				sprintf(t_resp->param, "%s\n",  "DISCONNECTED");
				spin_unlock_irqrestore(&cdev->lock, flags);				
				break;			    
			}
			
			if(!_android_dev->enabled)	{
				gadget->speed = 0;
			}
			sprintf(t_resp->param, "%s\n", usb_speed_string(gadget->speed));
			spin_unlock_irqrestore(&cdev->lock, flags);		
			break;
		case USB_RPMSG_GET_USB_STATE:			
			t_resp->cmd = USB_RPMSG_GET_USB_STATE;

			spin_lock_irqsave(&cdev->lock, flags);
			gadget = cdev->gadget;
			if(NULL == gadget)
			{
				printk("android_set_rpmsg_resp, gadget is NULL\n");
				sprintf(t_resp->param, "%s\n",  "DISCONNECTED");
				spin_unlock_irqrestore(&cdev->lock, flags);				
				break;			    
			}			
			if (cdev->config)
				sprintf(t_resp->param, "%s\n", "CONFIGURED"); 
			else if (!_android_dev->enabled)
				sprintf(t_resp->param, "%s\n",  "DISCONNECTED");
			else
				sprintf(t_resp->param, "%s\n",  "CONNECTED");
				
			spin_unlock_irqrestore(&cdev->lock, flags);			
			
			
			break;
		case USB_RPMSG_GET_USB_LINK_STATE:
			t_resp->cmd = USB_RPMSG_GET_USB_LINK_STATE;
			sprintf(t_resp->param, "%s\n", _android_dev->enabled ? "enabled" : "DISCONNECTED");
			
			break;
			//usb_set_rpmsg_resp(t_cmd->cmd, usb_resp);
		case USB_RPMSG_GET_USB_ENUM_MODE:
			t_resp->cmd = USB_RPMSG_GET_USB_ENUM_MODE;
			if(!_android_dev->enabled)	{
					sprintf(t_resp->param, "%s\n","DISCONNECTED");
					break;
			}
			switch(usb_cur_enum_mode){
				case USB_ENUM_MODE_DEBUG:
					sprintf(t_resp->param, "%s\n","debug mode");
					
					break;
				case USB_ENUM_MODE_USER:
					sprintf(t_resp->param, "%s\n","user mode");
					
					break;
				case USB_ENUM_MODE_FACTORY:
					sprintf(t_resp->param, "%s\n","factory mode");
					
					break;
				case USB_ENUM_MODE_AMT:
					sprintf(t_resp->param, "%s\n","amt mode");					
					break;
				case USB_ENUM_MODE_EYE_DIAGRAM:
					sprintf(t_resp->param, "%s\n","eye test mode");					
					break;
				default:
					sprintf(t_resp->param, "%s\n","invalid mode");
					break;
						
			}			
			
			break;
		case USB_RPMSG_GET_USB_CURR_CONFIG:
			
			t_resp->cmd = USB_RPMSG_GET_USB_CURR_CONFIG;
			spin_lock_irqsave(&cdev->lock, flags);
			if(!_android_dev->enabled)	{
					sprintf(t_resp->param, "%s\n","DISCONNECTED");
					spin_unlock_irqrestore(&cdev->lock, flags);
					break;
			}			
			list_for_each_entry(f, &_android_dev->enabled_functions, enabled_list)
			{	
				n = snprintf(buff, (max_len - n), "%s,", f->name);
				buff += n;
				data_len += n;
			}
			t_resp->param[data_len] = '\0';
			spin_unlock_irqrestore(&cdev->lock, flags);
			
			break;
		case USB_RPMSG_GET_USB_DL_STATE:
			t_resp->cmd = USB_RPMSG_GET_USB_DL_STATE;
			//spin_lock_irqsave(&cdev->lock, flags);
#ifndef CONFIG_SYSTEM_RECOVERY			
			//1:dl_off;0:dl_oen
			n = zDrvNand_ReadBootflag();			
			sprintf(t_resp->param, "%d\n",n);
#endif			
			//spin_unlock_irqrestore(&cdev->lock, flags);			
			break;
		//case :
		//	break;
		default:
			break;
	}
	
	mutex_unlock(&_android_dev->mutex);	
}
EXPORT_SYMBOL_GPL(android_set_rpmsg_resp);
#endif


static int __init init(void)
{
	struct android_dev *dev;
	int err;
	USBSTACK_DBG("REGISTER USB STACK DRIVER BEGIN");
	android_class = class_create(THIS_MODULE, "android_usb");
	if (IS_ERR(android_class))
		return PTR_ERR(android_class);

	dev = kzalloc(sizeof(*dev), GFP_KERNEL);
	if (!dev)
		return -ENOMEM;

	dev->disable_depth = 1;
	//dev->disable_depth = 0;
	dev->functions = supported_functions;
	INIT_LIST_HEAD(&dev->enabled_functions);
	INIT_WORK(&dev->work, android_work);
       INIT_WORK(&dev->usbmode, usb_mode_work);  
	USBSTACK_DBG("%s, %u", __func__, __LINE__);
	mutex_init(&dev->mutex);

	err = android_create_device(dev);
	if (err) {
		class_destroy(android_class);
		kfree(dev);
		return err;
	}

	_android_dev = dev;

	/* Override composite driver functions */
	composite_driver.setup = android_setup;
	composite_driver.disconnect = android_disconnect;

	err = usb_composite_probe(&android_usb_driver, android_bind);
	
	//close usb power at last
	usb_gadget_clear_selfpowered(dev->cdev->gadget);
	USBSTACK_DBG("REGISTER USB STACK DRIVER END");
	return err;
}
module_init(init);

static void __exit cleanup(void)
{
	usb_composite_unregister(&android_usb_driver);
	class_destroy(android_class);
	kfree(_android_dev);
	_android_dev = NULL;
}
module_exit(cleanup);
