/***********************************************************************
 * Copyright (C) 2015, ZTE Corporation.
 *
 * File Name:   leadt15ds26fb.h
 * File Mark:
 * Description: Adapted to LCM MODEL - LEAD T15DS26.
 * Others:      This LCM MODEL uses SPI-4. And this 4-lines-spi uses CS, CLK, D/CX and SDI. Do not using SDO, but instead of D/CX.
 * Version:     V1.0
 * Author:      YinWenguan
 * Date:        2015-6-19
 *
 * History 1:
 *     Date:	2016-4-21
 *     Version:
 *     Author:	MaXuelong
 *     Modification:	transplant it from 7510(uc) to 7520v2(linux)
 *
***********************************************************************/
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/fb.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/ctype.h>
#include <linux/dma-mapping.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/gpio.h>
#include <linux/delay.h>
#include <linux/uaccess.h>
#include <linux/vmalloc.h>
#include <linux/spi/spi.h>
#include <linux/wakelock.h>
#include <mach/gpio.h>
#include <mach/clk.h>
#include <mach/gpio_def.h>
#include <mach/gpio_cfg.h>
#include <mach/peri_cfg.h>
#include <linux/video/zx29_lcd.h>
#include <linux/mfd/zx234290.h>
#include <linux/irq.h>
#include <mach/pcu.h>
#include <linux/jiffies.h>
#include <linux/atomic.h>
#include <linux/soc/zte/pm/drv_idle.h>

#include <linux/slab.h>
#include <linux/debugfs.h>
#include <asm/uaccess.h>
#include <linux/reboot.h>
/*************************************************************************
 * Macro
 *************************************************************************/
#define  LCD_PRINFO(fmt,arg...)		no_printk(KERN_INFO "##%s, line:%d" fmt,__FUNCTION__,__LINE__,##arg)
#define  LCD_PRERR(fmt,arg...)		printk(KERN_ERR "##%s, line:%d" fmt,__FUNCTION__,__LINE__,##arg)


/* LCM manufacturer ID */
#define LCM_LEAD_ID			0x7C
#define LCM_ST7789V_ID		0x85
#define LCM_ST7789V_VERSION	0x8552   //240*240
#define LCM_GC930X_ID		0x00
#define LCM_GC930X_VERSION	0x9306  //240*320
#define LCM_ILI9342C_ID		0xE3

#define LCD_WIDTH			128
#define LCD_HEIGHT			160
#define LCD_BITS_PER_PIXEL	16
#define LCD_GRAM_SIZE		(LCD_WIDTH * LCD_HEIGHT * 2)

/* rotate display */
//#define LCD_ROTATE_180 /* need rotating 180C */
#define LCD_XS			0x00
#define LCD_XE			0x81
#ifdef LCD_ROTATE_180
#define LCD_MXMYRGB	0x1C
#define LCD_YS			0x00
#define LCD_YE			0x80
#else
#define LCD_MXMYRGB	0xC8
#define LCD_YS			0x00
#define LCD_YE			0x82
#endif/*LCD_ROTATE_180*/
#ifdef CONFIG_ARCH_ZX297520V3_FWP
#define LCD_PAGE_HEIGHT	8
#define LCD_PAGE_NUM		8

#define START_PAGE			0xB0
#define START_HIGH_BIT		0x10
#define START_LOW_BIT		0x00
#define START_LOW_PY		1
#endif
/* LOAD-BY-TRACE flag storing address.
    1. In normal start-up procedure, BOOTLOADER will clear all IRAM space. So the value of this address will be ZERO.
    2. In LOAD-BY-TRACE procedure, *.cmm script file have already set this address's content to 1. */
#define LCD_LOADBYTRACEFLAG_ADDR    (0x8201c804)

#define RS_L			gpio_set_value(PIN_SPI_DCX, GPIO_LOW)
#define RS_H			gpio_set_value(PIN_SPI_DCX, GPIO_HIGH)

/* read/write register's value */
#define LCD_REG_READ(REG)               (*((volatile unsigned int*)(REG)))
#define LCD_REG_WRITE(REG,value)        (*((volatile unsigned int*)(REG)) = value)

/* ioctrl command */
#define LEADT15DS26_SET_SLEEPMODE       _IOW('L', 1, unsigned int) /* enter or exit sleep-mode */
#define LEADT15DS26_SET_BACKLIGHT       _IOW('L', 2, unsigned int) /* turn on or off backlight */
#define LEADT15DS26_SET_BRIGHTNESS      _IOW('L', 3, unsigned int) /* set brightness value */
#define LEADT15DS26_FILLRECT      		_IOW('L', 4, unsigned int) /* fill rect region */

enum _lcm_operation_flag{
	LCD_LCM_DATA	 = 0x0,
	LCD_LCM_CMD	 = 0x1,
	LCD_LCM_SLEEP	 = 0x2,	
};

struct lcd_lcm_arg{
   enum  _lcm_operation_flag  flag; /* 0 data 1cmd 3sleep */
    int	arg; /*write data/cmd id/sleep time*/
};

#ifdef CONFIG_FB_LCD_TE_ON
static unsigned int g_sLcd_GpioSyn = 0;
static struct semaphore  gLcd_SyncSema;
volatile bool g_LcdSema_Flag = false;
volatile atomic_t g_Lcd_Fmark_count = {0};
volatile unsigned int g_Lcd_tick_count = 0;
#endif

struct lcd_lcm_arg lcd_lcm_ili9342c[]=
{
	{1,0xC8}, {0,0x03}, {0,0xFF}, {0,0x93}, {0,0x42},
	{1,0xE0}, {0,0xD0}, {0,0x00}, {0,0x03}, {0,0x09}, {0,0x13},
			  {0,0x1C}, {0,0x3A}, {0,0x55}, {0,0x48}, {0,0x18}, {0,0x12}, 
			  {0,0x0E}, {0,0x19}, {0,0x1E},  
	{1,0xE1}, {0,0xD0}, {0,0x00}, {0,0x03}, {0,0x09}, {0,0x05}, 
			  {0,0x25}, {0,0x3A}, {0,0x55}, {0,0x50}, {0,0x3D}, {0,0x1C},
			  {0,0x1D}, {0,0x1D}, {0,0x1E},  

	{1,0xB1}, {0,0x01}, {0,0x1B}, 
	{1,0xB4}, {0,0x01}, {0,0x02}, 
	{1,0xB6}, {0,0x02}, {0,0x0A}, {0,0xE0},
	
	{1,0x36}, {0,0x01}, {0,0x08}, 
	{1,0x3A}, {0,0x01}, {0,0x55}, 
	
	{1,0xC0}, {0,0x02}, {0,0x0F}, {0,0x0F},
	{1,0xC1}, {0,0x01}, {0,0x00}, 
	{1,0xC5}, {0,0x01}, {0,0xCC},
	{1,0x20}, {0,0x00}, 
	{1,0x11}, {0,0x00},
	{2, 120},	  
	{1,0x29}, {0,0x00}
};
struct lcd_lcm_arg lcd_lcm_st7789v[]=
{
	{1, 0x11}, 
	{2, 120},
	
	{1,0x36}, {0,0x00}, //0x0 -> 0x8
	{1,0x3A}, {0,0x55},
	
	{1,0xB2}, {0,0x0C}, {0,0x0C}, {0,0x00}, {0,0x33},{0,0x33},
	{1,0xB7}, {0,0x22}, 
	
	{1,0xBB}, {0,0x3C},  
	{1,0xC2}, {0,0x01},
	{1,0xC3}, {0,0x19}, 
	{1,0xC4}, {0,0x20},  
	{1,0xC6}, {0,0x0F},  
	{1,0xD0}, {0,0xA4}, {0,0xA1},
	{1,0xD6}, {0,0xA1},
	{1,0xE0}, {0,0x70}, {0,0x04}, {0,0x08}, {0,0x09}, {0,0x09},
			  {0,0x05}, {0,0x2A}, {0,0x33}, {0,0x41}, {0,0x07}, {0,0x13}, 
			  {0,0x13}, {0,0x29}, {0,0x2F},  
	{1,0xE1}, {0,0x70}, {0,0x03}, {0,0x09}, {0,0x0A}, {0,0x09}, 
			  {0,0x06}, {0,0x2B}, {0,0x34}, {0,0x41}, {0,0x07}, {0,0x12},
			  {0,0x14}, {0,0x28}, {0,0x2E},  
	{1,0x21}, 
	{1,0x29}, 
	{1,0x2C}
};

struct lcd_lcm_arg lcd_lcm_success[]=
{
	{1,0x11}, 
	{2, 150},
	{1,0x26}, {0,0x4},
	{1,0xB1}, {0,0x0B}, {0,0x14}, 
	{1,0xC0}, {0,0x10}, {0,0x00}, 
	{1,0xC1}, {0,0x05}, 
	{1,0xC5}, {0,0x46}, {0,0x40}, 
	{1,0xC7}, {0,0xBD}, 
	{1,0xEC}, {0,0x0C},
	{1,0x3a}, {0,0x05}, 
	{1,0x2A}, {0,0x00}, {0,0x00}, {0,0x00}, {0,0x7F}, 
	{1,0x2B}, {0,0x00}, {0,0x00}, {0,0x00},{0,0x9F}, 
	{1,0x36}, {0,LCD_MXMYRGB}, 
	{1,0xB7}, {0,0x00}, 
	{1,0xf2}, {0,0x1},
	{1,0xE0}, {0,0x3F}, {0,0x29}, {0,0x26}, {0,0x26}, {0,0x26},
			 {0,0x0C}, {0,0x51}, {0,0xB8}, {0,0x39}, {0,0x17}, {0,0x00}, 
			 {0,0x00}, {0,0x00}, {0,0x00}, {0,0x00}, 
	{1,0xE1}, {0,0x00}, {0,0x16}, {0,0x19}, {0,0x19}, {0,0x19}, 
			 {0,0x13}, {0,0x2E}, {0,0x47}, {0,0x46}, {0,0x08}, {0,0x3F},
			 {0,0x3F}, {0,0x3F}, {0,0x3F}, {0,0x3F}, 
	{1,0x29}, 
	{1,0x2C}, 
};

struct lcd_lcm_arg lcd_lcm_lead[]=
{
	{1,0x11}, 
	{2, 120},
	{1,0xB1}, {0,0x00}, {0,0x08}, {0,0x05},
	{1,0xB2}, {0,0x00}, {0,0x08}, {0,0x05},
	{1,0xB3}, {0,0x00}, {0,0x08}, {0,0x05}, {0,0x00},{0,0x08}, {0,0x05},
	{1,0xB4}, {0,0x00}, 
	{1,0xC0}, {0,0xA2}, {0,0x02}, {0,0x84},
	{1,0xC1}, {0,0xC5}, 
	{1,0xC2}, {0,0x0A}, {0,0x00},
	{1,0xC3}, {0,0x8A}, {0,0x2A},
	{1,0xC4}, {0,0x8A}, {0,0xEE}, 
	{1,0xC5}, {0,0x0E}, 
	{1,0x36}, {0,LCD_MXMYRGB}, 
	{1,0xB7}, {0,0x00}, 
	{1,0xf2}, {0,0x1},
	{1,0xE0}, {0,0x12}, {0,0x18}, {0,0x10}, {0,0x18}, {0,0x33},
			 {0,0x2c}, {0,0x25}, {0,0x28}, {0,0x28}, {0,0x27}, {0,0x00}, 
			 {0,0x00}, {0,0x00}, {0,0x00}, {0,0x00},
	{1,0xE1}, {0,0x12}, {0,0x18}, {0,0x10}, {0,0x18}, {0,0x2d}, 
			 {0,0x28}, {0,0x23}, {0,0x28}, {0,0x28}, {0,0x26}, {0,0x2f},
			 {0,0x3b}, {0,0x00}, {0,0x03}, {0,0x03}, {0,0x10},
	{1,0x2A}, {0,0x00}, {0,LCD_XS}, {0,0x00}, {0,LCD_XE}, 
	{1,0x2B}, {0,0x00}, {0,LCD_YS}, {0,0x00},{0,LCD_YE}, 		 
	{1,0x3A}, {0,0x05}, 
	{1,0x29}, 
	{1,0x2C}, 
	{2,200}
};
/**************************************************************************
 * Types
 **************************************************************************/
/* D/CX */
typedef enum
{
    LCD_DCX_CMD = 0,    /* command - low level0 */
    LCD_DCX_DATA,       /* data(parameter also as data) - high level1 */

    MAX_LCD_DCX_TYPE
} T_ZDrvLcd_DcxType; // transmission byte type

/* lcd mode */
typedef enum
{
    LCD_NORMAL,        /* normal mode */
    LCD_SLEEP,          /* sleep mode */

    MAX_LCD_MODE
} T_ZDrvLcd_Mode;

/* lead fb parameter */
struct leadt15ds26fb_par {
	struct spi_device *spi;
	struct fb_info *info;
	struct mutex io_lock;
	struct wake_lock wlock_write;	
	struct wake_lock wlock_ioctl;
	struct wake_lock wlock_backlight;
	unsigned int  lcd_id;
	unsigned short	driver_version;   
	bool		lcd_backlight;
	unsigned int	lcd_brightness;
	T_ZDrvLcd_Mode lcd_sleep_mode;
	int irq_num;
};

/**************************************************************************
 * Global  Variable
 **************************************************************************/
/* current LCM ID. Actually it is be read from controller. */
struct leadt15ds26fb_par g_leadt15ds26fb_par;

/* to save current status */
volatile static T_ZDrvLcd_DcxType g_sLcd_DcxType      = LCD_DCX_CMD; /* current D/CX pin status */
//volatile static T_ZDrvLcd_Mode    g_sLcd_Mode         = LCD_NORMAL;
//volatile static bool              g_sLcd_Backlight    = 0;			/* 0 - off; 1 - on */
//volatile static unsigned int      g_sLcd_Brightness   = 255;		/* [0,255]: 0 - darkest; 255 - brightest */

/* determine SPI type: actrul or simulated */
static bool use_actual_spi = true;
struct notifier_block pmu_reinit_notifier;
static unsigned int last_level = 0;

/**************************************************************************
 * Function Implementation
 **************************************************************************/

/*cs low: config spi pad gpio function, pull down cs pad. cs high: pull up cs, and restore spi function*/
extern void spi_gpio_3wire_cs(unsigned char level);
extern void spi_gpio_3wire_write8(unsigned char reg);
extern unsigned char spi_gpio_3wire_read8(void);


void write_st(__u8 rs, __u8 para)
{
	/* ݷѡ߻RS */

	if (rs)
	    {RS_H;}
	else
	    {RS_L;}
	
	spi_gpio_3wire_cs(GPIO_LOW);
	spi_gpio_3wire_write8(para);	
	spi_gpio_3wire_cs(GPIO_HIGH);
}

__u8 read_st(__u8 reg)
{
	__u8 i;
	volatile __u8 tmp;

	/* ȷͶ */
	spi_gpio_3wire_cs(GPIO_LOW);	
	RS_L;	
	spi_gpio_3wire_write8(reg);	
	/* лGPIOλ */
	RS_H;	
	tmp = spi_gpio_3wire_read8();
	spi_gpio_3wire_cs(GPIO_HIGH);

    return tmp;
}

/*******************************************************************************
 * Function:    leadt15ds26fb_delayms
 * Description: used to delay.
 * Parameters:
 *      (IN)
 *              ms: millisecond
 *      (OUT):
 *              None.
 * Returns:
 *
 *
 * Others:
 ********************************************************************************/
static inline void leadt15ds26fb_delayms(unsigned int ms)
{
    mdelay(ms);
}

/**************************************************************************
* Function: leadt15ds26fb_spi_config_dcx
* Description: used to configure D/CX pin. thus, this interface is just for SPI-4.
* Parameters:
*       (IN)
*               type    command or parameter/data
*       (OUT)
*               None.
* Returns:
*       0: successed.
*       < 0: others error code. For more details, please refer to the source code.
* Others:
*   - SPI-3's data packet contains a control bit D/CX and a transmission byte.
*   - SPI-4's data packet contains just transmission byte and
*      control bit D/CX is transferred by the D/CX pin.
**************************************************************************/
static int leadt15ds26fb_spi_config_dcx(struct spi_device *spi, T_ZDrvLcd_DcxType type)
{
	int ret = 0;
	struct zx29_lcd_platform_data *pdata = (struct zx29_lcd_platform_data *)spi->dev.platform_data;

	//LCD_PRINFO("lead sLcd_GpioDCX=%d\n", pdata->spi_dcx_gpio);
	gpio_set_value(pdata->spi_dcx_gpio, (type == LCD_DCX_DATA) ? GPIO_HIGH : GPIO_LOW);
	g_sLcd_DcxType = type;

    return ret;
}

/**************************************************************************
* Function: leadt15ds26fb_spi_write_cmd
* Description: used to send command-code to the LCD device IC.
* Parameters:
*       (IN)
*               cmd     command
*       (OUT)
*               None.
* Returns:
*       It returns zero on success, else a negative error code.
* Others:
*   - SPI-3's data packet contains a control bit D/CX and a transmission byte.
*   - SPI-4's data packet contains just transmission byte and
*      control bit D/CX is transferred by the D/CX pin.
**************************************************************************/
static int leadt15ds26fb_spi_write_cmd(struct spi_device *spi, unsigned char cmd)
{
    int ret = 0;
    unsigned char buf[1] = {cmd};

    if (use_actual_spi){
        /* adjust D/CX level: high - command */
        ret += leadt15ds26fb_spi_config_dcx(spi, LCD_DCX_CMD);

        /* then send 8bits cmd to SPI device */
        ret += spi_write(spi, buf, 1);
    }
    else{
        write_st(LCD_DCX_CMD, cmd);
    }

    return ret;
}

/**************************************************************************
* Function: leadt15ds26fb_spi_write_data/leadt15ds26fb_spi_write_datablock
* Description: used to send parameter/data to the LCD device IC.
* Parameters:
*       (IN)
*               data     parameter or RAM data.
*       (OUT)
*               None.
* Returns:
*       0: successed.
*       < 0: others error code. For more details, please refer to the source code.
* Others:
*   - SPI-3's data packet contains a control bit D/CX and a transmission byte.
*   - SPI-4's data packet contains just transmission byte and
*      control bit D/CX is transferred by the D/CX pin.
**************************************************************************/
static int leadt15ds26fb_spi_write_data(struct spi_device *spi, unsigned char data)
{
	int ret = 0;
	unsigned char buf[1] = {data};

	if (use_actual_spi){
        /* reduce D/CX change frequency */
		if (LCD_DCX_DATA != g_sLcd_DcxType){
		ret += leadt15ds26fb_spi_config_dcx(spi, LCD_DCX_DATA);
		}
	        /* then send 8bits data to SPI device */
	        ret += spi_write(spi, buf, 1);
	}
    else{
		write_st(LCD_DCX_DATA, data);
    }

    return ret;
}

static int leadt15ds26fb_spi_write_datablock(struct spi_device *spi, unsigned char* buf, unsigned int size)
{
	int ret = 0;
	unsigned int i = 0;

	if (use_actual_spi){
	/* reduce D/CX change frequency */
		if (LCD_DCX_DATA != g_sLcd_DcxType){
			ret += leadt15ds26fb_spi_config_dcx(spi, LCD_DCX_DATA);
		}

		/* then write data block to SPI device */
		ret += spi_write(spi, buf, size);
	}
	else{
		for (i = 0; i < size; i++)
		write_st(LCD_DCX_DATA, buf[i]);
	}

	return ret;
}

/**************************************************************************
* Function: leadt15ds26fb_reset
* Description: used to reset the LCD device opened previously.
* Parameters:
*       (IN)
*               None.
*       (OUT)
*               None.
* Returns:
*       None.
* Others:
*       None.
**************************************************************************/
static int leadt15ds26fb_reset(struct fb_info *info)
{
	struct leadt15ds26fb_par *par = (struct leadt15ds26fb_par *)(info->par);
	struct zx29_lcd_platform_data *pdata = (struct zx29_lcd_platform_data *)par->spi->dev.platform_data;
	int ret = 0;
	
	gpio_set_value(pdata->lcd_rst_gpio, GPIO_HIGH);
	leadt15ds26fb_delayms(10);

	gpio_set_value(pdata->lcd_rst_gpio, GPIO_LOW);
	leadt15ds26fb_delayms(10);

	gpio_set_value(pdata->lcd_rst_gpio, GPIO_HIGH);
	leadt15ds26fb_delayms(120);

	return ret;
}

/**************************************************************************
* Function: leadt15ds26fb_init_gpio
* Description: used to initialize GPIO usage.
* Parameters:
*       (IN)
*
*       (OUT)
*
* Returns:
*       0: successed.
*       < 0: others error code. For more details, please refer to the source code.
* Others:
**************************************************************************/
static int leadt15ds26fb_init_gpio(struct fb_info *info)
{
	int ret = 0;
	struct leadt15ds26fb_par *par = (struct leadt15ds26fb_par *)(info->par);
	struct zx29_lcd_platform_data *pdata = (struct zx29_lcd_platform_data *)par->spi->dev.platform_data;
#ifndef CONFIG_LCD_BLG_SINK
	/* backlight */
	ret = gpio_request(pdata->lcd_blg_gpio,"lcd_blgen");
	if(ret < 0)
		LCD_PRINFO("request lcd_blgen failed\n");
	zx29_gpio_function_sel(pdata->lcd_blg_gpio, pdata->lcd_blg_gpio_fun_sel);
	zx29_gpio_set_direction(pdata->lcd_blg_gpio, GPIO_OUT);
#endif
	/* reset */
	ret = gpio_request(pdata->lcd_rst_gpio,"lcd_reset");
	if(ret < 0)
		LCD_PRINFO("request lcd_reset failed\n");
	zx29_gpio_function_sel(pdata->lcd_rst_gpio, pdata->lcd_rst_gpio_fun_sel);
	zx29_gpio_set_direction(pdata->lcd_rst_gpio, GPIO_OUT);

	//ret = gpio_request(pdata->spi_dcx_gpio,"spi_dcx");
	//if(ret < 0)
		//LCD_PRINFO("request lcd_reset failed\n");
	zx29_gpio_function_sel(pdata->spi_dcx_gpio, pdata->spi_dcx_gpio_fun_sel);
	zx29_gpio_set_direction(pdata->spi_dcx_gpio, GPIO_OUT);

#ifdef  CONFIG_FB_LCD_TE_ON
	ret = gpio_request(PIN_LCD_TE, "lcd_syn");
	if(ret < 0)
		LCD_PRINFO("request lcd_syn failed\n");
	par->irq_num = gpio_to_irq(PIN_LCD_TE);
	zx29_gpio_config(PIN_LCD_TE, GPIO_TE_FUNC_SEL);
	zx29_gpio_set_inttype(PIN_LCD_TE, IRQ_TYPE_EDGE_RISING);
	zx29_gpio_pd_pu_set(PIN_LCD_TE, IO_CFG_PULL_DOWN);
	pcu_clr_irq_pending(par->irq_num);
#endif

    return ret;
}

#if 0
/**************************************************************************
* Function: leadt15ds26fb_init_backlight
* Description: used to initialize backlight usage.
* Parameters:
*       (IN)
*
*       (OUT)
*
* Returns:
*       0: successed.
*       < 0: others error code. For more details, please refer to the source code.
* Others:7510 control blg use blg controller.but 7520 not
**************************************************************************/
static int leadt15ds26fb_init_backlight(struct fb_info *info)
{
	struct leadt15ds26fb_par *par = (struct leadt15ds26fb_par *)(info->par);
	struct zx29_lcd_platform_data *pdata = (struct zx29_lcd_platform_data *)par->spi->dev.platform_data;
//#ifdef CONFIG_RECOVERY_SYSTEM
//#else
        gpio_set_value(pdata->lcd_blg_gpio,GPIO_HIGH);
//#endif
    return 0;
}
#endif

#ifdef  CONFIG_FB_LCD_TE_ON

static irqreturn_t LcdSync_irq_handler(int irq, void *dev_id)
{
    //printk("LcdSync_irq_handler irq=%d.\n", irq);
	disable_irq_nosync(irq);
    pcu_clr_irq_pending(irq);

    return IRQ_WAKE_THREAD;
}

/**************************************************************************
* Function: LcdSync_Isr
* Description: put lcd sync sema.
* Parameters:
*      (IN)
*               None.
*      (OUT)
*               None.
* Returns:
*       0: successed.
*       < 0: others error code. For more details, please refer to the source code.
* Others:
*       None.
**************************************************************************/
static irqreturn_t LcdSync_Isr(int irq, void *data)
{

	if((g_Lcd_Fmark_count.counter++) > 0xff)
	{
		g_Lcd_Fmark_count.counter = 10;
	}
    //printk("LcdSync_Isr irq=%d.\n", irq);
	if(g_LcdSema_Flag == true)
	{
		up(&gLcd_SyncSema);
		g_Lcd_tick_count = jiffies;//zOss_GetTickCount();
		g_LcdSema_Flag = false;		
	}
	
	enable_irq(irq);

	return IRQ_HANDLED;
}
#endif

/**************************************************************************
* Function: leadt15ds26fb_init_lcm
* Description: used to initialize lead(ST7735S)
* Parameters:
*      (IN)
*               None.
*      (OUT)
*               None.
* Returns:
*       0: successed.
*       < 0: others error code. For more details, please refer to the source code.
* Others:
*       None.
**************************************************************************/
static int leadt15ds26fb_init_lcm(struct fb_info *info)
{
	int ret = 0;
	int i = 0;
	int arg_count = 0;
	unsigned int uiRow, uiCol;
	struct leadt15ds26fb_par *par = (struct leadt15ds26fb_par *)(info->par);
	
	/* hardware reset */
	ret |= leadt15ds26fb_reset(info);

	/* read lcd module's manufacturer ID */
	//par->lcd_id= read_st(0xda);
	LCD_PRINFO("[LCD]leadt15ds26fb_init_lcm lcd_id = 0x%x.\n",par->lcd_id);	
	switch(par->lcd_id){
		
		case LCM_LEAD_ID:/* LCM: lead T15DDS26 */
			arg_count = sizeof(lcd_lcm_lead)/sizeof(struct lcd_lcm_arg);	
			for(i = 0; i< arg_count; i++){
				if(lcd_lcm_lead[i].flag == LCD_LCM_CMD)
					ret |= leadt15ds26fb_spi_write_cmd(par->spi, lcd_lcm_lead[i].arg);
				else if(lcd_lcm_lead[i].flag == LCD_LCM_DATA)
					ret = leadt15ds26fb_spi_write_data(par->spi, lcd_lcm_lead[i].arg);
				else if(lcd_lcm_lead[i].flag == LCD_LCM_SLEEP)
					leadt15ds26fb_delayms(lcd_lcm_lead[i].arg);
				else
					return -1;
			}
			break;
			
		case LCM_ST7789V_ID:/* LCM: st7789 */
			arg_count = sizeof(lcd_lcm_st7789v)/sizeof(struct lcd_lcm_arg);	
			for(i = 0; i< arg_count; i++){
				if(lcd_lcm_st7789v[i].flag == LCD_LCM_CMD)
					ret |= leadt15ds26fb_spi_write_cmd(par->spi, lcd_lcm_st7789v[i].arg);
				else if(lcd_lcm_st7789v[i].flag == LCD_LCM_DATA)
					ret = leadt15ds26fb_spi_write_data(par->spi, lcd_lcm_st7789v[i].arg);
				else if(lcd_lcm_st7789v[i].flag == LCD_LCM_SLEEP)
					leadt15ds26fb_delayms(lcd_lcm_st7789v[i].arg);
				else
					return -1;
			}
			break;
			
		case LCM_ILI9342C_ID:/* LCM: ILI9342C */
			arg_count = sizeof(lcd_lcm_ili9342c)/sizeof(struct lcd_lcm_arg); 
			for(i = 0; i< arg_count; i++){
				if(lcd_lcm_ili9342c[i].flag == LCD_LCM_CMD)
					ret |= leadt15ds26fb_spi_write_cmd(par->spi, lcd_lcm_ili9342c[i].arg);
				else if(lcd_lcm_ili9342c[i].flag == LCD_LCM_DATA)
					ret = leadt15ds26fb_spi_write_data(par->spi, lcd_lcm_ili9342c[i].arg);
				else if(lcd_lcm_ili9342c[i].flag == LCD_LCM_SLEEP)
					leadt15ds26fb_delayms(lcd_lcm_ili9342c[i].arg);
				else
					return -1;
			}
			break;
		default:/* LCM: success S93521A */
			arg_count = sizeof(lcd_lcm_success)/sizeof(struct lcd_lcm_arg);	
			for(i = 0; i< arg_count; i++){
				if(lcd_lcm_success[i].flag == LCD_LCM_CMD)
					ret |= leadt15ds26fb_spi_write_cmd(par->spi, lcd_lcm_success[i].arg);
				else if(lcd_lcm_success[i].flag == LCD_LCM_DATA)
					ret |= leadt15ds26fb_spi_write_data(par->spi, lcd_lcm_success[i].arg);
				else if(lcd_lcm_success[i].flag == LCD_LCM_SLEEP)
					leadt15ds26fb_delayms(lcd_lcm_success[i].arg);
				else
					return -1;
			}
			break;
	}
	
    if(LCM_ST7789V_ID == par->lcd_id)
	{
		ret |= leadt15ds26fb_spi_write_cmd(par->spi, 0x36);
		ret |= leadt15ds26fb_spi_write_data(par->spi, info->var.colorspace);
	}	
	
	if(ret !=0)
		LCD_PRINFO("[LCD][LEAD] lcd lcm init module,arg_count = %d.\n",arg_count);	
		
	/* to black */
	for (uiRow = 0; uiRow < info->var.xres; uiRow++)
	{
		for (uiCol = 0; uiCol < info->var.yres; uiCol++)
		{
			ret |= leadt15ds26fb_spi_write_data(par->spi, 0x00);
			ret |= leadt15ds26fb_spi_write_data(par->spi, 0x00);
		}
	}

	LCD_PRINFO("[LCD][LEAD]init lcd module.\n");

	return ret;
}

/**************************************************************************
* Function: Lcd_EnterSleepMode
* Description: used to let the LCD device into sleep mode.
* Parameters:
*       (IN)
*               bIn     enter or exit sleep mode
*       (OUT)
*               None.
* Returns:
*       0: successed.
*       < 0: others error code. For more details, please refer to the source code.
* Others:
*       None.
**************************************************************************/
static int leadt15ds26fb_set_sleep(struct fb_info *info, bool enable )
{
	int ret = 0;
	struct leadt15ds26fb_par *par = (struct leadt15ds26fb_par *)(info->par);

	/* exit sleep-mode */
	if ((!enable) && (LCD_SLEEP== par->lcd_sleep_mode)){
		ret = leadt15ds26fb_spi_write_cmd(par->spi, 0x11);
		leadt15ds26fb_delayms(120); // leave 120ms
		par->lcd_sleep_mode = LCD_NORMAL;
		LCD_PRINFO("[LCD][LEAD]sleep out.\n");
	}
	/* enter sleep-mode */
	else if(enable && (LCD_NORMAL == par->lcd_sleep_mode)){
		ret = leadt15ds26fb_spi_write_cmd(par->spi, 0x10);
		leadt15ds26fb_delayms(120); // leave 120ms
		par->lcd_sleep_mode = LCD_SLEEP;
		LCD_PRINFO("[LCD][LEAD]sleep in.\n");
	}

	return ret;
}

/**************************************************************************
* Function: leadt15ds26fb_set_backlight
* Description: used to turn on/off the LCD device's backlight.
* Parameters:
*       (IN)
*               None.
*       (OUT)
*               None.
* Returns:
*       0: successed.
*       < 0: others error code. For more details, please refer to the source code.
* Others:
*       None.
**************************************************************************/
static int leadt15ds26fb_set_backlight(struct fb_info *info, bool enable)
{
	struct leadt15ds26fb_par *par = (struct leadt15ds26fb_par *)(info->par);
	struct zx29_lcd_platform_data *pdata = (struct zx29_lcd_platform_data *)par->spi->dev.platform_data;
	int ret = 0;
	if (enable){
		/* turn on backlight */
		zx_cpuidle_set_busy(IDLE_FLAG_LCD);
		
#ifdef CONFIG_LCD_BLG_SINK
		ret = zx234297_set_sink(ZX234297_SINK1,1,SINK_CURRENT_5MA);
		if(ret < 0)
			LCD_PRERR("[LCD][LEAD] backlight on error!\n");			
#else
		gpio_set_value(pdata->lcd_blg_gpio,GPIO_HIGH);
#endif
#ifdef CONFIG_PM_WAKELOCKS
		/* when backlight is on, prevent system into sleep */
		wake_lock(&(par->wlock_backlight));
#endif
		if( LCD_SLEEP == par->lcd_sleep_mode){
			LCD_PRINFO("[LCD][LEAD] backlight on but  the lcd in sleep mode!!!!\n");
		}
	}
	else{
		/* turn off backlight */
		zx_cpuidle_set_free(IDLE_FLAG_LCD);	
#ifdef CONFIG_LCD_BLG_SINK
		ret = zx234297_set_sink(ZX234297_SINK1,0,SINK_CURRENT_5MA);
		if(ret < 0)
			LCD_PRERR("[LCD][LEAD] backlight off error!\n");			
#else
		gpio_set_value(pdata->lcd_blg_gpio,GPIO_LOW);
#endif
		last_level = 0;
#ifdef CONFIG_PM_WAKELOCKS
		/* when backlight is off, allow system into sleep */
		wake_unlock(&(par->wlock_backlight));
#endif
	}

	par->lcd_backlight= enable;
	LCD_PRINFO("[LCD][LEAD]backlight %s.\n", enable ? "on" : "off");
	return 0;
}


//add for reset.c
void leadt15ds26fb_setbacklight_before_reset(bool enable)
{
	struct leadt15ds26fb_par *par = (struct leadt15ds26fb_par *)(g_leadt15ds26fb_par.info->par);
	struct zx29_lcd_platform_data *pdata = (struct zx29_lcd_platform_data *)par->spi->dev.platform_data;
	int ret = 0;
#ifdef CONFIG_LCD_BLG_SINK
	ret = zx234297_set_sink(ZX234297_SINK1,enable,SINK_CURRENT_5MA);
	if(ret < 0)		
		LCD_PRERR("error.\n");
#else	
	//leadt15ds26fb_set_backlight(g_leadt15ds26fb_par.info, enable);
	 if (enable)    
		gpio_set_value(pdata->lcd_blg_gpio,GPIO_HIGH);
	 else
	 	gpio_set_value(pdata->lcd_blg_gpio,GPIO_LOW);
#endif	 
	LCD_PRINFO(" %s.\n", enable ? "on" : "off");

}
/*************************************************************************
*	Name:leadt15ds26fb_set_brightness
*	Description:
*	Note:not realize in the function
**************************************************************************/
SINT32 set_backlight(struct fb_info *info, int level)
{
    unsigned int i = 0;
    unsigned int j = 0;
    unsigned int real_level = 0;
    int ret = 0;
	
	struct leadt15ds26fb_par *par = (struct leadt15ds26fb_par *)(info->par);
	struct zx29_lcd_platform_data *pdata = (struct zx29_lcd_platform_data *)par->spi->dev.platform_data;
	
	if (level > 255 || level < 0)
	{
		return -1;
	}
	else if (200 < level && level <= 250)
	{
		real_level = 5;
	}
	else if (150 < level && level <= 200)
	{
		real_level = 7;
	}
	else if (100 < level && level <= 150)
	{
		real_level = 9;
	}
	else if (50 < level && level <= 100)
	{
		real_level = 11;
	}
	else if (25 < level && level <= 50)
	{
		real_level = 13;
	}
	else if (0 < level && level <= 25)
	{
		real_level = 15;
	}
	//real_level = level/16 - 2; 

	//real_level  = 16 - real_level; // 15 12 9 6 3
	
	LCD_PRINFO("lcd:pulse to set is  %d\n",real_level);
	
	//turn light on
	if (real_level > last_level)
	{
		j =  real_level -last_level;
	}
	//turn light off
	else if (real_level < last_level)
	{
		j =  real_level + 16 - last_level;
	}

	for(i =0 ; i < j;i++ )
	{
	
#ifdef CONFIG_LCD_BLG_SINK
		ret += zx234297_set_sink(ZX234297_SINK1,0,SINK_CURRENT_5MA);
		udelay(15);
		ret += zx234297_set_sink(ZX234297_SINK1,1,SINK_CURRENT_5MA);
		udelay(15);
		if(ret < 0)
			LCD_PRERR("error\n");			
#else	
		gpio_set_value(pdata->lcd_blg_gpio, GPIO_LOW);
		udelay(15);
		gpio_set_value(pdata->lcd_blg_gpio, GPIO_HIGH);
		udelay(15); //us
#endif
	}
	last_level = (last_level + j)%16;
	return 0;
}
static int leadt15ds26fb_set_brightness(struct fb_info *info, unsigned int level)
{
	int ret = 0;
	struct leadt15ds26fb_par *par = (struct leadt15ds26fb_par *)(info->par);

	if (level > 255)
    {
        return -EINVAL;
    }
	ret = set_backlight(info, level);
    //LCD_REG_WRITE(BLG_LIGHT_LEVEL_ADDR, level);
    
	LCD_PRINFO("[LCD][LEAD]leadt15ds26fb_set_brightness level %d.\n",level);
    par->lcd_brightness= level;

    return ret;
}

/**************************************************************************
* Function: leadt15ds26fb_set_ram_address
* Description: used to set specified start point of DISPLAY RAM for updating laterly
* Parameters:
*       (IN)
*               uiLeft      left postion
*               uiTop       top postion
*               uiRight     left postion
*               uiBottom    bottom postion
*       (OUT)
*               None.
* Returns:
*       0: successed.
*       < 0: others error code. For more details, please refer to the source code.
* Others:
*       None.
**************************************************************************/
static int leadt15ds26fb_set_ram_address(struct fb_info *info, unsigned short uiLeft, unsigned short uiTop, unsigned short uiRight, unsigned short uiBottom)
{
	int ret = 0;
	struct leadt15ds26fb_par *par = (struct leadt15ds26fb_par *)(info->par);
    
    ret |= leadt15ds26fb_spi_write_cmd(par->spi, 0x2A);
    ret |= leadt15ds26fb_spi_write_data(par->spi, 0x00);
    ret |= leadt15ds26fb_spi_write_data(par->spi, uiLeft % (info->var.xres) + info->var.xoffset);
    ret |= leadt15ds26fb_spi_write_data(par->spi, (uiRight>>8));
    ret |= leadt15ds26fb_spi_write_data(par->spi, (uiRight&0xff)); // new

    ret |= leadt15ds26fb_spi_write_cmd(par->spi, 0x2B);
    ret |= leadt15ds26fb_spi_write_data(par->spi, 0x00);
    ret |= leadt15ds26fb_spi_write_data(par->spi, uiTop % (info->var.yres) + info->var.yoffset);
    ret |= leadt15ds26fb_spi_write_data(par->spi, (uiBottom>>8));
    ret |= leadt15ds26fb_spi_write_data(par->spi, (uiBottom&0xff));

    return ret;
}

/**************************************************************************
* Function: leadt15ds26fb_mmap
* Description: used to remap framebuffer's address to user space.
* Parameters:
*       (IN)
*
*               uiLeft      left postion
*               uiTop       top postion
*               uiWidth     rectangle width
*               uiHeight    rectangle height
*       (OUT)
*               None.
* Returns:
*       0: successed.
*       < 0: others error code. For more details, please refer to the source code.
* Others:
*       Most drivers don't need their own mmap function.
**************************************************************************/
static int leadt15ds26fb_mmap(struct fb_info *info, struct vm_area_struct *vma)
{
	/* this is uClinux (no MMU) specific code */
	vma->vm_flags |= VM_DONTEXPAND; // | VM_DONTDUMP;
	vma->vm_start = info->screen_base;

	return 0;
}

#ifdef CONFIG_ARCH_ZX297520V3_FWP

int lcd_SetRamAddr(struct fb_info *info, unsigned short uiCol, unsigned short uiPage)
{
	int ret = 0;
	struct leadt15ds26fb_par *par = (struct leadt15ds26fb_par *)(info->par);

    /*set column address*/
    ret |= leadt15ds26fb_spi_write_cmd(par->spi, START_HIGH_BIT |((uiCol>>4) & 0x0F)); // 4λ
    ret |= leadt15ds26fb_spi_write_cmd(par->spi, (START_LOW_BIT | (uiCol & 0x0F))+START_LOW_PY);    // 4λ    

    /*set page address*/
    ret |= leadt15ds26fb_spi_write_cmd(par->spi, START_PAGE | (uiPage & 0x0F));

    return ret;
}

int lcd_WritePixel(struct fb_info *info, unsigned short rgb565,  unsigned short uiRow,  unsigned short uiCol)
{
	int ret = 0;
	unsigned short page_index = uiRow / LCD_PAGE_HEIGHT;
	unsigned short ram_offset = page_index * LCD_WIDTH + uiCol;
	unsigned short bit_shift = uiRow % LCD_PAGE_HEIGHT;

	if (rgb565 != 0 )
	*(info->screen_base + info->fix.smem_len + ram_offset) &= ~(0x01 << bit_shift);
	else 
	*(info->screen_base + info->fix.smem_len + ram_offset) |= (0x01 << bit_shift);

	return ret;
}


#endif

/**************************************************************************
* Function: leadt15ds26fb_write
* Description: used to update a specified rectangle area of the LCD device opened previously.
* Parameters:
*       (IN)
*               pBuf        data buffer
*               uiLeft      left postion
*               uiTop       top postion
*               uiWidth     rectangle width
*               uiHeight    rectangle height
*       (OUT)
*               None.
* Returns:
*       0: successed.
*       < 0: others error code. For more details, please refer to the source code.
* Others:
*       None.
**************************************************************************/
static ssize_t leadt15ds26fb_write(struct fb_info *info, const char __user *buf, size_t count, loff_t *ppos)
{
    int ret = 0;
    struct leadt15ds26fb_par *par = (struct leadt15ds26fb_par *)(info->par);
    unsigned long screen_pos = *ppos;
    unsigned char __iomem *dst;
	unsigned int block_nr;
	unsigned int nr;
	unsigned short  row, col, pixel_value;
	unsigned short cur_page;

    if (!count || !buf)
        return -EINVAL;

    if (screen_pos > (info->fix.smem_len))
        return -EFBIG;

    /* if LCD is in sleep-mode, return directly. */
    if(LCD_SLEEP == par->lcd_sleep_mode){
        return 0;
    }

    /* in case of reentry. */
    mutex_lock(&(par->io_lock));

    if (count + screen_pos > (info->fix.smem_len))
        count = (info->fix.smem_len) - screen_pos;

    dst = (void __force *) (info->screen_base + screen_pos);
#if 0
	//dst = info->screen_base;
	memcpy(dst, buf, count);
	//#else
	if (copy_from_user(dst, buf, count)){
		mutex_unlock(&(par->io_lock));
		return -EFAULT;
	}
#endif

#ifdef  CONFIG_FB_LCD_TE_ON
	again:
	
	/*detect TE sync irq times more than 3, the HW be regard as the LCD with TE.*/
	if (atomic_read(&g_Lcd_Fmark_count) > 3) 
	{
		g_LcdSema_Flag = true;	
		if(0 != down_timeout(&gLcd_SyncSema, msecs_to_jiffies(100)))
		{
			/*timeout 100ms, regard as no TE LCD.*/
			atomic_set(&g_Lcd_Fmark_count,0);
		}
		else
		{		
			/*wait for the next TE sync, if the wait time too long*/
			if(jiffies_to_msecs(jiffies - g_Lcd_tick_count) > 2)
				goto again;
		}
	}
#endif

    #ifdef CONFIG_PM_WAKELOCKS
    wake_lock(&(par->wlock_write)); /* prevant system into sleep for protecting DMA transmission. */
    #endif

#ifdef CONFIG_ARCH_ZX297520V3_FWP
	
	for (row = 0; row <info->var.yres; row++)
	{
		for (col = 0; col < info->var.xres; col++)
		{
			pixel_value = *( (unsigned short *)info->screen_base + (row *  info->var.xres + col));//screen_base

			lcd_WritePixel(info, pixel_value, row , col );
		}
	}   

	//zOss_Printf(SUBMDL_TEST, PRINT_LEVEL_NORMAL, "lcd_UpdateRect CONVERT tick = %d\n", zOss_GetTickCount);

	/* СRAMȥÿһpage */

	for ( cur_page =0; cur_page < LCD_PAGE_NUM; cur_page++ )
	{
		ret |= lcd_SetRamAddr(info, info->var.yoffset, cur_page );
		ret |= leadt15ds26fb_spi_write_datablock(par->spi,  (info->fix.smem_start + info->fix.smem_len  + cur_page*  info->var.xres),  info->var.xres);       
	}


#else
    ret |= leadt15ds26fb_set_ram_address(info, 0, 0, info->var.xres + info->var.xoffset - 1, info->var.yres + info->var.yoffset - 1); /* specifies new start-position */
    ret |= leadt15ds26fb_spi_write_cmd(par->spi, 0x2C);  /* send ram-write command and display data */

	block_nr = info->fix.smem_len / 65535 + 1;
	#ifdef CONFIG_ARCH_ZX297520V3_MIFI
	#else
	par->spi->bits_per_word = 32;
	spi_setup(par->spi);
    #endif
	for(nr = 0; nr < block_nr; nr++)
	{
		ret |= leadt15ds26fb_spi_write_datablock(par->spi, info->fix.smem_start + (info->fix.smem_len / block_nr * nr),(info->fix.smem_len) / block_nr);	
	}
	#ifdef CONFIG_ARCH_ZX297520V3_MIFI
	#else
	par->spi->bits_per_word = 8;
	spi_setup(par->spi);
	#endif
#endif
	
    #ifdef CONFIG_PM_WAKELOCKS
    wake_unlock(&(par->wlock_write));
    #endif

    mutex_unlock(&(par->io_lock));
    //LCD_PRINFO("[LCD][LEAD]leadt15ds26fb_write out\n");
    return count;
}

static void leadt15ds26fb_fillrect(struct fb_info *info, const struct fb_fillrect *rect)
{
    int ret = 0;
    struct leadt15ds26fb_par *par = (struct leadt15ds26fb_par *)(info->par);
	unsigned int block_nr;
	unsigned int nr;

    /* if LCD is in sleep-mode, return directly. */
    if(LCD_SLEEP == par->lcd_sleep_mode || rect == NULL || rect->width == 0 || rect->height == 0){
        return ;
    }
	
#ifdef  CONFIG_FB_LCD_TE_ON
	again:
	
	/*detect TE sync irq times more than 3, the HW be regard as the LCD with TE.*/
	if (atomic_read(&g_Lcd_Fmark_count) > 3) 
	{
		g_LcdSema_Flag = true;	
		if(0 != down_timeout(&gLcd_SyncSema, msecs_to_jiffies(100)))
		{
			/*timeout 100ms, regard as no TE LCD.*/
			atomic_set(&g_Lcd_Fmark_count,0);
		}
		else
		{		
			/*wait for the next TE sync, if the wait time too long*/
			if(jiffies_to_msecs(jiffies - g_Lcd_tick_count) > 2)
				goto again;
		}
	}
#endif

    #ifdef CONFIG_PM_WAKELOCKS
    wake_lock(&(par->wlock_write)); /* prevant system into sleep for protecting DMA transmission. */
    #endif

    ret |= leadt15ds26fb_set_ram_address(info, rect->dx, rect->dy, /*rect->dx + */info->var.xres - 1, rect->dy + rect->height - 1); /* specifies new start-position */
    ret |= leadt15ds26fb_spi_write_cmd(par->spi, 0x2C);  /* send ram-write command and display data */

	block_nr = (rect->width * rect->height *2) / 65504 + 1;
	#ifdef CONFIG_ARCH_ZX297520V3_MIFI
	#else
	par->spi->bits_per_word = 32;
	spi_setup(par->spi);
    #endif
	for(nr = 0; nr < block_nr; nr++)
	{
		ret |= leadt15ds26fb_spi_write_datablock(par->spi, info->fix.smem_start + (rect->width * rect->height * 2) / block_nr * nr, (rect->width * rect->height * 2) / block_nr);	
	}
	
	#ifdef CONFIG_ARCH_ZX297520V3_MIFI
	#else
	par->spi->bits_per_word = 8;
	spi_setup(par->spi);
	#endif
	
    #ifdef CONFIG_PM_WAKELOCKS
    wake_unlock(&(par->wlock_write));
    #endif

    //LCD_PRINFO("[LCD][LEAD]leadt15ds26fb_fillrect out\n");
}
static int leadt15ds26fb_ioctl(struct fb_info *info, unsigned int cmd, unsigned long arg)
{
    unsigned char ioctl_arg;
	struct fb_fillrect disp_rect = {0};
    struct leadt15ds26fb_par * par = (struct leadt15ds26fb_par *)(info->par);

    mutex_lock(&(par->io_lock));
	
#ifdef CONFIG_PM_WAKELOCKS
	wake_lock(&(par->wlock_ioctl)); /* prevant system into sleep for protecting DMA transmission. */
#endif
	
	LCD_PRINFO("[LCD][LEAD]leadt15ds26fb_ioctl enter cmd=%d\n",cmd);
    switch (cmd)
    {
        case LEADT15DS26_SET_SLEEPMODE:
        {
            if (copy_from_user(&ioctl_arg, (char *)arg, sizeof(unsigned char)))
        	{	
			#ifdef CONFIG_PM_WAKELOCKS
				wake_unlock(&(par->wlock_ioctl));
			#endif
				mutex_unlock(&(par->io_lock));
	            return -EFAULT;
        	}
            leadt15ds26fb_set_sleep(info, (bool)ioctl_arg);
            break;
        }
        case LEADT15DS26_SET_BACKLIGHT:
        {
            if (copy_from_user(&ioctl_arg, (char *)arg, sizeof(unsigned char)))
			{
			#ifdef CONFIG_PM_WAKELOCKS
				wake_unlock(&(par->wlock_ioctl));
			#endif
				mutex_unlock(&(par->io_lock));
				return -EFAULT;
			}
            leadt15ds26fb_set_backlight(info, (bool)ioctl_arg);
            break;
        }
        case LEADT15DS26_SET_BRIGHTNESS:
        {
            if (copy_from_user(&ioctl_arg, (char *)arg, sizeof(unsigned char)))
			{
			#ifdef CONFIG_PM_WAKELOCKS
				wake_unlock(&(par->wlock_ioctl));
			#endif
				mutex_unlock(&(par->io_lock));
				return -EFAULT;
			}
            leadt15ds26fb_set_brightness(info, ioctl_arg);
            break;
        }
        case LEADT15DS26_FILLRECT:
        {
            if (copy_from_user(&disp_rect, (struct fb_fillrect *)arg, sizeof(struct fb_fillrect)))
			{
	#ifdef CONFIG_PM_WAKELOCKS
				wake_unlock(&(par->wlock_ioctl));
	#endif
				mutex_unlock(&(par->io_lock));
				return -EFAULT;
			}
			
		if(disp_rect.dx >=320 &&disp_rect.dy >= 320 && disp_rect.color > 0 && disp_rect.rop > 0)
		{
				return -EFAULT;
		}
		
		if(disp_rect.width <= 320 && disp_rect.height <=320)
		{
	        leadt15ds26fb_fillrect(info, &disp_rect);
		}
		
            break;
        }
    default:
		#ifdef CONFIG_PM_WAKELOCKS
			wake_unlock(&(par->wlock_ioctl));
		#endif
			mutex_unlock(&(par->io_lock));
        return -EINVAL;
    }
	
#ifdef CONFIG_PM_WAKELOCKS
    wake_unlock(&(par->wlock_ioctl));
#endif

    mutex_unlock(&(par->io_lock));
    return 0;
}

static struct fb_ops leadt15ds26fb_ops = {
    .owner          = THIS_MODULE,
    .fb_read        = fb_sys_read, /* useless, so link to default read function */
    .fb_write       = leadt15ds26fb_write,
    .fb_ioctl       = leadt15ds26fb_ioctl,
	//.fb_fillrect	= leadt15ds26fb_fillrect,
    //.fb_mmap    = leadt15ds26fb_mmap,
};

static u64 leadt15ds26fb_dmamask = DMA_BIT_MASK(32);


int lcd_pmu_notifier_call(struct notifier_block * nb, unsigned long val, void * v)
{
	//struct leadt15ds26fb_par * par = (struct leadt15ds26fb_par *)(g_leadt15ds26fb_par.info->par);
	struct leadt15ds26fb_par * par = &g_leadt15ds26fb_par;
	bool backlight_status = par->lcd_backlight;
	
	LCD_PRINFO("[LCD][LEAD]in, val=%ld\n",val);
	mutex_lock(&(par->io_lock));
#ifdef CONFIG_PM_WAKELOCKS
	wake_lock(&(par->wlock_write)); /* prevant system into sleep for protecting DMA transmission. */
#endif

	leadt15ds26fb_set_backlight(par->info,0);//off
	leadt15ds26fb_init_lcm(par->info);
	leadt15ds26fb_set_backlight(par->info, backlight_status);//off
	
#ifdef CONFIG_PM_WAKELOCKS
	wake_unlock(&(par->wlock_write));
#endif
	mutex_unlock(&(par->io_lock));
	LCD_PRINFO(" [LCD][LEAD]END~~\n");
	return 0;
}

int lcd_reboot_event(struct notifier_block *this, unsigned long event, void *ptr)
{
	void *base;

	leadt15ds26fb_setbacklight_before_reset(0);

	return NOTIFY_DONE;
}

static struct notifier_block zx29_lcd_reboot_notifier = {
	.notifier_call = lcd_reboot_event
};

#ifdef CONFIG_DEBUG_FS
static ssize_t debugfs_lcd_write(struct file *file, const char __user *buf,size_t nbytes, loff_t *ppos)
{
	unsigned long val1, val2;
	u8 reg, value;
	int ret;
	void *kern_buf;
	struct seq_file	*s = file->private_data;
	struct leadt15ds26fb_par *lcd_par = s->private;

	kern_buf = kzalloc(nbytes, GFP_KERNEL);

	if (!kern_buf) {
		printk(KERN_INFO "[LCD][LEAD]: Failed to allocate buffer\n");
		return -ENOMEM;
	}

	if (copy_from_user(kern_buf, (void  __user *)buf, nbytes)) {
		kfree(kern_buf);
		return -ENOMEM;
	}
	//printk(KERN_INFO "%s input str=%s,nbytes=%d \n", __func__, kern_buf,nbytes);	
	
	int error = 0;

	LCD_PRINFO("[LCD][LEAD]leadfb_store enter\n");
	
	if (!lcd_par->info){
		//zx_fb_prn("g_leadt15ds26fb_par.info=NULL\n");
		kfree(kern_buf);
		LCD_PRINFO("[LCD][LEAD]g_leadt15ds26fb_par.info=NULL\n");
		return 0;
	}

	if (0 == strncmp(kern_buf, "reinit", 6)){
		leadt15ds26fb_init_lcm(lcd_par->info);
	}
	else if (0 == strncmp(kern_buf, "update", 6)){
		struct leadt15ds26fb_par *par = (struct leadt15ds26fb_par *)(lcd_par->info->par);
		extern unsigned char bmp_128_128_specify[];
		/* specifies new start-position */
		leadt15ds26fb_set_ram_address(lcd_par->info, 0, 0, (lcd_par->info->var.xres) - 1, (lcd_par->info->var.yres)- 1);
		/* send ram-write command and display data */
		leadt15ds26fb_spi_write_cmd(par->spi, 0x2C);
		memcpy(lcd_par->info->screen_base,bmp_128_128_specify,128*128*2);
		leadt15ds26fb_spi_write_datablock(par->spi, lcd_par->info->fix.smem_start, lcd_par->info->fix.smem_len);
	}
	else if (0 == strncmp(kern_buf, "sleepin", 7)){
		leadt15ds26fb_set_sleep(lcd_par->info, true);
	}
	else if (0 == strncmp(kern_buf, "sleepout", 8)){
		leadt15ds26fb_set_sleep(lcd_par->info, false);
	}
	else if (0 == strncmp(kern_buf, "blgon", 5)){
		leadt15ds26fb_set_backlight(lcd_par->info, true);
	}
	else if (0 == strncmp(kern_buf, "blgoff", 6)){
		leadt15ds26fb_set_backlight(lcd_par->info, false);
	}
	else if (0 == strncmp(kern_buf, "brn", 3)){
		leadt15ds26fb_set_brightness(lcd_par->info, 10);
		msleep(1000);
		leadt15ds26fb_set_brightness(lcd_par->info, 128);
		msleep(1000);
		leadt15ds26fb_set_brightness(lcd_par->info, 255);
	}
	else if (0 == strncmp(kern_buf, "simspion", 8)){
		use_actual_spi = false;
	}
	else if (0 == strncmp(kern_buf, "simspioff", 9)){
		use_actual_spi = true;
	}
	else if (0 == strncmp(kern_buf, "parinfo", 7)){
		LCD_PRINFO("g_leadt15ds26fb_par.spi = 0x%08x; g_leadt15ds26fb_par.info = 0x%08x\n", lcd_par->spi, lcd_par->info);
	}
	else{
		LCD_PRINFO("undefined cmd\n");
	}

	LCD_PRINFO("leadfb_store() done!\n");
	kfree(kern_buf);
	return error ? error : nbytes;
}

static int debugfs_lcd_show(struct seq_file *s, void *v)
{
	seq_printf(s, "%s  lcd_id = 0x%x,\n", __func__, g_leadt15ds26fb_par.lcd_id);
	seq_printf(s, "%s  sleep mode  %s,\n", __func__, g_leadt15ds26fb_par.lcd_sleep_mode == LCD_SLEEP ? "sleep":"normal");
	seq_printf(s, "%s  g_sLcd_Backlight  %s,\n", __func__, g_leadt15ds26fb_par.lcd_backlight? "on" : "off");

	return 0;
}

#define DEBUGFS_FILE_ENTRY(name) \  
static int debugfs_##name##_open(struct inode *inode, struct file *file) \  
{\  
return single_open(file, debugfs_##name##_show, inode->i_private); \  
}\  
\  
static const struct file_operations debugfs_##name##_fops = { \  
.owner= THIS_MODULE, \  
.open= debugfs_##name##_open, \  
.write=debugfs_##name##_write, \
.read= seq_read, \  
.llseek= seq_lseek, \  
.release= single_release, \  
}  

DEBUGFS_FILE_ENTRY(lcd);

static struct dentry *g_lcd_root;

static void debugfs_lcd_init(struct leadt15ds26fb_par  *lcd_par)
{
	struct dentry *root;
	struct dentry *node;		
	int i;
	
	if(!lcd_par)
		return;

	//create root
	root = debugfs_create_dir("lcd_zx29", NULL);
	if (!root)	{	
		dev_err(&(lcd_par->spi->dev), "debugfs_create_dir err=%d\n", IS_ERR(root));
		goto err;
	}

	//print regs;
	node = debugfs_create_file("lcd_lead", S_IRUGO | S_IWUGO, root, lcd_par,  &debugfs_lcd_fops);
	if (!node){
		dev_err(&(lcd_par->spi->dev), "debugfs_create_dir err=%d\n", IS_ERR(node));
		goto err;
	}
	
	g_lcd_root = (void *)root;
	return;
err:
	dev_err(&(lcd_par->spi->dev), "debugfs_spi_init err\n");
}

#endif

/* 豸̽⺯ */
static int leadt15ds26fb_probe(struct spi_device *spi, const struct spi_device_id *id)
{
	int ret = 0;
	struct fb_info *fbi;
	dma_addr_t handle;
	void *base;
	
	LCD_PRINFO("[LCD][LEAD]leadt15ds26fb_probe in.\n");

	fbi = framebuffer_alloc(0, &spi->dev); //framebuffer_alloc(sizeof(struct leadt15ds26fb_par), &spi->dev);
	if (!fbi){
		LCD_PRERR("[LCD][LEAD]Couldn't allocate the framebuffer.\n");
		return -ENOMEM;
	}

	fbi->par = &g_leadt15ds26fb_par;
	fbi->fbops= &leadt15ds26fb_ops;
	
	g_leadt15ds26fb_par.spi = spi;
	
	mutex_init(&g_leadt15ds26fb_par.io_lock);
	wake_lock_init(&g_leadt15ds26fb_par.wlock_write, WAKE_LOCK_SUSPEND, "lead_wake_lock_write");
	wake_lock_init(&g_leadt15ds26fb_par.wlock_ioctl, WAKE_LOCK_SUSPEND, "lead_wake_lock_ioctl");
	wake_lock_init(&g_leadt15ds26fb_par.wlock_backlight, WAKE_LOCK_SUSPEND, "lead_wake_lock_backlight");
	
	ret = leadt15ds26fb_init_gpio(fbi);
	
	g_leadt15ds26fb_par.lcd_id = read_st(0xda);
	g_leadt15ds26fb_par.driver_version = read_st(0xdb) << 8 + read_st(0xdc);
	
	if(LCM_LEAD_ID == g_leadt15ds26fb_par.lcd_id) 
	{
	#ifdef CONFIG_ARCH_ZX297520V3_PHONE
		fbi->var.xres = 128; 
		fbi->var.yres = 160;  
		fbi->var.xoffset= 0x00;
		fbi->var.yoffset= 0x00;
		#ifdef LCD_ROTATE_180
		fbi->var.colorspace = 0x14;
		#else
		fbi->var.colorspace = 0xC4;
		#endif
		fbi->var.bits_per_pixel = 16;
	#elif (defined CONFIG_FB_LCM_LEAD_128_128_F231E||defined CONFIG_LCD_BLG_SINK)
		fbi->var.xres = 128; 
		fbi->var.yres = 128;  
		fbi->var.xoffset= 0;
		fbi->var.yoffset= 32;
		fbi->var.bits_per_pixel = 16;
		fbi->var.colorspace = 0xC8;
	#else
		fbi->var.xres = 128; 
		fbi->var.yres = 128;  
		fbi->var.xoffset= 0x02;
		#ifdef LCD_ROTATE_180
		fbi->var.yoffset= 0x01;
		fbi->var.colorspace = 0x1C;
		#else
		fbi->var.yoffset = 0x03;
		fbi->var.colorspace = 0xC8;
		#endif
		fbi->var.bits_per_pixel = 16;
		spi->max_speed_hz = 26000000;
	#endif
	} 
	else if(LCM_ST7789V_ID == g_leadt15ds26fb_par.lcd_id) 
	{
		#ifdef CONFIG_ARCH_ZX297520V3_PHONE
		fbi->var.xres = 240; 
		fbi->var.yres = 320;  
		fbi->var.xoffset= 0;
		fbi->var.yoffset= 0;
		fbi->var.bits_per_pixel = 16;
		fbi->var.colorspace = 0x00;
		#else
		fbi->var.xres = 240; 
		fbi->var.yres = 240;  
		fbi->var.xoffset= 0;
		fbi->var.yoffset= 0;
		#ifdef CONFIG_FB_LCM_ST7789V_Z2
		fbi->var.yoffset= 40;
		#endif
		fbi->var.bits_per_pixel = 16;
		fbi->var.colorspace = 0x00;
		#endif
	} 
	else if(LCM_ILI9342C_ID == g_leadt15ds26fb_par.lcd_id) 
	{
		fbi->var.xres = 320; 
		fbi->var.yres = 240;  
		fbi->var.xoffset= 0;
		fbi->var.yoffset= 0;
		fbi->var.bits_per_pixel = 16;
		fbi->var.colorspace = 0;
	} 
	#ifdef CONFIG_ARCH_ZX297520V3_PHONE
	else if(LCM_GC930X_ID == g_leadt15ds26fb_par.lcd_id) 
	{
			fbi->var.xres = 240;
			fbi->var.yres = 320;
			fbi->var.xoffset= 0;
			fbi->var.yoffset= 0;
			fbi->var.bits_per_pixel = 16;
		#ifdef LCD_ROTATE_180
			fbi->var.colorspace = 0x1C;
		#else
			fbi->var.colorspace = 0xC8;
		#endif
	}
	#endif
	else
	{
#ifdef CONFIG_ARCH_ZX297520V3_FWP
		fbi->var.xres = 128;
		fbi->var.yres = 64;
		fbi->var.xoffset= 0;
		fbi->var.yoffset= 0;
		fbi->var.bits_per_pixel = 16;
#else
		fbi->var.xres = 128;
		fbi->var.yres = 160;
		fbi->var.xoffset= 0;
		fbi->var.yoffset= 0;
		fbi->var.bits_per_pixel = 16;
	#ifdef LCD_ROTATE_180
		fbi->var.colorspace = 0x1C;
	#else
		fbi->var.colorspace = 0xC8;
	#endif
#endif
	}
	
	fbi->var.red.length = 5;
	fbi->var.green.length = 6;
	fbi->var.blue.length = 5;

	fbi->fix.smem_len = fbi->var.xres * fbi->var.yres * fbi->var.bits_per_pixel / 8;
		
	LCD_PRINFO("[LCD][LEAD]leadt15ds26fb_probe in.lcd_id(0x%x),driver_version(0x%x)\n",g_leadt15ds26fb_par.lcd_id,g_leadt15ds26fb_par.driver_version);

	spi->dev.dma_mask		= &leadt15ds26fb_dmamask;
	spi->dev.coherent_dma_mask	= leadt15ds26fb_dmamask;
	#ifdef CONFIG_ARCH_ZX297520V3_FWP
	base = dma_alloc_coherent(&(spi->dev), fbi->fix.smem_len + fbi->var.xres * LCD_PAGE_NUM, &handle,
		      GFP_KERNEL);
	if (base == NULL) {
		printk(KERN_ERR "leadfb: unable to allocate screen "
		   "memory\n");
		framebuffer_release(fbi);
		return -ENOMEM;
	}
	memset(base, 0x00, fbi->fix.smem_len + fbi->var.xres * LCD_PAGE_NUM);

	#else
	base = dma_alloc_coherent(&(spi->dev), fbi->fix.smem_len, &handle,
		      GFP_KERNEL);
	if (base == NULL) {
		printk(KERN_ERR "leadfb: unable to allocate screen "
		   "memory\n");
		framebuffer_release(fbi);
		return -ENOMEM;
	}
       #endif

	fbi->screen_base = base;
	fbi->fix.smem_start = handle;
	
	g_leadt15ds26fb_par.lcd_sleep_mode = LCD_NORMAL;
	g_leadt15ds26fb_par.lcd_backlight =0;
	g_leadt15ds26fb_par.info = fbi;
	g_leadt15ds26fb_par.info->par = &g_leadt15ds26fb_par;
	
	ret = register_framebuffer(fbi);
	if (ret){
		LCD_PRERR("[LCD][LEAD]Couldn't register the framebuffer.\n");
	}
	
#if 0
	/* The LCM have been initialized in bootloader stage. So no necessary to init it again. */
	ret |= leadt15ds26fb_init_lcm(fbi);
#endif
	register_reboot_notifier(&zx29_lcd_reboot_notifier);

	pmu_reinit_notifier.notifier_call = lcd_pmu_notifier_call;
	zx234290_register_client(&pmu_reinit_notifier);

#if 0
	/* just for test */
	{
	extern unsigned char bmp_128_128_caliberate[];
	int ppos = 0;
	memcpy(base,bmp_128_128_caliberate,128*128*2);
	leadt15ds26fb_write(g_leadt15ds26fb_par.info, base, 128*128*2, &ppos);
	}
#endif

#ifdef CONFIG_FB_LCD_TE_ON
	sema_init(&gLcd_SyncSema, 0);
	ret = request_threaded_irq(((struct leadt15ds26fb_par *)fbi->par)->irq_num, LcdSync_irq_handler, LcdSync_Isr, IRQF_ONESHOT,
				   "lcdsync", &spi->dev);
	
	if (ret){
		LCD_PRERR("[LCD][LEAD]request_threaded_irq failed.\n");
	}
#endif

#if defined(CONFIG_DEBUG_FS)
	debugfs_lcd_init(&g_leadt15ds26fb_par);
#endif

	return 0;
}

static int leadt15ds26fb_remove(struct spi_device *spi)
{
	struct fb_info *fbi = spi_get_drvdata(spi);
	//struct leadt15ds26fb_par *par = fbi->par;
	
	if(!fbi)	
		return -EINVAL;
	unregister_framebuffer(fbi);
	dma_free_coherent(&(spi->dev), fbi->fix.smem_len, (void *)fbi->screen_base, (dma_addr_t)fbi->fix.smem_start);
	framebuffer_release(fbi);
#if defined(CONFIG_DEBUG_FS)
	if(g_lcd_root){
		printk(KERN_INFO "lcd_exit:debugfs_remove_recursive \n");
		debugfs_remove_recursive(g_lcd_root);
	}
#endif

	return 0;
}

static const struct spi_device_id leadt15ds26fb_id[] = {
    {"lead_t15ds26", 0 },
    { }
};

MODULE_DEVICE_TABLE(spi, leadt15ds26fb_id);

static struct spi_driver leadt15ds26_spi_driver = {
    .driver = {
        .name = "lead_t15ds26",
        .owner = THIS_MODULE,
    },
    .probe = leadt15ds26fb_probe,
    .remove = leadt15ds26fb_remove,
    .id_table = leadt15ds26fb_id,
};

static int __init leadt15ds26_spi_init(void)
{
	int ret;

	ret = spi_register_driver(&leadt15ds26_spi_driver);
	if (ret != 0){
		LCD_PRERR("[LCD][LEAD]Failed to register leadt15ds26_spi_driver : %d\n", ret);
	}
	
	LCD_PRINFO("[LCD][LEAD]spi_register_driver - leadt15ds26_spi.\n");

	return ret;
}

static void __exit leadt15ds26_spi_exit(void)
{
	spi_unregister_driver(&leadt15ds26_spi_driver);
}

module_init(leadt15ds26_spi_init);
module_exit(leadt15ds26_spi_exit);
