/*
 * File: sdio_func.h
 * Brief: Public APIs of sdio
 * include/linux/mmc/sdio_func.h
 *
 *  Copyright 2007-2008 Pierre Ossman
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 */

#ifndef LINUX_MMC_SDIO_FUNC_H
#define LINUX_MMC_SDIO_FUNC_H

#include <linux/device.h>
#include <linux/mod_devicetable.h>

#include <linux/mmc/pm.h>

struct mmc_card;
struct sdio_func;

typedef void (sdio_irq_handler_t)(struct sdio_func *);

/*
 * Structure used to hold embedded SDIO device data from platform layer
 */
struct sdio_embedded_func {
	uint8_t f_class;
	uint32_t f_maxblksize;
};

/*
 * SDIO function CIS tuple (unknown to the core)
 */
struct sdio_func_tuple {
	struct sdio_func_tuple *next;
	unsigned char code;
	unsigned char size;
	unsigned char data[0];
};

/*
 * SDIO function devices
 */
struct sdio_func {
	struct mmc_card		*card;		/* the card this device belongs to */
	struct device		dev;		/* the device */
	sdio_irq_handler_t	*irq_handler;	/* IRQ callback */
	unsigned int		num;		/* function number */

	unsigned char		class;		/* standard interface class */
	unsigned short		vendor;		/* vendor id */
	unsigned short		device;		/* device id */

	unsigned		max_blksize;	/* maximum block size */
	unsigned		cur_blksize;	/* current block size */

	unsigned		enable_timeout;	/* max enable timeout in msec */

	unsigned int		state;		/* function state */
#define SDIO_STATE_PRESENT	(1<<0)		/* present in sysfs */

	u8			tmpbuf[4];	/* DMA:able scratch buffer */

	unsigned		num_info;	/* number of info strings */
	const char		**info;		/* info strings */

	struct sdio_func_tuple *tuples;
};

#define sdio_func_present(f)	((f)->state & SDIO_STATE_PRESENT)

#define sdio_func_set_present(f) ((f)->state |= SDIO_STATE_PRESENT)

#define sdio_func_id(f)		(dev_name(&(f)->dev))

#define sdio_get_drvdata(f)	dev_get_drvdata(&(f)->dev)
#define sdio_set_drvdata(f,d)	dev_set_drvdata(&(f)->dev, d)
#define dev_to_sdio_func(d)	container_of(d, struct sdio_func, dev)

/*
 * SDIO function device driver
 */
struct sdio_driver {
	char *name;
	const struct sdio_device_id *id_table;

	int (*probe)(struct sdio_func *, const struct sdio_device_id *);
	void (*remove)(struct sdio_func *);

	struct device_driver drv;
};

#define to_sdio_driver(d)	container_of(d, struct sdio_driver, drv)

/**
 * SDIO_DEVICE - macro used to describe a specific SDIO device
 * @vend: the 16 bit manufacturer code
 * @dev: the 16 bit function id
 *
 * This macro is used to create a struct sdio_device_id that matches a
 * specific device. The class field will be set to SDIO_ANY_ID.
 */
#define SDIO_DEVICE(vend,dev) \
	.class = SDIO_ANY_ID, \
	.vendor = (vend), .device = (dev)

/**
 * SDIO_DEVICE_CLASS - macro used to describe a specific SDIO device class
 * @dev_class: the 8 bit standard interface code
 *
 * This macro is used to create a struct sdio_device_id that matches a
 * specific standard SDIO function type.  The vendor and device fields will
 * be set to SDIO_ANY_ID.
 */
#define SDIO_DEVICE_CLASS(dev_class) \
	.class = (dev_class), \
	.vendor = SDIO_ANY_ID, .device = SDIO_ANY_ID



/**
 * Brief Register SDIO drivers with SDIO bus.
 *
 * Param drv     Pointer to the struct sdio_driver instance.
 * Return Standard errno.
 * Retval 0 if successful.
 * Retval Negative errno code if failure.
 */ 
extern int sdio_register_driver(struct sdio_driver *drv);


/**
 * Brief Unregister SDIO driver from SDIO bus.
 *
 * Param drv     Pointer to the struct sdio_driver instance.
 * Return none.
 */ 
extern void sdio_unregister_driver(struct sdio_driver *drv);

/*
 * SDIO I/O operations
 */

/**
 * Brief  *	Claim a bus for a set of operations. The SDIO function given
 *	        is used to figure out which bus is relevant.
 * Param func     Pointer to the SDIO function that will be accessed.
 * Return none.
 */ 
extern void sdio_claim_host(struct sdio_func *func);
/**
 * Brief Release a bus for a certain SDIO function.
 *
 * Param func     Pointer to the SDIO function that will be accessed.
 * Return none
 * Note   Release a bus, allowing others to claim the bus for their
 *	      operations.
 */
extern void sdio_release_host(struct sdio_func *func);

/**
 * Brief Enables a SDIO function for usage.
 *
 * Param func     Pointer to the SDIO function to enable.
 * Return Standard errno
 * Retval 0 if successful.
 * Retval Negative errno code if failure.
 * Note   Powers up and activates a SDIO function so that register
 *	access is possible.
 */
extern int sdio_enable_func(struct sdio_func *func);
/**
 * Brief Disable a SDIO function.
 *
 * Param func     Pointer to the SDIO function to enable.
 * Return Standard errno.
 * Retval 0 if successful.(ѡ: ֵ˵)
 * Retval Negative errno code if failure.
 * Note   Powers down and deactivates a SDIO function. Register access
 *	to this function will fail until the function is reenabled.
 */
extern int sdio_disable_func(struct sdio_func *func);

/**
 * Brief Set the block size of an SDIO function
 *
 * Param func     Pointer to the SDIO function to change.
 * Param blksz    new block size or 0 to use the default.
 * Return Standard errno
 * Retval 0 if successful.
 * Retval Negative errno code if failure.
 * Note   The default block size is the largest supported by both the function
 *	and the host, with a maximum of 512 to ensure that arbitrarily sized
 *	data transfer use the optimal (least) number of commands.
 *
 *	A driver may call this to override the default block size set by the
 *	core. This can be used to set a block size greater than the maximum
 *	that reported by the card; it is the driver's responsibility to ensure
 *	it uses a value that the card supports.
 */
extern int sdio_set_block_size(struct sdio_func *func, unsigned blksz);

/**
 * Brief Claim the IRQ for a SDIO function.
 *
 * Param func     Pointer to the SDIO function.
 * Param handler IRQ handler callback.
 * Return Standard errno.
 * Retval 0 if successful.
 * Retval Negative errno code if failure.
 * Note   Claim and activate the IRQ for the given SDIO function. The provided
 *	handler will be called when that IRQ is asserted.  The host is always
 *	claimed already when the handler is called so the handler must not
 *	call sdio_claim_host() nor sdio_release_host().
 */
extern int sdio_claim_irq(struct sdio_func *func, sdio_irq_handler_t *handler);
/**
 * Brief Release the IRQ for a SDIO function.
 *
 * Param func     Pointer to the SDIO function.
 * Return Standard errno.
 * Retval 0 if successful.
 * Retval Negative errno code if failure.
 * Note   Disable and release the IRQ for the given SDIO function.
 */
extern int sdio_release_irq(struct sdio_func *func);

/**
 * Brief Pads a transfer size to a more optimal value.
 *
 * Param func     Pointer to the SDIO function.
 * Param sz original transfer size.
 * Return Returns the improved size, which might be unmodified.
 * Note   Pads the original data size with a number of extra bytes in
 *	order to avoid controller bugs and/or performance hits
 *	(e.g. some controllers revert to PIO for certain sizes).
 *
 *	If possible, it will also adjust the size so that it can be
 *	handled in just a single request.
 */
extern unsigned int sdio_align_size(struct sdio_func *func, unsigned int sz);

/**
 * Brief Read a single byte from a SDIO function
 *
 * Param func     Pointer to the SDIO function to access.
 * Param addr address to read
 * Param err_ret  optional status value from transfer
 *
 * Return Read value
 * Retval Read value
 * Retval 0xff if failed.
 * Note   Reads a single byte from the address space of a given SDIO
 *	function. If there is a problem reading the address, 0xff
 *	is returned and @err_ret will contain the error code.
 */
extern u8 sdio_readb(struct sdio_func *func, unsigned int addr, int *err_ret);
/**
 * Brief Read a single byte from a SDIO function
 *
 * Param func     Pointer to the SDIO function to access.
 * Param addr address to read
 * Param err_ret  optional status value from transfer
 * Param in value to add to argument
 *
 * Return Read value
 * Retval Read value
 * Retval 0xff if failed.
 * Note   Reads a single byte from the address space of a given SDIO
 *	function. If there is a problem reading the address, 0xff
 *	is returned and @err_ret will contain the error code.
 */
extern u8 sdio_readb_ext(struct sdio_func *func, unsigned int addr, int *err_ret,
	unsigned in);
/**
 * Brief Read a 16 bit integer from a SDIO function
 *
 * Param func     Pointer to the SDIO function to access.
 * Param addr address to read
 * Param err_ret  optional status value from transfer
 *
 * Return Read value
 * Retval Read value
 * Retval 0xffff if failed.
 * Note   Reads a 16 bit integer from the address space of a given SDIO
 *	function. If there is a problem reading the address, 0xffff
 *	is returned and @err_ret will contain the error code.
 */
extern u16 sdio_readw(struct sdio_func *func, unsigned int addr, int *err_ret);
/**
 * Brief Read a 32 bit integer from a SDIO function
 *
 * Param func     Pointer to the SDIO function to access.
 * Param addr address to read
 * Param err_ret  optional status value from transfer
 *
 * Return Read value
 * Retval Read value
 * Retval 0xffffffff if failed.
 * Note   Reads a 32 bit integer from the address space of a given SDIO
 *	function. If there is a problem reading the address,
 *	0xffffffff is returned and @err_ret will contain the error
 *	code.
 */
extern u32 sdio_readl(struct sdio_func *func, unsigned int addr, int *err_ret);

/**
 * Brief Read a chunk of memory from a SDIO function
 *
 * Param func     Pointer to the SDIO function to access.
 * Param dst buffer to store the data
 * Param addr  address to begin reading from
 * Param count  number of bytes to read
 *
 * Return Standard errno
 * Retval 0 if successful.
 * Retval Negative errno code if failure.
 * Note   Reads from the address space of a given SDIO function. Return
 *	value indicates if the transfer succeeded or not.
 */
extern int sdio_memcpy_fromio(struct sdio_func *func, void *dst,
	unsigned int addr, int count);
/**
 * Brief Read from a FIFO on a SDIO function
 *
 * Param func     Pointer to the SDIO function to access.
 * Param dst buffer to store the data
 * Param addr  address of (single byte) FIFO
 * Param count  number of bytes to read
 *
 * Return Standard errno
 * Retval 0 if successful.
 * Retval Negative errno code if failure.
 * Note   Reads from the specified FIFO of a given SDIO function. Return
 *	value indicates if the transfer succeeded or not.
 */
extern int sdio_readsb(struct sdio_func *func, void *dst,
	unsigned int addr, int count);

/**
 * Brief Write a single byte to a SDIO function
 *
 * Param func     Pointer to the SDIO function to access.
 * Param b byte to write
 * Param addr  address to write to
 * Param err_ret  optional status value from transfer
 *
 * Return none
 * Note   Writes a single byte to the address space of a given SDIO
 *	function. @err_ret will contain the status of the actual
 *	transfer.
 */
extern void sdio_writeb(struct sdio_func *func, u8 b,
	unsigned int addr, int *err_ret);
/**
 * Brief Write a 16 bit integer to a SDIO function
 *
 * Param func     Pointer to the SDIO function to access.
 * Param b integer to write
 * Param addr  address to write to
 * Param err_ret  optional status value from transfer
 *
 * Return none
 * Note   Writes a 16 bit integer to the address space of a given SDIO
 *	function. @err_ret will contain the status of the actual
 *	transfer.
 */
extern void sdio_writew(struct sdio_func *func, u16 b,
	unsigned int addr, int *err_ret);
/**
 * Brief Write a 32 bit integer to a SDIO function
 *
 * Param func     Pointer to the SDIO function to access.
 * Param b integer to write
 * Param addr  address to write to
 * Param err_ret  optional status value from transfer
 *
 * Return none
 * Note   Writes a 32 bit integer to the address space of a given SDIO
 *	function. @err_ret will contain the status of the actual
 *	transfer.
 */
extern void sdio_writel(struct sdio_func *func, u32 b,
	unsigned int addr, int *err_ret);

/**
 * Brief Write and read a byte from SDIO function
 *
 * Param func     Pointer to the SDIO function to access.
 * Param write_byte byte to write
 * Param addr  address to write to
 * Param err_ret  optional status value from transfer
 *
 * Return Read value
 * Retval Read value
 * Retval 0xff if failed.
 * Note   Performs a RAW (Read after Write) operation as defined by SDIO spec -
 *	single byte is written to address space of a given SDIO function and
 *	response is read back from the same address, both using single request.
 *	If there is a problem with the operation, 0xff is returned and
 *	@err_ret will contain the error code.
 */
extern u8 sdio_writeb_readb(struct sdio_func *func, u8 write_byte,
	unsigned int addr, int *err_ret);

/**
 * Brief  Write a chunk of memory to a SDIO function
 *
 * Param func     Pointer to the SDIO function to access.
 * Param addr address to start writing to
 * Param src  buffer that contains the data to write
 * Param count  number of bytes to write
 *
 * Return Standard errno
 * Retval 0 if successful.
 * Retval Negative errno code if failure.
 * Note   Writes to the address space of a given SDIO function. Return
 *	value indicates if the transfer succeeded or not.
 */
extern int sdio_memcpy_toio(struct sdio_func *func, unsigned int addr,
	void *src, int count);
/**
 * Brief Write to a FIFO of a SDIO function
 *
 * Param func     Pointer to the SDIO function to access.
 * Param addr address of (single byte) FIFO
 * Param src  buffer that contains the data to write
 * Param count  number of bytes to write
 *
 * Return Standard errno
 * Retval 0 if successful.
 * Retval Negative errno code if failure.
 * Note   Writes to the specified FIFO of a given SDIO function. Return
 *	value indicates if the transfer succeeded or not.
 */
extern int sdio_writesb(struct sdio_func *func, unsigned int addr,
	void *src, int count);

/**
 * Brief Read a single byte from SDIO function 0
 *
 * Param func     Pointer to an SDIO function of the card.
 * Param addr address to read
 * Param err_ret  optional status value from transfer
 *
 * Return Read value
 * Retval Read value
 * Retval 0xff if failure.
 * Note   Reads a single byte from the address space of SDIO function 0.
 *	If there is a problem reading the address, 0xff is returned
 *	and @err_ret will contain the error code.
 */
extern unsigned char sdio_f0_readb(struct sdio_func *func,
	unsigned int addr, int *err_ret);
/**
 * Brief Write a single byte to SDIO function 0
 *
 * Param dev     Pointer to an SDIO function of the card.
 * Param b byte to write
 * Param addr  address to write to
 * Param err_ret  optional status value from transfer
 *
 * Return none
 * Note   Only writes to the vendor specific CCCR registers (0xF0 -
 *	0xFF) are permiited; @err_ret will be set to -EINVAL for *
 *	writes outside this range.
 */
extern void sdio_f0_writeb(struct sdio_func *func, unsigned char b,
	unsigned int addr, int *err_ret);

/**
 * Brief Get host power management capabilities
 *
 * Param func     Pointer to the SDIO function attached to host.
 *
 * Return Returns a capability bitmask corresponding to power management
 *	features supported by the host controller that the card function
 *	might rely upon during a system suspend. 
 * Retval mmc_pm_flag_t
 * Note   Returns a capability bitmask corresponding to power management
 *	features supported by the host controller that the card function
 *	might rely upon during a system suspend.  The host doesn't need
 *	to be claimed, nor the function active, for this information to be
 *	obtained.
 */
extern mmc_pm_flag_t sdio_get_host_pm_caps(struct sdio_func *func);
/**
 * Brief Set wanted host power management capabilities
 *
 * Param func     Pointer to the SDIO function attached to host.
 * Param flags Numeric identification of the channel to configure
 * Param config  a capability bitmask corresponding to wanted host controller
 *	power management features for the upcoming suspend state
 *
 * Return Standard errno
 * Retval 0 if successful.
 * Retval -EINVAL if failure.
 * Note   Set a capability bitmask corresponding to wanted host controller
 *	power management features for the upcoming suspend state.
 *	This must be called, if needed, each time the suspend method of
 *	the function driver is called, and must contain only bits that
 *	were returned by sdio_get_host_pm_caps().
 *	The host doesn't need to be claimed, nor the function active,
 *	for this information to be set.
 */
extern int sdio_set_host_pm_flags(struct sdio_func *func, mmc_pm_flag_t flags);

#endif /* LINUX_MMC_SDIO_FUNC_H */
