
#ifndef LINUX_SENSOR_H
#define LINUX_SENSOR_H


/****************************************************************************
* 	                                        Include files
****************************************************************************/
#include <linux/ioctl.h>

/****************************************************************************
* 	                                        Macros
****************************************************************************/
#define CAMERA_ALLOC_BIG_MEM		1

#define	CAMERA_PIXEL_WIDTH_MAX		(640)
#define	CAMERA_PIXEL_HEIGHT_MAX		(480)
#define	CAMERA_BYTES_PER_PIXEL		(2)
#define CAMERA_BUFFER_NUM         	(3)
#define SSP_DMA_COUNT_LIMIT			(60 * 1024)

#define CAMERA_PICTURE_SIZE 			(CAMERA_PIXEL_WIDTH_MAX * CAMERA_PIXEL_HEIGHT_MAX * CAMERA_BYTES_PER_PIXEL)
#define CAMERA_DMA_COUNT_PER_PICTURE1		(CAMERA_PICTURE_SIZE / SSP_DMA_COUNT_LIMIT)
#define CAMERA_DMA_COUNT_PER_PICTURE2		(CAMERA_PICTURE_SIZE % SSP_DMA_COUNT_LIMIT ? 1 : 0)
#define CAMERA_DMA_COUNT_PER_PICTURE			(CAMERA_DMA_COUNT_PER_PICTURE1 + CAMERA_DMA_COUNT_PER_PICTURE2)

#define CAMERA_GC0310_PICTURE_SIZE 					CAMERA_PICTURE_SIZE
#define CAMERA_GC0310_DMA_COUNT_PER_PICTURE1		CAMERA_DMA_COUNT_PER_PICTURE1
#define CAMERA_GC0310_DMA_COUNT_PER_PICTURE2		CAMERA_DMA_COUNT_PER_PICTURE2
#define CAMERA_GC0310_DMA_COUNT_PER_PICTURE			CAMERA_DMA_COUNT_PER_PICTURE

#define	CAMERA_GC6133_PIXEL_WIDTH_MAX		(320)
#define	CAMERA_GC6133_PIXEL_HEIGHT_MAX		(240)
#define CAMERA_GC6133_PICTURE_SIZE 					(CAMERA_GC6133_PIXEL_WIDTH_MAX * CAMERA_GC6133_PIXEL_HEIGHT_MAX * CAMERA_BYTES_PER_PIXEL)
#define CAMERA_GC6133_DMA_COUNT_PER_PICTURE1		(CAMERA_GC6133_PICTURE_SIZE / SSP_DMA_COUNT_LIMIT)
#define CAMERA_GC6133_DMA_COUNT_PER_PICTURE2		(CAMERA_GC6133_PICTURE_SIZE % SSP_DMA_COUNT_LIMIT ? 1 : 0)
#define CAMERA_GC6133_DMA_COUNT_PER_PICTURE			(CAMERA_GC6133_DMA_COUNT_PER_PICTURE1 + CAMERA_GC6133_DMA_COUNT_PER_PICTURE2)

#if CAMERA_ALLOC_BIG_MEM
typedef unsigned char*  sensor_buff_addr;
#else
typedef unsigned char* sensor_buff_addr[CAMERA_BUFFER_NUM][CAMERA_DMA_COUNT_PER_PICTURE];
#endif

typedef unsigned int 	u32;
typedef unsigned short 	u16;

/****************************************************************************
* 	                                        Types
****************************************************************************/
enum sensor_id {
	SENSOR_MAIN,	   /* main sensor */
	SENSOR_SUB, 	   /* sub sensor */
	MAX_SENSOR_ID
};

enum sensor_type {
    SENSOR_OV9656,
    SENSOR_OV7670,
    SENSOR_MT9D113,
    SENSOR_MT9V113,
    SENSOR_HI704,
	SENSOR_GC6133,
	SENSOR_GC0310,

    MAX_SENSOR_DEV_TYPE
};

enum sensor_pixel_fmt {
    PIXEL_YCbCr400,
    PIXEL_YCbCr420,
    PIXEL_YUYV,
//    PIXEL_YCbCr422 = PIXEL_YUYV,
//    PIXEL_YCbCr422H = PIXEL_YUYV,
//    PIXEL_YCbCr422V = PIXEL_YUYV,
    PIXEL_YCbCr444,
    PIXEL_RGB_PLANAR,
    PIXEL_RGB555,
    PIXEL_RGB565,
    PIXEL_RGB666,
    PIXEL_RGB888,
    PIXEL_ARGB888,
    
    MAX_PIXEL_FORMAT
};

enum sensor_buff_nr
{
    FRAME_BUFFER0,
    FRAME_BUFFER1,
    FRAME_BUFFER2,
    FRAME_BUFFERNUM
};

enum sensor_state {
	SENSOR_OPEN= 0x0,
	SENSOR_START = 0x1,
	SENSOR_STANDBY = 0x2,
	SENSOR_PREP_CLOSE = 0x3,
	SENSOR_CLOSE = 0x4,
	MAX_SENSOR_STATE
};

enum sensor_output_size {
    OUTPUT_IMG_SXGA =0,
    OUTPUT_IMG_VGA,
    OUTPUT_IMG_QVGA,
    OUTPUT_IMG_QQVGA,
    OUTPUT_IMG_CIF ,
    OUTPUT_IMG_QCIF,
    OUTPUT_IMG_QQCIF ,
    OUTPUT_IMG_USVGA,
    MAX_OUTPUT_IMG_SIZE
};

enum sensor_night_level {
    NIGHT_LEVEL_0,             /* level 0*/
    NIGHT_LEVEL_1,             /* level 1 */
    NIGHT_LEVEL_2,             /* level 2 */    
    MAX_NIGHT_LEVEL
};

enum sensor_zoom_level {
    ZOOM_LEVEL_0,             /* level 0  */
    ZOOM_LEVEL_1,             /* level 1 */
    ZOOM_LEVEL_2,             /* level 2 */
    ZOOM_LEVEL_3,             /* level 3 */
    ZOOM_LEVEL_4,             /* level 4 */
    MAX_ZOOM_LEVEL
};

enum sensor_bright_level {
    BRI_LEVEL_0 = 0,         /* level 0 (darkest) */
    BRI_LEVEL_1,             /* level 1 */
    BRI_LEVEL_2,             /* level 2 */
    BRI_LEVEL_3,             /* level 3 */
    BRI_LEVEL_4,             /* level 4  */
    BRI_LEVEL_5,			 /* level 5  */
    BRI_LEVEL_6,			 /* level 6  */
    MAX_BRI_LEVEL
};

enum sensor_contrast_level {
    CON_LEVEL_0 = 0,       /* level 0 (thinnest) */
    CON_LEVEL_1,           /* level 1 */
    CON_LEVEL_2,           /* level 2 */
    CON_LEVEL_3,           /* level 3 */
    CON_LEVEL_4,           /* level 4 */
    CON_LEVEL_5,           /* level 5 */
    CON_LEVEL_6,           /* level 6 (sharpest) */
    MAX_CON_LEVEL
};

enum sensor_saturation_level {
    SAT_LEVEL_0,
    SAT_LEVEL_1,
    SAT_LEVEL_2,
    SAT_LEVEL_3,
    SAT_LEVEL_4,
    
    MAX_SAT_LEVEL
};

enum sensor_effect_mode {
    EFF_NORMAL = 0,
    EFF_ANTIQUE,
    EFF_BLUISH,
    EFF_GREENISH,
    EFF_REDISH,
    EFF_YELLOW,
    EFF_NEGATIVE,
    EFF_BLACK_AND_WHITE,
    EFF_BW_NEGATIVE,
    EFF_SEPIA_GREEN,
    MAX_EFF_TYPE
};

enum sensor_white_balance_mode {
    WB_AUTO = 0,
    WB_INSCANDSENCE,
    WB_FLUORESCENT,//CWF
   // WB_FLUORECWF, //cool white FLUORESCENT
    WB_TUNGSTEN,
    WB_SUNNY,    
    WB_CLOUDY,

    MAX_WB_MODE
};

enum sensor_light_mode {
    LIGTH_AUTO,     /* auto */
    LIGTH_SUNNY,    /* sunny */
    LIGTH_CLOUDY,   /* cloudy */
    LIGTH_OFFICE,   /* office */
    LIGTH_HOME,     /* home */
    LIGTH_NIGHT,    /* night */

    MAX_LIGTH_MODE
};

enum sensor_expo_level {
    EV_NEG_4_3 = 0,           /*  EV -2    */
    EV_NEG_3_3,           	  /*  EV -1.5 */
    EV_NEG_2_3,            	  /*  EV -1    */
    EV_NEG_1_3,            	  /*  EV -0.5 */
    EV_ZERO,            	  /*  EV 0      */
    EV_POS_1_3,            	  /*  EV +0.5 */
	EV_POS_2_3, 			  /*  EV +1    */
	EV_POS_3_3,            	  /*  EV +1.5 */
	EV_POS_4_3,            	  /*  EV +2 */

    MAX_EV_LEVEL
};

enum sensor_data_fmt {
    HAL_CAM_WORD_MODE_YUV,
    HAL_CAM_WORD_MODE_RGB,
    
    MAX_HAL_CAM_WORD_MODE
};

enum sensor_mirror_mode {
    MIR_NONE,
    MIR_V,
    MIR_H,
    MIR_V_H,
    
    MAX_MIR_TYPE
};

enum sensor_antiflicker_mode {
    FLICKER_50HZ,
	FLICKER_60HZ,
   
    MAX_FLICKER_TYPE
};

enum sensor_frame_rate {
    fps714  = 0,
    fps75      ,
    fps143     ,
    fps15      ,
    fps25      ,
    fps30      ,
   
    MAX_FPS_SIZE
};

struct sensor_yuv_buff {
   u32 *y;
   u32 *cb;
   u32 *cr;  
};

union sensor_image_data{
    struct sensor_yuv_buff yuv;
    u32* stream_databuf;
    u32* argb_databuf;
};

struct sensor_dev_info {
	enum sensor_type 	dev_type;	    /* the sensor now using*/
	enum sensor_id 		active_sensor; 	/* the sensor now using*/
	u16					width;	  		/* max pixel width of the actived sensor */
	u16					height;	  /* max pixel height of the actived sensor */
	u16         		pixel_bytes;
	int 				fuc_set_zoom;            /*whether to support set_zoom function */
	int 				fuc_set_imgsize; 		  /*whether to support set_img_Size function */
	int 				fuc_set_effect;          /*whether to support set_effect function */
	int 				fuc_set_mirrorflip;		/*whether to support set MirrorFlip function */
	int 				fuc_set_brightness;		/*whether to support set Brightness function */
	int 				fuc_set_contrast;		/*whether to support SetContrast function */
	int 				fuc_set_saturation;		/*whether to support SetSaturation function */
	int 				fuc_set_whitebalance;	/*whether to support SetWhiteBalance function */
	int 				fuc_set_lightmode;		/*whether to support SetLightMode function */
	int 				fuc_set_nightmode;		/*whether to support SetNightMode function */
	int 				fuc_set_framerate;		/*whether to support SetFrameRate function */
	int            	fuc_set_antiflicker; 
};

struct sensor_data_info {
	u32 width;
	u32 height;
	enum sensor_pixel_fmt fmt;
	enum sensor_buff_nr nr;
	union sensor_image_data buff;
};

struct sensor_yuv_buff_size {
    unsigned int     y_size;
    unsigned int     u_size;
    unsigned int     v_size;
};

union sensor_image_buff_size{
    unsigned int   stream_buff_size;
    unsigned int   argb_buff_size;
    struct sensor_yuv_buff_size buff_size;
};


#define IOCTL_CAM_START	_IO('c', 1)
#define IOCTL_CAM_STOP	_IO('c', 2)
#define IOCTL_CAM_SENSOR_GET_INFO	_IOR('c', 3,struct sensor_dev_info)
#define IOCTL_CAM_SET_EFFECT _IOW('c',4,enum sensor_effect_mode)
#define IOCTL_CAM_SET_MIRRORFLIP _IOW('c',5,enum sensor_mirror_mode)
#define IOCTL_CAM_SET_BRIGHTNESS _IOW('c',6,enum sensor_bright_level)
#define IOCTL_CAM_SET_CONTRAST _IOW('c',7,enum sensor_contrast_level)
#define IOCTL_CAM_SET_WHITE_BALANCE _IOW('c',8,enum sensor_white_balance_mode)
#define IOCTL_CAM_GET_ONEFRAME _IOR('c',9,struct sensor_data_info)
#define IOCTL_CAM_FREE_FRAME_BUFFER _IOW('c',10,enum sensor_buff_nr)
#define IOCTL_CAM_SET_ANTIFLICKER _IOW('c',11,enum sensor_antiflicker_mode)
#define IOCTL_CAM_SET_PREVIEW _IO('c',12)
#define IOCTL_CAM_SET_CAPTURE _IO('c',13)

#endif /*LINUX_SENSOR_H*/

