/* vi: set sw=4 ts=4: */
/*
 * Poweroff reboot and halt, oh my.
 *
 * Copyright 2006 by Rob Landley <rob@landley.net>
 *
 * Licensed under GPLv2, see file LICENSE in this source tree.
 */

//applet:IF_HALT(APPLET(halt, BB_DIR_SBIN, BB_SUID_DROP))
//applet:IF_HALT(APPLET_ODDNAME(poweroff, halt, BB_DIR_SBIN, BB_SUID_DROP, poweroff))
//applet:IF_HALT(APPLET_ODDNAME(reboot, halt, BB_DIR_SBIN, BB_SUID_DROP, reboot))

//kbuild:lib-$(CONFIG_HALT) += halt.o

//config:config HALT
//config:	bool "poweroff, halt, and reboot"
//config:	default y
//config:	help
//config:	  Stop all processes and either halt, reboot, or power off the system.
//config:
//config:config FEATURE_CALL_TELINIT
//config:	bool "Call telinit on shutdown and reboot"
//config:	default y
//config:	depends on HALT && !INIT
//config:	help
//config:	  Call an external program (normally telinit) to facilitate
//config:	  a switch to a proper runlevel.
//config:
//config:	  This option is only available if you selected halt and friends,
//config:	  but did not select init.
//config:
//config:config TELINIT_PATH
//config:	string "Path to telinit executable"
//config:	default "/sbin/telinit"
//config:	depends on FEATURE_CALL_TELINIT
//config:	help
//config:	  When busybox halt and friends have to call external telinit
//config:	  to facilitate proper shutdown, this path is to be used when
//config:	  locating telinit executable.

//usage:#define halt_trivial_usage
//usage:       "[-d DELAY] [-n] [-f]" IF_FEATURE_WTMP(" [-w]")
//usage:#define halt_full_usage "\n\n"
//usage:       "Halt the system\n"
//usage:     "\n	-d SEC	Delay interval"
//usage:     "\n	-n	Do not sync"
//usage:     "\n	-f	Force (don't go through init)"
//usage:	IF_FEATURE_WTMP(
//usage:     "\n	-w	Only write a wtmp record"
//usage:	)
//usage:
//usage:#define poweroff_trivial_usage
//usage:       "[-d DELAY] [-n] [-f]"
//usage:#define poweroff_full_usage "\n\n"
//usage:       "Halt and shut off power\n"
//usage:     "\n	-d SEC	Delay interval"
//usage:     "\n	-n	Do not sync"
//usage:     "\n	-f	Force (don't go through init)"
//usage:
//usage:#define reboot_trivial_usage
//usage:       "[-d DELAY] [-n] [-f]"
//usage:#define reboot_full_usage "\n\n"
//usage:       "Reboot the system\n"
//usage:     "\n	-d SEC	Delay interval"
//usage:     "\n	-n	Do not sync"
//usage:     "\n	-f	Force (don't go through init)"

#include "libbb.h"
#include "reboot.h"

#if ENABLE_FEATURE_WTMP
#include <sys/utsname.h>

static void write_wtmp(void)
{
	struct utmp utmp;
	struct utsname uts;
	/* "man utmp" says wtmp file should *not* be created automagically */
	/*if (access(bb_path_wtmp_file, R_OK|W_OK) == -1) {
		close(creat(bb_path_wtmp_file, 0664));
	}*/
	memset(&utmp, 0, sizeof(utmp));
	utmp.ut_tv.tv_sec = time(NULL);
	strcpy(utmp.ut_user, "shutdown"); /* it is wide enough */
	utmp.ut_type = RUN_LVL;
	utmp.ut_id[0] = '~'; utmp.ut_id[1] = '~'; /* = strcpy(utmp.ut_id, "~~"); */
	utmp.ut_line[0] = '~'; utmp.ut_line[1] = '~'; /* = strcpy(utmp.ut_line, "~~"); */
	uname(&uts);
	safe_strncpy(utmp.ut_host, uts.release, sizeof(utmp.ut_host));
	updwtmp(bb_path_wtmp_file, &utmp);
}
#else
#define write_wtmp() ((void)0)
#endif

/* Started by AICoder, pid:eecb340c7fw298714a9d0b3210b0288be1203207 */
static void get_app_name_by_pid(int pid, char *app_name, int app_name_len)
{
	char file_comm[256];
	FILE *pfile;
	size_t len;
	
	memset(file_comm, 0, sizeof(file_comm));
	snprintf(file_comm, sizeof(file_comm), "/proc/%d/comm", pid);

	pfile = fopen(file_comm, "r");
	if (pfile)
	{
		memset(app_name, 0, app_name_len);
		fgets(app_name, app_name_len, pfile);
		app_name[app_name_len-1] = '\0';
		app_name[strlen(app_name) - 1] = '\0'; //last byte is \n
		fclose(pfile);
	}
}

static int get_ppid(int pid) {
    char path[256];
    snprintf(path, sizeof(path), "/proc/%d/stat", pid);
    
    FILE *fp = fopen(path, "r");
    if (fp == NULL) {
        perror("fopen");
        return -1;
    }
    
    int ppid = -1;
    // 通过解析第4列（ppid）来获取父进程ID
    fscanf(fp, "%*d %*s %*c %d", &ppid);
    fclose(fp);
    return ppid;
}

static int get_reboot_caller(char *applet_name)
{
	int pid = get_ppid(getpid());
	char app_name[32];
	int app_name_len = sizeof(app_name);
	char cmd_str[256];
	int try_cnt = 0;

	while(1)
	{
		if (try_cnt > 5) {
			strcpy(app_name, "unkown");
			break;
		}
		try_cnt++;
		if (pid == 1) {
			get_app_name_by_pid(pid, app_name, app_name_len);
			break; //init
		}
		get_app_name_by_pid(pid, app_name, app_name_len);
		if ((strcmp(app_name, "sh") == 0) || (strcmp(app_name, "bash") == 0)) {
			//printf("shell %s continue %d\n", app_name, strlen(app_name));
			pid = get_ppid(pid); //sh continue
		} else {
			//printf("not sh break %s %d\n", app_name, strlen(app_name));
			break; //not sh
		}
	}

	memset(cmd_str, 0, sizeof(cmd_str));
	snprintf(cmd_str, sizeof(cmd_str), "nv set ap_reset_app=%s_%d; nv save", app_name, pid);
	system(cmd_str);
	printf("call %s by %s(%d)\n", applet_name, app_name, pid);

	return 0;
}
/* Ended by AICoder, pid:eecb340c7fw298714a9d0b3210b0288be1203207 */

int halt_main(int argc, char **argv) MAIN_EXTERNALLY_VISIBLE;
int halt_main(int argc UNUSED_PARAM, char **argv)
{
	static const int magic[] = {
		RB_HALT_SYSTEM,
		RB_POWER_OFF,
		RB_AUTOBOOT
	};
	static const smallint signals[] = { SIGUSR1, SIGUSR2, SIGTERM };

	int delay = 0;
	int which, flags, rc;

	/* Figure out which applet we're running */
	for (which = 0; "hpr"[which] != applet_name[0]; which++)
		continue;
#if ENABLE_INSTALL_TO_RECOVERY
#else
	get_reboot_caller(applet_name); //add by zxic, print parent proccess name and pid
#endif

	/* Parse and handle arguments */
	opt_complementary = "d+"; /* -d N */
	/* We support -w even if !ENABLE_FEATURE_WTMP,
	 * in order to not break scripts.
	 * -i (shut down network interfaces) is ignored.
	 */
	flags = getopt32(argv, "d:nfwi", &delay);

	sleep(delay);

	write_wtmp();

	if (flags & 8) /* -w */
		return EXIT_SUCCESS;

	if (!(flags & 2)) /* no -n */
		sync();

	/* Perform action. */
	rc = 1;
	if (!(flags & 4)) { /* no -f */
//TODO: I tend to think that signalling linuxrc is wrong
// pity original author didn't comment on it...
		if (ENABLE_FEATURE_INITRD) {
			/* talk to linuxrc */
			/* bbox init/linuxrc assumed */
			pid_t *pidlist = find_pid_by_name("linuxrc");
			if (pidlist[0] > 0)
				rc = kill(pidlist[0], signals[which]);
			if (ENABLE_FEATURE_CLEAN_UP)
				free(pidlist);
		}
		if (rc) {
			/* talk to init */
			if (!ENABLE_FEATURE_CALL_TELINIT) {
				/* bbox init assumed */
				rc = kill(1, signals[which]);
			} else {
				/* SysV style init assumed */
				/* runlevels:
				 * 0 == shutdown
				 * 6 == reboot */
				execlp(CONFIG_TELINIT_PATH,
						CONFIG_TELINIT_PATH,
						which == 2 ? "6" : "0",
						(char *)NULL
				);
				bb_perror_msg_and_die("can't execute '%s'",
						CONFIG_TELINIT_PATH);
			}
		}
	} else {
		rc = reboot(magic[which]);
	}

	if (rc)
		bb_perror_nomsg_and_die();
	return rc;
}
