#ifdef LED_SUPPORT
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <binder/Parcel.h>
#include <sys/socket.h>
#include <errno.h>
#include <unistd.h>
#include <cutils/jstring.h>
#include <sys/types.h>
#include <time.h>
#include <signal.h>
#include "led.h"
#include "led_inner.h"
#include <liblog/lynq_deflog.h>
#include "common.h"
#include "libled/lynq_led.h"
#include <include/lynq_uci.h>

#undef LOG_TAG
#define LOG_TAG "LED"


/*************************************************************
    Constants and Macros
*************************************************************/
#define GPIO_NETLED_CONNECT_REFLASH_TIMER   200
#define GPIO_NETLED_CREG_REFLASH_TIMER   800
#ifndef MOBILETEK_TARGET_PLATFORM_T106
#define MBTK_GPIO_NETLED_N            77
#else
#define MBTK_GPIO_LED  100
#define MBTK_GPIO_NETLED_N 92
#define MBTK_GPIO_STATUSLED_N 127
#endif

/*************************************************************
    Extern Function Declaration
*************************************************************/
void mbtk_netled_reflash_handle();

/*************************************************************
    Variables:local,extern
*************************************************************/
//static rex_timer_type mbtk_gpio_netled_timer;
static mbtk_netled_state_t  mbtk_netled_state;
/*************************************************************
    Definitions:enum,struct,union
*************************************************************/

/*=============================================
FUNCTION
    mbtk_get_netled_state
    
DESCRIPTION
    This function to return pointer of mbtk_netled_state.

DEPENDENCIES
    None

PARAMETERS
    None

RETURN VALUE
    &mbtk_netled_state

SIDE EFFECTS
    None
=============================================*/

mbtk_netled_state_t * mbtk_get_netled_state()
{
    return &mbtk_netled_state;
}

/*=============================================
FUNCTION
    mbtk_init_netled_state
    
DESCRIPTION
    This function to init mbtk_netled_state.

DEPENDENCIES
    None

PARAMETERS
    None

RETURN VALUE
    None

SIDE EFFECTS
    None
=============================================*/

void mbtk_init_netled_state()
{
    mbtk_netled_state_t * netled_state = NULL;
    netled_state = mbtk_get_netled_state();
    
    netled_state->gpio_netled_cs_flag = false;
    netled_state->gpio_netled_goingsleep_flag = false;
    netled_state->gpio_netled_ppp_flag = false;
    netled_state->gpio_netled_wwan_flag = false;
    
}

static timer_t s_mbtk_gpio_netled_timer;

static int s_mbtk_gpio_netled_timer_sig_value = 11;  /* 2,3,4 are used by network */

void start_led_timer(timer_t timer, int signal_value, int milliseconds) 
{
    RLOGD("start_led_timer(), timer_id=%ld, signal_value=%d, time=%d",(long)timer, signal_value, milliseconds);

    struct itimerspec expire;
    expire.it_interval.tv_sec = 0;
    expire.it_interval.tv_nsec = 0;
    expire.it_value.tv_sec = milliseconds/1000;
    expire.it_value.tv_nsec = (milliseconds%1000)*1000000;
	if (timer_settime(timer, 0, &expire, NULL) == -1) {
        RLOGE("timer_settime  failed reason=[%s]", strerror(errno));
    }
    
}

void stop_led_timer(timer_t timer, int signal_value) {
    RLOGD("stop_led_timer(), timer_id=%ld, signal_value=%d", (long)timer, signal_value);
    struct itimerspec timespec;
    if(timer_gettime(timer, &timespec) == -1) {
        RLOGE("stop_led_timer(), get time fail(%s)", strerror(errno));
        return;
    }
    RLOGD("stop_led_timer(), tv_sec=%ld, tv_nsec=%ld",timespec.it_value.tv_sec, timespec.it_value.tv_nsec);
    if((timespec.it_value.tv_sec == 0)  && (timespec.it_value.tv_nsec == 0) ) {
        RLOGD("stop_led_timer(), timer_id(%ld) had stopped, just return", (long)timer);
        return;
    } else {
        start_led_timer(timer, signal_value, 0);
    }
}

static pthread_mutex_t mtx = PTHREAD_MUTEX_INITIALIZER;
void led_timer_handler(sigval_t sig) 
{    
    RLOGD("led_timer_handler, sig_value: %d", sig.sival_int);
    int s;   
    s = pthread_mutex_lock(&mtx);
    if(s != 0) {
        RLOGE("led_timer_handler, pthead_mutex_lock fail");
    }   
    if(sig.sival_int == s_mbtk_gpio_netled_timer_sig_value) 
    {
        mbtk_netled_reflash_handle();
    }    
    s = pthread_mutex_unlock(&mtx);
    if(s != 0) {
        RLOGE("led_timer_handler, pthead_mutex_unlock fail");
    }   
}

void init_led_timer(timer_t* timer, int signal_value) 
{
    struct sigevent sevp;
    memset(&sevp, 0, sizeof(sevp));
    sevp.sigev_value.sival_int = signal_value;
    sevp.sigev_notify = SIGEV_THREAD;
    sevp.sigev_notify_function = led_timer_handler;

    if(timer_create(CLOCK_MONOTONIC, &sevp, timer) == -1) {
        RLOGE("init_led_timer()  failed reason=[%s]", strerror(errno));
    }
    RLOGD("init_led_timer(), timer_Id = %ld, signal_value=%d", (long)(*timer), signal_value);
}

void init_led_timer_all() 
{
    init_led_timer(&s_mbtk_gpio_netled_timer,s_mbtk_gpio_netled_timer_sig_value);   
}

void deinit_led_timer_all() 
{
    stop_led_timer(s_mbtk_gpio_netled_timer,s_mbtk_gpio_netled_timer_sig_value);    
}

/*=============================================
FUNCTION
    mbtk_netled_init
    
DESCRIPTION
    This function to init timer

DEPENDENCIES
    None

PARAMETERS
    param    not used

RETURN VALUE
    None

SIDE EFFECTS
    None
=============================================*/
void mbtk_netled_init()
{
    mbtk_init_netled_state();
    init_led_timer_all();   
}

/*=============================================
FUNCTION
    mbtk_netled_deinit
    
DESCRIPTION
    This function to init timer

DEPENDENCIES
    None

PARAMETERS
    param    not used

RETURN VALUE
    None

SIDE EFFECTS
    None
=============================================*/
void mbtk_netled_deinit()
{
    deinit_led_timer_all();   
}


/*=============================================
FUNCTION
    mbtk_get_at_netled_cmd
    
DESCRIPTION
    This function to return netled switch value.

DEPENDENCIES
    None

PARAMETERS
    None

RETURN VALUE
    int

SIDE EFFECTS
    None
=============================================*/

int mbtk_get_at_netled_cmd(void)
{
    return 1;
//    return (int) dsatutil_get_val(MBTK_AT_GPIO_CGNETLED_IDX,0,0,NUM_TYPE);
}

/*=============================================
FUNCTION
    mbtk_get_reg_net_status
    
DESCRIPTION
    This function that determine if net registers successfully.
    
DEPENDENCIES
    None

PARAMETERS
    param    not used

RETURN VALUE
    mbtk_netled_reg_net_status

SIDE EFFECTS
    None
=============================================*/

mbtk_netled_reg_net_status  mbtk_get_reg_net_status(void)
{
    if(false  == ril_get_if_insert_simcard())
    {
        RLOGE("ril_get_if_insert_simcard false" );
        return NET_REG_FAIL;
    }
   
    if(true == ril_get_if_3gpp_reg_success())
    {
        RLOGD("ril_get_if_3gpp_reg_success true");
        return NET_REG_SUCCESS;
    }

    if(true == ril_get_if_3gpp_data_reg_success())
    {
        RLOGD("ril_get_if_3gpp_data_reg_success true");
        return NET_REG_SUCCESS;
    }	

    return NET_REG_FAIL;
}

/*=============================================
FUNCTION
    mbtk_netled_get_socket_conn_status
    
DESCRIPTION
    This function that socket connects successfully.

DEPENDENCIES
    None

PARAMETERS
    param    not used

RETURN VALUE
    mbtk_netled_socket_conn_status

SIDE EFFECTS
    None
=============================================*/

mbtk_netled_socket_conn_status  mbtk_netled_get_socket_conn_status(void)
{
    mbtk_netled_socket_conn_status  socket_conn_status = MBTK_SOCKET_DISCONNECT; 
    mbtk_netled_state_t * netled_state =  mbtk_get_netled_state();    

    if(netled_state->gpio_netled_ppp_flag || netled_state->gpio_netled_wwan_flag)
    {
        RLOGD("GPIO: ppp wwan netled state sockNum %d,%d",
        netled_state->gpio_netled_ppp_flag, netled_state->gpio_netled_wwan_flag);
        
        socket_conn_status = MBTK_SOCKET_CONNECTED;
        return socket_conn_status;
    }   

    return socket_conn_status;

}

/*=============================================
FUNCTION
    mbtk_netled_get_current_state
    
DESCRIPTION
    get the netled status .

DEPENDENCIES
    None

PARAMETERS
    param    not used

RETURN VALUE
    mbtk_netled_status_type

SIDE EFFECTS
    None
=============================================*/
mbtk_netled_status_type mbtk_netled_get_current_state(void)
{
    mbtk_netled_reg_net_status reg_net_status = NET_REG_FAIL;
    mbtk_netled_status_type netled_status_type = GPIO_NETLED_OFF;
    mbtk_netled_socket_conn_status socket_conn_status = MBTK_SOCKET_DISCONNECT;
    int mbtk_netled_at_cmd_value = mbtk_get_at_netled_cmd();  //  0 -- close netled / 1 -- open netled
    mbtk_netled_state_t * netled_state =  mbtk_get_netled_state();
    
    if(mbtk_netled_at_cmd_value == 0 || 
       netled_state->gpio_netled_goingsleep_flag == true)
    {
        netled_status_type = GPIO_NETLED_OFF;
        return netled_status_type;
    }

    reg_net_status = mbtk_get_reg_net_status();
    if( NET_REG_FAIL == reg_net_status || true == netled_state->gpio_netled_cs_flag )
    {
        netled_status_type = GPIO_NETLED_LIGHT;
    }
    else if( NET_REG_SUCCESS == reg_net_status )
    {
        socket_conn_status = mbtk_netled_get_socket_conn_status();
        
        if( MBTK_SOCKET_CONNECTED == socket_conn_status )
        {
            netled_status_type = GPIO_NETLED_CONNECT;
        }
        else if( MBTK_SOCKET_DISCONNECT == socket_conn_status )
        {
            netled_status_type = GPIO_NETLED_REG;
        }
    } 
    return netled_status_type;
}



bool is_mbtk_timer_finish() {
    struct itimerspec timespec;
    if(timer_gettime(s_mbtk_gpio_netled_timer, &timespec) == -1) {
        RLOGD("%s(), get time fail(%s)", __FUNCTION__, strerror(errno));
        return true;
    }
    RLOGD("%s(), tv_sec=%ld, tv_nsec=%ld", __FUNCTION__,timespec.it_value.tv_sec, timespec.it_value.tv_nsec);
    if((timespec.it_value.tv_sec == 0)  && (timespec.it_value.tv_nsec == 0) ) {
        RLOGD("%s(), timer_id(%ld) had stopped", __FUNCTION__, (long)s_mbtk_gpio_netled_timer);
        return true;
    }
	return false;
}

/*=============================================
FUNCTION
    mbtk_netled_state_update
    
DESCRIPTION
    This function to reflash network led status and write gpio value

DEPENDENCIES
    None

PARAMETERS
    param     flag

RETURN VALUE
    None

SIDE EFFECTS
    None
=============================================*/
void mbtk_netled_state_update(mbtk_netled_reflash_type flag)
{
//  rex_timer_cnt_type prev_value;
    mbtk_netled_state_t * netled_state = NULL;
    netled_state = mbtk_get_netled_state();
    
    switch (flag)
    {
        case GPIO_NETLED_REFLASH_NORMAL:
             break;
        case GPIO_NETLED_CS_CALLING:
             netled_state->gpio_netled_cs_flag = true;
             break;
        case GPIO_NETLED_PPP_CONNECT:
             netled_state->gpio_netled_ppp_flag = true;
             break;
        case GPIO_NETLED_PPP_CLOSED:
             netled_state->gpio_netled_ppp_flag = false;
             break;
        case GPIO_NETLED_CS_CALLEND:
             netled_state->gpio_netled_cs_flag = false;
             break;
        case GPIO_NETLED_AP_GOINGSLEEP:
             netled_state->gpio_netled_goingsleep_flag = true;
             break;
        case GPIO_NETLED_AP_WAKEUP:
             netled_state->gpio_netled_goingsleep_flag = false;
             break;
        case GPIO_NETLED_WWAN_CONNECT:
             netled_state->gpio_netled_wwan_flag = true;
             break;
        case GPIO_NETLED_WWAN_CLOSED:
             netled_state->gpio_netled_wwan_flag = false;
             break;

    }

    //prev_value = rex_get_timer(&mbtk_gpio_netled_timer);
//    prev_value = timer_get_64(&mbtk_gpio_netled_timer, T_MSEC);
//    if( prev_value == 0)
	if((flag== GPIO_NETLED_AP_GOINGSLEEP) || is_mbtk_timer_finish())
    {
 //     MBTK_MSG1_HIGH("GPIO: rex_get_timer prev_value =%d",prev_value);
       //rex_set_timer(&mbtk_gpio_netled_timer, 1);
       //timer_set_64(&mbtk_gpio_netled_timer,1,0,T_MSEC);
        start_led_timer(s_mbtk_gpio_netled_timer, s_mbtk_gpio_netled_timer_sig_value, 1);        
    }

    return ;
}

/*=============================================
FUNCTION
    mbtk_netled_reflash_handle
    
DESCRIPTION
    This function to reflash network led status and write gpio value

DEPENDENCIES
    None

PARAMETERS
    param    not used

RETURN VALUE
    None

SIDE EFFECTS
    None
=============================================*/
#define LYNQ_UCI_SECTION "lynq_led"
#define LYNQ_UCI_NETLED_KEY "lynq_netled_on"
#define LED_ON 4

void mbtk_gpio_write_output(int gpio_port, int value)
{
#ifdef MOBILETEK_TARGET_PLATFORM_T106
    char com_exp[MBTK_GPIO_LED] = "";
    char com_dir[MBTK_GPIO_LED] = "";
    char com_val[MBTK_GPIO_LED] = "";
    char led_path[MBTK_GPIO_LED] = "";
    char netled_on[LED_ON] = "";
    lynq_get_value(LYNQ_UCI_FILE, LYNQ_UCI_SECTION, LYNQ_UCI_NETLED_KEY, netled_on);
    if(!atoi(netled_on)){
        system("echo 0 > /sys/class/gpio/gpio92/value");
        return ;
    }
    sprintf(com_exp, "echo %d > /sys/class/gpio/export", gpio_port);
    sprintf(com_dir, "echo out > /sys/class/gpio/gpio%d/direction", gpio_port);
    sprintf(com_val, "echo %d > /sys/class/gpio/gpio%d/value", value, gpio_port);
    sprintf(led_path, "/sys/class/gpio/gpio%d", gpio_port);

    if (access(led_path, F_OK) != 0) {
           system(com_exp);//92 net_Status,127 status
       }
    system(com_dir);
    system(com_val);
#else
    if(value==1)
    {
        system("echo 255 > /sys/class/leds/led9515:green:cellular-radio/brightness");
    }  
	else 
	{
	    system("echo 0 > /sys/class/leds/led9515:green:cellular-radio/brightness");
	}
#endif
}

/*=============================================
FUNCTION
    mbtk_netled_reflash_handle
    
DESCRIPTION
    This function to reflash network led status and write gpio value

DEPENDENCIES
    None

PARAMETERS
    param    not used

RETURN VALUE
    None

SIDE EFFECTS
    None
=============================================*/
void mbtk_netled_reflash_handle()
{
    mbtk_netled_status_type state = GPIO_NETLED_MAX;
    state = mbtk_netled_get_current_state();
    RLOGD("mbtk_netled_get_current_state  state = %d",state);
    static int led_val = 0;

    switch(state)
    {
        case GPIO_NETLED_OFF:
             {
                 if( 0 != led_val )
                 {
                     led_val = 0;
                     mbtk_gpio_write_output(MBTK_GPIO_NETLED_N, 0);
                 }
             }
             break;
        case GPIO_NETLED_REG:
             {
                led_val = ( (1 == led_val)? 0:1 );
                mbtk_gpio_write_output(MBTK_GPIO_NETLED_N, led_val);
                start_led_timer(s_mbtk_gpio_netled_timer, s_mbtk_gpio_netled_timer_sig_value, GPIO_NETLED_CREG_REFLASH_TIMER);
				
              //  timer_set_64(&mbtk_gpio_netled_timer,GPIO_NETLED_CREG_REFLASH_TIMER,0,T_MSEC);
                //rex_set_timer(&mbtk_gpio_netled_timer, GPIO_NETLED_CREG_REFLASH_TIMER);
             }
             break;
        case GPIO_NETLED_CONNECT:
             {
                 led_val = ( (1 == led_val)? 0:1 );
                 mbtk_gpio_write_output(MBTK_GPIO_NETLED_N, led_val);
        		 start_led_timer(s_mbtk_gpio_netled_timer, s_mbtk_gpio_netled_timer_sig_value, GPIO_NETLED_CONNECT_REFLASH_TIMER);
                // timer_set_64(&mbtk_gpio_netled_timer,GPIO_NETLED_CONNECT_REFLASH_TIMER,0,T_MSEC);
                 //rex_set_timer(&mbtk_gpio_netled_timer, GPIO_NETLED_CONNECT_REFLASH_TIMER);
             }
             break;
        case GPIO_NETLED_LIGHT:
             {
                 if( 1 != led_val )
                 {
                     led_val = 1;
                     mbtk_gpio_write_output(MBTK_GPIO_NETLED_N, 1);
                 }
             }
             break;
        default:
             RLOGE("GPIO: Unkown netled state !");
    }
}
#endif
