#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdint.h>
#include <log/log.h>
#include "lynq_monitor.h"
#include "lynq_monitor_internal.h"
#include "lynq_monitor_cfg.h"
#include "lynq_monitor_api.h"
#include "lynq_monitor_ping.h"
#include "lynq_monitor_nw.h"
#include "liblog/lynq_deflog.h"

#define USER_LOG_TAG "LYNQ_MONITOR"

static MonitorExceptionHandler s_ExceptionHandler = NULL; /**< 全局异常处理回调函数指针 */
int g_monitor_state = 0; /**< 全局状态 */
Lynq_Monitor_Config g_monitor_config = {0}; /**< 全局配置项 */

static void lynq_monitor_config_print(const Lynq_Monitor_Config* config)
{
    LYINFLOG("Lynq Monitor Config:\n");
    LYINFLOG("  Network Monitor Enabled: %d\n", config->network_monitor_enabled);
    LYINFLOG("  Device Monitor Enabled: %d\n", config->device_monitor_enabled);
    LYINFLOG("  Ping Monitor Enabled: %d\n", config->ping_monitor_enabled);
    LYINFLOG("  NW Monitor Period: %d\n", config->nw_monitor_period);
    LYINFLOG("  Ping Monitor Period: %d\n", config->ping_monitor_period);
    LYINFLOG("  Limit Reset Nad: %d\n", config->limit_reset_nad_1);
    LYINFLOG("  Limit Reset NW: %d\n", config->limit_reset_nw_1);
    LYINFLOG("  Limit Reset Data: %d\n", config->limit_reset_data_1);
    LYINFLOG("  Limit Reset Nad: %d\n", config->limit_reset_nad_2);
    LYINFLOG("  Limit Reset NW: %d\n", config->limit_reset_nw_2);
    LYINFLOG("  Limit Reset Data: %d\n", config->limit_reset_data_2);
    LYINFLOG("  Opt Reset Nad: %d\n", config->opt_reset_nad);
    LYINFLOG("  Opt Reset NW: %d\n", config->opt_reset_nw);
    LYINFLOG("  Opt Reset Data: %d\n", config->opt_reset_data);

    LYINFLOG("  Ping addr1: %s\n", config->ping_addr1);
    LYINFLOG("  Ping addr2: %s\n", config->ping_addr2);
    LYINFLOG("  Ping addr3: %s\n", config->ping_addr3);
}
static void lynq_monitor_config_init()
{
    g_monitor_config.network_monitor_enabled = lynq_cfg_get_i("network_monitor_enabled", 1);
    g_monitor_config.device_monitor_enabled = lynq_cfg_get_i("device_monitor_enabled", 1);
    g_monitor_config.ping_monitor_enabled = lynq_cfg_get_i("ping_monitor_enabled", 1);
    g_monitor_config.nw_monitor_period = lynq_cfg_get_i("nw_monitor_period", DEFAULT_NW_MONITOR_PERIOD);
    g_monitor_config.ping_monitor_period = lynq_cfg_get_i("ping_monitor_period", DEFAULT_PING_MONITOR_PERIOD);
    g_monitor_config.limit_reset_nad_1 = lynq_cfg_get_i("limit_reset_nad_1", LIMIT_RESET_NAD_1);
    g_monitor_config.limit_reset_nw_1 = lynq_cfg_get_i("limit_reset_nw_1", LIMIT_RESET_NW_1);
    g_monitor_config.limit_reset_data_1 = lynq_cfg_get_i("limit_reset_data_1", LIMIT_RESET_DATA_1);
    g_monitor_config.limit_reset_nad_2 = lynq_cfg_get_i("limit_reset_nad_2", LIMIT_RESET_NAD_2);
    g_monitor_config.limit_reset_nw_2 = lynq_cfg_get_i("limit_reset_nw_2", LIMIT_RESET_NW_2);
    g_monitor_config.limit_reset_data_2 = lynq_cfg_get_i("limit_reset_data_2", LIMIT_RESET_DATA_2);   
    g_monitor_config.opt_reset_nad = lynq_cfg_get_i("opt_reset_nad", 1);
    g_monitor_config.opt_reset_nw = lynq_cfg_get_i("opt_reset_nw", 2);
    g_monitor_config.opt_reset_data = lynq_cfg_get_i("opt_reset_data", 1);

    lynq_cfg_get_s("ping_addr1", g_monitor_config.ping_addr1, PING_ADDR_MAX_LEN, PING_ADDR1_DEFAULT);
    lynq_cfg_get_s("ping_addr2", g_monitor_config.ping_addr2, PING_ADDR_MAX_LEN, PING_ADDR2_DEFAULT);
    lynq_cfg_get_s("ping_addr3", g_monitor_config.ping_addr3, PING_ADDR_MAX_LEN, PING_ADDR3_DEFAULT);

    lynq_monitor_config_print(&g_monitor_config);
}
int lynq_monitor_init(const char* config_path)
{
    const char* config_file_path = config_path;
    int ret;
    if (config_path == NULL)
    {
        config_file_path = LYNQ_MONITOR_DEFAULT_CONFIG_PATH;
    }
    else if (access(config_path, R_OK) != 0)
    {
        config_file_path = LYNQ_MONITOR_DEFAULT_CONFIG_PATH;
    }
    //LYLOGSET(LOG_INFO);
    //LYLOGEINIT(USER_LOG_TAG);
    LYINFLOG("Initializing monitor with config at %s\n", config_file_path);
    lynq_config_load(config_file_path);
    lynq_monitor_config_init();

    if (g_monitor_config.network_monitor_enabled)
    {
        lynq_monitor_start_network_check_thread();
    }
    if (g_monitor_config.ping_monitor_enabled)
    {
        lynq_monitor_start_ping_check_thread();
    }
    return 0;
}

int lynq_monitor_register_exception_handler(MonitorExceptionHandler handler)
{
    s_ExceptionHandler = handler;
    LYINFLOG("Exception handler registered.\n");
    return 0;
}

int lynq_monitor_set_all_state(int state)
{
    if (state != 0 && state != 1)
    {
        return -1; // 参数错误
    }
    g_monitor_state = state;
    LYINFLOG("Setting all state to %d\n", state);
    return 0;
}

int lynq_monitor_set_monitor_state(int monitor_type,int enabled)
{
    if(monitor_type == 0)//network
    {
        g_monitor_config.network_monitor_enabled = enabled;
        LYINFLOG("Network monitoring set to %d\n", enabled);
    }
    else if(monitor_type == 1)//device
    {
        g_monitor_config.device_monitor_enabled = enabled;
        LYINFLOG("Device monitoring set to %d\n", enabled);
    }
    else if(monitor_type == 2)//ping
    {
        g_monitor_config.ping_monitor_enabled = enabled;
        LYINFLOG("Ping monitoring set to %d\n", enabled);
    }
    return 0;
}


// 异常处理的callback函数调用
static void lynq_monitor_handle_exception(int exception_type, int error_code, int opt_code)
{
    printf("Exception occurred: type=%d, code=%d, opt=%d\n", exception_type, error_code, opt_code);
    if (s_ExceptionHandler)
    {

        s_ExceptionHandler(exception_type, error_code, opt_code);
    }
    else
    {
        LYINFLOG("Exception occurred but no handler registered.\n");
    }
}

int lynq_monitor_check_state(void)
{
    if (g_monitor_state == 0)
    {
        return 1;
    }
    return 0;
}

int lynq_monitor_need_reset_nad(int exception_type, int error_code)
{
    LYINFLOG("lynq_monitor_need_reset_nad start\n");
    if (lynq_monitor_check_state())
    {
        return 0;
    }

    if(g_monitor_config.opt_reset_nad == 0)
    {
        return 0;
    }
    else if(g_monitor_config.opt_reset_nad == 1)
    {
        LYINFLOG("lynq_monitor_handle_exception ret 1\n");
        lynq_monitor_handle_exception(exception_type, error_code, LYNQ_OPT_RESET_NAD);
        return 1;
    }

    return -1;
}

int lynq_monitor_need_reset_data(int exception_type, int error_code)
{
    LYINFLOG("lynq_monitor_need_reset_data start\n");
    if (lynq_monitor_check_state())
    {
        return 0;
    }

    if(g_monitor_config.opt_reset_data == 0)
    {
        return 0;
    }
    else if(g_monitor_config.opt_reset_data == 1)
    {
        lynq_monitor_handle_exception(exception_type, error_code, LYNQ_OPT_RESET_DATA);
        return 1;
    }
 
    return -1;
}
int lynq_monitor_need_reset_flight_mode(int exception_type, int error_code)
{
    LYINFLOG("lynq_monitor_need_reset_flight_mode start\n");
    if (lynq_monitor_check_state())
    {
        return 0;
    }

    if(g_monitor_config.opt_reset_nw == 0)
    {
        return 0;
    }
    else if(g_monitor_config.opt_reset_nw == 1)
    {
        lynq_monitor_handle_exception(exception_type, error_code, LYNQ_OPT_RESET_NW);
        return 1;
    }
    else if(g_monitor_config.opt_reset_nw == 2)
    {
        lynq_monitor_reset_flight_mode();
        return 1;
    }
    
    return -1;
}

int lynq_monitor_need_reset_sim(int exception_type, int error_code)
{
    LYINFLOG("lynq_monitor_need_reset_data start\n");
    if (lynq_monitor_check_state())
    {
        return 0;
    }

    if(g_monitor_config.opt_reset_nw == 0)
    {
        return 0;
    }
    else if(g_monitor_config.opt_reset_nw == 1)
    {
        lynq_monitor_handle_exception(exception_type, error_code, LYNQ_OPT_RESET_SIM);
        return 1;
    }
    else if(g_monitor_config.opt_reset_nw == 2)
    {
        lynq_monitor_reset_sim();
        return 1;
    }

    return -1;
}

DEFINE_LYNQ_LIB_LOG(LYNQ_MONITOR)