/*============================================================================= 
#     FileName: lynq_gnss.c
#     Desc: about GNSS
#     Author: cz.li
#     Version: V1.0
#     LastChange: 2023-06-13 
#     History: 
=============================================================================*/

#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <errno.h>
#include <termios.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/types.h>
#include <unistd.h>
#include <pthread.h>
#include <termios.h>
#include <time.h>
#include <sys/timeb.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#include <dlfcn.h>
#include <stdint.h>
#include <linux/rpmsg/rpmsg_zx29.h>
#include <liblog/lynq_deflog.h>
#include "mbtk_gnss_internal.h"
#include "ringbuffer.h"
#include "lynq_qser_gnss.h"
#include "include/lynq_uci.h"
#include "include/lynq-qser-autosuspend.h"
DEFINE_LYNQ_LIB_LOG(LYNQ_QSER_GNSS)
#ifdef __cplusplus
extern "C" {
#endif
#define GNSS_WAKELOCK_NAME "GNSS_WAKELOCK"
int ret;
LYNQ_INJECT_TIME_INTO_T time_test;
int g_gnss_sync_enable_flag = 0;
int g_gnss_sync_done = 0;
int inited = 0;
int event = 0;
int lock_fd = 0;
static struct login_information *login_information_t = NULL;

int qser_Gnss_Init(uint32_t *h_gnss)
{
    ret = lynq_gnss_client_init(h_gnss);
    if(ret)
    {
        ALOGE("Init faild\n");
    }
#ifdef HD_GNSS
        login_information_t = (login_information*)malloc(sizeof(struct login_information));
        if(NULL == login_information_t)
        {
            ALOGE("malloc memory error\n");
            ret = lynq_gnss_client_deinit(h_gnss);
            if(ret)
            {
                ALOGE("Deinit faild\n");
            }
            return -1;
        }
        login_information_t->host = NULL;
        login_information_t->id = NULL;
        login_information_t->password = NULL;
#endif
    inited = 1;
    lock_fd = qser_wakelock_create(GNSS_WAKELOCK_NAME, strlen(GNSS_WAKELOCK_NAME));
    if(lock_fd < 0)
    {
        ALOGE("Create gnss wakelock failed %d",lock_fd);
    }
    if(qser_wakelock_lock(lock_fd) < 0)
    {
        ALOGE("Lock gnss wakelock failed");
    }
    return 0;
}

int qser_Gnss_Deinit(uint32_t h_gnss)
{
    if(!inited)
    {
        ALOGE("has not been initiated\n");
        return -1;
    }
    ret = lynq_gnss_client_deinit(h_gnss);
    if(ret)
    {
        ALOGE("Deinit faild\n");
    }
    if(login_information_t != NULL)
    {
        free(login_information_t);
        login_information_t = NULL;
    }
    inited = 0;

    if(qser_wakelock_unlock(lock_fd) < 0)
    {
        ALOGE("Unlock gnss wakelock failed");
    }
    if(qser_wakelock_destroy(lock_fd) < 0)
    {
        ALOGE("Destroy gnss wakelock failed");
    }

    return 0;
}

int qser_AddRxIndMsgHandler(gnss_handler_func_t handler_ptr,uint32_t h_gnss)
{
    if(!inited)
    {
        ALOGE("has not been initiated\n");
        return -1;
    }
    struct mbtk_gnss_handle_t *gnss_handle = (struct mbtk_gnss_handle_t *)h_gnss;
    if(0 == h_gnss && NULL == handler_ptr)
    {
        ALOGE("%s handler invalid.\n", __func__);
        return -1;
    }
    gnss_handle->gnss_handler_func = handler_ptr;
    return 0;
}

/**
 * @brief      qser_Set_Indications
 *
 * @details    Set Output Mode
 *             
 * @param      e_msg_id_t type
 *
 * @return     
 */

int qser_Set_Indications(uint32_t h_gnss,e_msg_id_t type)
{
    if(!inited)
    {
        ALOGE("has not been initiated\n");
        return -1;
    }
    struct mbtk_gnss_handle_t *gnss_handle = (struct mbtk_gnss_handle_t *)h_gnss;
    switch(type)
    {

    case 1:
    {
        gnss_handle->mode = 1;
        break;
    }

    case 3:
    {
        gnss_handle->mode = 3;
        break;
    }
    default:
        break;

    }
    return 0;
}

/**
 * @brief      qser_Gnss_InjectTime
 *
 * @details    injection time
 *             
 * @param      LYNQ_INJECT_TIME_INTO_T *time_info
 *
 * @return     
 */

/*need  set system time in advance*/
int qser_Gnss_InjectTime(uint32_t h_gnss,LYNQ_INJECT_TIME_INTO_T *time_info)
{
    if(!inited)
    {
        ALOGE("has not been initiated\n");
        return -1;
    }
    time_t tNOW =time(NULL);
    struct timeb tmNOW;
    struct tm *ptm = {0};
    ptm = gmtime(&tNOW);
    ftime(&tmNOW);
    time_info->year = (ptm->tm_year+1900);
    time_info->month = (ptm->tm_mon+1);
    time_info->day = ptm->tm_mday;
    time_info->hour = ptm->tm_hour;
    time_info->minute = ptm->tm_min;
    time_info->second = ptm->tm_sec;
    time_info->millisecond = tmNOW.millitm;
    char send_buf[128] = {0};
    printf("%d\n,%d\n,%d\n,%d\n,%d\n,%d\n,%d\r\n",time_info->year,time_info->month,time_info->day,\
           time_info->hour,time_info->minute,time_info->second,time_info->millisecond);
    snprintf(send_buf, sizeof(send_buf), "$AIDTIME,%d,%d,%d,%d,%d,%d,%d\r\n", time_info->year,time_info->month,time_info->day,\
             time_info->hour,time_info->minute,time_info->second,time_info->millisecond);
    ret = lynq_gnss_send_cmd(h_gnss, send_buf, sizeof(send_buf));
    if(ret < 0)
    {
        ALOGE("InjectTime send fail\n");
        return -1;
    }
    lynq_gnss_get_aidinfo(h_gnss);
    return 0;
}
int qser_Gnss_Start(uint32_t h_gnss)
{
    if(!inited)
    {
        ALOGE("has not been initiated\n");
        return -1;
    }
    struct mbtk_gnss_handle_t *gnss_handle = (struct mbtk_gnss_handle_t *)h_gnss;
    char gnss_sync_enable[24] = "";
    lynq_get_value("lynq_uci", "lynq_sync_time", "lynq_gnss_sync_time_enable" , gnss_sync_enable);
    g_gnss_sync_enable_flag = atoi(gnss_sync_enable);
    g_gnss_sync_done = 0;

    if (lynq_gnss_start() == -1)
        return -1;
#if defined(HD_GNSS)
        ret = set_baudrate(gnss_handle->dev_fd, B115200);
        if(-1 == ret)
        {
            return -2;
        }
#elif defined(HX_GNSS)
        ret = set_baudrate(gnss_handle->dev_fd, B9600);
        if(-1 == ret)
        {
            return -2;
        }
#endif
    return 0;
}

int qser_Gnss_Stop(uint32_t h_gnss)
{
    if(!inited)
    {
        ALOGE("has not been initiated\n");
        return -1;
    }
    lynq_open_gps(0);
    return 0;
}

/**
 * @brief      qser_Gnss_Set_Frequency
 *
 * @details    Change output frequency(just for 106E)
 *             
 * @param      frequency:1 2 5hz
 *
 * @return     
 */

int qser_Gnss_Set_Frequency(uint32_t h_gnss, int frequency)
{
    int ret;
    if(!inited)
    {
        ALOGE("has not been initiated\n");
        return -1;
    }

    ret = lynq_gnss_output_frq(frequency);
    if(ret < 0)
    {
        ALOGE("modify cmd send faild\n");
        return -1;
    }
    return 0;
}

/*清除辅助数据
bit0 - 清除星历
bit1 - reserve0
bit2 - 清除接收机位置和接收机时间
bit3 - reserve1
bit4 - 清除电离层修正参数和 UTC 参数
bit5 - reserve2
bit6 - reserve3
bit7 - 清除历书
*/

int qser_Gnss_Delete_Aiding_Data(uint32_t h_gnss,DELETE_AIDING_DATA_TYPE_T flags)
{
    char send_buf[60] = {0};
    if(!inited)
    {
        ALOGE("has not been initiated\n");
        return -1;
    }
    struct mbtk_gnss_handle_t *gnss_handle = (struct mbtk_gnss_handle_t *)h_gnss;
    switch(flags)
    {
    case 0:
    {
        snprintf(send_buf, sizeof(send_buf), "$RESET,0,h00\r\n");

        break;
    }
    case 1:
    {
        snprintf(send_buf, sizeof(send_buf), "$RESET,0,h01\r\n");

        break;
    }
    case 2:
    {
        snprintf(send_buf, sizeof(send_buf), "$RESET,0,h04\r\n");
        break;
    }
    case 3:
    {
        snprintf(send_buf, sizeof(send_buf), "$RESET,0,h10\r\n");
        break;
    }
    case 4:
    {
        snprintf(send_buf, sizeof(send_buf), "$RESET,0,h80\r\n");
        break;
    }
    case 5:
    {
        snprintf(send_buf, sizeof(send_buf), "$RESET,0,hFF\r\n");//冷启动
        break;
    }
    }
    ring_buffer_clean(&gnss_handle->ring_buffer);
    gnss_log("%s : %s\n", __FUNCTION__, send_buf);
    pthread_mutex_lock(&gnss_handle->_cond_mutex);
    gnss_handle->reset_state = 1;
    pthread_mutex_unlock(&gnss_handle->_cond_mutex);
    ret = lynq_gnss_send_cmd(h_gnss, send_buf, strlen(send_buf));
    if(ret < 0)
    {
        ALOGE("Delete_Aiding_Data cmd send faild");
        return -1;
    }
    ALOGE("reset OK!!!!!\n");

    ret = set_baudrate(gnss_handle->dev_fd, B9600);
    if(-1 == ret)
    {
        return  -2;
    }
    return 0;
}


/**
 * @brief      qser_Gnss_Server_Configuration
 *
 * @details    Configure user information for 106E-AGPS
 *             
 * @param      host id password (if the parameter is NULL,use default Configure)
 *
 * @return     
 */

int qser_Gnss_Server_Configuration(char *host, char *id, char *password)
{
    if(NULL == login_information_t)
    {
        ALOGE("don't have login information\n");
        return -1;
    }
    login_information_t->host = host;
    login_information_t->id = id;
    login_information_t->password = password;
    ALOGD("host:%s\nid:%s\npw:%s\n",login_information_t->host,login_information_t->id,login_information_t->password);
    return 0;
}

/**
 * @brief      qser_Gnss_download_tle
 *
 * @details    download_tle 106-AGPS tle,this interface not require init, but it must be able to dial up for internet access
 *             the download_tle will save in /mnt/userdata/agnss_tle
 * @param      NULL
 *
 * @return     
 */

int qser_Gnss_download_tle()
{
    int ret = -1;
    if(!inited)
    {
        ALOGE("has not been initiated\n");
        return -1;
    }
#if defined(HD_GNSS)
        ret = hd_gnss_download_tle(login_information_t->host, login_information_t->id, login_information_t->password);
        if(ret < 0)
        {
            ALOGE("download fail,ret = %d\n",ret);
            return -1;
        }
#elif defined(HX_GNSS)
        ret = mopen_gnss_download_tle();
        if(ret < 0)
        {
            ALOGE("download fail,ret = %d\n",ret);
            return -1;
        }
#endif
    return 0;
}

/**
 * @brief      qser_Gnss_injectEphemeris
 *
 * @details    inject Ephemeris 
 *             
 * @param      
 *
 * @return     
 */

int qser_Gnss_injectEphemeris(uint32_t h_gnss)
{
    int ret;
    if(!inited)
    {
        ALOGE("has not been initiated\n");
        return -1;
    }
#if defined(HD_GNSS)
    ret = hd_gnss_injects_aidpos(h_gnss);
    if(ret < 0)
    {
         return -1;
    }
#elif defined(HX_GNSS)
    ret = mopen_gnss_injects_aidpos(h_gnss);
    if(ret < 0)
    {
        return -1;
    }
#endif
    return 0;
}

int qser_Gnss_injectEphemeris_withpath(uint32_t h_gnss, char *path)
{
    int ret;
    if(!inited)
    {
        ALOGE("has not been initiated\n");
        return -1;
    }
    ret = lynq_gnss_Injection_ephemeris(h_gnss,path);
    if(ret < 0)
    {
        return -1;
    }

    return 0;
}

int qser_Gnss_setting_statement_type(uint32_t h_gnss, char *statement_type,int state)
{
    int ret;
    if(!inited)
    {
        ALOGE("has not been initiated\n");
        return -1;
    }
    ret = lynq_gnss_setting_statement_type(h_gnss,statement_type,state);
    if(ret < 0)
    {
        return -1;
    }

    return 0;
}

void atsvc_incb_entity(char *input,int length);
int lynq_at_cgps(int at_type,int gnss_state_type);
int lynq_at_cgpsnmea(int at_type,int gnss_state_type);
lynq_atsvc_outcb atsvc_gnss_outcb;
uint32_t h_gnss;
int at_gps_status = 0;
int at_gpsnmea_status = 0;
int pthread_state = 1;
int device_fd = -1;

int strUpper(char * str)
{
    int i=0;
    while(1)
    {
        if(str[i]=='\0')
        {
            break;
        }
        if(str[i]>='a'&&str[i]<='z')
        {
             str[i]=str[i]-32;
        }
        i++;
    }
    return 0;
}

static int channel_config(int device_fd)
{
    if(ioctl(device_fd, RPMSG_CREATE_CHANNEL, 512) < 0)
    {
        ALOGE("CREATE_CHANNEL Failed\n");
    }
    // 发送消息时，触发中断
    if(ioctl(device_fd, RPMSG_SET_INT_FLAG, NULL) < 0)
    {
        ALOGE("SET_INT_FLAG Failed\n");
        return -1;
    }
    // 阻塞的方式读；
    if(ioctl(device_fd, RPMSG_CLEAR_POLL_FLAG, NULL) < 0)
    {
        ALOGE("CLEAR_POLL_FLAG Failed\n");
        return -1;
    }
    return 0;
}

int gps_channel_init()
{
    int ret = 0;
    device_fd = open("/dev/armps_rpmsgch30", O_RDWR);//30 is AT-channel;31 is modem-channel
    ALOGD("device_fd:%d\n",device_fd);
    if (device_fd == -1)
    {
        ALOGE("Failed to open device file\n");
        return -1;
    }
    ret = channel_config(device_fd);
    if(ret < 0)
    {
        close(device_fd);
        return -1;
    }
    return 0;
}

time_t start;
time_t end;
int lynq_at_navsat(uint32_t h_gnss)
{
    struct mbtk_gnss_handle_t *gnss_handle = (struct mbtk_gnss_handle_t *)h_gnss;
    event = 1;
    char send_buf[] = {0xF1,0xD9,0x06,0x0C,0x00,0x00,0x12,0x3C};
    ret = write(gnss_handle->dev_fd, send_buf, sizeof(send_buf));
    if(ret < 0)
    {
        ALOGE("send navsat faild");
        return -1;
    }
    time(&start);
    while(event)
    {
        usleep(500000);
        time(&end);
        if(difftime(end,start) > 2)
        {
            return -1;
        }
    }
    
    return 0;
}

int lynq_at_gnssver(uint32_t h_gnss)
{
    struct mbtk_gnss_handle_t *gnss_handle = (struct mbtk_gnss_handle_t *)h_gnss;
    event = 2;
    char send_buf[] = {0xF1,0xD9,0x0A,0x04,0x00,0x00,0x0E,0x34};
    ret = write(gnss_handle->dev_fd, send_buf, sizeof(send_buf));
    if(ret < 0)
    {
        ALOGE("send gnssver faild");
        return -1;
    }
    time(&start);
    while(event)
    {
        usleep(500000);
        time(&end);
        if(difftime(end,start) > 2)
        {
            return -1;
        }
    }
    return 0;
}

int lynq_at_gnssota(int gnss_ver_symbol)
{
    FILE *fp;
    int mask = -1;
    char FileName[16];
    char cmd[64];
    char buf[256] = {0};
    gps_channel_init();
    lynq_open_gps(1);
    usleep(100000);
    if(gnss_ver_symbol == 1)
    {
        strcpy(FileName,"HD.bin");
    }
    else if(gnss_ver_symbol == 2)
    {
        strcpy(FileName,"HD_BD.bin");
    }
    else 
        return -1;
    sprintf(cmd,"lynq-gnss-update -B /data/gnss_update/%s",FileName);
    fp = popen(cmd, "r");
    if(fp == NULL)
    {
        ALOGE("error open\n");
        return -1;
    }
    while(fgets(buf, sizeof(buf) - 1, fp) != NULL)
    {
        write(device_fd, buf, strlen(buf));
        if(strncmp("upgrade sucess!\r\n",buf,strlen(buf)) == 0)
        {
            mask = 0;
        }
        memset(buf,0,sizeof(buf));
    }
    pclose(fp);
    return mask;
}

void at_callback(uint32_t h_loc,e_msg_id_t e_msg_id,void *pv_data,void *context_ptr)
{
    mopen_gnss_nmea_info_t  *pt_nmea = (mopen_gnss_nmea_info_t *)pv_data;
    write(device_fd, pt_nmea->nmea, strlen(pt_nmea->nmea));
}

int gnss_at_cmd_parse(char *cmd,char *parse_cmd,int* gnss_state_type,int* at_type)
{
    if (NULL == cmd || NULL == parse_cmd || NULL == at_type)
    {
        return -1;
    }
    int ret = 0;
    int at_type_jug = 0;
    int cmd_size;
    char cmd_buf[32] = {0};
    char buffer1[32] = {0};
    char buffer2[32] = {0};
    bzero(cmd_buf,32);
    bzero(buffer1,32);
    bzero(buffer2,32);
    cmd_size = strlen(cmd);
    strncpy(cmd_buf,cmd,cmd_size);
    strUpper(cmd_buf);
    ret = sscanf(cmd_buf, "%[^=]=%[^=]", buffer1,buffer2);
    if (ret == 1)
    {
        *at_type = 1;
        sscanf(buffer1, "%[^?]", buffer2);
        strcpy(parse_cmd,buffer2);
        return 0;
    }
    else if (ret == 2)
    {
        at_type_jug = strcmp(buffer2,"?");
        ALOGD("at_type_jug :%d\n",at_type_jug);
        if (at_type_jug == 0)
        {
            *at_type = 0;
            strcpy(parse_cmd,buffer1);
            return 0;
        }
        else 
        {
            *at_type = 2;
            ALOGD("Buffertest1:buffer1 :%s  buffer2 :%s\n",buffer1,buffer2);
            strcpy(parse_cmd,buffer1);
            *gnss_state_type = atoi(buffer2);
            ALOGD("buffer1 :%s  buffer2 :%d\n",parse_cmd,*gnss_state_type);
            return 0;
        }
    }
    else
    {
        ALOGE("unknown paramters");
        return -1;
    }
}

int lynq_at_cgps(int at_type,int gnss_state_type)
{
    int ret = 0;
    char cgps_at_res[64]={};
    if (at_type == 0)
    {
        strncpy(cgps_at_res,"+CGPS:(0,1)\r\n",64);
        atsvc_gnss_outcb(cgps_at_res,strlen(cgps_at_res),0);
        return 0;
    }

    else if(at_type == 1)
    {
        bzero(cgps_at_res,64);
        sprintf(cgps_at_res,"+CGPS:<%d>",at_gps_status);
        atsvc_gnss_outcb(cgps_at_res,strlen(cgps_at_res),0);
        return 0;
    }

    else if(at_type == 2)
    {
        if (at_gps_status != gnss_state_type)
        {
            at_gps_status = gnss_state_type;
        }
        else
        {
            strncpy(cgps_at_res,"+CGPS: same status\r\n",64);
            atsvc_gnss_outcb(cgps_at_res,strlen(cgps_at_res),0);
            return -1;
        }
        if (at_gps_status == 0)
        {
            qser_Gnss_Deinit(h_gnss);  
        }
        else if(at_gps_status == 1)
        {
            ret = gps_channel_init();
            if(ret < 0)
            {
                return -1;
            }
            qser_Gnss_Init(&h_gnss);
            qser_AddRxIndMsgHandler(&at_callback,h_gnss);
        }
        else
        {
            ALOGE("unknown at paramters");
            return -1;
        }
        return 0;
    }
}

int lynq_at_cgpsnmea(int at_type,int gnss_state_type)
{
    char cgpsnmea_at_res[64]={};
    if (at_type == 0)
    {
        sprintf(cgpsnmea_at_res,"+CGPSNMEA:(0,1)\r\n");
        atsvc_gnss_outcb(cgpsnmea_at_res,strlen(cgpsnmea_at_res),0);
        return 0;
    }
    else if(at_type == 1)
    {
        bzero(cgpsnmea_at_res,64);
        sprintf(cgpsnmea_at_res,"+CGPSNMEA:<%d>\r\n",at_gpsnmea_status);
        atsvc_gnss_outcb(cgpsnmea_at_res,strlen(cgpsnmea_at_res),0);
        return 0;
    }
    else if(at_type == 2)
    {
        if (at_gpsnmea_status != gnss_state_type)
        {
            at_gpsnmea_status = gnss_state_type;
        }
        else
        {
            sprintf(cgpsnmea_at_res,"+CGPSNMEA: same status\r\n");
            atsvc_gnss_outcb(cgpsnmea_at_res,strlen(cgpsnmea_at_res),0);
            return -1;
        }

        if (at_gpsnmea_status == 0)
        {
            qser_Gnss_Stop(h_gnss);
        }
        else if(at_gpsnmea_status == 1)
        {
            qser_Gnss_Start(h_gnss);
        }

        else
        {
            ALOGE("unknown at paramters");
            return -1;
        }
        return 0;
    }
    
}

int lynq_at_gnssmode(int gnss_state_type)
{
    char hot_start[] = {0xF1,0xD9,0x06,0x40,0x01,0x00,0x03,0x4A,0x24};
    char warm_start[] = {0xF1,0xD9,0x06,0x40,0x01,0x00,0x02,0x49,0x23};
    char cold_start[] = {0xF1,0xD9,0x06,0x40,0x01,0x00,0x01,0x48,0x22};
    int ret;
    int buf_length;
    char *send_buf = NULL;
    int fd = open(MBTK_GNSS_DEV, O_RDWR | O_NOCTTY | O_NDELAY);
    if(fd < 0) {
        ALOGE("open error\n");
        return -1;
    }
    switch(gnss_state_type)
    {
        case 1:
        {
            send_buf = cold_start;
            buf_length = sizeof(cold_start);
            break;
        }
        case 2:
        {
            send_buf = hot_start;
            buf_length = sizeof(hot_start);
            break;
        }
        case 3:
        {
            send_buf = warm_start;
            buf_length = sizeof(warm_start);
            break;
        }
        default:
        {
            close(fd);
            return -2;
        }
    }
    ret = write(fd, send_buf, buf_length);
    if(ret < 0)
    {
        ALOGE("modify cmd send faild\n");
        close(fd);
        return -1;
    }
    usleep(100000);
    close(fd);
    return 0;
}

void atsvc_incb_entity(char *input,int length)
{
    int res = 0;
    int income_at_type = 0;
    int gnss_state_type = -1;
    char gnss_at_cmd[64] = {0};
    char parse_atcmd[128] = {0}; 
    if(NULL == input)
    {
        sprintf(gnss_at_cmd,"+CME ERROR: 100\r\n");
        atsvc_gnss_outcb(gnss_at_cmd,strlen(gnss_at_cmd),0);
        return;
    }

    res = gnss_at_cmd_parse(input,parse_atcmd,&gnss_state_type,&income_at_type);
    if (res != 0)
    {
        ALOGE("parse at cmd error");
        sprintf(gnss_at_cmd,"+CME ERROR: 100\r\n");
        atsvc_gnss_outcb(gnss_at_cmd,strlen(gnss_at_cmd),0);
        return;
    }

    if (!strcmp(parse_atcmd, "AT+CGPS"))
    {
        res = lynq_at_cgps(income_at_type,gnss_state_type);
        if (res != 0)
        {
            sprintf(gnss_at_cmd,"+CME ERROR: 100\r\n");
            atsvc_gnss_outcb(gnss_at_cmd,strlen(gnss_at_cmd),0);
            return;
        }
    }
    else if (!strcmp(parse_atcmd, "AT+CGPSNMEA"))
    {
        res = lynq_at_cgpsnmea(income_at_type,gnss_state_type);
        if (res != 0)
        {
            sprintf(gnss_at_cmd,"+CME ERROR: 100\r\n");
            atsvc_gnss_outcb(gnss_at_cmd,strlen(gnss_at_cmd),0);
            return;
        }
    }
    else if (!strcmp(parse_atcmd, "AT+NAVSAT"))
    {
        res = lynq_at_navsat(h_gnss);
        if (res != 0)
        {
            sprintf(gnss_at_cmd,"+CME ERROR: 100\r\n");
            atsvc_gnss_outcb(gnss_at_cmd,strlen(gnss_at_cmd),0);
            return;
        }
    }
    else if (!strcmp(parse_atcmd, "AT+GNSSVER"))
    {
        res = lynq_at_gnssver(h_gnss);
        if (res != 0)
        {
            sprintf(gnss_at_cmd,"+CME ERROR: 100\r\n");
            atsvc_gnss_outcb(gnss_at_cmd,strlen(gnss_at_cmd),0);
            return;
        }
    }
    else if (!strcmp(parse_atcmd, "AT+GNSSOTA"))
    {
        res = lynq_at_gnssota(gnss_state_type);
        if (res != 0)
        {
            sprintf(gnss_at_cmd,"+CME ERROR: 100\r\n");
            atsvc_gnss_outcb(gnss_at_cmd,strlen(gnss_at_cmd),0);
            return;
        }
    }
    else if (!strcmp(parse_atcmd, "AT+GNSSMODE"))
    {
        res = lynq_at_gnssmode(gnss_state_type);
        if (res != 0)
        {
            sprintf(gnss_at_cmd,"+CME ERROR: 100\r\n");
            atsvc_gnss_outcb(gnss_at_cmd,strlen(gnss_at_cmd),0);
            return;
        }
    }
    else
    {
        ALOGE("unknown at cmd\r\n");
        sprintf(gnss_at_cmd,"+CME ERROR: 100\r\n");
        atsvc_gnss_outcb(gnss_at_cmd,strlen(gnss_at_cmd),0);
        return;
    }
    sprintf(gnss_at_cmd,"OK\r\n");
    atsvc_gnss_outcb(gnss_at_cmd,strlen(gnss_at_cmd),0);
    return;
}

lynq_atsvc_incb lynq_register_gnss(lynq_atsvc_outcb out_cb)
{
    if(NULL == out_cb)
    {
        ALOGE("out cb is null");
        return NULL;
    }
    atsvc_gnss_outcb = out_cb;
    ALOGE("gnss register success\r\n");
    return atsvc_incb_entity;
}

#ifdef __cplusplus
}
#endif
