#include <stdio.h>
#include <stdint.h>
#include <sys/types.h>
#include <sys/time.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include <signal.h>
#include <time.h>
#include <errno.h>
#include <fcntl.h>
#include <stdarg.h>
#include <climits>
#include <stdlib.h>
#include "liblog/lynq_deflog.h"
#include "lynq_data.h"
#include "lynq_data_list_buffer.h"
#include <include/lynq_uci.h>

#define LOG_UCI_FILE "lynq_uci"
#define LOG_UCI_MODULE "lynq_data"
#define USER_LOG_TAG "LYNQ_DATA_BUFFER"
#define DEFAULT_PERIOD 5    //default 5s

static timer_t s_list_buffer_timer;
static int s_list_buffer_valid = false;
static int s_state_buffer_valid_period;
lynq_data_call_response_v11_t dataCallList_buffer[LYNQ_APN_CHANNEL_MAX];
static int number_buffer=0;

void start_data_timer(timer_t timer, int milliseconds) 
{
    LYINFLOG("start_data_timer(), timer_id=%ld, time=%d",(long)timer, milliseconds);

    struct itimerspec expire;
    expire.it_interval.tv_sec = 0;
    expire.it_interval.tv_nsec = 0;
    expire.it_value.tv_sec = milliseconds/1000;
    expire.it_value.tv_nsec = (milliseconds%1000)*1000000;
    if (timer_settime(timer, 0, &expire, NULL) == -1) {
        LYERRLOG("timer_settime  failed reason=[%s]", strerror(errno));
    }
}
void stop_data_timer(timer_t timer) 
{
    LYINFLOG("stop_data_timer(), timer_id=%ld", (long)timer);
    struct itimerspec timespec;
    if(timer_gettime(timer, &timespec) == -1) {
        LYERRLOG("stop_data_timer(), get time fail(%s)", strerror(errno));
        return;
    }
    LYINFLOG("stop_data_timer(), tv_sec=%ld, tv_nsec=%ld",timespec.it_value.tv_sec, timespec.it_value.tv_nsec);
    if((timespec.it_value.tv_sec == 0)  && (timespec.it_value.tv_nsec == 0) ) {
        LYINFLOG("stop_data_timer(), timer_id(%ld) had stopped, just return", (long)timer);
        return;
    } else {
        start_data_timer(timer, 0);
    }
}

void data_timer_handler(sigval_t sig) 
{    
    LYDBGLOG("[%s] entry\n",__FUNCTION__);
    s_list_buffer_valid = false; 
}

int is_period_equal_not_valid(int period)
{
    return period<=0 || (period > INT_MAX/1000);
}

void init_data_timer_all()
{
    int ret;
    char tmp[20]={0};
    char lynq_state_buffer_valid_period[20]={0};
    struct sigevent sevp;
    memset(&sevp, 0, sizeof(sevp));
    sevp.sigev_notify = SIGEV_THREAD;
    sevp.sigev_notify_function = data_timer_handler;
    
    ret = lynq_get_value(LOG_UCI_FILE, LOG_UCI_MODULE, "data_list_valid_period", tmp);
    LYINFLOG("[%s] ret=%d, tmp is %s", __FUNCTION__, ret, tmp);
    if(ret == 0)
    {
        s_state_buffer_valid_period = atoi(tmp);
        if(is_period_equal_not_valid(s_state_buffer_valid_period))
        {
            s_state_buffer_valid_period = DEFAULT_PERIOD;
            snprintf(lynq_state_buffer_valid_period,sizeof(lynq_state_buffer_valid_period),"%d",DEFAULT_PERIOD);
            lynq_set_section(LOG_UCI_MODULE, LOG_UCI_MODULE);
            lynq_set_value(LOG_UCI_MODULE, "data_list_valid_period", lynq_state_buffer_valid_period);
        }
    }
    else 
    {
        s_state_buffer_valid_period = DEFAULT_PERIOD;
        snprintf(lynq_state_buffer_valid_period,sizeof(lynq_state_buffer_valid_period),"%d",DEFAULT_PERIOD);
        lynq_set_section(LOG_UCI_MODULE, LOG_UCI_MODULE);
        lynq_set_value(LOG_UCI_MODULE, "data_list_valid_period", lynq_state_buffer_valid_period);
    }
    LYINFLOG("[%s] s_state_buffer_valid_period = %d\n", __FUNCTION__, s_state_buffer_valid_period);
    if(timer_create(CLOCK_MONOTONIC, &sevp, &s_list_buffer_timer) == -1) {
        LYERRLOG("init_data_timer_all()  failed reason=[%s]", strerror(errno));
    }
    LYINFLOG("init_data_timer_all(), timer_Id = %ld\n", (long)(s_list_buffer_timer));
    s_list_buffer_valid = false;
    return;
}
void deinit_data_timer_all()
{
    stop_data_timer(s_list_buffer_timer);
    s_list_buffer_valid = false;

    return;
}

void set_list_buffer_valid(int valid)
{
    LYDBGLOG("[%s] entry\n",__FUNCTION__);

    s_list_buffer_valid = valid;
    if(s_list_buffer_valid==false)
    {
        stop_data_timer(s_list_buffer_timer);
    }
    else 
    {
        start_data_timer(s_list_buffer_timer, s_state_buffer_valid_period*1000);
    }
    LYINFLOG("[%s] valid is %d",__FUNCTION__,valid); 
    return;
}
void set_list_to_buf(lynq_data_call_response_v11_t dataCallList[LYNQ_APN_CHANNEL_MAX],int number)
{
    int ret;
    int i;

    LYDBGLOG("[%s] entry\n",__FUNCTION__);
    number_buffer = number;
    memcpy(dataCallList_buffer, dataCallList, sizeof(lynq_data_call_response_v11_t)*LYNQ_APN_CHANNEL_MAX);
    set_list_buffer_valid(true);

    return;
}
int get_list_from_buf(lynq_data_call_response_v11_t dataCallList[LYNQ_APN_CHANNEL_MAX],int *number)
{
    if(s_list_buffer_valid == false)
    {
        LYINFLOG("%s buffer not valid",__FUNCTION__);
        return 1;
    }
    *number = number_buffer;
    memcpy(dataCallList, dataCallList_buffer, sizeof(lynq_data_call_response_v11_t)*LYNQ_APN_CHANNEL_MAX);

    return 0;
}
