#include <stdint.h>
#include <stdio.h>
#include <string.h>

#include "liblog/lynq_deflog.h"

#ifdef __cplusplus
extern "C" {
#endif

#include "sc_audio.h"
#include "lynq-qser-audio.h"

#define AUDIO_INIT_MAX_TRY_CNT 100

#define SC_AUDIO_BE_DAI_MIN -1
#define SC_AUDIO_STREAM_FORMAT_INVALID 0

sc_audio_handle_t playback_handle = SC_AUDIO_INVALID_HANDLE;
sc_audio_handle_t capture_handle = SC_AUDIO_INVALID_HANDLE;

extern int sc_audio_set_codec_up_vol(int volume);
extern int sc_audio_get_codec_up_vol(int *p_volume);
extern int sc_audio_set_codec_down_vol(int volume);
extern int sc_audio_get_codec_down_vol(int *p_volume);

/********************************************************************
* @brief: _cb_onPlayer, typedef for a callback function that is called
          when an audio operation is performed
* @param int [IN]: The result of the audio operation, 0 if successful, non-zero if failed
* @return : void
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
typedef void (*_cb_onPlayer)(int);

/********************************************************************
* @brief: playback_state_cb, callback function that is called when the audio playback state changes
* @param handle [IN]: sc_audio_handle_t, the handle of the audio device
* @param params [IN]: void*, parameters for the callback function
* @param state [IN]: sc_audio_playback_state_e, the current state of audio playback
* @return : int, 0 if successful, non-zero if failed
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int playback_state_cb (sc_audio_handle_t handle, void *params, sc_audio_playback_state_e state)
{
    LYINFLOG("playback_state_cb handle:0x%lx state:%d\n", handle, state);
    return 0;
}

/********************************************************************
* @brief: capture_state_cb, callback function that is called when the audio capture state changes
* @param handle [IN]: sc_audio_handle_t, the handle of the audio device
* @param params [IN]: void*, parameters for the callback function
* @param state [IN]: sc_audio_capture_state_e, the current state of audio capture
* @return : int, 0 if successful, non-zero if failed
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int capture_state_cb (sc_audio_handle_t handle, void *params, sc_audio_capture_state_e state)
{
    LYINFLOG("capture_state_cb handle:0x%lx state:%d\n", handle, state);

    return 0;
}

/********************************************************************
* @brief: get_device_enum, function to convert a device string to its corresponding enum
* @param device [IN]: const char*, the name of the device
* @return : sc_audio_fe_pcm_dev_e, the enum corresponding to the device name
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
sc_audio_fe_pcm_dev_e get_device_enum(const char* device)
{
    if (strcmp(device, "device1") == 0)
    {
        return SC_AUDIO_FE_PCM_DEV_MULTIMEDIA1;
    }
    else if (strcmp(device, "device2") == 0)
    {
        return SC_AUDIO_FE_PCM_DEV_MULTIMEDIA2;
    }
    else
    {
        return SC_AUDIO_INVALID_HANDLE;
    }
}

/********************************************************************
* @brief: qser_AudPlayer_Open, open the audio device for playback
* @param device [IN]: char* device, the audio device to be opened for playback
* @param cb_fun [IN]: _cb_onPlayer, callback function to be called when the audio device is opened
* @return : int, 0 if successful, non-zero if failed
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_AudPlayer_Open(char* device, _cb_onPlayer cb_fun)
{
    int ret = SC_ERR_SUCCESS;
    int retry_cnt = 0;
    int audio_is_init = 0;

    sc_audio_fe_pcm_dev_e device_enum = get_device_enum(device); // Convert device string to enum

    while (AUDIO_INIT_MAX_TRY_CNT > retry_cnt)
    {
        ret = sc_audio_init();
        if (SC_ERR_NOT_READY == ret)
        {
            LYINFLOG("audio service is not ready, try again, try count = %d\n", (retry_cnt + 1));
            usleep(200 * 1000);
            retry_cnt++;
        }
        else if (SC_ERR_SUCCESS == ret)
        {
            LYINFLOG("Success to initialize audio service\n");
            audio_is_init = 1;
            break;
        }
        else
        {
            LYINFLOG("Failed to initialize audio service, ret = %d\n", ret);
            break;
        }
    }
    if (1 != audio_is_init)
    {
        LYINFLOG("Failed to initialize audio service\n");
        if (cb_fun != NULL)
        {
            cb_fun(-1);
        }
        return -1;
    }

    sc_audio_handle_t device_handle = sc_audio_playback_open(device_enum, SC_AUDIO_FE_PCM_DEV_MIN, SC_AUDIO_OWNER_ID_PLAYER);
    if (SC_AUDIO_INVALID_HANDLE == device_handle)
    {
        LYINFLOG("Failed to open device: %s\n", device);
        if (cb_fun != NULL)
        {
            cb_fun(-1);
        }
        return -1;
    }

    if (cb_fun != NULL)
    {
        cb_fun(0);
    }
    return 0;
}

/********************************************************************
* @brief: qser_AudPlayer_PlayFrmFile, play audio from file
* @param hdl [IN]: int, handle for the audio device or stream
* @param fd [IN]: const char*, file descriptor of the audio file
* @param offset [IN]: int, offset in the audio file
* @return : success 0, failed -1
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_AudPlayer_PlayFrmFile(int hdl, const char *fd, int offset)
{
    int error_line;
    sc_audio_pcm_config_t pcm_config;
    int ret = 0;
    sc_audio_owner_id owner_id = hdl;

    if(NULL== fd || 0 == strlen(fd))
    {
        error_line = __LINE__;
        goto exit;
    }

    playback_handle = sc_audio_playback_open(SC_AUDIO_FE_PCM_DEV_MULTIMEDIA2,SC_AUDIO_FE_PCM_DEV_MIN,owner_id);
    if (SC_AUDIO_INVALID_HANDLE == playback_handle)
    {
        error_line = __LINE__;
        goto exit;
    }

    if(strlen(fd))
    {
        ret = sc_audio_playback_file_prepare(playback_handle, fd, NULL, playback_state_cb, NULL);
        if (SC_ERR_SUCCESS != ret)
        {
            error_line = __LINE__;
            goto exit;
        }
    }

    ret = sc_audio_playback_play(playback_handle);
    if (SC_ERR_SUCCESS != ret)
    {
        error_line = __LINE__;
        goto exit;
    }

    return 0;
exit:
    LYINFLOG("qser_AudPlayer_PlayFrmFile error_line=%d\n",error_line);
    return -1;
}

/********************************************************************
* @brief: qser_AudPlayer_Play_Pcmbuf, play audio from PCM buffer
* @param period_size [IN]: int, size of the period for the PCM buffer, use default value if -1
* @param period_count [IN]: int, count of the period for the PCM buffer, use default value if -1
* @param num_channels [IN]: int, number of audio channels, use default value if -1
* @param sample_rate [IN]: int, sample rate for the audio, use default value if -1
* @param ownerid [IN]: int, ID of the audio owner, use default value if not in valid range
* @return : success 0, failed -1
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_AudPlayer_PlayPcmBuf(const unsigned char *pcm_data, int data_size, int period_size, \
                int period_count, int num_channels, int sample_rate, int ownerid)
{
    if (pcm_data == NULL)
    {
	LYINFLOG("Error: pcm_data parameter is a null pointer \n");
        return -1;
    }

    int error_line;
    int audio_is_init = 0;

    sc_audio_pcm_config_t pcm_config;
    int ret = 0;
    sc_audio_owner_id owner_id;

    if(ownerid <= SC_AUDIO_OWNER_ID_NONE || ownerid >= SC_AUDIO_OWNER_ID_MAX)
    {
        owner_id = SC_AUDIO_OWNER_ID_PLAYER;
    } else
    {
        owner_id = ownerid;
    }

    playback_handle = sc_audio_playback_open(SC_AUDIO_FE_PCM_DEV_MULTIMEDIA2,SC_AUDIO_FE_PCM_DEV_MIN, owner_id);
    if (SC_AUDIO_INVALID_HANDLE == playback_handle)
    {
        error_line = __LINE__;
        goto exit;
    }

    if(-1 == period_size || -1 == period_count || -1 == num_channels || -1 == sample_rate)
    {
        ret = sc_audio_playback_stream_prepare(playback_handle, NULL, playback_state_cb, NULL);
    } else
    {
        pcm_config.period_size = period_size;
        pcm_config.period_count = period_count;
        pcm_config.flags = 0;
        pcm_config.num_channels = num_channels;
        pcm_config.sample_rate = sample_rate;
        pcm_config.pcm_format = 2;

        ret = sc_audio_playback_stream_prepare(playback_handle, &pcm_config, playback_state_cb, NULL);
    }

    if (SC_ERR_SUCCESS != ret)
    {
        error_line = __LINE__;
        goto exit;
    }

    ret = sc_audio_playback_push_stream(playback_handle, (void *)pcm_data, data_size);
    if (SC_ERR_SUCCESS != ret)
    {
        error_line = __LINE__;
        goto exit;
    }

    ret = sc_audio_playback_play(playback_handle);
    if (SC_ERR_SUCCESS != ret)
    {
        error_line = __LINE__;
        goto exit;
    }

    return 0;

exit:
    if (SC_AUDIO_INVALID_HANDLE != playback_handle)
    {
        sc_audio_playback_close(playback_handle);
    }

    LYINFLOG("player_playback_play_pcmbuf error_line=%d\n",error_line);
    return -1;
}

/********************************************************************
* @brief: qser_AudRecorder_StartRecord_Custom, used to capture audio from a file and play it
* @param file [IN]: char*, the path of the audio file
* @param period_size [IN]: int, the period size of the PCM buffer
* @param period_count [IN]: int, the period count of the PCM buffer
* @param num_channels [IN]: int, the number of channels in the audio
* @param sample_rate [IN]: int, the sample rate of the audio
* @return : int, returns 0 if successful, returns -1 if failed
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_AudRecorder_StartRecord_Custom(char *file, int period_size, int period_count, \
                        int num_channels, int sample_rate)
{
    int error_line;
    sc_audio_pcm_config_t pcm_config;
    int ret = 0;

    if(NULL == file || 0 == strlen(file))
    {
        error_line = __LINE__;
        goto exit;
    }

    capture_handle = sc_audio_capture_open(SC_AUDIO_FE_PCM_DEV_MULTIMEDIA1, SC_AUDIO_BE_DAI_MIN);

    if (SC_AUDIO_INVALID_HANDLE == capture_handle)
    {
        error_line = __LINE__;
        goto exit;
    }

    memset(&pcm_config, 0, sizeof(sc_audio_pcm_config_t));

    if(-1 == period_size || -1 == period_count || -1 == num_channels || -1 == sample_rate)
    {
        ret = sc_audio_capture_file_prepare(capture_handle, file, SC_AUDIO_STREAM_FORMAT_INVALID, \
                        NULL, capture_state_cb, NULL);
    }
    else
    {
        pcm_config.period_size = period_size;
        pcm_config.period_count = period_count;
        pcm_config.flags = 0;
        pcm_config.num_channels = num_channels;
        pcm_config.sample_rate = sample_rate;
        pcm_config.pcm_format = 2;
        ret = sc_audio_capture_file_prepare(capture_handle, file, SC_AUDIO_STREAM_FORMAT_INVALID, \
                        &pcm_config, capture_state_cb, NULL);
    }

    if (SC_ERR_SUCCESS != ret)
    {
        error_line = __LINE__;
        goto exit;
    }

    ret = sc_audio_capture_record(capture_handle);

    if (SC_ERR_SUCCESS != ret)
    {
        error_line = __LINE__;
        goto exit;
    }

    return 0;

exit:
    if (SC_AUDIO_INVALID_HANDLE != capture_handle)
    {
        sc_audio_capture_close(capture_handle);
    }
    LYINFLOG("player_capture_play_file error_line=%d\n", error_line);
    return -1;
}

/********************************************************************
* @brief: qser_AudPlayer_Pause, pause the audio playback
* @param hdl [IN]: int, handle for the audio device or stream
* @return : success 0, failed -1
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_AudPlayer_Pause(int hdl)
{
    if (SC_AUDIO_INVALID_HANDLE == playback_handle)
    {
        LYINFLOG("qser_AudPlayer_Pause handle is invalid.\n");
        return -1;
    }
    if( sc_audio_playback_pause(playback_handle))
    {
	LYINFLOG("qser_AudPlayer_Pause sc_audio_playback_pause fail.\n");
        return -1;
    }
    return 0;
}

/********************************************************************
* @brief: qser_AudPlayer_Resume, resume the audio playback
* @param hdl [IN]: int, handle for the audio device or stream
* @return : success 0, failed -1
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_AudPlayer_Resume(int hdl)
{
    if (SC_AUDIO_INVALID_HANDLE == playback_handle)
    {
        LYINFLOG("qser_AudPlayer_Resume handle is invalid.\n");
        return -1;
    }
    if( sc_audio_playback_resume(playback_handle))
    {
        LYINFLOG("qser_AudPlayer_Resume sc_audio_playback_resume fail.\n");
        return -1;
   }
    return 0;
}

/********************************************************************
* @brief: qser_AudPlayer_Stop, stop the audio playback
* @param hdl [IN]: int, handle for the audio device or stream
* @return : void
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
void qser_AudPlayer_Stop(int hdl)
{
    if (SC_AUDIO_INVALID_HANDLE == playback_handle)
    {
	LYINFLOG("qser_AudPlayer_Stop handle is invalid.\n");
	return;
    }

    if( sc_audio_playback_stop(playback_handle))
    {
        LYINFLOG("qser_AudPlayer_Stop sc_audio_playback_stop fail.\n");
        return;
    }
}

/********************************************************************
* @brief: qser_AudPlayer_Close, close the audio playback
* @param hdl [IN]: int, handle for the audio device or stream
* @return : void
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
void qser_AudPlayer_Close(int hdl)
{
    if (SC_AUDIO_INVALID_HANDLE == playback_handle)
    {
        LYINFLOG("qser_AudPlayer_Close handle is invalid.\n");
        return;
    }
    sc_audio_playback_state_e state;
    if (sc_audio_playback_get_state(playback_handle, &state) == SC_ERR_SUCCESS)
    {
        if (state != SC_AUDIO_PLAYBACK_STATE_STOP && state != SC_AUDIO_PLAYBACK_STATE_IDLE)
        {
            if (sc_audio_playback_stop(playback_handle))
            {
                LYINFLOG("qser_AudPlayer_Close sc_audio_playback_stop fail.\n");
            }
        }
    }

    if (sc_audio_playback_close(playback_handle))
    {
        LYINFLOG("qser_AudPlayer_Close sc_audio_playback_close fail.\n");
        return;
    }

    playback_handle = SC_AUDIO_INVALID_HANDLE;
}

/********************************************************************
* @brief: Set uplink codec volume level
* @param volume [IN]: Volume level (0-based index, range: 0-11)
* @return : 0 if success, -1 for invalid parameters
*********************************************************************/
int qser_audio_set_codec_up_vol(int volume)
{
    if (volume < 0 || volume > 11) {
        LYINFLOG("%s: invalid volume=%d (valid range: 0-11)", __func__, volume);
        return -1;
    }

    int ret = sc_audio_set_codec_up_vol(volume);
    LYINFLOG("%s: %s volume=%d", __func__, (ret == 0) ? "success" : "failed", volume);
    return ret;
}

/********************************************************************
* @brief: Get current uplink codec volume level
* @param p_volume [OUT]: Pointer to store volume level
* @return : 0 if success, -1 for null pointer
*********************************************************************/
int qser_audio_get_codec_up_vol(int *p_volume)
{
    if (!p_volume) {
        LYINFLOG("%s: null pointer", __func__);
        return -1;
    }
    int ret = sc_audio_get_codec_up_vol(p_volume);
    LYINFLOG("%s: %s vol=%d", __func__, (ret == 0) ? "success" : "failed", *p_volume);
    return ret;
}

/********************************************************************
* @brief: Set downlink codec volume level
* @param volume [IN]: Volume level (0-based index, range: 0-11)
* @return : 0 if success, -1 for invalid parameters
*********************************************************************/
int qser_audio_set_codec_down_vol(int volume)
{
    if (volume < 0 || volume > 11) {
        LYINFLOG("%s: invalid volume=%d (valid range: 0-11)", __func__, volume);
        return -1;
    }

    int ret = sc_audio_set_codec_down_vol(volume);
    LYINFLOG("%s: %s volume=%d", __func__, (ret == 0) ? "success" : "failed", volume);
    return ret;
}

/********************************************************************
* @brief: Get current downlink codec volume level
* @param p_volume [OUT]: Pointer to store volume level
* @return : 0 if success, -1 for null pointer
*********************************************************************/
int qser_audio_get_codec_down_vol(int *p_volume)
{
    if (!p_volume) {
        LYINFLOG("%s: null pointer", __func__);
        return -1;
    }
    int ret = sc_audio_get_codec_down_vol(p_volume);
    LYINFLOG("%s: %s vol=%d", __func__, (ret == 0) ? "success" : "failed", *p_volume);
    return ret;
}

/********************************************************************
* @brief: qser_AudRecorder_Open, open the audio device for recording
* @param device [IN]: char* device, the audio device to be opened for recording
* @param cb_fun [IN]: _cb_onPlayer, callback function to be called when the audio device is opened
* @return : int, 0 if successful, non-zero if failed
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_AudRecorder_Open(char* device, _cb_onPlayer cb_fun)
{
    int ret = SC_ERR_SUCCESS;
    int retry_cnt = 0;
    int audio_is_init = 0;

    sc_audio_fe_pcm_dev_e device_enum = get_device_enum(device); // Convert device string to enum

    while (AUDIO_INIT_MAX_TRY_CNT > retry_cnt)
    {
        ret = sc_audio_init();
        if (SC_ERR_NOT_READY == ret)
        {
            LYINFLOG("audio service is not ready, try again, try count = %d\n", (retry_cnt + 1));
            usleep(200 * 1000);
            retry_cnt++;
        }
        else if (SC_ERR_SUCCESS == ret)
        {
            LYINFLOG("Success to initialize audio service\n");
            audio_is_init = 1;
            break;
        }
        else
        {
            LYINFLOG("Failed to initialize audio service, ret = %d\n", ret);
            break;
        }
    }
    if (1 != audio_is_init)
    {
        LYINFLOG("Failed to initialize audio service\n");
        if (cb_fun != NULL)
        {
            cb_fun(-1);
        }
        return -1;
    }

    sc_audio_handle_t device_handle = sc_audio_playback_open(device_enum, SC_AUDIO_FE_PCM_DEV_MIN, SC_AUDIO_OWNER_ID_PLAYER);
    if (SC_AUDIO_INVALID_HANDLE == device_handle)
    {
        LYINFLOG("Failed to open device: %s\n", device);
        if (cb_fun != NULL)
        {
            cb_fun(-1);
        }
        return -1;
    }

    if (cb_fun != NULL)
    {
        cb_fun(0);
    }
    return 0;
}

/********************************************************************
* @brief: qser_AudRecorder_StartRecord, play a file with capture
* @param hdl [IN]: int, handle for the audio device or stream
* @param fd [IN]: const char*, file descriptor of the audio file
* @param offset [IN]: int, offset in the audio file
* @return : int, 0 if successful, -1 if failed
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_AudRecorder_StartRecord(int hdl, const char *fd, int offset)
{
    int error_line;
    sc_audio_pcm_config_t pcm_config;
    int ret = 0;
    sc_audio_owner_id owner_id = hdl;

    if(NULL== fd || 0 == strlen(fd))
    {
        error_line = __LINE__;
        goto exit;
    }

    capture_handle = sc_audio_capture_open(SC_AUDIO_FE_PCM_DEV_MULTIMEDIA1,SC_AUDIO_BE_DAI_MIN);
    if (SC_AUDIO_INVALID_HANDLE == capture_handle)
    {
        error_line = __LINE__;
        goto exit;
    }

    memset(&pcm_config, 0, sizeof(sc_audio_pcm_config_t));

    ret = sc_audio_capture_file_prepare(capture_handle, fd, SC_AUDIO_STREAM_FORMAT_INVALID, NULL, capture_state_cb, NULL);
    if (SC_ERR_SUCCESS != ret)
    {
        error_line = __LINE__;
        goto exit;
    }

    ret = sc_audio_capture_record(capture_handle);
    if (SC_ERR_SUCCESS != ret)
    {
        error_line = __LINE__;
        goto exit;
    }

    return 0;
exit:
    LYINFLOG("qser_AudRecorder_StartRecord error_line=%d\n",error_line);
    return -1;
}

/********************************************************************
* @brief: qser_AudRecorder_Pause, pause the audio capture
* @return : int, 0 if successful, -1 if failed
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_AudRecorder_Pause(void)
{
    if (SC_AUDIO_INVALID_HANDLE == capture_handle)
    {
        LYINFLOG("qser_AudRecorder_Pause capture_handle is invalid.\n");
        return -1;
    }
    if( sc_audio_capture_pause(capture_handle))
    {
	LYINFLOG("qser_AudRecorder_Pause sc_audio_capture_pause fail.\n");
	return -1;
    }
    return 0;
}

/********************************************************************
* @brief: qser_AudRecorder_Resume, resume the audio capture
* @return : int, 0 if successful, -1 if failed
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_AudRecorder_Resume(void)
{
    if (SC_AUDIO_INVALID_HANDLE == capture_handle)
    {
        LYINFLOG("qser_AudRecorder_Resume capture_handle is invalid.\n");
        return -1;
    }
    if( sc_audio_capture_resume(capture_handle))
    {
        LYINFLOG("qser_AudRecorder_Resume sc_audio_capture_resume fail.\n");
        return -1;
    }
    return 0;
}

/********************************************************************
* @brief: qser_AudRecorder_Stop, stop the audio capture
* @return : void
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
void qser_AudRecorder_Stop(void)
{
    if (SC_AUDIO_INVALID_HANDLE == capture_handle)
    {
        LYINFLOG("qser_AudRecorder_Stop capture_handle is invalid.\n");
        return;
    }
    if( sc_audio_capture_stop(capture_handle))
    {
        LYINFLOG("qser_AudRecorder_Stop sc_audio_capture_stop fail.\n");
        return;
    }
}

/********************************************************************
* @brief: qser_AudRecorder_Close, close the audio capture
* @return : void
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
void qser_AudRecorder_Close(void)
{
    if (SC_AUDIO_INVALID_HANDLE == capture_handle)
    {
        LYINFLOG("qser_AudRecorder_Close capture_handle is invalid.\n");
        return;
    }
    sc_audio_capture_state_e state;
    if (sc_audio_capture_get_state(capture_handle, &state) == SC_ERR_SUCCESS)
    {
        if (state != SC_AUDIO_CAPTURE_STATE_STOP && state != SC_AUDIO_CAPTURE_STATE_IDLE)
        {
            if (sc_audio_capture_stop(capture_handle))
            {
                LYINFLOG("qser_AudRecorder_Close sc_audio_capture_stop fail.\n");
            }
        }
    }
    if( sc_audio_capture_close(capture_handle))
    {
        LYINFLOG("qser_AudRecorder_Close sc_audio_capture_close fail.\n");
        return;
    }

    capture_handle = SC_AUDIO_INVALID_HANDLE;
}

/********************************************************************
* @brief: qser_Audio_Deinit, deinitialize the audio system, stop and close any active playback or capture,
          and uninitialize the audio system
* @return : void
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
void qser_Audio_Deinit(void)
{
    if(SC_AUDIO_INVALID_HANDLE != playback_handle)
    {
        sc_audio_playback_stop(playback_handle);
        sc_audio_playback_close(playback_handle);
        playback_handle = SC_AUDIO_INVALID_HANDLE;
    }

    if(SC_AUDIO_INVALID_HANDLE != capture_handle)
    {
        sc_audio_capture_stop(capture_handle);
        sc_audio_capture_close(capture_handle);
        capture_handle = SC_AUDIO_INVALID_HANDLE;
    }

    sc_audio_uninit();
}

DEFINE_LYNQ_LIB_LOG(LYNQ_AUDIO)

#ifdef __cplusplus
}
#endif
