#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include "liblog/lynq_deflog.h"

#ifdef __cplusplus
extern "C" {
#endif

#include <cfg_api.h>
#include "sc_tel_types.h"
#include "sc_wifi.h"
#include "lynq-qser-wifi.h"
#define WIFI_ENABLE_FLAG_TIME_OUT 10
#define WIFI_ENABLE_FLAG_DEFAULT -100

static lynq_wifi_event_handle wifi_event_handle = NULL;
static lynq_wifi_event_handle_sta wifi_event_handle_sta = NULL;
static void *global_arg = NULL;
static int wifi_enable_flag = WIFI_ENABLE_FLAG_DEFAULT;
static bool lynq_wifi_enable = false;
#define WIFI_ENABLE_JUDGE() do{ \
    if(lynq_wifi_enable == false) \
    { \
        LYINFLOG("[%s ] wifi not enable\n", __func__); \
        return -1;  \
    } \
}while (0);

static lynq_wifi_event_handle wifi2_event_handle = NULL;
static lynq_wifi_event_handle_sta wifi2_event_handle_sta = NULL;
static void *global_arg_2 = NULL;
static int wifi2_enable_flag = WIFI_ENABLE_FLAG_DEFAULT;
static bool lynq_wifi2_enable = false;
#define WIFI2_ENABLE_JUDGE() do{ \
    if(lynq_wifi2_enable== false) \
    { \
        LYINFLOG("[%s ] wifi 2 not enable\n", __func__); \
        return -1;  \
    } \
}while (0);

/********************************************************************
* @brief: lynq_to_sc_auth_mode, The encryption mode of wifi is changed from api to platform
* @return :sc_wifi_auth_e, all
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
static sc_wifi_auth_e lynq_to_sc_auth_mode(lynq_wifi_auth_e auth_mode)
{
    sc_wifi_auth_e type;
    switch (auth_mode)
    {
        case LYNQ_WIFI_AUTH_OPEN:
            type = SC_WIFI_AUTH_OPEN;
            break;
        case LYNQ_WIFI_AUTH_WPA2_PSK:
            type = SC_WIFI_AUTH_WPA2_PSK;
            break;
        case LYNQ_WIFI_AUTH_WPA_WPA2_PSK_BOTH:
            type = SC_WIFI_AUTH_WPA_WPA2_PSK_BOTH;
            break;
        case LYNQ_WIFI_AUTH_WPA3_PSK:
            type = SC_WIFI_AUTH_WPA3_PSK;
            break;
        case LYNQ_WIFI_AUTH_WPA2_WPA3_PSK_BOTH:
            type = SC_WIFI_AUTH_WPA2_WPA3_PSK_BOTH;
            break;
        default:
            type = SC_WIFI_AUTH_MIN;
            break;
    }
    return type;
}

/********************************************************************
* @brief: sc_to_lynq_auth_mode, The wifi protocol moves from platform to api
* @return :lynq_wifi_auth_e, all
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
static lynq_wifi_auth_e sc_to_lynq_auth_mode(sc_wifi_auth_e auth_mode)
{
    lynq_wifi_auth_e type;
    switch (auth_mode)
    {
        case SC_WIFI_AUTH_OPEN:
            type = LYNQ_WIFI_AUTH_OPEN;
            break;
        case SC_WIFI_AUTH_WPA2_PSK:
            type = LYNQ_WIFI_AUTH_WPA2_PSK;
            break;
        case SC_WIFI_AUTH_WPA_WPA2_PSK_BOTH:
            type = LYNQ_WIFI_AUTH_WPA_WPA2_PSK_BOTH;
            break;
        case SC_WIFI_AUTH_WPA3_PSK:
            type = LYNQ_WIFI_AUTH_WPA3_PSK;
            break;
        case SC_WIFI_AUTH_WPA2_WPA3_PSK_BOTH:
            type = LYNQ_WIFI_AUTH_WPA2_WPA3_PSK_BOTH;
            break;
        default:
            type = LYNQ_WIFI_AUTH_MIN;
            break;
    }
    return type;
}

/********************************************************************
* @brief: lynq_to_sc_mode, The wifi protocol is transferred from api to platform
* @return :sc_wifi_ap_mode_type_e, all
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
static sc_wifi_ap_mode_type_e lynq_to_sc_mode(lynq_wifi_mode_type_e mode)
{
    sc_wifi_ap_mode_type_e type;
    switch (mode)
    {
        case LYNQ_WIFI_MODE_80211BGN:
            type = SC_WIFI_AP_MODE_80211BGN;
            break;
        case LYNQ_WIFI_MODE_80211BGNAX_2G:
            type = SC_WIFI_AP_MODE_80211BGNAX_2G;
            break;
        case LYNQ_WIFI_MODE_80211AN:
            type = SC_WIFI_AP_MODE_80211AN;
            break;
        case LYNQ_WIFI_MODE_80211ANAC:
            type = SC_WIFI_AP_MODE_80211ANAC;
            break;
        case LYNQ_WIFI_MODE_80211ANACAX_5G:
            type = SC_WIFI_AP_MODE_80211ANACAX_5G;
            break;
        default:
            type = SC_WIFI_AP_MODE_MIN;
            break;
    }
    return type;
}

/********************************************************************
* @brief: sc_to_lynq_mode, The encryption mode of wifi is changed from platform to api
* @return :lynq_wifi_mode_type_e, all
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
static lynq_wifi_mode_type_e sc_to_lynq_mode(sc_wifi_ap_mode_type_e mode)
{
    lynq_wifi_mode_type_e type;
    switch (mode)
    {
        case SC_WIFI_AP_MODE_80211BGN:
            type = LYNQ_WIFI_MODE_80211BGN;
            break;
        case SC_WIFI_AP_MODE_80211BGNAX_2G:
            type = LYNQ_WIFI_MODE_80211BGNAX_2G;
            break;
        case SC_WIFI_AP_MODE_80211AN:
            type = LYNQ_WIFI_MODE_80211AN;
            break;
        case SC_WIFI_AP_MODE_80211ANAC:
            type = LYNQ_WIFI_MODE_80211ANAC;
            break;
        case SC_WIFI_AP_MODE_80211ANACAX_5G:
            type = LYNQ_WIFI_MODE_80211ANACAX_5G;
            break;
        default:
            type = LYNQ_WIFI_MODE_MIN;
            break;
    }
    return type;
}

/********************************************************************
* @brief: sc_to_lynq_sta_status, sc_wifi_sta_status_t to lynq_wifi_sta_status_t
* @return :int, all
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
static int sc_to_lynq_sta_status(sc_wifi_sta_status_t *stat, lynq_wifi_sta_status_t *status_stat)
{
    status_stat->status = (lynq_wifi_sta_status_e)stat->status;
    status_stat->signal_level = stat->signal_level;
    status_stat->has_addr = stat->has_addr;
    status_stat->has_addr6 = stat->has_addr6;
    status_stat->reason_code = (lynq_wifi_reason_code_e)stat->reason_code;
    strncpy(status_stat->ifname, stat->ifname, sizeof(stat->ifname) - 1);
    strncpy(status_stat->ap_bssid, stat->ap_bssid, sizeof(stat->ap_bssid) - 1);
    if (status_stat->has_addr == 1)
    {
        strncpy(status_stat->addr.addr, stat->addr.addr, sizeof(status_stat->addr.addr) - 1);
        strncpy(status_stat->addr.netmask, stat->addr.netmask, sizeof(status_stat->addr.netmask) - 1);
        status_stat->addr.subnet_bits = stat->addr.subnet_bits;
        strncpy(status_stat->addr.gateway, stat->addr.gateway, sizeof(status_stat->addr.gateway) - 1);
        strncpy(status_stat->addr.dnsp, stat->addr.dnsp, sizeof(status_stat->addr.dnsp) - 1);
        strncpy(status_stat->addr.dnss, stat->addr.dnss, sizeof(status_stat->addr.dnss) - 1);
    }
    if (status_stat->has_addr6 == 1)
    {
        strncpy(status_stat->addr6.addr, stat->addr6.addr, sizeof(status_stat->addr6.addr) - 1);
        strncpy(status_stat->addr6.prefix, stat->addr6.prefix, sizeof(status_stat->addr6.prefix) - 1);
        status_stat->addr6.prefix_bits = stat->addr6.prefix_bits;
        strncpy(status_stat->addr6.gateway, stat->addr6.gateway, sizeof(status_stat->addr6.gateway) - 1);
        strncpy(status_stat->addr6.dnsp, stat->addr6.dnsp, sizeof(status_stat->addr6.dnsp) - 1);
        strncpy(status_stat->addr6.dnss, stat->addr6.dnss, sizeof(status_stat->addr6.dnss) - 1);
    }
    return 0;
}

/********************************************************************
* @brief: lynq_user_status, wifi startup callback
* @return : NA
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
static void lynq_user_status(sc_wifi_enable_status_e pre_status, sc_wifi_enable_status_e status)
{
    wifi_enable_flag = status;
    LYINFLOG("%s:%d,%d wifi_enable_flag:%d\n", __func__, pre_status, status, wifi_enable_flag);
}
static void lynq_user2_status(sc_wifi_enable_status_e pre_status, sc_wifi_enable_status_e status)
{
    wifi2_enable_flag = status;
    LYINFLOG("%s:%d,%d wifi2_enable_flag:%d\n", __func__, pre_status, status, wifi2_enable_flag);
}

/********************************************************************
* @brief: lynq_user_wifi_service_error, wifi service status callback
* @return : NA
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
static void lynq_user_wifi_service_error(int error)
{
	LYINFLOG("%s: %d\n", __func__, error);
}
static void lynq_user2_wifi_service_error(int error)
{
	LYINFLOG("%s: %d\n", __func__, error);
}

/********************************************************************
* @brief: lynq_user_ap_status, wifi Obtains the ap status callback
* @return : NA
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
static void lynq_user_ap_status(sc_wifi_ap_index_e index, sc_wifi_ap_status_e pre_status, sc_wifi_ap_status_t *p_msg)
{
    lynq_wifi_event_s event;
    if (wifi_event_handle != NULL && global_arg != NULL)
    {
        if (*(int *)global_arg == LYNQ_WIFI_EVENT_DISABLE_STATUS)
        {
            return;
        }
        LYINFLOG("%s:%d,%d,%s,%d\n", __func__, index, pre_status, p_msg->ifname, p_msg->status);
        event.id = LYNQ_WIFI_EVENT_AP_STATION;
        event.status = (lynq_wifi_status_e)p_msg->status;
        wifi_event_handle(&event, global_arg);
    }
    else
    {
        LYINFLOG("%s:%d,%d,%s,%d\n", __func__, index, pre_status, p_msg->ifname, p_msg->status);
    }
}
static void lynq_user2_ap_status(sc_wifi_ap_index_e index, sc_wifi_ap_status_e pre_status, sc_wifi_ap_status_t *p_msg)
{
    lynq_wifi_event_s event;
    if (wifi2_event_handle != NULL && global_arg_2 != NULL)
    {
        if (*(int *)global_arg_2 == LYNQ_WIFI_EVENT_DISABLE_STATUS)
        {
            return;
        }
        LYINFLOG("%s:%d,%d,%s,%d\n", __func__, index, pre_status, p_msg->ifname, p_msg->status);
        event.id = LYNQ_WIFI_EVENT_AP_STATION;
        event.status = (lynq_wifi_status_e)p_msg->status;
        wifi2_event_handle(&event, global_arg_2);
    }
    else
    {
        LYINFLOG("%s:%d,%d,%s,%d\n", __func__, index, pre_status, p_msg->ifname, p_msg->status);
    }
}

/********************************************************************
* @brief: lynq_user_ap_sta_conn_status, wifi as ap is sta information callback
* @return : NA
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
static void lynq_user_ap_sta_conn_status(sc_wifi_ap_index_e index, sc_wifi_sta_connect_status_t *p_msg)
{
    char hostname[32] = {0};
    lynq_wifi_event_s event;
    int ret = sc_wifi_get_hostname_by_mac(p_msg->macaddr, hostname, sizeof(hostname));
    if (ret == 0)
    {
        LYINFLOG("sta ip not assigned, try again later!\n");
    }
    if (wifi_event_handle != NULL && global_arg != NULL)
    {
        if (*(int *)global_arg == LYNQ_WIFI_EVENT_DISABLE_STATUS)
        {
            return;
        }
        LYINFLOG("%s:%d,%d,%s,%s\n", __func__, index, p_msg->is_connected, p_msg->macaddr, hostname);
        event.id = LYNQ_WIFI_EVENT_AP_STA_STATUS;
        event.ap_sta_info.connected = p_msg->is_connected;
        strncpy(event.ap_sta_info.mac, p_msg->macaddr,
                sizeof(event.ap_sta_info.mac) <= sizeof(p_msg->macaddr) ? sizeof(event.ap_sta_info.mac) : sizeof(p_msg->macaddr));
        strncpy(event.ap_sta_info.hostname, hostname, sizeof(hostname));
        wifi_event_handle(&event, global_arg);
    }
    else
    {
        LYINFLOG("%s:%d,%d,%s,%s\n", __func__, index, p_msg->is_connected, p_msg->macaddr, hostname);
    }
}
static void lynq_user2_ap_sta_conn_status(sc_wifi_ap_index_e index, sc_wifi_sta_connect_status_t *p_msg)
{
    char hostname[32] = {0};
    lynq_wifi_event_s event;
    int ret = sc_wifi_get_hostname_by_mac_ext(1, p_msg->macaddr, hostname, sizeof(hostname));
    if (ret == 0)
    {
        LYINFLOG("sta ip not assigned, try again later!\n");
    }
    if (wifi2_event_handle != NULL && global_arg_2 != NULL)
    {
        if (*(int *)global_arg_2 == LYNQ_WIFI_EVENT_DISABLE_STATUS)
        {
            return;
        }
        LYINFLOG("%s:%d,%d,%s,%s\n", __func__, index, p_msg->is_connected, p_msg->macaddr, hostname);
        event.id = LYNQ_WIFI_EVENT_AP_STA_STATUS;
        event.ap_sta_info.connected = p_msg->is_connected;
        strncpy(event.ap_sta_info.mac, p_msg->macaddr,
                sizeof(event.ap_sta_info.mac) <= sizeof(p_msg->macaddr) ? sizeof(event.ap_sta_info.mac) : sizeof(p_msg->macaddr));
        strncpy(event.ap_sta_info.hostname, hostname, sizeof(hostname));
        wifi2_event_handle(&event, global_arg_2);
    }
    else
    {
        LYINFLOG("%s:%d,%d,%s,%s\n", __func__, index, p_msg->is_connected, p_msg->macaddr, hostname);
    }
}
/********************************************************************
* @brief: print_sta_status, wifi gets the status callback in sta mode
* @return : NA
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
static void print_sta_status(sc_wifi_sta_status_t *p_msg)
{
    LYINFLOG("%s: %d, %s, %s, %d, %d\n", __func__, p_msg->status, p_msg->ifname, p_msg->ap_bssid,
             p_msg->signal_level, p_msg->reason_code);

    if (p_msg->has_addr == 1)
    {
        LYINFLOG("[%s]addr ip:%s, netmask:%s, subnet_bits:%d, gateway:%s, dnsp:%s, dnss:%s\n", __func__,
                 p_msg->addr.addr, p_msg->addr.netmask, p_msg->addr.subnet_bits,
                 p_msg->addr.gateway, p_msg->addr.dnsp, p_msg->addr.dnss);
    }

    if (p_msg->has_addr6 == 1)
    {
        LYINFLOG("[%s]addr6 ip:%s, prefix:%s, prefix_bits:%d, gateway:%s, dnsp:%s, dnss:%s\n", __func__,
                 p_msg->addr6.addr, p_msg->addr6.prefix, p_msg->addr6.prefix_bits,
                 p_msg->addr6.gateway, p_msg->addr6.dnsp, p_msg->addr6.dnss);
    }
    LYINFLOG("%s : sta_status end\n", __func__);
}

/********************************************************************
* @brief: lynq_user_sta_status_ind, wifi gets the status callback in sta mode
* @return : NA
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
static void lynq_user_sta_status_ind(sc_wifi_sta_status_e pre_status, 
  sc_wifi_sta_status_t *p_msg)
{	 
    if (wifi_event_handle != NULL && global_arg != NULL)
    {
        lynq_wifi_event_s event;
        if (*(int *)global_arg == LYNQ_WIFI_EVENT_DISABLE_STATUS)
        {
            return;
        }
        LYINFLOG("%s : user_sta_status_ind_cb pre:%d, cur:%d\n", __func__, pre_status, p_msg->status);
        event.id = LYNQ_WIFI_EVENT_STA_STATUS;
        if(p_msg->status == SC_WIFI_STA_STATUS_CONNECTED)
            event.sta_status = LYNQ_WIFI_STATION_CONNECTED;
        else if(p_msg->status == SC_WIFI_STA_STATUS_DISCONNECTED)
            event.sta_status = LYNQ_WIFI_STATION_DISCONNECTED;
        else
            event.sta_status = LYNQ_WIFI_STATION_DISABLE;
        sc_to_lynq_sta_status(p_msg, &event.sta_status_all);
        event.sta_status_all.sta_status = event.sta_status;
        wifi_event_handle(&event, global_arg);
    }
    else
    {
        print_sta_status(p_msg);
    }
	 
}
static void lynq_user2_sta_status_ind(sc_wifi_sta_status_e pre_status, 
  sc_wifi_sta_status_t *p_msg)
{	 
    if (wifi2_event_handle != NULL && global_arg_2 != NULL)
    {
        lynq_wifi_event_s event;
        if (*(int *)global_arg_2 == LYNQ_WIFI_EVENT_DISABLE_STATUS)
        {
            return;
        }
        LYINFLOG("%s : user_sta_status_ind_cb pre:%d, cur:%d\n", __func__, pre_status, p_msg->status);
        event.id = LYNQ_WIFI_EVENT_STA_STATUS;
        if(p_msg->status == SC_WIFI_STA_STATUS_CONNECTED)
            event.sta_status = LYNQ_WIFI_STATION_CONNECTED;
        else if(p_msg->status == SC_WIFI_STA_STATUS_DISCONNECTED)
            event.sta_status = LYNQ_WIFI_STATION_DISCONNECTED;
        else
            event.sta_status = LYNQ_WIFI_STATION_DISABLE;
        sc_to_lynq_sta_status(p_msg, &event.sta_status_all);
        event.sta_status_all.sta_status = event.sta_status;
        wifi2_event_handle(&event, global_arg_2);
    }
    else
    {
        print_sta_status(p_msg);
    }
	 
}
/********************************************************************
* @brief: lynq_user_sta_scan_result_ind, wifi gets the callback of sta mode traversing the hotspot
* @return : NA
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
static void lynq_user_sta_scan_result_ind(sc_wifi_sta_scan_list_t *p_msg)
{
    int i = 0;
    LYINFLOG("%s : user_sta_scan_result_ind_cb:%d\n", __func__, p_msg->cnt);
    if (p_msg->cnt <= 0)
    {
        return;
    }
    if (wifi_event_handle_sta != NULL)
    {
        lynq_wifi_sta_scan_list_t event;
        event.cnt = p_msg->cnt;
        for (i = 0; i < event.cnt; i++)
        {
            strncpy(event.info[i].essid, p_msg->info[i].essid, sizeof(p_msg->info[i].essid));
            strncpy(event.info[i].bssid, p_msg->info[i].bssid, sizeof(p_msg->info[i].bssid));
            event.info[i].auth = sc_to_lynq_auth_mode(p_msg->info[i].auth);
            event.info[i].cipher = (lynq_wifi_auth_wpa_psk_e)p_msg->info[i].cipher;
            event.info[i].channel = p_msg->info[i].channel;
            event.info[i].signal_level = p_msg->info[i].signal_level;
            event.info[i].frequency = p_msg->info[i].frequency;
            event.info[i].signal = p_msg->info[i].signal;
        }
        wifi_event_handle_sta(&event);
    }
    else
    {
        for (i = 0; i < p_msg->cnt; i++)
        {
            LYINFLOG("%s : ap[%d]:%s,%d,%d,%d,%s,%d,%d,%d\n", __func__, i, p_msg->info[i].essid, sc_to_lynq_auth_mode(p_msg->info[i].auth),
                     p_msg->info[i].cipher, p_msg->info[i].channel, p_msg->info[i].bssid, p_msg->info[i].signal_level,p_msg->info[i].frequency,p_msg->info[i].signal);
        }
    }
}
static void lynq_user2_sta_scan_result_ind(sc_wifi_sta_scan_list_t *p_msg)
{
    int i = 0;
    LYINFLOG("%s : user2_sta_scan_result_ind_cb:%d\n", __func__, p_msg->cnt);
    if (p_msg->cnt <= 0)
    {
        return;
    }
    if (wifi2_event_handle_sta != NULL)
    {
        lynq_wifi_sta_scan_list_t event;
        event.cnt = p_msg->cnt;
        for (i = 0; i < event.cnt; i++)
        {
            strncpy(event.info[i].essid, p_msg->info[i].essid, sizeof(p_msg->info[i].essid));
            strncpy(event.info[i].bssid, p_msg->info[i].bssid, sizeof(p_msg->info[i].bssid));
            event.info[i].auth = sc_to_lynq_auth_mode(p_msg->info[i].auth);
            event.info[i].cipher = (lynq_wifi_auth_wpa_psk_e)p_msg->info[i].cipher;
            event.info[i].channel = p_msg->info[i].channel;
            event.info[i].signal_level = p_msg->info[i].signal_level;
            event.info[i].frequency = p_msg->info[i].frequency;
            event.info[i].signal = p_msg->info[i].signal;
        }
        wifi2_event_handle_sta(&event);
    }
    else
    {
        for (i = 0; i < p_msg->cnt; i++)
        {
            LYINFLOG("%s : ap[%d]:%s,%d,%d,%d,%s,%d,%d,%d\n", __func__, i, p_msg->info[i].essid, sc_to_lynq_auth_mode(p_msg->info[i].auth),
                     p_msg->info[i].cipher, p_msg->info[i].channel, p_msg->info[i].bssid, p_msg->info[i].signal_level,p_msg->info[i].frequency,p_msg->info[i].signal);
        }
    }
}
/********************************************************************
* @brief: qser_wifi_work_mode_set, 2.4g or 5g working mode Settings
* @param type [IN]: lynq_wifi_work_mode_e, 2.4G or 5G working mode Settings
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi_work_mode_set(lynq_wifi_work_mode_e type)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_work_mode_e mode;
    ret = sc_wifi_work_mode_get(&mode);
    if (0 != ret)
    {
        LYERRLOG("[%s ] work_mode get ret = %d\n", __func__,ret);
        return ret;
    }
    if(mode == (sc_wifi_work_mode_e)type) //The same value is returned
    {
        return 0;
    }
    ret = sc_wifi_work_mode_set((sc_wifi_work_mode_e)type);
    if (0 != ret)
    {
        LYERRLOG("[%s ] work_mode set ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ]\n", __func__);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_work_mode_get, 2.4g or 5g working mode Gettings
* @param type [OUT]: lynq_wifi_work_mode_e, 2.4G or 5G working mode Gettings
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi_work_mode_get(lynq_wifi_work_mode_e *type)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_work_mode_e mode;
    ret = sc_wifi_work_mode_get(&mode);
    if (0 != ret)
    {
        LYERRLOG("[%s ] work_mode get ret = %d\n", __func__,ret);
        return ret;
    }
    *type = (lynq_wifi_work_mode_e)mode;
    LYINFLOG("[%s ]\n", __func__);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_register_handle, Register callback functions
* @param event_handle [IN]: lynq_wifi_event_handle, Register the ap event callback function
* @param event_handle_sta [IN]: Register sta's event callback function
* @param arg [IN]: void *, Not currently used, but cannot pass a null pointer
* @return :int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi_register_handle(lynq_wifi_event_handle event_handle, lynq_wifi_event_handle_sta event_handle_sta, void *arg)
{
    if((event_handle == NULL && event_handle_sta == NULL) || arg == NULL)
    {
        LYERRLOG("[%s ] NUll pointer event_handle = 0x%p arg = 0x%p\n", __func__, event_handle, arg);
        return -1;
    }
    if(*(int *)arg <= LYNQ_WIFI_EVENT_MIN || *(int *)arg >= LYNQ_WIFI_EVENT_MAX)
    {
        LYERRLOG("[%s ] The value of arg can only be an integer ranging from 0 to 3p, arg = %d\n", __func__, arg);
        return -1;
    }
    wifi_event_handle_sta = event_handle_sta;
    wifi_event_handle = event_handle;
    global_arg = arg;
    return 0;
}

/********************************************************************
* @brief: qser_wifi_enable, Enable WiFi function
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi_enable(void)
{
    if(lynq_wifi_enable)
    {
        LYERRLOG("[%s ] The wifi is enabled\n", __func__);
        return -1;
    }
    wifi_enable_flag = WIFI_ENABLE_FLAG_DEFAULT;
    char wifiAvailable[8] = {0};
    sc_cfg_get("wifiAvailable", wifiAvailable, sizeof(wifiAvailable));
    if (!strcmp(wifiAvailable, "0"))
    {
        LYERRLOG("[%s ] wifiAvailable has been set to 0. If WiFi is used, set 1 to enable it\n", __func__);
        return -1;
    }
    int ret = -1;
    ret = sc_wifi_init();
    if (0 != ret)
    {
        LYERRLOG("[%s ] init wifi ret = %d fail\n", __func__,ret);
        return ret;
    }

    //wifi
    ret = sc_wifi_set_enable_status_ind_cb(lynq_user_status);
    if (0 != ret)
    {
        LYERRLOG("[%s ] Request lynq_user_status  ret = %d fail\n", __func__,ret);
        return ret;
    }

    //ap
    ret = sc_wifi_ap_set_status_ind_cb(lynq_user_ap_status);
    if (0 != ret)
    {
        LYERRLOG("[%s ] Request lynq_user_ap_status  ret = %d fail\n", __func__,ret);
        return ret;
    }
    ret = sc_wifi_set_ap_sta_connect_ind_cb(lynq_user_ap_sta_conn_status);
    if (0 != ret)
    {
        LYERRLOG("[%s ] Request lynq_user_ap_sta_conn_status  ret = %d fail\n", __func__,ret);
        return ret;
    }

	//sta
    ret = sc_wifi_sta_set_status_ind_cb(lynq_user_sta_status_ind);
    if (0 != ret)
    {
        LYERRLOG("[%s ] Request lynq_user_sta_status_ind  ret = %d fail\n", __func__,ret);
        return ret;
    }
    ret = sc_wifi_sta_set_scan_result_ind_cb(lynq_user_sta_scan_result_ind);
    if (0 != ret)
    {
        LYERRLOG("[%s ] Request lynq_user_sta_scan_result_ind  ret = %d fail\n", __func__,ret);
        return ret;
    }
	//proxy
    ret = sc_wifi_set_service_error_cb(lynq_user_wifi_service_error);
    if (0 != ret)
    {
        LYERRLOG("[%s ] Request lynq_user_wifi_service_error  ret = %d fail\n", __func__,ret);
        return ret;
    }

    ret = sc_wifi_enable();
    if (0 != ret)
    {
        LYERRLOG("[%s ] enable wifi ret = %d fail\n", __func__,ret);
        return ret;
    }
    for(int i = 0; i < WIFI_ENABLE_FLAG_TIME_OUT; i++)
    {
        if(wifi_enable_flag == WIFI_ENABLE_FLAG_DEFAULT)
            sleep(1);
        else
            break;
    }
    switch (wifi_enable_flag)
    {
    case SC_WIFI_STATUS_DISABLED:
        ret = 1;
        break;
    case SC_WIFI_STATUS_ENABLED:
        ret = 0;
        break;
    case SC_WIFI_STATUS_INVALID_MAC:
        ret = 7;
        break;
    case SC_WIFI_STATUS_DEV_INIT_FAIL:
        ret = 6;
        break;
    case SC_WIFI_STATUS_FIRMWARE_CRASH:
        ret = 12;
        break;
    default:
        ret = -1;
        break;
    }
    LYINFLOG("[%s ] ret = %d\n", __func__,ret);
    if(ret != 0)
    {
        int ret_uninit = sc_wifi_uninit();
        if (0 != ret_uninit)
        {
            LYERRLOG("[%s ] uninit ret = %d\n", __func__,ret_uninit);
        }
    }
    else
        lynq_wifi_enable = true;
    return ret;
}

/********************************************************************
* @brief: qser_wifi_disable, Turn off WiFi
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi_disable(void)
{
    int ret = -1;
    wifi_enable_flag = WIFI_ENABLE_FLAG_DEFAULT;
    if(lynq_wifi_enable == false)
    {
        LYERRLOG("[%s ] WiFi not turned on\n", __func__);
        return -1;
    }
    ret = sc_wifi_disable();
    if (0 != ret)
    {
        LYERRLOG("[%s ] disable ret = %d\n", __func__,ret);
        return ret;
    }
    for(int i = 0; i < WIFI_ENABLE_FLAG_TIME_OUT; i++)
    {
        if(wifi_enable_flag == WIFI_ENABLE_FLAG_DEFAULT)
            sleep(1);
        else
            break;
    }
    switch (wifi_enable_flag)
    {
    case SC_WIFI_STATUS_DISABLED:
        ret = 0;
        break;
    case SC_WIFI_STATUS_INVALID_MAC:
        ret = 7;
        break;
    case SC_WIFI_STATUS_DEV_INIT_FAIL:
        ret = 6;
        break;
    case SC_WIFI_STATUS_FIRMWARE_CRASH:
        ret = 12;
        break;
    default:
        ret = -1;
        break;
    }
    if (0 != ret)
    {
        LYERRLOG("[%s ] disable ret = %d\n", __func__,ret);
        return ret;
    }
    lynq_wifi_enable = false;
    ret = sc_wifi_uninit();
    if (0 != ret)
    {
        LYERRLOG("[%s ] uninit ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ]\n", __func__);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_get_status, Example Query ap working status
* @param type [OUT]: lynq_wifi_ap_status_t *, ap working status
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi_ap_get_status(lynq_wifi_ap_index_e idx, lynq_wifi_ap_status_t *ap_stat)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_ap_status_t stat;
    ret = sc_wifi_ap_get_status((sc_wifi_ap_index_e)idx, &stat);
    if (0 != ret)
    {
        LYERRLOG("[%s ] wifi_ap_get  ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d ifname = %s status = %d bssid = %s \n", __func__, idx, stat.ifname, stat.status, stat.bssid);
    strncpy(ap_stat->ifname, stat.ifname, sizeof(stat.ifname) - 1);
    ap_stat->status = (lynq_wifi_status_e)stat.status;
    strncpy(ap_stat->bssid, stat.bssid, sizeof(stat.bssid) - 1);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_acl_set, Set the WiFi2.4G or 5G whitelist
* @param idx [IN]: int, Set 2.4G or 5G
* @param acl_rule [IN]: lynq_wifi_mac_acl_rule_e, Set the blacklist and whitelist mode
* @param mac_list [IN]: char *, Set the mac address of the whitelist
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi_ap_acl_set(lynq_wifi_ap_index_e idx, lynq_wifi_mac_acl_rule_e acl_rule, char *mac_list)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    if (mac_list == NULL)
    {
        LYERRLOG("[%s ] Null pointer mac_list = 0x%p \n", __func__, mac_list);
        return ret;
    }
    if(acl_rule == LYNQ_WIFI_MAC_ACL_RULE_BLACK)
        ret = sc_wifi_ap_acl_set((sc_wifi_ap_index_e)idx, SC_WIFI_MAC_ACL_RULE_WHITE, mac_list);
    ret = sc_wifi_ap_acl_set((sc_wifi_ap_index_e)idx, (sc_wifi_mac_acl_rule_e)acl_rule, mac_list);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d acl_rule = %d mac_list = %s \n", __func__, idx, acl_rule, mac_list);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_acl_get, Get the WiFi2.4G or 5G whitelist
* @param idx [IN]: int, Set 2.4G or 5G
* @param acl_rule [OUT]: lynq_wifi_mac_acl_rule_e *, Get the blacklist and whitelist mode
* @param mac_list [OUT]: char *, Get the mac address of the whitelist
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi_ap_acl_get(lynq_wifi_ap_index_e idx, lynq_wifi_mac_acl_rule_e *acl_rule, char *mac_list)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_ap_param_t param = {0};
    if (mac_list == NULL || acl_rule == NULL)
    {
        LYERRLOG("[%s ] Null pointer acl_rule = 0x%p mac_list = 0x%p \n", __func__, acl_rule, mac_list);
        return ret;
    }
    ret = sc_wifi_ap_param_get((sc_wifi_ap_index_e)idx, &param);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    *acl_rule = (lynq_wifi_mac_acl_rule_e)param.acl_rule;
    strncpy(mac_list, param.mac_list, sizeof(param.mac_list) - 1);
    LYINFLOG("[%s ] idx = %d acl_rule = %d mac_list = %s \n", __func__, idx, param.acl_rule, param.mac_list);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_ssid_set, Set the name of the ssid of 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param ssid [IN]: const char *, Set the ssid name
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi_ap_ssid_set(lynq_wifi_ap_index_e idx, const char *ssid)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    if (ssid == NULL)
    {
        LYERRLOG("[%s ] Null pointer ssid = 0x%p \n", __func__, ssid);
        return ret;
    }
    ret = sc_wifi_ap_ssid_set((sc_wifi_ap_index_e)idx, ssid);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d ssid = %s \n", __func__, idx, ssid);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_ssid_get, Get the name of the ssid of 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param ssid [OUT]: char *, Get the ssid name
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi_ap_ssid_get(lynq_wifi_ap_index_e idx, char *ssid)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_ap_param_t param = {0};
    ret = sc_wifi_ap_param_get((sc_wifi_ap_index_e)idx, &param);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d ssid = %s \n", __func__, idx, param.ssid);
    strncpy(ssid, param.ssid, sizeof(param.ssid) - 1);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_ssid_hide_set, Set whether the ssid of 2.4G or 5G is hidden
* @param idx [IN]: int, Set 2.4G or 5G
* @param hide [IN]: bool, Set whether the ssid is hidden
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi_ap_ssid_hide_set(lynq_wifi_ap_index_e idx, bool hide)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    ret = sc_wifi_ap_ssid_hidden_set((sc_wifi_ap_index_e)idx, (int)hide);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d hide = %d \n", __func__, idx, hide);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_ssid_hide_get, Get whether the ssid of 2.4G or 5G is hidden
* @param idx [IN]: int, Set 2.4G or 5G
* @param hide [OUT]: bool *, Get whether the ssid is hidden
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi_ap_ssid_hide_get(lynq_wifi_ap_index_e idx, bool *hide)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_ap_param_t param = {0};
    if (hide == NULL)
    {
        LYERRLOG("[%s ] Null pointer hide = 0x%p \n", __func__, hide);
        return ret;
    }
    ret = sc_wifi_ap_param_get((sc_wifi_ap_index_e)idx, &param);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d ssid_hide = %d \n", __func__, idx, param.ssid_hide);
    *hide = (bool)param.ssid_hide;
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_mode_set, Set the working protocol mode of 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param mode [IN]: lynq_wifi_mode_type_e, Set the working protocol mode
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi_ap_mode_set(lynq_wifi_ap_index_e idx, lynq_wifi_mode_type_e mode)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    ret = sc_wifi_ap_mode_set((sc_wifi_ap_index_e)idx, lynq_to_sc_mode(mode));
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d mode = %d \n", __func__, idx, mode);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_mode_get, Get the working protocol mode of 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param mode [OUT]: lynq_wifi_mode_type_e *, Get the working protocol mode
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi_ap_mode_get(lynq_wifi_ap_index_e idx, lynq_wifi_mode_type_e *mode)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_ap_param_t param = {0};
    if (mode == NULL)
    {
        LYERRLOG("[%s ] Null pointer mode = 0x%p \n", __func__, mode);
        return ret;
    }
    ret = sc_wifi_ap_param_get((sc_wifi_ap_index_e)idx, &param);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d mode = %d \n", __func__, idx, param.mode);
    *mode = sc_to_lynq_mode(param.mode);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_bandwidth_set, Set the bandwidth of 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param bandwidth [IN]: lynq_wifi_bandwidth_type_e, Set the bandwidth
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi_ap_bandwidth_set(lynq_wifi_ap_index_e idx, lynq_wifi_bandwidth_type_e bandwidth)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    ret = sc_wifi_ap_bandwidth_set((sc_wifi_ap_index_e)idx, (sc_wifi_bandwidth_e)bandwidth);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d bandwidth = %d \n", __func__, idx, bandwidth);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_bandwidth_get, Get the bandwidth of 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param bandwidth [OUT]: lynq_wifi_bandwidth_type_e *, Get the bandwidth
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi_ap_bandwidth_get(lynq_wifi_ap_index_e idx, lynq_wifi_bandwidth_type_e *bandwidth)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_ap_param_t param = {0};
    if (bandwidth == NULL)
    {
        LYERRLOG("[%s ] Null pointer bandwidth = 0x%p \n", __func__, bandwidth);
        return ret;
    }
    ret = sc_wifi_ap_param_get((sc_wifi_ap_index_e)idx, &param);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d bandwidth = %d \n", __func__, idx, param.bandwidth);
    *bandwidth = (lynq_wifi_bandwidth_type_e)param.bandwidth;
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_channel_set, Set the channel for 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param country_code [IN]: const char *, Set country code
* @param channel [IN]: int, Set the channel
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi_ap_channel_set(lynq_wifi_ap_index_e idx, const char *country_code, int channel)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    if (country_code == NULL)
    {
        LYERRLOG("[%s ] Null pointer country_code = 0x%p \n", __func__, country_code);
        return ret;
    }
    ret = sc_wifi_ap_cc_ch_set((sc_wifi_ap_index_e)idx, country_code, channel);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d country_code = %s channel = %d\n", __func__, idx, country_code, channel);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_channel_get, Get the channel for 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param country_code [OUT]: char *, Get country code
* @param channel [OUT]: int *, Get the channel
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi_ap_channel_get(lynq_wifi_ap_index_e idx, char *country_code, int *channel)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_ap_param_t param = {0};
    if (country_code == NULL || channel == NULL)
    {
        LYERRLOG("[%s ] Null pointer country_code = 0x%p channel = 0x%p\n", __func__, country_code, channel);
        return ret;
    }
    ret = sc_wifi_ap_param_get((sc_wifi_ap_index_e)idx, &param);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d country_code = %s channel = %d\n", __func__, idx, param.countrycode, param.channel);
    strncpy(country_code, param.countrycode, sizeof(param.countrycode) - 1);
    *channel = param.channel;
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_auth_set, Set the security authentication mode and password of 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param auth_mode [IN]: lynq_wifi_auth_e, Set the security authentication mode
* @param auth_passwd [IN]: const char *, Set password
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi_ap_auth_set(lynq_wifi_ap_index_e idx, lynq_wifi_auth_e auth_mode, const char *auth_passwd)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    if (auth_passwd == NULL)
    {
        LYERRLOG("[%s ] Null pointer auth_passwd = 0x%p\n", __func__, auth_passwd);
        return ret;
    }
    
    sc_wifi_ap_auth_t auth;
    auth.auth = lynq_to_sc_auth_mode(auth_mode);
    strncpy(auth.passwd, auth_passwd, sizeof(auth.passwd) - 1);
    ret = sc_wifi_ap_auth_set((sc_wifi_ap_index_e)idx, &auth);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d auth_mode = %d auth_passwd = %s\n", __func__, idx, auth_mode, "******");
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_auth_get, Get the security authentication mode and password of 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param auth_mode [OUT]: lynq_wifi_auth_e *, Get the security authentication mode
* @param auth_passwd [OUT]: char *, Get password
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi_ap_auth_get(lynq_wifi_ap_index_e idx, lynq_wifi_auth_e *auth_mode, char *auth_passwd)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_ap_param_t param = {0};
    if (auth_mode == NULL || auth_passwd == NULL)
    {
        LYERRLOG("[%s ] Null pointer auth_mode = 0x%p auth_passwd = 0x%p\n", __func__, auth_mode, auth_passwd);
        return ret;
    }
    ret = sc_wifi_ap_param_get((sc_wifi_ap_index_e)idx, &param);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d auth_mode = %d auth_passwd = %s \n", __func__, idx, param.auth.auth, param.auth.passwd);
    strncpy(auth_passwd, param.auth.passwd, sizeof(param.auth.passwd) - 1);
    
    *auth_mode = sc_to_lynq_auth_mode(param.auth.auth);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_auth_get_s, Get the security authentication mode , password group_rekey and  pairwise of 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param auth_mode [OUT]: lynq_wifi_ap_auth_t *, Get the security authentication mode
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi_ap_auth_get_s(lynq_wifi_ap_index_e idx, lynq_wifi_ap_auth_t *auth_mode)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_ap_param_t param = {0};
    if (auth_mode == NULL)
    {
        LYERRLOG("[%s ] Null pointer auth_mode = 0x%p\n", __func__, auth_mode);
        return ret;
    }
    ret = sc_wifi_ap_param_get((sc_wifi_ap_index_e)idx, &param);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d auth = %d passwd = %s group_rekey = %d pairwise = %d\n", __func__, idx, param.auth.auth, param.auth.passwd, param.auth.group_rekey, param.auth.pairwise);
    strncpy(auth_mode->passwd, param.auth.passwd, sizeof(param.auth.passwd) - 1);
    auth_mode->group_rekey = param.auth.group_rekey;
    auth_mode->pairwise = (lynq_wifi_auth_wpa_psk_e)param.auth.pairwise;
    auth_mode->auth = sc_to_lynq_auth_mode(param.auth.auth);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_max_sta_set, Set the maximum number of STAs for 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param max_sta_num [IN]: int, Set the maximum number
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi_ap_max_sta_set(lynq_wifi_ap_index_e idx, int max_sta_num)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    if(max_sta_num >= 32)
    {
        LYERRLOG("[%s ] fail max_sta_num = %d\n", __func__,max_sta_num);
        return -1;
    }
    ret = sc_wifi_ap_max_sta_num_set((sc_wifi_ap_index_e)idx, max_sta_num);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d max_sta_num = %d \n", __func__, idx, max_sta_num);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_max_sta_get, Get the maximum number of STAs for 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param max_sta_num [OUT]: int *, Get the maximum number
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi_ap_max_sta_get(lynq_wifi_ap_index_e idx, int *max_sta_num)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_ap_param_t param = {0};
    if (max_sta_num == NULL)
    {
        LYERRLOG("[%s ] Null pointer max_sta_num = 0x%p\n", __func__,max_sta_num);
        return ret;
    }
    ret = sc_wifi_ap_param_get((sc_wifi_ap_index_e)idx, &param);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d max_sta_num = %d \n", __func__, idx, param.max_sta_num);
    *max_sta_num = param.max_sta_num;
    return 0;
}

/********************************************************************
* @brief: qser_wifi_lanhost_get_list, Get sta device information received as an ap for 2.4G or 5G
* @param lynq_arrays [OUT]: lynq_lanhost_ts *, Get sta device information received as an ap
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi_lanhost_get_list(lynq_lanhost_ts *lynq_arrays)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    int i = 0;
    sc_lanhost_t array[32] = {0};
    if (lynq_arrays == NULL)
    {
        LYERRLOG("[%s ] Null pointer lynq_arrays = 0x%p\n", __func__, lynq_arrays);
        return ret;
    }
    lynq_arrays->array_len = 32;
    ret = sc_wifi_lanhost_get_list(array, &lynq_arrays->array_len);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__, ret);
        return ret;
    }
    LYINFLOG("[%s]ap_lanhost len[%d]\n", __func__, lynq_arrays->array_len);
    for (i = 0; i < lynq_arrays->array_len; i++)
    {
        LYINFLOG("[%s]Element : [%d] ifname = %s  macaddr = %s addr = %s name = %s uptime = %d\n", __func__, i,
                 array[i].ifname, array[i].macaddr, array[i].addr, array[i].name, array[i].uptime);
        strncpy(lynq_arrays->array[i].ifname, array[i].ifname, sizeof(lynq_arrays->array[i].ifname) - 1);
        strncpy(lynq_arrays->array[i].macaddr, array[i].macaddr, sizeof(lynq_arrays->array[i].macaddr) - 1);
        strncpy(lynq_arrays->array[i].addr, array[i].addr, sizeof(lynq_arrays->array[i].addr) - 1);
        strncpy(lynq_arrays->array[i].name, array[i].name, sizeof(lynq_arrays->array[i].name) - 1);
        lynq_arrays->array[i].uptime = array[i].uptime;
    }
    return 0;
}

/********************************************************************
* @brief: qser_wifi_get_ap_pkt_stats, Obtain data packets sent and received by ap for 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param pkt_stat [OUT]: lynq_wifi_pkt_stats_t *, Obtain data packets sent and received by ap
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi_get_ap_pkt_stats(lynq_wifi_ap_index_e idx, lynq_wifi_pkt_stats_t *pkt_stat)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_pkt_stats_t stat = {0};
    if (pkt_stat == NULL)
    {
        LYERRLOG("[%s ] Null pointer pkt_stat = 0x%p\n", __func__, pkt_stat);
        return ret;
    }
    idx = LYNQ_WIFI_AP_INDEX_AP0;
    ret = sc_wifi_get_ap_pkt_stats((sc_wifi_ap_index_e)idx, &stat);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__, ret);
        return ret;
    }
    LYINFLOG("[%s ]ap_pkt_get[%d] rx[%llu, %llu, %llu, %llu] tx[%llu, %llu, %llu, %llu]\n", __func__, idx,
             stat.rx_packets, stat.rx_bytes, stat.rx_errors, stat.rx_dropped,
             stat.tx_packets, stat.tx_bytes, stat.tx_errors, stat.tx_dropped);
    pkt_stat->rx_packets = stat.rx_packets;
    pkt_stat->rx_bytes = stat.rx_bytes;
    pkt_stat->rx_errors = stat.rx_errors;
    pkt_stat->rx_dropped = stat.rx_dropped;
    pkt_stat->tx_packets = stat.tx_packets;
    pkt_stat->tx_bytes = stat.tx_bytes;
    pkt_stat->tx_errors = stat.tx_errors;
    pkt_stat->tx_dropped = stat.tx_dropped;
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_start, Set the ap mode of 2.4G or 5G to enable
* @param idx [IN]: int, Set 2.4G or 5G
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi_ap_start(lynq_wifi_ap_index_e idx)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    ret = sc_wifi_ap_start((sc_wifi_ap_index_e)idx);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d \n", __func__, idx);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_stop, Disable ap mode for 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi_ap_stop(lynq_wifi_ap_index_e idx)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    ret = sc_wifi_ap_stop((sc_wifi_ap_index_e)idx);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d \n", __func__, idx);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_restart, Set the ap mode of 2.4G or 5G to restart
* @param idx [IN]: int, Set 2.4G or 5G
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi_ap_restart(lynq_wifi_ap_index_e idx)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    ret = sc_wifi_ap_stop((sc_wifi_ap_index_e)idx);
    if (0 != ret)
    {
        LYERRLOG("[%s ] stop ret = %d\n", __func__,ret);
        return ret;
    }
    ret = sc_wifi_ap_start((sc_wifi_ap_index_e)idx);
    if (0 != ret)
    {
        LYERRLOG("[%s ] start ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d \n", __func__, idx);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_sta_param_set, Set the ssid and password that you need to connect to the access point
* @param param_stat [IN]: sc_wifi_sta_param_t *, Set parameters such as ssid and password that you want to connect to the access point
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi_sta_param_set(lynq_wifi_sta_param_t *param_stat)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_sta_param_t stat = {0};
    stat.auth = lynq_to_sc_auth_mode(param_stat->auth);
    stat.pairwise = (sc_wifi_auth_wpa_psk_e )param_stat->pairwise;
    strncpy(stat.ssid, param_stat->ssid, sizeof(stat.ssid) - 1);
    strncpy(stat.passwd, param_stat->passwd, sizeof(stat.passwd) - 1);
    ret = sc_wifi_sta_param_set(&stat);
    if (0 != ret)
    {
        LYERRLOG("[%s ] sta_param_set ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] ret = %d \n", __func__, ret);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_sta_param_get, Get the ssid and password that you need to connect to the access point
* @param param_stat [OUT]: sc_wifi_sta_param_t *, Get parameters such as ssid and password that you want to connect to the access point
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi_sta_param_get(lynq_wifi_sta_param_t *param_stat)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_sta_param_t stat = {0};
    ret = sc_wifi_sta_param_get(&stat);
    if (0 != ret)
    {
        LYERRLOG("[%s ] sta_param_get ret = %d\n", __func__,ret);
        return ret;
    }
    param_stat->auth = sc_to_lynq_auth_mode(stat.auth);
    param_stat->pairwise = (lynq_wifi_auth_wpa_psk_e )stat.pairwise;
    strncpy(param_stat->ssid, stat.ssid, sizeof(stat.ssid) - 1);
    strncpy(param_stat->passwd, stat.passwd, sizeof(stat.passwd) -1);
    LYINFLOG("[%s ] ret = %d \n", __func__, ret);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_sta_get_status, Gets the status value associated with sta
* @param status_stat [OUT]: lynq_wifi_sta_status_t *, Gets the status value associated with sta
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi_sta_get_status(lynq_wifi_sta_status_t *status_stat)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_sta_status_t stat;
    ret = sc_wifi_sta_get_status(&stat);
    if (0 != ret)
    {
        LYERRLOG("[%s ] sta_param_get ret = %d\n", __func__, ret);
        return ret;
    }
    print_sta_status(&stat);
    sc_to_lynq_sta_status(&stat, status_stat);
    LYINFLOG("[%s ] ret = %d \n", __func__, ret);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_get_sta_pkt_stats, Obtain data packets sent and received by sta for 2.4G or 5G
* @param pkt_stat [OUT]: lynq_wifi_pkt_stats_t *, Obtain data packets sent and received by sta
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi_get_sta_pkt_stats(lynq_wifi_pkt_stats_t *pkt_stat)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_pkt_stats_t stat = {0};
    if (pkt_stat == NULL)
    {
        LYERRLOG("[%s ] Null pointer pkt_stat = 0x%p\n", __func__, pkt_stat);
        return ret;
    }
    ret = sc_wifi_get_sta_pkt_stats(&stat);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__, ret);
        return ret;
    }
    LYINFLOG("[%s ]sta_pkt_get rx[%llu, %llu, %llu, %llu] tx[%llu, %llu, %llu, %llu]\n", __func__, 
             stat.rx_packets, stat.rx_bytes, stat.rx_errors, stat.rx_dropped,
             stat.tx_packets, stat.tx_bytes, stat.tx_errors, stat.tx_dropped);
    pkt_stat->rx_packets = stat.rx_packets;
    pkt_stat->rx_bytes = stat.rx_bytes;
    pkt_stat->rx_errors = stat.rx_errors;
    pkt_stat->rx_dropped = stat.rx_dropped;
    pkt_stat->tx_packets = stat.tx_packets;
    pkt_stat->tx_bytes = stat.tx_bytes;
    pkt_stat->tx_errors = stat.tx_errors;
    pkt_stat->tx_dropped = stat.tx_dropped;
    return 0;
}

/********************************************************************
* @brief: qser_wifi_sta_start_scan, Scan for ap nodes and return them through callback
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi_sta_start_scan(void)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    ret = sc_wifi_sta_start_scan();
    if (0 != ret)
    {
        LYERRLOG("[%s ] scan ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] ret = %d \n", __func__, ret);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_sta_start, To enable sta mode, you need to enable ap mode first
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi_sta_start(void)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    ret = sc_wifi_sta_start();
    if (0 != ret)
    {
        LYERRLOG("[%s ] sta_start ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] ret = %d \n", __func__, ret);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_sta_stop, To disable sta mode.
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi_sta_stop(void)
{
    WIFI_ENABLE_JUDGE();
    int ret = -1;
    ret = sc_wifi_sta_stop();
    if (0 != ret)
    {
        LYERRLOG("[%s ] sta_stop ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] ret = %d \n", __func__, ret);
    return 0;
}


//wifi2

/********************************************************************
* @brief: qser_wifi_work_mode_set, 2.4g or 5g working mode Settings
* @param type [IN]: lynq_wifi_work_mode_e, 2.4G or 5G working mode Settings
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi2_work_mode_set(lynq_wifi_work_mode_e type)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_work_mode_e mode;
    ret = sc_wifi_work_mode_get_ext(1, &mode);
    if (0 != ret)
    {
        LYERRLOG("[%s ] work_mode get ret = %d\n", __func__,ret);
        return ret;
    }
    if(mode == (sc_wifi_work_mode_e)type) //The same value is returned
    {
        return 0;
    }
    ret = sc_wifi_work_mode_set_ext(1, (sc_wifi_work_mode_e)type);
    if (0 != ret)
    {
        LYERRLOG("[%s ] work_mode set ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ]\n", __func__);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_work_mode_get, 2.4g or 5g working mode Gettings
* @param type [OUT]: lynq_wifi_work_mode_e, 2.4G or 5G working mode Gettings
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi2_work_mode_get(lynq_wifi_work_mode_e *type)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_work_mode_e mode;
    ret = sc_wifi_work_mode_get_ext(1, &mode);
    if (0 != ret)
    {
        LYERRLOG("[%s ] work_mode get ret = %d\n", __func__,ret);
        return ret;
    }
    *type = (lynq_wifi_work_mode_e)mode;
    LYINFLOG("[%s ]\n", __func__);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_register_handle, Register callback functions
* @param event_handle [IN]: lynq_wifi_event_handle, Register the ap event callback function
* @param event_handle_sta [IN]: Register sta's event callback function
* @param arg [IN]: void *, Not currently used, but cannot pass a null pointer
* @return :int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi2_register_handle(lynq_wifi_event_handle event_handle, lynq_wifi_event_handle_sta event_handle_sta, void *arg)
{
    if((event_handle == NULL && event_handle_sta == NULL) || arg == NULL)
    {
        LYERRLOG("[%s ] NUll pointer event_handle = 0x%p arg = 0x%p\n", __func__, event_handle, arg);
        return -1;
    }
    if(*(int *)arg <= LYNQ_WIFI_EVENT_MIN || *(int *)arg >= LYNQ_WIFI_EVENT_MAX)
    {
        LYERRLOG("[%s ] The value of arg can only be an integer ranging from 0 to 3p, arg = %d\n", __func__, arg);
        return -1;
    }
    wifi2_event_handle_sta = event_handle_sta;
    wifi2_event_handle = event_handle;
    global_arg_2 = arg;
    return 0;
}

/********************************************************************
* @brief: qser_wifi_enable, Enable WiFi function
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi2_enable(void)
{
    if(lynq_wifi2_enable)
    {
        LYERRLOG("[%s ] The wifi is enabled\n", __func__);
        return -1;
    }
    wifi2_enable_flag = WIFI_ENABLE_FLAG_DEFAULT;
    char wifiAvailable[8] = {0};
    sc_cfg_get("wifiAvailable", wifiAvailable, sizeof(wifiAvailable));
    if (!strcmp(wifiAvailable, "0"))
    {
        LYERRLOG("[%s ] wifiAvailable has been set to 0. If WiFi is used, set 1 to enable it\n", __func__);
        return -1;
    }
    int ret = -1;
    ret = sc_wifi_init_ext(1);
    if (0 != ret)
    {
        LYERRLOG("[%s ] init wifi ret = %d fail\n", __func__,ret);
        return ret;
    }
    //wifi
    ret = sc_wifi_set_enable_status_ind_cb_ext(1, lynq_user2_status);
    if (0 != ret)
    {
        LYERRLOG("[%s ] Request lynq_user_status  ret = %d fail\n", __func__,ret);
        return ret;
    }

    //ap
    ret = sc_wifi_ap_set_status_ind_cb_ext(1, lynq_user2_ap_status);
    if (0 != ret)
    {
        LYERRLOG("[%s ] Request lynq_user_ap_status  ret = %d fail\n", __func__,ret);
        return ret;
    }
    ret = sc_wifi_set_ap_sta_connect_ind_cb_ext(1, lynq_user2_ap_sta_conn_status);
    if (0 != ret)
    {
        LYERRLOG("[%s ] Request lynq_user_ap_sta_conn_status  ret = %d fail\n", __func__,ret);
        return ret;
    }

	//sta
    ret = sc_wifi_sta_set_status_ind_cb_ext(1, lynq_user2_sta_status_ind);
    if (0 != ret)
    {
        LYERRLOG("[%s ] Request lynq_user_sta_status_ind  ret = %d fail\n", __func__,ret);
        return ret;
    }
    ret = sc_wifi_sta_set_scan_result_ind_cb_ext(1, lynq_user2_sta_scan_result_ind);
    if (0 != ret)
    {
        LYERRLOG("[%s ] Request lynq_user_sta_scan_result_ind  ret = %d fail\n", __func__,ret);
        return ret;
    }
	//proxy
    ret = sc_wifi_set_service_error_cb(lynq_user2_wifi_service_error);
    if (0 != ret)
    {
        LYERRLOG("[%s ] Request lynq_user_wifi_service_error  ret = %d fail\n", __func__,ret);
        return ret;
    }

    ret = sc_wifi_enable_ext(1);
    if (0 != ret)
    {
        LYERRLOG("[%s ] enable wifi ret = %d fail\n", __func__,ret);
        return ret;
    }
    for(int i = 0; i < WIFI_ENABLE_FLAG_TIME_OUT; i++)
    {
        if(wifi2_enable_flag == WIFI_ENABLE_FLAG_DEFAULT)
            sleep(1);
        else
            break;
    }
    switch (wifi2_enable_flag)
    {
    case SC_WIFI_STATUS_DISABLED:
        ret = 1;
        break;
    case SC_WIFI_STATUS_ENABLED:
        ret = 0;
        break;
    case SC_WIFI_STATUS_INVALID_MAC:
        ret = 7;
        break;
    case SC_WIFI_STATUS_DEV_INIT_FAIL:
        ret = 6;
        break;
    case SC_WIFI_STATUS_FIRMWARE_CRASH:
        ret = 12;
        break;
    default:
        ret = -1;
        break;
    }
    LYINFLOG("[%s ] ret = %d\n", __func__,ret);
    if(ret == 0)
        lynq_wifi2_enable = true;
    return ret;
}

/********************************************************************
* @brief: qser_wifi_disable, Turn off WiFi
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi2_disable(void)
{
    int ret = -1;
    wifi2_enable_flag = WIFI_ENABLE_FLAG_DEFAULT;
    if(lynq_wifi2_enable == false)
    {
        LYERRLOG("[%s ] WiFi not turned on\n", __func__);
        return -1;
    }
    ret = sc_wifi_disable_ext(1);
    if (0 != ret)
    {
        LYERRLOG("[%s ] disable ret = %d\n", __func__,ret);
        return ret;
    }
    for(int i = 0; i < WIFI_ENABLE_FLAG_TIME_OUT; i++)
    {
        if(wifi2_enable_flag == WIFI_ENABLE_FLAG_DEFAULT)
            sleep(1);
        else
            break;
    }
    switch (wifi2_enable_flag)
    {
    case SC_WIFI_STATUS_DISABLED:
        ret = 0;
        break;
    case SC_WIFI_STATUS_INVALID_MAC:
        ret = 7;
        break;
    case SC_WIFI_STATUS_DEV_INIT_FAIL:
        ret = 6;
        break;
    case SC_WIFI_STATUS_FIRMWARE_CRASH:
        ret = 12;
        break;
    default:
        ret = -1;
        break;
    }
    if (0 != ret)
    {
        LYERRLOG("[%s ] disable ret = %d\n", __func__,ret);
        return ret;
    }
    lynq_wifi2_enable = false;
    ret = sc_wifi_uninit_ext(1);
    if (0 != ret)
    {
        LYERRLOG("[%s ] uninit ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ]\n", __func__);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_get_status, Example Query ap working status
* @param type [OUT]: lynq_wifi_ap_status_t *, ap working status
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi2_ap_get_status(lynq_wifi_ap_index_e idx, lynq_wifi_ap_status_t *ap_stat)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_ap_status_t stat;
    ret = sc_wifi_ap_get_status_ext(1, (sc_wifi_ap_index_e)idx, &stat);
    if (0 != ret)
    {
        LYERRLOG("[%s ] wifi_ap_get  ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d ifname = %s status = %d bssid = %s \n", __func__, idx, stat.ifname, stat.status, stat.bssid);
    strncpy(ap_stat->ifname, stat.ifname, sizeof(stat.ifname) - 1);
    ap_stat->status = (lynq_wifi_status_e)stat.status;
    strncpy(ap_stat->bssid, stat.bssid, sizeof(stat.bssid) - 1);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_acl_set, Set the WiFi2.4G or 5G whitelist
* @param idx [IN]: int, Set 2.4G or 5G
* @param acl_rule [IN]: lynq_wifi_mac_acl_rule_e, Set the blacklist and whitelist mode
* @param mac_list [IN]: char *, Set the mac address of the whitelist
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi2_ap_acl_set(lynq_wifi_ap_index_e idx, lynq_wifi_mac_acl_rule_e acl_rule, char *mac_list)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    if (mac_list == NULL)
    {
        LYERRLOG("[%s ] Null pointer mac_list = 0x%p \n", __func__, mac_list);
        return ret;
    }
    if(acl_rule == LYNQ_WIFI_MAC_ACL_RULE_BLACK)
        ret = sc_wifi_ap_acl_set_ext(1, (sc_wifi_ap_index_e)idx, SC_WIFI_MAC_ACL_RULE_WHITE, mac_list);
    ret = sc_wifi_ap_acl_set_ext(1, (sc_wifi_ap_index_e)idx, (sc_wifi_mac_acl_rule_e)acl_rule, mac_list);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d acl_rule = %d mac_list = %s \n", __func__, idx, acl_rule, mac_list);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_acl_get, Get the WiFi2.4G or 5G whitelist
* @param idx [IN]: int, Set 2.4G or 5G
* @param acl_rule [OUT]: lynq_wifi_mac_acl_rule_e *, Get the blacklist and whitelist mode
* @param mac_list [OUT]: char *, Get the mac address of the whitelist
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi2_ap_acl_get(lynq_wifi_ap_index_e idx, lynq_wifi_mac_acl_rule_e *acl_rule, char *mac_list)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_ap_param_t param = {0};
    if (mac_list == NULL || acl_rule == NULL)
    {
        LYERRLOG("[%s ] Null pointer acl_rule = 0x%p mac_list = 0x%p \n", __func__, acl_rule, mac_list);
        return ret;
    }
    ret = sc_wifi_ap_param_get_ext(1, (sc_wifi_ap_index_e)idx, &param);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    *acl_rule = (lynq_wifi_mac_acl_rule_e)param.acl_rule;
    strncpy(mac_list, param.mac_list, sizeof(param.mac_list) - 1);
    LYINFLOG("[%s ] idx = %d acl_rule = %d mac_list = %s \n", __func__, idx, param.acl_rule, param.mac_list);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_ssid_set, Set the name of the ssid of 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param ssid [IN]: const char *, Set the ssid name
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi2_ap_ssid_set(lynq_wifi_ap_index_e idx, const char *ssid)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    if (ssid == NULL)
    {
        LYERRLOG("[%s ] Null pointer ssid = 0x%p \n", __func__, ssid);
        return ret;
    }
    ret = sc_wifi_ap_ssid_set_ext(1, (sc_wifi_ap_index_e)idx, ssid);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d ssid = %s \n", __func__, idx, ssid);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_ssid_get, Get the name of the ssid of 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param ssid [OUT]: char *, Get the ssid name
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi2_ap_ssid_get(lynq_wifi_ap_index_e idx, char *ssid)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_ap_param_t param = {0};
    ret = sc_wifi_ap_param_get_ext(1, (sc_wifi_ap_index_e)idx, &param);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d ssid = %s \n", __func__, idx, param.ssid);
    strncpy(ssid, param.ssid, sizeof(param.ssid) - 1);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_ssid_hide_set, Set whether the ssid of 2.4G or 5G is hidden
* @param idx [IN]: int, Set 2.4G or 5G
* @param hide [IN]: bool, Set whether the ssid is hidden
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi2_ap_ssid_hide_set(lynq_wifi_ap_index_e idx, bool hide)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    ret = sc_wifi_ap_ssid_hidden_set_ext(1, (sc_wifi_ap_index_e)idx, (int)hide);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d hide = %d \n", __func__, idx, hide);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_ssid_hide_get, Get whether the ssid of 2.4G or 5G is hidden
* @param idx [IN]: int, Set 2.4G or 5G
* @param hide [OUT]: bool *, Get whether the ssid is hidden
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi2_ap_ssid_hide_get(lynq_wifi_ap_index_e idx, bool *hide)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_ap_param_t param = {0};
    if (hide == NULL)
    {
        LYERRLOG("[%s ] Null pointer hide = 0x%p \n", __func__, hide);
        return ret;
    }
    ret = sc_wifi_ap_param_get_ext(1, (sc_wifi_ap_index_e)idx, &param);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d ssid_hide = %d \n", __func__, idx, param.ssid_hide);
    *hide = (bool)param.ssid_hide;
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_mode_set, Set the working protocol mode of 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param mode [IN]: lynq_wifi_mode_type_e, Set the working protocol mode
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi2_ap_mode_set(lynq_wifi_ap_index_e idx, lynq_wifi_mode_type_e mode)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    ret = sc_wifi_ap_mode_set_ext(1, (sc_wifi_ap_index_e)idx, lynq_to_sc_mode(mode));
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d mode = %d \n", __func__, idx, mode);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_mode_get, Get the working protocol mode of 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param mode [OUT]: lynq_wifi_mode_type_e *, Get the working protocol mode
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi2_ap_mode_get(lynq_wifi_ap_index_e idx, lynq_wifi_mode_type_e *mode)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_ap_param_t param = {0};
    if (mode == NULL)
    {
        LYERRLOG("[%s ] Null pointer mode = 0x%p \n", __func__, mode);
        return ret;
    }
    ret = sc_wifi_ap_param_get_ext(1, (sc_wifi_ap_index_e)idx, &param);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d mode = %d \n", __func__, idx, param.mode);
    *mode = sc_to_lynq_mode(param.mode);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_bandwidth_set, Set the bandwidth of 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param bandwidth [IN]: lynq_wifi_bandwidth_type_e, Set the bandwidth
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi2_ap_bandwidth_set(lynq_wifi_ap_index_e idx, lynq_wifi_bandwidth_type_e bandwidth)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    ret = sc_wifi_ap_bandwidth_set_ext(1, (sc_wifi_ap_index_e)idx, (sc_wifi_bandwidth_e)bandwidth);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d bandwidth = %d \n", __func__, idx, bandwidth);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_bandwidth_get, Get the bandwidth of 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param bandwidth [OUT]: lynq_wifi_bandwidth_type_e *, Get the bandwidth
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi2_ap_bandwidth_get(lynq_wifi_ap_index_e idx, lynq_wifi_bandwidth_type_e *bandwidth)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_ap_param_t param = {0};
    if (bandwidth == NULL)
    {
        LYERRLOG("[%s ] Null pointer bandwidth = 0x%p \n", __func__, bandwidth);
        return ret;
    }
    ret = sc_wifi_ap_param_get_ext(1, (sc_wifi_ap_index_e)idx, &param);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d bandwidth = %d \n", __func__, idx, param.bandwidth);
    *bandwidth = (lynq_wifi_bandwidth_type_e)param.bandwidth;
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_channel_set, Set the channel for 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param country_code [IN]: const char *, Set country code
* @param channel [IN]: int, Set the channel
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi2_ap_channel_set(lynq_wifi_ap_index_e idx, const char *country_code, int channel)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    if (country_code == NULL)
    {
        LYERRLOG("[%s ] Null pointer country_code = 0x%p \n", __func__, country_code);
        return ret;
    }
    ret = sc_wifi_ap_cc_ch_set_ext(1, (sc_wifi_ap_index_e)idx, country_code, channel);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d country_code = %s channel = %d\n", __func__, idx, country_code, channel);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_channel_get, Get the channel for 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param country_code [OUT]: char *, Get country code
* @param channel [OUT]: int *, Get the channel
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi2_ap_channel_get(lynq_wifi_ap_index_e idx, char *country_code, int *channel)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_ap_param_t param = {0};
    if (country_code == NULL || channel == NULL)
    {
        LYERRLOG("[%s ] Null pointer country_code = 0x%p channel = 0x%p\n", __func__, country_code, channel);
        return ret;
    }
    ret = sc_wifi_ap_param_get_ext(1, (sc_wifi_ap_index_e)idx, &param);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d country_code = %s channel = %d\n", __func__, idx, param.countrycode, param.channel);
    strncpy(country_code, param.countrycode, sizeof(param.countrycode) - 1);
    *channel = param.channel;
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_auth_set, Set the security authentication mode and password of 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param auth_mode [IN]: lynq_wifi_auth_e, Set the security authentication mode
* @param auth_passwd [IN]: const char *, Set password
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi2_ap_auth_set(lynq_wifi_ap_index_e idx, lynq_wifi_auth_e auth_mode, const char *auth_passwd)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    if (auth_passwd == NULL)
    {
        LYERRLOG("[%s ] Null pointer auth_passwd = 0x%p\n", __func__, auth_passwd);
        return ret;
    }
    
    sc_wifi_ap_auth_t auth;
    auth.auth = lynq_to_sc_auth_mode(auth_mode);
    strncpy(auth.passwd, auth_passwd, sizeof(auth.passwd) - 1);
    ret = sc_wifi_ap_auth_set_ext(1, (sc_wifi_ap_index_e)idx, &auth);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d auth_mode = %d auth_passwd = %s\n", __func__, idx, auth_mode, "******");
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_auth_get, Get the security authentication mode and password of 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param auth_mode [OUT]: lynq_wifi_auth_e *, Get the security authentication mode
* @param auth_passwd [OUT]: char *, Get password
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi2_ap_auth_get(lynq_wifi_ap_index_e idx, lynq_wifi_auth_e *auth_mode, char *auth_passwd)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_ap_param_t param = {0};
    if (auth_mode == NULL || auth_passwd == NULL)
    {
        LYERRLOG("[%s ] Null pointer auth_mode = 0x%p auth_passwd = 0x%p\n", __func__, auth_mode, auth_passwd);
        return ret;
    }
    ret = sc_wifi_ap_param_get_ext(1, (sc_wifi_ap_index_e)idx, &param);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d auth_mode = %d auth_passwd = %s \n", __func__, idx, param.auth.auth, param.auth.passwd);
    strncpy(auth_passwd, param.auth.passwd, sizeof(param.auth.passwd) - 1);
    
    *auth_mode = sc_to_lynq_auth_mode(param.auth.auth);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_auth_get_s, Get the security authentication mode , password group_rekey and  pairwise of 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param auth_mode [OUT]: lynq_wifi_ap_auth_t *, Get the security authentication mode
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi2_ap_auth_get_s(lynq_wifi_ap_index_e idx, lynq_wifi_ap_auth_t *auth_mode)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_ap_param_t param = {0};
    if (auth_mode == NULL)
    {
        LYERRLOG("[%s ] Null pointer auth_mode = 0x%p\n", __func__, auth_mode);
        return ret;
    }
    ret = sc_wifi_ap_param_get_ext(1, (sc_wifi_ap_index_e)idx, &param);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d auth = %d passwd = %s group_rekey = %d pairwise = %d\n", __func__, idx, param.auth.auth, param.auth.passwd, param.auth.group_rekey, param.auth.pairwise);
    strncpy(auth_mode->passwd, param.auth.passwd, sizeof(param.auth.passwd) - 1);
    auth_mode->group_rekey = param.auth.group_rekey;
    auth_mode->pairwise = (lynq_wifi_auth_wpa_psk_e)param.auth.pairwise;
    auth_mode->auth = sc_to_lynq_auth_mode(param.auth.auth);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_max_sta_set, Set the maximum number of STAs for 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param max_sta_num [IN]: int, Set the maximum number
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi2_ap_max_sta_set(lynq_wifi_ap_index_e idx, int max_sta_num)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    if(max_sta_num >= 32)
    {
        LYERRLOG("[%s ] fail max_sta_num = %d\n", __func__,max_sta_num);
        return -1;
    }
    ret = sc_wifi_ap_max_sta_num_set_ext(1, (sc_wifi_ap_index_e)idx, max_sta_num);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d max_sta_num = %d \n", __func__, idx, max_sta_num);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_max_sta_get, Get the maximum number of STAs for 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param max_sta_num [OUT]: int *, Get the maximum number
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi2_ap_max_sta_get(lynq_wifi_ap_index_e idx, int *max_sta_num)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_ap_param_t param = {0};
    if (max_sta_num == NULL)
    {
        LYERRLOG("[%s ] Null pointer max_sta_num = 0x%p\n", __func__,max_sta_num);
        return ret;
    }
    ret = sc_wifi_ap_param_get_ext(1, (sc_wifi_ap_index_e)idx, &param);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d max_sta_num = %d \n", __func__, idx, param.max_sta_num);
    *max_sta_num = param.max_sta_num;
    return 0;
}

/********************************************************************
* @brief: qser_wifi_lanhost_get_list, Get sta device information received as an ap for 2.4G or 5G
* @param lynq_arrays [OUT]: lynq_lanhost_ts *, Get sta device information received as an ap
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi2_lanhost_get_list(lynq_lanhost_ts *lynq_arrays)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    int i = 0;
    sc_lanhost_t array[32] = {0};
    if (lynq_arrays == NULL)
    {
        LYERRLOG("[%s ] Null pointer lynq_arrays = 0x%p\n", __func__, lynq_arrays);
        return ret;
    }
    lynq_arrays->array_len = 32;
    ret = sc_wifi_lanhost_get_list_ext(1, array, &lynq_arrays->array_len);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__, ret);
        return ret;
    }
    LYINFLOG("[%s]ap_lanhost len[%d]\n", __func__, lynq_arrays->array_len);
    for (i = 0; i < lynq_arrays->array_len; i++)
    {
        LYINFLOG("[%s]Element : [%d] ifname = %s  macaddr = %s addr = %s name = %s uptime = %d\n", __func__, i,
                 array[i].ifname, array[i].macaddr, array[i].addr, array[i].name, array[i].uptime);
        strncpy(lynq_arrays->array[i].ifname, array[i].ifname, sizeof(lynq_arrays->array[i].ifname) - 1);
        strncpy(lynq_arrays->array[i].macaddr, array[i].macaddr, sizeof(lynq_arrays->array[i].macaddr) - 1);
        strncpy(lynq_arrays->array[i].addr, array[i].addr, sizeof(lynq_arrays->array[i].addr) - 1);
        strncpy(lynq_arrays->array[i].name, array[i].name, sizeof(lynq_arrays->array[i].name) - 1);
        lynq_arrays->array[i].uptime = array[i].uptime;
    }
    return 0;
}

/********************************************************************
* @brief: qser_wifi_get_ap_pkt_stats, Obtain data packets sent and received by ap for 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @param pkt_stat [OUT]: lynq_wifi_pkt_stats_t *, Obtain data packets sent and received by ap
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int qser_wifi2_get_ap_pkt_stats(lynq_wifi_ap_index_e idx, lynq_wifi_pkt_stats_t *pkt_stat)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    sc_wifi_pkt_stats_t stat = {0};
    if (pkt_stat == NULL)
    {
        LYERRLOG("[%s ] Null pointer pkt_stat = 0x%p\n", __func__, pkt_stat);
        return ret;
    }
    idx = LYNQ_WIFI_AP_INDEX_AP0;
    ret = sc_wifi_get_ap_pkt_stats_ext(1, (sc_wifi_ap_index_e)idx, &stat);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__, ret);
        return ret;
    }
    LYINFLOG("[%s ]ap_pkt_get[%d] rx[%llu, %llu, %llu, %llu] tx[%llu, %llu, %llu, %llu]\n", __func__, idx,
             stat.rx_packets, stat.rx_bytes, stat.rx_errors, stat.rx_dropped,
             stat.tx_packets, stat.tx_bytes, stat.tx_errors, stat.tx_dropped);
    pkt_stat->rx_packets = stat.rx_packets;
    pkt_stat->rx_bytes = stat.rx_bytes;
    pkt_stat->rx_errors = stat.rx_errors;
    pkt_stat->rx_dropped = stat.rx_dropped;
    pkt_stat->tx_packets = stat.tx_packets;
    pkt_stat->tx_bytes = stat.tx_bytes;
    pkt_stat->tx_errors = stat.tx_errors;
    pkt_stat->tx_dropped = stat.tx_dropped;
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_start, Set the ap mode of 2.4G or 5G to enable
* @param idx [IN]: int, Set 2.4G or 5G
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi2_ap_start(lynq_wifi_ap_index_e idx)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    ret = sc_wifi_ap_start_ext(1, (sc_wifi_ap_index_e)idx);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d \n", __func__, idx);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_stop, Disable ap mode for 2.4G or 5G
* @param idx [IN]: int, Set 2.4G or 5G
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi2_ap_stop(lynq_wifi_ap_index_e idx)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    ret = sc_wifi_ap_stop_ext(1, (sc_wifi_ap_index_e)idx);
    if (0 != ret)
    {
        LYERRLOG("[%s ] ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d \n", __func__, idx);
    return 0;
}

/********************************************************************
* @brief: qser_wifi_ap_restart, Set the ap mode of 2.4G or 5G to restart
* @param idx [IN]: int, Set 2.4G or 5G
* @return : int, If equal to 0 succeeds, others fail
* @todo: NA
* @see: NA
* @warning: NA
*********************************************************************/
int  qser_wifi2_ap_restart(lynq_wifi_ap_index_e idx)
{
    WIFI2_ENABLE_JUDGE();
    int ret = -1;
    ret = sc_wifi_ap_stop_ext(1, (sc_wifi_ap_index_e)idx);
    if (0 != ret)
    {
        LYERRLOG("[%s ] stop ret = %d\n", __func__,ret);
        return ret;
    }
    ret = sc_wifi_ap_start_ext(1, (sc_wifi_ap_index_e)idx);
    if (0 != ret)
    {
        LYERRLOG("[%s ] start ret = %d\n", __func__,ret);
        return ret;
    }
    LYINFLOG("[%s ] idx = %d \n", __func__, idx);
    return 0;
}

DEFINE_LYNQ_LIB_LOG(LYNQ_WIFI)

#ifdef __cplusplus
}
#endif
