/**@file lynq-systime.cpp
 *
 * @brief Sync systime form different time source.
 *
 * @author sj.zhang
 *
 * @date 2023-08-15
 *
 * @version V1.0
 *
 * @copyright Copyright (c) MobileTek
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <include/lynq_uci.h>
#include "liblog/lynq_deflog.h"
#include "lynq_systime.h"

#ifdef __cplusplus
extern "C"
{
#endif

#define USER_LOG_TAG "LYNQ_SYSTIME"
#define LOG_LEVEL 4

#define LYNQ_SYNC_TIME_SECTION "lynq_sync_time"
#define LYNQ_MODEM_TIME_KEY "lynq_modem_sync_time_enable"
#define LYNQ_GNSS_TIME_KEY "lynq_gnss_sync_time_enable"
#define SNTP_PROCESS_RESULT  "sntp_process_result"
#define GNSS_PROCESS_RESULT  "gnss_process_result"
#define BUF_LEN 258
#define TIME_BUF 100
#define SYNC_TIME_SUCCESS 0
#define NTP_ALREADY_ENABLE 1
#define NTP_ALREADY_DISABLE 2

#define SYSTEM_FUNC_FAILED 7
#define ERROR_PARA 8
#define NV_SET_FAILED 9

#ifdef MOBILETEK_TARGET_PLATFORM_T106
#define NTP_RESTART_BUF "/etc/init.d/sntp restart"
#define NTP_STOP_BUF "/etc/init.d/sntp stop"
#define NTP_START_BUF "/etc/init.d/sntp start"
#define PGREP_NTP_DARMON "pgrep sntp"

#else
#define NTP_RESTART_BUF "systemctl restart ntpd"
#define NTP_STOP_BUF "systemctl stop ntpd"
#define NTP_START_BUF "systemctl start ntpd"
#define PGREP_NTP_DARMON "pgrep ntpd"

#endif

#ifdef MOBILETEK_TARGET_PLATFORM_T106
typedef int (*sc_rtc_time_get_cb)(unsigned int src_id, unsigned long ulsec);
extern int sc_rtc_timer_init(void);
extern int sc_rtc_timer_uninit(void);
extern int sc_rtc_time_set(int srcid);
extern int sc_rtc_time_get(int srcid, sc_rtc_time_get_cb rtc_notify);
extern int sc_cfg_get(char *name, char* buf, int bufLen);
extern int sc_cfg_set (char *name, char *value);
extern int get_terminate_flag(void);



char sync_from_rtc_cb_flag = 1;
char rtc_time_get_cb_flag = 1;
static unsigned long s_rtc_sec = 0;

static int sync_from_rtc_cb(unsigned int src_id, unsigned long ulsec)
{
    LYLOGSET(LOG_LEVEL);
    LYLOGEINIT(USER_LOG_TAG);
    LYDBGLOG("[%s][%d] enter.\n", __FUNCTION__, __LINE__);
    sync_from_rtc_cb_flag = 1;
    struct tm rtc_time = {0};
    time_t tmp = 0;
    char command[100] = "";
    tmp = (time_t)ulsec;
    //xy.he add for T106BUG-329 2023-12-13 start
    rtc_time = *localtime(&tmp);
    LYINFLOG("RTC time now is: %d-%d-%d,%d:%d:%d\n",
             rtc_time.tm_year + 1900, rtc_time.tm_mon + 1, rtc_time.tm_mday, rtc_time.tm_hour, rtc_time.tm_min, rtc_time.tm_sec);
    //set timezone
    tzset();
    printf("timezone = %d\n",timezone);
    //Change the time according to the time zone
    tmp = tmp - timezone;
    rtc_time = *localtime(&tmp);
    LYINFLOG("Local time now is: %d-%d-%d,%d:%d:%d\n",
             rtc_time.tm_year + 1900, rtc_time.tm_mon + 1, rtc_time.tm_mday, rtc_time.tm_hour, rtc_time.tm_min, rtc_time.tm_sec);
    snprintf(command, sizeof(command), "date --set=\'@%lu\' ", tmp);
    //xy.he add for T106BUG-329 2023-12-13 end
    int ret = system(command);
    if (ret != 0)
    {
        printf("Function system(\"%s\") failed.", command);
        sync_from_rtc_cb_flag = 0;
        return SYSTEM_FUNC_FAILED;
    }
    sync_from_rtc_cb_flag = 0;
    return 0;
}

static int rtc_time_get_cb(unsigned int src_id, unsigned long ulsec)
{
    LYLOGSET(LOG_LEVEL);
    LYLOGEINIT(USER_LOG_TAG);
    LYDBGLOG("[%s][%d] enter.\n", __FUNCTION__, __LINE__);
    rtc_time_get_cb_flag = 1;
    s_rtc_sec = ulsec;
    struct tm rtc_time = {0};
    time_t tmp = 0;
    tmp = (time_t)ulsec;
    rtc_time = *localtime(&tmp);
    LYINFLOG("RTC time now is: %d-%d-%d,%d:%d:%d\n",
             rtc_time.tm_year + 1900, rtc_time.tm_mon + 1, rtc_time.tm_mday, rtc_time.tm_hour, rtc_time.tm_min, rtc_time.tm_sec);
    rtc_time_get_cb_flag = 0;
    return 0;
}

int get_sntp_process_result()
{
    LYLOGSET(LOG_LEVEL);
    LYLOGEINIT(USER_LOG_TAG);
    char sntp_process_result[10] = {0};
    int ret = 0;

    ret = sc_cfg_get(SNTP_PROCESS_RESULT,sntp_process_result,sizeof(sntp_process_result));

    if(0 == ret)
    {
        if(strcmp(sntp_process_result,"success") != 0)
        {
            strcpy(sntp_process_result,"failed");
            LYINFLOG("sntp sync result:%s\n",sntp_process_result);
            return -1;
        }

        else
        {
            LYINFLOG("sntp sync result:%s\n",sntp_process_result);
            return SYNC_TIME_SUCCESS;
        }
    }

    else
    {
        LYERRLOG("read sntp_sync_result failed\n");
        return -1;
    }
    
}

int get_gnss_process_result()
{
    LYLOGSET(LOG_LEVEL);
    LYLOGEINIT(USER_LOG_TAG);
    char gnss_process_result[10] = {0};
    int ret = 0;

    ret = sc_cfg_get(GNSS_PROCESS_RESULT,gnss_process_result,sizeof(gnss_process_result));

    if(0 == ret)
    {
        if(strcmp(gnss_process_result,"success") != 0)
        {
            strcpy(gnss_process_result,"failed");
            LYINFLOG("gnss sync result:%s\n",gnss_process_result);
            return -1;
        }
        else
        {
            LYINFLOG("gnss sync result:%s\n",gnss_process_result);
            return SYNC_TIME_SUCCESS;
        }
    }
    else
    {
        LYERRLOG("read gnss_sync_result failed\n");
        return -1;
    }
    
}

int get_sync_time_result()
{
    LYLOGSET(LOG_LEVEL);
    LYLOGEINIT(USER_LOG_TAG);
    time_src_status_s time_src = {0};
    lynq_get_time_src_status(&time_src);

    LYINFLOG("time_src -> gnss = %d,time_src -> ntp = %d, time_src -> nitz = %d\n",time_src.gnss,time_src.ntp,time_src.nitz);

    if(time_src.gnss)
    {
        LYINFLOG("time_source is gnss\n");
        return get_gnss_process_result();
    }

    else if(time_src.ntp)
    {
        LYINFLOG("time_source is ntp\n");
        return get_sntp_process_result();
    }

    else if(time_src.nitz)
    {
        LYINFLOG("time_source is nitz\n");
        return 0;
    }

    else
    {
        LYINFLOG("don't set any time_source\n");
        return 0;
    }

}


int lynq_sync_time_from_rtc(void)
{
    LYLOGSET(LOG_LEVEL);
    LYLOGEINIT(USER_LOG_TAG);
    LYDBGLOG("[%s][%d] enter.\n", __FUNCTION__, __LINE__);
    int ret = -1;
    int srcid = -1;
    int i = 0;
    int count = 0;
    if ((srcid = sc_rtc_timer_init()) <= 0)
    {
        LYERRLOG("rtc_timer_demo:rtc_timer_init fail!\n");
        return -1;
    }
    ret = sc_rtc_time_get(srcid, sync_from_rtc_cb);
    if (ret != 0)
    {
        LYERRLOG("sc_rtc_time_get failed !\n");
        sc_rtc_timer_uninit();
        return -1;
    }

    // Wait for the callback function to finish executing. But 5 seconds tops.
    for (i = 25; i > 0; i--)
    {
        if (sync_from_rtc_cb_flag == 0)
            break;
        usleep(200000);
    }

    if (sync_from_rtc_cb_flag != 0)
    {
        LYERRLOG("rtc_get_time_cb failed\n");
        sc_rtc_timer_uninit();
        return -1;
    }
    LYINFLOG("sync from rtc success\n");
    ret = sc_rtc_timer_uninit();
    if (0 != ret)
    {
        LYERRLOG("uninit failed \n");
        return -1;
    }
    //xy.he add for T106BUG-413 2024-01-04
    while((get_terminate_flag() == 0) && (count <100))
    {
        usleep(10*1000);
        count ++;
    }
    return 0;
}

int lynq_set_rtc_time(void)
{
    LYLOGSET(LOG_LEVEL);
    LYLOGEINIT(USER_LOG_TAG);
    LYDBGLOG("[%s][%d] enter.\n", __FUNCTION__, __LINE__);
    int ret = -1;
    int srcid = -1;
    int count = 0;
    if ((srcid = sc_rtc_timer_init()) <= 0)
    {
        LYERRLOG("rtc_timer_demo:rtc_timer_init fail!\n");
        return -1;
    }
    ret = sc_rtc_time_set(srcid);
    if (ret != 0)
    {
        LYERRLOG("sc_rtc_time_set failed !\n");
        return -1;
    }
    ret = sc_rtc_timer_uninit();
    if (0 != ret)
    {
        LYERRLOG("uninit failed \n");
        return -1;
    }
    //xy.he add for T106BUG-413 2024-01-04
    while((get_terminate_flag() == 0) && (count <100))
    {
        usleep(10*1000);
        count ++;
    }
    printf("lynq_set_rtc_time_success\n");
    return 0;
}

int lynq_get_rtc_time(unsigned long *ulsec)
{
    LYLOGSET(LOG_LEVEL);
    LYLOGEINIT(USER_LOG_TAG);
    LYDBGLOG("[%s][%d] enter.\n", __FUNCTION__, __LINE__);
    int ret = -1;
    int srcid = -1;
    int i = 0;
    int count = 0;
    if (NULL == ulsec)
    {
        LYERRLOG("NULL parameter!\n");
        return ERROR_PARA;
    }
    if ((srcid = sc_rtc_timer_init()) <= 0)
    {
        LYERRLOG("rtc_timer_demo:rtc_timer_init fail!\n");
        return -1;
    }
    ret = sc_rtc_time_get(srcid, rtc_time_get_cb);
    if (ret != 0)
    {
        LYERRLOG("sc_rtc_time_get failed !\n");
        sc_rtc_timer_uninit();
        return -1;
    }

    // Wait for the callback function to finish executing. But 5 seconds tops.
    for (i = 25; i > 0; i--)
    {
        if (rtc_time_get_cb_flag == 0)
            break;
        usleep(200000);
    }
    if (rtc_time_get_cb_flag != 0)
    {
        LYERRLOG("rtc_get_time_cb failed\n");
        sc_rtc_timer_uninit();
        return -1;
    }
    *ulsec = s_rtc_sec;
    ret = sc_rtc_timer_uninit();
    if (0 != ret)
    {
        LYERRLOG("uninit failed \n");
        return -1;
    }
    //xy.he add for T106BUG-413 2024-01-04
    while((get_terminate_flag() == 0) && (count <100))
    {
        usleep(10*1000);
        count ++;
    }
    return 0;
}
#endif

int lynq_get_time_src_status(time_src_status_s *time_src)
{
    LYLOGSET(LOG_LEVEL);
    LYLOGEINIT(USER_LOG_TAG);
    LYDBGLOG("[%s][%d] enter.\n", __FUNCTION__, __LINE__);
    char nitz_enable[8] = "";
    char gnss_enable[8] = "";
    char buf[BUF_LEN] = "";
    if (NULL == time_src)
    {
        LYERRLOG("NULL parameter!\n");
        return ERROR_PARA;
    }
    // get ntp status.
    int ret = system(PGREP_NTP_DARMON);
    if (0 == ret)
    {
        time_src->ntp = 1;
    }
    else
    {
        time_src->ntp = 0;
    }

    // get nitz status
    sprintf(buf, "%s.%s.%s", LYNQ_UCI_FILE, LYNQ_SYNC_TIME_SECTION, LYNQ_MODEM_TIME_KEY);
    lynq_uci_get(buf, nitz_enable);
    time_src->nitz = atoi(nitz_enable);

    // get gnss status
    sprintf(buf, "%s.%s.%s", LYNQ_UCI_FILE, LYNQ_SYNC_TIME_SECTION, LYNQ_GNSS_TIME_KEY);
    lynq_uci_get(buf, gnss_enable);
    time_src->gnss = atoi(gnss_enable);

    return 0;
}

const char *lynq_read_version()
{
    return "SYSTIME_V1.0";
}

int user_set_time(char *date, char *time)
{
    LYLOGSET(LOG_LEVEL);
    LYLOGEINIT(USER_LOG_TAG);
    LYDBGLOG("[%s][%d] enter.\n", __FUNCTION__, __LINE__);
    char cmd_buf[TIME_BUF] = "";
    if (NULL == date || NULL == time)
    {
        LYERRLOG("NULL parameter!\n");
        return ERROR_PARA;
    }

    // set systime
    snprintf(cmd_buf, sizeof(cmd_buf), "date -s %s+%s", date, time);
    int ret = system(cmd_buf);
    if (ret != 0)
    {
        printf("Function system(\"%s\") failed.\n", cmd_buf);
        return SYSTEM_FUNC_FAILED;
    }
    return 0;
}

int ntp_sync_time(int enable)
{
    LYLOGSET(LOG_LEVEL);
    LYLOGEINIT(USER_LOG_TAG);
    LYDBGLOG("[%s][%d] enter.\n", __func__, __LINE__);
    int ret = 0;
    if (enable != 0 && enable != 1)
    {
        LYERRLOG("Parameter error! Only 0/1 allowed.\n");
        return ERROR_PARA;
    }
    ret = sc_cfg_set(SNTP_PROCESS_RESULT,"");
    if (0 != ret)
    {
        LYERRLOG("set SNTP_PROCESS_RESULT failed\n");
        return NV_SET_FAILED;
    }
    ret = system(PGREP_NTP_DARMON);
    if (0 == ret)
    {
        if (enable)
        {
            ret = system(NTP_RESTART_BUF);
            if (ret != 0)
            {
                printf("Function system(\"%s\") failed.", NTP_RESTART_BUF);
                return SYSTEM_FUNC_FAILED;
            }
        }
        else
        {
            ret = system(NTP_STOP_BUF);
            if (ret != 0)
            {
                printf("Function system(\"%s\") failed.", NTP_STOP_BUF);
                return SYSTEM_FUNC_FAILED;
            }
        }
    }
    else
    {
        if (enable)
        {
            // Only one time source is allowed to run simultaneously.
            modem_time_enable(0);
            gnss_time_enable(0);
            ret = system(NTP_START_BUF);
            if (ret != 0)
            {
                printf("Function system(\"%s\") failed.", NTP_START_BUF);
                return SYSTEM_FUNC_FAILED;
            }
        }
    }
    return SYNC_TIME_SUCCESS;
}

int modem_time_enable(int enable)
{
    LYLOGSET(LOG_LEVEL);
    LYLOGEINIT(USER_LOG_TAG);
    LYDBGLOG("[%s][%d] enter.\n", __FUNCTION__, __LINE__);
    char buf[BUF_LEN] = "";
    int ret = 0;
    if (enable != 0 && enable != 1)
    {
        LYERRLOG("Parameter error! Only 0/1 allowed.\n");
        return ERROR_PARA;
    }
    if (enable)
    {
        // Only one time source is allowed to run simultaneously.
        ntp_sync_time(0);
        gnss_time_enable(0);
    }
    sprintf(buf, "%s.%s.%s=%d", LYNQ_UCI_FILE, LYNQ_SYNC_TIME_SECTION, LYNQ_MODEM_TIME_KEY, enable);
    ret = lynq_uci_set(buf);

    return ret;
}

int gnss_time_enable(int enable)
{
    LYLOGSET(LOG_LEVEL);
    LYLOGEINIT(USER_LOG_TAG);
    LYDBGLOG("[%s][%d] enter.\n", __FUNCTION__, __LINE__);
    char buf[BUF_LEN] = "";
    char result[10] = "";
    int ret = 0;
    if (enable != 0 && enable != 1)
    {
        LYERRLOG("Parameter error! Only 0/1 allowed.\n");
        return ERROR_PARA;
    }
    ret = sc_cfg_set(GNSS_PROCESS_RESULT,"");
    if (0 != ret)
    {
        LYERRLOG("set GNSS_PROCESS_RESULT failed\n");
        return NV_SET_FAILED;
    }
    
    if (enable)
    {
        // Only one time source is allowed to run simultaneously.
        ntp_sync_time(0);
        modem_time_enable(0);
    }
    sprintf(buf, "%s.%s.%s=%d", LYNQ_UCI_FILE, LYNQ_SYNC_TIME_SECTION, LYNQ_GNSS_TIME_KEY, enable);
    ret = lynq_uci_set(buf);

    return ret;
}

DEFINE_LYNQ_LIB_LOG(LYNQ_SYSTIME)
#ifdef __cplusplus
}
#endif
