#include "lynq_deflog.h"
#include <stdarg.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <pthread.h>
#include <log/log.h>
#include <include/lynq_uci.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>

#ifdef MOBILETEK_TARGET_PLATFORM_T106
extern "C" {
#include <cfg_api.h>
extern int log_level(char *level_name);
}
#endif

// log entry for each module
struct log_module_entry
{
    LYNQ_WRITE_LOG_PTR log_ptr; // function ptr of write log
    log_level_enum level; // current log level of module;
    char module_name[32]; // log tag
    log_level_enum special_level; // defined by uci of [main module tag]__[module_tag]
    log_level_enum exe_level; // defined by uci of [main module tag]
    log_level_enum lib_level; // defined by uci of [module tag]
    log_level_enum global_level; // defined by uci of "global_level"

    log_module_entry() // constuctor of struct
    {
        level = LOG_INFO; // default log level when no uci value
        special_level = LOG_UNSET;
        lib_level = LOG_UNSET;
        exe_level = LOG_UNSET;
        global_level = LOG_UNSET;
    }

    void calc_level() // to calc the current log level
    {
        level = LOG_INFO;
        if (special_level != LOG_UNSET) // [exe]_[lib] level as the first consideration
        {
            level = special_level;
        }
        else if (exe_level != LOG_UNSET) // when [exe] level is set
        {
            level = exe_level;
        }
        else if (lib_level != LOG_UNSET) // when [lib] level is set
        {
            level = lib_level;
        }
        else if (global_level != LOG_UNSET) // global log level is set
        {
            level = global_level;
        }
    }
};


static pthread_mutex_t s_lynq_log_mutex = PTHREAD_MUTEX_INITIALIZER; //lock for module entry array
const int entry_count = 32; // max count of modules
static int s_curr_reg_count = 0; // current reg modules count
static int s_log_module_array_init_flag = 0;
static struct log_module_entry *s_log_module_entries[entry_count] = {0};
static volatile int s_main_module_entry_index = -1;

#define LOG_UCI_MODULE "lynq_log"
#define LOG_UCI_FILE "lynq_uci"

template <int n>
void lynq_write_log_func(log_level_enum level, const char *format, ...)
{
    log_module_entry *log_entry;
    va_list args;
    va_start(args,format);

    log_entry = s_log_module_entries[n];
    if (log_entry == NULL || level > log_entry->level)
        return;

    switch(level)
    {
        case LOG_VERBOSE:
#ifdef MOBILETEK_TARGET_PLATFORM_T106
            __android_log_vprint(ANDROID_LOG_ERROR,log_entry->module_name, format, args);
#else
            __android_log_vprint(ANDROID_LOG_VERBOSE,log_entry->module_name, format, args);
#endif
            break;
        case LOG_ERROR:
            __android_log_vprint(ANDROID_LOG_ERROR,log_entry->module_name, format, args);
            break;
        case LOG_WARNING:
            __android_log_vprint(ANDROID_LOG_WARN,log_entry->module_name, format, args);
            break;
        case LOG_INFO:
            __android_log_vprint(ANDROID_LOG_INFO,log_entry->module_name, format, args);
            break;
        case LOG_DEBUG:
            __android_log_vprint(ANDROID_LOG_DEBUG,log_entry->module_name, format, args);
            break;
        default :
            __android_log_print(ANDROID_LOG_DEBUG,log_entry->module_name, "Log Level is Error!!!!!!");
            break;
    }
    va_end(args);
    //return 0;
}

template <int n>
void reg_write_log_function()
{
    s_log_module_entries[n] = new struct log_module_entry;
    s_log_module_entries[n]->log_ptr = &lynq_write_log_func<n>;
    reg_write_log_function<n-1>();
}

template <>
void reg_write_log_function<0>()
{
    s_log_module_entries[0] = new struct log_module_entry;
    s_log_module_entries[0]->log_ptr = &lynq_write_log_func<0>;
}

const static char * LogLevelNameInfoTable[LOG_LEVEL_MAX] =
{
    "[VERBOSE]","[ERROR]","[WARNING]","[INFO]","[DEBUG]"
};

static log_level_enum convert_log_level_from_string(const char * level_string)
{
    for(int level=LOG_VERBOSE; level < LOG_LEVEL_MAX; level++)
    {
        if (strcmp(LogLevelNameInfoTable[level], level_string) == 0)
        {
            return (log_level_enum)level;
        }
    }
    return LOG_UNSET;
}

static log_level_enum get_uci_log_value(const char *key)
{
    char level_buf[64] = {0};
    if (key == NULL || key[0] == '\0')
    {
        return LOG_UNSET;
    }

    if (0 == lynq_get_value((char*)LOG_UCI_FILE, (char*)LOG_UCI_MODULE, (char*)key, level_buf))
    {
        return convert_log_level_from_string(level_buf);
    }

    return LOG_UNSET;
}

static void get_module_log_level(struct log_module_entry *entry) // calc module log level from uci
{
    int main_module_flag = 0;
    char uci_key[64] = {0};

    pthread_mutex_lock(&s_lynq_log_mutex);
    if (s_main_module_entry_index >= 0) //to check if this entry is main module if main module is set
    {
        strcpy(uci_key, s_log_module_entries[s_main_module_entry_index]->module_name);
        if (s_log_module_entries[s_main_module_entry_index] == entry)
        {
            main_module_flag = 1;
        }
    }
    pthread_mutex_unlock(&s_lynq_log_mutex);

    entry->global_level = get_uci_log_value("global_level");
    entry->exe_level = get_uci_log_value(uci_key);
    if (main_module_flag == 1) // if this entry is main module no need to get the special level
    {
        entry->lib_level = LOG_UNSET;
        entry->special_level = LOG_UNSET;
    }
    else
    {
        entry->lib_level = get_uci_log_value(entry->module_name);
        strcat(uci_key, "__");
        strcat(uci_key, entry->module_name);
        entry->special_level = get_uci_log_value(uci_key);
    }
    entry->calc_level();
}

static LYNQ_WRITE_LOG_PTR inner_log_function_init(const char *log_name, int main_module_flag)
{
    struct log_module_entry *entry = NULL;
    if(log_name == NULL)
    {
        log_name = "MAIN";
    }
    pthread_mutex_lock(&s_lynq_log_mutex);
    if (s_log_module_array_init_flag == 0) // to init module array if is not init
    {
        reg_write_log_function<entry_count-1>();
        s_log_module_array_init_flag = 1;
    }

    if (s_curr_reg_count < entry_count) // if module entry is not use out
    {
        entry = s_log_module_entries[s_curr_reg_count];
        if (main_module_flag == 1) // set s_main_module_entry_index when main module call
        {
            s_main_module_entry_index = s_curr_reg_count;
        }
        s_curr_reg_count++;

        entry->level = LOG_DEBUG;
        strncpy(entry->module_name, log_name, sizeof(entry->module_name));
    }
    pthread_mutex_unlock(&s_lynq_log_mutex);

    if (entry == NULL)
    {
        return NULL;
    }

    get_module_log_level(entry); // init the log level

    return entry->log_ptr;
}

/**
 * @brief lynq_log_function_init , this function called by lib
 * @param log_name
 * @return
 */
LYNQ_WRITE_LOG_PTR lynq_log_function_init(const char *log_name)
{
    return inner_log_function_init(log_name, 0);
}

/**
 * @brief lynq_log_configuration_init , this function called by main module
 * @param log_name
 */
void lynq_log_configuration_init(const char *log_name)
{
    log_level_enum exe_level;
    char exe_key[64] = {0};
    char special_key[64] = {0};

    pthread_mutex_lock(&s_lynq_log_mutex);
    for(int i=0; i < s_curr_reg_count; i++) // to check if log module is registered, bring it to main module
    {
        if (strcmp(log_name, s_log_module_entries[i]->module_name) == 0)
        {
            s_main_module_entry_index = i;
            break;
        }
    }
    pthread_mutex_unlock(&s_lynq_log_mutex);

    if (s_main_module_entry_index == -1) // if not registered yet, init a main module entry
    {
        inner_log_function_init(log_name, 1);
    }

    pthread_mutex_lock(&s_lynq_log_mutex);
    strcpy(exe_key, s_log_module_entries[s_main_module_entry_index]->module_name);
    exe_level = get_uci_log_value(exe_key);
    for(int i=0; i < s_curr_reg_count; i++) //recalc the module level when main module is set
    {
        s_log_module_entries[i]->exe_level = exe_level; // set the main module level
        if (i == s_main_module_entry_index)
            continue;

        sprintf(special_key, "%s__%s", exe_key, s_log_module_entries[i]->module_name);
        s_log_module_entries[i]->special_level = get_uci_log_value(special_key); // get the special level again
        s_log_module_entries[i]->calc_level();
    }
    pthread_mutex_unlock(&s_lynq_log_mutex);
}

void lynq_log_global_output(log_level_enum level,const char *format,...)
{
    if (s_main_module_entry_index == -1)
        return;

    va_list args;
    va_start(args,format);
    s_log_module_entries[s_main_module_entry_index]->log_ptr(level, format, args);
    va_end(args);
}

const char* lynq_read_log_version()
{
    return "LOG-V1.0";
}

#ifdef MOBILETEK_TARGET_PLATFORM_T106
/**
 * @brief Set the syslog file size
 *
 * @param syslog file size (M)
 * @return 0:success other:fail
 */
int lynq_syslog_set_file_size(int value)
{
    char lynq_syslog_data[64] = {0};

    if(value > 100)
    {
        value = 100;
    }
    else if( value < 1)
    {
        value = 1;
    }
    snprintf(lynq_syslog_data,sizeof(lynq_syslog_data),"%d",value);//Converts bytes to M
    if (0 != sc_cfg_set("syslog_file_size", lynq_syslog_data))
    {
        return -1;
    }
    return sc_cfg_save();
}

/**
 * @brief Get the syslog file size
 *
 * @param log data
 * @return 1-100:success -1:fial
 */
int lynq_syslog_get_file_size(void)
{
    int lynq_syslog_size = 0;
    char lynq_syslog_data[64] = {0};
    if(0 == sc_cfg_get("syslog_file_size", lynq_syslog_data, 64))
    {
        lynq_syslog_size = atoi(lynq_syslog_data);
        return lynq_syslog_size;
    }
    return -1;
}

/**
 * @brief Set the syslog file rotate
 *
 * @param syslog file rotate number
 *
 * @return 0:success other:fial
 */
int lynq_syslog_set_file_rotate(int value)
{
    char lynq_syslog_data[64] = {0};

    if(value < 0)
    {
        value = 0;
    }
    else if(value > 99)
    {
        value = 99;
    }

    snprintf(lynq_syslog_data,sizeof(lynq_syslog_data),"%d",value);
    if (0 != sc_cfg_set("syslog_file_num", lynq_syslog_data))
    {
        return -1;
    }
    return sc_cfg_save();
}

/**
 * @brief Example Set the number of syslog files cut
 *
 * @param log data
 * @return 0-99:success -1:fial
 */
int lynq_syslog_get_file_rotate(void)
{
    int lynq_syslog_rotate = 0;
    char lynq_syslog_data[64] = {0};
    if(0 == sc_cfg_get("syslog_file_num", lynq_syslog_data, 64))
    {
        lynq_syslog_rotate = atoi(lynq_syslog_data);
        return lynq_syslog_rotate;
    }
    return -1;
}

#else
/**
 * @brief Set the syslog file size
 *
 * @param syslog file size (M)
 * @return 0:success other:fial
 */
int lynq_syslog_set_file_size(int value)
{
    char lynq_syslog_data[64] = {0};

    if(value <10)
    {
      value = 10;
    }
    else if(value > 1024)
    {
        value = 1024;
    }
    snprintf(lynq_syslog_data,sizeof(lynq_syslog_data),"%d",(value*1024*1024));//Converts bytes to M
    return lynq_set_value(LOG_UCI_MODULE,"syslog_flie_size", lynq_syslog_data);
}

/**
 * @brief Get the syslog file size
 *
 * @param log data
 * @return 10-1024:success -1:fial
 */
int lynq_syslog_get_file_size(void)
{
    int lynq_syslog_size = 0;
    char lynq_syslog_data[64] = {0};
    if(0 == lynq_get_value(LOG_UCI_FILE, LOG_UCI_MODULE, "syslog_flie_size", lynq_syslog_data))
    {
        lynq_syslog_size = ((atoi(lynq_syslog_data)) / 1024) /1024;
        return lynq_syslog_size;
    }
    return -1;
}

/**
 * @brief Set the syslog file rotate
 *
 * @param syslog file rotate number
 *
 * @return 0:success other:fial
 */
int lynq_syslog_set_file_rotate(int value)
{
    char lynq_syslog_data[64] = {0};

    if(value < 0)
    {
        value = 0;
    }
    else if(value > 100)
    {
        value = 100;
    }

    snprintf(lynq_syslog_data,sizeof(lynq_syslog_data),"%d",value);
    return lynq_set_value(LOG_UCI_MODULE, "syslog_flie_rotate", lynq_syslog_data);
}

/**
 * @brief Example Set the number of syslog files cut
 *
 * @param log data
 * @return 10-1024:success -1:fial
 */
int lynq_syslog_get_file_rotate(void)
{
    int lynq_syslog_rotate = 0;
    char lynq_syslog_data[64] = {0};
    if(0 == lynq_get_value(LOG_UCI_FILE, LOG_UCI_MODULE, "syslog_flie_rotate", lynq_syslog_data))
    {
        lynq_syslog_rotate = atoi(lynq_syslog_data);
        return lynq_syslog_rotate;
    }
    return -1;
}
#endif

//log level api

int lynq_set_log_level(const char * module_name, log_level_enum level)
{
    char level_buf[64] = {0};
    if (module_name == NULL || module_name[0] == '\0')
    {
        __android_log_print(ANDROID_LOG_ERROR, "LYNQ_LOG", "lynq_set_log_level: bad module name");
        return -1;
    }

    if (level > LOG_UNSET && level < LOG_LEVEL_MAX)
    {
        sprintf(level_buf, "%s", LogLevelNameInfoTable[level]);
    }
    else if (level != LOG_UNSET)
    {
        __android_log_print(ANDROID_LOG_ERROR, "LYNQ_LOG", "unkown level %d", level);
        return -1;
    }

    return lynq_set_value((char*)LOG_UCI_MODULE, (char*)module_name, level_buf);
}

int lynq_get_log_level(const char * module_name, log_level_enum *level)
{
    char level_buf[64] = {0};
    if (level == NULL || module_name == NULL || module_name[0] == '\0')
    {
        __android_log_print(ANDROID_LOG_ERROR, "LYNQ_LOG", "lynq_get_log_level: bad param");
        return -1;
    }

    if (0 == lynq_get_value((char*)LOG_UCI_FILE, (char*)LOG_UCI_MODULE, (char*)module_name, level_buf))
    {
        *level = convert_log_level_from_string(level_buf);
        return 0;
    }

    return -1;
}

int lynq_set_special_log_level(const char * exe_name, const char * module_name, log_level_enum level)
{
    char special_name[128] = {0};
    if (exe_name == NULL || module_name == NULL || exe_name[0] == '\0' || module_name[0] == '\0')
    {
        __android_log_print(ANDROID_LOG_ERROR, "LYNQ_LOG", "lynq_set_special_log_level: bad param");
        return -1;
    }

    sprintf(special_name, "%s__%s", exe_name, module_name);
    return lynq_set_log_level(special_name, level);
}

int lynq_get_special_log_level(const char * exe_name, const char * module_name, log_level_enum *level)
{
    char special_name[128] = {0};
    if (exe_name == NULL || module_name == NULL || exe_name[0] == '\0' || module_name[0] == '\0')
    {
        __android_log_print(ANDROID_LOG_ERROR, "LYNQ_LOG", "lynq_get_special_log_level: bad param");
        return -1;
    }

    sprintf(special_name, "%s__%s", exe_name, module_name);
    return lynq_get_log_level(special_name, level);
}

int lynq_notify_recalc_log_level(pid_t pid)
{
    char cmd_buf[64];
    sprintf(cmd_buf, " [ -d /tmp/log_level/%d ] && kill -%d %d", pid, SIGUSR1, pid);
    return system(cmd_buf);
}

#define MAX_PIDS 128
int kill_pid(char *command,int signal)
{
    int pids[MAX_PIDS];
    int pid_count = 0;
    FILE *fp = popen(command, "r");
    if (fp == NULL)
    {
        perror("popen failed");
        return -1;
    }
    char line[1024];
    while(fgets(line, sizeof(line), fp) != NULL)
    {
        if (pid_count < MAX_PIDS)
        {
            pids[pid_count] = atoi(line);
            kill(pids[pid_count], signal);
            pid_count++;
        }  
    }
    pclose(fp);
    return 0;
}

int lynq_notify_recalc_log_level_bottom(char *log_level)
{
    sc_cfg_set("print_level", log_level);
	sc_cfg_set("syslog_level", log_level);

    const char* process_names[] = {
        "at_ctl",
        "mnet_whitelist_proxy",
        "rtc-service",
        "msm_svr",
        "zxic_mainctrl",
        "zxic_mmi",
        "lynq-ril-service"
    };
    const int process_count = sizeof(process_names) / sizeof(process_names[0]);
    
    
    char command[1024] = "";
    for (int i = 0; i < process_count; i++)
    {
        strcat(command, "pidof ");
        strcat(command, process_names[i]);
        if (i < process_count - 1)
            strcat(command, " && ");
    }
/*close zxic-process kill SIGUSR2*/
    kill_pid(command, SIGUSR2);
/*close lynq-process kill SIGUSR1*/
    kill_pid("ls /tmp/log_level/", SIGUSR1);
    return 0;
}


static void log_signal_handler(int signum) {
    if (SIGUSR1 != signum)
        return;
    // 处理信号
    log_level("syslog_level");
    log_level("print_level");
    for(int i=0; i < s_curr_reg_count; i++)
    {
        get_module_log_level(s_log_module_entries[i]);
    } 
}

void __attribute__((constructor)) lynq_log_init()
{

    pid_t pid;
    char cmd_buf[64];
    if (SIG_ERR != signal(SIGUSR1, log_signal_handler))
    {
        pid = getpid();
        sprintf(cmd_buf, "mkdir -p /tmp/log_level/%d", pid);
        system(cmd_buf);
    }

    pthread_mutex_lock(&s_lynq_log_mutex);
    if (s_log_module_array_init_flag == 0)
    {
        reg_write_log_function<entry_count-1>();
        s_log_module_array_init_flag = 1;
    }
    pthread_mutex_unlock(&s_lynq_log_mutex);
}

