/*******************************************************************************
 * Copyright (C) 2007, ZTE Corporation.
 *
 * File Name:    
 * File Mark:    
 * Description:  This source file contains the Answer To Reset (ATR) statemachine engine. 
 * Others:        
 * Version:       v0.1
 * Author:        mtr
 * Date:          2007-05-11
 * History 1:      
 *     Date: 
 *     Version:
 *     Author: 
 *     Modification:  
 * History 2: 
  ********************************************************************************/


/****************************************************************************
* 	                                           Include files
****************************************************************************/
#include "uicc_stub.h"

#include "uicc.h"
#include "uicc_ctrl.h"
#include "uicc_access.h"
#include "uicc_drv.h"
#include "uicc_trc.h"

#include "USIM_reg.h"
#include "drvs_ramlog.h"
#include "drvs_pm.h"


/****************************************************************************
* 	                                           Local Macros
****************************************************************************/
#define UICC_T_0                 0
#define UICC_T_1                 1
#define UICC_T_15               15

#define T_INITIAL_DELAY         5000 /* [us] The inital delay before actualy activating. */
#define T_WARM_RESET_PERIOD      2000 /* [us] The reset period of a warm reset (RST low period).*/
#define T_ERROR_DELAY           6000 /* [us] The waiting time once an error occured. */
#define T_DELAY_VCC_TO_IO        5000//1000 /* [us] The delay between activating Vcc and I/O. */  
#define T_DELAY_IO_TO_CLK        600 /* [us] The delay between activating I/O an clock. */  
#define T_MAX_ATR_DELAY         15000 /* [us] The time waiting for the card to activate (to start  transmitting ATR characters. */
#define T_EVALUATION_DELAY       5000 /* [us] Delay ensuring an appropriate evaluation distance. */

#define TS_DIRECT_CONVENTION  0x3B
#define TS_INVERSE_CONVENTION 0x03 /* The direct convention interpretation
                                      of the inverse convention TS (0x3F) */

#define TS_INVERSE_TRUE_VALUE 0x3F

/*  Interface character masks  */
#define TABCD_MASK (Y & 0xF0)
#define TBCD_MASK  (Y & 0xE0)
#define TCD_MASK   (Y & 0xC0)
#define TD_MASK    (Y & 0x80)
#define TC_MASK    (Y & 0x40)
#define TB_MASK    (Y & 0x20)
#define TA_MASK    (Y & 0x10)

/****************************************************************************
* 	                                           Local Types
****************************************************************************/
typedef enum
{
  NO_ANSWER,
  WRONG_TS,
  IWT_TIMEOUT,
  NO_TCK,
  WRONG_TCK,
  WRONG_TD1
} T_ATR_ERROR_CAUSE;


/****************************************************************************
* 	                                           Local Constants
****************************************************************************/

/****************************************************************************
* 	                                           Local Variables
****************************************************************************/
static T_ATR_ERROR_CAUSE  atr_error_cause;
static ubyte              check_sum;
static ubyte              remaining_hist_char;
static ubyte              current_indicated_protocol;
static ubyte              Y; 
static ubyte              pi1;

/* The following varialbles are ysed to form the profile of a given card. */
static ubyte  di;           /* Baud Rate adjustment factor [TA(1)] */
static ubyte  fi;           /* Clock Rate conversion factor [TA(1)] */
static ubyte  N;            /* Extra Guard Time [TC(1)] */
static ushort protocol;     /* Supported protocols [TD(i)] in a bit-field (T0=..0001, T1=..0010) [TD(i)] */
static ubyte  wi;           /* Work Waiting Time calculatiopn factor [TC(2)] */
static ubyte  ifs;          /* Information Field Size (IFSC & IFSD) [first T=1 TA(i)] */
static ubyte  cwi;          /* Character Waiting Time (CWI) [first T=1 TB(i)] */
static ubyte  bwi;          /* Block Waiting Time (BWI) [first T=1 TB(i)] */
static ubyte  edc;          /* Error Detection Code (EDC) [first T=1 TC(i)] */
static ushort xi_ui;        /* Clock Stop Mode (XI) and Class (voltage) Indicator (UI) [first T=1 TA(i)] */
static ushort ta2_specific; /* Specific mode (negotiable/specific) byte. */
static T_COM_CONVENTION  convention;  /* Indicates the communication convention (dir/inv). */

#ifndef UICC_NO_DEBUG
  static T_UICC_ATR_CTRL_STATE  debug          = UICC_ATR_NONE;
  static ubyte                  already_logged = 0;
#endif

/****************************************************************************
* 	                                           Local Function Prototypes
****************************************************************************/

/****************************************************************************
* 	                                          Global Constants
****************************************************************************/

/****************************************************************************
* 	                                          Global Variables
****************************************************************************/
  ubyte               UICC_nof_atr_characters;
  ubyte               UICC_atr_characters[UICC_MAX_ATR_CHARACTERS];
  T_UICC_CARD_PROFILE UICC_card_profile[UICC_NOF_CARD_SLOTS];

T_UICC_ATR_CTRL_STATE     UICC_atr_ctrl_state = UICC_ATR_READY;
extern VOID uiccDebug_Printf(char *fmt,...);

/****************************************************************************
* 	                                          Function Definitions
****************************************************************************/

/*******************************************************************************
* Function:... handle_history_and_tck
* Parameters:. None.
* Returns:.... The next state (history, tck or succeded.
* Description: This function is called when no more interface characters are
*              present in the ATR sequence. The function will the detirmine if
*              historical charatcers are to received, if a checksum character
*              is to be received of if the end of the ATR sequence has been
*              reached.
* Created:.... 29.08.01 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static T_UICC_ATR_CTRL_STATE handle_history_and_tck(void)
{
  T_UICC_ATR_CTRL_STATE next_state;

  if(remaining_hist_char)
  { 
    next_state = UICC_HISTORY;
  }
  else if(protocol > 1)  /* <---- T0 is the first bitfield in the variable */ 
  { /* Only if T=0 is indicated (possibly by default) byte TCK shall be absent. In all 
       other cases, byte TCK shall be present. */
    next_state = UICC_TCK;
  }
  else
  {
    next_state = UICC_ATR_SUCCEDED;

#ifdef UICC_NOT_TIMER
	next_state = UICC_ATR_READY;
	UICC_atr_ctrl_state = UICC_ATR_READY;

     /* Now evaluate the if some requirements has been violated. If that
        is the case the ATR sequence will be rejected!
        GSM 11.11 states that PI1 (TB1 parameter) must - if present - be
        equal to 0 or 5. Also the parameter N (TC1 parameter) must take on
        the values 0 or 0xFF in order to be accepted. Otherwise should the
        card be rejected. */

    if((((pi1 &= 0x1F) == 0) || ((pi1 &= 0x1F) == 5)) &&
     ((N == 0x00) || (N == 0xFF)))
    { /* ATR accepted! */
    /*===============*/

    if((ifs==0) || (ifs==0xFF))
    { /* Illigal values of IFS -> use default value. */
      ifs = UICC_DEFAULT_IFS;
    }

    /* Since the activation succeeded -> store the activation profile of the card. */
    UICC_card_profile[UICC_current_reader].di           = di;         
    UICC_card_profile[UICC_current_reader].fi           = fi;         
    UICC_card_profile[UICC_current_reader].N            = N;          
    UICC_card_profile[UICC_current_reader].T_indicated  = protocol;   
    UICC_card_profile[UICC_current_reader].wi           = wi;         
    UICC_card_profile[UICC_current_reader].ifsc         = ifs;        
    UICC_card_profile[UICC_current_reader].ifsc_used    = ifs;
    UICC_card_profile[UICC_current_reader].ifsd         = UICC_DEFAULT_IFS;        
    UICC_card_profile[UICC_current_reader].ifsd_used    = UICC_DEFAULT_IFS;        
    UICC_card_profile[UICC_current_reader].cwi          = cwi;        
    UICC_card_profile[UICC_current_reader].bwi          = bwi;        
    UICC_card_profile[UICC_current_reader].edc          = edc;        
    UICC_card_profile[UICC_current_reader].class_clock  = xi_ui;         
    UICC_card_profile[UICC_current_reader].ta2_specific = ta2_specific;         
    UICC_card_profile[UICC_current_reader].convention   = convention;                 

    UICC_card_profile[UICC_current_reader].voltage_settled      = FALSE; 
    UICC_card_profile[UICC_current_reader].used_clock_stop_mode = UICC_CLOCK_STOP_NOT_ALLOWED;

    UICC_card_ctrl(UICC_ATR_OK);
    }
    else
    { /* ATR rejected! */
    /*===============*/
    /* Signal the responsible access engine of the ATR result*/
    UICC_card_ctrl(UICC_ATR_FAILED);
    }		  
#else
    UICC_start_timer(T_EVALUATION_DELAY);
#endif

  }

  return(next_state);
}


/*******************************************************************************
* Function:... initiate_atr_variables
* Parameters:. None.
* Returns:.... None.
* Description: This function initializes the variables used by the ATR state-
*              machine.
* Created:.... 30.08.01 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static void initiate_atr_variables(void)
{
  #ifndef UICC_NO_DEBUG
    already_logged = 0;
  #endif

  UICC_nof_atr_characters    = 0;
  remaining_hist_char        = 0;
  fi                         = UICC_F_DEFAULT;
  di                         = UICC_D_DEFAULT;
  N                          = UICC_DEFAULT_N;
  current_indicated_protocol = UICC_T_0;
  wi                         = UICC_DEFAULT_WI;        
  ifs                        = UICC_DEFAULT_IFS;
  cwi                        = UICC_DEFAULT_CWI;        
  bwi                        = UICC_DEFAULT_BWI;
  edc                        = UICC_DEFAULT_EDC;
  xi_ui                      = 0x0000;
  ta2_specific               = 0x0000;
  
  /* If TD(1) is absent, the only offered protocol is T=0 and this is indicated in the 
     initial value 0x01. If TD(1) on the other hand is present, this initial value will 
     be overruled when analyzing the TD(1) indicated protocol. */
  protocol                   = 0x01; 
  pi1                        = 0;
}

/*******************************************************************************
* Function:... initial_uart_settings
* Parameters:. None.
* Returns:.... None.
* Description: This function initializes the UART.
* Created:.... 05.09.01 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static void initial_uart_settings(void)
{
  //clear FIFO interrupt register
  SIM_RUN_CTRL = 0;			
  SIM_FIFO_ICR = 0x3FF;
  
  UICC_disable_card_interface(); /* STOP the USIM block if it's running*/ 
  UICC_transmission_rate_control(93);  /* (F=372, D=1). */  
  UICC_enable_card_interface();
  UICC_switch_uart_clock_off();/*add 071219*/
  UICC_select_character_mode();
  UICC_select_direct_convention();
  UICC_disable_parity_error_detection();
  UICC_enable_usim_ok_isr();
  UICC_enable_character_interrupt();
  UICC_enable_parity_error_interrupt();
  UICC_enable_overrun_error_interrupt();
  UICC_disable_t0_end_interrupt();
}


/*******************************************************************************
* Function:... fetch_and_store_char
* Parameters:. None.
* Returns:.... The character from SIMRX.
* Description: This function is used to fetch and store the received ATR character.
* Created:.... 17.03.04 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static unsigned char fetch_and_store_char(void)
{
  /* Fetch the character and maintain the ATR storage variables. */
  if(UICC_nof_atr_characters < UICC_MAX_ATR_CHARACTERS) 
    return (UICC_atr_characters[UICC_nof_atr_characters++] = UICC_get_character());
  else
    return (UICC_get_character());
}

/*---------------------------------------------*/
/* Global functions.                           */
/*---------------------------------------------*/

/*******************************************************************************
* Function:... UICC_get_atr_historical_offset
* Parameters:. atr:     Pointer to the start of the ATR string.
*              nof_atr: Number of ATR characters.
* Returns:.... The offset in the ATR string in which the historical characters can 
*              be found.
* Description: Function for conveying the historical character offset within the 
*              ATR string.
* Created:.... 07.01.05 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
ubyte UICC_get_atr_historical_offset(ubyte *atr, ubyte nof_atr)
{  
  ubyte uicc_current, offset;
  ubyte found = FALSE;

  if((atr[1]&0x0F)==0)
  { /* No historical character available */
    offset=nof_atr;
  }
  else
  { /* Find the historical offset. */
    offset = uicc_current = 1;
	
    while(!found)
    {
      if(atr[uicc_current]&0x10) offset++; /* TA(i) present, count one. */
      if(atr[uicc_current]&0x20) offset++; /* TB(i) present, count one. */
      if(atr[uicc_current]&0x40) offset++; /* TC(i) present, count one. */
      if(atr[uicc_current]&0x80) 
      { /* TD(i) present, count one and point to TD(i+1). */
        offset++;
        uicc_current=offset;
        if(offset>nof_atr)
        { /* Safety check. */
          offset=nof_atr;
          found=TRUE;
        }
      }
      else
      { /* TD(i) NOT present -> offset points to the ATR position right before 
           the historical character(s). */
        offset++;
        found=TRUE;
      }
    }
  }

  return(offset);
}


/*******************************add 071219************************************************
* Function:... UICC_get_atr_protocols
* Parameters:. atr:     Pointer to the start of the ATR string.
*              nof_atr: Number of ATR characters.
* Returns:.... The found protocols.
* Description: Function for find the protocols indicated in the ATR string
*******************************************************************************/
extern ushort UICC_get_atr_protocols(ubyte *atr, ubyte nof_atr)
{
  ubyte uicc_current = 1, offset = 1, end_found = FALSE;
  ushort protocol = 0;

  while(!end_found)
  {
    if(atr[uicc_current]&0x10) offset++; /* TA(i) present, count one. */
    if(atr[uicc_current]&0x20) offset++; /* TB(i) present, count one. */
    if(atr[uicc_current]&0x40) offset++; /* TC(i) present, count one. */
    if(atr[uicc_current]&0x80) 
    { /* TD(i) present. */
      offset++;

      /* Extract the indicated protocol from the LBS nipple of TD(i). */
      protocol |= 1<<(atr[uicc_current]&0x0F); 
      
      uicc_current=offset; /* Point to TD(i+1). */

      if(offset>nof_atr)
      { /* Safety check. */
        end_found = TRUE;
      }
    }
    else
    { /* TD(i) NOT present -> no more TD character in the ATR string. */
      if(uicc_current == 1)
      { /* Format character includes no TD(1). */
        protocol = 0x01; /* Only offered protocol is T=0 - implicit. */
      }
      end_found = TRUE;
    }
  }
  return(protocol);
}
/*******************************************************************************
* Function:... UICC_ctrl_atr
* Parameters:. event: The event triggering the statemachine.
* Returns:.... -
* Description: This statemachine/engine is controlling the Answer To Reset (ATR)
*              activation sequence.
* Created:.... 29.08.01 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
void UICC_reset_atr(void)
{
  UICC_atr_ctrl_state = UICC_ATR_READY;
}

int atrcount = 0;
extern T_ZDrvPmic_Voltage gVsimType;


void wait_time_tmp(UINT32 t)
{	
  	while ((t--) > 0);
}
typedef struct _T_UiccDebugInfo
{
	CHAR debugInfo[41];
}
T_UiccDebugInfo;

T_UiccDebugInfo UiccAtrStatusInfo[] =
{
  "UICC_ATR_NONE",
  "UICC_ATR_READY",
  "UICC_COLD_ENABLING",
  "UICC_WARM_ENABLING",
  "UICC_ACTIVATING_VCC",

  "UICC_ACTIVATING_IO",
  "UICC_TS_INTERNAL_RESET",
  "UICC_TS_ACTIVE_LOW_RESET",
  "UICC_T0",
  "UICC_TA1", 

  "UICC_TB1", 
  "UICC_TC1", 
  "UICC_TD1",
  "UICC_TA2", 
  "UICC_TB2", 

  "UICC_TC2", 
  "UICC_TDI",
  "UICC_TAI", 
  "UICC_TBI", 
  "UICC_TCI",

  "UICC_HISTORY",
  "UICC_TCK",
  "UICC_ATR_SUCCEDED",
  "UICC_ATR_ERROR"
};


T_UiccDebugInfo UiccAtrEventInfo[] =
{
    "UICC_DUMMY_INT_EVENT",
  "UICC_TIMER_EXPIRED",          
  "UICC_GUARD_TIMER_EXPIRED",
  "UICC_CHARACTER_OK",
  "UICC_T0_END",
  "UICC_T1_END",
  "UICC_DMA_END",
  "UICC_PARITY_ERROR",
  "UICC_OVERRUN_ERROR",
  "UICC_CARD_IN",
  "UICC_CARD_OUT",
  "UICC_CHTIMEOUT",
  "UICC_T1_PARITY_ERROR",
  "UICC_T1_BWT_EXPIRED",
  "UICC_UNKNOWN_INT",
  "UICC_FIFO_INT",
  "UICC_RX_LSREQ",  
  "UICC_TX_LSREQ",  
  "UICC_RX_SREQ",  
  "UICC_TX_SREQ",    
  "UICC_RX_LBREQ",  
  "UICC_TX_LBREQ",  
  "UICC_RX_BREQ",  
  "UICC_TX_BREQ",  
  "UICC_TXF_OFL_IRQ",  
  "UICC_RXF_UFL_IRQ",
  "UICC_FIFO_UNKNOWN_INT",
    "UICC_LAST_INT_EVENT",
    /**"UICC_INITIATE_WARM_ATR",**/
    "UICC_INITIATE_COLD_ATR",
    "UICC_RESET_ATR",
    "UICC_LAST_ATR_EVENT"
};


void UICC_ctrl_atr(T_UICC_ATR_EVENTS event)
{


#ifdef UICC_DEBUG_INF
 UICC_debug[UICC_debug_count][0] = 1;  
UICC_debug[UICC_debug_count][1] = UICC_atr_ctrl_state;
UICC_debug[UICC_debug_count][2] = event;
UICC_debug_count = (UICC_debug_count+1) % 200;
#endif
if(atrcount++ > 1000)
	atrcount = 0;
int i = 0;
      
  i = UICC_atr_ctrl_state;
    
  zDrvRamlog_PRINTF(RAMLOG_MOD_UICC,"[%d]state=%s,event=%s\n", zOss_GetTickCount(), (char *)(&UiccAtrStatusInfo[i]),(char *)(&UiccAtrEventInfo[event]));
  uiccramlog_Printf("[%d]state=%s,event=%s\n", zOss_GetTickCount(), (char *)(&UiccAtrStatusInfo[i]),(char *)(&UiccAtrEventInfo[event]));

  switch(UICC_atr_ctrl_state)
  {
    /*=========================*/
    /* Statemachine not active */
    /*=========================*/
    case UICC_ATR_READY:
      switch(event)
      {
        case UICC_INITIATE_COLD_ATR:
          initiate_atr_variables();
          UICC_atr_ctrl_state = UICC_COLD_ENABLING;
          SIM_LDO &= ~(0x1<<1);
          UICC_activate_vcc(UICC_current_reader, UICC_card_profile[UICC_current_reader].used_voltage, 1);
          UICC_get_character(); /* To clear the Rx register. */
          initial_uart_settings();
         // SIM_CTRL |= 0x2000;
          UICC_start_timer(T_INITIAL_DELAY);          
		  
#ifdef UICC_NOT_TIMER
		  UICC_control_card_access(UICC_TIMER_EXPIRED);
#endif

          break;

        case UICC_INITIATE_WARM_ATR:
          initiate_atr_variables();
          UICC_atr_ctrl_state = UICC_WARM_ENABLING;
	   SIM_LDO &= ~(0x1<<1);
          initial_uart_settings();
          UICC_deactivate_rst(UICC_current_reader);      /* Begin the RESET deactivation period. */
          UICC_get_character(); /* To clear the Rx register. */
          UICC_start_timer(T_WARM_RESET_PERIOD);
#ifdef UICC_NOT_TIMER
		  UICC_control_card_access(UICC_TIMER_EXPIRED);
#endif

	  
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          break;
      }
      break;

    /*===============================*/
    /* Activating in COLD reset mode */
    /*===============================*/
    case UICC_COLD_ENABLING:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          UICC_atr_ctrl_state = UICC_ACTIVATING_VCC;
		  #ifdef MTNET
//		  HAL_TestPrintf("will call UICC_activate_vcc\n");
		  
		  zDrvRamlog_PRINTF(RAMLOG_MOD_UICC, "will call UICC_activate_vcc\n");
   		  uiccramlog_Printf("will call UICC_activate_vcc\n");
		  #endif
          UICC_activate_vcc(UICC_current_reader, UICC_card_profile[UICC_current_reader].used_voltage, 2);
          //UICC_start_timer(T_DELAY_VCC_TO_IO);
#ifdef UICC_NOT_TIMER
		  UICC_control_card_access(UICC_TIMER_EXPIRED);
#endif

  
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          break;
      }
      break;

    /*===============================*/
    /* Activating in WARM reset mode */
    /*===============================*/
    case UICC_WARM_ENABLING:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          UICC_atr_ctrl_state = UICC_TS_ACTIVE_LOW_RESET;
          UICC_activate_rst(UICC_current_reader);
		  UICC_switch_uart_clock_on();//09.11.05 add
          UICC_start_timer(T_MAX_ATR_DELAY);
#ifdef UICC_NOT_TIMER
	   UICC_control_card_access(UICC_TIMER_EXPIRED);
#endif
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          break;
      }
      break;

    /*================*/
    /* Activating Vcc */
    /*================*/
    case UICC_ACTIVATING_VCC:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          UICC_atr_ctrl_state = UICC_ACTIVATING_IO;
		  
	   if(gVsimType ==  PM_VOLT_3_000)
			SIM_LDO &= ~(0x1<<1);
	   else if(gVsimType == PM_VOLT_1_800)
	   		SIM_LDO |= (0x1<<1); 

          //wait_time_tmp(5000);
		 
          UICC_enable_io(UICC_current_reader);
          UICC_start_timer(T_DELAY_IO_TO_CLK);
#ifdef UICC_NOT_TIMER
		  UICC_control_card_access(UICC_TIMER_EXPIRED);
#endif

 
          break;

        default:
          UICC_switch_uart_clock_off();
          UICC_get_character(); /* To clear the Rx register. */
          UICC_atr_ctrl_state = UICC_ATR_ERROR;
          atr_error_cause = NO_ANSWER;
          UICC_start_timer(T_ERROR_DELAY);
          break;
      }
      break;

    /*================*/
    /* Activating I/O */
    /*================*/
    case UICC_ACTIVATING_IO:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          UICC_atr_ctrl_state = UICC_TS_INTERNAL_RESET;
          UICC_select_direct_convention();
          UICC_disable_parity_error_detection();
          UICC_activate_clk(UICC_current_reader);
          UICC_switch_uart_clock_on();/*add by wangxia 080513*/
          UICC_start_timer(T_MAX_ATR_DELAY);
#ifdef UICC_NOT_TIMER
		  UICC_control_card_access(UICC_TIMER_EXPIRED);
#endif

  
          break;

        default:
          UICC_switch_uart_clock_off();
          UICC_get_character(); /* To clear the Rx register. */
          UICC_atr_ctrl_state = UICC_ATR_ERROR;
          atr_error_cause = NO_ANSWER;
          UICC_start_timer(T_ERROR_DELAY);
          break;
      }
      break;

    /*====================================*/
    /* Activating an Internal Reset card. */
    /*====================================*/
    case UICC_TS_INTERNAL_RESET:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          /* Well it was not an internal reset card. Let's see if it will work if RST is raised! */
          UICC_atr_ctrl_state = UICC_TS_ACTIVE_LOW_RESET;
          UICC_activate_rst(UICC_current_reader);	
		uiccDebug_Printf("%d c cold reset UICC_TS_ACTIVE_LOW_RESET\n",atrcount);
          UICC_start_timer(T_MAX_ATR_DELAY);
          break;

        case UICC_CHARACTER_OK:
          /* Check the TS character value. */
          if (UICC_get_character() == TS_DIRECT_CONVENTION)
          {  /* TS was valid - await next ATR chahacter. */
            UICC_atr_ctrl_state = UICC_T0;
            UICC_atr_characters[UICC_nof_atr_characters++] = TS_DIRECT_CONVENTION;
            convention = UICC_DIRECT;
            UICC_enable_parity_error_detection();
            UICC_start_timer(UICC_T_IWT);
			
	    uiccDebug_Printf("%d c cold reset UICC_CHARACTER_OK\n",atrcount);

          }
          else
          {  /* The character was not valid - ATR failed. */
            #ifndef UICC_NO_DEBUG
              debug = UICC_atr_ctrl_state;
            #endif
            UICC_switch_uart_clock_off();
            UICC_atr_ctrl_state = UICC_ATR_ERROR;
            atr_error_cause = NO_ANSWER; /*change  071219*/
            UICC_start_timer(T_ERROR_DELAY);

	    uiccDebug_Printf("%d c cold reset UICC_CHARACTER_OK1111\n",atrcount);
          }
          break;

        case UICC_PARITY_ERROR:
          /* Check the TS character value. */
          if (UICC_get_character() == TS_INVERSE_CONVENTION)
          {  /* TS was valid - await next ATR chahacter. */
            UICC_atr_ctrl_state = UICC_T0;
            UICC_atr_characters[UICC_nof_atr_characters++] = TS_INVERSE_TRUE_VALUE;
            convention = UICC_INVERSE;
            UICC_enable_parity_error_detection();
            UICC_select_inverse_convention();
            UICC_start_timer(UICC_T_IWT);

	    uiccDebug_Printf("uicc cold reset UICC_PARITY_ERROR\n");
          }
          else
          {  /* The character was not valid - ATR failed. */
            #ifndef UICC_NO_DEBUG
              debug = UICC_atr_ctrl_state;
            #endif
            UICC_switch_uart_clock_off();
            UICC_atr_ctrl_state = UICC_ATR_ERROR;
            atr_error_cause = NO_ANSWER; /*change 071219*/
            UICC_start_timer(T_ERROR_DELAY);

	    uiccDebug_Printf("uicc cold reset UICC_PARITY_ERROR1111\n");
          }
          break;

        case UICC_OVERRUN_ERROR:
        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          break;
      }
      break;

    /*======================================*/
    /* Activating an Active Low reset card. */
    /*======================================*/
    case UICC_TS_ACTIVE_LOW_RESET:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          /* Well it was not an active low reset either (at least there where no answer from the card) - ATR failed. */
          UICC_atr_ctrl_state = UICC_ATR_ERROR;
          atr_error_cause = NO_ANSWER;
          UICC_start_timer(T_ERROR_DELAY);
          break;

        case UICC_CHARACTER_OK:
          /* Check the TS character value. */
          if (UICC_get_character() == TS_DIRECT_CONVENTION)
          {  /* TS was valid - await next ATR chahacter. */
            UICC_atr_ctrl_state = UICC_T0;
            UICC_atr_characters[UICC_nof_atr_characters++] = TS_DIRECT_CONVENTION;
            convention = UICC_DIRECT;
            UICC_enable_parity_error_detection();
            UICC_start_timer(UICC_T_IWT);
          }
          else
          {  /* The character was not valid - ATR failed. */
            #ifndef UICC_NO_DEBUG
              debug = UICC_atr_ctrl_state;
            #endif
            UICC_atr_ctrl_state = UICC_ATR_ERROR;
            atr_error_cause = WRONG_TS;
            UICC_start_timer(T_ERROR_DELAY);
          }
          break;

        case UICC_PARITY_ERROR:
          /* Check the TS character value. */
          if (UICC_get_character() == TS_INVERSE_CONVENTION)
          {  /* TS was valid - await next ATR chahacter. */
            UICC_atr_ctrl_state = UICC_T0;
            UICC_atr_characters[UICC_nof_atr_characters++] = TS_INVERSE_TRUE_VALUE;
            convention = UICC_INVERSE;
            UICC_enable_parity_error_detection();
            UICC_select_inverse_convention();
            UICC_start_timer(UICC_T_IWT);
          }
          else
          {  /* The character was not valid - ATR failed. */
            #ifndef UICC_NO_DEBUG
              debug = UICC_atr_ctrl_state;
            #endif
            UICC_atr_ctrl_state = UICC_ATR_ERROR;
            atr_error_cause = WRONG_TS;
            UICC_start_timer(T_ERROR_DELAY);
          }
           break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          break;
      }
      break;

    /*===========================*/
    /* Await format character T0 */
    /* -------------------------- =======*/
    /* T0: - nof historical characters   */
    /*     - TA1-TD1 presence indication */
    /*===================================*/
    case UICC_T0:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
             /* No answer received within the Initial Waiting Time window - ATR failed. */
            #ifndef UICC_NO_DEBUG
              debug = UICC_atr_ctrl_state;
            #endif
            UICC_atr_ctrl_state = UICC_ATR_ERROR;
            atr_error_cause = IWT_TIMEOUT;
            UICC_start_timer(T_ERROR_DELAY);
          break;

        case UICC_CHARACTER_OK:
          check_sum = Y = fetch_and_store_char();

          remaining_hist_char = Y & 0x0F; /* Store the number of history characters */

          UICC_start_timer(UICC_T_IWT);

          /* Now examine the presence of the Interface Characters */
          if (TABCD_MASK == 0x00)
            UICC_atr_ctrl_state = handle_history_and_tck();
          else if(TA_MASK)
            UICC_atr_ctrl_state = UICC_TA1;
          else if(TB_MASK)
            UICC_atr_ctrl_state = UICC_TB1;
          else if(TC_MASK)
            UICC_atr_ctrl_state = UICC_TC1;
          else if(TD_MASK) 
            UICC_atr_ctrl_state = UICC_TD1;
          break;

        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(UICC_T_IWT);
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          break;
      }
      break;

    /*==========================*/
    /* Await interface byte TA1 */
    /* ------------------------ ===============*/
    /* TA1: - Clock rate convetion factor ~ FI */
    /*      - Baud rate adjustment factor ~ DI */
    /*=========================================*/
    case UICC_TA1:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          /* No answer received within the Initial Waiting Time window - ATR failed. */
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          UICC_atr_ctrl_state = UICC_ATR_ERROR;
              atr_error_cause = IWT_TIMEOUT;
          UICC_start_timer(T_ERROR_DELAY);
          break;

        case UICC_CHARACTER_OK:
        {
          ubyte temp;
          
          check_sum ^= temp = fetch_and_store_char();

          /* Mask the factors out */
          di = temp & 0x0F;
          fi = (temp & 0xF0) >> 4;

          UICC_start_timer(UICC_T_IWT);

          if (TBCD_MASK == 0x00)
            UICC_atr_ctrl_state = handle_history_and_tck();
          else if(TB_MASK)
            UICC_atr_ctrl_state = UICC_TB1;
          else if(TC_MASK)
            UICC_atr_ctrl_state = UICC_TC1;
          else if(TD_MASK)
            UICC_atr_ctrl_state = UICC_TD1;
          break;
        }
        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(UICC_T_IWT);
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          break;
      }
      break;

    /*==========================*/
    /* Await interface byte TB1 */
    /* ------------------------ ========*/
    /* TB1: - Programming current ~ II  */
    /*      - Programming voltage ~ PI1 */    
    /*==================================*/
    case UICC_TB1:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          /* No answer received within the Initial Waiting Time window - ATR failed. */
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          UICC_atr_ctrl_state = UICC_ATR_ERROR;
          atr_error_cause = IWT_TIMEOUT;
          UICC_start_timer(T_ERROR_DELAY);
          break;

        case UICC_CHARACTER_OK:
          check_sum ^= pi1 = fetch_and_store_char();

          UICC_start_timer(UICC_T_IWT);

          if (TCD_MASK == 0x00)
            UICC_atr_ctrl_state = handle_history_and_tck();
          else if(TC_MASK)
            UICC_atr_ctrl_state = UICC_TC1;
          else if(TD_MASK)
            UICC_atr_ctrl_state = UICC_TD1;
          break;

        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(UICC_T_IWT);
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          break;
      }
      break;

    /*==========================*/
    /* Await interface byte TC1 */
    /* ------------------------ ==========*/
    /* TC1: - Extra Guard Time factor ~ N */
    /*====================================*/
    case UICC_TC1:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          /* No answer received within the Initial Waiting Time window - ATR failed. */
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          UICC_atr_ctrl_state = UICC_ATR_ERROR;
          atr_error_cause = IWT_TIMEOUT;
          UICC_start_timer(T_ERROR_DELAY);
          break;

        case UICC_CHARACTER_OK:
          check_sum ^= N = fetch_and_store_char();

          UICC_start_timer(UICC_T_IWT);

          if(TD_MASK)
            UICC_atr_ctrl_state = UICC_TD1;
          else
            UICC_atr_ctrl_state = handle_history_and_tck();
          break;

        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(UICC_T_IWT);
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          break;
      }
      break;

    /*==========================*/
    /* Await interface byte TD1 */
    /* ------------------------ ==========*/
    /* TD1: - First offered protocol      */
    /*      - TA2-TD2 presence indication */
    /*====================================*/
    case UICC_TD1:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          /* No answer received within the Initial Waiting Time window - ATR failed. */
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          UICC_atr_ctrl_state = UICC_ATR_ERROR;
          UICC_start_timer(T_ERROR_DELAY);
          break;

        case UICC_CHARACTER_OK:
          check_sum ^= Y = fetch_and_store_char();

          current_indicated_protocol = Y & 0x0F; /* Store the latest indicated protocol */ 

          if (current_indicated_protocol == 0x0F)
          { /*T=15 is forbidden ETSI TS 102 230 Testcase 5.5.5.2*/
            UICC_atr_ctrl_state = UICC_ATR_ERROR;
            atr_error_cause = WRONG_TD1;
            UICC_start_timer(T_ERROR_DELAY);
            break;
          }

          /* Store the indicated protocol in bit-field */
          protocol = (1<<current_indicated_protocol);

          UICC_start_timer(UICC_T_IWT);

          if (TABCD_MASK == 0x00)
            UICC_atr_ctrl_state = handle_history_and_tck();
          else if(TA_MASK)
            UICC_atr_ctrl_state = UICC_TA2;
          else if(TB_MASK)
            UICC_atr_ctrl_state = UICC_TB2;
          else if(TC_MASK)
            UICC_atr_ctrl_state = UICC_TC2;
          else if(TD_MASK)
            UICC_atr_ctrl_state = UICC_TDI;
          break;

        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(UICC_T_IWT);
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          break;
      }
      break;

    /*==========================*/
    /* Await interface byte TA2 */
    /* ------------------------ ===================*/
    /* TA2: - Operation mode (specific/negotiable) */
    /*                                             */
    /*=============================================*/
    case UICC_TA2:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          /* No answer received within the Initial Waiting Time window - ATR failed. */
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          UICC_atr_ctrl_state = UICC_ATR_ERROR;
          atr_error_cause = IWT_TIMEOUT;
          UICC_start_timer(T_ERROR_DELAY);
          break;

        case UICC_CHARACTER_OK:
        {
          ubyte temp;
                  
          check_sum ^= temp = fetch_and_store_char();

          ta2_specific = temp | 0xFF00; /* The MSB value indicates that the byte has been received. */ 

          UICC_start_timer(UICC_T_IWT);

          if (TBCD_MASK == 0x00)
            UICC_atr_ctrl_state = handle_history_and_tck();
          else if(TB_MASK)
            UICC_atr_ctrl_state = UICC_TB2;
          else if(TC_MASK)
            UICC_atr_ctrl_state = UICC_TC2;
          else if(TD_MASK)
            UICC_atr_ctrl_state = UICC_TDI;
          break;
        }
        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(UICC_T_IWT);
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          break;
      }
      break;

    /*==========================*/
    /* Await interface byte TB2 */
    /* ------------------------ ====================*/
    /* TB2: - Alternative programming voltage ~ PI2 */
    /*==============================================*/
    case UICC_TB2:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          /* No answer received within the Initial Waiting Time window - ATR failed. */
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          UICC_atr_ctrl_state = UICC_ATR_ERROR;
          atr_error_cause = IWT_TIMEOUT;
          UICC_start_timer(T_ERROR_DELAY);
          break;

        case UICC_CHARACTER_OK:
          check_sum ^= fetch_and_store_char();

          UICC_start_timer(UICC_T_IWT);

          if (TCD_MASK == 0x00)
            UICC_atr_ctrl_state = handle_history_and_tck();
          else if(TC_MASK)
            UICC_atr_ctrl_state = UICC_TC2;
          else if(TD_MASK)
            UICC_atr_ctrl_state = UICC_TDI;
          break;

        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(UICC_T_IWT);
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          break;
      }
      break;

    /*==========================*/
    /* Await interface byte TC2 */
    /* ------------------------ ======*/
    /* TC2: - Work Waiting Time ~ WWT */
    /*================================*/
    case UICC_TC2:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          /* No answer received within the Initial Waiting Time window - ATR failed. */
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          UICC_atr_ctrl_state = UICC_ATR_ERROR;
          atr_error_cause = IWT_TIMEOUT;
          UICC_start_timer(T_ERROR_DELAY);
          break;

        case UICC_CHARACTER_OK:
          check_sum ^= wi = fetch_and_store_char();

          UICC_start_timer(UICC_T_IWT);

          if(TD_MASK)
            UICC_atr_ctrl_state = UICC_TDI;
          else
            UICC_atr_ctrl_state = handle_history_and_tck();
          break;

        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(UICC_T_IWT);
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          break;
      }
      break;

    /*==========================*/
    /* Await interface byte TDi */
    /* ------------------------ ==================*/
    /* TDi: - Next offered protocol               */
    /*      - TA(i+1)-TD(i+1) presence indication */
    /*============================================*/
    case UICC_TDI:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          /* No answer received within the Initial Waiting Time window - ATR failed. */
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          UICC_atr_ctrl_state = UICC_ATR_ERROR;
          atr_error_cause = IWT_TIMEOUT;
          UICC_start_timer(T_ERROR_DELAY);
          break;

        case UICC_CHARACTER_OK:
          check_sum ^= Y = fetch_and_store_char();

          current_indicated_protocol = Y & 0x0F;

          /* Store the indicated protocol on top of the previously indicated protocol. */
          protocol |= (1<<current_indicated_protocol);

          UICC_start_timer(UICC_T_IWT);

          if (TABCD_MASK == 0x00)
            UICC_atr_ctrl_state = handle_history_and_tck();
          else if(TA_MASK)
            UICC_atr_ctrl_state = UICC_TAI;
          else if(TB_MASK)
            UICC_atr_ctrl_state = UICC_TBI;
          else if(TC_MASK)
            UICC_atr_ctrl_state = UICC_TCI;
          else if(TD_MASK)
            UICC_atr_ctrl_state = UICC_TDI;
          break;

        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(UICC_T_IWT);
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          break;
      }
      break;

    /*==========================*/
    /* Await interface byte TAi */
    /* ------------------------ ==========================*/
    /* TAi (for i>2):                                     */
    /*    If TD(i-1) indicates T=1  -> T=1 Specific:      */
    /*      - Information Field Size ~ IFS (IFSC & IFSD)  */
    /*    If TD(i-1) indicates T=15 -> Global I/F byte:   */
    /*      - Clock Ctop Indicator ~ XI                   */
    /*      - Class (voltage) Indicator ~ UI              */
    /*====================================================*/
    case UICC_TAI:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          /* No answer received within the Initial Waiting Time window - ATR failed. */
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          UICC_atr_ctrl_state = UICC_ATR_ERROR;
          atr_error_cause = IWT_TIMEOUT;
          UICC_start_timer(T_ERROR_DELAY);
          break;

        case UICC_CHARACTER_OK:
        {
          ubyte temp;  

          check_sum ^= temp = fetch_and_store_char();

          switch(current_indicated_protocol)
          { 
            case UICC_T_1:  /* Last indicated protocol is T=1 */
              ifs = temp; /* Holds Information Field Size (IFS).*/
              break;
            case UICC_T_15: /* Last indicated protocol is T=15 */         
              xi_ui = temp | 0xFF00; /* XI holds clockstop characteristics wheras UI holds class 
                                        characteristics. MSB indicates the presence of the character. */
              break;
          }

          UICC_start_timer(UICC_T_IWT);

          if (TBCD_MASK == 0x00)
            UICC_atr_ctrl_state = handle_history_and_tck();
          else if(TB_MASK)
            UICC_atr_ctrl_state = UICC_TBI;
          else if(TC_MASK)
            UICC_atr_ctrl_state = UICC_TCI;
          else if(TD_MASK)
            UICC_atr_ctrl_state = UICC_TDI;
          break;
        }
        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(UICC_T_IWT);
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          break;
      }
      break;

    /*==========================*/
    /* Await interface byte TBi */
    /* ------------------------ =====================*/
    /* TBi (for i>2):                                */
    /*    If TD(i-1) indicates T=1  -> T=1 Specific: */
    /*      - Character Waiting Time ~ CWI           */
    /*      - Block Waiting Time ~ BWT               */
    /*===============================================*/
    case UICC_TBI:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          /* No answer received within the Initial Waiting Time window - ATR failed. */
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          UICC_atr_ctrl_state = UICC_ATR_ERROR;
          atr_error_cause = IWT_TIMEOUT;
          UICC_start_timer(T_ERROR_DELAY);
          break;
            
        case UICC_CHARACTER_OK:
        {
          ubyte temp;
          
          check_sum ^= temp = fetch_and_store_char();

          if(current_indicated_protocol == UICC_T_1)
          { /* Last indicated protocol is T=1 */
            cwi = temp & 0x0F;         /* Store CWI */
            bwi = (temp & 0xF0) >> 4;  /* Store BWI */
          }

          UICC_start_timer(UICC_T_IWT);

          if (TCD_MASK == 0x00)
            UICC_atr_ctrl_state = handle_history_and_tck();
          else if(TC_MASK)
            UICC_atr_ctrl_state = UICC_TCI;
          else if(TD_MASK)
            UICC_atr_ctrl_state = UICC_TDI;
          break;
        }
        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(UICC_T_IWT);
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          break;
      }
      break;

    /*==========================*/
    /* Await interface byte TCi */
    /* ------------------------ =====================*/
    /* TCi (for i>2):                                */
    /*    If TD(i-1) indicates T=1  -> T=1 Specific: */
    /*      - Error Detection Code ~ EDC             */
    /*===============================================*/
    case UICC_TCI:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          /* No answer received within the Initial Waiting Time window - ATR failed. */
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          UICC_atr_ctrl_state = UICC_ATR_ERROR;
          atr_error_cause = IWT_TIMEOUT;
          UICC_start_timer(T_ERROR_DELAY);
          break;

        case UICC_CHARACTER_OK:
        {
          ubyte temp;

          check_sum ^= temp = fetch_and_store_char();

          if(current_indicated_protocol == UICC_T_1)
          { /* Last indicated protocol is T=1 */
            edc = temp & 0x01;      /* Store EDC - only first bit is valid. */
          }

          UICC_start_timer(UICC_T_IWT);

          if(TD_MASK)
            UICC_atr_ctrl_state = UICC_TDI;
          else
            UICC_atr_ctrl_state = handle_history_and_tck();
          break;
        }
        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(UICC_T_IWT);
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          break;
      }
      break;

    /*============================*/
    /* Await history character(s) */
    /*============================*/
    case UICC_HISTORY:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          /* No answer received within the Initial Waiting Time window - ATR failed. */
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          UICC_atr_ctrl_state = UICC_ATR_ERROR;
          atr_error_cause = IWT_TIMEOUT;
          UICC_start_timer(T_ERROR_DELAY);
          break;

        case UICC_CHARACTER_OK:
          check_sum ^= fetch_and_store_char();
           
          remaining_hist_char--;  /* Decrement the number of historical characters */

          UICC_start_timer(UICC_T_IWT);

          UICC_atr_ctrl_state = handle_history_and_tck();
          break;

        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(UICC_T_IWT);
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          break;
      }
      break;

    /*==============================*/
    /* Await checksum character TCK */
    /*==============================*/
    case UICC_TCK:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          /* No answer received within the Initial Waiting Time window - ATR failed. */
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          UICC_atr_ctrl_state = UICC_ATR_ERROR;
          atr_error_cause = NO_TCK;
          UICC_start_timer(T_ERROR_DELAY);
          break;

        case UICC_CHARACTER_OK:
          check_sum ^= fetch_and_store_char();

          /* The accumulated check_sum value exlusive ored with the actual
             received checksum character TCK must be zero if NO error had
             occured. If not zero, the ATR sequence has not been successful. */
          if(check_sum)
          {
            #ifndef UICC_NO_DEBUG
              debug = UICC_atr_ctrl_state;
            #endif
            UICC_atr_ctrl_state = UICC_ATR_ERROR;
            atr_error_cause = WRONG_TCK;
            UICC_start_timer(T_ERROR_DELAY);
#ifdef UICC_NOT_TIMER
		  UICC_control_card_access(UICC_TIMER_EXPIRED);
#endif

		
          }
          else
          {
            UICC_atr_ctrl_state = UICC_ATR_SUCCEDED;
            UICC_start_timer(T_EVALUATION_DELAY);
#ifdef UICC_NOT_TIMER
		  UICC_control_card_access(UICC_TIMER_EXPIRED);
#endif

		
          }
          break;

        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(UICC_T_IWT);
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          break;
      }
      break;

    /*==========================================*/
    /* Await timeout if ATR sequence had FAILED */
    /*==========================================*/
    case UICC_ATR_ERROR:
            
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
           UICC_switch_uart_clock_off(); 
          UICC_atr_ctrl_state = UICC_ATR_READY;
          /* Signal the main control statemachine of the ATR result*/
          switch(atr_error_cause)
          {
            case NO_ANSWER:
              UICC_card_ctrl(UICC_ATR_FAILED_NO_ANSWER);
              break;
 
            case WRONG_TS:
            case IWT_TIMEOUT:
            case NO_TCK:
            case WRONG_TCK:
            case WRONG_TD1:
            default:
              UICC_card_ctrl(UICC_ATR_FAILED);
              break;
          }
          break;

        #if 0
        case UICC_CHARACTER_OK:
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          fetch_and_store_char(); /* To clear the Rx register. */
          UICC_start_timer(T_ERROR_DELAY);
          break;

        case UICC_OVERRUN_ERROR:
          UICC_switch_uart_clock_off();
          
        case UICC_PARITY_ERROR:
            
          UICC_get_character(); /* To clear the Rx register. */
          UICC_start_timer(T_ERROR_DELAY);
          break;
        #endif
        case UICC_OVERRUN_ERROR:           
          UICC_switch_uart_clock_off();
          break;
        case UICC_CHARACTER_OK:
        case UICC_PARITY_ERROR:         
          fetch_and_store_char(); /* To clear the Rx register. */
          if(UICC_nof_atr_characters < UICC_MAX_ATR_CHARACTERS)
            UICC_start_timer(T_ERROR_DELAY);
          
          break;

        default:           
          break;
      }
      break;

    /*===============================================*/
    /* Await timeout if ATR sequence was SUCCESSFULL */
    /*===============================================*/
    case UICC_ATR_SUCCEDED:
      
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          UICC_atr_ctrl_state = UICC_ATR_READY;

          /* Now evaluate the if some requirements has been violated. If that
             is the case the ATR sequence will be rejected!
             GSM 11.11 states that PI1 (TB1 parameter) must - if present - be
             equal to 0 or 5. Also the parameter N (TC1 parameter) must take on
             the values 0 or 0xFF in order to be accepted. Otherwise should the
             card be rejected. */

          if((((pi1 &= 0x1F) == 0) || ((pi1 &= 0x1F) == 5)) &&
             ((N == 0x00) || (N == 0xFF)))
          { /* ATR accepted! */
            /*===============*/

            if((ifs==0) || (ifs==0xFF))
            { /* Illigal values of IFS -> use default value. */
              ifs = UICC_DEFAULT_IFS;
            }
 
            /* Since the activation succeeded -> store the activation profile of the card. */
            UICC_card_profile[UICC_current_reader].di           = di;         
            UICC_card_profile[UICC_current_reader].fi           = fi;         
            UICC_card_profile[UICC_current_reader].N            = N;          
            UICC_card_profile[UICC_current_reader].T_indicated  = protocol;   
            UICC_card_profile[UICC_current_reader].wi           = wi;         
            UICC_card_profile[UICC_current_reader].ifsc         = ifs;        
            UICC_card_profile[UICC_current_reader].ifsc_used    = ifs;
            UICC_card_profile[UICC_current_reader].ifsd         = UICC_DEFAULT_IFS;        
            UICC_card_profile[UICC_current_reader].ifsd_used    = UICC_DEFAULT_IFS;        
            UICC_card_profile[UICC_current_reader].cwi          = cwi;        
            UICC_card_profile[UICC_current_reader].bwi          = bwi;        
            UICC_card_profile[UICC_current_reader].edc          = edc;        
            UICC_card_profile[UICC_current_reader].class_clock  = xi_ui;         
            UICC_card_profile[UICC_current_reader].ta2_specific = ta2_specific;         
            UICC_card_profile[UICC_current_reader].convention   = convention;                 

            UICC_card_profile[UICC_current_reader].voltage_settled      = FALSE; 
            UICC_card_profile[UICC_current_reader].used_clock_stop_mode = UICC_CLOCK_STOP_NOT_ALLOWED;
  
            UICC_card_ctrl(UICC_ATR_OK);
          }
          else
          { /* ATR rejected! */
            /*===============*/
            /* Signal the responsible access engine of the ATR result*/
            UICC_card_ctrl(UICC_ATR_FAILED);
          }
          break;

        case UICC_CHARACTER_OK:
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif

          UICC_get_character(); /* To clear the Rx register. */
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_atr_ctrl_state;
          #endif
          break;
      }
      break;

    default:
      #ifndef UICC_NO_DEBUG
        debug = UICC_atr_ctrl_state;
      #endif
      break;
  }

  #ifndef UICC_NO_DEBUG
    if(debug != UICC_ATR_NONE)
    {
      if(!already_logged)
      {
        already_logged++;

        UICC_log_data_ptr[0]  = debug;
        UICC_log_data_ptr[1]  = event;
        UICC_log_data_ptr[2]  = UICC_nof_atr_characters;
        UICC_log_data_ptr[3]  = remaining_hist_char;
        UICC_log_data_ptr[4]  = fi;
        UICC_log_data_ptr[5]  = di;
        UICC_log_data_ptr[6]  = xi_ui;
        UICC_log_data_ptr[8]  = pi1;
        UICC_log_data_ptr[9]  = protocol;      
        UICC_log_data_ptr[10] = current_indicated_protocol;
        UICC_log_data_ptr[11] = wi;
        UICC_log_data_ptr[12] = N;
        UICC_log_data_ptr[13] = ifs;
        UICC_log_data_ptr[14] = cwi;
        UICC_log_data_ptr[15] = bwi;
        UICC_log_data_ptr[16] = edc;
        UICC_log_data_ptr[17] = check_sum;
        UICC_log_data_ptr[18] = ta2_specific;
        UICC_raise_exception(TRAP_UICC_ATR, UICC_ANSWER_TO_RESET_FAILED_1, 20); 
      }
      debug = UICC_ATR_NONE;
    }
  #endif
}
                                                 /* End of file.              */
