/**
 * @file drvs_codec.h 
 * @brief Public APIs of Codec drivers
 *
 * Copyright (C) 2017 Sanechips Technology Co., Ltd.
 * @author Xinqiang Xu <xu.xinqiang@sanechips.com.cn>
 * @ingroup si_cp_drv_id
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation. 
 *
 */

#ifndef _DRV_CODEC_TLV3100_H
#define _DRV_CODEC_TLV3100_H


/****************************************************************************
* 	                        Include files
****************************************************************************/
#include "drvs_i2s.h"
#include "drvs_voiceprocess.h"

/*******************************************************************************
 *                             Macro definitions                               *
 ******************************************************************************/
 
/*******************************************************************************
 *                             Type definitions                                *
 ******************************************************************************/
typedef unsigned long  T_ZDrvCodec_Handle;

typedef enum {
	VOICE_SOURCE,
    AUDIO_SOURCE,
    EXTERN_AUDIO,
    
    SOURCE_MAX
} T_ZDrvCodec_SourceType;

typedef enum {
    AUDIO_I2S_NONE,
    AUDIO_I2S0,
    AUDIO_I2S1,
    
    AUDIO_I2S_MAX
} T_ZDrvCodec_AudioI2SChannel;

typedef enum {
    I2S_MASTER_MODE,
    I2S_SLAVE_MODE,
    
    I2S_MODE_MAX
} T_ZDrvCodec_AudioI2SMode;

typedef enum {
    AUDIO_SAMPLE_8K,
    AUDIO_SAMPLE_11_025_KHZ,          /* Use 11.025 kHz sampling rate. */
    AUDIO_SAMPLE_12_KHZ,              /* Use 12 kHz sampling rate. */
    AUDIO_SAMPLE_16K,
    AUDIO_SAMPLE_22_050_KHZ,          /* Use 22.050 kHz sampling rate. */
    AUDIO_SAMPLE_24_KHZ,              /* Use 24 kHz sampling rate. */
    AUDIO_SAMPLE_32K,
    AUDIO_SAMPLE_44_1K,
    AUDIO_SAMPLE_48K,
    AUDIO_SAMPLE_64_KHZ,              /* Use 64 kHz sampling rate */
    AUDIO_SAMPLE_96_KHZ,              /* Use 96 kHz sampling rate. */
    
    AUDIO_SAMPLE_MAX
} T_ZDrvCodec_AudioSample;

typedef enum {
    AUDIO_INPUT_HANDSET,
    AUDIO_INPUT_SPEAKER,
    AUDIO_INPUT_HEADSET,
    AUDIO_INPUT_BLUETOOTH,
    AUDIO_INPUT_FM_ANATODIG,
    AUDIO_INPUT_FM_ANATOANA,
    
    AUDIO_INPUT_MAX
} T_ZDrvCodec_InDevice;

typedef enum {
    AUDIO_OUTPUT_RECEIVER,
    AUDIO_OUTPUT_SPEAKER,
    AUDIO_OUTPUT_HEADSET,
    AUDIO_OUTPUT_BLUETOOTH,
    AUDIO_OUTPUT_HEADSET_SPEAKER,
    AUDIO_OUTPUT_FM_ANATODIG_HEADSET,
    AUDIO_OUTPUT_FM_ANATODIG_SPEAKER,
    AUDIO_OUTPUT_FM_ANATOANA_HEADSET,
    AUDIO_OUTPUT_FM_ANATOANA_SPEAKER,
    
    AUDIO_OUTPUT_MAX
} T_ZDrvCodec_OutDevice;

typedef enum {
    CLSD_GAIN_PLUS_6DB,
    CLSD_GAIN_PLUS_12DB,
    CLSD_GAIN_PLUS_18DB,
    CLSD_GAIN_PLUS_24DB,
    
    CLSD_GAIN_MAX
} T_ZDrvCodec_ClsD_Vol;

typedef enum {
    CLSAB_GAIN_PLUS_0DB,
    CLSAB_GAIN_PLUS_1DB,
    CLSAB_GAIN_PLUS_2DB,
    CLSAB_GAIN_PLUS_3DB,
    CLSAB_GAIN_PLUS_4DB,
    CLSAB_GAIN_PLUS_5DB,
    CLSAB_GAIN_PLUS_6DB,
    CLSAB_GAIN_PLUS_7DB,
    CLSAB_GAIN_PLUS_8DB,
    CLSAB_GAIN_PLUS_9DB,
    
    CLSAB_GAIN_MAX
} T_ZDrvCodec_ClsAB_Vol;

typedef enum {
    CODEC_INPATH_DIG_GAIN_MINUS_12DB = 0,
    CODEC_INPATH_DIG_GAIN_MINUS_11DB,
    CODEC_INPATH_DIG_GAIN_MINUS_10DB,
    CODEC_INPATH_DIG_GAIN_MINUS_9DB,
    CODEC_INPATH_DIG_GAIN_MINUS_8DB,
    CODEC_INPATH_DIG_GAIN_MINUS_7DB,
    CODEC_INPATH_DIG_GAIN_MINUS_6DB,
    CODEC_INPATH_DIG_GAIN_MINUS_5DB,
    CODEC_INPATH_DIG_GAIN_MINUS_4DB,
    CODEC_INPATH_DIG_GAIN_MINUS_3DB,
    CODEC_INPATH_DIG_GAIN_MINUS_2DB,
    CODEC_INPATH_DIG_GAIN_MINUS_1DB,
    CODEC_INPATH_DIG_GAIN_0DB,
    CODEC_INPATH_DIG_GAIN_PLUS_1DB,
    CODEC_INPATH_DIG_GAIN_PLUS_2DB,
    CODEC_INPATH_DIG_GAIN_PLUS_3DB,
    CODEC_INPATH_DIG_GAIN_PLUS_4DB,
    CODEC_INPATH_DIG_GAIN_PLUS_5DB,
    CODEC_INPATH_DIG_GAIN_PLUS_6DB,
    CODEC_INPATH_DIG_GAIN_PLUS_7DB,
    CODEC_INPATH_DIG_GAIN_PLUS_8DB,
    CODEC_INPATH_DIG_GAIN_PLUS_9DB,
    CODEC_INPATH_DIG_GAIN_PLUS_10DB,
    CODEC_INPATH_DIG_GAIN_PLUS_11DB,
    CODEC_INPATH_DIG_GAIN_PLUS_12DB,
    CODEC_INPATH_DIG_GAIN_PLUS_13DB,
    CODEC_INPATH_DIG_GAIN_PLUS_14DB,
    CODEC_INPATH_DIG_GAIN_PLUS_15DB,
    CODEC_INPATH_DIG_GAIN_PLUS_16DB,
    CODEC_INPATH_DIG_GAIN_PLUS_17DB,
    CODEC_INPATH_DIG_GAIN_PLUS_18DB,
    CODEC_INPATH_DIG_GAIN_PLUS_19DB,
    CODEC_INPATH_DIG_GAIN_PLUS_20DB,
    
    CODEC_INPATH_DIG_GAIN_MAX
} T_ZDrvCodec_InPath_Digital_Gain;

typedef enum {
    CODEC_OUTPATH_DIG_GAIN_PLUS_24DB = 0,
    CODEC_OUTPATH_DIG_GAIN_PLUS_23DB,
    CODEC_OUTPATH_DIG_GAIN_PLUS_22DB,
    CODEC_OUTPATH_DIG_GAIN_PLUS_21DB,
    CODEC_OUTPATH_DIG_GAIN_PLUS_20DB,
    CODEC_OUTPATH_DIG_GAIN_PLUS_19DB,
    CODEC_OUTPATH_DIG_GAIN_PLUS_18DB,
    CODEC_OUTPATH_DIG_GAIN_PLUS_17DB,
    CODEC_OUTPATH_DIG_GAIN_PLUS_16DB,
    CODEC_OUTPATH_DIG_GAIN_PLUS_15DB,
    CODEC_OUTPATH_DIG_GAIN_PLUS_14DB,
    CODEC_OUTPATH_DIG_GAIN_PLUS_13DB,
    CODEC_OUTPATH_DIG_GAIN_PLUS_12DB,
    CODEC_OUTPATH_DIG_GAIN_PLUS_11DB,
    CODEC_OUTPATH_DIG_GAIN_PLUS_10DB,
    CODEC_OUTPATH_DIG_GAIN_PLUS_9DB,
    CODEC_OUTPATH_DIG_GAIN_PLUS_8DB,
    CODEC_OUTPATH_DIG_GAIN_PLUS_7DB,
    CODEC_OUTPATH_DIG_GAIN_PLUS_6DB,
    CODEC_OUTPATH_DIG_GAIN_PLUS_5DB,
    CODEC_OUTPATH_DIG_GAIN_PLUS_4DB,
    CODEC_OUTPATH_DIG_GAIN_PLUS_3DB,
    CODEC_OUTPATH_DIG_GAIN_PLUS_2DB,
    CODEC_OUTPATH_DIG_GAIN_PLUS_1DB,
    CODEC_OUTPATH_DIG_GAIN_0DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_1DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_2DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_3DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_4DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_5DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_6DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_7DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_8DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_9DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_10DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_11DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_12DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_13DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_14DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_15DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_16DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_17DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_18DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_19DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_20DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_21DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_22DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_23DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_24DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_25DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_26DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_27DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_28DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_29DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_30DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_31DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_32DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_33DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_34DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_35DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_36DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_37DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_38DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_39DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_40DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_41DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_42DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_43DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_44DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_45DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_46DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_47DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_48DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_49DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_50DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_51DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_52DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_53DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_54DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_55DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_56DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_57DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_58DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_59DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_60DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_61DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_62DB,
    CODEC_OUTPATH_DIG_GAIN_MINUS_63DB,
    
    CODEC_OUTPATH_DIG_GAIN_MAX
} T_ZDrvCodec_OutPath_Digital_Gain;

typedef enum {
    MICBIAS_OUT_POWERDOWN = 0,
    MICBIAS_OUT_2V,
    MICBIAS_OUT_2_5V,
    MICBIAS_OUT_AVDD,
    
    MICBIAS_OUT_MAX
} T_ZDrvMicBias_Ctrl;

/**
 * @brief describe the codec agc Parameter 
 * @param onOff				agc function enable
 * @param targetLevel		the Level of target
 * @param attackTime 		attack Time	 	
 * @param decayTime 		decay Time
 * @param noiseDebounce		noise Debounce	 	 
 * @param signalDebounce	signal Debounce	
 * @param noiseThreshold	the Threshold of the noise	
 * @param maxgain	 		the max gain 	
 */
typedef struct _T_AudCodec_AGC_Parameter {
	SINT8 onOff;
	SINT8 targetLevel;
	SINT8 attackTime;
	SINT8 decayTime;
	SINT8 noiseDebounce;
	SINT8 signalDebounce;
	SINT8 noiseThreshold;
	
	SINT8 maxgain;
} T_ZDrvAudCodec_AGC_Parameter;

/**
 * @brief describe the codec drc Parameter 
 * @param onOff			drc function enable
 * @param threshold		threshold value 
 * @param hysteresis 	hysteresis value	 	
 * @param holdTime 		hold Time
 * @param attackRate	attack Rate	 	 
 * @param decayRate	 	decay Rate	 	
 */
typedef struct _T_AudCodec_DRC_Parameter {
	SINT8 onOff;
	SINT8 threshold;
	SINT8 hysteresis;
	SINT8 holdTime;
	SINT8 attackRate;
	SINT8 decayRate;
} T_ZDrvAudCodec_DRC_Parameter;

typedef enum {
    CODEC_DATA_LEGTH_8BIT = 0,
    CODEC_DATA_LEGTH_16BIT,
    CODEC_DATA_LEGTH_20BIT,
    CODEC_DATA_LEGTH_24BIT,
    CODEC_DATA_LEGTH_32BIT,
    
    CODEC_DATA_LEGTH_MAX
} T_ZDrvCodec_DataLength;

typedef enum {
    CODEC_DATA_TRACK_DOUBLE = 0,
    CODEC_DATA_TRACK_LEFT,
    CODEC_DATA_TRACK_RIGHT,
    
    CODEC_DATA_TRACK_MAX
} T_ZDrvCodec_DataTrack;

typedef enum {
    CODEC_BUS_I2S,
    CODEC_BUS_RJF,   /* Right-Justified Mode */
    CODEC_BUS_LJF,   /* Left-Justified Mode */
    CODEC_BUS_DSP,
    CODEC_BUS_TDM,
    CODEC_BUS_MAX
} T_AudCodec_BusMode;

#define T_ZDrvCodec_BusMode  T_AudCodec_BusMode

/**
 * @brief describe the i2s transfer protocol 
 * @param audI2SChannel		Audio I2S Channel
 * @param masterSlave		Audio I2S Mode;		 
 * @param busMode 			bus mode		 	
 * @param dataLength 		Codec Data Length
 * @param dataTrack	 		codec data type	 	
 */
typedef struct _T_Codec_UseProtocol {
	T_ZDrvCodec_AudioI2SChannel audI2SChannel;
	T_ZDrvCodec_AudioI2SMode masterSlave;
	T_ZDrvCodec_BusMode busMode;
	T_ZDrvCodec_DataLength dataLength;
	T_ZDrvCodec_DataTrack dataTrack;
} T_ZDrvCodec_UseProtocol;

/*******************************************************************************
 *                       Global variable declarations                          *
 ******************************************************************************/
 

/*******************************************************************************
 *                       Global function declarations                          *
 ******************************************************************************/
/**
 * @brief Open codec,set parameters.
 *
 * @param handle	Pointer to T_ZDrvCodec_Handle.
 * @param srcType 	Pointer to T_ZDrvCodec_SourceType
 *
 * @return	 0-DRV_SUCCESS, other-error
 */
SINT32 zDrvCodec_Open(T_ZDrvCodec_Handle *handle, T_ZDrvCodec_SourceType srcType);

/**
* @brief	close codec, power off.
*
* @param	handle    T_ZDrvCodec_Handle
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_Close(T_ZDrvCodec_Handle *handle);

/**
* @brief	Set i2s protocol.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	channel   T_ZDrvCodec_AudioI2SChannel
* @param	masterSlave	T_ZDrvCodec_AudioI2SMode
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetProtocal(T_ZDrvCodec_Handle handle, T_ZDrvCodec_AudioI2SChannel channel, T_ZDrvCodec_AudioI2SMode masterSlave);

/**
* @brief	Set i2s clock.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	sample   T_ZDrvCodec_AudioSample
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetClock(T_ZDrvCodec_Handle handle, T_ZDrvCodec_AudioSample sample);

/**
* @brief	reset codec.
*
* @param	handle	 T_ZDrvCodec_Handle
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_Reset(T_ZDrvCodec_Handle handle);

/**
* @brief	Set the input path of the codec.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	dev   T_ZDrvCodec_InDevice
* @param	onoff   BOOL
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetInPath(T_ZDrvCodec_Handle handle, T_ZDrvCodec_InDevice dev, BOOL onoff);

/**
* @brief	Set the output path of the codec.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	dev   T_ZDrvCodec_InDevice
* @param	onoff   BOOL
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetOutPath(T_ZDrvCodec_Handle handle, T_ZDrvCodec_OutDevice dev, BOOL onoff);

/**
* @brief	Enable the audio path.
*
* @param	handle	 T_ZDrvCodec_Handle
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_AudioEnable(T_ZDrvCodec_Handle handle);

/**
* @brief	Disable the audio path.
*
* @param	handle	 T_ZDrvCodec_Handle
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_AudioDisable(T_ZDrvCodec_Handle handle);

/**
* @brief	Set the ClsD volume.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	vol	 T_ZDrvCodec_ClsD_Vol
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_Set_ClsD_Vol(T_ZDrvCodec_Handle handle, T_ZDrvCodec_ClsD_Vol vol);

/**
* @brief	Set the ClsAB volume.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	vol  UINT8
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_Set_ClsAB_Vol(T_ZDrvCodec_Handle handle, UINT8 vol);

/**
* @brief	Set the  output Analog volume of the Output path.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	vol  SINT8
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_Set_Outpath_Ana_Vol(T_ZDrvCodec_Handle handle, SINT8 vol);

/**
* @brief	Set the  Analog volume of the Input path.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	vol  SINT8
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_Set_Inpath_Ana_Vol(T_ZDrvCodec_Handle handle, SINT8 vol);

/**
* @brief	Set the  Digital volume of the Input path.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	invol    T_ZDrvCodec_InPath_Digital_Gain
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_Set_Inpath_Dig_Vol(T_ZDrvCodec_Handle handle, T_ZDrv_VpVol vol);

/**
* @brief	Set the  Digital volume of the Output path.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	invol    T_ZDrv_VpVol
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_Set_Outpath_Dig_Vol(T_ZDrvCodec_Handle handle, T_ZDrv_VpVol vol);

/**
* @brief	Set the Side Tone.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	onoff    BOOL
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetSideTone(T_ZDrvCodec_Handle handle, BOOL onoff);

/**
* @brief	Set the Mic Bias.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	micBiasCtrl    T_ZDrvMicBias_Ctrl
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetMicBias(T_ZDrvCodec_Handle handle, T_ZDrvMicBias_Ctrl micBiasCtrl);

/**
* @brief	Set the Input Agc.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	AGCPara    Poiter to T_ZDrvAudCodec_AGC_Parameter
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetInputAgc(T_ZDrvCodec_Handle handle, T_ZDrvAudCodec_AGC_Parameter *AGCPara);

/**
* @brief	Set the Output Agc.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	DRCPara    Poiter to T_ZDrvAudCodec_DRC_Parameter
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetOutputDrc(T_ZDrvCodec_Handle handle, T_ZDrvAudCodec_DRC_Parameter *DRCPara);

/**
* @brief	Set the Loop Back.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	onoff    BOOL
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetLoopBack(T_ZDrvCodec_Handle handle, BOOL onoff);

/**
* @brief	Set the mute function in the input path.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	onoff	 BOOL
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetInpathMute(T_ZDrvCodec_Handle handle, BOOL onoff);

/**
* @brief	Set the mute function in the output path.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	onoff	 BOOL
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetOutpathMute(T_ZDrvCodec_Handle handle, BOOL onoff);

/**
* @brief	read data to codec
*
* @param	regPage     the page number of the register
* @param	regAddress  register address
* @param	regValue    register value
*
* @return	0-DRV_SUCCESS, other-error
*/
//SINT32 codec_I2CRead(UINT8 regPage, UINT8 regAddress, UINT8 *regValue);

#endif/*#ifndef _DRV_CODEC_TLV3100_H*/

