/**
 *   Copyright (C) Infineon Technologies Denmark A/S. All rights reserved.
 *
 * This document contains proprietary information belonging to Infineon
 * Technologies Denmark A/S. Passing on and copying of this document, use
 * and communication of its contents is not permitted without prior written
 * authorisation.
 *
 * Description:
 *   Contains dispatcher and functions for volume control
 *
 * Revision Information:
 *   File name: \dwddrv\AUD\src\aud_volume.c
 *   Version: \main\16
 *   Date: 2007-10-01 15:59:28
 *   Responsible: johansek
 *   Comment:
 *     Added Steon2 changes under project define.
 */

/*******************************************************************************
*
*                               Include files
*
*******************************************************************************/
//#include "scttypes.h"           /* must be included in all files */
//#include "ms.h"                 /* must be included in all files */
#include "bastypes.h"           /* must be included in all files */
#if defined (TRAP_HANDLING)
//#include "trap.h"               /* audio define for trap handling */
#endif
//#include "dsp.h"                /* for DSP command types */
#include "aud_drv.h"            /* audio types and functions */
#include "aud_data.h"           /* data describing the paths */
#include "aud_com.h"            /* types and functions used only within the audio driver */
#include "aud_gain_lib.h"       /* structures and functions for writing DSP commands */
#include "aud_path_control.h"   /* aud_path_settings_result */
#include "aud_volume.h"         /* header file for volume control */
//#include "aud_fm_radio.h"       /* calling the state machine for muting of FM radio */
//#include "aud_tty.h"            /* for checking if TTY is active */
#include "drvs_general.h"
#include "drvs_io_voice.h"

#include "aud_nv.h"
#include "drvs_teak.h"

/*******************************************************************************
*
*                         Internal/private variables
*
*******************************************************************************/
static aud_resource_volume_type aud_vol_resource_volume[aud_resource_end]; /* resource volume and mute status - used by driver */
static aud_volume_enum          aud_vol_master_volume = aud_volume_99; /* always contains the latest master volume set by application */
static aud_afe_earpc_gain_enum  aud_vol_master_volume_analog = afe_earpc_gainMinus6db; /* contains the AFE gain reduction , 0, -6, -12, -18 dB for the master volume */
static U16                      aud_vol_master_volume_digital = 0x2000; /* contains the digital master volume gain */
static aud_mute_enum            aud_vol_master_mute = aud_mute_disable; /* always contains the latest master mute status set by Apoxi/MMI */

extern aud_eep_static_type aud_EEP_static;

/*******************************************************************************
*
*                         Internal/private functions
*
*******************************************************************************/
static void aud_vol_set_AFE_and_digital_gain_values(aud_volume_enum vol_step);
static word aud_vol_calc_resource_volume(aud_volume_enum vol_step);
static void aud_vol_mute_external_hardware(aud_mute_enum disable_enable);
static void aud_vol_program_master_gains(void);
static void aud_vol_program_resource_gains(aud_resource_enum resource);
void aud_voice_record_channel(T_ZDrvVoice_RecordChannel channel );

/*******************************************************************************
*
*                           Global/public functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: aud_volume_dispatcher
* Return.....: TRUE if the dispatcher handles the event, FALSE otherwise
* Description: Dispatcher for volume events (muting and volume level).
* Created....: 07.03.2007 by KBJ
* Modified...: 03.04.2007 by KBJ (reset mute status when resource is released)
------------------------------------------------------------------------------*/
BOOL aud_volume_dispatcher(
    aud_event_enum event,   /* the event to be handled */
    void* p_data)           /* data for handling the event */
{
    BOOL func_ret = TRUE; /* used to tell whether the event has been handled by this state machine */
    disp_to_SM_data_type *p_SM_data = (disp_to_SM_data_type*)p_data; /* contains the structure passed from the dispatcher */
    U32 parm1=p_SM_data->parm1;
    U32 parm2=p_SM_data->parm2;
    U32 parm3=p_SM_data->parm3;

    /* Send string : at+xl1set="l1 15l" To enable this LLT trace */
#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_volume, 11, 0,                        /* LLT params */
                     's', 0x01,                                                             /* LIS params */
                     aud_log_lis_raw_data, aud_log_lis_display_data, aud_log_lis_llt_trace, /* LIS params */
                     (U32) aud_get_SM_state(aud_resource_speech), 
                     (U32) event,
                     (U32) p_SM_data->parm1,
                     (U32) p_SM_data->parm2,
                     (U32) p_SM_data->parm3,
                     (U32) p_SM_data->parm4,
                     (U32) p_SM_data->parm5,
                     (U32) p_SM_data->parm6,
                     (U32) p_SM_data->parm7,
                     (U32) p_SM_data->parm8,
                     (U32) E_audio_llt_type_id_0 );
 #endif   
    switch(event)
    {
        case E_mute_resource:
            if(aud_mute_enable == parm2) /* enable muting */
            {
                if(aud_direction_resource_uplink==parm3 || aud_direction_resource_updownlink==parm3)
                {
                    aud_vol_resource_volume[parm1].resource_muted_UL = aud_mute_enable;
                }
                if(aud_direction_resource_downlink==parm3 || aud_direction_resource_updownlink==parm3)
                {
                    aud_vol_resource_volume[parm1].resource_muted_DL = aud_mute_enable;
                }
            }
            else /* disable muting */
            {
                if(aud_direction_resource_uplink==parm3 || aud_direction_resource_updownlink==parm3)
                {
                    aud_vol_resource_volume[parm1].resource_muted_UL = aud_mute_disable;
                }
                if(aud_direction_resource_downlink==parm3 || aud_direction_resource_updownlink==parm3)
                {
                    aud_vol_resource_volume[parm1].resource_muted_DL = aud_mute_disable;
                }
            }
            aud_vol_program_resource_gains((aud_resource_enum)parm1);
            break;

        case E_mute_master:
            if(aud_mute_enable == parm1)
            {
                aud_vol_master_mute = aud_mute_enable;
                aud_vol_program_master_gains(); /* muting is done through the digital gain cells */
                aud_vol_mute_external_hardware(aud_mute_enable);
            }
            else
            {
                aud_vol_master_mute = aud_mute_disable;
                aud_vol_set_AFE_and_digital_gain_values(aud_vol_master_volume); /* programs both analog and digital gains */
                aud_vol_mute_external_hardware(aud_mute_disable);
            }
            break;

        case E_set_resource_volume:
            aud_vol_resource_volume[parm1].volume_level=(aud_volume_enum)parm2;
            aud_vol_program_resource_gains((aud_resource_enum)parm1);
            break;

        case E_set_master_volume:
            aud_vol_set_AFE_and_digital_gain_values((aud_volume_enum)parm1); /* programs both analog and digital gains */
            break;

        case E_resource_release:
            /* reset mute status when the resource is released */
            aud_vol_resource_volume[parm1].resource_muted_UL = aud_mute_disable;
            aud_vol_resource_volume[parm1].resource_muted_DL = aud_mute_disable;
            break;

        case E_resource_allocate:
        case E_info_hw_available:
            break; /* dummy - do nothing */

        default:
            func_ret = FALSE; /* the event has not been handled */
            break;
    }
    return (func_ret);
}

/*------------------------------------------------------------------------------
* Function...: aud_volume_init
* Return.....: -
* Description: Initialises the data needed to control volume and muting.
* Created....: 07.03.2007 by KBJ
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
void aud_volume_init(void)
{
    /* initialise resource volume and mute status */
    U8 i;
    for(i=0; i<aud_resource_end; i ++)
    {
        aud_vol_resource_volume[i].volume_level      = aud_volume_50;
        aud_vol_resource_volume[i].resource_muted_UL = aud_mute_disable;
        aud_vol_resource_volume[i].resource_muted_DL = aud_mute_disable;
    }
    aud_vol_resource_volume[aud_resource_speech].volume_level          = aud_volume_95;
    aud_vol_resource_volume[aud_resource_tone_generator].volume_level  = aud_volume_30;

    /* set DSP structures that are not programmed from elsewhere */
    aud_dsp_gain_parms.speech_mix    = aud_constant_gain_values.speech_mix_dl; /* make sure that speech is not affected when DSP tones are generated */
    aud_dsp_gain_parms.speech_mix_ul = aud_constant_gain_values.speech_mix_ul; /* make sure that speech is not affected when DSP tones are generated */

    /* program gain cells for master volume */
    aud_vol_set_AFE_and_digital_gain_values(aud_vol_master_volume);
}

/*------------------------------------------------------------------------------
* Function...: aud_calc_side_tone_volume
* Return.....: the value for the side tone gain
* Description: Calculates the gain for side tone gain cell and scales it with a
*              factor that depends on the active paths.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
word aud_calc_side_tone_volume(
    aud_volume_enum vol_step) /* the volume step to balance against (usually master volume) */
{
    U32 volume;
    vol_step = (aud_volume_enum)((U8) aud_volume_100 - ((U8)vol_step - 1)); /* the -1 ensures we never reaches volume step 0 = mute */
    volume   = (U32) aud_vol_calc_resource_volume(vol_step); /* contains range check */
    volume   = (volume*aud_path_settings_result.downlink_parms.side_tone_fact)/1000; /* scale the side tone - per thousand. Select factor 0 for muting */
    return ((word)volume);   
}

/*------------------------------------------------------------------------------
* Function...: aud_update_resource_gain
* Return.....: -
* Description: Update the gain cells associated with a resource. Called when
*              the resource is (de)activated.
* Created....: 09.03.2007 by KBJ
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
void aud_update_resource_gain(
    aud_resource_enum resource) /* the resource whose state has changed */
{
    aud_vol_program_resource_gains(resource);
}

/*******************************************************************************
*
*                     Getter functions for master volume
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: aud_get_master_volume
* Return.....: volume step for the master volume
* Description: Returns the master volume.
* Created....: 07.03.2007 by KBJ
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
aud_volume_enum aud_get_master_volume(void)
{
    return (aud_vol_master_volume); /* volume level */
}
#ifdef _USE_CUT_TEAKLIT_CODE
/*------------------------------------------------------------------------------
* Function...: aud_get_master_volume_analog
* Return.....: analog part of the master volume
* Description: Returns the analog gain settings for the master volume.
* Created....: 07.03.2007 by KBJ
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
aud_afe_earpc_gain_enum aud_get_master_volume_analog(void)
{
    return (aud_vol_master_volume_analog); /* analog gain setting */
}
#endif
/*------------------------------------------------------------------------------
* Function...: aud_get_master_volume_digital
* Return.....: digital part of the master volume
* Description: Returns the digital gain settings for the master volume.
* Created....: 07.03.2007 by KBJ
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
U16 aud_get_master_volume_digital(void)
{
    return (aud_vol_master_volume_digital); /* digital gain setting */
}

/*------------------------------------------------------------------------------
* Function...: aud_get_master_mute_status
* Return.....: master mute status
* Description: Returns the mute status for the master volume.
* Created....: 07.03.2007 by KBJ
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
aud_mute_enum aud_get_master_mute_status(void)
{
    return (aud_vol_master_mute);
}

/*******************************************************************************
*
*                     Getter functions for resource volume
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: aud_get_resource_mute_status_uplink
* Return.....: resource mute status
* Description: Returns the uplink mute status for a specified resource.
* Created....: 07.03.2007 by KBJ
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
aud_mute_enum aud_get_resource_mute_status_uplink(
    aud_resource_enum resource) /* the resource for which the mute status is requested */
{
    return (aud_vol_resource_volume[resource].resource_muted_UL);
}
#ifdef _USE_CUT_TEAKLIT_CODE
/*------------------------------------------------------------------------------
* Function...: aud_get_resource_mute_status_downlink
* Return.....: resource mute status
* Description: Returns the downlink mute status for a specified resource.
* Created....: 07.03.2007 by KBJ
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
aud_mute_enum aud_get_resource_mute_status_downlink(
    aud_resource_enum resource) /* the resource for which the mute status is requested */
{
    return (aud_vol_resource_volume[resource].resource_muted_DL);
}

/*------------------------------------------------------------------------------
* Function...: aud_get_resource_volume
* Return.....: volume step for a resource
* Description: Returns the volume step for a specified resource.
* Created....: 07.03.2007 by KBJ
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
aud_volume_enum aud_get_resource_volume(
    aud_resource_enum resource) /* the resource for which the volume step is requested */
{
    return (aud_vol_resource_volume[resource].volume_level);
}
#endif
/*******************************************************************************
*
*                  Internal/private function implementations
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: aud_vol_set_AFE_and_digital_gain_values
* Return.....: -
* Description: Calculates the Master volume values and sets it.
*           Gain in dB:  G = vol/4
*           Analog Gain: AG = (int)(Gain_dB/(3.13))-(int)(Gain_dB/(6.26))-(int)(Gain_dB/(21.76))
*           Digital Gain: DG = G - (AG*6)-3.5
*           Parameter to dig. gain cell: PD = 0x4000 * (10^(DG/20)). Calculated using look-up table
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
static void aud_vol_set_AFE_and_digital_gain_values(
    aud_volume_enum vol_step) /* the master volume step to program */
{
    float Gain_dB=0;
    U8 Analog_Gain=0;

    /* Parameter for Digital gain - result */
    const int PD[100] = {11270,11599,11938,12286,12645,13014,13394,13785,14188,14602,    
                                   15029,15467,7978,8211,8451,8698,8952,9213,9482,9759,
                                   10044,10338,10639,10950,11270,11599,11938,12286,12645,13014,
                                   13394,13785,14188,14602,15029,15467,15919,8211,8451,8698,
                                   8952,9213,9482,9759,10044,10338,10639,10950,11270,11599,
                                   11938,12286,12645,13014,13394,13785,14188,14602,15029,15467,
                                   15919,16384,8451,8698,8952,9213,9482, 9759,10044,10338,
                                   10639,10950,11270,11599,11938,12286,12645,13014,13394,13785,
                                   14188,14602,15029,15467,15919,16384,16862,17355,17862,18383,
                                   18920,19472,20041,20626,21229,21848,22486,23143,23819,24514};

    const aud_afe_earpc_gain_enum afe_earpc_gains[afe_earpc_gain_end] =
                {afe_earpc_gainMinus54db, afe_earpc_gainMinus48db, afe_earpc_gainMinus42db, afe_earpc_gainMinus36db,
                afe_earpc_gainMinus30db, afe_earpc_gainMinus24db, afe_earpc_gainMinus18db, afe_earpc_gainMinus12db,
                afe_earpc_gainMinus6db, afe_earpc_gain0db}; /* only -18 dB to 0 dB are used */

#if defined (AUD_3D_AMP_PRESENT)
    const S8 aud_3d_amp_gains[afe_earpc_gain_end] = {-48, -42, -36, -30, -24, -18, -12, -6, 0,  6};
#endif /* AUD_3D_AMP_PRESENT */
   
    /* update master volume */
    aud_vol_master_volume = vol_step;

    /* calculate digital gain */
    if(vol_step > aud_volume_0 && vol_step <= aud_volume_100)
    {
        Gain_dB = (float)vol_step/4.0;
        Analog_Gain = (int) (Gain_dB/(3.13)) -(int) (Gain_dB/(6.26)) - (int) (Gain_dB/(21.76)); 
        aud_vol_master_volume_digital = PD[vol_step-1];
    }
    else
    {
        Analog_Gain = 0; /* equals floor function for positive numbers */
        aud_vol_master_volume_digital = 0;
    }
    aud_vol_program_master_gains(); /* digital part */

    /* calculate analog gain */
    aud_vol_master_volume_analog = afe_earpc_gains[Analog_Gain+afe_earpc_gainMinus18db];
    /*[Begin]no use. [lvwenhua-2010/7/28]*/
    #if 0
    aud_program_afe_rx_gain(aud_vol_master_volume_analog); /* analog part */
    #endif
    /*[End] [lvwenhua-2010/7/28]*/

#if defined (AUD_3D_AMP_PRESENT)
    /* If 3D amp is mounted, control the volume here instead of Master Vol for EPPA1 and EPPA2 outputs */
    AUD_control_3D_amp(2*aud_3d_amp_gains[Analog_Gain+afe_earpc_gainMinus18db]); /* times 2 because of 0.5dB units */
#endif

#ifdef AUDIO_LLT
    aud_common_trace(llt_group_audio, llt_type_audio_volume, 3, 0,                          /* LLT params */
                     'G', 0x48,                                                             /* LIS params */
                     aud_log_lis_display, aud_log_lis_display_data, aud_log_lis_llt_trace,  /* LIS params */
                     (U32) vol_step,
                     (U32) aud_vol_master_volume_digital, 
                     (U32) aud_vol_master_volume_analog );
#endif
}

/*------------------------------------------------------------------------------
* Function...: aud_vol_calc_resource_volume
* Return.....: the gain value corresponding to a volume step
* Description: Calculate the gain value for a resource based on a volume step.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
static word aud_vol_calc_resource_volume(
    aud_volume_enum vol_step) /* the volume step to transform to a gain value */
{
    const int PD_res[100] = 
    {    1892, 1947, 2004, 2063, 2123, 2185, 2249, 2314, 2382, 2451,
         2523, 2597, 2673, 2751, 2831, 2914, 2999, 3086, 3176, 3269,
         3365, 3463, 3564, 3668, 3775, 3885, 3999, 4116, 4236, 4359,
         4487, 4618, 4753, 4891, 5034, 5181, 5332, 5488, 5648, 5813,
         5983, 6158, 6338, 6523, 6713, 6909, 7111, 7319, 7532, 7752,
         7979, 8211, 8451, 8698, 8952, 9213, 9482, 9759,10044,10338,
        10639,10950,11270,11599,11938,12286,12645,13014,13394,13785,
        14188,14602,15029,15467,15919,16384,16862,17355,17862,18383,
        18920,19472,20041,20626,21229,21848,22486,23143,23819,24514,
        25230,25967,26725,27506,28309,29135,29986,30862,31763,32690
    };

    if(vol_step<=0) /* if below aud_volume_1 = 1 return 0, that means mute the gain cell */
    {
        return (0);
    }

    if(vol_step>100)
    {
        return (PD_res[99]); /* over the top => return max */
    }
    else
    {
        return (PD_res[vol_step-1]); /* -1 because aud_volume steps counts from 1 and not 0 */
    }
}

/*------------------------------------------------------------------------------
* Function...: aud_vol_mute_external_hardware
* Return.....: -
* Description: When master volume is muted, also mute external hardware.
* Created....: 07.03.2007 by KBJ
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
static void aud_vol_mute_external_hardware(
    aud_mute_enum disable_enable) /* whether the external hardware should be muted or unmuted */
{
    if(aud_mute_enable == disable_enable)
    {
#if defined (AUD_EXTERNAL_RINGER)
        AudRingerSM(E_ringer_mute_amplifier, NULL);
#endif
#if defined (FM_RADIO_PRESENT)
        AudFMRadioSM(E_radio_mute,NULL);
#endif
    }
    else
    {
#if defined (AUD_EXTERNAL_RINGER)
        AudRingerSM(E_ringer_unmute_amplifier, NULL);
#endif
#if defined (FM_RADIO_PRESENT)
        AudFMRadioSM(E_radio_unmute,NULL);
#endif
    }
}

/*------------------------------------------------------------------------------
* Function...: aud_vol_program_master_gains
* Return.....: -
* Description: Program digital gain cells that are controlled by master volume.
* Created....: 07.03.2007 by KBJ
* Modified...: 01.10.2007 by KBJ (added Steon2 exceptions)
------------------------------------------------------------------------------*/
extern aud_path_downlink_buffer_type aud_path_active_downlink_paths;
static void aud_vol_program_master_gains(void)
{
    /* scal_i2s1 */
    /*set scal_I2S1 from NV[Begin] [lvwenhua-2010/9/16]*/
    #if 0
    U16 scal_i2s1_values[afe_earpc_gain_end] = {0,0,0,0,0,0,0x07FF,0x0FFF,0x1FFF,0x3FFF}; /* for simulating analog gain in I2S1 (..., -18, -12, -6, 0dB) */
    aud_dsp_gain_parms.scal_I2S1 = scal_i2s1_values[aud_vol_master_volume_analog];
    #endif
    /*[End] [lvwenhua-2010/9/16]*/
    /* scal_rec */
    if(aud_mute_enable == aud_vol_master_mute)
    {
        aud_dsp_gain_parms.scal_rec = 0;
    }
    else
    {
        /*set scal_rec from NV[Begin] [lvwenhua-2010/9/16]*/
        #if 0
        aud_dsp_gain_parms.scal_rec = aud_vol_master_volume_digital/2;
        #endif
        aud_dsp_gain_parms.scal_rec = (word) aud_EEP_static.aud_scal_rec_parms[aud_path_active_downlink_paths.path[0]].scal_rec;
        /*[End] [lvwenhua-2010/9/16]*/
    }
    /*[Begin]vb gain will be set at the ending of this function, so this setting can be cancelled. [lvwenhua-2010/7/28]*/
    #if 0
    aud_dsp_set_vb_gains();
    #endif
    /*[End] [lvwenhua-2010/7/28]*/

    /* mix_i2s1 */
    if(aud_mute_enable == aud_vol_master_mute)
    {
        aud_dsp_cbuf_gain_parms.mix_i2s1 = 0;
    }
    else
    {
        aud_dsp_cbuf_gain_parms.mix_i2s1 = aud_vol_master_volume_digital;
    }
    /* mix_afe (media to downlink - even if mute is enabled, there should be audible data in downlink) */
#if defined (STEON2_PROJECT)
    aud_steon2_exceptions(); /* For the Steon2 project, this gain cell is controlled differently */
#else
    aud_update_mix_afe();
#endif /* STEON2_PROJECT */
    aud_dsp_set_cbuf_gains();
    /* side tone (balanced against the master volume) */
    aud_update_side_tone();
    aud_dsp_set_vb_gains();
}
#ifdef _USE_CUT_TEAKLIT_CODE
/*------------------------------------------------------------------------------
* Function...: aud_voice_record_channel_select_function
* Return.....: -
* Description: Program the channel of selecting that belong to voice record.
* Created....: 10.17.2012 by lbg
------------------------------------------------------------------------------*/
void aud_voice_record_channel(T_ZDrvVoice_RecordChannel channel )
{
    if(channel == VOICE_RECORD_UPLINK)
    {
                    aud_dsp_vm_parms.alpha0 = aud_constant_gain_values.alpha0;    
	             aud_dsp_vm_parms.alpha1 = 0;
    }
    else if(channel == VOICE_RECORD_DOWNLINK)
    {
       aud_dsp_vm_parms.alpha0 = 0;
    	aud_dsp_vm_parms.alpha1 = aud_constant_gain_values.alpha1;
    }
    else if(channel == VOICE_RECORD_UPLINK_AND_DOWNLINK)
    {
	 aud_dsp_vm_parms.alpha0 = aud_constant_gain_values.alpha0; 
	 aud_dsp_vm_parms.alpha1 = aud_constant_gain_values.alpha1;
    }
    else
    {
        aud_dsp_vm_parms.alpha0 = 0;
	 aud_dsp_vm_parms.alpha1 = 0;	
    }

    aud_dsp_set_vm_gains();
}
#endif

/*------------------------------------------------------------------------------
* Function...: aud_vol_program_resource_gains
* Return.....: -
* Description: Program the gain cells that belong to a specific resource.
* Created....: 07.03.2007 by KBJ
* Modified...: 01.10.2007 by KBJ (added Steon2 exceptions)
------------------------------------------------------------------------------*/
static void aud_vol_program_resource_gains(
    aud_resource_enum resource) /* the resource whose gain settings should be updated */
{
    /* General rule for setting the gain cell associated with a particular resource:              */
    /* if the resource is active and not muted -> set gain cell to the calculated resource volume */
    /* otherwise                               -> set the gain cell to 0 (mute)                   */
    switch(resource)
    {
        /* Speech + TTY */
        case aud_resource_speech:
        case aud_resource_tty:
            /* beta0 is shared between speech and TTY */
			#if 0
            if(Aud_TTY_enabled())
            {
                aud_dsp_vm_parms.beta0 = aud_constant_gain_values.beta0; /* resource volume for TTY is fixed and independent of mute status */
            }
            else 
				#endif
				if(TRUE==aud_SM_active(aud_resource_speech) && aud_mute_disable==aud_vol_resource_volume[aud_resource_speech].resource_muted_DL)
            {
                aud_dsp_vm_parms.beta0 = aud_vol_calc_resource_volume(aud_vol_resource_volume[aud_resource_speech].volume_level);
            }
            else
            {
                aud_dsp_vm_parms.beta0 = 0;
            }
            /* mic_mute */
            if(TRUE==aud_SM_active(aud_resource_speech) && aud_mute_enable==aud_vol_resource_volume[aud_resource_speech].resource_muted_UL)
            {
                aud_dsp_gain_parms.mic_mute_on = 1;
            }
            else
            {
                aud_dsp_gain_parms.mic_mute_on = 0;
            }
            /* side_tone_fact */
            aud_update_side_tone();
            aud_dsp_set_vb_gains();
            aud_dsp_set_vm_gains();
            break;

        /* Tone generator */
        case aud_resource_tone_generator:
            /* tone_fact + ton_mix_ul + ton_mix_dl */
            aud_dsp_gain_parms.tone_fact = aud_vol_calc_resource_volume(aud_vol_resource_volume[aud_resource_tone_generator].volume_level);
            if(TRUE==aud_SM_active(aud_resource_tone_generator) && aud_mute_disable==aud_vol_resource_volume[aud_resource_tone_generator].resource_muted_UL)
            {
                aud_dsp_gain_parms.ton_mix_ul = aud_constant_gain_values.ton_mix_ul;
            }
            else
            {
                aud_dsp_gain_parms.ton_mix_ul = 0; /* Mute tone generation UL */
            }
            if(TRUE==aud_SM_active(aud_resource_tone_generator) && aud_mute_disable==aud_vol_resource_volume[aud_resource_tone_generator].resource_muted_DL)
            {
                aud_dsp_gain_parms.ton_mix_dl = aud_constant_gain_values.ton_mix_dl; /* Do not affect tone generation DL */
            }
            else
            {
                aud_dsp_gain_parms.ton_mix_dl = 0; /* Mute tone generation DL */
            }
            aud_dsp_set_vb_gains();
            break;

        /* Ringer + External amplifier for speaker phone + Yamaha Music API */
        case aud_resource_ringer:
        case aud_resource_amplifier:
        case aud_resource_mapi:
#if defined(AUD_EXTERNAL_RINGER)
            if( (TRUE==aud_SM_active(aud_resource_ringer) && aud_mute_disable==aud_vol_resource_volume[aud_resource_ringer].resource_muted_DL)
                || (TRUE==aud_SM_active(aud_resource_amplifier) && aud_mute_disable==aud_vol_resource_volume[aud_resource_amplifier].resource_muted_DL)
                || (TRUE==aud_SM_active(aud_resource_mapi) && aud_mute_disable==aud_vol_resource_volume[aud_resource_mapi].resource_muted_DL))
            {
                AudRingerSM(E_ringer_unmute_amplifier, NULL);
            }
            else
            {
                AudRingerSM(E_ringer_mute_amplifier, NULL);
            }
            AudRingerSM(E_ringer_set_volume, NULL);
#endif
            break;

        /* Vibrator */
        case aud_resource_vibrator:
            break;

        /* FM Radio */
        case aud_resource_radio:
#if defined (FM_RADIO_PRESENT)
            if(TRUE==aud_SM_active(aud_resource_radio) && aud_mute_disable==aud_vol_resource_volume[aud_resource_radio].resource_muted_DL)
            {
                AudFMRadioSM(E_radio_unmute, NULL);
            }
            else
            {
                AudFMRadioSM(E_radio_mute, NULL);
            }
            AudFMRadioSM(E_radio_set_volume, NULL);
#endif
            break;

        /* Voice memo + PCM recorders */
        case aud_resource_record_vm:
        case aud_resource_record_pcm:
            /* alpha0 + alpha1 are shared between vm recorder and pcm recorder */
	     /*add on the platform of 7502 for setting channel of record online voice
            if(TRUE==aud_SM_active(aud_resource_record_vm) || TRUE==aud_SM_active(aud_resource_record_pcm))
            {
                aud_dsp_vm_parms.alpha0 = aud_constant_gain_values.alpha0; 
                aud_dsp_vm_parms.alpha1 = aud_constant_gain_values.alpha1;
            }
            else
            {
                aud_dsp_vm_parms.alpha0 = 0;
                aud_dsp_vm_parms.alpha1 = 0;
            }
            */
            if(TRUE==aud_SM_active(aud_resource_record_vm))
            {
                if(aud_mute_enable==aud_vol_resource_volume[aud_resource_record_vm].resource_muted_UL)
                {
                    aud_dsp_vm_parms.alpha0 = 0; /* if muting enabled, mute input from uplink direction */
                }
                if(aud_mute_enable==aud_vol_resource_volume[aud_resource_record_vm].resource_muted_DL)
                {
                    aud_dsp_vm_parms.alpha1 = 0; /* if muting enabled, mute input from downlink direction */
                }
            }
            else if(TRUE==aud_SM_active(aud_resource_record_pcm))
            {
                if(aud_mute_enable==aud_vol_resource_volume[aud_resource_record_pcm].resource_muted_UL)
                {
                    aud_dsp_vm_parms.alpha0 = 0; /* if muting enabled, mute input from uplink direction */
                }
                if(aud_mute_enable==aud_vol_resource_volume[aud_resource_record_pcm].resource_muted_DL)
                {
                    aud_dsp_vm_parms.alpha1 = 0; /* if muting enabled, mute input from downlink direction */
                }
            }
            aud_dsp_set_vm_gains();
            break;

        /* Voice Memo playback */
        case aud_resource_playback_vm:
            /* beta1 + gamma1 */
            if(TRUE==aud_SM_active(aud_resource_playback_vm) && aud_mute_disable==aud_vol_resource_volume[aud_resource_playback_vm].resource_muted_DL)
            {
                aud_dsp_vm_parms.beta1  = aud_vol_calc_resource_volume(aud_vol_resource_volume[aud_resource_playback_vm].volume_level);
                aud_dsp_vm_parms.gamma1 = aud_constant_gain_values.gamma1;
            }
            else
            {
                aud_dsp_vm_parms.beta1  = 0;
                aud_dsp_vm_parms.gamma1 = 0;
            }
            aud_dsp_set_vm_gains();
            break;

        /* Midi player + MP3 player */
        case aud_resource_playback_mp3:
        case aud_resource_midi_player:
            /* scal_sapp is shared between midi player and mp3 player */
            if(TRUE==aud_SM_active(aud_resource_midi_player))
            {
                if(aud_mute_disable==aud_vol_resource_volume[aud_resource_midi_player].resource_muted_DL)
                {
                    aud_dsp_cbuf_gain_parms.scal_sapp = aud_vol_calc_resource_volume(aud_vol_resource_volume[aud_resource_midi_player].volume_level);
                }
                else
                {
                    aud_dsp_cbuf_gain_parms.scal_sapp = 0;
                }
            }
            else if(TRUE==aud_SM_active(aud_resource_playback_mp3))
            {
                if(aud_mute_disable==aud_vol_resource_volume[aud_resource_playback_mp3].resource_muted_DL)
                {
                    aud_dsp_cbuf_gain_parms.scal_sapp = aud_vol_calc_resource_volume(aud_vol_resource_volume[aud_resource_playback_mp3].volume_level);
                }
                else
                {
                    aud_dsp_cbuf_gain_parms.scal_sapp = 0;
                }
            }
            else
            {
                aud_dsp_cbuf_gain_parms.scal_sapp = 0;
            }
            aud_dsp_set_cbuf_gains();
            break;

        /* PCM channel - has been replaced by I2S2_rx and I2S2_Tx resources */
        case aud_resource_PCM_channel:
            break;

        /* PCM player */
        case aud_resource_playback_pcm:
            /* scal_pcm */
            if(TRUE==aud_SM_active(aud_resource_playback_pcm) && aud_mute_disable==aud_vol_resource_volume[aud_resource_playback_pcm].resource_muted_DL)
            {
                aud_dsp_cbuf_gain_parms.scal_pcm = aud_vol_calc_resource_volume(aud_vol_resource_volume[aud_resource_playback_pcm].volume_level);
            }
            else
            {
                aud_dsp_cbuf_gain_parms.scal_pcm = 0;
            }
            aud_dsp_set_cbuf_gains();
            break;

        /* Inband resource, also known as non-baseband resource */
        case aud_resource_inband:
#if defined (STEON2_PROJECT)		
            aud_steon2_exceptions(); /* For the Steon2 project, there is an exception for the inband resource (used by external FM radio) */
            aud_dsp_set_vb_gains();
#endif /* STEON2_PROJECT */
            break;

        /* SM Power backspeaker */
        case aud_resource_smpower_backspeaker:
            break;

        /* SBC Encoder */
        case aud_resource_sbc_encoder:
            break;

        /* I2S2 Rx */
        case aud_resource_I2S2_Rx:
            /* scal_ext + lambda1 + kappa1 */
            if(TRUE==aud_SM_active(aud_resource_I2S2_Rx))
            {
                aud_dsp_cbuf_gain_parms.scal_ext = aud_vol_calc_resource_volume(aud_vol_resource_volume[aud_resource_I2S2_Rx].volume_level);
            }
            else
            {
                aud_dsp_cbuf_gain_parms.scal_ext = 0;
            }
            if(TRUE==aud_SM_active(aud_resource_I2S2_Rx) && aud_mute_disable==aud_vol_resource_volume[aud_resource_I2S2_Rx].resource_muted_DL)
            {
                aud_dsp_gain_parms.kappa1 = aud_vol_calc_resource_volume(aud_vol_resource_volume[aud_resource_I2S2_Rx].volume_level);
            }
            else
            {
                aud_dsp_gain_parms.kappa1 = 0;
            }
            if(TRUE==aud_SM_active(aud_resource_I2S2_Rx) && aud_mute_disable==aud_vol_resource_volume[aud_resource_I2S2_Rx].resource_muted_UL)
            {
                aud_dsp_gain_parms.lambda1 = aud_constant_gain_values.lambda1;
            }
            else
            {
                aud_dsp_gain_parms.lambda1 = 0;
            }
#if defined (STEON2_PROJECT)
            aud_steon2_exceptions(); /* For the Steon2 project, there is an exception for mix_afe and mix_pcmrec */
#endif /* STEON2_PROJECT */
#if defined (aud_old_I2S_interface)
            /* The handle to aud_resource_PCM_channel was mapped to aud_resource_I2S2_Rx by the interface.
             * aud_resource_PCM_channel should also have its mute and volume status changed (required by test cases) */
            aud_vol_resource_volume[aud_resource_PCM_channel].resource_muted_DL = aud_vol_resource_volume[aud_resource_I2S2_Rx].resource_muted_DL;
            aud_vol_resource_volume[aud_resource_PCM_channel].resource_muted_UL = aud_vol_resource_volume[aud_resource_I2S2_Rx].resource_muted_UL;
            aud_vol_resource_volume[aud_resource_PCM_channel].volume_level = aud_vol_resource_volume[aud_resource_I2S2_Rx].volume_level;
#endif /* aud_old_I2S_interface */
            aud_dsp_set_cbuf_gains();
            aud_dsp_set_vb_gains();
            break;

        /* I2S2 Tx */
        case aud_resource_I2S2_Tx:
            /* delta0 + delta1 */
            if(TRUE==aud_SM_active(aud_resource_I2S2_Tx) && aud_mute_disable==aud_vol_resource_volume[aud_resource_I2S2_Tx].resource_muted_DL)
            {
                aud_dsp_gain_parms.delta1 = aud_constant_gain_values.delta1;
            }
            else
            {
                aud_dsp_gain_parms.delta1 = 0;
            }
            if(TRUE==aud_SM_active(aud_resource_I2S2_Tx) && aud_mute_disable==aud_vol_resource_volume[aud_resource_I2S2_Tx].resource_muted_UL)
            {
                aud_dsp_gain_parms.delta0 = aud_constant_gain_values.delta0;
            }
            else
            {
                aud_dsp_gain_parms.delta0 = 0;
            }
#if defined (STEON2_PROJECT)
            aud_steon2_exceptions(); /* new entry points */
#endif /* STEON2_PROJECT */
            aud_dsp_set_vb_gains();
            break;

        default:
            AUD_ERROR_REPORTING(aud_error_grade2, TRAP_AUD_ERROR, "Unknown resource: %d", resource);
            break;
    }
}

/*------------------------------------------------------------------------------
* Function...: aud_steon2_exceptions
* Return.....: -
* Description: Control of gain cells that are not controlled in the same way on
*              Steon2 as on the platform.
* Modified...: 11.07.2007 by KBJ (added muting to the fm radio/inband resource)
------------------------------------------------------------------------------*/
#if defined (STEON2_PROJECT)
void aud_steon2_exceptions(void)
{
/*
    These exceptions exist on the Steon2 project:
    1) The FM radio volume uses scal_mic which is used by the path system unless the inband resource is active.
    2) The I2S2 Rx resource uses mix_afe and mix_pcmrec (media to uplink+downlink paths are not used).
    3) New gain cells for controlling I2S2 Resources located in the frame-based part.
*/

    /* The first exception is regarding the FM radio (controlled by inband resource) */
    U8 cnt = 0;
    BOOL fm_radio_active = FALSE;
    /* check whether aud_external_stereo_input path is set */
    while(aud_uplink_source_end != aud_get_uplink_path_with_priority(cnt))
    {
        if(aud_external_stereo_input == aud_get_uplink_path_with_priority(cnt))
        {
            fm_radio_active = TRUE;
        }
        cnt++;
    }
    /* if aud_external_stereo_input is set, scal_mic is controlled by inband resource volume */
    if(TRUE == fm_radio_active)
    {
        if( (aud_mute_disable==aud_vol_resource_volume[aud_resource_inband].resource_muted_UL)
            && (aud_mute_disable==aud_vol_resource_volume[aud_resource_inband].resource_muted_DL) )
        {
            aud_dsp_gain_parms.scal_mic = aud_vol_calc_resource_volume(aud_vol_resource_volume[aud_resource_inband].volume_level)/4; /* divide by 4 because 0dB is at 0x1FFF, not 0x7FFF */
        }
        else
        {
            aud_dsp_gain_parms.scal_mic = 0;
        }
    }
    else /* otherwise, scal_mic is controlled by paths as usual */
    {
        aud_dsp_gain_parms.scal_mic = aud_path_settings_result.uplink_parms.p_uplink_gain_cells->scal_mic;
    }

    /* The second exception concerns I2S2 Rx resource mute */
    /* For Steon, mix_afe is used to mute/unmute I2S2 Rx - External entry point */
    if(TRUE==aud_SM_active(aud_resource_I2S2_Rx) && aud_mute_disable==aud_vol_resource_volume[aud_resource_I2S2_Rx].resource_muted_DL)
    {
        aud_dsp_cbuf_gain_parms.mix_afe = aud_vol_master_volume_digital; /* not muted, set to master volume digital gain */
    }
    else /* media_to_downlink is not used on Steon2 */
    {
        aud_dsp_cbuf_gain_parms.mix_afe = 0;
    }
    /* For Steon, media to uplink is not controlled by paths, but is I2S2 Rx resource mute */
    if(TRUE==aud_SM_active(aud_resource_I2S2_Rx) && aud_mute_disable==aud_vol_resource_volume[aud_resource_I2S2_Rx].resource_muted_UL)
    {
        aud_dsp_cbuf_gain_parms.mix_pcmrec = aud_constant_gain_values.mix_pcmrec;
    }
    else /* media_to_uplink is not used on Steon2 */
    {
        aud_dsp_cbuf_gain_parms.mix_pcmrec = 0;
    }

    /* The third exception has to do with the new entry points for I2S2 and the new gain cells for (un)muting them */
    /* I2S2 Rx entry point MMS FB */
    if(TRUE==aud_SM_active(aud_resource_I2S2_Rx) && aud_mute_disable==aud_vol_resource_volume[aud_resource_I2S2_Rx].resource_muted_DL)
    {
        aud_dsp_gain_parms.beta1_i2s2 = aud_vol_calc_resource_volume(aud_vol_resource_volume[aud_resource_I2S2_Rx].volume_level);
    }
    else
    {
        aud_dsp_gain_parms.beta1_i2s2 = 0;
    }
    if(TRUE==aud_SM_active(aud_resource_I2S2_Rx) && aud_mute_disable==aud_vol_resource_volume[aud_resource_I2S2_Rx].resource_muted_UL)
    {
        aud_dsp_gain_parms.gamma1_i2s2 = aud_constant_gain_values.lambda1;
    }
    else
    {
        aud_dsp_gain_parms.gamma1_i2s2 = 0;
    }
    /* I2S2 Tx entry point MMS FB */
    if(TRUE==aud_SM_active(aud_resource_I2S2_Tx) && aud_mute_disable==aud_vol_resource_volume[aud_resource_I2S2_Tx].resource_muted_DL)
    {
        aud_dsp_gain_parms.alfa1_i2s2 = aud_constant_gain_values.delta1;
    }
    else
    {
        aud_dsp_gain_parms.alfa1_i2s2 = 0;
    }
    if(TRUE==aud_SM_active(aud_resource_I2S2_Tx) && aud_mute_disable==aud_vol_resource_volume[aud_resource_I2S2_Tx].resource_muted_UL)
    {
        aud_dsp_gain_parms.alfa0_i2s2 = aud_constant_gain_values.delta0;
    }
    else
    {
        aud_dsp_gain_parms.alfa0_i2s2 = 0;
    }
}
#endif /* STEON2_PROJECT */

