//SPDX-License-Identifier: MediaTekProprietary
/*
 * Copyright (C) 2006 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#define LOG_TAG "libuciwrapper"
// #define LOG_NDEBUG 0

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
//#include <cutils/sockets.h>
#include <errno.h>
#include <assert.h>

#include <cutils/properties.h>
#include <stdbool.h>
#include <inttypes.h>
#include <log/log.h>

int8_t property_get_bool(const char *key, int8_t default_value) {
    if (!key) {
        return default_value;
    }

    int8_t result = default_value;
    char buf[PROPERTY_VALUE_MAX] = {'\0',};

    int len = property_get(key, buf, "");
    if (len == 1) {
        char ch = buf[0];
        if (ch == '0' || ch == 'n') {
            result = false;
        } else if (ch == '1' || ch == 'y') {
            result = true;
        }
    } else if (len > 1) {
        if (!strcmp(buf, "no") || !strcmp(buf, "false") || !strcmp(buf, "off")) {
            result = false;
        } else if (!strcmp(buf, "yes") || !strcmp(buf, "true") || !strcmp(buf, "on")) {
            result = true;
        }
    }

    return result;
}

// Convert string property to int (default if fails); return default value if out of bounds
static intmax_t property_get_imax(const char *key, intmax_t lower_bound, intmax_t upper_bound,
                                  intmax_t default_value) {
    if (!key) {
        return default_value;
    }

    intmax_t result = default_value;
    char buf[PROPERTY_VALUE_MAX] = {'\0',};
    char *end = NULL;

    int len = property_get(key, buf, "");
    if (len > 0) {
        int tmp = errno;
        errno = 0;

        // Infer base automatically
        result = strtoimax(buf, &end, /*base*/0);
        if ((result == INTMAX_MIN || result == INTMAX_MAX) && errno == ERANGE) {
            // Over or underflow
            result = default_value;
            ALOGV("%s(%s,%" PRIdMAX ") - overflow", __FUNCTION__, key, default_value);
        } else if (result < lower_bound || result > upper_bound) {
            // Out of range of requested bounds
            result = default_value;
            ALOGV("%s(%s,%" PRIdMAX ") - out of range", __FUNCTION__, key, default_value);
        } else if (end == buf) {
            // Numeric conversion failed
            result = default_value;
            ALOGV("%s(%s,%" PRIdMAX ") - numeric conversion failed",
                  __FUNCTION__, key, default_value);
        }

        errno = tmp;
    }

    return result;
}

int64_t property_get_int64(const char *key, int64_t default_value) {
    return (int64_t)property_get_imax(key, INT64_MIN, INT64_MAX, default_value);
}

int32_t property_get_int32(const char *key, int32_t default_value) {
    return (int32_t)property_get_imax(key, INT32_MIN, INT32_MAX, default_value);
}

#define _REALLY_INCLUDE_SYS__SYSTEM_PROPERTIES_H_
#include <sys/_system_properties.h>

#include <uci.h>
#define UCI_CONFIG_FILE "/etc/config/radio_property"

static pthread_mutex_t property_mutex = PTHREAD_MUTEX_INITIALIZER;

int property_set(const char *key, const char *new_value) {

    if (NULL == key || NULL == new_value) {
        return 0;
    }

    pthread_mutex_lock(&property_mutex);

    int retValue = 0;
    int value = 0;
    struct uci_context *_ctx = uci_alloc_context();
    struct uci_ptr ptr;
    struct uci_package *pkg = NULL;
    memset(&ptr, 0, sizeof(ptr));

    // replace '.' with '_' to avoid uci r/w error
    char tmp_string[PROPERTY_KEY_MAX];
    int str_i = 0;
    strncpy(tmp_string, key, PROPERTY_KEY_MAX - 1);
    for (str_i = 0; str_i < PROPERTY_KEY_MAX; str_i++) {
        if (tmp_string[str_i] == '.') {
            tmp_string[str_i] = '_';
        }
    }

    ptr.package = "radio_property";
    ptr.section = "property";
    ptr.option = tmp_string;
    if (strlen(new_value) == 0) {
        ptr.value = ":empty";
    } else {
        ptr.value = new_value;
    }
    value = uci_set(_ctx, &ptr);
    if (value == 0) {
        value = uci_commit(_ctx, &ptr.p, false);
        if (value == 0) {
            retValue = 1;
            ALOGV("property_set ok. %s=%s return %d\n", key, new_value, value);
        } else {
            ALOGE("property_set uci_commit fail %s=%s return %d\n", key, new_value, value);
        }
    } else {
        ALOGE("uci_set fail %s=%s return %d\n", key, new_value, value);
    }
    uci_unload(_ctx, ptr.p);
    uci_free_context(_ctx);
    _ctx = NULL;

    pthread_mutex_unlock(&property_mutex);
    return retValue;
}


int property_get(const char *key, char *value, const char *default_value) {
    if (NULL == key || NULL == value) {
        return 0;
    }

    pthread_mutex_lock(&property_mutex);

    int retValue = 0;
    struct uci_context *uciCtx = uci_alloc_context();
    const char *pValueData = NULL;
    struct uci_package *pkg = NULL;
    struct uci_element *e;

    // replace '.' with '_' to avoid uci r/w error
    char tmp_string[PROPERTY_KEY_MAX];
    int str_i = 0;
    strncpy(tmp_string, key, PROPERTY_KEY_MAX - 1);
    for (str_i = 0; str_i < PROPERTY_KEY_MAX; str_i++) {
        if (tmp_string[str_i] == '.') {
            tmp_string[str_i] = '_';
        }
    }


    if (UCI_OK != uci_load(uciCtx, UCI_CONFIG_FILE, &pkg)) {
        if (default_value) {
            int len = strlen(default_value);
            memcpy(value, default_value, len);
            value[len] = '\0';
            retValue = 1;
        }
        ALOGE("%s(), uci load fail, file: %s\n", __FUNCTION__, UCI_CONFIG_FILE);
        goto cleanup;
    }

    uci_foreach_element(&pkg->sections, e) {
        struct uci_section *s = uci_to_section(e);
        if (NULL != (pValueData = uci_lookup_option_string(uciCtx, s, tmp_string))) {
            if (!strncmp(pValueData, ":empty", strlen(":empty"))) {
                value[0] = '\0';
            } else {
                strncpy(value, pValueData, strlen(pValueData));
            }
            retValue = 1;
            ALOGV("property_get, %s: %s\n", key, value);
        }
    }
    if (!retValue) {
        if (default_value) {
            int len = strlen(default_value);
            memcpy(value, default_value, len);
            value[len] = '\0';
            retValue = 1;
            ALOGV("property_get use default value, %s: %s\n", key, value);
        }
    }

    uci_unload(uciCtx, pkg);

cleanup:
    uci_free_context(uciCtx);
    uciCtx = NULL;

    pthread_mutex_unlock(&property_mutex);
    return retValue;
}


struct property_list_callback_data {
    void (*propfn)(const char *key, const char *value, void *cookie);
    void *cookie;
};

static void property_list_callback(const prop_info *pi, void *cookie) {
    char name[PROP_NAME_MAX];
    char value[PROP_VALUE_MAX];
    struct property_list_callback_data *data = cookie;

    //__system_property_read(pi, name, value);
    data->propfn(name, value, data->cookie);
}

int property_list(
    void (*propfn)(const char *key, const char *value, void *cookie),
    void *cookie) {
    struct property_list_callback_data data = { propfn, cookie };
    return 0;
}
