/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright (C) 2009, 2010 ARM Limited
 *
 * Author: Will Deacon <will.deacon@arm.com>
 */

/*
 * HW_breakpoint: a unified kernel/user-space hardware breakpoint facility,
 * using the CPU's debug registers.
 */
#ifdef CONFIG_HW_BREAKPOINT_MANAGE
#include <linux/string.h>
#include <linux/proc_fs.h>
#include <linux/ctype.h>
#include <linux/seq_file.h>
#include <asm/uaccess.h>
#include <linux/hw_breakpoint.h>
#include <linux/hw_breakpoint_manage.h>

hw_brk_pool_t 		*hw_brks_pool;
u32					breakpoint_step_flag	= 0;
u32					breakpoint_far_val		= 0;
static hw_context_t wp_bp_context_table[HW_MAX_BREAKPOINT_COUNT] 	= {{0}};
static char 		shell_cmd_result[HW_BREAKPOINT_RESULT_MAX_LEN] 	= {0};

/* Number of BRP/WRP registers on this CPU. */
static int	arm_brps_num;
static int	arm_wrps_num;

static void	hw_print_regs(void);
static void	hw_printk_list(void);
static u32 	bcr_read(u32 num);
static void bcr_write(u32 num, u32 value);
static u32 	bvr_read(u32 num);
static void bvr_write(u32 num, u32 value);
static u32 	wcr_read(u32 num);
static void wcr_write(u32 num, u32 value);
static u32 	wvr_read(u32 num);
static void wvr_write(u32 num, u32 value);

void hw_breakpoint_enable_program(u32 num)
{
    u32 bcr;
    bcr = bcr_read(num);
    bcr |= 0x1;
	bcr_write(num, bcr);
}

void hw_breakpoint_disable_program(u32 num)
{
    u32 bcr;
    
    bcr = bcr_read(num);
    if (bcr & 0x1){
        bcr &= ~(0x1);
        bcr_write(num, bcr);
    }
}

void hw_breakpoint_enable_data(u32 num)
{
    u32 wcr;
		
    wcr = wcr_read(num);
    wcr |= 0x1;
	wcr_write(num, wcr);
}

void hw_breakpoint_disable_data(u32 num)
{
    u32 wcr;
    wcr = wcr_read(num);
    if (wcr & 0x1){
        wcr &= ~(0x1);
		wcr_write(num, wcr);
    }
}

void hw_breakpoint_do_action(u8 enableflag)
{
	int 				i 		   = 0;
	struct perf_event 	**slots_bp = NULL;
	struct perf_event 	**slots_wp = NULL;

    if (!hw_brks_pool->enable)
        return;

	slots_wp = get_wrps_regs();
	slots_bp = get_brps_regs();

	/*Index 0 is reserved for step breakpoint. */
	for (i = 1; i < arm_brps_num; i++) {
		if (slots_bp[i] != NULL){
			if(enableflag == HW_BREAKPOINT_ENABLE_REGS)
				hw_breakpoint_enable_program(i);
			else
				hw_breakpoint_disable_program(i);
		}
	}
	for (i = 0; i < arm_wrps_num; i++) {
	   if (slots_wp[i] != NULL){
			if(enableflag == HW_BREAKPOINT_ENABLE_REGS)
				hw_breakpoint_enable_data(i);
			else
				hw_breakpoint_disable_data(i);
		}
	}
}


bool hw_breakpoint_match_databreak(u32 address)
{
	int 				i 				= 0;
	int 				cond_start_data	= 0;
	int 				cond_end_data	= 0;
	struct 	perf_event 	*wp		  		= NULL;
    struct 	perf_event 	**slots_wp 		= NULL;
	bool 				ret 			= HW_BREAKPOINT_FAIL;
	unsigned long		wp_addr  		= 0;
	__u64				watch_addr  	= 0;

	slots_wp = get_wrps_regs();
	for (i = 0; i < arm_brps_num; i++) {
		rcu_read_lock();
      	wp = slots_wp[i];
		if (!wp)
			continue;

	    if (address >= wp->attr.bp_addr && address < (wp->attr.bp_addr + 0x4)){
	   		watch_addr 		= wp->attr.bp_addr;
			cond_start_data = wp->attr.cond_start_data;
			cond_end_data   = wp->attr.cond_end_data;
			wp_addr		= (unsigned long)watch_addr;
			switch (wp->attr.bp_cond){
			case HW_BREAKPOINT_COND_IN_RANGE:
				
				if ((wp_addr & 0x3) == 0){
				    if ((HW_BREAKPOINT_REG32(wp_addr) >= (unsigned long)cond_start_data)
						&&(HW_BREAKPOINT_REG32(wp_addr) <= (unsigned long)cond_end_data)){					
							rcu_read_unlock();
					        return HW_BREAKPOINT_SUCCESS; /*break*/
				   	}
				    else
				        continue;
				}
				else if ((wp_addr & 0x1) == 0){
				    if ((HW_BREAKPOINT_REG16(wp_addr) >= (cond_start_data & 0xFFFF))
						&& (HW_BREAKPOINT_REG16(wp_addr) <= (cond_end_data & 0xFFFF))){
						rcu_read_unlock();
						return HW_BREAKPOINT_SUCCESS; /*break*/
					}
				    else
				        continue;
				}
				else{
				     if ((HW_BREAKPOINT_REG8(wp_addr) >= (cond_start_data & 0xFF))
						&&(HW_BREAKPOINT_REG8(wp_addr) <= (cond_end_data & 0xFF))){
						rcu_read_unlock();
						return HW_BREAKPOINT_SUCCESS; /*break*/
					}
				    else
				        continue;
				}
				break;
			case HW_BREAKPOINT_COND_OUT_RANGE:
				if ((wp_addr & 0x3) == 0){
				     if ((HW_BREAKPOINT_REG32(wp_addr) < (long)cond_start_data)
						||(HW_BREAKPOINT_REG32(wp_addr) > (long)cond_end_data)){
						rcu_read_unlock();
						return HW_BREAKPOINT_SUCCESS; /*break*/
					}
				    else
				        continue;
				}
				else if ((wp_addr & 0x1) == 0){
				    if ((HW_BREAKPOINT_REG16(wp_addr) < (cond_start_data & 0xFFFF))
						||(HW_BREAKPOINT_REG16(wp_addr) > (cond_end_data & 0xFFFF))){
						rcu_read_unlock();
						return HW_BREAKPOINT_SUCCESS; /*break*/
					}
				    else
				        continue;
				}
				else{
				     if ((HW_BREAKPOINT_REG8(wp_addr) < (cond_start_data & 0xFF))
						||(HW_BREAKPOINT_REG8(wp_addr) > (cond_end_data & 0xFF))){
						rcu_read_unlock();
						return HW_BREAKPOINT_SUCCESS; /*break*/
					}
				    else
				        continue;
				}
				break;
			default:
				break;
		    }			
		}
	}
	rcu_read_unlock();
	return ret;
}

static u32 bcr_read(u32 num)
{
	return arm_bpwp_regs_read(ARM_BASE_BCR + num);
}

static void bcr_write(u32 num, u32 value)
{
	arm_bpwp_regs_write(ARM_BASE_BCR + num, value);
}

static u32 bvr_read(u32 num)
{
	return arm_bpwp_regs_read(ARM_BASE_BVR + num);
}

static void bvr_write(u32 num, u32 value)
{
	arm_bpwp_regs_write(ARM_BASE_BVR + num, value);
}

static u32 wcr_read(u32 num)
{
	return arm_bpwp_regs_read(ARM_BASE_WCR + num);
}

static void wcr_write(u32 num, u32 value)
{
	arm_bpwp_regs_write(ARM_BASE_WCR + num, value);
}

static u32 wvr_read(u32 num)
{
	return arm_bpwp_regs_read(ARM_BASE_WVR + num);
}

static void wvr_write(u32 num, u32 value)
{
	arm_bpwp_regs_write(ARM_BASE_WVR + num, value);
}

u32 dscr_read(void)
{
	unsigned int value = 0;
#if defined (__GNUC__)
	asm volatile("mrc p14,0,%0,c0,c2,2" : "=r" (value));
#elif defined(__ARMCC_VERSION)
	__asm { mrc p14,0,value,c0,c1,0; }
#endif
	return value;
}

void dscr_write(u32  value)
{
#if defined (__GNUC__)
	// asm volatile("mcr p14,0,%0,c0,c1,0" :: "r" (value));DBGDSCRINT;
	asm volatile("mcr p14,0,%0,c0,c2,2" :: "r" (value));
#elif defined(__ARMCC_VERSION)
	__asm { mcr p14,0,value,c0,c1,0; }
#endif
}

static u32 oslar_read(void)  
{
    u32 value = 0;
    asm volatile("mrc p14,0,%0,c1,c0,4" : "=r" (value));
    return value;
}

u32 didr_read(void)
{
    u32 value = 0;

#if defined (__GNUC__)
	asm volatile("mrc p14,0,%0,c0,c0,0" : "=r" (value));
#elif defined(__ARMCC_VERSION)
	__asm { mrc p14,0,value,c0,c0,0; }
#endif
    return value;
}

void oslar_write(u32 value)
{
    asm volatile("mcr p14,0,%0,c1,c0,4" :: "r" (value));
}

u32 far_read(void)
{
    u32 value = 0;
#if defined (__GNUC__)
        asm volatile("mrc p15,0,%0,c6,c0,0" : "=r" (value));
#elif defined(__ARMCC_VERSION)
        __asm { mrc p15, 0, value, c6, c0, 0; }
#endif
    return value;
}


static int hw_breakpoint_del_data(u32 num)
{
	struct perf_event 		**wrps 		= NULL; 
	struct perf_event       **brps		= NULL;	
	struct perf_event 		**slots_wp 	= NULL; 
	struct perf_event 		**slots_bp 	= NULL; 
    u8                      step_index 	= 0;

	if (num >= arm_wrps_num){
		printk("[Hw breakpoint]Error param: num is out of range 0-%d\n", arm_wrps_num);
		return HW_BREAKPOINT_ERR_NUM_OUTOFRANGE;
	}
	
	slots_wp = get_wrps_regs();
	slots_bp = get_brps_regs();

	wrps = &slots_wp[num];
	
	/*stepϵʹõ0*/
	step_index 	= HW_BREAKPOINT_STEP_REGS_INDEX;
	brps 		= &slots_bp[step_index];
	
	if (*wrps == NULL){
		printk("[hw breakpoint]del data fail, index %d watchpoint is NULL\n",num);
		return -EPERM;
	}
	
	if (*brps != NULL){
		if (slots_wp[num]->attr.step_addr ==    \
			slots_bp[step_index]->attr.step_addr){			
			bcr_write(step_index, 0);	
			bvr_write(step_index, 0);
			*brps = NULL;
		}
	}	
	unregister_hw_breakpoint(*wrps);
	wcr_write(num, 0);
	wvr_write(num, 0);
	*wrps = NULL;	
	return HW_BREAKPOINT_SUCCESS;
}

static int hw_breakpoint_del_program(u32 num)
{
	struct perf_event	**slots_bp = NULL; 
	struct perf_event 	**brps     = NULL;	

	if (num == 0 || num >= arm_brps_num){
		printk("Error param:num 0 is reserve for step breakpoint,num %d is out of range 1-%d,\n",num, arm_brps_num);
		return HW_BREAKPOINT_ERR_NUM_OUTOFRANGE;
	}
		
	slots_bp = get_brps_regs();
	
	brps = &slots_bp[num];
	if (*brps == NULL){
		printk("[hw breakpoint]del data fail, index %d breakpoint is NULL\n",num);
		return -EPERM;
	}
	
	unregister_hw_breakpoint(*brps);
	bcr_write(num, 0);
	bvr_write(num, 0);

	*brps = NULL;
	return HW_BREAKPOINT_SUCCESS;
}


/**
 * hw_register_set_watchponit - register a watchponit in the kernel
 * @wp_param: breakpoint attributes
 * @callback_fun: callback to trigger when we hit the watchponit
 *
 * @return a set of per_cpu pointers to perf events
 */

static int hw_register_set_watchponit(hw_brk_setdata_param_t *wp_param, perf_overflow_handler_t callback_fun)
{
	struct perf_event 				*wp				= NULL;		
	struct perf_event 				**slots_wp		= NULL; 
	struct perf_event_attr 			attr 			= {0};
	struct perf_event * __percpu 	*hw_handler_wp 	= NULL;
	int 							i 				= 0;

	if (wp_param->num >= arm_wrps_num){
		snprintf(shell_cmd_result, HW_BREAKPOINT_RESULT_MAX_LEN, "[HW-BRPS]Error param: num is out of range 0-%d\n", (arm_wrps_num - 1));
		return HW_BREAKPOINT_ERR_NUM_OUTOFRANGE;
	}

	hw_breakpoint_enable();
	slots_wp = get_wrps_regs();

	for (i = 0; i < arm_wrps_num; i++) {
      	wp = slots_wp[i];
		if (wp == NULL)
			continue;
			
		if ((wp->attr.bp_addr  == wp_param->addr)
			&&(wp->attr.bp_len == wp_param->addr_len)
			&&(wp->attr.bp_type == wp_param->type)
			&&(wp->attr.bp_cond == wp_param->cond)
			&&(wp->attr.cond_start_data == wp_param->start_value)
			&&(wp->attr.cond_end_data == wp_param->end_value)){
			
			printk("[HW-BRPS]registered watchpoint already in hw watchpoint pool\n");
			snprintf(shell_cmd_result, HW_BREAKPOINT_RESULT_MAX_LEN, "[HW-BRPS]registered watchpoint already in hw watchpoint pool\n");
			return -EPERM;
		}

		if(wp->attr.bp_index == wp_param->num){
			hw_breakpoint_del_data(wp_param->num);
		}	
	}

	hw_breakpoint_init(&attr);
	
	attr.bp_addr		 = wp_param->addr;	
	attr.bp_len			 = wp_param->addr_len;
	attr.bp_type		 = wp_param->type;
	attr.bp_index 		 = wp_param->num;
	attr.bp_cond 		 = wp_param->cond;
	attr.cond_start_data = wp_param->start_value; 	
	attr.cond_end_data	 = wp_param->end_value; 
	
	hw_handler_wp = register_wide_hw_breakpoint(&attr, callback_fun, NULL);
	if (IS_ERR((void __force *)hw_handler_wp)) {
		snprintf(shell_cmd_result,HW_BREAKPOINT_RESULT_MAX_LEN,"Breakpoint registration failed\n");
		return PTR_ERR((void __force *)hw_handler_wp);
	}	
	return 0;
}

/**
 * hw_register_set_breakponit - register a breakpoint in the kernel
 * @bp_param: breakpoint attributes
 * @callback_fun: callback to trigger when we hit the breakpoint
 *
 * @return a set of per_cpu pointers to perf events
 */
static int hw_register_set_breakponit(hw_brk_setprogram_param_t *bp_param, perf_overflow_handler_t callback_fun)
{
	struct perf_event 				*bp        = NULL;	
	struct perf_event 				**slots_bp = NULL; 
	struct perf_event_attr			attr;	
	struct perf_event * __percpu 	*hw_handler_bp = NULL;
    unsigned char           i  = 0;

	if (bp_param->num == 0 || bp_param->num >= arm_brps_num){
		printk("[Hw breakpoint]Error param:num 0 is reserve for step breakpoint,num %d is out of range 1-%d,\n", bp_param->num, (arm_brps_num-1));
		snprintf(shell_cmd_result, HW_BREAKPOINT_RESULT_MAX_LEN,"[Hw breakpoint]Error param:num 0 is reserve for step breakpoint,num %d is out of range 1-%d,\n", bp_param->num, (arm_brps_num-1));
		return HW_BREAKPOINT_ERR_NUM_OUTOFRANGE;
	}
	
	hw_breakpoint_enable();
	
	slots_bp = get_brps_regs();
	for (i = 0; i < arm_brps_num; i++) {
      	bp = slots_bp[i];
		if (bp == NULL)
			continue;
		if(bp->attr.bp_addr == bp_param->addr){
			printk("[Hw breakpoint]register fail! breakpoint item already in hw breakpoint pool\n");
			snprintf(shell_cmd_result, HW_BREAKPOINT_RESULT_MAX_LEN,"[Hw breakpoint]register fail! breakpoint item already in hw breakpoint pool\n");
			return -EPERM;
		}
		
		if(bp->attr.bp_index == bp_param->num){
			hw_breakpoint_del_program(bp_param->num);
		}	
	}
	
	hw_breakpoint_init(&attr);
	
	attr.bp_index = bp_param->num;
	attr.bp_addr  = bp_param->addr;
	attr.bp_len   = bp_param->len;;
	attr.bp_type  = HW_BREAKPOINT_X;

	hw_handler_bp = register_wide_hw_breakpoint(&attr, callback_fun, NULL);
	if (IS_ERR((void __force *)hw_handler_bp)){
		printk(KERN_INFO "Breakpoint registration failed!\n");		
		snprintf(shell_cmd_result, HW_BREAKPOINT_RESULT_MAX_LEN,"Breakpoint registration failed!\n");
		return PTR_ERR((void __force *)hw_handler_bp);
	}
	return 0;
}


static void hw_breakpoint_print_commandformat(void)
{    
	printk("/*---------------------------------------------------------*/\n");
	printk("Breakpoint command format:\n");
	printk("/*brk setprogram [num] [address] [len]*/\n");
	printk("/*brk setdata [num] [address] [len] [type] [cond] [value]--[value]*/\n");  
	printk("/*brk deldata [num]*/\n");
	printk("/*brk delprogram [num]*/\n");
	printk("/*brk list*/\n");	
	printk("/*brk regs*/\n");
	printk("Example:\n");
	printk("/*brk setprogram 0 0xC26AE940 4*/\n");
	printk("/*brk setdata 0 0xC26AE940 4 w  i 0x00000000--0xFFFFFFFF*/\n");
	printk("/*brk setdata 1 0xC26AE940 4 r  o 0x00000020--0x00000020*/\n");
	printk("Program Breakpoint Count: %d\n", arm_brps_num);
	printk("Data Breakpoint Count: %d\n", arm_wrps_num);
	printk("/*---------------------------------------------------------*/\n");
}

/**
 * hw_brk_handler - breakpoint call back function
 * @bp: breakpoint event
 * @data: perf data
 * @regs: regs ptr
 */
static void hw_brk_handler(struct perf_event *bp,
			       struct perf_sample_data *data,
			       struct pt_regs *regs)
{
	dump_stack();
	panic("[Hw breakpoint panic]\n");
}

static int hw_get_watchpoint_params(hw_brk_setdata_param_t	*wp_attr, const char *hw_info_buf)
{
	u32 					count		= 0;		
	int 					ret 		= 0;	
	u32 					addr_len	= 0;
	char					type[HW_BREAKPOINT_MODE_MAX_LEN] = {0};
	char					cond[HW_BREAKPOINT_COND_MAX_LEN] = {0}; 

    if(wp_attr == NULL || hw_info_buf == NULL)
		return -EINVAL;
	
	count = sscanf (hw_info_buf, 
					"%d %llx %d %s %s %8x--%8x", 
					(int *)&(wp_attr->num), &(wp_attr->addr), 
					&addr_len, type, cond, 
					&(wp_attr->start_value), &(wp_attr->end_value));

	if (count != HW_BREAKPOINT_SETDATA_PARAM_NUM)
		return HW_BREAKPOINT_ERR_PARMAM_NUM;

	if ((addr_len !=  HW_BREAKPOINT_LEN_1) && (addr_len !=	HW_BREAKPOINT_LEN_2)
		&&(addr_len !=	HW_BREAKPOINT_LEN_4) && (addr_len !=  HW_BREAKPOINT_LEN_8))
		return HW_BREAKPOINT_ERR_ADDR_LEN;

	wp_attr->addr_len = addr_len;

	if (!strcmp((const char *)type, "r"))	 
		wp_attr->type = HW_BREAKPOINT_R;
	else if (!strcmp((const char *)type, "w"))
		wp_attr->type = HW_BREAKPOINT_W;
	else if (!strcmp((const char *)type, "rw"))
		wp_attr->type = HW_BREAKPOINT_RW;
	else
		return HW_BREAKPOINT_ERR_TYPE;

	if (!strcmp((const char *)cond, "i"))	
		wp_attr->cond = HW_BREAKPOINT_COND_IN_RANGE;
	else if (!strcmp((const char *)cond, "o")
			|| (!strcmp((const char *)cond, "O")))
		wp_attr->cond = HW_BREAKPOINT_COND_OUT_RANGE;
	else 
		return HW_BREAKPOINT_ERR_COND;

	return ret;
}
/**
 * hw_breakpoint_register_entry - register to wp_on_reg and bp_on_reg
 * @hw_cmd_type: cmd type buf
 * @hw_info_buf: breakpoint info buf
 * Returns -errno, or 0 for success.
 *	brk setdata num address r/w/rw i/o start--end	
 *	numbreakpoint index
 *	address:breakpoint watchpoint addr
 *	r/w/rwbrakpoint type: read(r)write(w)read/write(rw)
 *	i/ocond:(i:in range)(o: out of range)
 *	start--endvalue range	
 *	example: 
 *	brk setdata 0 0xc1234560 4 w i 0x00000000--0xFFFFFFFF  //set watchpoint 0len is 4, monitor 0xc1234560 addrwhen writed to any valuepanic
 *	brk setdata 1 0xc1234560 4 w i 0x00000020--0x00000020  //set watchpoint 0len is 4, monitor 0xc1234560 addrwhen writed to 0x20, panic
*/

static int hw_breakpoint_register_entry(const char *hw_cmd_type, const char *hw_info_buf)
{
	u32  					count		= 0;		
	int 					ret 		= 0;	
	u32 					index 		= 0;
    hw_brk_setdata_param_t 	wp_param 	= {0};
	
	if (hw_cmd_type == NULL || hw_info_buf == NULL){
		hw_breakpoint_print_commandformat();
		return -EINVAL;
	}
	
	memset(shell_cmd_result, 0, sizeof(shell_cmd_result));
	
	if (!strcmp((const char *)hw_cmd_type, "setdata")){
	    ret = hw_get_watchpoint_params(&wp_param, hw_info_buf);
		if (ret)
			goto err_param;
		
		ret = hw_register_set_watchponit(&wp_param, hw_brk_handler);
		if (ret)
			goto err_out;
		else
			hw_brks_pool->active_num_wrps++;
	}
	
	else if (!strcmp((const char *)hw_cmd_type, "setprogram")){		
		
		hw_brk_setprogram_param_t bp_param = {0};
		
		count = sscanf(hw_info_buf, "%d %llx %d", (int *)&bp_param.num, &bp_param.addr, &bp_param.len);
		if (count != HW_BREAKPOINT_SETPROGRAM_PARAM_NUM){
			ret = HW_BREAKPOINT_ERR_PARMAM_NUM;
			goto err_param; 
		}
		if ((bp_param.len != HW_BREAKPOINT_LEN_2)&&(bp_param.len !=	HW_BREAKPOINT_LEN_4)){
			ret = HW_BREAKPOINT_ERR_ADDR_LEN;
			goto err_param; 
		}
		ret = hw_register_set_breakponit(&bp_param, hw_brk_handler);
		if (ret)
			goto err_out;
		else
			hw_brks_pool->active_num_brps++;
	}
	else if (!strcmp((const char *)hw_cmd_type, "deldata")){
		count = sscanf(hw_info_buf, "%d", &index);
		if (count != HW_BREAKPOINT_DEL_PARAM_NUM){
			ret = HW_BREAKPOINT_ERR_PARMAM_NUM;
			goto err_param; 
		}
		ret = hw_breakpoint_del_data(index);
		if (ret)
			goto err_out;
		else
			hw_brks_pool->active_num_wrps--;
	}
	else if (!strcmp((const char *)hw_cmd_type, "delprogram")){
		count = sscanf(hw_info_buf, "%d", &index);
		if (count != HW_BREAKPOINT_DEL_PARAM_NUM){
			ret = HW_BREAKPOINT_ERR_PARMAM_NUM;
			goto err_param; 
		}
		ret = hw_breakpoint_del_program(index);
		if (ret)
			goto err_out;
		else
			hw_brks_pool->active_num_brps--;
	}
	else if (!strcmp((const char *)hw_cmd_type, "list"))
		hw_printk_list();
	else if (!strcmp((const char *)hw_cmd_type, "regs"))
		hw_print_regs();
	else{
		ret = HW_BREAKPOINT_ERR_CMD;
		goto err_param; 
	}
	printk("Shell cmd %s SUCCESS\n", hw_cmd_type);
	snprintf(shell_cmd_result, HW_BREAKPOINT_RESULT_MAX_LEN,"Shell cmd %s SUCCESS\n", hw_cmd_type);
	return ret;

err_param:    
  	printk("breakpoint: params error reason: 0x%02x\n", ret);
  	snprintf(shell_cmd_result, HW_BREAKPOINT_RESULT_MAX_LEN,"breakpoint: params error reason: 0x%02x\n", ret);
    return -EINVAL;
err_out:   
  	printk("breakpoint set fail, register ret =0x%02x:\n", ret);   
  	snprintf(shell_cmd_result, HW_BREAKPOINT_RESULT_MAX_LEN,"breakpoint set fail, register ret =0x%02x:\n", ret); 
    return -EINVAL;
}

/**
 * hw_breakpoint_write - extract hw breakpoint data from a user string
 * @file: the file to extract from
 * @user_buf: the buffer to extract from
 * @size: the length of the buffer
 * @off: .
 *
 * Returns -errno, or 0 for success.
 */
static int hw_breakpoint_write(struct file *file, const char __user * user_buf, size_t size, loff_t * off)
{
	char 	*hwbrk_buf 		= NULL;
	char    *comms_ptr		= NULL;
	char    *hw_info_buf	= NULL;	
	char    *param_ptr      = NULL;	
	int     len 			= 0;
	int     pre_len 		= 0;
	u32 	hw_info_len 	= 0;			
	u32 	comms_size 	    = 0;
	int		retval 			= -EFAULT;
	char    hw_cmd_type[HW_BREAKPOINT_CMD_TYPE_LEN]	= {0};

	hwbrk_buf = kmalloc(size + 1, GFP_KERNEL);	
	if (hwbrk_buf == NULL)
		return -ENOMEM;
    
	if (copy_from_user(hwbrk_buf, user_buf, size)){
		kfree(hwbrk_buf);
		return -EINVAL;
	}
	hwbrk_buf[size] = '\0';

	hw_info_buf = kmalloc(HW_BREAKPOINT_BUF_MAX_LEN, GFP_KERNEL);
	if (hw_info_buf == NULL){		
		kfree(hwbrk_buf);
		return -ENOMEM;
	}

	param_ptr = strstr(hwbrk_buf, HW_BREAKPOINT_SHELLCMD_PREFIX);
	if (param_ptr != NULL){
		pre_len = param_ptr - hwbrk_buf + strlen(HW_BREAKPOINT_SHELLCMD_PREFIX);
		comms_ptr = param_ptr + pre_len;
		comms_size = size - pre_len;
	}
	else { 					
		kfree(hwbrk_buf);	
		kfree(hw_info_buf);
		printk("Err: Shellcmd prefix should be:%s \n", HW_BREAKPOINT_SHELLCMD_PREFIX);
		return -EINVAL;
	}
	while (*hwbrk_buf != '\0' ||  comms_size > 0){		
		len = 0;
		for (; *comms_ptr != '\0' ; comms_ptr++, comms_size--){
			if (!HW_SPECIAL_CHARACTER(*comms_ptr) && isprint(*comms_ptr))
				len++;

			else if ((HW_SPECIAL_CHARACTER(*comms_ptr)|| !isprint(*comms_ptr))&& len > 0){
				break;
			}
		}

		if (len > 0){
			strncpy(hw_cmd_type, comms_ptr - len, len);
			hw_cmd_type[len] = '\0';
			hw_info_len = strlen(hwbrk_buf) - pre_len - len;
			strncpy(hw_info_buf, comms_ptr, hw_info_len);
			hw_info_buf[hw_info_len] = '\0';
			retval = hw_breakpoint_register_entry(hw_cmd_type, hw_info_buf);
			break;
		}
		else {
		 	printk("breakpoint: params error set fail\n");  
			hw_breakpoint_print_commandformat();
			kfree(hwbrk_buf);	
			kfree(hw_info_buf);
    		return -EINVAL;	
		}
	}
	
	*off += size;
	kfree(hwbrk_buf);	
	kfree(hw_info_buf);
	retval = size;
    return retval;
}

static void hw_print_wpbp_regs(void)
{
	int 				i 		   = 0;
	struct perf_event 	**slots_bp = NULL;
	struct perf_event 	**slots_wp = NULL;
	struct perf_event 	**slot_t   = NULL;

	slots_wp = get_wrps_regs();
	slots_bp = get_brps_regs();
		
	printk("[HW breakpoint]watchpoint count = %d\n", arm_wrps_num);	

	for (i = 0; i < arm_wrps_num; i++) {
	   slot_t = &slots_wp[i];
	   printk("[HW breakpoint]wp_on_regs[%d] = 0x%08x\n", i, (u32)*slot_t);
	   if (slots_wp[i] != NULL)
	   		printk("[HW breakpoint]wp_on_regs[%d] addr = 0x%llx, cond =%d, cond_start = 0x%08x, cond_end = 0x%08x\n",
	   		i, slots_wp[i]->attr.bp_addr, slots_wp[i]->attr.bp_cond, slots_wp[i]->attr.cond_start_data, slots_wp[i]->attr.cond_end_data); 
	}

	printk("[HW breakpoint]breakpoint count = %d\n", arm_brps_num);	
	/* Print the breakpoint. */
	for (i = 0; i < arm_brps_num; i++) {
	   	slot_t = &slots_bp[i];
		printk("[HW breakpoint]bp_on_regs[%d] = 0x%08x\n", i, (u32)*slot_t); 
		if (slots_bp[i] != NULL)
			printk("[HW breakpoint]bp_on_regs[%d] addr = 0x%llx\n",i, slots_bp[i]->attr.bp_addr);
	}
}

void hw_print_regs(void)
{
	u32 dscr	= dscr_read();
	int i		= 0;

    if (!hw_brks_pool->enable)
		return;

	printk("[HW breakpoint]watchpoint count = %d, breakpoint count = %d\n", arm_wrps_num, arm_brps_num);	
	printk("[HW breakpoint]dscr= 0x%x\n", dscr);
	
	for (i = 0; i< arm_wrps_num; i++)
		printk("[HW breakpoint]WCR%d = %d WVR%d = %x\n", i, wcr_read(i), i, wvr_read(i));
	for (i = 0; i< arm_wrps_num; i++)		
		printk("[HW breakpoint]BCR%d = %d BVR%d = %x\n", i, bcr_read(i), i, bvr_read(i));
}

static void hw_printk_list(void)
{
	if (!hw_brks_pool->enable){        
		printk("Breakpoint not enable!\n");
		return;
	}
	
	printk("/*---------------------------------------------------------*/\n");
	printk("Program Breakpoints:\n");
	if (!hw_brks_pool->active_num_brps && !hw_brks_pool->active_num_wrps){
		printk("No breakpoints exist now!\n");
		return;
	}
	//hw_print_regs();
	hw_print_wpbp_regs();
}


void hw_breakpoint_context_save(void)
{
    int index = 0;

    if(hw_brks_pool->active_num_brps == 0 
		&& hw_brks_pool->active_num_wrps == 0 )
		return;
	
    wp_bp_context_table[index].dscr = dscr_read();	   

	for (index = 0; index < arm_brps_num; index++) {
		wp_bp_context_table[index].bcr = bcr_read(index);
		wp_bp_context_table[index].bvr = bvr_read(index);
	}

	for (index = 0; index < arm_wrps_num; index++) {		
		wp_bp_context_table[index].wcr = wcr_read(index);
		wp_bp_context_table[index].wvr = wvr_read(index);	
	}
}

void hw_breakpoint_restore_context(void)
{
	u32 index	= 0;

	if(hw_brks_pool->active_num_brps == 0 
		&& hw_brks_pool->active_num_wrps == 0)
		return;
	
	hw_breakpoint_enable();

	for (index = 0; index < arm_brps_num; index++){
		/* Setup the control register. */
		bcr_write(index, wp_bp_context_table[index].bcr);
		
		/* Setup the address register. */		
		bvr_write(index, wp_bp_context_table[index].bvr);
	}

	for (index = 0; index < arm_wrps_num; index++){
		
		/* Setup the control register. */
		wcr_write(index, wp_bp_context_table[index].wcr);
		
		/* Setup the address register. */		
		wvr_write(index, wp_bp_context_table[index].wvr);		
	}
}


void hw_breakpoint_enable(void)
{
    u32 dscr;
    u32 val;
    
    dscr = dscr_read();
    if (!(dscr & ARM_DSCR_MDBGEN)){
        dscr |= ARM_DSCR_MDBGEN;
        dscr_write(dscr);
    }
	val = HW_ARM_OSLAR_UNLOCKED;
	//val = oslar_read();
	//if((val & HW_ARM_OSLAR_LOCKED)){		
	//	val = HW_ARM_OSLAR_UNLOCKED;
		oslar_write(val);
	//}
	
	hw_brks_pool->enable = 1;
}
/**
 * hw_breakpoint_pool_init - init the global hw breakpoint pool 
 *
 * Returns -errno.
 */
int hw_breakpoint_pool_init(void)
{
	hw_brks_pool = (hw_brk_pool_t *)kmalloc(sizeof(hw_brk_pool_t), GFP_KERNEL);
	if (hw_brks_pool == NULL)
		return -ENOMEM;   
	
	hw_brks_pool->active_num_wrps = 0;	
	hw_brks_pool->active_num_brps = 0;	
	hw_brks_pool->wrps_on_reg = get_wrps_regs();
	hw_brks_pool->brps_on_reg = get_brps_regs();
	get_wrps_brps_nums(&arm_wrps_num, &arm_brps_num);

	return HW_BREAKPOINT_SUCCESS;
}


/**
 * hw_breakpoint_info_show - show the registered breakpoints
 * @m: proc file structure
 * @v: 
 *
 * Returns -errno, or 0 for success.
 */
static int hw_breakpoint_info_show(struct seq_file *m, void *v)
{
	int 				i 		   = 0;
	struct perf_event 	**slots_bp = NULL;
	struct perf_event 	**slots_wp = NULL;

	slots_wp = get_wrps_regs();
	slots_bp = get_brps_regs();

	seq_printf(m, "[HW breakpoint]watchpoint count = %d\n", arm_wrps_num);	

	for (i = 0; i < arm_wrps_num; i++) {
	   seq_printf(m, "[HW breakpoint]wp_on_regs[%d] = 0x%08x\n", i, (u32)slots_wp[i]);
	   if (slots_wp[i] != NULL)
	   		seq_printf(m, "[HW breakpoint]wp_on_regs[%d] addr = 0x%llx, cond =%d, cond_start = 0x%08x, cond_end = 0x%08x\n",
	   		i, slots_wp[i]->attr.bp_addr, slots_wp[i]->attr.bp_cond, slots_wp[i]->attr.cond_start_data, slots_wp[i]->attr.cond_end_data); 
	}
	
	seq_printf(m, "[HW breakpoint]breakpoint count = %d\n", arm_brps_num);	
	
	/* Print the breakpoint. */
	for (i = 0; i < arm_brps_num; i++) {
		seq_printf(m, "[HW breakpoint]bp_on_regs[%d] = 0x%08x\n", i, (u32)slots_wp[i]); 
		if (slots_bp[i] != NULL)
			seq_printf(m, "[HW breakpoint]bp_on_regs[%d] addr = 0x%llx\n",i, slots_bp[i]->attr.bp_addr);
	}
	
	seq_printf(m,"%s\n", "Result is");
	seq_printf(m,"%s\n", shell_cmd_result);
	return 0;
}

/**
 * hw_breakpoint_open - open function
 * @inode: file inode
 * @file: file descriptor
 *
 * Returns -errno, or 0 for success.
 */
static int hw_breakpoint_open(struct inode *inode, struct file *file)
{
	return single_open(file, hw_breakpoint_info_show, NULL);
}

/**
 * hwbreakpoint_proc_fops - proc file ops
 * @open: open the registed proc file.
 * @read: read the registed proc file.
 * @write: write to the registed proc file.
 *
 * Returns -errno, or 0 for success.
 */
static const struct file_operations hwbreakpoint_proc_fops = {
	.open		= hw_breakpoint_open,
	.read		= seq_read,
	.write		= hw_breakpoint_write
};

static int __init hw_breakpoint_proc_init(void)
{
   	proc_create("hw_breakpoint_info", 0, NULL, &hwbreakpoint_proc_fops);
	return 0;
}

module_init(hw_breakpoint_proc_init);

#endif /*CONFIG_HW_BREAKPOINT_MANAGE*/

