/*
 *  zx297510-pm.h - zx297510 power management interface.
 *
 *  Written by zxp.
 *
 */

#ifndef _ZX297510_PM_H
#define _ZX297510_PM_H

#define pm_reg_sync_write(v, a) \
        do {    \
            iowrite32(v, a);    \
            dsb(); \
        } while (0)

#define pm_read_reg(addr) \
    ioread32(addr)

#define pm_write_reg(addr, val)   \
	pm_reg_sync_write(val, addr)

#define pm_set_reg(addr, val) \
    pm_reg_sync_write(pm_read_reg(addr) | (val), addr)

#define pm_clr_reg(addr, val) \
    pm_reg_sync_write(pm_read_reg(addr) & ~(val), addr)


#define pm_reg_sync_write_16(v, a) \
        do {    \
            iowrite16(v, a);    \
            dsb(); \
        } while (0)

#define pm_read_reg_16(addr) \
    ioread16(addr)

#define pm_write_reg_16(addr, val)   \
	pm_reg_sync_write_16(val, addr)

/* these regs define may be moved to clk module, now for debug */
#define	TOP_624_CFG1_REG 						(unsigned)(ZX29_TOP_VA+0x8)
#define	TOP_UFI_CFG1_REG 						(unsigned)(ZX29_TOP_VA+0x10)
#define	TOP_UFI_CFG2_REG 						(unsigned)(ZX29_TOP_VA+0x14)
#define	TOP_UFI_DELAY_CFG_REG 					(unsigned)(ZX29_TOP_VA+0x30)
#define	TOP_UFI_SEL_REG 						(unsigned)(ZX29_TOP_VA+0x44)
#define	TOP_UFI_DIV_REG 						(unsigned)(ZX29_TOP_VA+0x6C)

#define PLL_BIT_PWRDN     						(1U << 31)
#define PLL_BIT_LD     							(1U << 30)

#define ZX29_TIMER1_VA     						(ZX297510_TIMER1_BASE - ZX297510_A2LSP_BASE + ZX29_A2LSP_VA)
#define ZX29_TIMER3_VA     						(ZX297510_TIMER3_BASE - ZX297510_A1_BASE + ZX29_A1_VA)
#define TIMER3_CUR_VALUE_REG					(unsigned)(ZX29_TIMER3_VA+0x18)
#define TIMER3_LOAD_VALUE_REG					(unsigned)(ZX29_TIMER3_VA+0x08)

/* for debug uart clk */
#define ZX29_A1CRM_VA	   						(ZX297510_A1CRM_BASE - ZX297510_A1_BASE + ZX29_A1_VA)
#define	A1_CRM_PCLK_EN_REG						(unsigned)(ZX29_A1CRM_VA + 0xC)
#define	A1_CRM_WCLK_EN_REG						(unsigned)(ZX29_A1CRM_VA + 0x10)

#define	A1_CRM_UART0_BIT						(1 << 0)
#define	A1_CRM_UART1_BIT						(1 << 1)

/* for ddr clock gate */
#define ZX29_SOCCRM_VA	   						(ZX297510_SOCCRM_BASE - ZX297510_A1_BASE + ZX29_A1_VA)

#define	LPDDR_CLK_EN_REG						(unsigned)(ZX29_SOCCRM_VA + 0x2C)

#define	A9_DDRCTRL_PORT1_CLKEN					(1U << 17)
#define	A9_DDRCTRL_PORT2_CLKEN					(1U << 18)
#define	A9_DDRCTRL_PORT3_CLKEN					(1U << 19)

#define	INVALID_INT_NUM							(0xFFFF)

#define	__SLEEP_TIME_1s__						((s64)(1000000))
#define	__SLEEP_TIME_1m__						(__SLEEP_TIME_1s__*60)
#define	__SLEEP_TIME_1h__						(__SLEEP_TIME_1m__*60)
#define	__MAX_SLEEP_TIME__						(__SLEEP_TIME_1h__*18)

/* iram is for axi_freq and debug */
#define IRAM_CHANGE_AXI_BASE       				(ZX29_IRAM02_VA+0x7000)

/*  
 *	flag   : m0 set 0 when get request,  ap set 1 to request m0
 *  target : ap axi target, defined by zx297510_axi_freq
 *  cur    : current axi freq, defined by zx297510_axi_freq
 *  ack    : m0 set 1 when done request,  ap set 0 before new request
**/
#define AXI_PS2M0_FLAG         					(IRAM_CHANGE_AXI_BASE + 0x0)
#define AXI_PS2M0_TARGET           				(IRAM_CHANGE_AXI_BASE + 0x2)
#define AXI_PHY2M0_FLAG         				(IRAM_CHANGE_AXI_BASE + 0x4)
#define AXI_PHY2M0_TARGET          				(IRAM_CHANGE_AXI_BASE + 0x6)
#define AXI_AP2M0_FLAG           				(IRAM_CHANGE_AXI_BASE + 0x8)
#define AXI_AP2M0_TARGET           				(IRAM_CHANGE_AXI_BASE + 0xa)
#define AXI_ZSP2M0_FLAG         				(IRAM_CHANGE_AXI_BASE + 0xc)
#define AXI_ZSP2M0_TARGET          				(IRAM_CHANGE_AXI_BASE + 0xe)

#define AXI_CURRENT_FREQ            			(IRAM_CHANGE_AXI_BASE + 0x10)

#define AXI_M02PS_ACK           				(IRAM_CHANGE_AXI_BASE + 0x12)
#define AXI_M02PHY_ACK           				(IRAM_CHANGE_AXI_BASE + 0x14)
#define AXI_M02AP_ACK           				(IRAM_CHANGE_AXI_BASE + 0x16)
#define AXI_M02ZSP_ACK           				(IRAM_CHANGE_AXI_BASE + 0x18)

#define	IRAM_SIGN_FLAG_FOR_A9					(0xA5A55A5A)
#define IRAM_ADDR_FOR_A9         				(IRAM_CHANGE_AXI_BASE + 0x20)	
#define	sign_a9_run_flag()						pm_write_reg(IRAM_ADDR_FOR_A9, IRAM_SIGN_FLAG_FOR_A9)

/*test flag*/
#define IRAM_ADDR_FOR_SLEEP_CNT					(IRAM_CHANGE_AXI_BASE+0x20)
#define IRAM_ADDR_FOR_WAKE_CNT					(IRAM_CHANGE_AXI_BASE+0x24)
#define IRAM_ADDR_FOR_CHIPSLEEPEN_CNT			(IRAM_CHANGE_AXI_BASE+0x28)
#define IRAM_ADDR_FOR_g_ChipSleepFlag			(IRAM_CHANGE_AXI_BASE+0x2C)
#define IRAM_ADDR_FOR_g_ChipRealCanSleepTime	(IRAM_CHANGE_AXI_BASE+0x30)

#define IRAM_ADDR_FOR_DENALI_DDRCTRL_89_CKE1	(IRAM_CHANGE_AXI_BASE+0x34)
#define IRAM_ADDR_FOR_DENALI_DDRCTRL_89_CKE2	(IRAM_CHANGE_AXI_BASE+0x38)

#define IRAM_ADDR_FOR_AP_CONFIG					(IRAM_CHANGE_AXI_BASE+0x3C)
#define IRAM_ADDR_FOR_PS_CONFIG					(IRAM_CHANGE_AXI_BASE+0x40)
#define IRAM_ADDR_FOR_PHY_CONFIG				(IRAM_CHANGE_AXI_BASE+0x44)

#define IRAM_ADDR_FOR_PLL_624_208_CONFIG1		(IRAM_CHANGE_AXI_BASE+0x48)
#define IRAM_ADDR_FOR_PLL_CPU_UFI_CONFIG1		(IRAM_CHANGE_AXI_BASE+0x4C)
#define IRAM_ADDR_FOR_PLL_CPU_BUS_CONFIG1		(IRAM_CHANGE_AXI_BASE+0x50)
#define IRAM_ADDR_FOR_PLL_TD_LTE_CONFIG1		(IRAM_CHANGE_AXI_BASE+0x54)

#define IRAM_ADDR_FOR_SYSTEM_CONFIG_REG			(IRAM_CHANGE_AXI_BASE+0x58)

#define IRAM_ADDR_FOR_PS1						(IRAM_CHANGE_AXI_BASE+0x60)
#define IRAM_ADDR_FOR_PS2						(IRAM_CHANGE_AXI_BASE+0x64)
#define IRAM_ADDR_FOR_PS3						(IRAM_CHANGE_AXI_BASE+0x68)
#define IRAM_ADDR_FOR_PS4						(IRAM_CHANGE_AXI_BASE+0x6C)

#define IRAM_ADDR_FOR_PS5						(IRAM_CHANGE_AXI_BASE+0x70)
#define IRAM_ADDR_FOR_PS6						(IRAM_CHANGE_AXI_BASE+0x74)
#define IRAM_ADDR_FOR_PS7						(IRAM_CHANGE_AXI_BASE+0x78)
#define IRAM_ADDR_FOR_PS8						(IRAM_CHANGE_AXI_BASE+0x7C)

#define IRAM_ADDR_FOR_PS9						(IRAM_CHANGE_AXI_BASE+0x80)
#define IRAM_ADDR_FOR_PS10						(IRAM_CHANGE_AXI_BASE+0x84)
#define IRAM_ADDR_FOR_PS11						(IRAM_CHANGE_AXI_BASE+0x88)
#define IRAM_ADDR_FOR_PS12						(IRAM_CHANGE_AXI_BASE+0x8C)

#define		SLEEP_CODE_ADDR						(ZX29_IRAM6_VA+0x200)
#define		SLEEP_CODE_LENGTH					(0x80)

/* copied from zx297510_uart.h */ 
#define ZX29_UART0_VA	   						(ZX297510_UART0_BASE - ZX297510_A1_BASE + ZX29_A1_VA)
#define ZX29_UART1_VA	   						(ZX297510_UART1_BASE - ZX297510_A1_BASE + ZX29_A1_VA)

#define zx297510_UART_IBRD						0x24	/* Integer baud rate divisor register. */
#define zx297510_UART_FBRD						0x28	/* Fractional baud rate divisor register. */
#define zx297510_UART_LCRH						0x30	/* Line control register. */
#define zx297510_UART_CR						0x34	/* Control register. */
#define zx297510_UART_IFLS						0x38	/* Interrupt fifo level select. */
#define zx297510_UART_IMSC						0x40	/* Interrupt mask. */
#define zx297510_UART_ICR						0x4c	/* Interrupt clear register. */
#define zx297510_UART_DMACR						0x50	/* DMA control register. */

/*------ uart control reg -----*/
#define UART_CR_CTSEN							(1<<15)	/* CTS hardware flow control */
#define UART_CR_RTSEN							(1<<14)	/* RTS hardware flow control */
#define UART_CR_OUT2							(1<<13)	/* OUT2 */
#define UART_CR_OUT1							(1<<12)	/* OUT1 */
#define UART_CR_RTS								(1<<11)	/* RTS */
#define UART_CR_DTR								(1<<10)	/* DTR */
#define UART_CR_RXE								(1<<9)	/* receive enable */
#define UART_CR_TXE								(1<<8)	/* transmit enable */
#define UART_CR_LBE								(1<<7)	/* loopback enable */
#define UART_CR_SIRLP							(1<<2)	/* SIR low power mode */
#define UART_CR_SIREN							(1<<1)	/* SIR enable */
#define UART_CR_UARTEN							(1<<0)	/* UART enable */

//#define	DEBUG_UART0
#define	DEBUG_UART1
#ifdef DEBUG_UART0
#define	debug_uart_base()						ZX29_UART0_VA
#define	no_use_uart_base()						ZX29_UART1_VA
#else
#define	debug_uart_base()						ZX29_UART1_VA
#define	no_use_uart_base()						ZX29_UART0_VA
#endif

struct pm_wake_cause_info
{
	unsigned 	pcu_index;
	unsigned 	gic_index;
	char *		int_name;
};

struct zx_uart_context
{
	unsigned int	ibrd;		/*0x24	 Integer baud rate divisor register. */
	unsigned int	fbrd;		/*0x28	 Fractional baud rate divisor register. */
	unsigned int	lcrh;		/*0x30	 Line control register. */
	unsigned int	cr;			/*0x34	 Control register. */
	unsigned int	ifls;		/*0x38	 Interrupt fifo level select. */
	unsigned int	imsc;		/*0x40	 Interrupt mask. */	
	unsigned int	dmacr;		/*0x50	 DMA control register. */		
};

struct zx_timer_context
{
	unsigned int	cfg;		/*0x04	 config register. */
	unsigned int	load;		/*0x08	 load register. */
	unsigned int	start;		/*0x0C	 timer start register. */
	unsigned int	count;		/*0x18	 current counter register. */	
};

/* for device or sw to restore */
struct zx_suspend_context
{
	struct zx_uart_context  uart;
	struct zx_timer_context timer;
};

#define	pm_ufi_pll_pwrdn()		pm_set_reg(TOP_UFI_CFG1_REG, PLL_BIT_PWRDN)
#define	pm_ufi_pll_pwron()									\
do {                                            			\
    pm_clr_reg(TOP_UFI_CFG1_REG, PLL_BIT_PWRDN); 			\
    while(!(pm_read_reg(TOP_UFI_CFG1_REG)&PLL_BIT_LD));		\
} while (0)	


/*used as pm common interface*/
#define zx_pcu_init 				zx297510_pcu_init
#define zx_set_pcu					zx297510_set_pcu
#define zx_clear_pcu				zx297510_clear_pcu

#define zx_unmask_wakeup_interrupt	zx297510_unmask_wakeup_interrupt
#define zx_interrupt_mask_restore	zx297510_interrupt_mask_restore

extern void zx297510_pm_clk_init(void);
#define zx_clk_init					zx297510_pm_clk_init

extern void zx297510_save_crm(u32 *pointer, u32 crm_base);
extern void zx297510_restore_crm(u32 *pointer, u32 crm_base);
	
extern void restore_crm(u32 crm_base);

extern void pm_switch_clk_to_26m(void);
extern void pm_switch_clk_from_26m(void);
extern void save_ufi_clk(void);

extern void pm_set_pll_used(void);

extern s64 pm_get_remainder_time(void);
extern void setup_timer_wakeup(s64 us);

extern unsigned pm_get_wakeup_int_no(void);
extern char * pm_get_wakeup_int_name(void);
extern void pm_get_wake_cause(void);

extern void debug_uart_suspend(void);
extern void debug_uart_resume(void);

extern void pm_timer_suspend(void); 
extern void pm_timer_resume(void);

extern void zx_pm_pre_suspend(void);
extern void zx_pm_post_suspend(void);

extern void zx_pm_trace_out(void);

#endif /*_ZX297510_PM_H*/
