/*
 * ZTE cpu context save&restore driver
 *
 * Copyright (C) 2013 ZTE Ltd.
 * 	by zxp
 *
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/types.h>
#include <linux/suspend.h>

#include <asm/memory.h>

#include <mach/zx-pm.h>


struct zx_pm_main_table zx_pm_main_table;
unsigned pm_device_memory [PM_MEMORY_SIZE/4];

extern volatile int sleep_ret_flag[];

/**
 * This is where the reset vector jumps to.
 *
 */
static unsigned get_device_memory(unsigned size);
/*********************************************************************
 * FUNCTION DEFINATIONS
 ********************************************************************/
/**
 *  zx_pm_context_init - initial context for cpu suspend and resume.
 *  
 *  initial the struct variable for context. 
 */
int zx_pm_context_init(void)
{
    int i;
	struct zx_cluster_context* temp_cluster_context;
	struct zx_cpu_context* temp_cpu_context;

    pr_info("[SLP] Power/PM_CONTEXT_INIT \n");

	/**the iram:0 shadow ram in ddr.
	 * when cpu is reset, first run in iram address 0,
	 * then we will jump to resume code in ddr
	 */
	zx_pm_main_table.ddr_ram_addr              = (u32)((void *)cpu_wake_up);
	zx_pm_main_table.ddr_ram_size              = SHADOW_RAM_SIZE;

	zx_pm_main_table.reset_code_addr           = (u32)((void *)cpu_reset_handler);
    zx_pm_main_table.num_cpus                  = MAX_CPU_NUM;

	zx_pm_main_table.scu_address	           = (u32)SCU_ADDRESS;
	zx_pm_main_table.ic_dist_address	       = (u32)SCU_ADDRESS + IC_DISTRIBUTOR_OFFSET;
	zx_pm_main_table.ic_interface_address      = (u32)SCU_ADDRESS + IC_INTERFACE_OFFSET;	
	zx_pm_main_table.l2_address                = (u32)L2_CONTROLLER_ADDRESS;
	zx_pm_main_table.crm_address               = (u32)A9_CRM_ADDRESS;
	
	zx_pm_main_table.scu_address_p	           = (u32)SCU_ADDRESS_P;
	zx_pm_main_table.l2_address_p              = (u32)L2_CONTROLLER_ADDRESS_P;
	zx_pm_main_table.crm_address_p             = (u32)A9_CRM_ADDRESS_P;	
  
	/*cluster info*/
	temp_cluster_context                       = (void *)get_device_memory(sizeof(struct zx_cluster_context));
	temp_cluster_context->flags                = 0;
	temp_cluster_context->saved_items          = 0;
	temp_cluster_context->gic_dist_shared_data = (void *)get_device_memory(GIC_DIST_SHARED_DATA_SIZE);
	temp_cluster_context->l2_data              = (void *)get_device_memory(L2_DATA_SIZE);
	temp_cluster_context->scu_data             = (void *)get_device_memory(SCU_DATA_SIZE);
	temp_cluster_context->global_timer_data    = (void *)get_device_memory(GLOBAL_TIMER_DATA_SIZE);
	temp_cluster_context->crm_data             = (void *)get_device_memory(CRM_DATA_SIZE);
	temp_cluster_context->power_state          = CPU_POWER_MODE_RUN;
	
	zx_pm_main_table.cluster_context           = temp_cluster_context;

	/*cpu info*/
	for(i=0;i<zx_pm_main_table.num_cpus;i++)
	{
		temp_cpu_context                        = (void *)get_device_memory(sizeof(struct zx_cpu_context));
		temp_cpu_context->flags                 = 0;
		temp_cpu_context->saved_items           = 0;
		temp_cpu_context->control_data          = (void *)get_device_memory(CONTROL_DATA_SIZE);
		temp_cpu_context->pmu_data              = (void *)get_device_memory(PMU_DATA_SIZE);
		temp_cpu_context->timer_data            = (void *)get_device_memory(TIMER_DATA_SIZE);
		temp_cpu_context->vfp_data              = (void *)get_device_memory(VFP_DATA_SIZE);
		temp_cpu_context->gic_interface_data    = (void *)get_device_memory(GIC_INTERFACE_DATA_SIZE);
		temp_cpu_context->gic_dist_private_data = (void *)get_device_memory(GIC_DIST_PRIVATE_DATA_SIZE);
		temp_cpu_context->banked_registers      = (void *)get_device_memory(BANKED_REGISTERS_SIZE);
		temp_cpu_context->cp15_data             = (void *)get_device_memory(CP15_DATA_SIZE);
		temp_cpu_context->debug_data            = (void *)get_device_memory(DEBUG_DATA_SIZE);
		temp_cpu_context->mmu_data              = (void *)get_device_memory(MMU_DATA_SIZE);
		temp_cpu_context->other_data            = (void *)get_device_memory(OTHER_DATA_SIZE);
		temp_cpu_context->power_state           = CPU_POWER_MODE_RUN;
		temp_cpu_context->sleep_type            = CPU_SLEEP_TYPE_NULL;

		zx_pm_main_table.cpu_context[i]         = temp_cpu_context;
	}

	pr_info("[SLP] Power/PM_CONTEXT_INIT END\n");

    return 0;
}

/**
 * sleep_type - idle/suspend.
 * 
 * set the context flag will be saved according to sleep type.
 * 
 */
int zx_set_context_level (cpu_sleep_type_t sleep_type)
{
    unsigned cpu_id;

	cpu_id = read_cpuid();
	zx_pm_main_table.cur_cpu = cpu_id;	

	if(CPU_SLEEP_TYPE_LP1 == sleep_type)  //suspend
	{
		zx_pm_main_table.cluster_context->flags = LP1_MG_SAVE_FLAG;
		zx_pm_main_table.cpu_context[cpu_id]->flags = LP1_CPU_SAVE_FLAG;

		zx_pm_main_table.cpu_context[cpu_id]->power_state  = CPU_POWER_MODE_SHUTDOWN;
		zx_pm_main_table.cluster_context->power_state = CPU_POWER_MODE_SHUTDOWN;
	}
	else if(CPU_SLEEP_TYPE_IDLE_LP2 == sleep_type)//deep idle
	{
		if(0 == cpu_id)
		{
			zx_pm_main_table.cluster_context->flags = LP2_MG_SAVE_FLAG;
			zx_pm_main_table.cpu_context[cpu_id]->flags = LP2_CPU0_SAVE_FLAG;

			zx_pm_main_table.cpu_context[cpu_id]->power_state  = CPU_POWER_MODE_DORMANT;
			zx_pm_main_table.cluster_context->power_state = CPU_POWER_MODE_DORMANT;
		}
		else
		{
			zx_pm_main_table.cluster_context->flags = MG_SAVE_FLAG_NULL;
			zx_pm_main_table.cpu_context[cpu_id]->flags = LP2_CPUX_SAVE_FLAG;

			zx_pm_main_table.cpu_context[cpu_id]->power_state  = CPU_POWER_MODE_DORMANT;
		}
	}
	else if(CPU_SLEEP_TYPE_LP3 == sleep_type)//wfi only
	{
		zx_pm_main_table.cluster_context->flags = MG_SAVE_FLAG_NULL;
		zx_pm_main_table.cpu_context[cpu_id]->flags = MG_SAVE_FLAG_NULL;

		zx_pm_main_table.cpu_context[cpu_id]->power_state  = CPU_POWER_MODE_STANDBY;
		zx_pm_main_table.cluster_context->power_state = CPU_POWER_MODE_STANDBY;		
	}

	zx_pm_main_table.cpu_context[cpu_id]->sleep_type = sleep_type;

	return 0;
}

/**
 * This function saves all the context that will be lost 
 * when a CPU and cluster enter a low power state.
 *
 */
void zx_pm_save_context(void)
{
    struct zx_cpu_context *context;
    struct zx_cluster_context *cluster_context;
	int is_secure = true;	
	int ret=0;

	context         = zx_pm_main_table.cpu_context[zx_pm_main_table.cur_cpu];
	cluster_context = zx_pm_main_table.cluster_context;
	
	if(context->flags&CPU_SAVE_TIMERS)
	{
		save_a9_timers(context->timer_data,
			           zx_pm_main_table.scu_address+PRIVATE_TWD_OFFSET);
		context->saved_items |= CPU_SAVE_TIMERS;
	}

	if(context->flags&CPU_SAVE_PMU)
	{
		save_performance_monitors(context->pmu_data);
		context->saved_items |= CPU_SAVE_PMU;
	}

	if(context->flags&CPU_SAVE_VFP)
	{
		save_vfp(context->vfp_data);
		context->saved_items |= CPU_SAVE_VFP;
	}

	/*only for smp */
	if(zx_pm_main_table.ic_interface_address)
	{
        save_gic_interface(context->gic_interface_data,
						   zx_pm_main_table.ic_interface_address,
						   is_secure);
        ret = save_gic_distributor_private(context->gic_dist_private_data, 
									 		zx_pm_main_table.ic_dist_address,
									 		is_secure);
	}

	if(ret == -1)
	{
		BUG_ON(0);
		while(1);		
	}

	if(context->flags&CPU_SAVE_DEBUG)
    {
        save_v7_debug(context->debug_data);
		context->saved_items |= CPU_SAVE_DEBUG;
    }


	save_banked_registers(context->banked_registers);
    save_cp15(context->cp15_data);
	
    if (context->flags&CPU_SAVE_OTHER)
    {
		save_a9_other(context->other_data, is_secure);
		context->saved_items |= CPU_SAVE_OTHER;		
    }

    if (cluster_context->flags&CPU_SAVE_GLOBAL_TIMER)
    {
        save_a9_global_timer(cluster_context->global_timer_data, 
							 zx_pm_main_table.scu_address+GLOBAL_TIMER_OFFSET);
        cluster_context->saved_items |= CPU_SAVE_GLOBAL_TIMER;
    }

    if(cluster_context->flags&CPU_SAVE_GIC)
    {
        ret = save_gic_distributor_shared(cluster_context->gic_dist_shared_data, 
										zx_pm_main_table.ic_dist_address, 
										is_secure);
		cluster_context->saved_items |= CPU_SAVE_GIC;
    }	
/*
	if(ret == -1)
	{
		while(1);
	}
*/

    save_control_registers(context->control_data, is_secure);
    save_mmu(context->mmu_data);
	
    if (cluster_context->flags&CPU_SAVE_SCU)
    {
        save_a9_scu(cluster_context->scu_data, zx_pm_main_table.scu_address);
        cluster_context->saved_items |= CPU_SAVE_SCU;
    }

    if (cluster_context->flags&CPU_SAVE_L2)
    {
        save_pl310(cluster_context->l2_data, zx_pm_main_table.l2_address);
        cluster_context->saved_items |= CPU_SAVE_L2;
    }	

    if (cluster_context->flags&CPU_SAVE_CRM)
    {
        cluster_context->saved_items |= CPU_SAVE_CRM;
		save_crm(cluster_context->crm_data, zx_pm_main_table.crm_address);
    }

	/* save ufi clk */
	save_ufi_clk();

	/*saved completely*/
	sleep_ret_flag[zx_pm_main_table.cur_cpu] = 0;
}

/** 
 * This function restores all the context that was lost 
 * when a CPU and cluster entered a low power state. It is called shortly after
 * reset, with the MMU and data cache off.
 *
 * note:before MMU is enable, all address should convert to PA
 */
void zx_pm_restore_context(void)
{
    unsigned cpu_id;
    struct zx_cpu_context *context;
    struct zx_cluster_context *cluster_context;
	int is_secure = true;
	
	cpu_id = read_cpuid();

	context         	= zx_pm_main_table.cpu_context[cpu_id];
	cluster_context		= zx_pm_main_table.cluster_context;

    if (cluster_context->saved_items & CPU_SAVE_CRM)
    {
		restore_crm(cluster_context->crm_data, zx_pm_main_table.crm_address);
		cluster_context->saved_items &= ~CPU_SAVE_CRM;
    }	

    if (cluster_context->saved_items & CPU_SAVE_SCU)
    {
        restore_a9_scu(cluster_context->scu_data, zx_pm_main_table.scu_address);
		cluster_context->saved_items &= ~CPU_SAVE_SCU;
    }

    if (cluster_context->saved_items & CPU_SAVE_L2)
    {
        restore_pl310(cluster_context->l2_data, 
					  zx_pm_main_table.l2_address, 
					  cluster_context->power_state == CPU_POWER_MODE_DORMANT);
		cluster_context->saved_items &= ~CPU_SAVE_L2;		
    }

    /* Next get the MMU back on */
    restore_mmu(context->mmu_data);
    restore_control_registers(context->control_data, is_secure);
    /* 
     * MMU and L1 and L2 caches are on, we may now read/write any data.
     * Now we need to restore the rest of this CPU's context 
     */

    /* Restore shared items if necessary */
    if (cluster_context->saved_items & CPU_SAVE_GIC)
    {
        gic_distributor_set_enabled(false, zx_pm_main_table.ic_dist_address);
        restore_gic_distributor_shared(cluster_context->gic_dist_shared_data, zx_pm_main_table.ic_dist_address, is_secure);
        gic_distributor_set_enabled(true, zx_pm_main_table.ic_dist_address);
		
        restore_gic_distributor_private(context->gic_dist_private_data, zx_pm_main_table.ic_dist_address, is_secure);
        restore_gic_interface(context->gic_interface_data, zx_pm_main_table.ic_interface_address, is_secure);
		cluster_context->saved_items &= ~CPU_SAVE_GIC;
    }
    if (cluster_context->saved_items & CPU_SAVE_GLOBAL_TIMER)
    {
        restore_a9_global_timer(cluster_context->global_timer_data, 
								zx_pm_main_table.scu_address+GLOBAL_TIMER_OFFSET);
		cluster_context->saved_items &= ~CPU_SAVE_GLOBAL_TIMER;
    }
	

    /* Get the debug registers restored, so we can debug most of the APPF code sensibly! */    
    if (context->saved_items&CPU_SAVE_DEBUG)
    {
        restore_v7_debug(context->debug_data);
		context->saved_items &= ~CPU_SAVE_DEBUG;
    }

    if (context->saved_items&CPU_SAVE_OTHER)
    {
		restore_a9_other(context->other_data, is_secure);
		context->saved_items &= ~CPU_SAVE_OTHER;
    }

    restore_cp15(context->cp15_data);
    restore_banked_registers(context->banked_registers);

    if (context->saved_items&CPU_SAVE_VFP)
    {
        restore_vfp(context->vfp_data);
		context->saved_items &= ~CPU_SAVE_VFP;		
    }

    if (context->saved_items&CPU_SAVE_TIMERS)
    {
        restore_a9_timers(context->timer_data, 
						  zx_pm_main_table.scu_address+PRIVATE_TWD_OFFSET);
		context->saved_items &= ~CPU_SAVE_TIMERS;
    }

    if (context->saved_items&CPU_SAVE_PMU)
    {
        restore_performance_monitors(context->pmu_data);
		context->saved_items &= ~CPU_SAVE_PMU;		
    }

	cluster_context->power_state = CPU_POWER_MODE_RUN;
	context->power_state		 = CPU_POWER_MODE_RUN;

	/*restore completely*/
	sleep_ret_flag[cpu_id] = 1;
}
/** 
 * This function restores the context that lost when cpu not power down correctly.
 *
 */
void zx_pm_restore_abort_context(void)
{
    unsigned cpu_id;
    struct zx_cpu_context *context;
    struct zx_cluster_context *cluster_context;
	
	cpu_id = read_cpuid();
	context         	= zx_pm_main_table.cpu_context[cpu_id];
	cluster_context		= zx_pm_main_table.cluster_context;
	
    if (cluster_context->saved_items & CPU_SAVE_CRM)
    {
		restore_crm(cluster_context->crm_data, zx_pm_main_table.crm_address);
		cluster_context->saved_items &= ~CPU_SAVE_CRM;
    }	
	
    if (cluster_context->saved_items & CPU_SAVE_SCU)
    {
		cluster_context->saved_items &= ~CPU_SAVE_SCU;
    }
	
    if (cluster_context->saved_items & CPU_SAVE_L2)
    {
		cluster_context->saved_items &= ~CPU_SAVE_L2;		
    }
        
    /* 
     * MMU and L1 and L2 caches are on, we may now read/write any data.
     * Now we need to restore the rest of this CPU's context 
     */

    /* Restore shared items if necessary */
    if (cluster_context->saved_items & CPU_SAVE_GIC)
    {
		cluster_context->saved_items &= ~CPU_SAVE_GIC;
    }

    if (cluster_context->saved_items & CPU_SAVE_GLOBAL_TIMER)
    {
        restore_a9_global_timer(cluster_context->global_timer_data, 
								zx_pm_main_table.scu_address+GLOBAL_TIMER_OFFSET);
		cluster_context->saved_items &= ~CPU_SAVE_GLOBAL_TIMER;
    }
	

    /* Get the debug registers restored, so we can debug most of the APPF code sensibly! */    
    if (context->saved_items&CPU_SAVE_DEBUG)
    {
		context->saved_items &= ~CPU_SAVE_DEBUG;
    }

    if (context->saved_items&CPU_SAVE_OTHER)
    {
		context->saved_items &= ~CPU_SAVE_OTHER;
    }

    if (context->saved_items&CPU_SAVE_VFP)
    {
        restore_vfp(context->vfp_data);
		context->saved_items &= ~CPU_SAVE_VFP;		
    }

    if (context->saved_items&CPU_SAVE_TIMERS)
    {
        restore_a9_timers(context->timer_data, 
						  zx_pm_main_table.scu_address+PRIVATE_TWD_OFFSET);
		context->saved_items &= ~CPU_SAVE_TIMERS;
    }

    if (context->saved_items&CPU_SAVE_PMU)
    {
        restore_performance_monitors(context->pmu_data);
		context->saved_items &= ~CPU_SAVE_PMU;		
    }

	cluster_context->power_state = CPU_POWER_MODE_RUN;
	context->power_state		 = CPU_POWER_MODE_RUN;

	/*restore completely*/
	sleep_ret_flag[cpu_id] = 1;
}


/**
 * Simple Device memory allocator function.
 * Returns start address of allocated region
 * Allocates region of size bytes, size will be rounded up to multiple of sizeof(long long)
 * Memory is zero-initialized.
 *
 * This function is from ARM.
 */
static long long *device_memory = (void *)pm_device_memory;
static unsigned get_device_memory(unsigned size)
{
    static unsigned watermark = 0;
    static unsigned total_size  = 0;	
    unsigned ret, chunks_required;
    
    ret = watermark;
    chunks_required = (size + sizeof(long long) - 1) / sizeof(long long);
    watermark += chunks_required;
    
    if (watermark >= PM_MEMORY_SIZE / sizeof(long long))
    {
    	pr_info("[SLP] error alloc size: %d Bytes \n", size);
    	BUG_ON(0);
        while(1);
        return 0;  /* No output possible, so loop */		
    }

	total_size += size;
	pr_info("[SLP] alloc size: %d Bytes , total size %d Bytes\n", size, total_size);

	
    return (unsigned) &device_memory[ret];
}


/*===================================================================
 *==  iram address allocation  ======================================
 *===================================================================
 *========= 0      ~~  0x1FF : reset code area    ===================
 *========= 0x200  ~~  0x27F : code area_1        ===================
 *========= 0x280  ~~  0x2ff : code area_2        ===================
 *========= 0x300  ~~  999   : reserved[0]        ===================
 *========= 1000   ~~  1003  : code addr[0]       ===================
 *========= 1004   ~~  1007  : code addr[1]       ===================
 *========= 1008   ~~  1011  : debug[0]           ===================
 *========= 1012   ~~  1015  : debug[1]           ===================
 *========= 1016   ~~  1019  : sleep time         ===================
 *========= 1020   ~~  1023  : reserved[1]        ===================
 *===================================================================
 */

/** 
 * This function tell the wakeup code address to CPU_M0 for cpu waked up from deep sleep(shutdown or dormant),
 * the wakeup code will exist in iram (address 0), but iram will power down when AP sub-system or
 * whole chip power down, so A9 tell CPU_M0 the address, before A9 reset, CPU_M0 will copy the code
 * from this address to the iram.
 *
 * This code should call after zx_pm_context_init.
 */
void zx_set_wakeup_address(void *wakeup_addr)
{
    pr_info("[SLP] Power/WAKEUP_ADDRESS \n");	
	
	/*copy the ddr reset code to iram*/
    pr_info("[SLP] wakeup addr: %x, ddr_ram_addr: %x, ddr_ram_size: %d \n", (unsigned)wakeup_addr, zx_pm_main_table.ddr_ram_addr, zx_pm_main_table.ddr_ram_size);	
	memcpy(wakeup_addr, (unsigned char *)(zx_pm_main_table.ddr_ram_addr), zx_pm_main_table.ddr_ram_size);

	*(unsigned *)(wakeup_addr+1000)	= __pa(zx_pm_main_table.reset_code_addr);	
    pr_info("[SLP] cpu_reset_handler: %x \n", *(unsigned *)(wakeup_addr+1000));	

	/* copy sleep code to iram */
	memcpy((void *)SLEEP_CODE_ADDR, (void *)do_sleep_cpu, SLEEP_CODE_LENGTH);
}

/** 
 * get current sleep_type helper function.
 *
 * This code only used pm internel.
 */
cpu_sleep_type_t pm_get_sleep_type(void)
{
	return zx_pm_main_table.cpu_context[zx_pm_main_table.cur_cpu]->sleep_type;
}

/** 
 * init lp for scu/l2.
 *
 * This code only used pm internel.
 */
void pm_init_l2_and_scu(void)
{
	init_lp_of_scu(zx_pm_main_table.scu_address);
	init_lp_of_l2(zx_pm_main_table.l2_address);
}
