/*
 * ZTE cpu sleep driver
 *
 * Copyright (C) 2013 ZTE Ltd.
 * 	by zxp
 *
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/types.h>
#include <linux/suspend.h>

#include <mach/gpio.h>
#include <mach/zx-pm.h>
#include <asm/suspend.h>

/* used to return the value, if 0 represent sleeping process ok,
 * if 1 represent reset and restore back. 
 */
volatile int sleep_ret_flag[MAX_CPU_NUM] = {0};

static pm_wake_reason_t pm_wake_reason;

extern struct zx_pm_main_table zx_pm_main_table;

/**
 * suspend platform device, for example: gpio, uart and so on.
 * 
 */
int zx_board_suspend(void)
{
	//gpio

	debug_uart_suspend();

	
	return 0;
}

/**
 * resume debug uartGPIO and other device out of A9.
 * 
 */
int zx_board_resume(void)
{
	//gpio
	debug_uart_resume();	

	//uart
	
	return 0;
}

/**
 * close clocks and power domains that PCU does not controls.
 * 
 */
int zx_dpm_suspend(void)
{
	
	return 0;
}

/**
 * resume debug uartGPIO and other device out of A9.
 * 
 */
int zx_dpm_resume(void)
{
	
	return 0;
}

/**
 * set cpu power state before do wfi.
 * 
 * cpu_context.power_state should filled in before call this function.
 */
static void set_power_state(void)
{
	set_status_a9_scu(zx_pm_main_table.cur_cpu, 
					  zx_pm_main_table.cpu_context[zx_pm_main_table.cur_cpu]->power_state,
					  zx_pm_main_table.scu_address);	
}

#ifdef  CONFIG_ZX_PM_DEBUG_TIME
unsigned int suspend_start_time;
unsigned int suspend_enter_time;
unsigned int suspend_exit_time;
unsigned int suspend_finish_time;
unsigned int suspend_cur_time;
unsigned int zx_get_cur_time(void)
{
	return ioread32(ZX29_TIMER1_VA+0x18);
}
#endif
static void zx_sleep_before_wfi(void)
{
	pm_switch_clk_to_26m();	
}

static void zx_sleep_after_wfi(void)
{
	pm_switch_clk_from_26m();	
}

/**
 * before power off the cpu, need clean the cache first.
 * 
 */
static inline void pm_clean_cache(void)
{
	u32  power_state = zx_pm_main_table.cluster_context->power_state;

	/* deal l1&l2 cache */
	disable_clean_inv_dcache_v7_l1();

	if(0==zx_pm_main_table.cur_cpu)
	{
		if(CPU_POWER_MODE_SHUTDOWN==power_state)
		{
			/* clean & invalidate L2 */
			clean_pl310(zx_pm_main_table.l2_address);
		}
		else if(CPU_POWER_MODE_DORMANT==power_state)
		{
			/* clean L2 */
			clean_pl310(zx_pm_main_table.l2_address);
		}
	}
}

/**
 * when return pointor is stored,
 * cpu will poweroff now.
 */
static int zx_finish_suspend(unsigned long param)
{

	pm_clean_cache();
	
#ifdef  CONFIG_ZX_PM_DEBUG_TIME
	pm_write_reg(ZX29_IRAM6_VA+1016, zx_get_cur_time());
#endif

	exit_coherency();

	set_power_state();

	zx_sleep_before_wfi();

	zx_jump_addr((unsigned long)SLEEP_CODE_ADDR);

#ifdef  CONFIG_ZX_PM_DEBUG_TIME
	suspend_enter_time = pm_read_reg(ZX29_IRAM6_VA+1016);
	suspend_exit_time = zx_get_cur_time();
#endif

	zx_sleep_after_wfi();

	/* when the sleep is abnormal exit, can run here */
	return 1;
}

/**
 * when cpu sleep procedure is abort,
 * then return true.
 */
static inline bool is_cpu_sleep_abort(void)
{
    unsigned int cpu_id;
    struct zx_cpu_context *context;
	
    cpu_id 	= read_cpuid();
	context	= zx_pm_main_table.cpu_context[cpu_id];

	if (context->power_state != CPU_POWER_MODE_RUN)
		return true;

	return false;
}

/**
 * cpu sleep stage may be abort for some interrupt or event pending.
 * This function is used to deal this situation.
 * 
 * if state is CPU_POWER_MODE_RUN, indicated not back from restore,
 * so sleep stage is abort.
 */
static void cpu_check_sleep_abort(void)
{
    if (is_cpu_sleep_abort())
    {
		/*only enable l1-cache, for l2&scu not power down,
		  then code can run well */
    	enable_cache();
	
		zx_pm_restore_abort_context();

		/* clear scu power state */
		set_power_state();
		
		pm_set_wakeup_reason(WR_WAKE_SRC_ABNORMAL);

		zx_trace_flag2(0x66aa);
    }
	else
	{
	}
}


/**
 * cpu enter&resume interface code.
 * 
 * sleep_type -- CPU_SLEEP_TYPE_LP1/CPU_SLEEP_TYPE_IDLE_LP2
 */
#ifdef	CONFIG_PM_SLEEP
void zx_enter_sleep(cpu_sleep_type_t sleep_type)
{
#ifdef  CONFIG_ZX_PM_DEBUG_TIME
	suspend_start_time = zx_get_cur_time();
#endif

	zx_set_context_level(sleep_type);

	/* set&enable PCU for interrupt/clock/powerdomain/pll/iram */
	zx_set_pcu();

	if(CPU_SLEEP_TYPE_LP3 == sleep_type)
	{
		do_wfi();
	}
	else
	{
/*=================================================================
 *=== the following code is for dormant or shutdown
 *=================================================================
 */
		zx_pm_save_context();

		if(!cpu_suspend(0, zx_finish_suspend))
		{
#ifdef  CONFIG_ZX_PM_DEBUG_TIME
			suspend_enter_time = pm_read_reg(ZX29_IRAM6_VA+1016);
			suspend_exit_time = zx_get_cur_time();
#endif			
			zx_sleep_after_wfi();
			zx_pm_restore_context();
		}
		
		/*exit from sleep*/
	    join_coherency();
		
	    cpu_check_sleep_abort();
/*=================================================================
 *=== ending with dormant or shutdown
 *=================================================================
 */
	}

	/* get cause of exiting sleep */
	pm_get_wake_cause();
	
	zx_clear_pcu();

#ifdef  CONFIG_ZX_PM_DEBUG_TIME
	suspend_finish_time = zx_get_cur_time();
//	pr_info("[SLP] suspend time: start:%d enter:%d total:%d\n",suspend_start_time, suspend_enter_time, suspend_start_time-suspend_enter_time);
//	pr_info("[SLP] suspend time: exit:%d finish:%d total:%d \n",suspend_exit_time, suspend_finish_time, suspend_exit_time-suspend_finish_time);	
#endif	
}
#else
void zx_enter_sleep(cpu_sleep_type_t sleep_type){}
#endif

/**
 * get wakeup reason .
 * 
 * 
 */
pm_wake_reason_t pm_get_wakeup_reason(void)
{
	return pm_wake_reason;
}

/**
 * set wakeup reason .
 * 
 * 
 */
void pm_set_wakeup_reason(pm_wake_reason_t reason)
{
	pm_wake_reason = reason;
}
