/*
 * arch/arm/mach-zx297510/zx297510-clock.c
 *
 *  Copyright (C) 2013 ZTE-TSP
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 */

#include <linux/module.h>

#include <asm/irq.h>
#include <asm/mach/arch.h>
#include <asm/mach/map.h>

#include <mach/clock.h>

#define ZX297510_A1_CRM (0)
#define ZX297510_LSP_CRM (1)
#define ZX297510_SOC_CRM (2)
#define ZX297510_TOP_CRM (3)

#define ZX29_A1CRM_VA	   (ZX297510_A1CRM_BASE - ZX297510_A1_BASE + ZX29_A1_VA)
#define ZX29_SOCCRM_VA	   (ZX297510_SOCCRM_BASE - ZX297510_A1_BASE + ZX29_A1_VA)
#define Zx29_LSPCPRM_VA	   (ZX297510_LSPCPRM_BASE - ZX297510_A2LSP_BASE + ZX29_A2LSP_VA) 
#define Zx29_TOPCPRM_VA	   (ZX29_TOP_VA) 

#define ZX29_PLL_UFI_BASE  (ZX29_TOP_VA + 0x10)

#define DEFINE_ZX29_CLK(hwclk,clkops,num,names) \
    struct clk * parents_##hwclk[num]; \
    static struct clk hwclk##_clk = { \
    .name = #hwclk "_clk", \
    .hw = &hwclk.hw, \
    .ops = &clkops, \
    .num_parents = num, \
    .parent_names = names, \
    .parents = parents_##hwclk, \
    .flags = CLK_AUTO_ROUND_PARENT, \
}

struct zx29_reg_conf {
    u8 reg_class;
    u8 reg_offset;
    u8 reg_bit_offset;
    u8 reg_bit_size;
};

struct zx29_hwclk {
    struct clk_hw   hw;
    struct zx29_reg_conf     clk_en_reg;
    struct zx29_reg_conf     clk_sel_reg;
    struct zx29_reg_conf     clk_div_reg;
    struct zx29_reg_conf     clk_gate_reg;
};

#define to_zx29_hwclk(_hw) container_of(_hw, struct zx29_hwclk, hw)

static void __iomem * zx297510_clk_get_regbase(u8 reg_class)
{
	void __iomem *regbase = NULL;
    
    switch(reg_class){
		case ZX297510_A1_CRM: regbase=ZX29_A1CRM_VA;break;
		case ZX297510_SOC_CRM: regbase=ZX29_SOCCRM_VA;break;
		case ZX297510_LSP_CRM: regbase=Zx29_LSPCPRM_VA;break;
		case ZX297510_TOP_CRM: regbase=Zx29_TOPCPRM_VA;break;
		default: 
            return NULL;
    }
    return regbase;
}

static int zx297510_clk_enable(struct clk_hw *hw)
{
	unsigned int data = 0;
	void __iomem *regbase = NULL;
	void __iomem *regaddr = NULL;
    struct zx29_hwclk *zx29clk = to_zx29_hwclk(hw);
    
    if(zx29clk->clk_en_reg.reg_bit_size == 0)
        return 0;

    regbase = zx297510_clk_get_regbase(zx29clk->clk_en_reg.reg_class);
    regaddr = regbase+zx29clk->clk_en_reg.reg_offset*4;
    
	data=ioread32(regaddr);
	data |= 1<<zx29clk->clk_en_reg.reg_bit_offset;
	iowrite32(data,regaddr);

    return 0;
}

static void zx297510_clk_disable(struct clk_hw *hw)
{
	unsigned int data = 0;
	void __iomem *regbase = NULL;
	void __iomem *regaddr = NULL;
    struct zx29_hwclk *zx29clk = to_zx29_hwclk(hw);

    if(zx29clk->clk_en_reg.reg_bit_size == 0)
        return ;
    
    regbase = zx297510_clk_get_regbase(zx29clk->clk_en_reg.reg_class);
    regaddr = regbase+zx29clk->clk_en_reg.reg_offset*4;
    
	data=ioread32(regaddr);
	data &= ~(1<<zx29clk->clk_en_reg.reg_bit_offset);
	iowrite32(data,regaddr);

    return ;
}

static int zx297510_is_enabled(struct clk_hw *hw)
{
	unsigned int data = 0;
	void __iomem *regbase = NULL;
	void __iomem *regaddr = NULL;
    struct zx29_hwclk *zx29clk = to_zx29_hwclk(hw);

    if(zx29clk->clk_en_reg.reg_bit_size == 0)
        return 1;
    
    regbase = zx297510_clk_get_regbase(zx29clk->clk_en_reg.reg_class);
    regaddr = regbase+zx29clk->clk_en_reg.reg_offset*4;
    
	data=ioread32(regaddr);
	data>>=zx29clk->clk_en_reg.reg_bit_offset;

    if(data & 0x1)
        return 1;
    else
        return 0;
}

static int zx297510_clk_set_rate(struct clk_hw *hw, unsigned long new_rate)
{
    unsigned long parent_rate = 0;
    unsigned long clk_div = 0;
 	void __iomem *regbase = NULL;
	void __iomem *regaddr = NULL;
	unsigned int data = 0;
    struct zx29_hwclk *zx29clk = to_zx29_hwclk(hw);
   
    if(zx29clk->hw.clk == NULL || zx29clk->hw.clk->parent == NULL)
        return -EINVAL;
    if(zx29clk->clk_div_reg.reg_bit_size == 0)
        return -EINVAL;

    parent_rate = zx29clk->hw.clk->parent->rate;
    clk_div = parent_rate/new_rate;
    if(clk_div >= (1<<zx29clk->clk_div_reg.reg_bit_size)) {
        clk_div = (1<<zx29clk->clk_div_reg.reg_bit_size);
    }
    
    if(clk_div) {
        clk_div--;
    }
    
    regbase = zx297510_clk_get_regbase(zx29clk->clk_div_reg.reg_class);
    regaddr = regbase+zx29clk->clk_div_reg.reg_offset*4;
    clk_div &= ((1<<(zx29clk->clk_div_reg.reg_bit_size))-1);/*clear invalid data*/
   
	data=ioread32(regaddr);
	data &= ~(((1<<(zx29clk->clk_div_reg.reg_bit_size))-1)<<zx29clk->clk_div_reg.reg_bit_offset);
    data |= clk_div<<zx29clk->clk_div_reg.reg_bit_offset;
    iowrite32(data,regaddr);

    return 0;
}

static int zx297510_clk_set_rate_x2(struct clk_hw *hw, unsigned long new_rate)
{
    unsigned long parent_rate = 0;
    unsigned long clk_div = 0;
 	void __iomem *regbase = NULL;
	void __iomem *regaddr = NULL;
	unsigned int data = 0;
    struct zx29_hwclk *zx29clk = to_zx29_hwclk(hw);
    unsigned long i = 0;
   
    if(zx29clk->hw.clk == NULL || zx29clk->hw.clk->parent == NULL)
        return -EINVAL;
    if(zx29clk->clk_div_reg.reg_bit_size == 0)
        return -EINVAL;

    parent_rate = zx29clk->hw.clk->parent->rate;
    clk_div = parent_rate/new_rate;
    for (i = 0; i < (1<<zx29clk->clk_div_reg.reg_bit_size); i++) {
        if (clk_div <= (0x1<<i))
            break;
    }

    if (i >= (1<<zx29clk->clk_div_reg.reg_bit_size))
        i = (1<<zx29clk->clk_div_reg.reg_bit_size) - 1;
    
    regbase = zx297510_clk_get_regbase(zx29clk->clk_div_reg.reg_class);
    regaddr = regbase+zx29clk->clk_div_reg.reg_offset*4;
   
	data=ioread32(regaddr);
	data &= ~(((1<<(zx29clk->clk_div_reg.reg_bit_size))-1)<<zx29clk->clk_div_reg.reg_bit_offset);
    data |= i<<zx29clk->clk_div_reg.reg_bit_offset;
    iowrite32(data,regaddr);

    return 0;
}

static unsigned long zx297510_clk_recalc_rate(struct clk_hw *hw,
					unsigned long parent_rate)
{
 	void __iomem *regbase = NULL;
	void __iomem *regaddr = NULL;
	unsigned int data = 0;
    struct zx29_hwclk *zx29clk = to_zx29_hwclk(hw);
    
    if(zx29clk->clk_div_reg.reg_bit_size == 0) {
        if(zx29clk->hw.clk->num_parents)
            return parent_rate;
        else
            return 0;
    }
    
    regbase = zx297510_clk_get_regbase(zx29clk->clk_div_reg.reg_class);
    regaddr = regbase+zx29clk->clk_div_reg.reg_offset*4;    
	data=ioread32(regaddr);
    data >>= zx29clk->clk_div_reg.reg_bit_offset;
	data &= ((1<<(zx29clk->clk_div_reg.reg_bit_size))-1);
    
    return parent_rate/(data+1);
}

static unsigned long zx297510_clk_recalc_rate_x2(struct clk_hw *hw,
					unsigned long parent_rate)
{
 	void __iomem *regbase = NULL;
	void __iomem *regaddr = NULL;
	unsigned int data = 0;
    struct zx29_hwclk *zx29clk = to_zx29_hwclk(hw);
    
    if(zx29clk->clk_div_reg.reg_bit_size == 0) {
        if(zx29clk->hw.clk->num_parents)
            return parent_rate;
        else
            return 0;
    }
    
    regbase = zx297510_clk_get_regbase(zx29clk->clk_div_reg.reg_class);
    regaddr = regbase+zx29clk->clk_div_reg.reg_offset*4;    
	data=ioread32(regaddr);
    data >>= zx29clk->clk_div_reg.reg_bit_offset;
	data &= ((1<<(zx29clk->clk_div_reg.reg_bit_size))-1);
    
    return parent_rate/(0x1<<data);
}


static int zx297510_clk_set_parent(struct clk_hw *hw, u8 index)
{
 	void __iomem *regbase = NULL;
	void __iomem *regaddr = NULL;
	unsigned int data = 0;
    struct zx29_hwclk *zx29clk = to_zx29_hwclk(hw);
    
    if(zx29clk->hw.clk->num_parents==0 || index>=zx29clk->hw.clk->num_parents)
        return -EINVAL;
    if(zx29clk->clk_sel_reg.reg_bit_size == 0)
        return 0; /*default index 0 as parent*/
    
    regbase = zx297510_clk_get_regbase(zx29clk->clk_sel_reg.reg_class);
    regaddr = regbase+zx29clk->clk_sel_reg.reg_offset*4;
    index &= ((1<<(zx29clk->clk_sel_reg.reg_bit_size))-1);/*clear invalid data*/
    
	data=ioread32(regaddr);
	data &= ~(((1<<(zx29clk->clk_sel_reg.reg_bit_size))-1)<<zx29clk->clk_sel_reg.reg_bit_offset);
    data |= index<<zx29clk->clk_sel_reg.reg_bit_offset;
    iowrite32(data,regaddr);

    return 0;
}

static long zx297510_clk_round_rate(struct clk_hw *hw, unsigned long rate,
					unsigned long * best_parent_ratee)
{
    unsigned long parent_rate = 0;
    unsigned long clk_div = 0;
    struct zx29_hwclk *zx29clk = to_zx29_hwclk(hw);
   
    if(zx29clk->hw.clk == NULL || zx29clk->hw.clk->parent == NULL)
        return 0;
    if(zx29clk->clk_div_reg.reg_bit_size == 0)
        return 0;

    parent_rate = zx29clk->hw.clk->parent->rate;
    clk_div = parent_rate/rate;
    if(clk_div >= (1<<zx29clk->clk_div_reg.reg_bit_size)) 
    {
        clk_div = (1<<zx29clk->clk_div_reg.reg_bit_size);
    }
    if(!clk_div)
    {
        return parent_rate;
    }

    return parent_rate/(clk_div);
}

static long zx297510_clk_round_rate_x2(struct clk_hw *hw, unsigned long rate,
					unsigned long * best_parent_ratee)
{
    unsigned long parent_rate = 0;
    unsigned long clk_div = 0;
    struct zx29_hwclk *zx29clk = to_zx29_hwclk(hw);
    unsigned long i = 0;
   
    if(zx29clk->hw.clk == NULL || zx29clk->hw.clk->parent == NULL)
        return 0;
    if(zx29clk->clk_div_reg.reg_bit_size == 0)
        return 0;

    parent_rate = zx29clk->hw.clk->parent->rate;
    clk_div = parent_rate/rate;
    for (i = 0; i < (1<<zx29clk->clk_div_reg.reg_bit_size); i++) {
        if (clk_div <= (0x1<<i))
            break;
    }

    if (i >= (1<<zx29clk->clk_div_reg.reg_bit_size))
        i = (1<<zx29clk->clk_div_reg.reg_bit_size) - 1;
    
    return parent_rate/(0x1<<i);
}


static int zx297510_set_auto_gate(struct clk_hw *hw, bool enable)
{
	unsigned int data = 0;
	void __iomem *regbase = NULL;
	void __iomem *regaddr = NULL;
    struct zx29_hwclk *zx29clk = to_zx29_hwclk(hw);

    if(zx29clk->clk_gate_reg.reg_bit_size == 0)
        return -EINVAL;
    
    regbase = zx297510_clk_get_regbase(zx29clk->clk_gate_reg.reg_class);
    regaddr = regbase+zx29clk->clk_gate_reg.reg_offset*4;
    
	data=ioread32(regaddr);
    if(enable)
	    data |= 1<<zx29clk->clk_gate_reg.reg_bit_offset;
    else
	    data &= ~(1<<zx29clk->clk_gate_reg.reg_bit_offset);
	iowrite32(data,regaddr);

    return 0;
}
static u8 zx297510_get_parent(struct clk_hw *hw)
{
	unsigned int data = 0;
	void __iomem *regbase = NULL;
	void __iomem *regaddr = NULL;
    struct zx29_hwclk *zx29clk = to_zx29_hwclk(hw);

    if(zx29clk->hw.clk->num_parents < 2 || zx29clk->clk_sel_reg.reg_bit_size ==0)
        return 0;
    
    regbase = zx297510_clk_get_regbase(zx29clk->clk_sel_reg.reg_class);
    regaddr = regbase+zx29clk->clk_sel_reg.reg_offset*4;
    
	data = ioread32(regaddr);
    data >>= zx29clk->clk_sel_reg.reg_bit_offset;
    data &= (1<<(zx29clk->clk_sel_reg.reg_bit_size))-1;
    
    return data;
}
static struct clk_ops peripheral_clk_ops = {
    .enable = zx297510_clk_enable,
    .disable = zx297510_clk_disable,
    .set_auto_gate = zx297510_set_auto_gate,
    .set_rate = zx297510_clk_set_rate,
    .recalc_rate = zx297510_clk_recalc_rate,
    .set_parent = zx297510_clk_set_parent,
    .round_rate = zx297510_clk_round_rate,
    .get_parent = zx297510_get_parent,
    .is_enabled = zx297510_is_enabled,
};

static struct clk_ops cpu_clk_ops = {
    .enable = zx297510_clk_enable,
    .disable = zx297510_clk_disable,
    .set_auto_gate = zx297510_set_auto_gate,
    .set_rate = zx297510_clk_set_rate_x2,
    .recalc_rate = zx297510_clk_recalc_rate_x2,
    .set_parent = zx297510_clk_set_parent,
    .round_rate = zx297510_clk_round_rate_x2,
    .get_parent = zx297510_get_parent,
    .is_enabled = zx297510_is_enabled,
};

static int clk_rootclk_enable(struct clk_hw *hw)
{
    return 0;
}
static void clk_rootclk_disable(struct clk_hw *hw)
{
    return ;
}

static int clk_rootclk_is_enabled(struct clk_hw *hw)
{
    return 1;
}

static int clk_rootclk_set_rate(struct clk_hw *hw, unsigned long new_rate)
{
    return 0;
}

static unsigned long clk_axi_recalc_rate(struct clk_hw *hw,
					unsigned long parent_rate)
{
	void __iomem *regaddr = NULL;
	unsigned int data = 0;
	unsigned int axi_rate = 0;

    regaddr = (ZX29_TOP_VA+0x54);
	data = ioread32(regaddr);
    data &= 0x3;
    switch (data) {
        case 0: axi_rate = 104000000;break;
        case 1: axi_rate = 26000000;break;
        case 2: axi_rate = 122880000;break;
        case 3: axi_rate = 156000000;break;
        default:
            break;      
    }
    
    regaddr = (ZX29_TOP_VA+0x7c);
	data = ioread32(regaddr);
    data &= 0x3;
    return axi_rate/(0x1<<data);
}

void clk_lsp_com_work_clk_init(struct clk_hw *hw)
{
	void __iomem *regaddr = NULL;
	unsigned int data = 0;

    regaddr = (ZX29_SOCCRM_VA+0x04);
	data = ioread32(regaddr);
    data |= 0x1<<2; /*lsp sel work clk 96m*/
    iowrite32(data,regaddr);;

    regaddr = (Zx29_LSPCPRM_VA);
	data = ioread32(regaddr);
    data &= ~0x1; /*lsp comm work clk sel 96m*/
    iowrite32(data,regaddr);;
    
    return ;
}

static unsigned long clk_pll_recalc_rate(void __iomem * pll_reg_base, unsigned int pll_fref)
{
	unsigned int pll_config1 = 0;
	unsigned int pll_config2 = 0;
    unsigned int pll_refdiv = 0;
    unsigned int pll_fbdiv = 0;
    unsigned int pll_frac = 0;
    unsigned int pll_postdiv1 = 0;
    unsigned int pll_postdiv2 = 0;
    unsigned int foutpostdiv = 0;
    
	pll_config1 = ioread32(pll_reg_base);
	pll_config2 = ioread32(pll_reg_base+0x4);

    pll_refdiv = (pll_config1>>18)&((0x1<<6)-1);
    pll_fbdiv = (pll_config1>>6)&((0x1<<12)-1);
    pll_frac = (pll_config2)&((0x1<<24)-1);
    pll_postdiv1 = (pll_config1>>3)&((0x1<<3)-1);
    pll_postdiv2 = (pll_config1)&((0x1<<3)-1);

    foutpostdiv = (pll_fref/pll_refdiv)*(pll_fbdiv+pll_frac);
    foutpostdiv = foutpostdiv/pll_postdiv1/pll_postdiv2;

    return foutpostdiv;  
}

static unsigned long clk_pll_ps_recalc_rate(struct clk_hw *hw,
					unsigned long parent_rate)
{    
    return clk_pll_recalc_rate(ZX29_TOP_VA + 0x18, 26000000)/2;
}

static unsigned long clk_pll_ufi_recalc_rate(struct clk_hw *hw,
					unsigned long parent_rate)
{
    return clk_pll_recalc_rate(ZX29_PLL_UFI_BASE, 26000000);
}

static int clk_pll_ufi_enable(struct clk_hw *hw)
{
	unsigned int data = 0;

    data = ioread32(ZX29_PLL_UFI_BASE);
    data &= ~(0x1<<31); /*pll enable*/
    iowrite32(data, ZX29_PLL_UFI_BASE);
    /*wait pll lock*/
    while(!(ioread32(ZX29_PLL_UFI_BASE) & (0x1<<30)));
    return 0;
}
static void clk_pll_ufi_disable(struct clk_hw *hw)
{
	unsigned int data = 0;

    data = ioread32(ZX29_PLL_UFI_BASE);
    data |= (0x1<<31); /*pll disable*/
    data &= ~(0x1<<30);
    iowrite32(data, ZX29_PLL_UFI_BASE);
    return ;
}
static int clk_pll_ufi_is_enabled(struct clk_hw *hw)
{
    if (ioread32(ZX29_PLL_UFI_BASE) & (0x1<<31))
        return 0;
    else
        return 1;
}


struct clk_ops root_clk_ops = {
    .enable = clk_rootclk_enable,
	.disable = clk_rootclk_disable,
	.is_enabled = clk_rootclk_is_enabled,
	.set_rate = clk_rootclk_set_rate,
};

struct clk_ops axi_clk_ops = {
    .enable = clk_rootclk_enable,
	.disable = clk_rootclk_disable,
	.is_enabled = clk_rootclk_is_enabled,
	.set_rate = clk_rootclk_set_rate,
	.recalc_rate = clk_axi_recalc_rate,
};

struct clk_ops lsp_com_clk_ops = {
    .enable = clk_rootclk_enable,
	.disable = clk_rootclk_disable,
	.is_enabled = clk_rootclk_is_enabled,
	.set_rate = clk_rootclk_set_rate,
	.init = clk_lsp_com_work_clk_init,
};

struct clk_ops pll_ps_clk_ops = {
    .enable = clk_rootclk_enable,
	.disable = clk_rootclk_disable,
	.is_enabled = clk_rootclk_is_enabled,
	.set_rate = clk_rootclk_set_rate,
	.recalc_rate = clk_pll_ps_recalc_rate,
};

struct clk_ops pll_ufi_clk_ops = {
    .enable = clk_pll_ufi_enable,
	.disable = clk_pll_ufi_disable,
	.is_enabled = clk_pll_ufi_is_enabled,
	.set_rate = clk_rootclk_set_rate,
	.recalc_rate = clk_pll_ufi_recalc_rate,
};

/*
*   root clk
*/
static struct clk work_clk_104m = {
    .name = NAME_work_clk_104m,
	.rate = 104*1000*1000,
	.ops = &root_clk_ops,
	.flags = CLK_IS_ROOT,
};
static struct clk main_clk_26m = {
    .name = NAME_main_clk_26m,
	.rate = 26*1000*1000,
	.ops = &root_clk_ops,
	.flags = CLK_IS_ROOT,
};
static struct clk standby_clk_32k = {
    .name = NAME_standby_clk_32k,
	.rate = 32768,
	.ops = &root_clk_ops,
	.flags = CLK_IS_ROOT,
};
static struct clk lsp_com_work_clk = {
    .name = "lsp_com_work_clk",
	.rate = 96*1000*1000,
	.ops = &lsp_com_clk_ops,
	.flags = CLK_IS_ROOT,
};
static struct clk pll_ps_clk = {
    .name = "pll_ps_clk",
	.rate = 0,
	.ops = &pll_ps_clk_ops,
	.flags = CLK_IS_ROOT,
};
static struct clk pll_ufi_clk = {
    .name = "pll_ufi_clk",
	.rate = 0,
	.ops = &pll_ufi_clk_ops,
	.flags = CLK_IS_ROOT | CLK_IGNORE_UNUSED,
};
static struct clk pll_624_clk = {
    .name = "pll_624_clk",
	.rate = 624000000,
	.ops = &root_clk_ops,
	.flags = CLK_IS_ROOT,
};

/*virtual*/
static struct clk apb_clk = {
    .name = "apb_clk",
	.rate = 0,
	.ops = &root_clk_ops,
	.flags = CLK_IS_ROOT,
};
char*  virtual_pclk_parents[1] = {NAME_apb_clk};

static struct clk axi_clk = {
    .name = "axi_clk",
	.rate = 0,
	.ops = &axi_clk_ops,
	.flags = CLK_IS_ROOT,
};

char*  matrix_ahb_clk_parents[1] = {"axi_clk"};
static struct zx29_hwclk matrix_ahb = {
	.clk_en_reg={ZX297510_SOC_CRM,0,0,0},
	.clk_sel_reg={ZX297510_SOC_CRM,0,0,0},
	.clk_div_reg={ZX297510_SOC_CRM,0,8,3},
};
DEFINE_ZX29_CLK(matrix_ahb,peripheral_clk_ops,1,matrix_ahb_clk_parents);

char*  matrix_apb_clk_parents[1] = {"matrix_ahb_clk"};
static struct zx29_hwclk matrix_apb = {
	.clk_en_reg={ZX297510_SOC_CRM,0,0,0},
	.clk_sel_reg={ZX297510_SOC_CRM,0,0,0},
	.clk_div_reg={ZX297510_SOC_CRM,0,16,3},
};
DEFINE_ZX29_CLK(matrix_apb,peripheral_clk_ops,1,matrix_apb_clk_parents);

char*  lsp_apb_clk_parents[1] = {"matrix_ahb_clk"};
static struct zx29_hwclk lsp_apb = {
	.clk_en_reg={ZX297510_LSP_CRM,0,0,0},
	.clk_sel_reg={ZX297510_LSP_CRM,0,0,0},
	.clk_div_reg={ZX297510_LSP_CRM,8,0,4},
};
DEFINE_ZX29_CLK(lsp_apb,peripheral_clk_ops,1,lsp_apb_clk_parents);

/*
*   a1_uart0
*/
static struct zx29_hwclk uart0_apb = {
	.clk_en_reg={ZX297510_A1_CRM,3,0,1},
	.clk_sel_reg={ZX297510_A1_CRM,0,0,0},
	.clk_div_reg={ZX297510_A1_CRM,0,0,0},
};
DEFINE_ZX29_CLK(uart0_apb,peripheral_clk_ops,1,virtual_pclk_parents);

char*  uart_wclk_parents[1] = {NAME_work_clk_104m};
static struct zx29_hwclk uart0_work = {
	.clk_en_reg={ZX297510_A1_CRM,4,0,1},
	.clk_sel_reg={ZX297510_A1_CRM,0,0,0},
	.clk_div_reg={ZX297510_A1_CRM,0,0,0},
};
DEFINE_ZX29_CLK(uart0_work,peripheral_clk_ops,1,uart_wclk_parents);
/*
*   a1_timer3
*/
static struct zx29_hwclk timer3_apb = {
	.clk_en_reg={ZX297510_A1_CRM,3,7,1},
	.clk_sel_reg={ZX297510_A1_CRM,0,0,0},
	.clk_div_reg={ZX297510_A1_CRM,0,0,0},
};
DEFINE_ZX29_CLK(timer3_apb,peripheral_clk_ops,1,virtual_pclk_parents);

char*  timer_A1_wclk_parents[2] = {NAME_work_clk_104m,NAME_standby_clk_32k};
static struct zx29_hwclk timer3_work = {
	.clk_en_reg={ZX297510_A1_CRM,4,7,1},
	.clk_sel_reg={ZX297510_A1_CRM,0,15,1},
	.clk_div_reg={ZX297510_A1_CRM,0,8,4},
};
DEFINE_ZX29_CLK(timer3_work,peripheral_clk_ops,2,timer_A1_wclk_parents);

/*
*   lsp_timer1
*/
static struct zx29_hwclk timer1_apb = {
	.clk_en_reg={ZX297510_LSP_CRM,0,0,0},
	.clk_sel_reg={ZX297510_LSP_CRM,0,0,0},
	.clk_div_reg={ZX297510_LSP_CRM,0,0,0},
};
DEFINE_ZX29_CLK(timer1_apb,peripheral_clk_ops,1,virtual_pclk_parents);

char*  timer_LSP_wclk_parents[2] = {NAME_main_clk_26m,NAME_standby_clk_32k};
static struct zx29_hwclk timer1_work = {
	.clk_en_reg={ZX297510_LSP_CRM,11,14,1},
	.clk_sel_reg={ZX297510_LSP_CRM,0,5,1},
	.clk_div_reg={ZX297510_LSP_CRM,9,20,4},
};
DEFINE_ZX29_CLK(timer1_work,peripheral_clk_ops,2,timer_LSP_wclk_parents);
/*
*   ssp
*/
static struct zx29_hwclk ssp0_apb = {
	.clk_en_reg={ZX297510_A1_CRM,3,2,1},
	.clk_sel_reg={ZX297510_A1_CRM,0,0,0},
	.clk_div_reg={ZX297510_A1_CRM,0,0,0},
};
DEFINE_ZX29_CLK(ssp0_apb,peripheral_clk_ops,1,virtual_pclk_parents);

char*  ssp_wclk_parents[1] = {NAME_work_clk_104m};
static struct zx29_hwclk ssp0_work = {
	.clk_en_reg={ZX297510_A1_CRM,4,2,1},
	.clk_sel_reg={ZX297510_A1_CRM,0,0,0},
	.clk_div_reg={ZX297510_A1_CRM,1,0,4},
	.clk_gate_reg={ZX297510_A1_CRM,7,0,1},
};
DEFINE_ZX29_CLK(ssp0_work,peripheral_clk_ops,1,ssp_wclk_parents);

/*
*   rtc
*/
static struct zx29_hwclk rtc_apb = {
	.clk_en_reg={ZX297510_SOC_CRM,3,17,1},
	.clk_sel_reg={ZX297510_SOC_CRM,0,0,0},
	.clk_div_reg={ZX297510_SOC_CRM,0,0,0},
};
DEFINE_ZX29_CLK(rtc_apb,peripheral_clk_ops,1,virtual_pclk_parents);

char*  rtc_wclk_parents[1] = {NAME_standby_clk_32k};
static struct zx29_hwclk rtc_work = {
	.clk_en_reg={ZX297510_SOC_CRM,0,0,0},
	.clk_sel_reg={ZX297510_SOC_CRM,0,0,0},
	.clk_div_reg={ZX297510_SOC_CRM,0,0,0},
};
DEFINE_ZX29_CLK(rtc_work,peripheral_clk_ops,1,rtc_wclk_parents);

/*
*   usb0
*/
static struct zx29_hwclk usb0_ahb = {
	.clk_en_reg={ZX297510_SOC_CRM,3,2,1},
	.clk_sel_reg={ZX297510_SOC_CRM,0,0,0},
	.clk_div_reg={ZX297510_SOC_CRM,0,0,0},
};
DEFINE_ZX29_CLK(usb0_ahb,peripheral_clk_ops,0,NULL);

static struct zx29_hwclk usb0_work = {
	.clk_en_reg={ZX297510_SOC_CRM,4,16,1},
	.clk_sel_reg={ZX297510_SOC_CRM,0,0,0},
	.clk_div_reg={ZX297510_SOC_CRM,0,0,0},
};
DEFINE_ZX29_CLK(usb0_work,peripheral_clk_ops,0,NULL);
/*
*   usb1
*/
#ifdef _USE_USB1
static struct zx29_hwclk usb1_ahb = {
	.clk_en_reg={ZX297510_SOC_CRM,3,28,1},
	.clk_sel_reg={ZX297510_SOC_CRM,0,0,0},
	.clk_div_reg={ZX297510_SOC_CRM,0,0,0},
};
DEFINE_ZX29_CLK(usb1_ahb,peripheral_clk_ops,0,NULL);

static struct zx29_hwclk usb1_work = {
	.clk_en_reg={ZX297510_SOC_CRM,4,17,1},
	.clk_sel_reg={ZX297510_SOC_CRM,0,0,0},
	.clk_div_reg={ZX297510_SOC_CRM,0,0,0},
};
DEFINE_ZX29_CLK(usb1_work,peripheral_clk_ops,0,NULL);
#endif
/*
*   dma0
*/
#if 0
static struct zx29_hwclk dma_hwx = {
	.clk_en_reg={ZX297510_SOC_CRM,3,5,1},
	.clk_sel_reg={ZX297510_SOC_CRM,0,0,0},
	.clk_div_reg={ZX297510_SOC_CRM,0,0,0},
};
DEFINE_ZX29_CLK(dma_hwx,peripheral_clk_ops,0,NULL);

static struct zx29_hwclk dma_apb = {
	.clk_en_reg={ZX297510_SOC_CRM,3,6,1},
	.clk_sel_reg={ZX297510_SOC_CRM,0,0,0},
	.clk_div_reg={ZX297510_SOC_CRM,0,0,0},
};
DEFINE_ZX29_CLK(dma_apb,peripheral_clk_ops,0,NULL);
#endif

/*
*   blg
*/
static struct zx29_hwclk blg_apb = {
	.clk_en_reg={ZX297510_LSP_CRM,11,11,1},
	.clk_sel_reg={ZX297510_LSP_CRM,0,0,0},
	.clk_div_reg={ZX297510_LSP_CRM,0,0,0},
};
DEFINE_ZX29_CLK(blg_apb,peripheral_clk_ops,1,virtual_pclk_parents);

char*  blg_wclk_parents[2] = {NAME_main_clk_26m,NAME_standby_clk_32k};
static struct zx29_hwclk blg_work = {
	.clk_en_reg={ZX297510_LSP_CRM,3,6,1},
	.clk_sel_reg={ZX297510_LSP_CRM,0,3,1},
	.clk_div_reg={ZX297510_LSP_CRM,8,4,10},
};
DEFINE_ZX29_CLK(blg_work,peripheral_clk_ops,2,blg_wclk_parents);

/*
*   sdmmc
*/
static struct zx29_hwclk sdmmc_apb = {
	.clk_en_reg={ZX297510_LSP_CRM,11,7,1},
	.clk_sel_reg={ZX297510_LSP_CRM,0,0,0},
	.clk_div_reg={ZX297510_LSP_CRM,0,0,0},
};
DEFINE_ZX29_CLK(sdmmc_apb,peripheral_clk_ops,1,virtual_pclk_parents);

char*  sdmmc_wclk_parents[1] = {"lsp_com_work_clk"};
static struct zx29_hwclk sdmmc_work = {
	.clk_en_reg={ZX297510_LSP_CRM,0,0,0},
	.clk_sel_reg={ZX297510_LSP_CRM,0,0,0},
	.clk_div_reg={ZX297510_LSP_CRM,9,0,4},
};
DEFINE_ZX29_CLK(sdmmc_work,peripheral_clk_ops,1,sdmmc_wclk_parents);

/*
*   ufi cpu
*/
char*  ufi_clk_parents[4] = {"pll_ufi_clk", NAME_main_clk_26m, "pll_624_clk", "pll_ps_clk"};
static struct zx29_hwclk cpu_work = {
	.clk_en_reg={ZX297510_TOP_CRM,0,0,0},
	.clk_sel_reg={ZX297510_TOP_CRM,17,0,2},
	.clk_div_reg={ZX297510_TOP_CRM,27,0,2},
};
DEFINE_ZX29_CLK(cpu_work,cpu_clk_ops,4,ufi_clk_parents);

struct clk_lookup periph_clocks_lookups[] = {
	CLK_ZX29_CONFIG(NULL, NAME_work_clk_104m, &work_clk_104m),
	CLK_ZX29_CONFIG(NULL, NAME_main_clk_26m, &main_clk_26m),
	CLK_ZX29_CONFIG(NULL, NAME_standby_clk_32k, &standby_clk_32k),
	CLK_ZX29_CONFIG(NULL, NAME_apb_clk, &apb_clk),
	CLK_ZX29_CONFIG(NULL, "axi_clk", &axi_clk),
	CLK_ZX29_CONFIG(NULL, "matrix_ahb_clk", &matrix_ahb_clk),
	CLK_ZX29_CONFIG(NULL, "matrix_apb_clk", &matrix_apb_clk),
	CLK_ZX29_CONFIG(NULL, "lsp_apb_clk", &lsp_apb_clk),
	CLK_ZX29_CONFIG(NULL, "lsp_com_work_clk",  &lsp_com_work_clk),
	CLK_ZX29_CONFIG(NULL, "pll_ps_clk", &pll_ps_clk),
	CLK_ZX29_CONFIG(NULL, "pll_ufi_clk", &pll_ufi_clk),
	CLK_ZX29_CONFIG(NULL, "pll_624_clk", &pll_624_clk),
	CLK_ZX29_CONFIG("zx297510_uart.0", "work_clk", &uart0_work_clk),
	CLK_ZX29_CONFIG("zx297510_uart.0", "apb_clk",  &uart0_apb_clk),
	CLK_ZX29_CONFIG("zx297510_timer1", "work_clk", &timer1_work_clk),
	CLK_ZX29_CONFIG("zx297510_timer1", "apb_clk", &timer1_apb_clk),
	CLK_ZX29_CONFIG("zx297510_timer3", "work_clk", &timer3_work_clk),
	CLK_ZX29_CONFIG("zx297510_timer3", "apb_clk",  &timer3_apb_clk),
	CLK_ZX29_CONFIG("zx297510_ssp.0", "work_clk", &ssp0_work_clk),
	CLK_ZX29_CONFIG("zx297510_ssp.0", "apb_clk",  &ssp0_apb_clk),
	CLK_ZX29_CONFIG("zx297510_rtc.0", "apb_clk", &rtc_apb_clk),
	CLK_ZX29_CONFIG("zx297510_rtc.0", "work_clk", &rtc_work_clk),
	CLK_ZX29_CONFIG("zx297510_usb.0", "ahb_clk",  &usb0_ahb_clk),
	CLK_ZX29_CONFIG("zx297510_usb.0", "work_clk", &usb0_work_clk),
#ifdef _USE_USB1
	CLK_ZX29_CONFIG("zx297510_usb.1", "ahb_clk",  &usb1_ahb_clk),
	CLK_ZX29_CONFIG("zx297510_usb.1", "work_clk", &usb1_work_clk),
#endif
	CLK_ZX29_CONFIG("zx297510_blg.0", "apb_clk",  &blg_apb_clk),
	CLK_ZX29_CONFIG("zx297510_blg.0", "work_clk",  &blg_work_clk),
	CLK_ZX29_CONFIG("zx297510_sd.0", "apb_clk",  &sdmmc_apb_clk),
	CLK_ZX29_CONFIG("zx297510_sd.0", "work_clk",  &sdmmc_work_clk),
	//CLK_ZX29_CONFIG("zx297510_dma", "apb_clk",  &dma_hwx_clk),
	//CLK_ZX29_CONFIG("zx297510_dma", "axi_clk",  &dma_apb_clk),
	CLK_ZX29_CONFIG(NULL, "cpu_clk",  &cpu_work_clk),
};

unsigned int periph_clocks_lookups_num=ARRAY_SIZE(periph_clocks_lookups);

