/*
 * arch/arm/mach-zx297520v2/include/mach/i2c_private.h
 *
 * Copyright (C) 2015 ZTE-TSP.
 *
 * This file can only include by i2c-zx29.c.
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

#ifndef _I2C_PRIVATE_H
#define _I2C_PRIVATE_H

#include <mach/board.h>
#include <mach/i2c.h>
#include <linux/wakelock.h>
#include <linux/semaphore.h>

/*I2C register offset address */
#define I2C_VERSION       		0x00
#define I2C_CMD           		0x04
#define I2C_MST_CODE          	0x08
#define I2C_DEVICE_H          	0x0c
#define I2C_DEVICE_L          	0x10
#define I2C_CLK_DIV_FS        	0x14
#define I2C_CLK_DIV_HS        	0x18
#define I2C_TXF_CTRL         	0x1c
#define I2C_RXF_CTRL    		0x20
#define I2C_DATA    			0x24
#define I2C_STATUS        		0x28
#define I2C_TXF_STATUS  		0x2c
#define I2C_RXF_STATUS  		0x30

/*I2C cmd bit mask*/
#define CMD_I2C_MODE  	 		0x01
#define CMD_TENBIT_MODE  		0x02
#define CMD_SPEED_MODE   		0x04
#define CMD_IRQ_MASK     		0x08
#define CMD_I2C_RW       		0x10
#define CMD_I2C_RW_FORMAT		0x20
#define CMD_I2C_START 	 		0x40

#define CMD_I2C_FMT_WR			0
#define CMD_I2C_FMT_RD			CMD_I2C_RW
#define CMD_I2C_FMT_CMB			CMD_I2C_RW_FORMAT
#define CMD_I2C_FMT_MSK			(CMD_I2C_RW_FORMAT | CMD_I2C_RW)

/*I2C TXF CTRL bit  mask*/
#define WFIFO_RST   			0x80
#define WFIFO_SIZE  			0x1f

/*I2C RXF CTRL bit mask*/
#define RFIFO_RST   			0x80
#define RFIFO_SIZE  			0x1f

/*I2C status mask*/
#define IRQ_ACK   				0x80
#define I2C_STAT_BUSY 			0x40
#define IRQ_RXF_FULL     		0x20
#define IRQ_TXF_EMPTY    		0x10
#define IRQ_TIME_OUT    		0x08
#define IRQ_ERR_DATA 			0x04
#define IRQ_ERR_DEVICE 			0x02
#define IRQ_TRANS_DONE 			0x01
#define IRQ_NOFINISH_IDLE  		0x00
#define IRQ_STAT_MASK  			0x3f
/*I2C TXF status mask*/
#define WFIFO_FULL     			0x01
#define WFIFO_EMPTY    			0x02
#define WFIFO_DATA     			0xfc

/*I2C RXF status mask*/
#define RFIFO_FULL     			0x01
#define RFIFO_EMPTY    			0x02
#define RFIFO_DATA     			0xfc

#define	I2C0_SCL_PIN			ZX29_GPIO_46
#define	I2C0_SDA_PIN			ZX29_GPIO_47
#define	I2C1_SCL_PIN			ZX29_GPIO_48
#define	I2C1_SDA_PIN			ZX29_GPIO_49

/* i2c controller state */
enum zx29_i2c_state {
	STATE_IDLE,
	STATE_START,
	STATE_READ,
	STATE_WRITE,
	STATE_STOP,
};

struct zx29_i2c {
	char					name[16];
	unsigned int 			id;
	spinlock_t		       	lock;
	struct semaphore		msg_complete;
	unsigned int		   	suspended;
    struct wake_lock        psm_lock;

    struct hrtimer          hr_timer;
    ktime_t                 ktime;

	struct i2c_msg		   	*msg;
//	unsigned int		   	msg_num;
//	unsigned int		   	msg_idx;
	unsigned int		   	msg_ptr;
	unsigned int		   	buf_remaining;

	unsigned int		   	tx_setup;
	unsigned int		   	irq;
	unsigned int		   	gpio;

	enum zx29_i2c_state		state;
	unsigned long		    clkrate;
	unsigned int		   	reg_status;

	void __iomem		   	*regs;
	struct clk		       	*clk;
	struct clk		       	*pclk;
	struct device		   	*dev;
	//struct resource	   	*ioarea;
	struct i2c_adapter  	adap;

	struct zx29_i2c_platform_data	*pdata;
	int						gpios[2];
};


/*
  * config pin as i2c function.
  *
  * i2c:   struct of i2c info
  *
  */
static inline void	i2c_config_pins(struct zx29_i2c *i2c)
{
	int ret=0;

	switch(i2c->id)
	{
		case 0:
			ret=gpio_request(I2C0_SCL_PIN,i2c->dev->kobj.name);
			if(ret)
				BUG();

			ret=gpio_request(I2C0_SDA_PIN,i2c->dev->kobj.name);
			if(ret)
				BUG();

			zx29_gpio_config(I2C0_SCL_PIN, GPIO46_SCL0);
			zx29_gpio_config(I2C0_SDA_PIN, GPIO47_SDA0);
			break;

		case 1:
			ret=gpio_request(I2C1_SCL_PIN,i2c->dev->kobj.name);
			if(ret)
				BUG();

			ret=gpio_request(I2C1_SDA_PIN,i2c->dev->kobj.name);
			if(ret)
				BUG();

		    zx29_gpio_config(I2C1_SCL_PIN, GPIO48_SCL1);
			zx29_gpio_config(I2C1_SDA_PIN, GPIO49_SDA1);
			break;

		default:
			break;
	}
}
#define	i2c_get_reg(d, reg)		(zx_read_reg(d->regs + reg))

/* irq enable/disable functions */
#define i2c_disable_irq(d)		\
	zx_clr_reg(d->regs + I2C_CMD, CMD_IRQ_MASK)

#define i2c_enable_irq(d)		\
	zx_set_reg(d->regs + I2C_CMD, CMD_IRQ_MASK)

#define	i2c_clr_irq_ack(d)		\
	zx_write_reg(d->regs+I2C_STATUS, IRQ_ACK)

#define	i2c_get_irq_status(d)	\
	zx_read_reg(d->regs+I2C_STATUS)

/*
 * rest fifo
 * in zx297502 read-modify-write is needed.because in i2c init step, whole system was reset.
 * and then bit 0 was set.if write 0x6 directly, bit0 will be clear. system
 * will not work .
 * in zx297510 reset bit must be clear after be set.
 */
#define zx29_i2c_reset_fifo(d)		\
{\
	zx_set_reg(d->regs + I2C_TXF_CTRL, WFIFO_RST);	\
	zx_set_reg(d->regs + I2C_RXF_CTRL, RFIFO_RST);	\
}
#define zx29_i2c_reset_tx_fifo(d)		\
{\
	zx_set_reg(d->regs + I2C_TXF_CTRL, WFIFO_RST);	\
}
#define zx29_i2c_reset_rx_fifo(d)		\
{\
	zx_set_reg(d->regs + I2C_RXF_CTRL, RFIFO_RST);	\
}

#define	i2c_set_clk_div(d, div)		\
    zx_write_reg(d->regs+I2C_CLK_DIV_FS, div)


/*
 * set xfer address
 *
 * flag: true-set high address
  */
#define	i2c_set_addr(d, a, flag)	\
{\
    if (flag){\
		zx_write_reg(d->regs+I2C_DEVICE_H, (a>>7)&0x7);\
	}\
	zx_write_reg(d->regs+I2C_DEVICE_L, a&0x7f);\
}

#define i2c_get_bus_status(d)		\
	 (zx_read_reg(d->regs+I2C_STATUS) & I2C_STAT_BUSY)

#define i2c_get_rx_fifo_length(d)	\
	((zx_read_reg(d->regs+I2C_RXF_STATUS) & 0xff)>>2)

#define i2c_get_tx_fifo_length(d)	\
	((zx_read_reg(d->regs+I2C_TXF_STATUS) & 0xff)>>2)

#define i2c_set_rx_fifo_depth(d, depth)	\
	zx_write_reg(d->regs+I2C_RXF_CTRL, depth)

#define i2c_set_tx_fifo_depth(d, depth)	\
	zx_write_reg(d->regs+I2C_TXF_CTRL, depth)

#define	i2c_write_data(d, v)	zx_write_reg(d->regs+I2C_DATA, v)

#define	i2c_read_data(d)		zx_read_reg(d->regs+I2C_DATA)

#define	i2c_get_cmd(d)			zx_read_reg(d->regs+I2C_CMD)

#define	i2c_set_cmd(d, v)		zx_write_reg(d->regs+I2C_CMD, v)
#endif
