/* drivers/android/netlog_console.c
 *
 * Copyright (C) 2007-2008 Google, Inc.
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This prognetlog is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

#include <linux/console.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/sched.h>
#include <linux/platform_device.h>
#include <linux/proc_fs.h>
#include <linux/string.h>
#include <linux/uaccess.h>
#include <linux/io.h>
#include <asm/uaccess.h>
#include <linux/interrupt.h>
#include <linux/notifier.h>
#include <linux/rculist.h>
#include <linux/kernel.h>
#include <linux/poll.h>

/* netlog专用buf的大?*/
#define BUFLEN (8 * 1024)
#define LOG_BUF_MASK (BUFLEN - 1)
#define LOG_BUF(idx) (netbuf[(idx) & LOG_BUF_MASK])
#define log_buf_len BUFLEN

static char fmtbuf[1024];
static unsigned log_start;
static unsigned log_end;
static char netbuf[BUFLEN];

DECLARE_WAIT_QUEUE_HEAD(netlog_wait);
static DEFINE_RAW_SPINLOCK(netlog_lock);

static void emit_log_char(char c)
{
	//raw_spin_lock_irq(&netlog_lock);
	LOG_BUF(log_end) = c;
	log_end++;

	if (log_end - log_start > log_buf_len)
		log_start = log_end - log_buf_len;
}

static ssize_t netlog_read(struct file *file, char __user *buf,
		size_t len, loff_t *offset)
{
	unsigned i;
	char c;
	int error;
	unsigned long flags;

	error = -EINVAL;
	if (!buf )//cov_2|| len < 0)
		goto out;

	error = 0;
	if (!len)
		goto out;
	if (!access_ok(VERIFY_WRITE, buf, len)) {
		error = -EFAULT;
		goto out;
	}
	error = wait_event_interruptible(netlog_wait, (log_start - log_end));
	if (error)
		goto out;

	i = 0;
	raw_spin_lock_irqsave(&netlog_lock, flags);
	while (!error && (log_start != log_end) && i < len) {
		c = LOG_BUF(log_start);
		log_start++;
		raw_spin_unlock_irqrestore(&netlog_lock, flags);
		error = __put_user(c, buf);
		buf++;
		i++;
		cond_resched();
		raw_spin_lock_irqsave(&netlog_lock, flags);
	}
	raw_spin_unlock_irqrestore(&netlog_lock, flags);
	if (!error)
		error = i;
out:
	return error;
}

static int netlog_open(struct inode * inode, struct file * file)
{
	return 0;
}

static unsigned int netlog_poll(struct file *file, poll_table *wait)
{
	poll_wait(file, &netlog_wait, wait);
	if (log_end - log_start)
		return POLLIN | POLLRDNORM;
	return 0;
}

static const struct file_operations netlog_file_ops = {
	.owner = THIS_MODULE,
	.read = netlog_read,
	.open = netlog_open,
	.poll = netlog_poll,
	.llseek = generic_file_llseek,
};

static size_t log_prefix(const char *p, unsigned int *level, char *special)
{
	unsigned int lev = 0;
	char sp = '\0';
	size_t len;

	if (p[0] != '<' || !p[1])
		return 0;
	if (p[2] == '>') {
		/* usual single digit level number or special char */
		switch (p[1]) {
			case '0' ... '7':
				lev = p[1] - '0';
				break;
			case 'c': /* KERN_CONT */
			case 'd': /* KERN_DEFAULT */
				sp = p[1];
				break;
			default:
				return 0;
		}
		len = 3;
	} else {
		/* multi digit including the level and facility number */
		char *endp = NULL;

		lev = (simple_strtoul(&p[1], &endp, 10) & 7);
		if (endp == NULL || endp[0] != '>')
			return 0;
		len = (endp + 1) - p;
	}

	/* do not accept special char if not asked for */
	if (sp && !special)
		return 0;

	if (special) {
		*special = sp;
		/* return special char, do not touch level */
		if (sp)
			return len;
	}

	if (level)
		*level = lev;
	return len;
}


static int new_text_line = 1;
#if 0
int net_vprintk(const char *fmt, va_list args)
{
	unsigned long flags;
	char *p;
	int printed_len = 0;
	/* 暂时所有的都打?*/
	int current_log_level = 7;
	char special;
	size_t plen;

	raw_spin_lock_irqsave(&netlog_lock, flags);

	printed_len += vscnprintf(fmtbuf, sizeof(fmtbuf), fmt, args);	
	p = fmtbuf;
	plen = log_prefix(p, &current_log_level, &special);
	if (plen) {
		p += plen;

		switch (special) {
			case 'c': 
				plen = 0;
				break;
			case 'd': 
				plen = 0;
			default:
				if (!new_text_line) {
					emit_log_char('\n');
					new_text_line = 1;
				}
		}
	}

	for (; *p; p++) {
		if (new_text_line) {
			new_text_line = 0;

			if (plen) {
				/* Copy original log prefix */
				int i;

				for (i = 0; i < plen; i++)
					emit_log_char(fmtbuf[i]);
				printed_len += plen;
			} else {
				/* Add log prefix */
				emit_log_char('<');
				emit_log_char(current_log_level + '0');
				emit_log_char('>');
				printed_len += 3;
			}
			{
				/* Add the current time stamp */
				char tbuf[50], *tp;
				unsigned tlen;
				unsigned long long t;
				unsigned long nanosec_rem;

				t = cpu_clock(smp_processor_id());
				nanosec_rem = do_div(t, 1000000000);
				tlen = sprintf(tbuf, "[%5lu.%06lu] ",
						(unsigned long) t,
						nanosec_rem / 1000);

				for (tp = tbuf; tp < tbuf + tlen; tp++)
					emit_log_char(*tp);
				printed_len += tlen;
			}

			if (!*p)
				break;
		}
		emit_log_char(*p);
		if (*p == '\n')
			new_text_line = 1;
	}
	raw_spin_unlock_irqrestore(&netlog_lock, flags);
	if(log_start != log_end)
		wake_up_interruptible(&netlog_wait);
	return printed_len;
}

int net_print(const char *fmt, ...)
{
	va_list args;
	int r;

	va_start(args, fmt);
	r = net_vprintk(fmt, args);
	va_end(args);
	return r;
}
#endif
static int __init netlog_late_init(void)
{
	proc_create("netlog", S_IRUSR, NULL, &netlog_file_ops);
	return 0;
}

late_initcall(netlog_late_init);
