#include <linux/spinlock_types.h>
#include <linux/spinlock.h>

#include <linux/compiler.h>
#include <linux/kernel.h>
#include <linux/bug.h>
#include <linux/slab.h>
#include <linux/mm.h>
#include <linux/mman.h>
#include <linux/mmzone.h>
#include <linux/vmstat.h>


//#define SKB_TRACE /*for trace the max size and number*/
typedef unsigned long skb_addr_t;

#define SKB_POOL_AUTO_EXTEND
#define SKB_SYS_POOL_NR             (32U)
#define SKB_SYS_POOL_BATCH_NR       (100U)


#define SKB_SYS_POOL_0_SIZE     	(256U) /*skb*/
#define SKB_SYS_POOL_0_NR        	(1U)
#define SKB_SYS_POOL_0_PATCH_NR  	(10U)
#define SKB_SYS_POOL_0_RESERVE_NR  	(100U)


#define SKB_SYS_POOL_1_SIZE     	(900U)/*skb_data*/
#define SKB_SYS_POOL_1_NR        	(1U)
#define SKB_SYS_POOL_1_PATCH_NR  	(5U)
#define SKB_SYS_POOL_1_RESERVE_NR  	(10U)


#define SKB_SYS_POOL_2_SIZE     	(2000U)
#define SKB_SYS_POOL_2_NR        	(1U)
#define SKB_SYS_POOL_2_PATCH_NR  	(10U)
#define SKB_SYS_POOL_2_RESERVE_NR  	(40U)


#define SKB_SYS_POOL_3_SIZE     	(4000U)
#define SKB_SYS_POOL_3_NR        	(1U)
#define SKB_SYS_POOL_3_PATCH_NR  	(5U)
#define SKB_SYS_POOL_3_RESERVE_NR  	(40U)


#define SKB_SYS_POOL_4_SIZE     	(0)
#define SKB_SYS_POOL_4_NR        	(0)
#define SKB_SYS_POOL_4_PATCH_NR  	(0)
#define SKB_SYS_POOL_4_RESERVE_NR  	(0)


#define SKB_SYS_POOL_5_SIZE     	(0)
#define SKB_SYS_POOL_5_NR        	(0)
#define SKB_SYS_POOL_5_PATCH_NR  	(1)
#define SKB_SYS_POOL_5_RESERVE_NR  	(0)


#define SKB_SYS_POOL_6_SIZE     	(0)
#define SKB_SYS_POOL_6_NR       	(0)
#define SKB_SYS_POOL_6_PATCH_NR  	(1)
#define SKB_SYS_POOL_6_RESERVE_NR  	(0)


#define SKB_SYS_POOL_7_SIZE     	(0)
#define SKB_SYS_POOL_7_NR       	(0)
#define SKB_SYS_POOL_7_PATCH_NR 	(1)
#define SKB_SYS_POOL_7_RESERVE_NR  	(0)


#define SKB_SYS_POOL_8_SIZE     	(0)
#define SKB_SYS_POOL_8_NR       	(0)
#define SKB_SYS_POOL_8_PATCH_NR 	(1)
#define SKB_SYS_POOL_8_RESERVE_NR  	(0)

#define pool_test_bit(nr, val)   ((val) & (1UL<<(nr)))
#define pool_set_bit(nr, val)    ((val) |= (1UL<<(nr)))
#define pool_clear_bit(nr, val)  ((val) &= ~(1UL<<(nr)))

#define array_start(a)      (&(a)[0])
#define array_nr(a)         (sizeof(a) / sizeof((a)[0]))
#define array_end(a)        (&(a)[array_nr(a)])
#define array_index(a, e)   ((e) - array_start(a))

#define SKB_POOL_RESERVE		1
#define SKB_POOL_NORESERVE		0

typedef struct skb_pool_config_t skb_pool_config_t;
typedef struct skb_pool_t skb_pool_t;
typedef struct skb_pool_node_t skb_pool_node_t;
static spinlock_t  skb_sys_pool_spinlock;

#ifdef SKB_DBG_POOL
# define skb_pool_magic_top(node, size)                                     \
    ((void *)((skb_addr_t)(node) + sizeof(skb_pool_node_impl_t) + round_up(size, SKB_POOL_ALIGN_SIZE)))
#endif

#ifdef SKB_DBG_POOL
# define SKB_POOL_MAGIC          (0x5A5A5A5A)
# define SKB_POOL_MAGIC_NR      (2)
# define SKB_POOL_MAGIC_SIZE    (sizeof(skb_addr_t) * SKB_POOL_MAGIC_NR)
#else
# define SKB_POOL_MAGIC_SIZE    (0)
#endif

#define skb_pool_ptr(node)                                                  \
    ((void *)((skb_addr_t)(node) + sizeof(skb_pool_node_impl_t)))

#define skb_pool_node(ptr)                                                  \
    ((skb_pool_node_impl_t *)((skb_addr_t)(ptr) - sizeof(skb_pool_node_impl_t)))

#define skb_pool_node_size(obj_size)                                        \
    (sizeof(skb_pool_node_impl_t) + round_up(obj_size, SKB_POOL_ALIGN_SIZE) + SKB_POOL_MAGIC_SIZE)

#define skb_pool_size(obj_size, obj_nr)                                     \
    (sizeof(skb_pool_impl_t))
    
#define is_aligned(x, a)    (((x) & ((typeof(x))(a) - 1)) == 0)

#define ptr_is_aligned(x)   ((x) != NULL && is_aligned((skb_addr_t)x, sizeof(skb_addr_t)))

/*******************************************************************************
*                                Ͷ                                  *
*******************************************************************************/
typedef unsigned int  skb_count_t;
typedef unsigned int size_t;
typedef unsigned int u32;

#define SKB_POOL_ALIGN_SIZE     (sizeof(skb_addr_t))

typedef struct skb_pool_node_impl_t skb_pool_node_impl_t;

struct skb_pool_node_impl_t
{
    skb_pool_node_impl_t    *free_next;
#ifdef _USE_VEHICLE_DC
	u32 padding[15];//for cacheline
#endif
#ifdef SKB_DBG_POOL
    struct list_head        alloc_node;
    const char              *file;
    unsigned long           tick;
    u32                     line;
    skb_addr_t              *magic_top;
    skb_addr_t              magic_bottom[SKB_POOL_MAGIC_NR];
#endif
} skb_align_data;

typedef struct
{
    skb_pool_node_impl_t    *free_head;
    size_t                   obj_size;
    skb_count_t              obj_cur_nr;     //ǰ

//#ifdef SKB_POOL_AUTO_EXTEND
    skb_count_t             obj_nr;          //ܹ
//    skb_count_t             obj_max_nr;
    skb_count_t             obj_batch_nr;
	skb_count_t				obj_reserve_nr;

//#endif

#ifdef SKB_DBG_POOL
    struct list_head        alloc_head;
    skb_count_t             obj_used_cnt;     //ǰʹ(obj_nr - obj_cur_nr)
    skb_count_t             obj_max_used_cnt; //ʹ(ֵ)
#endif

#ifdef SKB_TRACE
    size_t                  obj_real_size[2048];
#endif
} skb_pool_inner_t;

#define skb_aligned(x)      __attribute__((aligned(x)))

#define skb_align_data          skb_aligned(sizeof(skb_addr_t))
typedef struct
{
    spinlock_t  lock;
    skb_pool_inner_t    inner;
} skb_align_data skb_pool_impl_t;


struct skb_pool_config_t
{
    void        *addr;      /* base address     */
    size_t      obj_size;   /* object size      */
    skb_count_t obj_nr;     /* object number    */
    size_t      patch_nr;
    size_t      reserve_nr;
};

/*******************************************************************************
*                             ⲿ                                  *
*******************************************************************************/
extern unsigned long wm_min_pages;

/*******************************************************************************
*                                                                 *
*******************************************************************************/
static unsigned long skb_sys_pool_bitmap;
static skb_pool_inner_t *skb_sys_pool_inner[SKB_SYS_POOL_NR];

static skb_count_t skb_sys_pool_nr;
static size_t skb_sys_pool_sizes[SKB_SYS_POOL_NR];

#if (SKB_SYS_POOL_0_NR > 0)
     static u8 skb_sys_pool_0[skb_pool_size(SKB_SYS_POOL_0_SIZE, SKB_SYS_POOL_0_NR)] skb_align_data;
#endif

#if (SKB_SYS_POOL_1_NR > 0)
     static u8 skb_sys_pool_1[skb_pool_size(SKB_SYS_POOL_1_SIZE, SKB_SYS_POOL_1_NR)] skb_align_data;
#endif

#if (SKB_SYS_POOL_2_NR > 0)
     static u8 skb_sys_pool_2[skb_pool_size(SKB_SYS_POOL_2_SIZE, SKB_SYS_POOL_2_NR)] skb_align_data;
#endif

#if (SKB_SYS_POOL_3_NR > 0)
     static u8 skb_sys_pool_3[skb_pool_size(SKB_SYS_POOL_3_SIZE, SKB_SYS_POOL_3_NR)] skb_align_data;
#endif

#if (SKB_SYS_POOL_4_NR > 0)
     static u8 skb_sys_pool_4[skb_pool_size(SKB_SYS_POOL_4_SIZE, SKB_SYS_POOL_4_NR)] skb_align_data;
#endif

#if (SKB_SYS_POOL_5_NR > 0)
     static u8 skb_sys_pool_5[skb_pool_size(SKB_SYS_POOL_5_SIZE, SKB_SYS_POOL_5_NR)] skb_align_data;
#endif

#if (SKB_SYS_POOL_6_NR > 0)
     static u8 skb_sys_pool_6[skb_pool_size(SKB_SYS_POOL_6_SIZE, SKB_SYS_POOL_6_NR)] skb_align_data;
#endif

#if (SKB_SYS_POOL_7_NR > 0)
     static u8 skb_sys_pool_7[skb_pool_size(SKB_SYS_POOL_7_SIZE, SKB_SYS_POOL_7_NR)] skb_align_data;
#endif

#if (SKB_SYS_POOL_8_NR > 0)
     static u8 skb_sys_pool_8[skb_pool_size(SKB_SYS_POOL_8_SIZE, SKB_SYS_POOL_8_NR)] skb_align_data;
#endif

static struct skb_pool_config_t  skb_sys_pool_config[] = 
{
#if (SKB_SYS_POOL_0_NR > 0)
    {
        (void *)array_start(skb_sys_pool_0),        /* base address     */
        SKB_SYS_POOL_0_SIZE,                        /* object size      */
        SKB_SYS_POOL_0_NR,                          /* object number    */
	    SKB_SYS_POOL_0_PATCH_NR,
	    SKB_SYS_POOL_0_RESERVE_NR,
    },
#endif

#if (SKB_SYS_POOL_1_NR > 0)
    {
        (void *)array_start(skb_sys_pool_1),        /* base address     */
        SKB_SYS_POOL_1_SIZE,                        /* object size      */
        SKB_SYS_POOL_1_NR,                          /* object number    */
	    SKB_SYS_POOL_1_PATCH_NR,
	    SKB_SYS_POOL_1_RESERVE_NR,
    },
#endif

#if (SKB_SYS_POOL_2_NR > 0)
    {
        (void *)array_start(skb_sys_pool_2),        /* base address     */
        SKB_SYS_POOL_2_SIZE,                        /* object size      */
        SKB_SYS_POOL_2_NR,                          /* object number    */
	    SKB_SYS_POOL_2_PATCH_NR,
	    SKB_SYS_POOL_2_RESERVE_NR,
    },
#endif

#if (SKB_SYS_POOL_3_NR > 0)
    {
        (void *)array_start(skb_sys_pool_3),        /* base address     */
        SKB_SYS_POOL_3_SIZE,                        /* object size      */
        SKB_SYS_POOL_3_NR,                          /* object number    */
		SKB_SYS_POOL_3_PATCH_NR,
		SKB_SYS_POOL_3_RESERVE_NR,
    },
#endif

#if (SKB_SYS_POOL_4_NR > 0)
    {
        (void *)array_start(skb_sys_pool_4),        /* base address     */
        SKB_SYS_POOL_4_SIZE,                        /* object size      */
        SKB_SYS_POOL_4_NR,                          /* object number    */
	    SKB_SYS_POOL_4_PATCH_NR,
	    SKB_SYS_POOL_4_RESERVE_NR,
    },
#endif

#if (SKB_SYS_POOL_5_NR > 0)
    {
        (void *)array_start(skb_sys_pool_5),        /* base address     */
        SKB_SYS_POOL_5_SIZE,                        /* object size      */
        SKB_SYS_POOL_5_NR,                          /* object number    */
		SKB_SYS_POOL_5_PATCH_NR,
		SKB_SYS_POOL_5_RESERVE_NR,
    },
#endif

#if (SKB_SYS_POOL_6_NR > 0)
    {
        (void *)array_start(skb_sys_pool_6),        /* base address     */
        SKB_SYS_POOL_6_SIZE,                        /* object size      */
        SKB_SYS_POOL_6_NR,                          /* object number    */
		SKB_SYS_POOL_6_PATCH_NR,
		SKB_SYS_POOL_6_RESERVE_NR,
    },
#endif

#if (SKB_SYS_POOL_7_NR > 0)
    {
        (void *)array_start(skb_sys_pool_7),        /* base address     */
        SKB_SYS_POOL_7_SIZE,                        /* object size      */
        SKB_SYS_POOL_7_NR,                          /* object number    */
	    SKB_SYS_POOL_7_PATCH_NR,
	    SKB_SYS_POOL_7_RESERVE_NR,
    },
#endif

#if (SKB_SYS_POOL_8_NR > 0)
    {
        (void *)array_start(skb_sys_pool_8),        /* base address     */
        SKB_SYS_POOL_8_SIZE,                        /* object size      */
        SKB_SYS_POOL_8_NR,                          /* object number    */
	    SKB_SYS_POOL_8_PATCH_NR, 
	    SKB_SYS_POOL_8_RESERVE_NR,
    },
#endif
};

/*******************************************************************************
*                                ȫֺ                                  *
*******************************************************************************/
#ifdef SKB_DBG_POOL
int skb_pool_magic_check(skb_pool_node_impl_t *node)
{
    int cnt;

    BUG_ON(node == NULL);

    for (cnt = 0x00; cnt < SKB_POOL_MAGIC_NR; cnt++)
    {
        BUG_ON(node->magic_bottom[cnt] != SKB_POOL_MAGIC);
        BUG_ON(*(node->magic_top + cnt) != SKB_POOL_MAGIC);
    }

    return 0;
}

static inline void skb_pool_magic_init(skb_pool_node_impl_t *node)
{
    int cnt;

    BUG_ON(node == NULL);

    for (cnt = 0x00; cnt < SKB_POOL_MAGIC_NR; cnt++)
    {
        node->magic_bottom[cnt] = SKB_POOL_MAGIC;
        *(node->magic_top + cnt) = SKB_POOL_MAGIC;
    }
}

#else
static inline int skb_pool_magic_check(skb_pool_node_impl_t *node) { return 0; }
static inline void skb_pool_magic_init(skb_pool_node_impl_t *node) {}
#endif

static inline int is_memory_enough(void)
{
	return (global_page_state(NR_FREE_PAGES) > wm_min_pages);
}

int skb_pool_add_inner(skb_pool_inner_t *inner, unsigned int obj_nr)
{
    u32 alloc_size;
    skb_count_t cnt;
    skb_pool_node_impl_t *node = NULL;

    BUG_ON(inner == NULL || obj_nr == 0);

    inner->obj_cur_nr += obj_nr;
    inner->obj_nr += obj_nr;
    alloc_size = skb_pool_node_size(inner->obj_size);

    // ǰжʣڴǷ
    if (is_memory_enough())
        node = (skb_pool_node_impl_t *)kmalloc(alloc_size, GFP_ATOMIC);

    if (node == NULL)
	{
        inner->obj_cur_nr -= obj_nr;
        inner->obj_nr -= obj_nr;
        return -1;
    }
    inner->free_head = node;
#ifdef SKB_DBG_POOL
    node->magic_top = skb_pool_magic_top(node, inner->obj_size);
    skb_pool_magic_init(node);
#endif

    for (cnt = 1; cnt < obj_nr; cnt++)
    {
        // ǰжʣڴǷ
        if (is_memory_enough())
            node->free_next = (skb_pool_node_impl_t *)kmalloc(alloc_size, GFP_ATOMIC);
        else
            node->free_next = NULL;   /* no mem */
        if (node->free_next == NULL)
        {
            inner->obj_nr -= obj_nr;
            inner->obj_nr += cnt;
            inner->obj_cur_nr = cnt;
            return 0;
        }

        node = node->free_next;
#ifdef SKB_DBG_POOL
        node->magic_top = skb_pool_magic_top(node, inner->obj_size);
        skb_pool_magic_init(node);
#endif
    }
    node->free_next = NULL;
    return 0;
}

static void skb_pool_delete_inner(skb_pool_inner_t *inner, unsigned int type)
{
    skb_pool_node_impl_t *node;
    skb_pool_node_impl_t *next_free;

    BUG_ON(inner == NULL); 
    node = inner->free_head;
    switch(type)
    {
        case SKB_POOL_RESERVE:
            while (inner->obj_cur_nr > inner->obj_reserve_nr)
            {
                next_free = node->free_next;
                kfree(node);
                inner->obj_nr--;
                inner->obj_cur_nr--;
                node = next_free;
            }
            inner->free_head = node;
            break;
        case SKB_POOL_NORESERVE:
            inner->obj_nr -= inner->obj_cur_nr;
            while (node != NULL)
            {   
                next_free = node->free_next;
                kfree(node);
                inner->obj_cur_nr--;
                node = next_free;
            }
            inner->free_head = NULL;
            break;
        default:
            break;
    }
    return;
}

void skb_sys_pool_delete(void)
{   
    int index;
    unsigned long flags;
 
    spin_lock_irqsave(&skb_sys_pool_spinlock, flags);

    for (index = 0; index < skb_sys_pool_nr; index++)
    {
        if (skb_sys_pool_inner[index] != NULL)
        {
            skb_pool_delete_inner(skb_sys_pool_inner[index], SKB_POOL_NORESERVE);
        }
    }

    spin_unlock_irqrestore(&skb_sys_pool_spinlock, flags);
}

skb_pool_inner_t *skb_pool_create_inner(
                    skb_pool_inner_t *inner,
                    size_t           obj_size,
                    skb_count_t      obj_nr,
                    skb_count_t      obj_batch_nr,
                    skb_count_t      obj_reserve_nr)
{
    BUG_ON(!ptr_is_aligned(inner) || obj_size <= 0 || obj_nr <= 0);

    inner->obj_nr = 0;
    inner->obj_batch_nr = obj_batch_nr;
	inner->obj_reserve_nr = obj_reserve_nr;

#ifdef SKB_TRACE
    memset(inner->obj_real_size, 0, sizeof(inner->obj_real_size));
#endif

#ifdef SKB_DBG_POOL
    INIT_LIST_HEAD(&inner->alloc_head);
    inner->obj_max_used_cnt = 0;
    inner->obj_used_cnt = 0;
#endif

    inner->free_head = NULL;
    inner->obj_size = obj_size;
    inner->obj_cur_nr = 0;

    if (skb_pool_add_inner(inner, obj_nr) < 0)
        return NULL;

    return inner;
}

static inline int skb_sys_pool_match_by_size(u32 size)
{
    int index;

    for (index = 0; index < skb_sys_pool_nr; index++)
    {
        if (size > skb_sys_pool_sizes[index])
            continue;
        return index;
    }

    return -1;
}

size_t skb_sys_pool_size(const void *ptr)
{
    int index;
    skb_pool_node_impl_t *node;

    node = skb_pool_node(ptr);
    index = *(int*)node;

    return skb_sys_pool_inner[index]->obj_size;
}

void *skb_sys_pool_alloc(
                    size_t      size
#ifdef SKB_DBG_POOL
                    ,const char *file,
                    u32         line
#endif
                    )
{
    int index;
    unsigned long flags;
    skb_pool_inner_t *inner;
    skb_pool_node_impl_t *node;

    BUG_ON(size == 0);

    spin_lock_irqsave(&skb_sys_pool_spinlock, flags);

    index = skb_sys_pool_match_by_size(size);
    if (index == -1)
    {
        skb_pool_inner_t *inner;

        if (skb_sys_pool_nr >= SKB_SYS_POOL_NR)
        {
            spin_unlock_irqrestore(&skb_sys_pool_spinlock, flags);
            return NULL;
        }
        inner = (skb_pool_inner_t*)kzalloc(sizeof(skb_pool_inner_t), GFP_ATOMIC);
        if (inner == NULL)
        {
            spin_unlock_irqrestore(&skb_sys_pool_spinlock, flags);
        	return NULL;
        }
        if ((skb_sys_pool_inner[skb_sys_pool_nr] = skb_pool_create_inner(inner, size, 1, 1, 0)) == NULL)
        {
        	kfree(inner);
            spin_unlock_irqrestore(&skb_sys_pool_spinlock, flags);
            return NULL;
        }

        skb_sys_pool_sizes[skb_sys_pool_nr] = size;
        pool_set_bit(skb_sys_pool_nr, skb_sys_pool_bitmap);
        index = skb_sys_pool_nr;
        skb_sys_pool_nr++;
    }

    inner = skb_sys_pool_inner[index];
    node = inner->free_head;

    if (node == NULL)
    {
        if (skb_pool_add_inner(inner, inner->obj_batch_nr) < 0)
        {
            spin_unlock_irqrestore(&skb_sys_pool_spinlock, flags);
            return NULL;	
	    }
    }

    inner->obj_cur_nr--;
    node = inner->free_head;
    inner->free_head = node->free_next;
    *(int *)node = index;

#ifdef SKB_DBG_POOL
    list_add_tail(&node->alloc_node, &inner->alloc_head);
    inner->obj_used_cnt++;
    if (inner->obj_max_used_cnt < inner->obj_used_cnt)
        inner->obj_max_used_cnt = inner->obj_used_cnt;
#endif

#ifdef SKB_TRACE
    if (size < 2048)
    {
        inner->obj_real_size[size]++;
    }
    else
    {
        inner->obj_real_size[0]++;
    }
#endif

    spin_unlock_irqrestore(&skb_sys_pool_spinlock, flags);

#ifdef SKB_DBG_POOL
    BUG_ON(skb_pool_magic_check(node) != 0);

    node->file  = file;
    node->line  = line;
    node->tick  = jiffies;
#endif

    return skb_pool_ptr(node);
}

/*******************************************************************************
* :     ϵͳڴͷ
* ˵:     
*   ()  ptr:    ͷŵڴַ
                file:   ļ
                line:   к
*   ()  void
*   ֵ:     void
* ˵:     void
*******************************************************************************/
void skb_sys_pool_free(
                    void        *ptr
#ifdef SKB_DBG_POOL
                    ,const char *file,
                    u32         line
#endif
                    )
{
    int index;
    unsigned long flags;
    skb_pool_inner_t *inner;
    skb_pool_node_impl_t *node;

    BUG_ON(ptr == NULL);

    spin_lock_irqsave(&skb_sys_pool_spinlock, flags);

    node = skb_pool_node(ptr);
    index = *(int *)node;
    BUG_ON(index == -1 || index > skb_sys_pool_nr || skb_sys_pool_inner[index] == NULL);

    inner = skb_sys_pool_inner[index];
    node->free_next = inner->free_head;
    inner->free_head = node;
    inner->obj_cur_nr++;
    
#ifdef SKB_DBG_POOL
    list_del(&node->alloc_node);
    inner->obj_used_cnt--;
#endif

    if (inner->obj_cur_nr > inner->obj_reserve_nr)
        skb_pool_delete_inner(skb_sys_pool_inner[index], SKB_POOL_RESERVE);

    spin_unlock_irqrestore(&skb_sys_pool_spinlock, flags);

#ifdef SKB_DBG_POOL
    BUG_ON(skb_pool_magic_check(node) != 0);
#endif
}

void skb_sys_pool_init_impl(skb_pool_config_t *cfg, skb_count_t cfg_nr)
{
    u32 cnt;

    BUG_ON( cfg == NULL || cfg_nr > SKB_SYS_POOL_NR);

    for (cnt = 0x00; cnt < cfg_nr; cnt++, cfg++)
    {
        skb_sys_pool_inner[cnt] = 
            skb_pool_create_inner(
                (skb_pool_inner_t *)cfg->addr,
                cfg->obj_size,
                cfg->obj_nr,
                cfg->patch_nr,
                cfg->reserve_nr);
        skb_sys_pool_sizes[cnt] = cfg->obj_size;
        skb_sys_pool_nr++;
        pool_set_bit(cnt , skb_sys_pool_bitmap);

        if (cnt && cnt > 1)
            BUG_ON(cfg->obj_size < skb_sys_pool_sizes[cnt - 1]);
    }
}

int __init skb_sys_pool_init(void)
{
    spin_lock_init(&skb_sys_pool_spinlock);

    skb_sys_pool_init_impl(array_start(skb_sys_pool_config), array_nr(skb_sys_pool_config));

    return 0;
}

core_initcall(skb_sys_pool_init);
//late_initcall(skb_sys_pool_init);

