/*
 * zx297510 CPU idle Routines
 *
 * Copyright (C) 2013 ZTE, Ltd.
 * Shine Yu <yu.xiang5@zte.com.cn>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/sched.h>
#include <linux/export.h>
#include <linux/kernel.h>
#include <linux/cpu.h>
#include <linux/tick.h>
#include <linux/suspend.h>
#include <linux/interrupt.h>

#include <mach/zx-pm.h>

#ifdef CONFIG_ZX_PM_DEBUG
struct zx_idle_stats 	idle_stats;
#endif

struct cpuidle_driver zx_idle_driver = {
	.name = 	"zx_idle",
	.owner =	THIS_MODULE,
	.en_core_tk_irqen	= 0,					/* no use cpuidle time keeping */
};

static bool deep_idle_disabled_by_suspend = 0;
static u32 	deep_idle_disabled_by_debug 	= 0;
static u32 print_enabled_by_debug = 0;
DEFINE_PER_CPU(struct cpuidle_device, zx_idle_dev);


/**
 * idle_can_enter_deep_sleep - check can enter deep sleep state?
 *
 *
 */
static int idle_can_enter_deep_sleep(void)
{
	s64 request = ktime_to_us(tick_nohz_get_sleep_length());

	/* can not enter deep sleep now */
	if (deep_idle_disabled_by_suspend)
		return false;

	/* This mode only can be entered when other core's are offline */
	if(deep_idle_disabled_by_debug || num_online_cpus() > 1)
		return false;

	if(request < 0)
		return false;

	return true;
}


static int zx_pm_idle_prepare(struct cpuidle_device *dev,
		struct cpuidle_driver *drv, int index)
{
	int new_index = index;

	if(new_index == drv->safe_state_index)
		return new_index;

	if(!idle_can_enter_deep_sleep())
	{
		new_index = drv->safe_state_index;
		return new_index;
	}

	return new_index;
}

/**
 * zx_enter_idle
 * @dev: cpuidle device
 * @state: The target state to be programmed
 *
 * Idle function for C1 state, WFI on a single CPU.
 * Called with irqs off, returns with irqs on.
 * Returns the amount of time spent in the low power state.
 */
int zx_enter_idle(struct cpuidle_device *dev,
				struct cpuidle_driver *drv,
				int index)
{
	ktime_t entry_time, exit_time;
	s64 idle_time;
	int new_index = index;

	local_irq_disable();
	local_fiq_disable();

	entry_time = ktime_get();

/*=================================================================
 *=======begin enter idle sleep====================================
 *=================================================================
 */
	new_index = zx_pm_idle_prepare(dev, drv, index);

	index = zx_pm_idle_enter(new_index);

/*=================================================================
 *=======end enter idle sleep======================================
 *=================================================================
 */
	exit_time = ktime_get();

	local_fiq_enable();
	local_irq_enable();

	idle_time = ktime_to_us(ktime_sub(exit_time, entry_time));

	dev->last_residency = (int)idle_time;

	if(print_enabled_by_debug != 0)
	{
		printk(KERN_INFO "[CPUIDLE] exit idle: idle time= %d , enter level= %d !\n", (u32)idle_time, index);
	}

	return index;
}



static int idle_pm_notify(struct notifier_block *nb,
	unsigned long event, void *dummy)
{
#ifdef CONFIG_PM_SLEEP
	if (event == PM_SUSPEND_PREPARE)
		deep_idle_disabled_by_suspend = true;
	else if (event == PM_POST_SUSPEND)
		deep_idle_disabled_by_suspend = false;
#endif

	return NOTIFY_OK;
}

static struct notifier_block idle_pm_notifier =
{
	.notifier_call = idle_pm_notify,
};


/**
 * zx_cpuidle_init - Init routine for zx29xx idle
 *
 * Registers the cpuidle driver with the cpuidle
 * framework with the valid set of states.
 */
int __init zx_cpuidle_init(void)
{
	int cpu_id;
	struct cpuidle_device *device;
	struct cpuidle_driver *drv = &zx_idle_driver;

	/* Setup cpuidle driver */
	drv->state_count = zx_fill_cpuidle_data(drv);
	cpuidle_register_driver(drv);

	/* Setup cpuidle device for each cpu */
	for_each_cpu(cpu_id, cpu_online_mask)
	{
		device = &per_cpu(zx_idle_dev, cpu_id);
		device->cpu = cpu_id;

		if (cpu_id == 0)
			device->state_count = drv->state_count;
		else
			device->state_count = 1;	/* None boot cpu Support IDLE only now ! */

		if (cpuidle_register_device(device))
		{
			printk(KERN_ERR "[CPUIDLE] register device failed\n,");
			return -EIO;
		}
	}

	register_pm_notifier(&idle_pm_notifier);

	printk(KERN_INFO "[CPUIDLE] register device OK\n,");

	return 0;
}

#if 0
static void __exit zx_cpuidle_exit(void)
{
	unregister_pm_notifier(&idle_pm_notifier);
	cpuidle_unregister_driver(&zx_idle_driver);
}

late_initcall(zx_cpuidle_init);
module_exit(zx_cpuidle_exit);
#endif

#ifdef CONFIG_ZX_PM_DEBUG
static char*  lp2_debug_show(char *s)
{
	int i;

	s += sprintf(s, "%-30s%8s %8s %8s %8s\n", " ", "cpu0","cpu1","cpu2","cpu3");
	s += sprintf(s, "%s\n", "---------------------------------------------------------------");
	s += sprintf(s, "%-30s%8u %8u %8u %8u\n", "lp3 in count:",
		idle_stats.lp3_count[0],
		idle_stats.lp3_count[1],
		idle_stats.lp3_count[2],
		idle_stats.lp3_count[3]);

	s += sprintf(s, "%-30s%8u %8u %8u %8u\n", "lp2 in count:",
		idle_stats.lp2_count[0],
		idle_stats.lp2_count[1],
		idle_stats.lp2_count[2],
		idle_stats.lp2_count[3]);

	s += sprintf(s, "%-30s%8u %8u %8u %8u\n", "lp2 completed:",
		idle_stats.lp2_completed_count[0],
		idle_stats.lp2_completed_count[1],
		idle_stats.lp2_completed_count[2],
		idle_stats.lp2_completed_count[3]);

	s += sprintf(s, "%-30s%7u%% %7u%% %7u%% %7u%%\n", "lp2 completed%:",
		idle_stats.lp2_completed_count [0]* 100 / (idle_stats.lp2_count[0] ?: 1),
		idle_stats.lp2_completed_count [1]* 100 / (idle_stats.lp2_count[1] ?: 1),
		idle_stats.lp2_completed_count [2]* 100 / (idle_stats.lp2_count[2] ?: 1),
		idle_stats.lp2_completed_count [3]* 100 / (idle_stats.lp2_count[3] ?: 1));
	s += sprintf(s, "%-30s%8u\n", "all idle count:", idle_stats.idle_count);

	s += sprintf(s, "\n%-30s%8llu %8llu %8llu %8llu ms\n", "cpu ready time:",
		div64_u64(idle_stats.cpu_wants_lp2_time[0], 1000),
		div64_u64(idle_stats.cpu_wants_lp2_time[1], 1000),
		div64_u64(idle_stats.cpu_wants_lp2_time[2], 1000),
		div64_u64(idle_stats.cpu_wants_lp2_time[3], 1000));

	s += sprintf(s, "%-30s%8llu %8llu %8llu %8llu ms\n", "lp2 in time:",
		div64_u64(idle_stats.in_lp2_time[0], 1000),
		div64_u64(idle_stats.in_lp2_time[1], 1000),
		div64_u64(idle_stats.in_lp2_time[2], 1000),
		div64_u64(idle_stats.in_lp2_time[3], 1000));

	s += sprintf(s, "%-30s%7d%% %7d%% %7d%% %7d%%\n", "lp2 in time%:",
		(int)(idle_stats.cpu_wants_lp2_time[0] ?
			div64_u64(idle_stats.in_lp2_time[0] * 100,
			idle_stats.cpu_wants_lp2_time[0]) : 0),
		(int)(idle_stats.cpu_wants_lp2_time[1] ?
			div64_u64(idle_stats.in_lp2_time[1] * 100,
			idle_stats.cpu_wants_lp2_time[1]) : 0),
		(int)(idle_stats.cpu_wants_lp2_time[2] ?
			div64_u64(idle_stats.in_lp2_time[2] * 100,
			idle_stats.cpu_wants_lp2_time[2]) : 0),
		(int)(idle_stats.cpu_wants_lp2_time[3] ?
			div64_u64(idle_stats.in_lp2_time[3] * 100,
			idle_stats.cpu_wants_lp2_time[3]) : 0));

	s += sprintf(s, "\n\n%3s %20s %6s %10s\n",
		"int", "name", "count", "last count");
	s += sprintf(s, "%s", "--------------------------------------------\n");
	for (i = 0; i < NR_IRQS; i++) {
		if (idle_stats.lp2_int_count[i] == 0)
			continue;
		s += sprintf(s, "%3d %20s %6d %10d\n",
			i - NR_BOARD_IRQS,
			irq_to_desc(i)->action ? irq_to_desc(i)->action->name ?: "???" : "???",
			idle_stats.lp2_int_count[i],
			idle_stats.lp2_int_count[i] - idle_stats.last_lp2_int_count[i]);

		idle_stats.last_lp2_int_count[i] = idle_stats.lp2_int_count[i];
	};

	return s;
}

/******************************************************
 *** 1 -- lp2 ******************************
 ******************************************************
 */
static ssize_t lp2_show(struct kobject *kobj, struct kobj_attribute *attr,
			  char *buf)
{
	char *s = buf;

	s = lp2_debug_show(s);


	return (s - buf);
}

static ssize_t lp2_store(struct kobject *kobj, struct kobj_attribute *attr,

			   const char *buf, size_t n)
{

	int error = 0;


	return error ? error : n;
}

zte_pm_attr(lp2);


/*=============================================================================
 *========  /sys/zte_pm/cpuidle/disable_lp2  ==================================
 *=============================================================================
 */
static ssize_t disable_lp2_show(struct kobject *kobj, struct kobj_attribute *attr,
			  char *buf)
{
	char *s = buf;

	s += sprintf(s, "%s %d\n", "[CPUIDLE] deep_idle_disabled_by_debug:", deep_idle_disabled_by_debug);

	return (s - buf);
}

/* usage: "echo 1 > disable_lp2" */
static ssize_t disable_lp2_store(struct kobject *kobj, struct kobj_attribute *attr,
			   const char *buf, size_t n)
{
	int error = 0;
	long temp;

	if(strict_strtol(buf, 0, &temp))
		error = -EINVAL;

	deep_idle_disabled_by_debug = temp;

	return error ? error : n;
}
zte_pm_attr(disable_lp2);


/*=============================================================================
 *========  /sys/zte_pm/cpuidle/enable_print  ==================================
 *=============================================================================
 */
static ssize_t enable_print_show(struct kobject *kobj, struct kobj_attribute *attr,
			  char *buf)
{
	char *s = buf;

	s += sprintf(s, "%s %d\n", "[CPUIDLE] print_enabled_by_debug:", print_enabled_by_debug);

	return (s - buf);
}

/* usage: "echo 1 > enable_print" */
static ssize_t enable_print_store(struct kobject *kobj, struct kobj_attribute *attr,
			   const char *buf, size_t n)
{
	int error = 0;
	long temp;

	if(strict_strtol(buf, 0, &temp))
		error = -EINVAL;

	print_enabled_by_debug = temp;

	return error ? error : n;
}

zte_pm_attr(enable_print);

static struct attribute * g[] =
{
	&lp2_attr.attr,
	&disable_lp2_attr.attr,
	&enable_print_attr.attr,
	NULL,
};


static struct attribute_group idle_attr_group =
{
	.attrs = g,
};
/**
 * idle_debug_init
 * create cpuidle sysfs, we can use cat /sys/zte_pm/cpuidle/lp2 command view debug info
 *
 */
static struct kobject *idle_kobj;

int __init idle_debug_init(void)
{
	int ret;

	idle_kobj = kobject_create_and_add("cpuidle", pm_debug_kobj);
	if (!idle_kobj)
		return -ENOMEM;

    ret = sysfs_create_group(idle_kobj, &idle_attr_group);
    if (ret)
    {
        pr_info("[CPUIDLE] sysfs_create_group ret %d\n", ret);
		return ret;
    }

	return 0;
}

#endif

