/*
 * ZTE zx297510 dvfs driver
 *
 * Copyright (C) 2013 ZTE Ltd.
 * 	by zxp
 *
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/err.h>
#include <linux/clk.h>
#include <linux/io.h>
#include <linux/slab.h>
#include <linux/cpufreq.h>
#include <linux/suspend.h>

#include "mach/icp.h"
#include "mach/clock.h"
#include "mach/zx-cpufreq.h"
#include "mach/zx-pm.h"

/* for count change time by M0 */
#define DEBUG_CPUFREQ_TIME	1

//#undef CONFIG_AXI_FREQ
#ifdef CONFIG_AXI_FREQ
static DEFINE_MUTEX(axifreq_lock);

#define	get_cur_axi()				pm_read_reg_16(AXI_CURRENT_FREQ)
#define	set_target_axi(f)			pm_write_reg_16(AXI_AP2M0_TARGET, f)
#define	set_axi_req()				pm_write_reg_16(AXI_AP2M0_FLAG, 1)
#define	clr_axi_ack()				pm_write_reg_16(AXI_M02AP_ACK, 0)

#if 0
#define	WAIT_AXI_ACK_TIMEOUT		(jiffies + msecs_to_jiffies(2))	/* wait 2 ms, we count max 200us also */
#define wait_axi_ack(timeout)		while(!pm_read_reg_16(AXI_M02AP_ACK) && time_before(jiffies, timeout))
#else
#define	WAIT_AXI_ACK_TIMEOUT		(500)			/* wait 500us, we count max 200us also */
static void wait_axi_ack(unsigned timeout)
{
	ktime_t begin_time = ktime_get();
	
	while(!pm_read_reg_16(AXI_M02AP_ACK) && (unsigned)ktime_to_us(ktime_sub(ktime_get(), begin_time))<timeout);
}
#endif

#ifdef CONFIG_ZX_PM_DEBUG
static unsigned axi_freq_table[]=
{
    26000,
    39000,
    52000,
    61440,
    78000,
    104000,
    122880,
    156000,
};

static unsigned get_axi_freq(void)
{
	void __iomem *regaddr = NULL;
	unsigned int data = 0;
	unsigned int axi_rate = 0;

    regaddr = (ZX29_TOP_VA+0x54);
	data = ioread32(regaddr);
    data &= 0x3;
    switch (data) {
        case 0: axi_rate = 104000000;break;
        case 1: axi_rate = 26000000;break;
        case 2: axi_rate = 122880000;break;
        case 3: axi_rate = 156000000;break;
        default:
            break;      
    }
    
    regaddr = (ZX29_TOP_VA+0x7c);
	data = ioread32(regaddr);
    data &= 0x3;
    return axi_rate/(0x1<<data);
}
static void debug_axi_clk_info(unsigned req_axi)
{
	unsigned 	real_axi = 	get_axi_freq()/1000;
	
	pr_info("axi req[KHz]   	axi cur[KHz]       axi read[KHz]\n");	
	pr_info("%d[KHz]          %d[KHz]           %d[KHz]\n", axi_freq_table[req_axi], axi_freq_table[get_cur_axi()], real_axi);
}
#endif

static int send_msg_to_m0(void)
{
	unsigned char ap_m0_buf[4];		/* the icp interface need a buffer */
    T_ZDrvRpMsg_Msg Icp_Msg;
	int				ret;	

    Icp_Msg.actorID = M0_ID;
    Icp_Msg.chID 	= channel_1;
    Icp_Msg.flag 	= 1;		/* 1- means send an icp interrupt> */
    Icp_Msg.buf 	= ap_m0_buf;
    Icp_Msg.len 	= 0x4;	

	ret = zDrvRpMsg_Write(&Icp_Msg);
	if(Icp_Msg.len == ret)
		return 0;
	else
		return ret;
}
/**
 * request to change axi freq.
 *
 * axi_freq: input freq enum
 */
int request_axi_freq(zx297510_axi_freq axi_freq)
{
    unsigned current_axi_freq = get_cur_axi();
#if DEBUG_CPUFREQ_TIME	
	ktime_t begin_time, end_time;
	s64 total_time;
#endif	

	set_target_axi(axi_freq);

    if(axi_freq != current_axi_freq)
    {
		/* request freq */
		clr_axi_ack();
		set_axi_req();

#if DEBUG_CPUFREQ_TIME	
		begin_time = ktime_get();
#endif
			
		if(!send_msg_to_m0())
			/* wait axi freq changed ok! we will set a timeout for safety~ */
			wait_axi_ack(WAIT_AXI_ACK_TIMEOUT);		

#if DEBUG_CPUFREQ_TIME	
		end_time = ktime_get();
		total_time = ktime_to_us(ktime_sub(end_time, begin_time));
		pr_info("[CPUFREQ] total axi time: %d \n",(u32)total_time);		
#endif
    }	

//	debug_axi_clk_info(axi_freq);

	return 0;
}

static int axi_freq_change_allowed(void)
{
	/* can not change axi freq */
	
	return true;
}

/**
 * input cpu freq [KHz].
 */
static zx297510_axi_freq request_axi_freq_by_cpu(unsigned int freq)
{
	if(freq >= 600*1000)
		return AXI_FREQ_156M;
	else if(freq >= 400*1000)
		return AXI_FREQ_104M;
	else
		return AXI_FREQ_78M;
}

/**
 * set axi freq . 
 *
 * we will do this by M0.
 */
static int set_axi_frequency_by_cpu(unsigned int freq)
{
	zx297510_axi_freq axi_freq = request_axi_freq_by_cpu(freq);
	
	if(!axi_freq_change_allowed())
		return 0;

	/* set new freq */
	return request_axi_freq(axi_freq);
}
#endif

static int zx297510_set_frequency(unsigned int old_index,
				     unsigned int new_index);

#define CPUFREQ_LEVEL_END	L5

static int max_support_idx = L0;
static int min_support_idx = (CPUFREQ_LEVEL_END - 1);
static int cpufreq_driver_inited = 0;
static struct clk *cpu_clk;

/* in fact, zx297510 can not adjust core voltage, we reserved it for future if it be... */
static unsigned int zx297510_volt_table[CPUFREQ_LEVEL_END] = {
	1250000, 1150000, 1050000, 975000, 950000,
};

static struct cpufreq_frequency_table zx297510_freq_table[] = {
	{L0, 800*1000},
	{L1, 624*1000},
	{L2, 400*1000},
	{L3, 156*1000},
	{L4, 26*1000},
	{0, CPUFREQ_TABLE_END},
};

#ifdef CONFIG_ZX_PM_DEBUG

/* for debug clock state */
static char * ufi_pll_str[]=
{
	"pll_ufi",		
	"pll_main",	
	"pll_624",
	"pll_ps",
};
static void debug_cpu_clk_info(void)
{
	unsigned 	cpu_clock;
	char * 		pll_used;
	unsigned 	ufi_clk_div;

	ufi_clk_div = 1 << (pm_read_reg(TOP_UFI_DIV_REG)&0x3);
	pll_used 	= ufi_pll_str[pm_read_reg(TOP_UFI_SEL_REG)&3];
	cpu_clock	= clk_get_rate(cpu_clk) / 1000;
	
	pr_info("cpu_clock[KHz]   	pll_used       ufi_clk_div\n");	
	pr_info("%d   	          %s           %d\n", cpu_clock, pll_used, ufi_clk_div);	
}

static struct zx_dvfs_info test_dvfs_info;
void cpufreq_test(unsigned int old_index, unsigned int new_index)
{
	zx297510_cpufreq_init(&test_dvfs_info);

	zx297510_set_frequency(old_index, new_index);
}

#else
static void debug_cpu_clk_info(void){}
static void debug_axi_clk_info(unsigned req_axi){}
#endif


/**
 * set freq according to index of freq_table. 
 * 
 */
static int zx297510_set_frequency(unsigned int old_index,
				     unsigned int new_index)
{
	int	ret = 0;

	if(old_index == new_index)
		return ret;

	ret = clk_set_rate(cpu_clk, zx297510_freq_table[new_index].frequency * 1000);
	if (ret) 
		pr_info("[CPUFREQ] Failed to set rate %dkHz: %d\n", zx297510_freq_table[new_index].frequency, ret);

	/* set pcu pll used */
	pm_set_pll_used();	

	pr_info("[CPUFREQ] set cpufreq:old index:%d new index:%d \n", old_index, new_index);
	debug_cpu_clk_info();

#ifdef CONFIG_AXI_FREQ
	mutex_lock(&axifreq_lock);
	set_axi_frequency_by_cpu(zx297510_freq_table[new_index].frequency);
	mutex_unlock(&axifreq_lock);	
#endif	
		
	return ret;	
}

int zx297510_cpufreq_init(struct zx_dvfs_info *info)
{
	if(cpufreq_driver_inited)
		return 0;

	cpu_clk = clk_get(NULL, "cpu_clk");
	if (IS_ERR(cpu_clk))
	{
		pr_info("[CPUFREQ] get cpu_clk error \n");
		return PTR_ERR(cpu_clk);
	}

	info->freq_cur_idx 		= L2;
	info->pll_safe_idx 		= L2;
	info->max_support_idx 	= max_support_idx;
	info->min_support_idx 	= min_support_idx;
	info->cpu_clk 			= cpu_clk;
	info->volt_table 		= zx297510_volt_table;
	info->freq_table 		= zx297510_freq_table;
	info->set_freq 			= zx297510_set_frequency;

	cpufreq_driver_inited = 1;

	return 0;
}
EXPORT_SYMBOL(zx297510_cpufreq_init);

#ifdef CONFIG_AXI_FREQ
/**
 * zx_axifreq_pm_notifier - acquire axifreq in suspend-resume context
 *			
 * @notifier
 * @pm_event
 * @v
 *
 */
static int zx_axifreq_pm_notifier(struct notifier_block *notifier,
				       unsigned long pm_event, void *v)
{
	mutex_lock(&axifreq_lock);
	
	switch (pm_event) 
	{
	case PM_SUSPEND_PREPARE:
		request_axi_freq(AXI_FREQ_39M);
		break;

	case PM_POST_SUSPEND:
		request_axi_freq(AXI_FREQ_104M);
		break;
	}
	
	mutex_unlock(&axifreq_lock);

	return NOTIFY_OK;
}

static struct notifier_block zx_axifreq_nb = 
{
	.notifier_call = zx_axifreq_pm_notifier,
};

static int __init zx297510_axifreq_init(void)
{
	int ret = 0;

	ret = zDrvRpMsg_CreateChannel(M0_ID, channel_1, 0x20);
	if(ret)
	{
		pr_info("[CPUFREQ] Failed create psm icp channel, err:%d ! \n", ret);
		return -EPERM;
	}
	pr_info("[CPUFREQ] Success create psm icp channel!!! \n");

	/* pm notify */
	register_pm_notifier(&zx_axifreq_nb);

	request_axi_freq(AXI_FREQ_104M);		

	return 0;
}

late_initcall(zx297510_axifreq_init);
#endif	

