/*
 * zx297510 CPU idle Routines
 *
 * Copyright (C) 2013 ZTE, Ltd.
 * Shine Yu <yu.xiang5@zte.com.cn>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/sched.h>
#include <linux/export.h>
#include <linux/kernel.h>
#include <linux/cpu.h>
#include <linux/tick.h>
#include <linux/suspend.h>
#include <linux/interrupt.h>

#include <mach/zx-pm.h>

#define	WHOLE_CHIP_EXIT_LATENCY		(4000) 		/* us */

#define LP2_DEFAULT_EXIT_LATENCY	(1600) 		/* us */
#define	LP2_MIN_POWER_OFF_TIME		(500)		/* us */

#define LP2_DELTA_EXIT_LATENCY		(100) 		/* us -- for timer setting refresh time, should > 2us.
                                                         Donnot modify this. */

static s64 	zx_idle_sleeptime = 0;
static struct cpuidle_driver *cur_idle_drv;

#ifdef CONFIG_ZX_PM_DEBUG
extern struct zx_idle_stats 	idle_stats;
#endif

static struct cpuidle_state zx297510_cpuidle_set[] __initdata =
{
	/* LP3 -- wfi */
	[ZX_IDLE_CSTATE_LP3] =
	{
		.enter				= zx_enter_idle,
		.exit_latency		= 2,
		.target_residency	= 5,
		.flags				= CPUIDLE_FLAG_TIME_VALID,
		.name				= "LP3",
		.desc				= "clock gating(WFI)",
	},
	/* LP2 -- dormant  */
	[ZX_IDLE_CSTATE_LP2] =
	{
		.enter				= zx_enter_idle,
		.exit_latency		= LP2_DEFAULT_EXIT_LATENCY,
		.target_residency	= LP2_DEFAULT_EXIT_LATENCY+LP2_MIN_POWER_OFF_TIME/*+WHOLE_CHIP_EXIT_LATENCY*/,
		.flags				= CPUIDLE_FLAG_TIME_VALID,
		.name				= "LP2",
		.desc				= "DORMANT",
	},
};

int __init zx_fill_cpuidle_data(struct cpuidle_driver *drv)
{
	int i, max_cpuidle_state;

	max_cpuidle_state = sizeof(zx297510_cpuidle_set) / sizeof(struct cpuidle_state);

	for (i = 0; i < max_cpuidle_state; i++)
		memcpy(&drv->states[i], &zx297510_cpuidle_set[i],	sizeof(struct cpuidle_state));

	drv->safe_state_index = ZX_IDLE_CSTATE_LP3;

	cur_idle_drv = drv;

	return max_cpuidle_state;
}


/**
 * 	idle_update_sleep_param
 *
 *	when exit from one lp2 level sleep, update the exit_latency/target_residency.
 */
/* our idle exit lattency may */
#if 0
static unsigned int lp2_exit_latencies[MAX_CPU_NUM];
static void idle_update_sleep_param(void)
{

}
#endif
/**
 * 	idle_set_sleeptime
 *
 *	set the wakeup timer
 *
 *  sleep_time  (us)
 */
extern void setup_timer_wakeup(s64 us);
static void idle_set_sleeptime(s64 sleep_time)
{
	/* set timer */
	setup_timer_wakeup(sleep_time);

	zx_idle_sleeptime = sleep_time;
}

/**
 * 	idle_get_sleeptime
 *
 *	for PCU sleeptime
 */
s64 idle_get_sleeptime(void)
{
	return zx_idle_sleeptime;
}

/**
 * 	idle_unmask_interrupt
 *
 *
 */
static void idle_unmask_interrupt(void)
{
}

static void idle_unmask_interrupt_restore(void)
{
}

static unsigned int idle_get_exit_latency(int index)
{
	struct cpuidle_state* state = &(cur_idle_drv->states[index]);

	return state->exit_latency;
}

/**
 * 	zx_enter_deep_idle
 *
 *	enter lp2 mode
 */
extern s64 pm_get_remainder_time(void);

static int zx_enter_deep_idle(int index)
{
	bool sleep_completed = false;
	pm_wake_reason_t wake_reason;
	s64 sleep_time;
	s64 request = ktime_to_us(tick_nohz_get_sleep_length());
//	s64 remainder_timer = pm_get_remainder_time();
	ktime_t entry_time, exit_time;
	s64 idle_time;
	entry_time = ktime_get();

#ifdef CONFIG_ZX_PM_DEBUG
	unsigned cpu = read_cpuid();

	//idle_stats.cpu_ready_count[cpu]++;
 	idle_stats.tear_down_count[cpu]++;
#endif

//	pr_info("[SLP] request:%d remainder: %d\n",(u32)request, (u32)remainder_timer);
/*=================================================================
 *=======begin enter deep sleep====================================
 *=================================================================
 */
	/*  */
	idle_unmask_interrupt();

	/* set wakeup timer */
	sleep_time = request - idle_get_exit_latency(index);
	if (sleep_time > LP2_DELTA_EXIT_LATENCY)
		idle_set_sleeptime(sleep_time);
	else
		return zx_pm_idle_enter(ZX_IDLE_CSTATE_LP3);

#ifdef CONFIG_ZX_PM_DEBUG
	idle_stats.lp2_count[cpu]++;
#endif

	/* sleep */
	zx_enter_sleep(CPU_SLEEP_TYPE_IDLE_LP2);

	/* get wakeup cause */
	wake_reason = pm_get_wakeup_reason();
	if (wake_reason != WR_WAKE_SRC_ABNORMAL)
	{
		sleep_completed = true;
	}
	else
	{
#ifdef CONFIG_ZX_PM_DEBUG
		int irq = 0;
		irq = zx29_gic_pending_interrupt();
		idle_stats.lp2_int_count[irq]++;
#endif
	}

	/*  */
	idle_unmask_interrupt_restore();
/*=================================================================
 *=======end enter deep sleep======================================
 *=================================================================
 */
	exit_time = ktime_get();
	idle_time = ktime_to_us(ktime_sub(exit_time, entry_time));

#ifdef CONFIG_ZX_PM_DEBUG

	//idle_stats.cpu_wants_lp2_time[cpu] += idle_time;
	idle_stats.in_lp2_time[cpu] += idle_time;

	if (sleep_completed)
		idle_stats.lp2_completed_count[cpu]++;
#endif

//	pr_info("[CPUIDLE] idle time: request:%d  \n",(u32)request);

	return index;
}

/**
 * zx_enter_lowpower - Programs cpu to enter the specified state
 * @dev: cpuidle device
 * @state: The target state to be programmed
 *
 * Called from the CPUidle framework to program the device to the
 * specified low power state selected by the governor.
 * Called with irqs off, returns with irqs on.
 * Returns the amount of time spent in the low power state.
 */
int zx_pm_idle_enter(int index)
{
#ifdef CONFIG_ZX_PM_DEBUG
	idle_stats.idle_count++;
#endif
	if(ZX_IDLE_CSTATE_LP2 == index)
	{
		return zx_enter_deep_idle(index);
	}
	else
	{
#ifdef CONFIG_ZX_PM_DEBUG
		unsigned cpu = read_cpuid();
		idle_stats.lp3_count[cpu]++;
#endif
		cpu_do_idle();
		return ZX_IDLE_CSTATE_LP3;
	}
}
