/*
 * arch/arm/mach-zx297510/include/mach/timex.h
 *
 *  Copyright (C) 2013 ZTE-TSP
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#ifndef __ASM_ARCH_TIMEX_H
#define __ASM_ARCH_TIMEX_H

#include <mach/board.h>
#include <asm/io.h>

#define CLOCK_TICK_RATE		(3250000)

/*  timer register offset  */
#define CONFIG_REG   				(0x04)
#define LOAD_REG     				(0x08)
#define START_REG    				(0x0C)
#define REFRESH_REG  				(0x10)
#define ACK_REG  					(0x14)
#define CUR_VALUE    				(0x18)

#define TIMER_PRESCALE(ptv)			(ptv << 5)
#define TIMER_AUTORELOAD			(1 << 1)


/*
 *  name       power domain  wakeup   clock  	usage
 *
 * ap_timer0	PD_CORE	               26M		ϵͳtick
 * ap_timer4	PD_CORE	               26M		
 * ap_timer3	PD_AO	               32K		clock source,ʡʱ䲹 (persistent_timer)
 * ap_timer1	PD_AO	     pcu	   32K		psm wakeup(idle)
 * ap_timer2	PD_AO	     pcu	   26M		udelay
 *
 */

#define SOURCE_CLK_PERSISTENT		1

/*when use fpga platform, we can only use 32K*/
#ifdef CONFIG_ARCH_ZX297520V2FPGA
#define SOURCE_CLOCK_RATE 			(32768)
#define EVENT_CLOCK_RATE  			(32768)
#else
#if SOURCE_CLK_PERSISTENT
#define SOURCE_CLOCK_RATE 			(32768)
#else
#define SOURCE_CLOCK_RATE 			(3250000)     /* 26M/8/1 */
#endif
#define EVENT_CLOCK_RATE  			(3250000)     /* 26M/8/1 */
#define DELAY_CLOCK_RATE  			(26000000)     /* 26M */
#endif
#define PERSISTENT_TIMER_CLOCK_RATE (32768)
#define WAKE_TIMER_CLOCK_RATE  		(32768)

#if SOURCE_CLK_PERSISTENT
#define CLOCKSOURCE_NAME  			"zx29_ap_timer3"
#else
#define CLOCKSOURCE_NAME  			"zx29_ap_timer4"
#endif
#define CLOCKEVENT_NAME   			"zx29_ap_timer0"
#define PERSISTENT_TIMER_NAME  		"zx29_ap_timer3"
#define WAKE_TIMER_NAME   			"zx29_ap_timer1"

#define DELAY_TIMER_NAME   			"zx29_ap_timer2"

#if SOURCE_CLK_PERSISTENT
#define CLOCKSOURCE_BASE 			(ZX_AP_TIMER3_BASE)
#else
#define CLOCKSOURCE_BASE 			(ZX_AP_TIMER4_BASE)
#endif
#define CLOCKEVENT_BASE  			(ZX_AP_TIMER0_BASE)
#define PERSISTENT_TIMER_BASE 		(ZX_AP_TIMER3_BASE)
#define WAKE_TIMER_BASE 			(ZX_AP_TIMER1_BASE)
#define CLOCKDELAY_BASE 			(ZX_AP_TIMER2_BASE)

#define CLOCKEVENT_INT  			(AP_TIMER0_INT)
#define WAKE_TIMER_INT  			(AP_TIMER1_INT)
#define PCU_WAKE_TIMER_INT  		(PCU_AP_TIMER1_INT)

#if NEW_LINUX_FRAME
#define	timer_get_clk(dev, con)		clk_get(dev, con)
#else
#define	timer_get_clk(dev, con)		clk_get_sys(dev, con)
#endif

/*
 *refresh configure register
 */
static inline void refresh_config_reg(void __iomem *base)
{
	unsigned int tmp=0;
	unsigned long flags;
	volatile int cnt;

	local_irq_save(flags);

	tmp=ioread32(base+REFRESH_REG);
	
	while(((ioread32(base+ACK_REG)>>4) & 0xf) != (tmp & 0xf))
	{
		cnt = 10;
		while(cnt--);
	}

	tmp ^=0xC;
	iowrite32(tmp,base+REFRESH_REG);

	local_irq_restore(flags);
	
}

/*
 *refresh load register
 */
static inline void refresh_load_reg(void __iomem *base)
{
	unsigned int tmp=0;
	unsigned long flags;
	volatile int cnt;

	local_irq_save(flags);

	tmp=ioread32(base+REFRESH_REG);

	while(((ioread32(base+ACK_REG)>>4) & 0xf) != (tmp & 0xf))
	{
		cnt = 10;
		while(cnt--);
	}

	tmp ^= 0x3;
	iowrite32(tmp,base+REFRESH_REG);

	local_irq_restore(flags);

}

/*
 *refresh load register and configure register
 */
static inline void refresh_config_load_reg(void __iomem *base) 
{
	unsigned int tmp=0;
	unsigned long flags;
	volatile int cnt;

	local_irq_save(flags);

	tmp=ioread32(base+REFRESH_REG);

	while(((ioread32(base+ACK_REG)>>4) & 0xf) != (tmp & 0xf))
	{
		cnt = 10;
		while(cnt--);
	}
	
	tmp ^= 0xf;
	iowrite32(tmp,base+REFRESH_REG);

	local_irq_restore(flags);
	
}

/*
 * get cur_value of the timer
 */
static inline u32 read_timer_clk(void __iomem *base)
{
	return ioread32(base+CUR_VALUE);
}

/*
 * set reload mode
 */
static inline void timer_set_mode(void __iomem *base, bool periodic)
{
	unsigned int tmp = ioread32(base+CONFIG_REG);
	
	if(periodic)
		tmp |= TIMER_AUTORELOAD;
	else
		tmp &= ~TIMER_AUTORELOAD;
	iowrite32(tmp, base+CONFIG_REG);

	refresh_config_reg(base);
}

/*
 * start timer
 */
static inline void timer_start(void __iomem *base)
{
	iowrite32(0x1, base+START_REG);
}

/*
 * stop timer
 */
static inline void timer_stop(void __iomem *base)
{
	iowrite32(0x0, base+START_REG);
}

/*
 * set load value(unit is cycle)
 */
static inline void timer_set_load(void __iomem *base, u32 load_val)
{
	iowrite32(load_val, base+LOAD_REG);
	refresh_load_reg(base);
}

/*
 * set prescale, default is 0
 * 1--div 2   2--div 4 ...
 */
static inline void timer_set_prescale(void __iomem *base, u32 prescale)
{
	unsigned int tmp = ioread32(base+CONFIG_REG);

	tmp &= ~TIMER_PRESCALE(0x1F);
	tmp |= TIMER_PRESCALE(prescale);

	iowrite32(tmp, base+CONFIG_REG);

	refresh_config_reg(base);
}
#endif


