/**
 * Copyright (C) 2017 Sanechips Technology Co., Ltd.
 * @author Yanan Liu <liu.yanan@sanechips.com.cn>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

/*******************************************************************************
 *                           Include header files                              *
 ******************************************************************************/

#define _GNU_SOURCE


#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <sys/prctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/vfs.h>
#include <dirent.h>
#include <fcntl.h>

#include <assert.h>
#include <errno.h>
#include <unistd.h>
#include <time.h>
#include <netdb.h>



#include "cfg_api.h"
#include "fota_dm_utils.h"
#include "fota_dm_macros.h"
#include "fota_common.h"
#include "rtc_timer.h"
#include "softap_api.h"
/*******************************************************************************
 *                             Macro definitions                               *
 ******************************************************************************/
#define OPT_RECURSIVE 1
#define OPT_FORCE 2

#define SYS_UNLOCK_PATH "/sys/power/wake_unlock"
#define SYS_LOCK_PATH  "/sys/power/wake_lock"
#define NS_PER_SECONDS 1000000000UL

#define FOTA_DM_TIME_OUT_WAKE_LOCK "fota_dm_timeout"
#define FOTA_WAKE_LOCK_AUTO_TIMEOUT (3)

/*******************************************************************************
 *                             Type definitions                                *
 ******************************************************************************/
#define FOTA_DM_TIMER_ID 0
/*******************************************************************************
 *                        Local function declarations                          *
 ******************************************************************************/

/*******************************************************************************
 *                         Local variable definitions                          *
 ******************************************************************************/

/*******************************************************************************
 *                        Global variable definitions                          *
 ******************************************************************************/

/*******************************************************************************
 *                      Inline function implementations                        *
 ******************************************************************************/

/*******************************************************************************
 *                      Local function implementations                         *
 ******************************************************************************/
/*read file to buffer*/
int fota_dm_read_file(const char*path, char*buf, size_t sz)
{
    int fd = -1;
    size_t cnt;

    fd = open(path, O_RDONLY, 0);
    if(fd < 0) {
        LOGE("failed to open %s: %s\n", path, strerror(errno));
        cnt = -1;
        goto err;
    }
    cnt = read(fd, buf, sz - 1);
    if(cnt <= 0) {

        LOGE("failed to read %s: %s\n", path, strerror(errno));
        cnt = -1;
        goto err;
    }
    buf[cnt] = '\0';
    if(buf[cnt - 1] == '\n') {
        cnt--;
        buf[cnt] = '\0';
    }
err:
    if(fd >= 0)
        close(fd);
    return cnt;
}

/*read an integer from file*/
int fota_dm_read_file_int(const char* path, int *val)
{
    char buf[32];
    int ret ;
    int tmp;
    char *end;
    ret = fota_dm_read_file(path, buf, sizeof(buf));
    if(ret < 0)
        return -1;
		
	// kw CXX.ERRNO.NOT_CHECKED
	errno = 0;
    tmp = strtol(buf, &end, 0);
	if (errno == ERANGE) {
		/* Handle error */
		LOGE("strtol errno %d: %s\n", errno, strerror(errno));
//		return -1;
	} 
	
#if 0	
    if((end == buf) || ((end < buf + sizeof(buf)) && (*end != '\0'
                        && *end != '\0')))
        goto err;
#else
    if((end == buf) || ((end < buf + sizeof(buf)) && (*end != '\0'))){
         goto err;
	}
                      
    
#endif	
    *val = tmp;
    return 0;
err:
    return -1;
}
/*write string to file*/
int fota_dm_write_file(const char*path, const char*value, int size)
{
    int fd = open(path, O_WRONLY | O_CREAT | O_SYNC, 777);
    if(fd < 0) {
        LOGE("failed to open %s: %s\n", path, strerror(errno));
        return -1;
    }
    if(write(fd, value, size) != size) {
        LOGE("failed to write %s:%s\n", path, strerror(errno));
        close(fd);
        return -1;
    }
    close(fd);
    return 0;
}
/*write int to file */
int fota_dm_write_file_int(const char*path, int value)
{
    char buffer[32]  = {0};
    sprintf(buffer, "%d", value);
    return fota_dm_write_file(path, buffer, strlen(buffer));

}
/*delete file or dir recursive*/
int unlink_recursive(const char*name, int flags)
{
    struct stat st;
    DIR *dir;
    struct dirent *de;
    int fail = 0;
    if(lstat(name, &st) < 0)
        return ((flags & OPT_FORCE)  && errno == ENOENT) ? 0 : -1;
    /* not a dir, unlink*/
    if(! S_ISDIR(st.st_mode))
        return unlink(name);
    /*dir, recursive unlink*/
    dir = opendir(name);
    if(dir == NULL)
        return -1;
    errno = 0;
    while((de = readdir(dir)) != NULL) {
        char dn[PATH_MAX];
        if(!strcmp(de->d_name, ".") || !strcmp(de->d_name, ".."))
            continue;
        snprintf(dn, PATH_MAX, "%s/%s", name, de->d_name);
        if(unlink_recursive(dn, flags) < 0) {
            if(!(flags & OPT_FORCE)) {
                fail = 1;
                break;
            }
        }
        errno = 0;
    }
    if(fail || errno < 0) {
        int save = errno;
        closedir(dir);
        errno = save;
        return -1;
    }
    if(closedir(dir) < 0)
        return -1;
    return rmdir(name);
}

/*******************************************************************************
 *                      Global function implementations
*******************************************************************************/

/*set process name*/
void fota_dm_set_process_name(const char* name)
{
    if((name == NULL) || (*name == '\0'))
        return;
    prctl(PR_SET_NAME, name, 0, 0, 0);
}
/*set thread name*/
void fota_dm_set_thread_name(const char*name)
{
    /*we should use pthread_setname_np, but it is not portable, user prctl instead*/
    fota_dm_set_process_name(name);
}
/* mkdir recursive*/
int fota_dm_mkdirs(const char* path, mode_t mode)
{
    char buf[128];
    const char *slash;
    const char *p = path;
    int width;
    int ret;

    struct stat info;
    while((slash = strchr(p, '/')) != NULL) {
        width = slash - path;
        p = slash + 1;
        if(width < 0)
            break;
        if(width == 0)
            continue;
        if((unsigned int)width > sizeof(buf) - 1) {
            assert(0);
            return -1;
        }
        memcpy(buf, path, width);
        buf[width] = 0;
        if(stat(buf, &info) != 0) {
            ret = mkdir(buf, mode);
            if(ret && errno != EEXIST)
                return ret;
        }
    }
    ret = mkdir(path, mode);
    if(ret && errno != EEXIST)
        return ret;
    return 0;
}
/*check if file or dir exists*/
int fota_dm_is_file_exist(const char* path)
{
    if((path == NULL) || (*path == '\0'))
        return FALSE;
    if(access(path, R_OK) != 0)
        return FALSE;
    return TRUE;
}

/* read nv string value */
int fota_dm_get_nv_string(char* nv_name, char* nv_value, size_t buf_size)
{
    char buff[FOTA_CFG_GET_ITEMLEN] = {0};
    if(NULL == nv_value || buf_size <= 0)
        return -1;
    sc_cfg_get(nv_name, buff, sizeof(buff));
    strncpy(nv_value, buff, buf_size - 1);
    nv_value[buf_size - 1] = '\0';
    return 0;
}
int fota_dm_set_nv_string(char *nv_name, char* nv_value)
{
    sc_cfg_set(nv_name, nv_value);
    return 0;
}

/* read nv int value*/
int fota_dm_get_nv_int(char* nv_name)
{
    char buff[FOTA_CFG_GET_ITEMLEN] = {0};
    int value = 0;
    sc_cfg_get(nv_name, buff, sizeof(buff));
    value = atoi(buff);
    return value;
}
/* set an int value to nv*/
int fota_dm_set_nv_int(char* nv_name, int value)
{
    char buff[FOTA_CFG_GET_ITEMLEN] = {0};
    sprintf(buff, "%d", value);
    sc_cfg_set(nv_name, buff);
    return 0;
}
int fota_dm_set_nv_long(char* nv_name, long value)
{
    char buff[FOTA_CFG_GET_ITEMLEN] = {0};
    sprintf(buff, "%ld", value);
    sc_cfg_set(nv_name, buff);
    return 0;
}

/*check roaming status*/
static NetworkStatus fota_dm_get_roaming_status(void)
{
    char status[FOTA_CFG_GET_ITEMLEN] = {0};
    fota_dm_get_nv_string(NV_SIMCARD_ROAM, status, sizeof(status));
    if(strcmp(ROAM_INTERNATIONAL, status) == 0) {
        return ROAMING_INTERNATIONAL;
    }
    if(strcmp(ROAM_INTERNAL, status) == 0) {
        return ROAMING_INTERNAL;
    }
    if(strcmp(ROAM_HOME, status) == 0) {
        return ROAMING_HOME;
    }
    return ROAMING_NA;
}

/*check is roaming*/
int fota_dm_is_roaming(void)
{
    NetworkStatus status = fota_dm_get_roaming_status();
    if(status == ROAMING_INTERNAL || status == ROAMING_INTERNATIONAL) {
        return TRUE;
    }
    return FALSE;
}

/*if check/download version allowed in roaming*/
int fota_dm_allow_roaimg_update(void)
{
    int value = fota_dm_get_nv_int(NV_DM_FOTA_ALLOW_ROAMING);
    return value == 1 ? TRUE : FALSE;
}

//get voltage value
int fota_dm_get_voltage(int *voltage)
{
    int tmp = 0;
    int ret = 0;

    ret = fota_dm_read_file_int(CHARGER_VOLTAGE_FILE, &tmp);
    if(ret < 0) {

        *voltage = 0;
        return -1;
    }

    *voltage = tmp;
    return 0;
}

//check whether enough battery for upgrade
int fota_dm_is_power_enough(void)
{
    int vol = 0 ;
    int ret = 0;
    if(!fota_dm_is_file_exist(CHARGER_VOLTAGE_FILE)) {
        /*MDL or the device is charging, no CHARGER_VOLTAGE_FILE exist*/
        LOGD("MDL or device is charging, CHARGER_VOLTAGE_FILE not exist\n");
        return TRUE;
    } else {
        ret = fota_dm_get_voltage(&vol);
        if(ret < 0)
            return FALSE;
        return ((vol > 3570) ? TRUE : FALSE);
    }
}

//get update status from file
int fota_dm_get_update_status(int *fota_status)
{

    int status = 0;
    int ret = 0;
    if(!fota_dm_is_file_exist(FOTA_UPDATE_STATUS_FILE)) {
        *fota_status = -1;
        return -1;
    }
    ret = fota_dm_read_file_int(FOTA_UPDATE_STATUS_FILE, &status);
    if(ret < 0) {
        *fota_status = -1;
        return -1;
    }
    *fota_status = status;
    return 0;
}

//clear update status in file
int fota_dm_clear_update_status(void)
{
    LOGD("clear update status in /cache/zte_fota/update_status to ZTE_DUA_NO_NEED_UPDATE \n");
    return fota_dm_write_file_int(FOTA_UPDATE_STATUS_FILE, ZTE_DUA_NO_NEED_UPDATE);

}

//get IMEI from NV
int fota_dm_get_imei(char* imei, size_t size)
{
    return fota_dm_get_nv_string(NV_IMEI, imei, size);
}
//get TOKEN from NV
int fota_dm_get_fota_token(char *fota_token , size_t size, char* suffix)
{
    char buf[64] = {0};
    snprintf(buf, 64, "%s%s", NV_FOTA_TOKEN, (suffix!=NULL) ? suffix:"");
    return fota_dm_get_nv_string(buf, fota_token, size);
}
int fota_dm_get_dl_url(char* url, size_t size, char* suffix)
{
    char buf[64] = {0};
    snprintf(buf, 64, "%s%s", NV_FOTA_DL_URL, (suffix!=NULL) ? suffix:"");
    return fota_dm_get_nv_string(buf, url, size);
}
int fota_dm_get_chk_url(char* url, size_t size, char *suffix)
{
    char buf[64] = {0};
    snprintf(buf, 64, "%s%s", NV_FOTA_CHK_URL, (suffix!=NULL) ? suffix:"");
    return fota_dm_get_nv_string(buf, url, size);
}
int fota_dm_get_reg_url(char* url, size_t size, char* suffix)
{
    char buf[64] = {0};
    snprintf(buf, 64, "%s%s", NV_FOTA_REG_URL, (suffix!=NULL) ? suffix:"");
    return fota_dm_get_nv_string(buf, url, size);
}
int fota_dm_get_report_dlr_url(char* url, size_t size, char* suffix)
{
    char buf[64] = {0};
    snprintf(buf, 64, "%s%s", NV_FOTA_REPORT_DLR_URL, (suffix!=NULL) ? suffix:"");
    return fota_dm_get_nv_string(buf, url, size);
}
int fota_dm_get_report_upgr_url(char* url, size_t size, char* suffix)
{
    char buf[64] = {0};
    snprintf(buf, 64, "%s%s", NV_FOTA_REPORT_UPGR_URL, (suffix!=NULL) ? suffix:"");
    return fota_dm_get_nv_string(buf, url, size);
}
int fota_dm_get_report_sales_url(char* url, size_t size, char* suffix)
{
    char buf[64] = {0};
    snprintf(buf, 64, "%s%s", NV_FOTA_REPORT_SALES_URL, (suffix!=NULL) ? suffix:"");
    return fota_dm_get_nv_string(buf, url, size);
}

//get inner version from NV
int fota_dm_get_inner_version(char* version, size_t size)
{
    return fota_dm_get_nv_string(NV_CR_INNER_VERSION, version, size);
}
//get OEM info from NV
int fota_dm_get_fota_oem(char* fota_oem, size_t size)
{
    return fota_dm_get_nv_string(NV_FOTA_OEM, fota_oem, size);
}


//get device type from NV
int fota_dm_get_fota_device_type(char *fota_device_type, size_t size)
{
    return fota_dm_get_nv_string(NV_FOTA_DEVICE_TYPE, fota_device_type, size);
}
//get platform from NV
int fota_dm_get_platform(char *fota_platform, size_t size)
{
    return fota_dm_get_nv_string(NV_FOTA_PLATFORM, fota_platform, size);
}
//get models from NV
int fota_dm_get_models(char * fota_models, size_t size)
{
    return fota_dm_get_nv_string(NV_FOTA_MODELS, fota_models, size);
}



//get product_id from NV
int fota_dm_get_product_id(char * fota_product_id, size_t size)
{
    return fota_dm_get_nv_string(NV_FOTA_PRODUCTID, fota_product_id, size);
}

//get product_secret from NV
int fota_dm_get_product_secret(char * fota_product_secret, size_t size)
{
    return fota_dm_get_nv_string(NV_FOTA_PRODUCTSECRET, fota_product_secret, size);
}

//get app_version from NV
int fota_dm_get_app_version(char * fota_app_version, size_t size)
{
    return fota_dm_get_nv_string(NV_FOTA_APPVERSION, fota_app_version, size);
}

//get network_type from NV
int fota_dm_get_network_type(char * fota_network_type, size_t size)
{
    return fota_dm_get_nv_string(NV_FOTA_NETWORKTYPE, fota_network_type, size);
}

//remove files or dirs
int fota_dm_remove(const char*path)
{
    return unlink_recursive(path, OPT_FORCE | OPT_RECURSIVE);
}


//call zxic_fota to verify packages
int  fota_dm_verify_package(void)
{
    system("/usr/bin/free");
	if(get_modem_info("AT+CFUN=0\r\n", NULL, NULL) < 0)  // cov M CHECKED_RETURN
	{
	    LOGE("get_modem_info AT+CFUN=0 fail!\n");
	}
	system("echo 7 > /proc/sys/vm/drop_caches");
	system("/usr/bin/free");
	system("rm -rf /var/log");
    return system("fota_upi  --upgrade  verify");
}
//call zxic_fota to update recovery
int fota_dm_update_recoveryfs(void)
{
    return system("fota_upi --upgrade  recovery");
}
//get update_flag from NV
int fota_dm_get_update_flag(int *fota_flag)
{
    int ret = fota_dm_get_nv_int(NV_FOTA_UPGRADE_FLAG);
    *fota_flag = ret;
    return 0;
}
//set update_flag to NV
void fota_dm_set_update_flag(void)
{
    fota_dm_set_nv_int(NV_FOTA_UPGRADE_FLAG, 1);
}
//clear update_flags in NV
void fota_dm_clear_update_flag(void)
{
    fota_dm_set_nv_int(NV_FOTA_UPGRADE_FLAG, 0);
}


//get free space size in bytes
size_t  fota_dm_get_free_space(const char* path)
{
    struct statfs sf;
    if(statfs(path, &sf) != 0) {
        LOGE("failed to statfs %s:%s\n", path, strerror(errno));
        return -1;
    }
    return sf.f_bsize * sf.f_bfree;
}

//rename files
int fota_dm_rename_file(const char*oldpath, const char*newpath)
{
    int ret = 0;
    if(NULL == oldpath || NULL == newpath)
        return 0;
    ret = rename(oldpath, newpath);
    if(ret < 0) {
        if(errno == EXDEV) {
            LOGD("cross devices rename\n");
            //TODO:cross devices rename
        }
        LOGE("failed to rename %s->%s:%s\n",  oldpath, newpath, strerror(errno));
        return -1;
    }
    return 0;
}

int fota_dm_sync_file(char *file_path)
{
	int file_fd = -1;
	int ret  = -1;

	if (NULL == file_path) {
		LOGE("Input para is invalid, null point!");
		return -1;
	}


	file_fd = open(file_path, O_RDWR | O_SYNC);
	if (file_fd < 0) {
		LOGE("Open file fail, error:%s, file path:%s", strerror(errno), file_path);
		return -1;
	}


	ret = fsync(file_fd);
	if (ret < 0) {
		LOGE("Sync Failed:%s, file path:%s", strerror(errno), file_path);
		goto error;
	}


	ret = 0;
	goto end;

error:
	ret = -1;

end:
	close(file_fd);

	file_fd = -1;

	return ret;
}



//get file size in bytes
size_t fota_dm_get_file_size(const char* path)
{
    struct stat statbuf = {0};
    if(stat(path, &statbuf) != 0) {
        return 0;
    }
    return statbuf.st_size;
}

//reboot system
void fota_dm_reboot(void)
{
	//رӦ쳣˳
	if(access("/sbin/app_monitor.sh", F_OK) == 0)
   	{
        system("app_monitor.sh off");
   	}
    system("reboot");
}


int fota_dm_is_network_avaiable(void)
{
#if 0
    int ret = default_route_check();
    if(ret == 1)
        return TRUE;
#else
    unsigned long ip = 0;
    struct hostent *host = gethostbyname(FOTA_DLSVR_DOMAIN);
    if(host != NULL) {
        ip = *((unsigned long*)host->h_addr);
        if(ip != 0)
            return TRUE;
    }
#endif
    return FALSE;
}

int fota_dm_wait_network_ready(int time_interval, int retry_times)
{
    while(retry_times--) {
        if(fota_dm_is_network_avaiable() == TRUE)
            return TRUE;
        sleep(time_interval);
    }
    return FALSE;
}


int fota_dm_is_auto_polling_on(void)
{
    int value = fota_dm_get_nv_int(NV_DM_POLLINGSWITCH);
    return value == 1 ? TRUE : FALSE;
}
int fota_dm_is_poweron_auto_check(void)
{
    int value = fota_dm_get_nv_int(NV_FOTA_PWRON_AUTO_CHECK);
    return value == 1 ? TRUE : FALSE;
}

int fota_dm_add_rtc_alarm(int seconds, int alarm_id)
{
    int ret = 0;
    ret = rtc_timer_add(seconds, alarm_id , MODULE_ID_DM_WEBUI_AT);
    if(ret < 0)
        LOGE("failed to create rtc timer\n");
    LOGD("ret=%d, time interval=%d\n", ret, seconds);
    return ret;
}

int fota_dm_add_utc_alarm(struct tm* sec, int alarm_id)
{
    int ret = 0;
    ret = rtc_timer_add_utc(sec, alarm_id , MODULE_ID_DM_WEBUI_AT, 0);
    if(ret < 0)
        LOGE("failed to create utc alarm\n");
    LOGD("ret=%d, time=%s\n", ret, asctime(sec));
    return ret;
}
int fota_dm_del_alarm(int alarm_id)
{
    return rtc_timer_del(alarm_id, MODULE_ID_DM_WEBUI_AT);
}
int fota_dm_is_force_package(void)
{
    int value  = fota_dm_get_nv_int(NV_FOTA_VERSION_FORCE_INSTALL);
    return  value == 1 ? TRUE : FALSE;
}

int fota_dm_set_force_package(void)
{
    return fota_dm_set_nv_int(NV_FOTA_VERSION_FORCE_INSTALL, 1);
}

int fota_dm_clear_force_package(void)
{
    return fota_dm_set_nv_int(NV_FOTA_VERSION_FORCE_INSTALL, 0);
}


int fota_dm_set_download_progress(int progress)
{
    return fota_dm_set_nv_int(NV_FOTA_PKG_DL_SIZE, progress);
}



int fota_dm_wake_unlock(const char* lock_name)
{
    if(!lock_name)
        return -1;
    LOGD("%s\n", lock_name);
    return fota_dm_write_file(SYS_UNLOCK_PATH, lock_name, strlen(lock_name));
}

int fota_dm_wake_lock(const char* lock_name)
{

    if(!lock_name)
        return -1;
    LOGD("%s\n", lock_name);
    return fota_dm_write_file(SYS_LOCK_PATH, lock_name, strlen(lock_name));
}

int fota_dm_timeout_lock_sec(const char* lock_name, int sec)
{
    char buff[128] = {0};
    unsigned long interval = sec * NS_PER_SECONDS;
    if(!lock_name)
        return -1;
    LOGD("%s, %d\n", lock_name, sec);
    snprintf(buff, sizeof(buff) - 1, "%s %lu", lock_name, interval);
    LOGD("buff:%s\n", buff);
    return fota_dm_write_file(SYS_LOCK_PATH, buff, strlen(buff));
}


int fota_dm_timeout_wakelock()
{
	return fota_dm_timeout_lock_sec(FOTA_DM_TIME_OUT_WAKE_LOCK, FOTA_WAKE_LOCK_AUTO_TIMEOUT);
}



