/**
 * @file flags_api.c
 * @brief flagsӿʵ
 *
 * Copyright (C) 2023 Sanechips Technology Co., Ltd.
 * @author 
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation. ѡGPLv2 Licence
 *
 */


/*******************************************************************************
 *                           Include header files                              *
 ******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <mtd/mtd-abi.h>

#include "pub_flags.h"

#include "flags_log.h"
#include "flags_api.h"


/*******************************************************************************
 *                             Macro definitions                               *
 ******************************************************************************/
#define MAX_PATH_LEN (256)

#define PARTITION_NAME_FLAGS "flags"

#define FLAGS_INIT_VALID_BLOCKS_NUM	(8)

#define GOOD_BLOCK (0)
#define BAD_BLOCK (1)

#define FILE_PATH_PROC_CMDLINE "/proc/cmdline"

#define PROC_CMDLINE_SYSTEM_A_FLAG   "system=system_a"
#define PROC_CMDLINE_SYSTEM_B_FLAG   "system=system_b"

#define SYSTEM_INDEX_UNKNOWN (-1)
#define SYSTEM_INDEX_1 (1)
#define SYSTEM_INDEX_2 (2)

#define CRC_LE_BITS 64
#define CRC32_POLY_LE 0xedb88320
#define LE_TABLE_ROWS (CRC_LE_BITS/8)
#define LE_TABLE_SIZE 256


/*******************************************************************************
 *                             Type definitions                                *
 ******************************************************************************/
typedef struct
{
    unsigned int mtd_totalsize;				// mtd device total size
    unsigned int mtd_pageperblock;			// mtd device page per block
    unsigned int mtd_blocksize;				// mtd device block size
    unsigned int mtd_pagesize;				// mtd device page size
    unsigned int mtd_oobsize;				// mtd device oob size
    int parti_file_desc;					// partition update file description
} partition_mtd_info_t;


typedef enum
{
    DEVICE_MTD = 0,
    DEVICE_ZFTL = 1,
    DEVICE_MTD_BLOCK,
} device_type_t;


/*******************************************************************************
 *						   Local variable definitions						   *
 ******************************************************************************/


/*******************************************************************************
 *						  Global variable definitions						   *
 ******************************************************************************/
static unsigned int crc32table_le[LE_TABLE_ROWS][256];


/*******************************************************************************
 * 					   Local function declarations							   *
 ******************************************************************************/
static void crc32init_le_generic(const unsigned int polynomial, unsigned int (*tab)[256]);
static unsigned int crc32_body(unsigned int crc, unsigned char const *buf, size_t len, const unsigned int (*tab)[256]);
static unsigned int crc32_le_generic(unsigned int crc, unsigned char const *p, size_t len, const unsigned int (*tab)[256]);

static int mtd_get(const char *i_parti_name, device_type_t device_type, char *o_mtd_path, unsigned int o_mtd_path_len);
static int write_flags_info(partition_mtd_info_t *p_mtd_info, int index, unsigned char *content, int len);

static int get_flags_info(T_FLAGS_INFO *p_main, int *p_main_index, T_FLAGS_INFO *p_backup, int *p_backup_index);
static int set_flags_info(T_FLAGS_INFO *p_flags_info, int *p_main_index, int *p_backup_index);

static void copy_flags_info(T_FLAGS_INFO *dst, T_FLAGS_INFO *src);

static int get_current_system();


/*******************************************************************************
 * 					 Local function implementations 						   *
 ******************************************************************************/
static void crc32init_le_generic(const unsigned int polynomial, unsigned int (*tab)[256])
{
	unsigned i, j;
	unsigned int crc = 1;

	tab[0][0] = 0;

	for (i = LE_TABLE_SIZE >> 1; i; i >>= 1) {
		crc = (crc >> 1) ^ ((crc & 1) ? polynomial : 0);
		for (j = 0; j < LE_TABLE_SIZE; j += 2 * i)
			tab[0][i + j] = crc ^ tab[0][j];
	}
	for (i = 0; i < LE_TABLE_SIZE; i++) {
		crc = tab[0][i];
		for (j = 1; j < LE_TABLE_ROWS; j++) {
			crc = tab[0][crc & 0xff] ^ (crc >> 8);
			tab[j][i] = crc;
		}
	}
}


static unsigned int crc32_body(unsigned int crc, unsigned char const *buf, size_t len, const unsigned int (*tab)[256])
{
#define DO_CRC(x) crc = t0[(crc ^ (x)) & 255] ^ (crc >> 8)
#define DO_CRC4 (t3[(q) & 255] ^ t2[(q >> 8) & 255] ^ \
		   t1[(q >> 16) & 255] ^ t0[(q >> 24) & 255])
#define DO_CRC8 (t7[(q) & 255] ^ t6[(q >> 8) & 255] ^ \
		   t5[(q >> 16) & 255] ^ t4[(q >> 24) & 255])

	const unsigned int *b;
	size_t    rem_len;

	const unsigned int *t0=tab[0], *t1=tab[1], *t2=tab[2], *t3=tab[3];
	const unsigned int *t4 = tab[4], *t5 = tab[5], *t6 = tab[6], *t7 = tab[7];
	unsigned int q;
	
	if ((long)buf & 3 && len)
	{
		do
		{
			DO_CRC(*buf++);
		} while ((--len) && ((long)buf)&3);
	}
	
	rem_len = len & 7;
	len = len >> 3;
	b = (const unsigned int *)buf;
	
	for (--b; len; --len)
	{
		q = crc ^ *++b;
		crc = DO_CRC8;
		q = *++b;
		crc ^= DO_CRC4;
	}
	
	len = rem_len;
	if (len)
	{
		unsigned char *p = (unsigned char *)(b + 1) - 1;
		do
		{
			DO_CRC(*++p); /* use pre increment for speed */
		} while (--len);
	}
	
	return crc;
#undef DO_CRC
#undef DO_CRC4
#undef DO_CRC8
}


static unsigned int crc32_le_generic(unsigned int crc, unsigned char const *p, size_t len, const unsigned int (*tab)[256])
{
	crc = crc32_body(crc, p, len, tab);
	return crc;
}


static int mtd_get(const char *i_parti_name, device_type_t device_type, char *o_mtd_path, unsigned int o_mtd_path_len)
{
    FILE *fp_mtd = 0;
    char buf[128];
    char *line_str;

    if (!o_mtd_path_len)
    {
        return -1;
    }

    fp_mtd = fopen("/proc/mtd", "r+");
    if (NULL == fp_mtd)
    {
        flags_err("open file error: %s", strerror(errno));
        return -1;
    }
    // printf("[libmtd]: partition name:%s\n", i_parti_name);

    while (1)
    {
        int matches = 0;
        char mtdname[64] = {0};
        int mtdnum = 0;
        unsigned int mtdsize, mtderasesize;
        memset(buf, 0, sizeof(buf));
        line_str = fgets(buf, sizeof(buf) - 1, fp_mtd);

        if (NULL == line_str)
        {
            flags_err("get info from mtd error: %s", strerror(errno));
            fclose(fp_mtd);
            return -1;
        }
        // mtd5: 00100000 00020000 "fotaflag"
        matches = sscanf(buf, "mtd%d: %x %x \"%63[^\"]",
                         &mtdnum, &mtdsize, &mtderasesize, mtdname);
        mtdname[63] = '\0';

        if ((matches == 4) && (strcmp(mtdname, i_parti_name) == 0))
        {
            memset(o_mtd_path, 0, o_mtd_path_len);
            if (device_type == DEVICE_MTD_BLOCK)
            {
                snprintf(o_mtd_path, o_mtd_path_len, "/dev/mtdblock%d", mtdnum);
            }
            else if (device_type == DEVICE_MTD)
            {
                snprintf(o_mtd_path, o_mtd_path_len, "/dev/mtd%d", mtdnum);
            }
            else if (device_type == DEVICE_ZFTL)
            {
                snprintf(o_mtd_path, o_mtd_path_len, "/dev/zftl%d", mtdnum);
            }
            else
            {
                flags_err("unknown device type %d", device_type);
                fclose(fp_mtd);
                return -1;
            }
            // printf("[libmtd]: o_mtd_path=[%s]\n", o_mtd_path);
            break;
        }
    }
    fclose(fp_mtd);
    return 0;
}


static int write_flags_info(partition_mtd_info_t *p_mtd_info, int index, unsigned char *content, int len)
{
    struct erase_info_user erase_info;
    int write_len = 0;
    long long  offs = 0;

    erase_info.start = index * p_mtd_info->mtd_blocksize;
    erase_info.length = p_mtd_info->mtd_blocksize;

    offs = (long long)index * p_mtd_info->mtd_blocksize;

    if (ioctl(p_mtd_info->parti_file_desc, WRITEENABLE, 0) != 0)
    {
        flags_err("failed to enable mtd writeable, errno=%d, strerror=%s", errno, strerror(errno));
		
        return -1;
    }

    if (ioctl(p_mtd_info->parti_file_desc, MEMGETBADBLOCK, &offs) != 0)
    {
        flags_err("ioctl [MEMGETBADBLOCK] block: %d, change to be bad block, errno=%d, strerror=[%s]", index, errno, strerror(errno));
		
        return BAD_BLOCK;
    }

    if (ioctl(p_mtd_info->parti_file_desc, MEMERASE, &erase_info) < 0)
    {
        flags_err("ioctl [MEMERASE] block: %d fail, errno=%d, strerror=[%s]", index, errno, strerror(errno));
		
        return -1;
    }

    if (ioctl(p_mtd_info->parti_file_desc, MEMGETBADBLOCK, &offs) != 0)
    {
        flags_err("ioctl [MEMGETBADBLOCK] block:%d , change to be bad block, errno=%d, strerror=[%s]", index, errno, strerror(errno));
		
        return BAD_BLOCK;
    }

    if (lseek(p_mtd_info->parti_file_desc, index * p_mtd_info->mtd_blocksize, SEEK_SET) < 0)
    {
        flags_err("lseek fail, errno=%d, strerror=[%s]", errno, strerror(errno));
		
        return -1;
    }

    write_len = write(p_mtd_info->parti_file_desc, content, p_mtd_info->mtd_blocksize);

    if (write_len != p_mtd_info->mtd_blocksize)
    {
        flags_err("write flash fail, errno=%d, strerror=[%s]", errno, strerror(errno));
		
        return -1;
    }

    if (ioctl(p_mtd_info->parti_file_desc, WRITEDISABLE, 0) != 0)
    {
        flags_err("failed to disable mtd writeable, errno=%d, strerror=%s", errno, strerror(errno));
    }
	
    return 0;
}


static int get_flags_info(T_FLAGS_INFO *p_main, int *p_main_index, T_FLAGS_INFO *p_backup, int *p_backup_index)
{
    int ret = -1;
    int fd_dst = -1;
    char mtd_path[MAX_PATH_LEN] = {0};
    struct mtd_info_user meminfo = {0};
    partition_mtd_info_t mtd_info = {0};
    int index  = 0;
    int block_num = 0;
    int good_index = 0;
    int read_len  = 0;

    long long offs = 0;

    int block_flag = GOOD_BLOCK;

    ret = mtd_get(PARTITION_NAME_FLAGS, DEVICE_MTD, mtd_path, MAX_PATH_LEN);
    if (ret < 0)
    {
        flags_err("partition [%s] not find", PARTITION_NAME_FLAGS);
        goto error;
    }

    if ((fd_dst = open(mtd_path, O_RDWR | O_SYNC)) < 0)
    {
        flags_err("open flash error, errno=%d, strerror=[%s]", errno, strerror(errno));
        goto error;
    }
	
    mtd_info.parti_file_desc = fd_dst;

    if (ioctl(fd_dst, MEMGETINFO, &meminfo) != 0)
    {
        flags_err("get flash info error, errno=%d, strerror=[%s]", errno, strerror(errno));
        goto error_close;
    }

    mtd_info.mtd_blocksize		= meminfo.erasesize;
    mtd_info.mtd_oobsize		= meminfo.oobsize;
    mtd_info.mtd_pageperblock 	= meminfo.erasesize / meminfo.writesize;
    mtd_info.mtd_pagesize 		= meminfo.writesize;
    mtd_info.mtd_totalsize		= meminfo.size;

    block_num = mtd_info.mtd_totalsize / mtd_info.mtd_blocksize;

    for (index = 0; (good_index < 2 && index < block_num); index++)
    {

        offs = index * mtd_info.mtd_blocksize;
        if (ioctl(mtd_info.parti_file_desc, MEMGETBADBLOCK, &offs) == 0)
        {
            block_flag = GOOD_BLOCK;
        }
        else
        {
            flags_err("flags block [%d] is bad, errno=%d, strerror=[%s]", index, errno, strerror(errno));
            block_flag =  BAD_BLOCK;
        }

        if (block_flag == GOOD_BLOCK)
        {
            if (lseek(mtd_info.parti_file_desc, index * mtd_info.mtd_blocksize, SEEK_SET) < 0)
            {
                flags_err("lseek error, errno=%d, strerror=[%s]", errno, strerror(errno));
                goto error_close;
            }

            if (good_index == 0)
            {
                read_len = read(mtd_info.parti_file_desc, (unsigned char*)p_main, sizeof(T_FLAGS_INFO));
                *p_main_index = index;
            }
            else if (good_index == 1)
            {
                read_len = read(mtd_info.parti_file_desc, (unsigned char*)p_backup, sizeof(T_FLAGS_INFO));
                *p_backup_index = index;
            }
            else
            {
                break;
            }

            if (read_len < sizeof(T_FLAGS_INFO))
            {
                flags_err("read len (%d) < need len (%d)", read_len, sizeof(T_FLAGS_INFO));
                goto error_close;
            }

            good_index++;
        }

    }

    close(fd_dst);

    return 0;

error_close:
    close(fd_dst);

error:
    return -1;
}


static int set_flags_info(T_FLAGS_INFO *p_flags_info, int *p_main_index, int *p_backup_index)
{
    int ret = -1;
    int fd_dst = -1;
    char mtd_path[MAX_PATH_LEN] = {0};
    struct mtd_info_user meminfo = {0};
    partition_mtd_info_t mtd_info = {0};

    unsigned char *real_write_content = NULL;

    int index  = 0;
    int block_num = 0;
    int main_index = *p_main_index;
    int back_index = *p_backup_index;

    ret = mtd_get(PARTITION_NAME_FLAGS, DEVICE_MTD, mtd_path, MAX_PATH_LEN);
    if (ret < 0)
    {
        flags_err("partition [%s] not found", PARTITION_NAME_FLAGS);
		
        return -1;
    }

    if ((fd_dst = open(mtd_path, O_RDWR | O_SYNC)) < 0)
    {
        flags_err("open flash error, errno=%d, strerror=[%s]", errno, strerror(errno));
        return -1;
    }

    if (ioctl(fd_dst, MEMGETINFO, &meminfo) != 0)
    {
        flags_err("get flash info error, errno=%d, strerror=[%s]", errno, strerror(errno));
        goto error;
    }

    mtd_info.parti_file_desc 	= fd_dst;
    mtd_info.mtd_blocksize		= meminfo.erasesize;
    mtd_info.mtd_oobsize		= meminfo.oobsize;
    mtd_info.mtd_pageperblock 	= meminfo.erasesize / meminfo.writesize;
    mtd_info.mtd_pagesize 		= meminfo.writesize;
    mtd_info.mtd_totalsize		= meminfo.size;

    block_num = mtd_info.mtd_totalsize / mtd_info.mtd_blocksize;

    real_write_content = (unsigned char *)malloc(mtd_info.mtd_blocksize);
    if (NULL == real_write_content)
    {
        flags_err("malloc block fail");
        goto error;
    }

    memset(real_write_content, 0xFF, mtd_info.mtd_blocksize);
    memcpy(real_write_content, (char *)p_flags_info, sizeof(T_FLAGS_INFO));
	
    flags_log("begin to write main flags");

    for (index = 0; index < block_num; index++)
    {
        if (index == back_index)
        {
            continue;
        }

        ret = write_flags_info(&mtd_info, index, real_write_content, mtd_info.mtd_blocksize);
        if (ret == 0)
        {
            // дɹ˳µһλ
            flags_log("main flags location: [%d]->[%d]", main_index, index);
            main_index = index;
            break;
        }
        else if (ret == BAD_BLOCK)
        {
            // 飬һ
            flags_log("flags block index [%d] is bad", index);
            continue;
        }
        else
        {
            flags_err("write main flags fail");
            main_index = -1;
            break;
        }
    }

    flags_log("begin to write backup flags");

    for (index = 0; index < block_num; index++)
    {
        if (index == main_index)
        {
            continue;
        }

        flags_log("write backup to [%d] block", index);
        ret = write_flags_info(&mtd_info, index, real_write_content, mtd_info.mtd_blocksize);
        if (ret == 0)
        {
            // дɹ˳µһλ
            flags_log("backup flags location: [%d]->[%d]", back_index, index);
            back_index = index;
            break;
        }
        else if (ret == BAD_BLOCK)
        {
            // 飬һ
            continue;
        }
        else
        {
            flags_err("write backup flags fail");
            back_index = -1;
            break;
        }
    }

    if (main_index == -1 && back_index == -1)
    {
        goto error;
    }
    else
    {
        ret = 0;
        goto end;
    }

error:
    ret = -1;

end:
    close(fd_dst);
	
	if(NULL != real_write_content)
	{
		free(real_write_content);
		real_write_content = NULL;
	}
	
    return ret;
}


static void copy_flags_info(T_FLAGS_INFO *dst, T_FLAGS_INFO *src)
{
	memcpy(dst, src, sizeof(T_FLAGS_INFO));

	return;
}


static int get_current_system()
{
    char buf[1024] = {0};

    char *para = NULL;
    int matches = 0;

    FILE *fd_cmd = NULL;
    char *line_str = NULL;

    int current_system = -1;


    fd_cmd = fopen(FILE_PATH_PROC_CMDLINE, "r");
    if (!fd_cmd)
    {
        printf("Open file %s error, error:%s", FILE_PATH_PROC_CMDLINE, strerror(errno));
        return SYSTEM_INDEX_UNKNOWN;
    }

    while (!feof(fd_cmd))
    {
        memset(buf, 0, sizeof(buf));
        line_str = fgets(buf, sizeof(buf), fd_cmd);

        if (NULL == line_str)
        {
            printf("get info from /proc/cmdline error:%s", strerror(errno));
            goto end;
        }

        printf("buff:%s", buf);

        para = strtok(buf, " ");
        while (para)
        {
            printf("para:%s", para);
            if (strncmp(para, PROC_CMDLINE_SYSTEM_A_FLAG, strlen(PROC_CMDLINE_SYSTEM_A_FLAG)) == 0)
            {
                current_system = SYSTEM_INDEX_1;
                goto end;
            }
            else if (strncmp(para, PROC_CMDLINE_SYSTEM_B_FLAG, strlen(PROC_CMDLINE_SYSTEM_B_FLAG)) == 0)
            {
                current_system = SYSTEM_INDEX_2;
                goto end;
            }
            else
            {
                //:
            }
            para = strtok(NULL, " ");
        }
    }

end:
    if (fd_cmd)
    {
        fclose(fd_cmd);
    }
	
    return current_system;
}


/*******************************************************************************
 * 					 Global function implementations						   *
 ******************************************************************************/
int flags_init()
{
	T_FLAGS_INFO flags_info = {0};
	int main_index = 0;
	int back_index = 1;
	int ret = -1;
	int fd_dst = -1;
	char mtd_path[MAX_PATH_LEN] = {0};
	struct mtd_info_user meminfo = {0};
	partition_mtd_info_t mtd_info = {0};

	unsigned char *real_write_content = NULL;
	T_FLAGS_INFO *p_write = NULL;

	int index  = 0;
	int block_num = 0;
	int already_write = 0;
	
	unsigned int crc_32 = 0;
	
    flags_info.magic_start = FLAGS_MAGIC;
	
    flags_info.boot_fota_flag.boot_to = DUAL_SYSTEM;
    flags_info.boot_fota_flag.fota_status = 0;
    flags_info.boot_fota_flag.system.status = DUALSYSTEM_STATUS_BOOTABLE;
    flags_info.boot_fota_flag.system.try_cnt = 0;
    flags_info.boot_fota_flag.system2.status = DUALSYSTEM_STATUS_BOOTABLE;
    flags_info.boot_fota_flag.system2.try_cnt = 0;

    flags_info.boot_env.dualsys_type = DUALSYSTEM_AB;

	flags_info.ubifs_status.fs_status = 0;
	
    flags_info.magic_end = FLAGS_MAGIC;

	crc32init_le();
	crc_32 = crc32_le(0, (unsigned char const *)(&flags_info), sizeof(flags_info));
	flags_log("init crc_32=%u", crc_32);

	flags_info.crc32 = crc_32;

	ret = mtd_get(PARTITION_NAME_FLAGS, DEVICE_MTD, mtd_path, MAX_PATH_LEN);
	if (ret < 0)
	{
		flags_err("partition [%s] not found", PARTITION_NAME_FLAGS);
		
		return -1;
	}

	if ((fd_dst = open(mtd_path, O_RDWR | O_SYNC)) < 0)
	{
		flags_err("open flash error, errno=%d, strerror=[%s]", errno, strerror(errno));
		
		return -1;
	}

	if (ioctl(fd_dst, MEMGETINFO, &meminfo) != 0)
	{
		flags_err("get flash info error, errno=%d, strerror=[%s]", errno, strerror(errno));
		
		goto error;
	}

	mtd_info.parti_file_desc 	= fd_dst;
	mtd_info.mtd_blocksize		= meminfo.erasesize;
	mtd_info.mtd_oobsize		= meminfo.oobsize;
	mtd_info.mtd_pageperblock	= meminfo.erasesize / meminfo.writesize;
	mtd_info.mtd_pagesize		= meminfo.writesize;
	mtd_info.mtd_totalsize		= meminfo.size;

	block_num = mtd_info.mtd_totalsize / mtd_info.mtd_blocksize;

	real_write_content = (unsigned char *)malloc(mtd_info.mtd_blocksize);
	if (NULL == real_write_content)
	{
		flags_err("malloc for block fail");
		
		goto error;
	}

	memset(real_write_content, 0xFF, mtd_info.mtd_blocksize);
	memcpy(real_write_content, (char *)(&flags_info), sizeof(T_FLAGS_INFO));

	for (index = 0; index < block_num; index++)
	{
		if (already_write >= FLAGS_INIT_VALID_BLOCKS_NUM)
		{
			break;
		}

		ret = write_flags_info(&mtd_info, index, real_write_content, mtd_info.mtd_blocksize);
		if (ret == 0)
		{
			already_write++;
			flags_log("write init valid block num: %d", already_write);
			
			continue;
		}
		else if (BAD_BLOCK == ret)
		{
			continue;
		}
	}

	if (already_write >= 2)
	{
		flags_log("init system status success, alread write block: %d", already_write);
		ret = 0;
	}
	else
	{
		flags_log("init system status fail, alread write block: %d", already_write);
		ret = -1;
	}

	goto end;

error:
	ret = -1;

end:
	close(fd_dst);
	
	if(NULL != real_write_content)
	{
		free(real_write_content);
		real_write_content = NULL;
	}
	
	return ret;
}


int flags_get(T_FLAGS_INFO *p_flags_info)
{
    T_FLAGS_INFO main_flag = {0};
    T_FLAGS_INFO backup_flag = {0};
	T_FLAGS_INFO p_flags_info_tmp = {0};

    int main_index = 0;
    int backup_index = 1;

	unsigned int crc32_main = 0;
	unsigned int crc32_backup = 0;
	
	unsigned int crc32_le_main = 0;
	unsigned int crc32_le_backup = 0;

	if (NULL == p_flags_info)
	{
		flags_err("invalid param NULL");
		
		return -1;
	}

    if (get_flags_info(&main_flag, &main_index, &backup_flag, &backup_index) != 0)
    {
    	flags_err("get flags info fail");
		
        return -1;
    }
	
	flags_log("main_flag crc32=%u", main_flag.crc32);
	flags_log("backup_flag crc32=%u", backup_flag.crc32);
	
	if ((0 == main_flag.crc32) && (0 == backup_flag.crc32))
	{
		if ((FLAGS_MAGIC == main_flag.magic_start) && (FLAGS_MAGIC == main_flag.magic_end))
		{
			if ((FLAGS_MAGIC == backup_flag.magic_start) && (FLAGS_MAGIC == backup_flag.magic_end))
			{
				memcpy(&p_flags_info_tmp, &main_flag, sizeof(T_FLAGS_INFO));
				crc32init_le();
				p_flags_info_tmp.crc32 = 0;
				p_flags_info_tmp.crc32 = crc32_le(0, (unsigned char const *)(&p_flags_info_tmp), sizeof(T_FLAGS_INFO));
				flags_log("old version, set crc32=%u", p_flags_info_tmp.crc32);
				
				if (set_flags_info(&p_flags_info_tmp, &main_index, &backup_index) != 0)
				{
					flags_err("old version, set flags info fail");
					return -1;
				}
				
				copy_flags_info(p_flags_info, &main_flag);
				return 0;
			}
		}
	}

	crc32_main = main_flag.crc32;
	crc32_backup = backup_flag.crc32;

	main_flag.crc32 = 0;
	backup_flag.crc32 = 0;
	
	crc32init_le();
	
	crc32_le_main = crc32_le(0, (unsigned char const *)(&main_flag), sizeof(main_flag));
	flags_log("crc32_le_main crc32=%u", crc32_le_main);
	
	crc32_le_backup = crc32_le(0, (unsigned char const *)(&backup_flag), sizeof(backup_flag));
	flags_log("crc32_le_backup crc32=%u", crc32_le_backup);

    if (crc32_main == crc32_le_main)
    {
        copy_flags_info(p_flags_info, &main_flag);
		
        return 0;
    }
	
    if (crc32_backup == crc32_le_backup)
    {
        copy_flags_info(p_flags_info, &backup_flag);
		
        return 0;
    }

    flags_err("do not find valid flags info");
	
    return -1;
}


int flags_set(T_FLAGS_INFO *p_flags_info)
{
	T_FLAGS_INFO main_flag = {0};
    T_FLAGS_INFO backup_flag = {0};
    int main_index = 0;
    int backup_index = 1;
	
	if (NULL == p_flags_info)
	{
		flags_err("invalid param NULL");
		
		return -1;
	}

	if ((FLAGS_MAGIC != p_flags_info->magic_start) || (FLAGS_MAGIC != p_flags_info->magic_end))
	{
		flags_err("invalid magic");
		
		return -1;
	}
	
    if (get_flags_info(&main_flag, &main_index, &backup_flag, &backup_index) != 0)
    {
    	flags_err("get flags info fail");
		
        return -1;
    }

	crc32init_le();
	p_flags_info->crc32 = 0;
	p_flags_info->crc32 = crc32_le(0, (unsigned char const *)p_flags_info, sizeof(T_FLAGS_INFO));
	flags_log("set crc32=%u", p_flags_info->crc32);

    if (set_flags_info(p_flags_info, &main_index, &backup_index) != 0)
    {
        flags_err("set ubifs status fail");
		
        return -1;
    }

	return 0;
}


int flags_get_ubifs_status(T_UBIFS_STATUS *p_ubifs_status)
{
	T_FLAGS_INFO p_flags_info = {0};
	
	if (NULL == p_ubifs_status)
	{
		flags_err("invalid param NULL");
		return -1;
	}

	if (0 == flags_get(&p_flags_info))
	{
		p_ubifs_status->fs_status = p_flags_info.ubifs_status.fs_status;
		strncpy(p_ubifs_status->fs_mtd_name, p_flags_info.ubifs_status.fs_mtd_name, sizeof(p_ubifs_status->fs_mtd_name));
		strncpy(p_ubifs_status->fs_ubi_vol_name, p_flags_info.ubifs_status.fs_ubi_vol_name, sizeof(p_ubifs_status->fs_ubi_vol_name));
		
		return 0;
	}
	else
	{
	    flags_err("do not find valid flags info");
	    return -1;
	}
}


int flags_set_ubifs_status(T_UBIFS_STATUS *p_ubifs_status)
{
	T_FLAGS_INFO p_flags_info = {0};
	
	if (NULL == p_ubifs_status)
	{
		flags_err("invalid param NULL");
		return -1;
	}

	if (0 != flags_get(&p_flags_info))
	{
		flags_err("get ubifs status invalid");
		return -1;
	}
	
	memcpy(&(p_flags_info.ubifs_status), p_ubifs_status, sizeof(T_UBIFS_STATUS));

	if (0 != flags_set(&p_flags_info))
	{
		flags_err("set ubifs status fail");
		return -1;
	}
	
    return 0;
}


unsigned int flags_get_nvroflag(void)
{
	T_FLAGS_INFO t_flag = {0};

	if (flags_get(&t_flag) != 0)
		return NVRO_INVALID;
	return t_flag.nvro_flag;
}


int flags_set_nvroflag(unsigned int flag)
{
	T_FLAGS_INFO t_flag = {0};

	if (flags_get(&t_flag) != 0)
		return -1;
	if (t_flag.nvro_flag == flag)
		return 0;
	if (flag == NVRO_RESTORING)
	{
		if (t_flag.nvro_flag != NVRO_BACKED_UP)
		{
			printf("[error]flags nvro only NVRO_BACKED_UP switch to NVRO_RESTORING\n");
			return -1;
		}
	}
	t_flag.nvro_flag = flag;

	return flags_set(&t_flag);
}


int flags_get_current_system()
{
    int current = get_current_system();
	
    if (current == 1)
    {
        return DUAL_SYSTEM;
    }
    else if (current == 2)
    {
        return DUAL_SYSTEM2;
    }

    return -1;
}


/* APIڵ⣬ʽ벻ʹ */
int flags_get_nocrc(T_FLAGS_INFO *p_flags_info)
{
    T_FLAGS_INFO main_flag = {0};
    T_FLAGS_INFO backup_flag = {0};
    int main_index = 0;
    int backup_index = 1;

	if (NULL == p_flags_info)
	{
		flags_err("invalid param NULL");
		return -1;
	}

    if (get_flags_info(&main_flag, &main_index, &backup_flag, &backup_index) != 0)
    {
    	flags_err("get flags info fail");
        return -1;
    }
	
	if (0 != memcmp(&main_flag, &backup_flag, sizeof(T_FLAGS_INFO)))
	{
	    flags_err("main flag and backup flag are different");
        return -1;
	}
	
	copy_flags_info(p_flags_info, &main_flag);
	p_flags_info->crc32 = main_flag.crc32;

    return 0;
}


/* APIڵ⣬ʽ벻ʹ */
int flags_set_nocrc(T_FLAGS_INFO *p_flags_info)
{
	T_FLAGS_INFO main_flag = {0};
    T_FLAGS_INFO backup_flag = {0};
    int main_index = 0;
    int backup_index = 1;
	
	if (NULL == p_flags_info)
	{
		flags_err("invalid param NULL");
		return -1;
	}

	if ((FLAGS_MAGIC != p_flags_info->magic_start) || (FLAGS_MAGIC != p_flags_info->magic_end))
	{
		flags_err("invalid magic");
		return -1;
	}
	
    if (get_flags_info(&main_flag, &main_index, &backup_flag, &backup_index) != 0)
    {
    	flags_err("get flags info fail");
        return -1;
    }

    if (set_flags_info(p_flags_info, &main_index, &backup_index) != 0)
    {
        flags_err("set ubifs status fail");
        return -1;
    }

	return 0;
}


void crc32init_le(void)
{
	crc32init_le_generic(CRC32_POLY_LE, crc32table_le);
}


unsigned int crc32_le(unsigned int crc, unsigned char const *p, size_t len)
{
	return crc32_le_generic(crc, p, len, (const unsigned int (*)[256])crc32table_le);
}

