
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <stdint.h>
#include <stdlib.h>
#include <errno.h>
#include <termios.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/types.h>
#include <pthread.h>
#include <termios.h>
#include <time.h>
#include <sys/ioctl.h>
#include <malloc.h>
#include <sys/time.h>
#include "mbtk_gnss_internal.h"
#include "ringbuffer.h"
#ifdef __cplusplus
extern "C" {
#endif


struct mbtk_gnss_handle_t
{
    int dev_fd;
    pthread_t  uart_pthread;
    pthread_t  gnss_pthread;
    gnss_handler_func_t gnss_handler_func;
    int mode; // 0 - stop, 1 - single, 2 - periodic, 3 - start
    pthread_mutex_t _cond_mutex;
    int reset_state;
    int inited;
    ring_buffer_t ring_buffer;
    int getap_status;
    char *rb;
    e_msg_id_t e_msg_id;
    
#if TTFF_TEST
    pthread_t ttff_pid;
    int location_state;
#endif
    /********************
    存储handle的地址指针
    phandle = &handle
    handle  = mbtk_gnss_handle
    *********************/
    uint32_t *phandle; // handle的地址指针
};
#if TTFF_TEST
struct mbtk_gnss_ttff_t
{
    int type;
    int timeout_sec;
    int test_count;
    int average_loc_time;
};
// ttff 测试 flag
static int location_test = 0;
static pthread_mutex_t loc_cond_mutex_r = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t loc_sync_cond = PTHREAD_COND_INITIALIZER;
#endif
volatile int nmea_state = 0;
volatile int nmea_reading = 0;
static struct mbtk_gnss_cmd_msg_t mbtk_gnss_cmd_msg_map[] = {
{ 1,     "$OK",         NULL,                 0},
{ 2,     "$Fail",       NULL,                 0},
{ 3,     "RMC",         get_gnss_time_info,   1},
{ 4,     "GGA",         get_gnss_time_info,   1},
{ 5,     "$PDTINFO",    get_gnss_device_info, 0},
{ 6,     "$CFGNMEA",    get_gnss_device_info, 0},
{ 7,     "$CFGPRT",     gnss_uart_info,       0},
{ 8,     "$CFGAID",     get_gnss_agnss_state, 0},
{ 9,     "$ANTSTAT",    NULL,                 0},
#if TTFF_TEST
{10,     "GSA",         gnss_gsa_info,        1},
#endif
};


/**
 *  \brief strstr_n
 *
 *  find string return number
 *
 *  \param param
 *  \return return type
 */
int strstr_n(const char *s1, const char *s2)
{
    int n;
    int strlen = 0;

    if(*s2)
    {
        while(*s1)
        {
            for(n = 0; *(s1+n) == *(s2 + n); n++)
            {
                if(!*(s2 + n + 1))
                {
                    strlen++;
                    return strlen;
                }
            }
            s1++;
            strlen++;
        }
        return 0;
    }
    else
        return 0;
}

/**
 * @brief      gnss_get_para_from_nmea
 *
 * @details    从buf里面得到第num个逗号所在的位置
 *
 * @param      param
 *
 * @return     0~0xfe,代表逗号所在位置的偏移.
 *             0xff,代表不存在第cx个逗号
 */
static int gnss_get_para_from_nmea(const char *data, char *out_data, int num)
{
    int i = 0;
    int n[2] = {0};
    int tmp;

    // 找到第num个","，结果放到 n[0]
    for (i = 0; i < num; ++i) {
        tmp = strstr_n(&data[n[0]], ",");
        if(0 == tmp) {
            gnss_log("%s %d : error\n", __func__, __LINE__);
            gnss_log("error: [%d] %s\n", num, data);
            return -1;
        }
        n[0] += tmp;
    }
    if ((n[1] = strstr_n(&data[n[0]], ",")) ||
            (n[1] = strstr_n(&data[n[0]], "*")) &&
            (n[1] > 1)) {
        memcpy(out_data, &data[n[0]], n[1] - 1);
    } else {
        gnss_log("%s %d : error [%d]\n" , __func__, __LINE__, n[1]);
        gnss_log("error: [%d] %s\n", num, data);
        return -1;
    }

    return 0;
}
/**
* @brief      get_timestamp
*
* @details    
*
* @param      param
*
* @return     return type
*/
static time_t get_timestamp(char *time)
{
    char tmp_char[4] = {0};
    struct tm* tmp_time = (struct tm*)malloc(sizeof(struct tm));

    memset(tmp_time, 0, sizeof(struct tm));
    memset(tmp_char, 0, sizeof(tmp_char));
    memcpy(tmp_char, &time[4], 2);
    tmp_time->tm_sec = atoi(tmp_char);
    memcpy(tmp_char, &time[2], 2);
    tmp_time->tm_min = atoi(tmp_char);
    memcpy(tmp_char, &time[0], 2);
    tmp_time->tm_hour = atoi(tmp_char);
    memcpy(tmp_char, &time[6], 2);
    tmp_time->tm_mday = atoi(tmp_char);
    memcpy(tmp_char, &time[8], 2);
    tmp_time->tm_mon = atoi(tmp_char);
    memcpy(tmp_char, &time[10], 2);
    tmp_time->tm_year = 100 + atoi(tmp_char);

    time_t _t = mktime(tmp_time);//按当地时区解析tmp_time
    // gnss_log("timestamp: %ld\n",_t);
    free(tmp_time);

    return _t;
}
/**
 * @brief      get_gnss_device_info
 *
 * @details    获取设备信息
 *
 * @param      type: 5-从$PDTINFO获取
 *                   6-从$CFGNMEA获取
 *
 * @return     return type
 */
static void get_gnss_device_info(int type, char *str, void *usr_ptr)
{
    char tmp_str[32] = {0};
    int i, ret;

    if(5 == type) // define  mbtk_gnss_cmd_msg_map
    {
        // $PDTINFO get product info
        char *tmp_ptr[6] = {mopen_gnss_device_info.product_name,
                            mopen_gnss_device_info.dev_config,
                            mopen_gnss_device_info.hw_ver,
                            mopen_gnss_device_info.fw_ver,
                            mopen_gnss_device_info.pn,
                            mopen_gnss_device_info.sn
                           };
        for (i = 0; i < 6; ++i) {
            memset(tmp_str, 0, sizeof(tmp_str));
            // get product name
            ret = gnss_get_para_from_nmea(str, tmp_str, i + 1);
            if(ret)
                continue;
            memcpy(tmp_ptr[i], tmp_str, strlen(tmp_str));
        }
        gnss_log("*************************\n");
        gnss_log("-Pn: %s\n dc: %s\n hv: %s\n fw: %s\n pn: %s\n sn: %s\n ",
                 mopen_gnss_device_info.product_name,
                 mopen_gnss_device_info.dev_config,
                 mopen_gnss_device_info.hw_ver,
                 mopen_gnss_device_info.fw_ver,
                 mopen_gnss_device_info.pn,
                 mopen_gnss_device_info.sn);
    }
    if(6 == type) // define  mbtk_gnss_cmd_msg_map
    {
        // $CFGNMEA get nmea version
        memset(tmp_str, 0, sizeof(tmp_str));
        ret = gnss_get_para_from_nmea(str, tmp_str, 1);
        if(ret)
            return;
        memcpy(mopen_gnss_device_info.nmea_ver, tmp_str, strlen(tmp_str));
        mopen_gnss_device_info.device_info_valid = TRUE;
    }
}

static short int from_hex(char a)
{
    if (a >= 'A' && a <= 'F')
        return a - 'A' + 10;
    else if (a >= 'a' && a <= 'f')
        return a - 'a' + 10;
    else
        return a - '0';
}

static int str_to_hex(char *str)
{
    unsigned char str_len = strlen(str);
    int i;
    int ret = 0;

    for (i = 0; i < str_len; ++i) {
        ret = ret * 16 + from_hex(str[i]);
    }
    return ret;
}

static void get_gnss_agnss_state(int cmd, char *str, void *data)
{

    int ret;
    char tmp_str[10] = {0};
    int agps;
    static int count = 0;

    ret = gnss_get_para_from_nmea(str, tmp_str, 2);
    if(ret) {
        printf("\n%s[%d] error!\n" , __FUNCTION__, __LINE__);
    }
    agps = str_to_hex(tmp_str);
    gnss_log("\n%s[%d] agnss: %s[%x]\n" , __FUNCTION__, __LINE__, tmp_str, agps);
    if(0 == agps && count < 5) {
        lynq_gnss_get_aidinfo((uint32)mbtk_gnss_handle);
        count++;
    } else {
        printf("\nagnss: %s\n", str);
        count = 0;
    }
}
// 1节=1海里/小时=1.852公里/小时
/**
 * @brief      function description
 *
 * @details    获取位置信息
 *
 * @param      type: 1-从$RMC获取
 *                   2-从$GGA获取
 * @return     return type
 */
static void get_gnss_loc_info(int type, char *str,
                              struct mopen_location_info_t *_mopen_location_info)

{
    char tmp_str[32] = {0};
    int ret;

    if(1 == type)
    {
        // $xxRMC get product info
        memset(tmp_str, 0, sizeof(tmp_str));
        // get product name
        ret = gnss_get_para_from_nmea(str, tmp_str, 7);
        if(ret)
            return;
        _mopen_location_info->speed = atof(tmp_str);
        memset(tmp_str, 0, sizeof(tmp_str));
        // get product name
        ret = gnss_get_para_from_nmea(str, tmp_str, 8);
        if(ret)
            return;
        _mopen_location_info->bearing = atof(tmp_str);
    }
    else if(2 == type)
    {
        // $XXGGA get product info
        memset(tmp_str, 0, sizeof(tmp_str));
        // get product name
        ret = gnss_get_para_from_nmea(str, tmp_str, 2);
        if(ret)
            return;
        _mopen_location_info->latitude = atof(tmp_str);
        memset(tmp_str, 0, sizeof(tmp_str));
        // get product name
        ret = gnss_get_para_from_nmea(str, tmp_str, 4);
        if(ret)
            return;
        _mopen_location_info->longitude = atof(tmp_str);
        memset(tmp_str, 0, sizeof(tmp_str));
        // get product name
        ret = gnss_get_para_from_nmea(str, tmp_str, 9);
        if(ret)
            return;
        _mopen_location_info->altitude = atof(tmp_str);
    }
}

static void get_gnss_time_info(int cmd, char *str, void *data)
{
    int ret;
    char param[36] = {0};
    struct mopen_location_info_t *mopen_location_info_ptr = (struct mopen_location_info_t *)data;

    if (3 == cmd) {
        memset(param, 0, sizeof(param));
        // get time
        ret = gnss_get_para_from_nmea(str, param, 1);
        if(ret)
            return;
        // get date
        ret = gnss_get_para_from_nmea(str, &param[6], 9);
        if(ret)
            return;

        mopen_location_info_ptr->timestamp = get_timestamp(param);
        get_gnss_loc_info(1, str, mopen_location_info_ptr);
    } else if(4 == cmd) /* GGA */{
        get_gnss_loc_info(2, str, mopen_location_info_ptr);
        ret = gnss_get_para_from_nmea(str, param, 7);
        if(ret)
            return;
        char no_sv = (char)atoi(param);
        gnss_log("SV number: %d, %d\n", g_no_sv, no_sv);
        /*
          只能在临时固件下，才能获取APdata星历数据
          在6颗卫星保存文件，每增加2颗保存一次。
         */
        if (1 == firmware_extren_state &&
                g_no_sv < (no_sv - 1) && no_sv > 5) {

            g_no_sv = no_sv;
            lynq_gnss_get_ap_data();
        }
    }
}

static void gnss_uart_info(int cmd, char *str, void *data)
{
    int ret;
    char tmp_str[12] = {0};

    // $CFGPRT,1,h0,9600,129,3*57
    ret = gnss_get_para_from_nmea(str, tmp_str, 3);
    if(ret)
        return;
    mopen_gnss_device_info.usrt_bandrate = atoi(tmp_str);
    gnss_log("CFGPRT: %s\n", str);
    gnss_log("Uart bandrate: %d\n" , mopen_gnss_device_info.usrt_bandrate);
    gnss_log("*************************\n");
}
/**
 *  \brief function description
 *
 *  Detailed 处理gnss数据
 *
 *  \param param
 *  \return return type
 */
static void process_gnss_callback(struct mbtk_gnss_handle_t *handle, char *data, int data_len)
{
    int ret = 0;
    int i = 0;
    static struct mopen_location_info_t mopen_location_info;
    static int64_t tmp_time = 0;
    mopen_gnss_nmea_info_t nmea_info;

    memset(&nmea_info, 0, sizeof(nmea_info));
    if(0 == tmp_time)
        memset(&mopen_location_info, 0, sizeof(mopen_location_info));

    for (i = 0;
         i < (sizeof(mbtk_gnss_cmd_msg_map) / sizeof(struct mbtk_gnss_cmd_msg_t));
         ++i) {
        if(strstr_n(data, mbtk_gnss_cmd_msg_map[i].cmd_str)) {
            if(mbtk_gnss_cmd_msg_map[i].gnss_msg_func)
                mbtk_gnss_cmd_msg_map[i].gnss_msg_func(mbtk_gnss_cmd_msg_map[i].index,
                                                       data, &mopen_location_info);
            break;
        }
    }
    if(0 == mbtk_gnss_cmd_msg_map[i].is_continue)
        return;

    tmp_time = mopen_location_info.timestamp;
    nmea_info.timestamp = mopen_location_info.timestamp;
    nmea_info.length = data_len;
    memcpy(nmea_info.nmea, data, data_len);
    //gnss_log("nmea:[%d] %s", data_len, data);
    if( handle->mode == 3 && nmea_info.timestamp &&handle->gnss_handler_func)
    {
        handle->gnss_handler_func((uint32)handle, E_MT_LOC_MSG_ID_NMEA_INFO, &nmea_info, NULL);
    }
    if(handle->gnss_handler_func && handle->mode == 1 &&
            mopen_location_info.latitude &&
            mopen_location_info.longitude &&
            mopen_location_info.altitude &&
            mopen_location_info.timestamp &&
            mopen_location_info.speed)
    {
        
        handle->gnss_handler_func((uint32)handle, E_MT_LOC_MSG_ID_LOCATION_INFO, &mopen_location_info, NULL);
        memset(&mopen_location_info, 0, sizeof(mopen_location_info));
    }

    return;
}

/**
 *  \brief get_gnss_from_str
 *
 *  Detailed 从串口数据解析出每条消息
 *
 *  \param param
 *  \return return type
 */
static int get_gnss_from_str(struct mbtk_gnss_handle_t *handle, char *data, int data_len)
{
    char *tail = NULL;
    static int seek = 0;
    // 等待 OK, 如果20条结果没有等到，就异常
    static int reset_count = 0;
    int i = 0, ret = -1;

    if (handle->reset_state)
    {
        // 等待 reset 回复的 OK
        if(0 != memcmp(data, "$OK", 3) && reset_count < 20) {
            printf("gnss reset invalid: [%s]\n", data);
            reset_count++;
            return -1;
        }
        if (reset_count > 19) {
            printf("%s: device reset timeout!!!\n", __FUNCTION__);
            LOGI("%s: device reset timeout!!!\n", __FUNCTION__);
        }
        reset_count = 0;
        gnss_log("reset ok: %s\n", data);
#if BAUDRATE_115200
        ret = mopen_uart_change(handle->dev_fd, 0);
        if(ret) {
            printf("reset Uart set 115200 error\n");
        }
#endif
        pthread_mutex_lock(&handle->_cond_mutex);
        handle->reset_state = 0;
        pthread_mutex_unlock(&handle->_cond_mutex);
    }

    if((data[0] == '$' || data[0] == '#') &&
            data[data_len - 1] == '\n' &&
            data_len < 128) {
        process_gnss_callback(handle, data, data_len);
    } else {
        gnss_log("nmea error: %s\n", data);
    }

    return 1;
}

void mopen_gnss_NonBlock(int fd, int cmd)
{
    int flags;

    flags = fcntl(fd, F_GETFL, 0);
    if(cmd)
        flags |= O_NONBLOCK;
    else
        flags &= ~O_NONBLOCK;

    fcntl(fd, F_SETFL, flags);
}

int set_baudrate(int fd, int baudrate)
{
    struct termios options, oldtio;

    if(fcntl(fd, F_SETFL, 0) < 0) {
        printf("fcntl failed!\n");
        return -1;
    }

    if(tcgetattr(fd, &oldtio) != 0) {
        printf("setup serial error!\n");
        return -1;
    }

    /* Get the current options for the port... */
    tcgetattr(fd, &options);

    /* Set the baud rates to baudrate... */
    cfsetispeed(&options, baudrate);
    cfsetospeed(&options, baudrate);
    tcsetattr(fd, TCSANOW, &options);

    if (0 != tcgetattr(fd, &options))
    {
        printf("get options error!\n");
        return -1;
    }

    /*
     * 8bit Data,no partity,1 stop bit...
     */
    options.c_cflag &= ~PARENB;//无奇偶校验
    options.c_cflag &= ~CSTOPB;//停止位，1位
    options.c_cflag &= ~CSIZE; //数据位的位掩码
    options.c_cflag |= CS8;    //数据位，8位

    cfmakeraw(&options);

    /*
     * Set the new options for the port...
     */
    if (tcsetattr(fd, TCSANOW, &options) != 0)
    {
        printf("setup serial error!\n");
        return -1 ;
    }
    return 0 ;
}
/*
  自适应波特率设置
 */
static int auto_set_uart_baudrate(int fd)
{
    char rbuf[512];
    int rByte = 0;
    int b[3] = {B115200, B9600, 0};
    int ret = B9600;
    struct timeval time_m, time_n;
    // 时间超时标志
    int timeout_sign = 0;
    // 先测试默认的9600波特率
    SET_TIME_OUT(3);
    do {
        gettimeofday(&time_n, NULL);
        if(time_n.tv_sec > time_m.tv_sec) {
            //printf("Baudrate--test-9600--- timeout!\n");
            if(timeout_sign)
                break;
            set_baudrate(fd, B115200);
            ret = B115200;
            timeout_sign = 1;
            SET_TIME_OUT(3);
            continue;
        }

        if(select_read(fd, 1) > 0)
            usleep(50000);
        else
            continue;

        rByte = deal_read(fd,&rbuf,sizeof(rbuf));
        if(rByte > 0) {
            gnss_log("Auto Baudrate[%d]%s\n", rByte, rbuf);
            if(strstr(rbuf, "$"))
                return ret;
            memset(rbuf, 0, sizeof(rbuf));
        } else {
            printf("*** read error\n");
        }
    }while(1);

    return 0;
}

int mopen_gnss_open(char *dev, int baudrate)
{
    int ret;
    int fd = 0;

    fd = open(dev, O_RDWR | O_NOCTTY | O_NDELAY);
    if(fd < 0) {
        printf("open error\n");
        return -1;
    }
    gnss_log("curent dev: %s, fd: %d \n", dev, fd);
    printf("curent dev: %s, fd: %d \n", dev, fd);
    if(baudrate) {
        gnss_log("set baudrate: %d \n", baudrate);
        ret = set_baudrate(fd, baudrate);
        if(-1 == ret) {
            close(fd);
            return -1;
        }
    } else {
        set_baudrate(fd, B9600);
        printf("baudrate ok\n");
    }

    return fd;
}

static int mopen_gnss_read(int fd, char* buf, unsigned int buf_len)
{
    buf_len=(buf_len > MBTK_UART_RECV_BUFFER_SIZE ? MBTK_UART_RECV_BUFFER_SIZE : buf_len);
    return read(fd, buf, buf_len);
}

int mopen_gnss_write(int fd, const char* buf, unsigned int buf_len)
{
    size_t size;
    size_t size_to_wr;
    ssize_t size_written;
    if(MBTK_UART_SEND_BUFFER_MAX < buf_len)
    {
        return -1;
    }
    for(size = 0; size < buf_len;)
    {
        size_to_wr = buf_len - size;
        if( size_to_wr > MBTK_UART_SEND_BUFFER_MAX)
            size_to_wr = MBTK_UART_SEND_BUFFER_MAX;
        size_written = write(fd, &buf[size], size_to_wr);
        if (size_written==-1)
        {
            return -1;
        }
        printf("send cmd: %s\n", &buf[size]);
        size += size_written;
        if(size_written != size_to_wr)
        {
            return size;
        }
    }
    return size;
}

int mopen_gnss_close(int fd)
{
    return close(fd);
}

static void gnss_info_pthread(void* hdl)
{
    struct mbtk_gnss_handle_t *gnss_handle = (struct mbtk_gnss_handle_t *)hdl;
    int ret = 0, i;
    char tmp;
    char tmp_arr[128] = {0};

    pthread_detach(pthread_self());

    memset(tmp_arr, 0, sizeof(tmp_arr));
    while(mbtk_gnss_handle->inited)
    {
        for (i = 0; i < 256; ++i) {
            if (0 == mbtk_gnss_handle->inited)
                goto exit;
            ret = ring_buffer_peek(&gnss_handle->ring_buffer, &tmp, i);
            if (0 == ret) {
                usleep(300000);
                gnss_log("ring_buffer_peek ringbuffer read error\n");
                i--;
                continue;
            }
            if (tmp == '\n') {
                break;
            }
        }

        if (i > (256 - 2))
            continue;

        ret = ring_buffer_dequeue_arr(&gnss_handle->ring_buffer, tmp_arr, i + 1);

        if(ret > 0 && 0 == mbtk_gnss_handle->getap_status) {
            // gnss_log("NEMA:[%d] %s", ret, tmp_arr);
            get_gnss_from_str(gnss_handle, tmp_arr, ret);
            memset(tmp_arr, 0, sizeof(tmp_arr));
        } else {
            gnss_log("ringbuffer read error\n");
        }
        usleep(5000);
    }
exit:
    pthread_exit(NULL);
}

static void gnss_uart_pthread(void* hdl)
{
    struct mbtk_gnss_handle_t *gnss_handle = (struct mbtk_gnss_handle_t *)hdl;
    int ret = 0;
    char buf[MBTK_UART_RECV_BUFFER_SIZE] = {0};

    pthread_detach(pthread_self());

    memset(buf, 0, sizeof(buf));
    mbtk_gnss_handle->getap_status = 0;

    while(mbtk_gnss_handle->inited)
    {
        while(mbtk_gnss_handle->getap_status){
            // 在读AP_DATA星历时，不能输出NMEA
            usleep(100000);
        }
        if(nmea_state == 1)
        {
            nmea_reading = 1;
            ret = mopen_gnss_read(gnss_handle->dev_fd, buf, MBTK_UART_RECV_BUFFER_SIZE);
            nmea_reading = 0;
            if(ret > 0) {
                printf("%s\n",buf);
                ring_buffer_queue_arr(&gnss_handle->ring_buffer, buf, ret);
                memset(buf, 0, sizeof(buf));
            } else {
                gnss_log("read error\n");
            }
        }
        usleep(100000);
    }

    pthread_exit(NULL);
}

ssize_t deal_read(int fd, void *buf, size_t count)
{
    int ret = 0;

#if 0
    ret = read(fd, buf, count);
    return ret;
#else
    while (1)
    {
        ret = read(fd, buf, count);
        if (ret == 0) {
            printf("read serial return 0, please check serial device.\n");
            exit(-5);
        }
        if(ret < 0) {
            if ((errno == EAGAIN) || (errno == EINTR)) {
                printf("read serial return -1, errno = %d, retry.\n", errno);
                continue;
            } else {
                printf("read serial return -1, errno = %d, please check serial device.\n", errno);
                exit(-5);
            }
        }
        return ret;
    }
#endif
}

static int select_read( int fd, int timeout ) //1ms
{
    fd_set set;
    struct timeval t;
    int ret;
    int i = timeout;

    do {
        FD_ZERO(&set);
        FD_SET(fd, &set);
        t.tv_sec = 0;
        t.tv_usec = 100;

        ret = select(FD_SETSIZE, &set, NULL, NULL, &t );
        if(ret == 0) continue;
        if(ret < 0 && errno == EINTR)continue;
        else return ret;
    } while(i--);

    return ret;
}

// check: 是否需要校验
static int mopen_uart_change(int fd, int check)
{
    int rByte = 0, i = 20;
    char name[32];
    char rbuf[1024];

    sprintf(name,"$CFGPRT,1,h0,115200,129,3\r\n");
    rByte = write( fd, name, strlen(name));
    tcdrain(fd);//阻塞直到发送完成

    set_baudrate(fd, B115200);
    usleep(200000);
    tcflush(fd, TCIFLUSH);//清空缓存区数据

    if (0 == check)
        return 0;
    do{
        rByte = 0;
        memset(rbuf, 0, sizeof(rbuf));
        if(select_read(fd, 1) > 0) {
            rByte = deal_read(fd, &rbuf, sizeof(rbuf) - 1);
            rbuf[rByte] = 0;
            gnss_log("%s: %s", __FUNCTION__, rbuf);
        }
        if(strstr(rbuf, "$")) {
            return 0;
        } else {
            gnss_log("%d rByte = %d, [%s]\n", 20 - i, rByte, rbuf);
        }
        usleep(5000 * 100);
    }while(i--);

    return -1;
}

#define GPS_DEV "/sys/devices/soc.0/d4000000.apb/mbtk-dev-op/gps_power"

int lynq_open_gps(int state)
{
    char cmd[128] = "echo '86' > /sys/class/gpio/export && cd /sys/class/gpio/gpio86";
    int ret;

    if(0 == state)
    {
        nmea_state = 0;
        for(int cont = 0; cont < 200; cont++)
        {
            if (nmea_reading == 0)
            {
                break;
            }
            usleep(10000);
        }
        ret = exec_cmd(cmd,NULL);
        if(0 != ret)
        {
            printf("stop fail %x\n", ret);
        }
        char cmd2[128] = "echo '0' > /sys/class/gpio/gpio86/value && echo '86' > /sys/class/gpio/unexport";
        ret = exec_cmd(cmd2,NULL);
        if(0 != ret)
        {
            printf("stop fail %x\n", ret);
        }   
    }

    if(1 == state)
    {
        nmea_state = 1;

        ret = exec_cmd(cmd,NULL);
        if(0 != ret)
        {
            printf("start fail %x\n", ret);
            return -1;
        }
        char cmd3[128] = "echo '1' > /sys/class/gpio/gpio86/value && echo '86' > /sys/class/gpio/unexport";
        ret = exec_cmd(cmd3,NULL);
        if(0 != ret)
        {
            printf("start fail %x\n", ret);
            return -1;
        }
    }
    return 0;
}

int lynq_gnss_client_init(uint32 *ph_gnss)
{
    int ret;
    if(ph_gnss == NULL) {
        printf("ARG error or has inited.");
        return -1;
    }

    if (mbtk_gnss_handle) {
        printf("GNSS has inited.");
        *ph_gnss = (uint32)mbtk_gnss_handle;
        return 0;
    }
    mbtk_gnss_handle = (mbtk_gnss_handle_t*)malloc(sizeof(struct mbtk_gnss_handle_t));
    if(NULL == mbtk_gnss_handle)
    {
        printf("malloc memory error\n");
        return -3;
    }
    memset(mbtk_gnss_handle, 0, sizeof(struct mbtk_gnss_handle_t));
    memset(&mopen_gnss_device_info, 0, sizeof(mopen_gnss_device_info));
    sleep(1);
    lynq_open_gps(0);

    mbtk_gnss_handle->dev_fd = mopen_gnss_open(MBTK_GNSS_DEV, 0);
    ret = auto_set_uart_baudrate(mbtk_gnss_handle->dev_fd);
    if(-1 == ret) {
        ret = -2;
        goto err;
    } else if (B9600 == ret) {
        ret = mopen_uart_change(mbtk_gnss_handle->dev_fd, 1);
        if(ret)
        {
            printf("GNSS Uart set B115200 error\n");
            mopen_gnss_close(mbtk_gnss_handle->dev_fd);
            return -1;
        }
    }
    
    printf("Gnss Config Uart Baudrate Successful.\n");
    
    mbtk_gnss_handle->rb = (char*)malloc(MBTK_UART_RECV_BUFFER_SIZE);
    if(NULL == mbtk_gnss_handle->rb)
    {
        printf("malloc memory error\n");
        return -1;
    }

    ring_buffer_init(&mbtk_gnss_handle->ring_buffer,
                     mbtk_gnss_handle->rb,
                     MBTK_UART_RECV_BUFFER_SIZE);
    mbtk_gnss_handle->inited = 1;
    pthread_mutex_init(&mbtk_gnss_handle->_cond_mutex, NULL);
    pthread_create(&mbtk_gnss_handle->uart_pthread, NULL, (void* (*)(void*))gnss_uart_pthread, (void *)mbtk_gnss_handle);
    pthread_create(&mbtk_gnss_handle->gnss_pthread, NULL, (void* (*)(void*))gnss_info_pthread, (void *)mbtk_gnss_handle);
    mopen_gnss_get_ant_state_info((uint32)mbtk_gnss_handle);
    lynq_gnss_get_device_info((uint32)mbtk_gnss_handle);
    mopen_gnss_get_nmea_config((uint32)mbtk_gnss_handle);
    lynq_gnss_get_uart((uint32)mbtk_gnss_handle);
    lynq_gnss_get_aidinfo((uint32)mbtk_gnss_handle);
    *ph_gnss = (uint32)mbtk_gnss_handle;
    mbtk_gnss_handle->phandle = ph_gnss;

    return 0;
err:
    mopen_gnss_close(mbtk_gnss_handle->dev_fd);
    lynq_open_gps(0);
    firmware_extren_state = 0;
    if (mbtk_gnss_handle) free(mbtk_gnss_handle);
    mbtk_gnss_handle = NULL;

    return ret;
}
static int _kill_pthread(pthread_t pid, int kill)
{
    int ret;

    if (kill) {
        ret = pthread_cancel(pid);
        pthread_join(pid, NULL);
    }
    do{
        ret = pthread_kill(pid, 0);
        if(ret == ESRCH)
            gnss_log("The specified thread does not exist or has terminated\n");
        else if(ret == EINVAL)
            gnss_log("Useless signal\n");
        else
            gnss_log("The thread exists\n");
        usleep(100000);
    }while(0 == ret);

    return 0;
}
int lynq_gnss_client_deinit(uint32 h_gnss)
{
    int ret;
    struct mbtk_gnss_handle_t *gnss_handle = (struct mbtk_gnss_handle_t *)h_gnss;

    if(h_gnss == NULL)
    {
        gnss_log("ARG error or not inited.");
        printf("ARG error or not inited.\n");
        return -1;
    }
    gnss_handle->inited = 0;
#if TTFF_TEST
    // ttff测试线程在运行，而且不是临时固件模式
    if (gnss_handle->ttff_pid &&
            0 == firmware_extren_state &&
            0 == location_test) {
        gnss_log("kill thread ttff.\n");
        _kill_pthread(gnss_handle->ttff_pid, 1);
    }
#endif
    gnss_log("kill thread info 0.\n");
    _kill_pthread(gnss_handle->gnss_pthread, 0);
    gnss_log("kill thread uart.\n");
    _kill_pthread(gnss_handle->uart_pthread, 0);
    mopen_gnss_close(gnss_handle->dev_fd);
    ret = lynq_open_gps(0);
    if(ret)
    {
        printf("GNSS close init error\n");
        return -1;
    }

    firmware_extren_state = 0;
    if (gnss_handle->rb) free(gnss_handle->rb);
    free((void *)h_gnss);
    return 0;
}

int lynq_gnss_send_cmd(uint32 h_gnss, const char *cmd, int cmd_len)
{
    struct mbtk_gnss_handle_t *gnss_handle = (struct mbtk_gnss_handle_t *)h_gnss;
    return  mopen_gnss_write(gnss_handle->dev_fd, cmd, cmd_len);
}


int lynq_gnss_get_device_info(uint32 h_gnss)
{
    int ret;
    char *send_buf = "$PDTINFO\r\n";
    if(0 == h_gnss)
    {
        printf("%s handler invalid.\n", __func__);
        return -1;
    }

    ret = lynq_gnss_send_cmd(h_gnss, send_buf, strlen(send_buf));
    if(ret < 0)
    {
        printf("lynq_gnss_client_init FAIL.	ret:%d\n",ret);
        return -1;
    }

    return 0;
}


/**
 * @brief      lynq_gnss_get_aidinfo
 *
 * @details    查询辅助数据状态
 *             $CFGAID,0,D7FBFBDF,00000000,08*47
 * @param      param
 *
 * @return     return type
 */
int lynq_gnss_get_aidinfo(uint32 h_gnss)
{
    int ret;
    // char *send_buf = "$AIDINFO\r\n";
    char *send_buf = "$CFGAID,0\r\n";
    if(0 == h_gnss)
    {
        printf("%s handler invalid.\n", __func__);
        return -1;
    }

    ret = lynq_gnss_send_cmd(h_gnss, send_buf, strlen(send_buf));
    if(ret < 0)
    {
        printf("%s FAIL. ret:%d\n", __FUNCTION__, ret);
        return -1;
    }
    return 0;
}
/**
 * @brief      lynq_gnss_get_uart
 *
 * @details    get uart config info.
 *             $CFGPRT,1,h0,9600,129,3*57
 *
 * @param      param
 *
 * @return     return type
 */
int lynq_gnss_get_uart(uint32 h_gnss)
{
    int ret;
    char *send_buf = "$CFGPRT,1\r\n";
    // char *send_buf = "$CFGPRT,2\r\n";
    if(0 == h_gnss)
    {
        printf("%s handler invalid.\n", __func__);
        return -1;
    }

    ret = lynq_gnss_send_cmd(h_gnss, send_buf, strlen(send_buf));
    if(ret < 0)
    {
        printf("lynq_gnss_client_init FAIL.	ret:%d\n",ret);
        return -1;
    }
    return 0;
}

int mopen_gnss_set_uart(uint32 h_gnss, int baudrate)
{
    int ret;
    char send_buf[28] = {0};
    // char *send_buf = "$CFGPRT,1,h0,9600,1,3\r\n";

    if(0 == h_gnss)
    {
        printf("%s handler invalid.\n", __func__);
        return -1;
    }
    sprintf(send_buf, "$CFGPRT,1,h0,%d,1,3\r\n", baudrate);
    gnss_log("%s %s", __FUNCTION__, send_buf);
    ret = lynq_gnss_send_cmd(h_gnss, send_buf, strlen(send_buf));
    if(ret < 0)
    {
        printf("lynq_gnss_client_init FAIL.	ret:%d\n",ret);
        return -1;
    }
    return 0;
}


int mopen_gnss_get_ant_state_info(uint32 h_gnss)
{
    int ret;
    char *cmd1_buf = "$ANTSTAT,1\r\n";
    char *cmd2_buf = "$ANTSTAT1\r\n";

    if(0 == h_gnss)
    {
        printf("%s handler invalid.\n", __func__);
        return -1;
    }
    ret = lynq_gnss_send_cmd(h_gnss, cmd1_buf, strlen(cmd1_buf));
    if(ret < 0)
    {
        gnss_log("%s FAIL. ret:%d\n", __func__, ret);
        return -1;
    }
    ret = lynq_gnss_send_cmd(h_gnss, cmd2_buf, strlen(cmd2_buf));
    if(ret < 0)
    {
        gnss_log("%s FAIL. ret:%d\n", __func__, ret);
        return -1;
    }
    return 0;
}

int mopen_gnss_get_nmea_config(uint32 h_gnss)
{
    int ret;
    char *send_buf = "$CFGNMEA\r\n";

    if(0 == h_gnss)
    {
        printf("%s handler invalid.\n", __func__);
        return -1;
    }
    ret = lynq_gnss_send_cmd(h_gnss, send_buf, strlen(send_buf));
    if(ret < 0)
    {
        gnss_log("%s FAIL. ret:%d\n", __func__, ret);
        return -1;
    }
    return 0;
}

int mopen_gnss_add_rx_msg_handler(uint32 h_gnss, gnss_handler_func_t handler_ptr)
{
    struct mbtk_gnss_handle_t *gnss_handle = (struct mbtk_gnss_handle_t *)h_gnss;
    if(0 == h_gnss && NULL == handler_ptr)
    {
        printf("%s handler invalid.\n", __func__);
        return -1;
    }
    gnss_handle->gnss_handler_func = handler_ptr;
    return 0;
}

#define AGNSS_TLE_FILE  "/tmp/agnss_tle"




#define AGNSS_CONFIG_FILE  "/etc/mbtk/gps.conf"


/**
 * @brief
 *
 * @details    detailed description
 *
 * @param      mode
 *                  0 : stop
 *                  1 : 输出一次坐标
 *                  2 : periodic
 *                  3 : 输出nmea数据到回调函数
 *    0 - stop, 1 - single, 2 - periodic, 3 - start
 *
 * @return     return type
 */
int mopen_gnss_set_mode(uint32 h_gnss, int mode)
{
    struct mbtk_gnss_handle_t *gnss_handle = (struct mbtk_gnss_handle_t *)h_gnss;
    if(0 == h_gnss)
    {
        printf("%s handler invalid.\n", __func__);
        return -1;
    }
    gnss_handle->mode = mode;

    return 0;
}

int mopen_gnss_print_version(uint32 h_gnss)
{
    printf("*************************\n");
    printf("-Pn: %s\n dc: %s\n hv: %s\n fw: %s\n pn: %s\n sn: %s\n ",
           mopen_gnss_device_info.product_name,
           mopen_gnss_device_info.dev_config,
           mopen_gnss_device_info.hw_ver,
           mopen_gnss_device_info.fw_ver,
           mopen_gnss_device_info.pn,
           mopen_gnss_device_info.sn);
    printf("Uart bandrate: %d\n" , mopen_gnss_device_info.usrt_bandrate);
    printf("*************************\n");

    return 0;
}

/**
* @brief 使用popen调用终端并获取执行结果
*
* @param[in] cmd 命令内容
* @param[out] result 保存结果的地址
* @return 0或1 执行状态，成功或失败
*/
int exec_cmd(const char *cmd, char *result)
{
    FILE *pipe = popen(cmd, "r");
    if(!pipe)
        return -1;

    char buffer[256] = {0};
    while(!feof(pipe))
    {
        if(fgets(buffer, 256, pipe))
        {
            memset(buffer, 0, sizeof(buffer));
        }
    }
    pclose(pipe);
    return 0;
}

#define GNSS_AP_DATA_FILE  "/etc/mbtk/rtm.bin"

int lynq_gnss_get_ap_data(void)
{
    int state = 0;
    uint32 *ph_gnss = NULL;
    gnss_handler_func_t cb;
    int current_mode;
    const char* cmd = "mbtk_gnss_update getap -d /dev/ttyS2 -b 115200 -a /etc/mbtk/rtm.bin";

    if(access(GNSS_AP_DATA_FILE, F_OK) != -1) {
        unlink(GNSS_AP_DATA_FILE);
    }
    mbtk_gnss_handle->getap_status = 1;
    sleep(1);
    printf("Mopen Gnss Get Ap Data -> \n");
    int ret = exec_cmd(cmd, NULL);

    usleep(100000);
    mbtk_gnss_handle->getap_status = 0;
    if(0 != ret) {
        printf("Gnss getap result: %x\n", ret);
        return -1;
    }
    LOGI("%s %d: %d.\n", __FUNCTION__, __LINE__, ret);
    return ret;
}
/*
  sync : 1
 */
int mopen_gnss_firmware_update(void)
{
    int state = 0;
    uint32 *ph_gnss = NULL;
    gnss_handler_func_t cb;
    int current_mode;
    const char* cmd_1 = "mbtk_gnss_update downbl -d /dev/ttyS2 \
            -l /mnt/userdata/bootloader_r3.0.0_build6773_uartboot_921600.pkg";

            const char* cmd_2 = "mbtk_gnss_update sendap -d /dev/ttyS2 -b 921600 -a /etc/mbtk/rtm.bin";

    const char* cmd_3 = "mbtk_gnss_update downfw -d /dev/ttyS2 -b 921600\
            -f /mnt/userdata/UC6228CI-R3.2.10.100Build8019_mfg.pkg";
            // /etc/mbtk/UC6228CI-R3.4.0.0Build7258_mfg.pkg

            if (mbtk_gnss_handle) {
            printf("%s gnss thread runing!!!\n", __func__);
            if (mbtk_gnss_handle->gnss_handler_func)
            cb = mbtk_gnss_handle->gnss_handler_func;

            ph_gnss = mbtk_gnss_handle->phandle;
            current_mode = mbtk_gnss_handle->mode;
            // 主线程是否在运行
            if (mbtk_gnss_handle->gnss_pthread)
            state = 1;
            lynq_gnss_client_deinit((uint32)mbtk_gnss_handle);
}
            printf("Mopen Gnss Bootloader Update -> \n");
    int ret = exec_cmd(cmd_1, NULL);
    if(0 != ret) {
        printf("Gnss update result: %x\n", ret);
        return -1;
    }

    /*if(access(GNSS_AP_DATA_FILE, F_OK) != -1)
    {
        printf("Mopen Gnss Send AP Data  -> \n");
        ret = exec_cmd(cmd_2, NULL);
        if(0 != ret) {
            printf("Gnss update result: %x\n", ret);
        }
    }*/

    printf("Mopen Gnss Firmware Update -> \n");
    ret = exec_cmd(cmd_3, NULL);
    if(0 != ret) {
        printf("Gnss update result: %x\n", ret);
        return -2;
    }

    if (0 == firmware_extren_state)
        firmware_extren_state = 1;
    if (state && ph_gnss) {
        ret = lynq_gnss_client_init(ph_gnss);
        if (cb) {
            mbtk_gnss_handle->gnss_handler_func = cb;
            mbtk_gnss_handle->mode = current_mode;
        }
    }

    LOGI("%s %d: %d.\n", __FUNCTION__, __LINE__, ret);
    return ret;
}

/*update ephemeris from network*/
#define AGNSS_TLE_FILE "/mnt/userdata/agnss_tle" //deposit ephemeris

/*Injection ephemeris AGNSS_TLE_FILE*/
int lynq_gnss_Injection_ephemeris(uint32 h_gnss)
{
    int ret;
    int agnss_fd = 0;
    int size = 0;

    if(0 == h_gnss)
    {
        printf("%s handler invalid.\n", __func__);
        return -1;
    }

    agnss_fd = open(AGNSS_TLE_FILE, O_RDWR);
    if (agnss_fd <= 0)
    {
        printf("%s open file FAIL. errno:%d\n", __FUNCTION__, errno);
        return -1;
    }
    char* databuf = (char*)malloc(128);
    if(databuf == NULL)
    {
        gnss_log("%s malloc() fail.", __FUNCTION__);
        return -1;
    }
    memset(databuf, 0, 128);
    while(0 < (size = read(agnss_fd, databuf, 128)))
    {
        gnss_log("%s Write[%d]\r\n", __FUNCTION__, size);
        ret = lynq_gnss_send_cmd(h_gnss, databuf, size);
        if(ret < 0)
        {
            printf("send cmd FAIL. ret:%d\n",  ret);
            break;
        }
        memset(databuf, 0, 128);
    }
    close(agnss_fd);
    free(databuf);
    return 0;

}
#ifdef __cplusplus
}
#endif
