#include <stdio.h>
#include <stdint.h>  /*just for uint64_t in lynq_query_cell_info 20220819 hq*/
#include <sys/types.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include <signal.h>
#include <errno.h>
#include <fcntl.h>
#include <stdarg.h>
#include <climits>
#include "liblog/lynq_deflog.h"
#include "lynq_network.h"
#include "lynq_module_common.h"
#include "lynq_module_state_buffer.h"

#define USER_LOG_TAG "LYNQ_NETWORK"

static timer_t s_state_buffer_timer[NETWORK_STATE_BUFFER_TYPE_MAX];

static int s_state_buffer_sig_value[NETWORK_STATE_BUFFER_TYPE_MAX]={2,3,4};

static int s_state_buffer_valid[NETWORK_STATE_BUFFER_TYPE_MAX]={false,false,false};

static char* s_state_buffer_file[NETWORK_STATE_BUFFER_TYPE_MAX]={
     "/mnt/userdata/ril/network/.network_register_voice_buffer",
     "/mnt/userdata/ril/network/.network_register_data_buffer",
     "/mnt/userdata/ril/network/.network_register_ims_buffer"
};

#define STATE_BUFFER_CONFIG_FILE  "/mnt/userdata/ril/network/.state_buffer_cfg"
typedef struct{
    int valid_period[NETWORK_STATE_BUFFER_TYPE_MAX];   
}network_state_buffer_cfg;
static network_state_buffer_cfg s_state_buffer_cfg;
static pthread_mutex_t mtx = PTHREAD_MUTEX_INITIALIZER;

void get_state_buffer_timer_length_from_file()
{
    int fd_backup;
    int ret;
    memset(&s_state_buffer_cfg,0,sizeof (s_state_buffer_cfg));
    fd_backup = open(STATE_BUFFER_CONFIG_FILE,O_RDONLY | O_CREAT,0777);
    if (fd_backup < 0) {
        LYERRLOG("%s open file faile, error is %s\n",__func__,strerror(errno));       
        return;
    }
      
    ret = read(fd_backup,&s_state_buffer_cfg,sizeof (s_state_buffer_cfg));
    if(ret<=0)
    {
        LYERRLOG("%s read fail, ret is %d",__func__,ret);
        memset(&s_state_buffer_cfg,0,sizeof (s_state_buffer_cfg));
    } 
    else 
    {
        LYINFLOG("%s suc",__func__);  
    }    
    close(fd_backup);   
   
    return;   
}

int set_state_buffer_timer_length_to_file()
{
    int fd_backup;
    int ret;
    fd_backup = open(STATE_BUFFER_CONFIG_FILE,O_WRONLY | O_CREAT,0777);
    if (fd_backup < 0) {
        LYERRLOG("%s open file fail, error is %s\n",__func__,strerror(errno));       
        return -1;
    }
      
    ret = write(fd_backup,&s_state_buffer_cfg,sizeof (s_state_buffer_cfg));
    if(ret<=0)
    {
        LYERRLOG("%s write fail, ret is %d",__func__,ret);         
    } 
    else 
    {
        LYINFLOG("%s suc",__func__);  
    }    
    close(fd_backup);   
   
    return ret>0? 0:-2;   
}

void start_network_timer(timer_t timer, int signal_value, int milliseconds) {
    LYINFLOG("start_network_timer(), timer_id=%ld, signal_value=%d, time=%d",(long)timer, signal_value, milliseconds);

    struct itimerspec expire;
    expire.it_interval.tv_sec = 0;
    expire.it_interval.tv_nsec = 0;
    expire.it_value.tv_sec = milliseconds/1000;
    expire.it_value.tv_nsec = (milliseconds%1000)*1000000;
    if (timer_settime(timer, 0, &expire, NULL) == -1) {
        LYERRLOG("timer_settime  failed reason=[%s]", strerror(errno));
    }
}

void stop_network_timer(timer_t timer, int signal_value) {
    LYINFLOG("stop_network_timer(), timer_id=%ld, signal_value=%d", (long)timer, signal_value);
    struct itimerspec timespec;
    if(timer_gettime(timer, &timespec) == -1) {
        LYERRLOG("stop_network_timer(), get time fail(%s)", strerror(errno));
        return;
    }
    LYINFLOG("stop_network_timer(), tv_sec=%ld, tv_nsec=%ld",timespec.it_value.tv_sec, timespec.it_value.tv_nsec);
    if((timespec.it_value.tv_sec == 0)  && (timespec.it_value.tv_nsec == 0) ) {
        LYINFLOG("stop_network_timer(), timer_id(%ld) had stopped, just return", (long)timer);
        return;
    } else {
        start_network_timer(timer, signal_value, 0);
    }
}

void network_timer_handler(sigval_t sig) {    
    LYINFLOG("network_timer_handler, sig_value: %d", sig.sival_int);
    int s;   
    s = pthread_mutex_lock(&mtx);
    if(s != 0) {
        LYERRLOG("network_timer_handler, pthead_mutex_lock fail");
    }   
    for(int i=0;i<NETWORK_STATE_BUFFER_TYPE_MAX;i++)
    {
        if(sig.sival_int == s_state_buffer_sig_value[i]) 
        {
            s_state_buffer_valid[i]=false; 
            break;
        }        
    }    
    s = pthread_mutex_unlock(&mtx);
    if(s != 0) {
        LYERRLOG("network_timer_handler, pthread_mutex_unlock fail");
    }
}

void init_network_timer(timer_t* timer, int signal_value) 
{
    struct sigevent sevp;
    memset(&sevp, 0, sizeof(sevp));
    sevp.sigev_value.sival_int = signal_value;
    sevp.sigev_notify = SIGEV_THREAD;
    sevp.sigev_notify_function = network_timer_handler;

    if(timer_create(CLOCK_MONOTONIC, &sevp, timer) == -1) {
        LYERRLOG("init_network_timer()  failed reason=[%s]", strerror(errno));
    }
    LYINFLOG("init_network_timer(), timer_Id = %ld, signal_value=%d", (long)(*timer), signal_value);
}

void init_network_timer_all() {

    for(int i=0;i<NETWORK_STATE_BUFFER_TYPE_MAX;i++){
        init_network_timer(&s_state_buffer_timer[i],s_state_buffer_sig_value[i]); 
        s_state_buffer_valid[i] = false;
    }
    get_state_buffer_timer_length_from_file();
}

void deinit_network_timer_all() 
{
    for(int i=0;i<NETWORK_STATE_BUFFER_TYPE_MAX;i++){
        stop_network_timer(s_state_buffer_timer[i],s_state_buffer_sig_value[i]); 
        s_state_buffer_valid[i] = false;
    }
}

int is_period_equal_not_valid(int period)
{
    return period<=0 || (period > INT_MAX/1000);
}

void set_state_buffer_valid(lynq_network_state_buffer_type type,int valid)
{
    if(is_period_equal_not_valid(s_state_buffer_cfg.valid_period[type]))
    {
        valid=false;
    }
    s_state_buffer_valid[type] = valid;
    if(s_state_buffer_valid[type]==false)
    {
        stop_network_timer(s_state_buffer_timer[type], s_state_buffer_sig_value[type]);
    }
    else 
    {
        start_network_timer(s_state_buffer_timer[type], s_state_buffer_sig_value[type],s_state_buffer_cfg.valid_period[type]*1000);
    }
    LYINFLOG("%s type %d valid is %d",__func__,type,valid);    
}

int get_state_buffer_valid(lynq_network_state_buffer_type type)
{
    return s_state_buffer_valid[type];
}

void set_state_to_buf(lynq_network_state_buffer_type type,...)
{
    int ret;
    int fd_backup;
    
    if(is_period_equal_not_valid(s_state_buffer_cfg.valid_period[type]))
    {
        LYINFLOG("%s type %d timer length %d is equal not valid",__func__,type,s_state_buffer_cfg.valid_period[type]);
        return;
    }
        
    fd_backup = open(s_state_buffer_file[type],O_WRONLY | O_CREAT,0777);

    if (fd_backup < 0) {
        LYERRLOG("%s Error opening registrate buffer file: %s\n",__func__,strerror(errno));
        return ;
    }

    va_list pArgs;
    va_start(pArgs, type);

    int int_num;
    int i,int_value;

    int_num = va_arg(pArgs, int);
    LYINFLOG("%s int_num is %d",__func__,int_num);
    for(i=0;i<int_num;i++)
    {
        int_value=va_arg(pArgs, int);
        LYINFLOG("%s int_value is %d",__func__,int_value);
        ret = write(fd_backup,&int_value,sizeof (int));
        if(ret<=0)
        {
            LYERRLOG("%s write int %d value %d fail, ret is %d",__func__,i,int_value,ret);
            goto set_state_error_done;  
        }   
        
    }

    int stringNum;
    char * p_value;
    int stringLen;

    stringNum = va_arg(pArgs, int);
    LYINFLOG("%s stringNum is %d",__func__,stringNum);
    for(i=0;i<stringNum;i++)
    {
        p_value=va_arg(pArgs, char*);
        stringLen=strlen(p_value)+1;
        LYINFLOG("%s p_value is %s,stringlen is %d",__func__,p_value,stringLen);
        ret = write(fd_backup,&stringLen,sizeof(int));
        if(ret<=0)
        {
            LYERRLOG("%s write %i stringlen %d fail, ret is %d",__func__,i,stringLen,ret);
            goto set_state_error_done;
        }   
        ret = write(fd_backup,p_value,stringLen);  
        if(ret<=0)
        {
            LYERRLOG("%s write string %d stringlen %d fail, ret is %d",__func__,i, stringLen,ret);
            goto set_state_error_done;  
        }   
    }   

    va_end(pArgs);

    sync();
    close(fd_backup); 
    set_state_buffer_valid(type,true);  
    LYINFLOG("%s type %d suc",__func__,type);
    return;

set_state_error_done:    
    set_state_buffer_valid(type,false);
    close(fd_backup); 
    return;   
}

int get_state_from_buf(lynq_network_state_buffer_type type,...)
{
    int ret;
    int fd_backup;
    
    if(!get_state_buffer_valid(type))
    {
        LYINFLOG("%s type %d buffer not valid",__func__,type);
        return 1;
    }

    fd_backup = open(s_state_buffer_file[type], O_RDONLY);

    if (fd_backup < 0) {
        LYERRLOG("%s Error opening type %d buffer file: %s\n",__func__,type,strerror(errno));
        set_state_buffer_valid(type,false); 
        return -1;
    }

    va_list pArgs;
    va_start(pArgs, type);
    int int_p_num; 
    int i;
    int* int_p_value;

    int_p_num= va_arg(pArgs, int); 
    LYINFLOG("%s int_num is %d",__func__,int_p_num);
    for(i=0;i<int_p_num;i++)
    {
        int_p_value=va_arg(pArgs, int*);        
        ret = read(fd_backup,int_p_value,sizeof (int));
        LYINFLOG("%s int_p_value is %d",__func__,(*int_p_value));
        if(ret<=0)
        {
            LYERRLOG("%s read int %d fail, ret is %d",__func__,i,ret);
            goto get_state_error_done;  
        }   
        
    }

    int stringNum;
    char * p_value;
    int stringLen;

    stringNum = va_arg(pArgs, int);
    LYINFLOG("%s stringNum is %d",__func__,stringNum);
    for(i=0;i<stringNum;i++)
    {
        ret = read(fd_backup,&stringLen,sizeof (int));
        if(ret<=0)
        {
            LYERRLOG("%s get %d string len fail, ret is %d",__func__,i,ret);
            goto get_state_error_done;
        }
        LYINFLOG("%s stringlen is %d",__func__,stringLen);
        p_value=va_arg(pArgs, char*);          
        ret = read(fd_backup,p_value,stringLen);
        if(ret<=0)
        {
            LYERRLOG("%s get %d string fail, stringlen is %d,  ret is %d",__func__, stringLen,ret);
            goto get_state_error_done;
        } 
        LYINFLOG("%s p_value is %s",__func__,p_value);
    } 
    va_end(pArgs);

    close(fd_backup);   
    LYINFLOG("%s type %d suc",__func__,type);
    return 0;  
    
get_state_error_done:
    set_state_buffer_valid(type,false); 
    close(fd_backup); 
    return -1;
}

int lynq_set_state_buffer_valid_period_of_network(lynq_network_state_buffer_type type,int valid_period)
{
    if(type>=NETWORK_STATE_BUFFER_TYPE_MAX || type <0)
    {
        LYERRLOG("%s type %d error",__func__,type);
        return -1;
    }
    if(valid_period>INT_MAX/1000 || valid_period <0)
    {
        LYERRLOG("%s type %d error",__func__,valid_period);
        return -2;
    }
   
    network_state_buffer_cfg cfg_back=s_state_buffer_cfg;
    s_state_buffer_cfg.valid_period[type]=valid_period;    
    int ret=set_state_buffer_timer_length_to_file();
    if(ret!=0)
    {
        s_state_buffer_cfg=cfg_back;
    }
    else 
    {
        set_state_buffer_valid(type,false);   
    }
    LYINFLOG("%s type %d period %d ret is %d",__func__,type,valid_period,ret);
    return ret;
}

int lynq_get_state_buffer_valid_period_of_network(lynq_network_state_buffer_type type)
{
    if(type>=NETWORK_STATE_BUFFER_TYPE_MAX || type <0)
    {
        LYERRLOG("%s type %d error",__func__,type);
        return 0;
    }
    return s_state_buffer_cfg.valid_period[type];
}
