/*
 * 
 * Copyright (C)  2023
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details
 *
 * FileName : log_usb.c
 * This program Capture module's trace log.
 */

#include "log_usb.h"

int zLogAgt_Usb_Close(usbdev_t *udev)
{
    if (udev == NULL)
        return -1;
    
    if (udev->ttyfd >= 0) {
        close(udev->ttyfd);
        udev->ttyfd = -1;
    }
    return 0;
}

static int poll_wait(int poll_fd, short events, int timeout_msec)
{
    struct pollfd pollfds[] = {{poll_fd, events, 0}};

    int ret = poll(pollfds, 1, timeout_msec);
    if (ret == 0) {
        return ETIMEDOUT;
    } else if (ret < 0) {
        dbg_time("%s errno:%d(%s) \n", __func__, errno, strerror(errno));
    }

    if (pollfds[0].revents & (POLLERR | POLLHUP | POLLNVAL)) {
        return EIO;
    }

    if (pollfds[0].revents & (events)) {
        return 0;
    }

    return EIO;
}

int zLogAgt_Usb_Write(usbdev_t *udev, char *pbuf, int len)
{
    int write_size = 0;
    int timeout_msec = 3000;

    while (write_size < len)
    {
        int retval = -1;
   
        if (udev->ttyfd >= 0) {
            if (poll_wait(udev->ttyfd, POLLOUT, timeout_msec)) {
                break;
            }
            retval = write(udev->ttyfd, pbuf + write_size, len - write_size);
            if (retval < 0 && errno != EINTR) {
                dbg_time("%s write:%d, errno:%d(%s)\n", __func__, retval, errno, strerror(errno));
                return retval;
            }
        }
        write_size += retval;        
    }

    return write_size;
}

int zLogAgt_Usb_Read(usbdev_t *udev, char *pbuf, int len)
{
    int read_size = 0;
    int timeout_msec = 1000;

    while (read_size < len)
    {
        int retval = -1;
        if (udev->ttyfd >= 0) {
            if (poll_wait(udev->ttyfd, POLLIN, timeout_msec)) {
                break;
            }
            
            retval = read(udev->ttyfd, pbuf + read_size, len - read_size);
            if (retval <= 0 && errno != EINTR) {
                dbg_time("%s read:%d, errno:%d(%s)\n", __func__, retval, errno, strerror(errno));
                return retval;
            }
        }
        read_size += retval;
    }
    return read_size;
}

