/***********************************************************************
* Copyright (C) 2019, Sanechips Corporation.
*
* File Name: 	 zx29_kpd5x6_soft.c
* File Mark:  	 None
* Description:   None
* Others:
* Version:
* Author:
* Date:			2019-11-19
*
* History 1:  	Primary edition
*     Date:
* 	 Version:
*     Author:
*     Modification:
**********************************************************************/
/*include area*/
#include <linux/module.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/types.h>
#include <linux/input.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/platform_device.h>
#include <linux/io.h>
#include <mach/pcu.h>

#include <linux/clk.h>
#include <linux/gpio.h>
#include <mach/gpio.h>
#include <linux/irq.h>
#include <linux/delay.h>
#include <linux/wakelock.h>
#include <linux/soc/zte/pm/drv_idle.h>
#include <linux/input/zx29_keypad_5x6.h>

/*define area*/
#define KPD_MATRIX_SCAN_INTERVAL    (33)            /* millisecond */
#define KEYPAD_DBG_EN				(0)

#if KEYPAD_DBG_EN
#define KEYPAD_DBG(fmt, args...)	pr_info(fmt , ## args)
#else
#define KEYPAD_DBG(fmt, args...)
#endif
#define DBG_ENTER()                 KEYPAD_DBG("%s Enter.\n", __FUNCTION__)
#define DBG_EXIT()                  KEYPAD_DBG("%s Exit.\n", __FUNCTION__)

/* #define INT_POSEDGE                 (0x2) */       /* 10: raise edge */
#define KEY_STATUS_UP               0
#define KEY_STATUS_DOWN             1
#define	KEY_RELEASED				KEY_STATUS_UP
#define	KEY_PRESSED					KEY_STATUS_DOWN

/*global & static variable area*/
static void __iomem *KEY_BASE_ADDR_VIR;
static void __iomem *RM_MOD_RST1_REG_VIR;
//static void __iomem* RM_MOD_CLKEN2_REG_VIR;
//static void __iomem* PIN_MUX_CTRLREG13_REG_VIR;

typedef struct {
    uint8_t	st_rpt;	/* report */
    uint8_t	st_crt;	/* current */
} T_ZDrvKpd_MatrixStatus;

struct kpd_gpio_desc {
	unsigned int gpio;
	gpio_func_id func_key;
	gpio_func_id func_gpio;
	char *desc;
};

static const struct kpd_gpio_desc kpd_pins[] = {
	{83,	GPIO83_KEY_COL0,	GPIO83_GPIO83,	"kpd_col0"},
	{84,	GPIO84_KEY_COL1,	GPIO84_GPIO84,	"kpd_col1"},
	{85,	GPIO85_KEY_ROW0,	GPIO85_GPIO85,	"kpd_row0"},
	{86,	GPIO86_KEY_ROW1,	GPIO86_GPIO86,	"kpd_row1"},

	{66,	GPIO66_KEY_COL2,	GPIO66_GPIO66,	"kpd_col2"},
	{67,	GPIO67_KEY_COL3,	GPIO67_GPIO67,	"kpd_col3"},
	{68,	GPIO68_KEY_COL4,	GPIO68_GPIO68,	"kpd_col4"},
	{69,	GPIO69_KEY_ROW2,	GPIO69_GPIO69,	"kpd_row2"},
	{70,	GPIO70_KEY_ROW3,	GPIO70_GPIO70,	"kpd_row3"},
	{71,	GPIO71_KEY_ROW4,	GPIO71_GPIO71,	"kpd_row4"},

	{126,	GPIO126_KEY_COL2,	GPIO126_GPIO126,	"kpd_col2"},
	{127,	GPIO127_KEY_COL3,	GPIO127_GPIO127,	"kpd_col3"},
	{128,	GPIO128_KEY_COL4,	GPIO128_GPIO128,	"kpd_col4"},
	{129,	GPIO129_KEY_COL5,	GPIO129_GPIO129,	"kpd_col5"},
	{130,	GPIO130_KEY_ROW2,	GPIO130_GPIO130,	"kpd_row2"},
	{131,	GPIO131_KEY_ROW3,	GPIO131_GPIO131,	"kpd_row3"},
	{132,	GPIO132_KEY_ROW4,	GPIO132_GPIO132,	"kpd_row4"}
}; /* keypad gpio info */

static const struct kpd_gpio_desc *kpd_board_rows[KPD_MAX_ROW];
static const struct kpd_gpio_desc *kpd_board_cols[KPD_MAX_COLUMN];

static T_ZDrvKpd_MatrixStatus  s_kpdMatrixStatus[KPD_MAX_ROW][KPD_MAX_COLUMN] = {};
static struct wake_lock keypad_wake_lock;

static void zx29_key_set_active(void)
{
	zx_cpuidle_set_busy(IDLE_FLAG_KPD5X6);
	wake_lock(&keypad_wake_lock);
}

static void zx29_key_set_idle(void)
{
	wake_unlock(&keypad_wake_lock);
	zx_cpuidle_set_free(IDLE_FLAG_KPD5X6);
}

static void zx29_key_param_init(unsigned interval, unsigned debounce, unsigned inten)
{
	DBG_ENTER();

	if (!(KEY_BASE_ADDR_VIR))
		return;

	__raw_writel(interval, KEY_BASE_ADDR_VIR + KEY_INTERVAL);
	__raw_writel(debounce, KEY_BASE_ADDR_VIR + KEY_DEBOUNCE);
	__raw_writel(inten, KEY_BASE_ADDR_VIR + KEY_INT_EN);
	__raw_writel(1, KEY_BASE_ADDR_VIR + KEY_CFG_UPT);

	DBG_EXIT();
}

static void zx29_key_state_print(int count)
{
	int i, j;
	int key_state_index = 0, temp;
	char key_state[64];

	for (i = 0; i < KPD_MAX_ROW; i++) {
		temp = key_state_index;
		key_state[key_state_index++] = 'R';
		key_state[key_state_index++] = '0' + i;
		key_state[key_state_index++] = ':';
		key_state[key_state_index++] = '\t';
		for (j = 0; j < KPD_MAX_COLUMN; j++) {
			if (s_kpdMatrixStatus[i][j].st_crt == KEY_PRESSED) {
				key_state[key_state_index++] = 'C';
				key_state[key_state_index++] = '0' + j;
				key_state[key_state_index++] = '\t';
				if (key_state_index > sizeof(key_state) - 10)
					break;
			}
		}
		if ((key_state_index - temp) == 4)
			key_state_index = temp;
		else {
			key_state[key_state_index - 1] = '\n';
			if (key_state_index > sizeof(key_state) - 10)
				break;
		}
	}
	key_state[key_state_index] = 0;
	pr_info("kpd %d keys pressed:\n%s", count, key_state);

	return;
}

static void zx29_key_reset(void)
{
	unsigned int reg = 0;

	DBG_ENTER();

	if (!(RM_MOD_RST1_REG_VIR))
		return;

	reg = __raw_readl(RM_MOD_RST1_REG_VIR);
	//rm Key apb sw reset
	reg &= ~(1 << RM_MODRST1_KEY_SWPRST_POS);
	//rm Key work sw reset
	reg &= ~(1 << RM_MODRST1_KEY_SWWRST_POS);
	__raw_writel(reg, RM_MOD_RST1_REG_VIR);

	reg = __raw_readl(RM_MOD_RST1_REG_VIR);
	//rm Key apb sw unreset
	reg |= (1 << RM_MODRST1_KEY_SWPRST_POS);
	//rm Key work sw unreset
	reg |= (1 << RM_MODRST1_KEY_SWWRST_POS);
	__raw_writel(reg, RM_MOD_RST1_REG_VIR);

	DBG_EXIT();
}


static void zx29_key_pin_config(struct platform_device *pdev)
{
	struct zx29_5x6_keypad_platform_data *pdata = dev_get_platdata(&pdev->dev);
	uint32_t i, j, temp;
	int ret;

	DBG_ENTER();

/*pin config*/
	for (i = 0; i < ARRAY_SIZE(pdata->pin_col_row); i++) {
		if (pdata->pin_col_row[i] == 0)
			continue;
		for (j = 0; j < ARRAY_SIZE(kpd_pins); j++) {
			if (pdata->pin_col_row[i] == kpd_pins[j].gpio)
				break;
		}
		if (j == ARRAY_SIZE(kpd_pins))
			continue;
		ret = gpio_request(kpd_pins[j].gpio, kpd_pins[j].desc);
		if (ret < 0) {
			pr_err("kpd request gpio%d failed.\n", kpd_pins[j].gpio);
			continue;
		}

		if (!memcmp(kpd_pins[j].desc, "kpd_row", sizeof("key_row") - 1)) {
			temp = kpd_pins[j].desc[sizeof("key_row") - 1] - '0';
			kpd_board_rows[temp] = &kpd_pins[j];
		} else {
			temp = kpd_pins[j].desc[sizeof("key_col") - 1] - '0';
			kpd_board_cols[temp] = &kpd_pins[j];
		}
	}

	for (i = 0; i < ARRAY_SIZE(kpd_board_rows); i++) {
		if (!kpd_board_rows[i])
			continue;
		zx29_gpio_pd_pu_set(kpd_board_rows[i]->gpio, IO_CFG_PULL_UP);
		zx29_gpio_config(kpd_board_rows[i]->gpio, kpd_board_rows[i]->func_key);
		gpio_direction_input(kpd_board_rows[i]->gpio);
	}

	for (i = 0; i < ARRAY_SIZE(kpd_board_cols); i++) {
		if (!kpd_board_cols[i])
			continue;
		zx29_gpio_pd_pu_set(kpd_board_cols[i]->gpio, IO_CFG_PULL_DISABLE);
		zx29_gpio_config(kpd_board_cols[i]->gpio, kpd_board_cols[i]->func_gpio);
		gpio_direction_output(kpd_board_cols[i]->gpio, 0);
	}

/*end of pin config*/
	DBG_EXIT();

	return;
}

static void zx29_kpd5x6_matrix_mode(void)
{
	int i = 0;

	/* put row pins in keypad matrix mode */
	for (i = 0; i < ARRAY_SIZE(kpd_board_rows); i++) {
		if (!kpd_board_rows[i])
			continue;
		zx29_gpio_config(kpd_board_rows[i]->gpio, kpd_board_rows[i]->func_key);
	}

	/* have column pins output LOW in GPIO mode */
	for (i = 0; i < ARRAY_SIZE(kpd_board_cols); i++) {
		if (!kpd_board_cols[i])
			continue;
		gpio_direction_output(kpd_board_cols[i]->gpio, 0);
	}

	return;
}

static void zx29_kpd5x6_soft_mode(void)
{
	int i = 0;

	/* put row pins in GPIO input mode */
	for (i = 0; i < ARRAY_SIZE(kpd_board_rows); i++) {
		if (!kpd_board_rows[i])
			continue;
		zx29_gpio_config(kpd_board_rows[i]->gpio, kpd_board_rows[i]->func_gpio);
	}

	/* put column pins in Hi-Z output mode */
	for (i = 0; i < ARRAY_SIZE(kpd_board_cols); i++) {
		if (!kpd_board_cols[i])
			continue;
		gpio_direction_input(kpd_board_cols[i]->gpio);
	}

	return;
}

/**
 * @function bsp_udelay
 *
 * @brief Busy wait the specified number of microseconds.
 */
static void bsp_udelay(int us)
{
    volatile int i = 0;
    us <<= 8;
    while (us)
    {
        i++;
        us--;
    }
}

static unsigned int zx29_kpd5x6_check_status(void)
{
	unsigned int i, j;
	unsigned int key_pressed_cnt = 0;

	for (i = 0; i < ARRAY_SIZE(kpd_board_cols); i++) {
		if (!kpd_board_cols[i])
			continue;
		gpio_direction_output(kpd_board_cols[i]->gpio, 0);

		udelay(30);
		//bsp_udelay(20);
		//msleep(5);

		for (j = 0; j < ARRAY_SIZE(kpd_board_rows); j++) {
			if (!kpd_board_rows[j])
				continue;

			if (!gpio_get_value(kpd_board_rows[j]->gpio)) {
				s_kpdMatrixStatus[j][i].st_crt = KEY_PRESSED;
				key_pressed_cnt++;
			} else {
				s_kpdMatrixStatus[j][i].st_crt = KEY_RELEASED;
			}
		}

		gpio_direction_input(kpd_board_cols[i]->gpio);
	}

    return key_pressed_cnt;
}

static void zx29_kpd5x6_report_released_keys(struct zx29_5x6_kpd *keypad)
{
	struct zx29_5x6_keypad_platform_data *pdata = dev_get_platdata(&keypad->pdev->dev);
	unsigned char i = 0, j = 0;

	for (i = 0; i < KPD_MAX_ROW; i++) {
		for (j = 0; j < KPD_MAX_COLUMN; j++) {
			if ((s_kpdMatrixStatus[i][j].st_rpt == KEY_PRESSED) && \
				(s_kpdMatrixStatus[i][j].st_crt == KEY_RELEASED)) {
				s_kpdMatrixStatus[i][j].st_rpt = KEY_RELEASED;

				if (pdata->key_map[i][j] != KEY_RESERVED) {
					pr_info("kpd up code=%d", pdata->key_map[i][j]);
					input_report_key(keypad->input, pdata->key_map[i][j], KEY_STATUS_UP);
				}
			}
		}
	}

	input_sync(keypad->input);
}

static void zx29_kpd5x6_report_pressed_keys(struct zx29_5x6_kpd *keypad)
{
	struct zx29_5x6_keypad_platform_data *pdata = dev_get_platdata(&keypad->pdev->dev);
	unsigned char i = 0, j = 0;

	for (i = 0; i < KPD_MAX_ROW; i++) {
		for (j = 0; j < KPD_MAX_COLUMN; j++) {
			if ((s_kpdMatrixStatus[i][j].st_rpt == KEY_RELEASED) && \
				(s_kpdMatrixStatus[i][j].st_crt == KEY_PRESSED)) {
				s_kpdMatrixStatus[i][j].st_rpt = KEY_PRESSED;

				if (pdata->key_map[i][j] != KEY_RESERVED) {
					pr_info("kpd down code=%d", pdata->key_map[i][j]);
					input_report_key(keypad->input, pdata->key_map[i][j], KEY_STATUS_DOWN);
				}
			}
		}
	}

	input_sync(keypad->input);
}

/***sys driver layer***/

/* Initializing the kp Module */
static int zx29_5x6_keypad_initialize(struct zx29_5x6_kpd *keypad)
{
	int error = 0;

	DBG_ENTER();

	KEY_BASE_ADDR_VIR = ioremap(KEY_BASE_ADDR, KEY_REG_TOTAL);

	RM_MOD_RST1_REG_VIR = ioremap(RM_MOD_RST1_REG, 0x4);
#if 0
	RM_MOD_CLKEN2_REG_VIR = ioremap(RM_MOD_CLKEN2_REG, 0x4);
#endif
	//PIN_MUX_CTRLREG13_REG_VIR = ioremap(PIN_MUX_CTRLREG13_REG, 0x4);

	if (!KEY_BASE_ADDR_VIR || !RM_MOD_RST1_REG_VIR) {
		error = -ENOMEM;
		pr_err("could not remap register memory");
		goto exit;
	}

    zx29_key_pin_config(keypad->pdev);
	zx29_key_reset();
   // key_clken(1);
	zx29_key_param_init(0, 10, 1);

	KEYPAD_DBG("------------------>>keypad register read begin<<----------------\n");
	KEYPAD_DBG("KEY_INTERVAL = %08x\n",__raw_readl(KEY_BASE_ADDR_VIR + KEY_INTERVAL));
	KEYPAD_DBG("KEY_DEBOUNCE = %08x\n",__raw_readl(KEY_BASE_ADDR_VIR + KEY_DEBOUNCE));
	KEYPAD_DBG("KEY_INT_EN = %08x\n",__raw_readl(KEY_BASE_ADDR_VIR + KEY_INT_EN));
	KEYPAD_DBG("KEY_CFG_UPT = %08x\n",__raw_readl(KEY_BASE_ADDR_VIR + KEY_CFG_UPT));
	KEYPAD_DBG("KEY_INDEX = %08x\n",__raw_readl(KEY_BASE_ADDR_VIR + KEY_INDEX));
	KEYPAD_DBG("KEY_STATE1 = %08x\n",__raw_readl(KEY_BASE_ADDR_VIR + KEY_STATE1));
	KEYPAD_DBG("KEY_STATE2 = %08x\n",__raw_readl(KEY_BASE_ADDR_VIR + KEY_STATE2));
	KEYPAD_DBG("KEY_STATE3 = %08x\n",__raw_readl(KEY_BASE_ADDR_VIR + KEY_STATE3));
	//KEYPAD_DBG("PIN_MUX_CTRLREG13_REG = %08x\n",__raw_readl(PIN_MUX_CTRLREG13_REG_VIR));
	KEYPAD_DBG("------------------>>keypad register read end<<----------------\n");
exit:
	DBG_EXIT();
	return error;
}

/*interrupt handle thread*/
static irqreturn_t zx29_5x6_keypad_interrupt(int irq, void *dev_id)
{
	int key_count;
	int cnt_last = 0;

	zx29_key_set_active();
	DBG_ENTER();

	zx29_kpd5x6_soft_mode();

	while (1) {
		key_count = zx29_kpd5x6_check_status();

		zx29_kpd5x6_report_released_keys(dev_id);

		if (key_count == 0)
			break;

		/* If there's 4 or more keys pressed, there maybe a ghost key. */
		if (key_count < 4)
			zx29_kpd5x6_report_pressed_keys(dev_id);
		else if (key_count != cnt_last) {
			cnt_last = key_count;
			zx29_key_state_print(key_count);
		}

		msleep(KPD_MATRIX_SCAN_INTERVAL);
	}

	zx29_kpd5x6_matrix_mode();

	pcu_clr_irq_pending(irq);

	DBG_EXIT();
	zx29_key_set_idle();

	return IRQ_HANDLED;
}

/*probe*/
static int zx29_5x6_keypad_probe(struct platform_device *pdev)
{
	struct input_handle *handle;
	struct zx29_5x6_kpd *keypad;
	struct zx29_5x6_keypad_platform_data *pdata = dev_get_platdata(&pdev->dev);
	struct input_dev *key_dev;
	int error = 0;
	int i, j;
	struct clk *pclk=NULL;
	struct clk *wclk=NULL;
	struct resource *res_irq;

	DBG_ENTER();

	if (!pdata) {
		return -EINVAL;
	}

	keypad = kzalloc(sizeof(struct zx29_5x6_kpd), GFP_KERNEL);
	if (!keypad) {
		return -ENOMEM;
	}
	keypad->pdev = pdev;

	/*set the clk*/
	pclk = clk_get_sys("zx29_kpd.0","apb_clk");
	if (IS_ERR(pclk))
		panic("failed to get kpd pclk.");
	clk_prepare_enable(pclk);

	wclk = clk_get_sys("zx29_kpd.0","work_clk");
	if (IS_ERR(wclk))
		panic("failed to get kpd wclk.");
	//clk_set_rate(wclk,32000);
	clk_prepare_enable(wclk);

	res_irq = platform_get_resource(pdev, IORESOURCE_IRQ, 0);
	if (!res_irq) {
		error = -EINVAL;
		goto fail0;
	}
	keypad->irq = res_irq->start;

	key_dev = input_allocate_device();
	if (!key_dev) {
		error = -1;
		goto fail0;
	}
	keypad->input = key_dev;

	/* __set_bit(EV_REP, key_dev->evbit); */
	__set_bit(EV_KEY, key_dev->evbit);

    for (i = 0; i < KPD_MAX_ROW; i++) {
	    for (j = 0; j < KPD_MAX_COLUMN; j++) {
		if (pdata->key_map[i][j])
        	set_bit(pdata->key_map[i][j], key_dev->keybit);
	    }
    }

	key_dev->name = pdev->name;

	error = input_register_device(keypad->input);
	if (error < 0) {
		goto fail1;
	}
	handle = container_of(key_dev->h_list.next, struct input_handle, d_node);
	/* printk(KERN_INFO "%s registered %s/%p.\n", __FUNCTION__, handle->name, handle->private); */

	error = zx29_5x6_keypad_initialize(keypad);
	if (error < 0) {
		goto fail2;
	}

	platform_set_drvdata(pdev, keypad);

	//spin_lock_init(&kpd_lock);
	wake_lock_init(&keypad_wake_lock, WAKE_LOCK_SUSPEND, "keypad_5X6");

    pcu_clr_irq_pending(keypad->irq);
	error = request_threaded_irq(keypad->irq, NULL, zx29_5x6_keypad_interrupt, IRQF_TRIGGER_RISING | IRQF_ONESHOT, pdev->name, keypad);
	if (error < 0) {
		goto fail2;
	}
	irq_set_irq_wake(keypad->irq, 1);
    printk("probe keypad: irq:%d\n", keypad->irq);

	goto end;
/*
fail3:
	free_irq(keypad->irq, keypad); */
fail2:
	input_unregister_device(keypad->input);
fail1:
	input_free_device(key_dev);
fail0:
	kfree(keypad);
end:
	DBG_EXIT();
	return error;
}

/*remove*/
static int zx29_5x6_keypad_remove(struct platform_device *pdev)
{
	struct zx29_5x6_keypad_platform_data *pdata = dev_get_platdata(&pdev->dev);
	int i;
	struct zx29_5x6_kpd *keypad = platform_get_drvdata(pdev);

	DBG_ENTER();

	free_irq(keypad->irq, keypad);

	input_unregister_device(keypad->input);

	input_free_device(keypad->input);

	kfree(keypad);

	wake_lock_destroy(&keypad_wake_lock);

	for (i = 0; i < KPD_MAX_COLUMN + KPD_MAX_ROW; i++) {
		if (pdata->pin_col_row[i] == 0)
			continue;
		gpio_free(pdata->pin_col_row[i]);
	}

	DBG_EXIT();
	return 0;
}

static struct platform_driver zx29_5x6_keypad_driver =
{
	.probe	= zx29_5x6_keypad_probe,
	.remove	= zx29_5x6_keypad_remove,
	.driver	= {
		.name	=	"zx29_keypad",
		.owner  =	THIS_MODULE,
	},
};

static int __init zx29_5x6_keypad_init(void)
{
	return platform_driver_register(&zx29_5x6_keypad_driver);
}

static void __exit zx29_5x6_keypad_exit(void)
{
	platform_driver_unregister(&zx29_5x6_keypad_driver);
}

module_init(zx29_5x6_keypad_init);
module_exit(zx29_5x6_keypad_exit);

MODULE_AUTHOR("Sanechips");
MODULE_DESCRIPTION("Sanechips-7520v3_5x6_keypad Driver");
MODULE_LICENSE("GPL");

