/*
 * zx234290-regulator.c  --  ZTE zx234290
 *
 * Copyright 2013 ZTE Corporation.
 *
 * Author: Dong Jian <>
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under  the terms of the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the License, or (at your
 *  option) any later version.
 *
 * This driver is based on wm8350 implementation.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/err.h>
#include <linux/platform_device.h>
#include <linux/regulator/driver.h>
#include <linux/regulator/machine.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/gpio.h>
#include <linux/mfd/zx234290.h>

//#define ZX234290_REG_VCOIN	16	/* 0x02	*/
//#define ZX234290_REG_VRF	17	/* fixed	*/

/* Number of step-down converters available */
#define ZX234290_NUM_DCDC	5

/* Number of LDO voltage regulators  available */
#define ZX234290_NUM_LDO	11

/* Number of total regulators available */
#define ZX234290_NUM_REGULATOR		(ZX234290_NUM_DCDC + ZX234290_NUM_LDO)


struct zx_info {
	const char *name;
};

static struct zx_info zx234290_regs[] = {
	{
		.name = "LDO1",
	},
	{
		.name = "LDO2",
	},
	{
		.name = "LDO3",
	},
	{
		.name = "LDO4",
	},
	{
		.name = "LDO5",
	},
	{
		.name = "LDO6",
	},
	{
		.name = "LDO7",
	},
	{
		.name = "LDO8",
	},
	{
		.name = "LDO9",
	},
	{
		.name = "LDO10",
	},
	{
		.name = "LDO11",
	},
	{
		.name = "DCDC0",
	},
	{
		.name = "DCDC1",
	},
	{
		.name = "DCDC2",
	},
	{
		.name = "DCDC3",
	},
	{
		.name = "DCDC4",
	},
};

struct zx234290_reg {
	struct regulator_desc desc[ZX234290_NUM_REGULATOR];
	struct zx234290 *mfd;
	struct regulator_dev *rdev[ZX234290_NUM_REGULATOR];
	struct zx_info *info[ZX234290_NUM_REGULATOR];
	/* for read/write access */
	struct mutex io_lock;
	int mode;
	int (*get_ctrl_reg)(int);
	int normal_mode_reg;
	int sleep_mode_reg;
};


/* LDO output enable mask */
#define	ZX234290_EN1_LDO1_EN	BIT(0)
#define	ZX234290_EN1_LDO2_EN	BIT(1)
#define	ZX234290_EN1_LDO3_EN	BIT(2)
#define	ZX234290_EN1_LDO4_EN	BIT(3)
#define	ZX234290_EN1_LDO5_EN	BIT(4)
#define	ZX234290_EN1_LDO6_EN	BIT(5)
#define	ZX234290_EN1_LDO7_EN	BIT(6)
#define	ZX234290_EN1_LDO8_EN	BIT(7)

#define	ZX234290_EN2_LDO9_EN	BIT(0)
#define	ZX234290_EN2_LDO10_EN	BIT(1)
#define	ZX234290_EN2_LDO11_EN	BIT(2)
#define	ZX234290_EN2_BUCK0_EN	BIT(3)
#define	ZX234290_EN2_BUCK1_EN	BIT(4)
#define	ZX234290_EN2_BUCK2_EN	BIT(5)
#define	ZX234290_EN2_BUCK3_EN	BIT(6)
#define	ZX234290_EN2_BUCK4_EN	BIT(7)


#if 1
static const int ldo5_voltage_map[] = {
	3300,
};

/* VLA - ldo1/2/3/10	*/
static const int ldoa_voltage_map[] = {
	   0,  800,  850,  900,  950, 1000, 1050, 1100,
	1200, 1500, 1800, 2000, 2500, 2800, 3000, 3300,
};

/* VLB - VCOIN 2bit	*/
static const int ldob_voltage_map[] = {
	1800, 2500, 3000, 3300,
};

/* VLAC - ldo11	*/
static const int ldoc_voltage_map[] = {
	   0,  800,  825,  850,  875,  900,  925,  950,
	 975, 1000, 1025, 1050, 1075, 1100, 1800, 2800,
};

/* VLD - ldo4/6/7/8/9	*/
static const int ldod_voltage_map[] = {
	   0, 1500, 1600, 1800, 1850, 2000, 2005, 2500,
	2550, 2700, 2750, 2800, 2850, 2900, 2950, 3000,
};

/* VBA - BUCK0/1 	5bit 0x1f	*/
static const int bucka_voltage_map[] = {
	   0,  700,  725,  750,  775,  800,  825,  850,
	 875,  900,	 925,  950,  975, 1000, 1025, 1050,
	1075, 1100,	1125, 1150, 1175, 1200, 1225, 1250,
	1250, 1250, 1250, 1250, 1250, 1250, 1250, 1250,
};

/* VBC - BUCK2 		4bit 0x0f  0000 1111	*/
static const int buckc_voltage_map[] = {
	   0,  900,  950, 1000, 1050, 1100,
	1150, 1200, 1250, 1300, 1350, 1400,
	1450, 1500, 1550, 1600,
};

/*  VBD-BUCK3	*/
static const int buck3_voltage_map[] = {
	1600,
};

/*  VBE-BUCK4	*/
static const int buck4_voltage_map[] = {
	3600,
};

/* BUCK3/4 EXTERNAL ADJUSTABLE	*/
/* LDO5- FIXED	*/
/* VLA-ldo1/2/3/10  VLB-VCOIN-2bit VLAC-ldo11 VLD-ldo4/6/7/8/9	*/
static const int *ldo_voltage_map[] = {
	ldoa_voltage_map,	/* ldo1	*/
	ldoa_voltage_map,
	ldoa_voltage_map,	/* ldo3	*/
	ldod_voltage_map,	/* ldo4	*/
	ldo5_voltage_map,	/* ldo5	*/
	ldod_voltage_map,	/* ldo6	*/
	ldod_voltage_map,
	ldod_voltage_map,
	ldod_voltage_map,	/* ldo9	*/
	ldoa_voltage_map,	/* ldo10	*/
	ldoc_voltage_map,	/* ldo11	*/
	ldob_voltage_map,	/* coin	*/
};

/* VBA-BUCK0/1  VBC-BUCK2    EXTERNAL ADJUSTABLE-BUCK3/4 	*/
static const int *buck_voltage_map[] = {
	bucka_voltage_map,	/* buck0	*/
	bucka_voltage_map,	/* buck1	*/
	buckc_voltage_map,	/* buck2	*/
	buck3_voltage_map,	/* buck3    */
	buck4_voltage_map,	/* buck4	*/
};

#if 1
static const int ldo_output_enable_mask[] = {
	ZX234290_EN1_LDO1_EN,
	ZX234290_EN1_LDO2_EN,
	ZX234290_EN1_LDO3_EN,
	ZX234290_EN1_LDO4_EN,
	ZX234290_EN1_LDO5_EN,
	ZX234290_EN1_LDO6_EN,
	ZX234290_EN1_LDO7_EN,
	ZX234290_EN1_LDO8_EN,

	ZX234290_EN2_LDO9_EN,
	ZX234290_EN2_LDO10_EN,
	ZX234290_EN2_LDO11_EN,
	ZX234290_EN2_BUCK0_EN,
	ZX234290_EN2_BUCK1_EN,
	ZX234290_EN2_BUCK2_EN,
	ZX234290_EN2_BUCK3_EN,
	ZX234290_EN2_BUCK4_EN,
};

/* ø Ǹ  zx234290_get_ctrl_register	*/
static const int ldo_output_enable_addr[] = {
	ZX234290_REG_ADDR_LDO_EN1,
	ZX234290_REG_ADDR_LDO_EN2,
};
#endif

#endif

#define ZX234290_COIN_VOL_MASK 0x02

#define ZX234290_LDO_VOL_VALUE_MAP(x) (ldo_voltage_map[x])
#define ZX234290_LDO_VOL_MASK(x) (((x) % 2) ? 0xf0 : 0x0f)

/* enable	*/
#define ZX234290_LDO_OUTPUT_ENABLE_MASK(x) (ldo_output_enable_mask[x])

#define ZX234290_BUCK_VOL_VALUE_MAP(x) (buck_voltage_map[x])
#define ZX234290_BUCK_VOL_MASK(x) (((x) != 13) ? 0x1f : 0x0f)	/* 0001 1111  or 0000 1111	*/

/* ͨLDO id ȡLDO ʹ Ĵ	*/
static int zx234290_get_ctrl_register(int id)
{
	if (id >= ZX234290_REG_LDO1 && id <= ZX234290_REG_LDO8)
		return ZX234290_REG_ADDR_LDO_EN1;
	else if (id >= ZX234290_REG_LDO9 && id <= ZX234290_REG_DCDC4)
		return ZX234290_REG_ADDR_LDO_EN2;
	else
		return -EINVAL;
}

/* ѡѹüĴ*/
static int zx234290_get_sel_register(struct zx234290_reg *pmic, int id)
{
	u8 reg = 0;

	switch(id) {
	case ZX234290_REG_DCDC0:	/* 0x1f	*/
		reg = ZX234290_REG_ADDR_BUCK0_VOL;
	case ZX234290_REG_DCDC1:	/* 0x1f	*/
		reg = ZX234290_REG_ADDR_BUCK1_VOL;
		break;
	case ZX234290_REG_DCDC2:	/* 0x0f	*/
		reg = ZX234290_REG_ADDR_BUCK2_VOL;
		break;
	case ZX234290_REG_LDO1:	/* 0x0f	*/
	case ZX234290_REG_LDO2: /* 0xf0	*/
		reg = ZX234290_REG_ADDR_LDO12_VOL;
		break;
	case ZX234290_REG_LDO3:	/* 0x0f	*/
	case ZX234290_REG_LDO4: /* 0xf0	*/
		reg = ZX234290_REG_ADDR_LDO34_VOL;
		break;
	case ZX234290_REG_LDO6: /* 0xf0	*/
		reg = ZX234290_REG_ADDR_LDO6_VOL;
		break;
	case ZX234290_REG_LDO7:	/* 0x0f	*/
	case ZX234290_REG_LDO8: /* 0xf0	*/
		reg = ZX234290_REG_ADDR_LDO78_VOL;
		break;
	case ZX234290_REG_LDO9:	/* 0x0f	*/
	case ZX234290_REG_LDO10: /* 0xf0	*/
		reg = ZX234290_REG_ADDR_LDO910_VOL;
		break;
	case ZX234290_REG_LDO11: /* 0xf0	*/
		reg = ZX234290_REG_ADDR_LDO11_VOL;
		break;
	default:
		return -EINVAL;
	}

	return reg;
}

static int zx234290_get_mode_regiters(struct zx234290_reg *pmic, int id)
{
	switch (id) {
	case ZX234290_REG_DCDC0:
	case ZX234290_REG_DCDC1:
		pmic->normal_mode_reg = ZX234290_REG_ADDR_BUCK01_MODE;/* 0x0c	*/
		pmic->normal_mode_reg = ZX234290_REG_ADDR_BUCK01_MODE; /* 0x03 */
		break;
	case ZX234290_REG_DCDC2:
	case ZX234290_REG_DCDC3:
		pmic->normal_mode_reg = ZX234290_REG_ADDR_BUCK23_MODE;
		pmic->normal_mode_reg = ZX234290_REG_ADDR_BUCK23_MODE;
		break;
	case ZX234290_REG_DCDC4:
		pmic->normal_mode_reg = ZX234290_REG_ADDR_BUCK4_MODE;
		pmic->normal_mode_reg = ZX234290_REG_ADDR_BUCK4_MODE;
		break;
	case ZX234290_REG_LDO1:
	case ZX234290_REG_LDO2:
	case ZX234290_REG_LDO3:
	case ZX234290_REG_LDO4:
		pmic->normal_mode_reg = ZX234290_REG_ADDR_LDO1234_MODE;
		pmic->normal_mode_reg = ZX234290_REG_ADDR_LDO1234_MODE;
		break;
	case ZX234290_REG_LDO5:
	case ZX234290_REG_LDO6:
	case ZX234290_REG_LDO7:
	case ZX234290_REG_LDO8:
		pmic->normal_mode_reg = ZX234290_REG_ADDR_LDO5678_MODE;
		pmic->normal_mode_reg = ZX234290_REG_ADDR_LDO5678_MODE;
		break;
	case ZX234290_REG_LDO9:
	case ZX234290_REG_LDO10:
	case ZX234290_REG_LDO11:
		pmic->normal_mode_reg = ZX234290_REG_ADDR_LDO91011_MODE;
		pmic->normal_mode_reg = ZX234290_REG_ADDR_LDO91011_MODE;
		break;
	default:
		return -EINVAL;
	}

	return 0;
}

#if 1	/* enable	*/
static int zx234290_reg_is_enabled(struct regulator_dev *dev)
{
	struct zx234290_reg *pmic = rdev_get_drvdata(dev);
	struct zx234290 *mfd = pmic->mfd;
	int reg, value, id = rdev_get_id(dev);

	if (id < ZX234290_REG_LDO1 || id > ZX234290_REG_DCDC4)
		return -EINVAL;

	reg = pmic->get_ctrl_reg(id);
	if (reg < 0)
		return reg;

	value = zx234290_reg_read(mfd, reg);
	if (value < 0)
		return value;

	return value & (ZX234290_LDO_OUTPUT_ENABLE_MASK(id));
}

static int zx234290_reg_enable(struct regulator_dev *dev)
{
	struct zx234290_reg *pmic = rdev_get_drvdata(dev);
	struct zx234290 *mfd = pmic->mfd;
	int id = rdev_get_id(dev);
	int reg;

	if (id < ZX234290_REG_LDO1 || id > ZX234290_REG_DCDC4)
		return -EINVAL;

	/* ȡʹܼĴ	*/
	reg = pmic->get_ctrl_reg(id);
	if (reg < 0)
		return reg;

	/*  ZX234290_REG_ENABLED 1000 0000 λ  1	*/
	return zx234290_set_bits(mfd, reg, ZX234290_LDO_OUTPUT_ENABLE_MASK(id));
}

static int zx234290_reg_disable(struct regulator_dev *dev)
{
	struct zx234290_reg *pmic = rdev_get_drvdata(dev);
	struct zx234290 *mfd = pmic->mfd;
	int id = rdev_get_id(dev), reg;

	reg = pmic->get_ctrl_reg(id);
	if (reg < 0)
		return reg;

	return zx234290_clear_bits(mfd, reg, ZX234290_LDO_OUTPUT_ENABLE_MASK(id));
}
#endif

static int zx234290_set_voltage_sel(struct regulator_dev *dev,
					 unsigned selector)
{
	struct zx234290_reg *pmic = rdev_get_drvdata(dev);
	struct zx234290 *mfd = pmic->mfd;
	int id = rdev_get_id(dev);
	int value;
	u8 reg;

	reg = zx234290_get_sel_register(pmic, id);
	value = zx234290_reg_read(mfd, reg);

	if (id >= ZX234290_REG_LDO1 && id <= ZX234290_REG_LDO11 && id != ZX234290_REG_LDO5)
	{
		value &= ~(ZX234290_LDO_VOL_MASK(id));
		if (id%2)selector *=16;	/* leftshit 5bits	*/
	}
	else if (id >= ZX234290_REG_DCDC1 && id <= ZX234290_REG_DCDC2)
		value &= ~(ZX234290_BUCK_VOL_MASK(id));
	else
		return -EINVAL;

	return zx234290_reg_write(mfd, reg, selector | value);
}



#if 1
static int zx234290_ldo_list_voltage(struct regulator_dev *dev, unsigned selector)
{
	int ldo = rdev_get_id(dev) - ZX234290_REG_LDO1;
	return 1000 * ZX234290_LDO_VOL_VALUE_MAP(ldo)[selector];
}

static int zx234290_ldo_get_voltage(struct regulator_dev *dev)
{
	struct zx234290_reg *pmic = rdev_get_drvdata(dev);
	struct zx234290 *mfd = pmic->mfd;
	int id = rdev_get_id(dev);
	int reg, vsel;

	/* ѡѹüĴ	*/
	reg = zx234290_get_sel_register(pmic, id);
	if (reg < 0)
		return reg;

	/* ȡѹ	*/
	vsel = zx234290_reg_read(mfd, reg);
	vsel &= ZX234290_LDO_VOL_MASK(id); /* 0000 1111	*/
	/* ldo2/4/6/8/10 4λ	*/
	if(id%2)
		vsel/=16;

	/* ȡvsel slector ѹ  Ȼͨlistصѹ	*/
	return zx234290_ldo_list_voltage(dev, vsel);
}

/* Operations permitted on LDOx */
static struct regulator_ops zx234290_ops_ldo = {
	.is_enabled = zx234290_reg_is_enabled,
	.enable = zx234290_reg_enable,
	.disable = zx234290_reg_disable,
	.get_voltage = zx234290_ldo_get_voltage,
	.set_voltage_sel = zx234290_set_voltage_sel,
	.list_voltage = zx234290_ldo_list_voltage,
};
#endif



#if 1
static int zx234290_dcdc_list_voltage(struct regulator_dev *dev, unsigned index)
{
	int buck = rdev_get_id(dev) - ZX234290_REG_DCDC0;
	return 1000 * buck_voltage_map[buck][index];
}

#if 0
static int zx234290_set_mode(struct regulator_dev *dev, unsigned int mode)
{
	struct zx234290_reg *pmic = rdev_get_drvdata(dev);
	struct zx234290 *mfd = pmic->mfd;
	int normal_mode, sleep_mode, id = rdev_get_id(dev);

	zx234290_get_mode_regiters(pmic, id);

	normal_mode = zx234290_reg_read(mfd, pmic->normal_mode_reg);
	sleep_mode = zx234290_reg_read(mfd, pmic->sleep_mode_reg);

	normal_mode &= DCDCCTRL_DCDC_MODE_MASK;
	sleep_mode &= DCDC_AVS_ECO_MASK;

	switch (mode) {
	case REGULATOR_MODE_FAST:
		/* Verify if mode alredy set */
		if (normal_mode && !sleep_mode)
			break;
		zx234290_set_bits(mfd, pmic->normal_mode, DCDCCTRL_DCDC_MODE_MASK);
		zx234290_clear_bits(mfd, pmic->eco_reg, DCDC_AVS_ECO_MASK);
		break;
	case REGULATOR_MODE_NORMAL:
	case REGULATOR_MODE_IDLE:
		if (!normal_mode && !sleep_mode)
			break;
		zx234290_clear_bits(mfd, pmic->normal_mode, DCDCCTRL_DCDC_MODE_MASK);
		zx234290_clear_bits(mfd, pmic->eco_reg, DCDC_AVS_ECO_MASK);
		break;
	case REGULATOR_MODE_STANDBY:
		if (!normal_mode && sleep_mode)
			break;
		zx234290_clear_bits(mfd, pmic->normal_mode, DCDCCTRL_DCDC_MODE_MASK);
		zx234290_set_bits(mfd, pmic->eco_reg, DCDC_AVS_ECO_MASK);
		break;
	default:
		return -EINVAL;
	}

	return 0;
}

static unsigned int zx234290_get_mode(struct regulator_dev *dev)
{
	struct zx234290_reg *pmic = rdev_get_drvdata(dev);
	struct zx234290 *mfd = pmic->mfd;
	int pwm_mode, eco, mode = 0, id = rdev_get_id(dev);

	zx234290_get_mode_regiters(pmic, id);

	pwm_mode = zx234290_reg_read(mfd, pmic->normal_mode_reg);
	eco = zx234290_reg_read(mfd, pmic->sleep_mode_reg);

	pwm_mode &= DCDCCTRL_DCDC_MODE_MASK;
	eco &= DCDC_AVS_ECO_MASK;

	if (pwm_mode && !eco)
		mode = REGULATOR_MODE_FAST;
	else if (!pwm_mode && !eco)
		mode = REGULATOR_MODE_NORMAL;
	else if (!pwm_mode && eco)
		mode = REGULATOR_MODE_STANDBY;

	return mode;
}
#endif

static int zx234290_dcdc_get_voltage(struct regulator_dev *dev)
{
	struct zx234290_reg *pmic = rdev_get_drvdata(dev);
	struct zx234290 *mfd = pmic->mfd;
	int id = rdev_get_id(dev);
	int reg, vsel;

	/* ѡѹüĴ	*/
	reg = zx234290_get_sel_register(pmic, id);
	if (reg < 0)
		return reg;

	/* ȡѹ	*/
	vsel = zx234290_reg_read(mfd, reg);
	vsel &= ZX234290_BUCK_VOL_MASK(id); /* 0001 1111	*/

	/* ȡvsel slector ѹ  Ȼͨlistصѹ	*/
	return zx234290_dcdc_list_voltage(dev, vsel);
}

/* Operations permitted on DCDCx */
static struct regulator_ops zx234290_ops_dcdc = {
	.is_enabled = zx234290_reg_is_enabled,
	.enable = zx234290_reg_enable,
	.disable = zx234290_reg_disable,
/*	.set_mode = zx234290_set_mode,
	.get_mode = zx234290_get_mode,
*/	.get_voltage = zx234290_dcdc_get_voltage,
	.set_voltage_sel = zx234290_set_voltage_sel,
	.list_voltage = zx234290_dcdc_list_voltage,
};
#endif

static __devinit int zx234290_probe(struct platform_device *pdev)
{
	struct zx234290 *zx234290 = dev_get_drvdata(pdev->dev.parent);
	struct zx_info *info;
	struct regulator_init_data *reg_data;
	struct regulator_dev *rdev;
	struct zx234290_reg *pmic;
	struct zx234290_board *pmic_plat_data;
	struct zx234290_platform_data *zx234290_plat_data;

	int i, err;

	pr_info("%s hello \n", __func__);

	pmic_plat_data = dev_get_platdata(pdev->dev.parent);
	if (!pmic_plat_data)
		return -EINVAL;

	pmic = kzalloc(sizeof(*pmic), GFP_KERNEL);
	if (!pmic)
		return -ENOMEM;

	mutex_init(&pmic->io_lock);
	pmic->mfd = zx234290;
	platform_set_drvdata(pdev, pmic);

	pmic->get_ctrl_reg = &zx234290_get_ctrl_register;
	info = zx234290_regs;

#if 1
	pmic->desc[0].n_voltages =  // 1/2/3/10
	pmic->desc[1].n_voltages =
	pmic->desc[2].n_voltages =
	pmic->desc[9].n_voltages = ARRAY_SIZE(ldoa_voltage_map);
	pmic->desc[3].n_voltages =
	pmic->desc[5].n_voltages =
	pmic->desc[6].n_voltages =
	pmic->desc[7].n_voltages =
	pmic->desc[8].n_voltages =	ARRAY_SIZE(ldod_voltage_map); // 4/6/7/8/9
	pmic->desc[10].n_voltages = ARRAY_SIZE(ldoc_voltage_map); // 11
	pmic->desc[11].n_voltages =
	pmic->desc[12].n_voltages = ARRAY_SIZE(bucka_voltage_map); // buck0/1
	pmic->desc[13].n_voltages = ARRAY_SIZE(buckc_voltage_map); // buck2
	pmic->desc[4].n_voltages =
	pmic->desc[14].n_voltages =
	pmic->desc[15].n_voltages = 1;
#endif


	for (i = 0; i < ZX234290_NUM_REGULATOR; i++, info++) {
		/* Register the regulators */
		pmic->info[i] = info;
		pmic->desc[i].name = info->name;
		pmic->desc[i].id = i;
		pmic->desc[i].ops = (i > ZX234290_REG_LDO11 ?
			&zx234290_ops_dcdc : &zx234290_ops_ldo);
		pmic->desc[i].type = REGULATOR_VOLTAGE;
		pmic->desc[i].owner = THIS_MODULE;

		zx234290_plat_data= pmic_plat_data->regulator_pdata[i];
		reg_data = zx234290_plat_data->reg_init_data;

		rdev = regulator_register(&pmic->desc[i],
					zx234290->dev, reg_data, pmic, NULL);


		if (IS_ERR(rdev)) {
			dev_err(zx234290->dev,
				"failed to register %s regulator\n",
				pdev->name);
			err = PTR_ERR(rdev);
			goto err;
		}

		/* Save regulator for cleanup */
		pmic->rdev[i] = rdev;
	}
	return 0;

err:
	while (--i >= 0)
		regulator_unregister(pmic->rdev[i]);

	kfree(pmic);
	return err;
}

static int __devexit zx234290_remove(struct platform_device *pdev)
{
	struct zx234290_reg *zx234290_reg = platform_get_drvdata(pdev);
	int i;

	for (i = 0; i < ZX234290_NUM_REGULATOR; i++)
		regulator_unregister(zx234290_reg->rdev[i]);

	kfree(zx234290_reg);
	return 0;
}

static struct platform_driver zx234290_driver = {
	.driver = {
		.name = "zx234290-regulators",
		.owner = THIS_MODULE,
	},
	.probe = zx234290_probe,
	.remove = __devexit_p(zx234290_remove),
};

static int __init zx234290_init(void)
{
	pr_info("%s hello\n", __func__);
	return platform_driver_register(&zx234290_driver);
}
subsys_initcall(zx234290_init);

static void __exit zx234290_cleanup(void)
{
	platform_driver_unregister(&zx234290_driver);
}
module_exit(zx234290_cleanup);

MODULE_AUTHOR("DongJian");
MODULE_DESCRIPTION("ZX234290 voltage regulator driver");
MODULE_LICENSE("GPL v2");
MODULE_ALIAS("platform:zx234290-regulator");
