/*
 * efuse_zx.c
 *
 *  Copyright (C) 2015 ZTE-TSP
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 */

#include <linux/init.h>
#include <linux/types.h>
#include <linux/mm.h>
#include <linux/kernel.h>
#include <mach/board.h>
#include <mach/iomap.h>
#include <mach/debug.h>
#include <linux/io.h>
#include <mach/spinlock.h>
#include <mach/pcu.h>
#include <linux/mutex.h>
#include <linux/device.h>
#include <linux/module.h>
#include <linux/cdev.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include <linux/soc/zte/efuse/efuse_zx.h>
#include <asm/setup.h>
#include <asm/io.h>
#include <asm/system.h>
#include <linux/slab.h>
#include <linux/miscdevice.h>

#define EFUSE_RAM_BASE (ZX_EFUSE_BASE+0x40)

#define SECURE_EN_OFFSET     0
#define PUB_KEY_HASH_OFFSET  1
#define DEVICE_ID_OFFSET     5

#define CHIP_COM_EN_FLAG	0xF86302FF
#define CHIP_COM_DIS_FLAG	0xF8630200

#define CHIP_SPE_NYB_EN_FLAG		0x1E871EFF
#define CHIP_SPE_NYB_DIS_FLAG		0x1E871E00
#define CHIP_SPE_NYC_EN_FLAG		0x1E871FFF
#define CHIP_SPE_NYC_DIS_FLAG		0x1E871F00
#define CHIPE_SPE_APM_EN_FLAG		0x1E8720FF
#define CHIPE_SPE_APM_DIS_FLAG	0x1E872000
#define CHIPECO_SPE_APM_EN_FLAG	0x1E8721FF
#define CHIPECO_SPE_APM_DIS_FLAG	0x1E872100
#define CHIPECO_SPE_NYB_EN_FLAG	0x1E8722FF
#define CHIPECO_SPE_NYB_DIS_FLAG	0x1E872200
#define CHIPECO_SPE_NYC_EN_FLAG	0x1E8723FF
#define CHIPECO_SPE_NYC_DIS_FLAG	0x1E872300
#define CHIPECO_SPE_256M_EN_FLAG	0x1E8724FF
#define CHIPECO_SPE_256M_DIS_FLAG	0x1E872400


#define EFUSE_PROGRAM_MAX_CNT	5

typedef volatile struct
{
	u32 secure_flag;
	u32 puk_hash[4];
	u32 secureDevId[3];
} s_EfuseStruct;

typedef struct
{
	u32 uiPubKeyRsaE[32];
	u32 uiPubKeyRsaN[32];
} s_ImageHeader;

typedef volatile struct
{
	u32 version;					/*汾żĴ*/
	u32 ctrl;					/*ƼĴ[0]start/busy  [3:1]op_code  [4]chip_sel  [5]read_mode  [6]int_en*/
	u32 addr;					/*ַĴ[7:0]*/
	u32 pdata;					/*ݼĴ[31:0]*/
	u32 rdata;					/*ݼĴ[31:0]*/
	u32 status;					/*״̬Ĵ[0]int_status  [1]cache_valid*/
	u32 reserved0[2];
	u32 timing1;					/*ʱĴ1*/
	u32 timing2;
	u32 timing3;
	u32 timing4;
	u32 reserved1[4];
	u32 cache[8];				/*Ĵ1*/
} s_EfuseRegister;

struct mutex efuseMutex;

/*******************************************************************************
* Function:     efuse_Program
* Description:  program 32bits data
* ʱɿڲɲַADDRĴԣ
* ʱʹADDR[2:0]λԣдʱֻʹADDR[7:5]ַλɿԶɣ
* дʱʹADDR[7:0]
* Parameters:
*   Input:
*       None
*   Output:
*       None
* Returns:
*       None
* Others:
*******************************************************************************/
static int efuse_Program(u32 cache_offset, u32 data)
{
	s_EfuseRegister * pReg = (s_EfuseRegister *)(ZX_EFUSE_BASE);
	int i = 0;
	emsf_lock_id softLock = EFUSE_SFLOCK;

	for(i = 0; i < EFUSE_PROGRAM_MAX_CNT; i++) {
		/*********************write***********************/
		/*step1, wait operation done*/
		soft_spin_lock(softLock);
		
		while(pReg->ctrl & 0x1);/*bit0=1 ctl is busy*/

		/*step2, program data & addr*/
		pReg->addr = (cache_offset<<5);
		pReg->pdata = data;

		/*step3, select efuse0 32bits program*/
		pReg->ctrl = (0<<4)|(2<<1)|1;/*cs0 program 32bit*/

		/*step4, wait operation done*/
		while(pReg->ctrl & 0x1);/* bit0=1 ctl is busy*/

		/*step5, select efuse1 32bits program*/
		pReg->ctrl = (1<<4)|(2<<1)|1;/*cs1 program 32bit*/
		
		/*step6, wait operation done*/
		while(pReg->ctrl & 0x1);/*bit0=1 ctl is busy*/	
		
		/*********************read************************/
		/*step7, select efuse0 read all*/
		pReg->ctrl = 1;
		/*step8, wait cache valid*/
		while(!(pReg->status & 0x2));/* bit0=1 ctl is busy*/
		
		soft_spin_unlock(softLock);

		/********************compare**********************/
		if(SECURE_EN_OFFSET == cache_offset) {
			if(data == (pReg->cache[cache_offset] & 0xFF))
				return 0;
		}else{
			if(data == pReg->cache[cache_offset])
				return 0;
		}
	
	}
	
	return -1;
}

/*******************************************************************************
* Function:     efuse_ReadEn
* Description:  
* Parameters:
*   Input:
*       None
*   Output:
*       None
* Returns:
*       None
* Others:
*******************************************************************************/
static void efuse_ReadEn(void)
{
	s_EfuseRegister * pReg = (s_EfuseRegister *)(ZX_EFUSE_BASE);

	emsf_lock_id	softLock = EFUSE_SFLOCK;

	/*step1, wait operation done*/
	soft_spin_lock(softLock);

	while(pReg->ctrl & 0x1);/*bit0=1 ctl is busy*/

	/*step2, select efuse0 read all*/
	pReg->ctrl = 1;

	/*step3, wait cache valid*/
	while(!(pReg->status & 0x2));/* bit0=1 ctl is busy*/

	soft_spin_unlock(softLock);

}

/*******************************************************************************
* Function:     zDrvEfuse_GetSecureMsg
* Description:  get pub key from iram, load by uboot
* Parameters:
*   Input:
*       None
*   Output:
*       None
* Returns:
*       None
* Others:
*******************************************************************************/
void zDrvEfuse_GetSecureMsg(T_ZDrvEfuse_Secure *secure)
{
	u32 i = 0;
	s_ImageHeader *s_header = (s_ImageHeader *)(SECURE_PUK_BASE);
	s_EfuseStruct *s_efuse  = (s_EfuseStruct *)(EFUSE_RAM_BASE);

	mutex_lock(&efuseMutex);

	efuse_ReadEn();

	for(i = 0; i < 32; i ++)
	{
		secure->pubKeyRsaE[i] = s_header->uiPubKeyRsaE[i];
		secure->pubKeyRsaN[i] = s_header->uiPubKeyRsaN[i];
	}

	secure->secureFlag = s_efuse->secure_flag;

	for(i = 0; i < 4; i ++)
	{
		secure->pubKeyHash[i] = s_efuse->puk_hash[i];
	}

	for(i = 0; i < 3; i ++)
	{
		secure->secureDevId[i] = s_efuse->secureDevId[i];
	}

	mutex_unlock(&efuseMutex);
}
EXPORT_SYMBOL(zDrvEfuse_GetSecureMsg);
/*******************************************************************************
* Function:     zDrvEfuse_SetSecureMsg
* Description:  set efuse value
* Parameters:
*   Input:
*       None
*   Output:
*       None
* Returns:
*       None
* Others:
*******************************************************************************/
s32 zDrvEfuse_SetSecureMsg(E_ZDrvEfuse_SecureMsg secure_msg, u32 *secure_buf)
{
	u32 i = 0;
	T_ZDrvEfuse_Secure secure = {0};
	u32 tmpbuf[4] = {0};
	int ret = 0;
	
	if(secure_msg >= MAX_ENUM)
	{
		return -EINVAL;
	}

	zDrvEfuse_GetSecureMsg(&secure);

	if(((secure_msg == SECURE_EN)&&(secure.secureFlag & 0xff))||
	  ((secure_msg == PUB_KEY_HASH)&&(memcmp(secure.pubKeyHash,tmpbuf,sizeof(secure.pubKeyHash)) != 0))||
	  ((secure_msg == DEVICE_ID)&&(memcmp(secure.secureDevId,tmpbuf,sizeof(secure.secureDevId)) != 0)))
		return -1;

	mutex_lock(&efuseMutex);
	
	switch(secure_msg)
	{
		case SECURE_EN:
		{
			ret =efuse_Program(SECURE_EN_OFFSET,*secure_buf);
			if(ret < 0){
				mutex_unlock(&efuseMutex);	
				return -1;
			}
			break;
		}
		case PUB_KEY_HASH:
		{
			for(i = 0; i < 4; i ++)
			{
				ret = efuse_Program((PUB_KEY_HASH_OFFSET+i),*(secure_buf+i));
				if(ret < 0){
					mutex_unlock(&efuseMutex);	
					return -1;
				}
			}
			break;
		}
		case DEVICE_ID:
		{
			for(i = 0; i < 3; i ++)
			{
				ret =efuse_Program((DEVICE_ID_OFFSET+i),*(secure_buf+i));
				if(ret < 0){
					mutex_unlock(&efuseMutex);					
					return -1;
				}
			}
			break;
		}
		default:
			break;
	}

	mutex_unlock(&efuseMutex);

	return 0;
}

int zx29_sfuse_set_secure(E_ZDrvEfuse_SecureMsg secure_msg, u32 *secure_buf)
{
	int ret = 0;
	int	len = 0;
	void *buf=NULL;

	switch(secure_msg)
	{
		case SECURE_EN:
			len = sizeof(u32);
			break;
		case PUB_KEY_HASH:
			len = 4 * sizeof(u32);
			break;
		case DEVICE_ID:
			len = 3 * sizeof(u32);
			break;

		default:
			return -EINVAL;
	}

	buf = kmalloc(len, GFP_KERNEL);
	if(NULL == buf)
		return -ENOSPC;
	
	ret = copy_from_user(buf, secure_buf, len);
	if(ret)
		printk("%s: copy user failed\n",__func__);
	
	ret = zDrvEfuse_SetSecureMsg(secure_msg, buf);
	kfree(buf);

	return ret;
}


static long zx_efuse_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	int ret = 0;
	T_ZDrvEfuse_Secure secure;

	if (arg == NULL)
		return -EFAULT;

	switch(cmd)
	{
		case EFUSE_GET_DATA:
			zDrvEfuse_GetSecureMsg(&secure);
			ret = copy_to_user((void *)arg, &secure, sizeof(T_ZDrvEfuse_Secure));
			if(ret)
				printk("%s: copy user failed\n",__func__);
			break;
		case EFUSE_SET_SECURE_EN:
			ret = zx29_sfuse_set_secure(SECURE_EN, (u32 *)arg);
			break;
		case EFUSE_SET_PUB_KEY_HASH:
			ret = zx29_sfuse_set_secure(PUB_KEY_HASH, (u32 *)arg);
			break; 
		case EFUSE_SET_DEVICE_ID:
			ret = zx29_sfuse_set_secure(DEVICE_ID, (u32 *)arg);
			break; 
		default:
			return -EPERM;
		}
	
	return ret;
}

static const struct file_operations	zx_efuse_ops = {
	.owner 		= THIS_MODULE,
	.unlocked_ioctl = zx_efuse_ioctl,
};

static struct miscdevice zx_efuse_miscdev = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= "efuse",
	.fops		= &zx_efuse_ops,
};


static int __init zx_efuse_init(void)
{
	int ret = 0;

	mutex_init(&efuseMutex);
	
	ret = misc_register(&zx_efuse_miscdev);
	if (ret) {
		printk(KERN_ERR"%s: efuse failed to register miscdev (ret = %d)\n", __FILE__, ret);
		return ret;
	}

	printk("[xxx] efuse dev inited! \n");	

	return ret;
}

module_init(zx_efuse_init);

