/*
 * arch/arm/mach-zx297510/zx-cpufreq.c
 *
 * Author:
 *	dongjian <dong.jian@zte.com.cn>
 *
 * Copyright (C) 2010-2013 ZTE CORPORATION. All rights reserved.
 *
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/sched.h>
#include <linux/cpufreq.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/err.h>
#include <linux/clk.h>
#include <linux/io.h>
#include <linux/suspend.h>
#include <linux/cpu.h>
#include <linux/slab.h>

#include <asm/system.h>

//#include "mach/clock.h"
#include "zx-pm.h"

zx29xx_cpufreq_init_cb 	zx29xx_cpufreq_init;

static struct zx_dvfs_info *zx_info;

static struct cpufreq_frequency_table *freq_table;
static struct cpufreq_freqs freqs;

static unsigned long target_cpu_speed[CONFIG_NR_CPUS];
DEFINE_MUTEX(cpufreq_lock);
static unsigned int locking_frequency;
static bool frequency_locked;
extern unsigned int freq_change_enabled_by_startup;
extern unsigned int cpu_dfs_is_not_allowed;
 
#ifdef CONFIG_PM
static int zx_cpufreq_suspend(struct cpufreq_policy *policy)
{
	return 0;
}

static int zx_cpufreq_resume(struct cpufreq_policy *policy)
{
	return 0;
}
#endif

int zx_verify_speed(struct cpufreq_policy *policy)
{
	return cpufreq_frequency_table_verify(policy, freq_table);
}

unsigned int zx_getspeed(unsigned int cpu)
{
	unsigned int rate;
	
	if (cpu >= CONFIG_NR_CPUS)
		return 0;

	rate = clk_get_rate(zx_info->cpu_clk) / 1000;

	//pr_info("[CPUFREQ] get_cpu_rate: %d\n", rate);

	return rate;
}


unsigned long zx_cpu_lowest_speed(void) 
{
	unsigned long rate = ULONG_MAX;
	int i;

	for_each_online_cpu(i)
		rate = min(rate, target_cpu_speed[i]);
	
	return rate;
}

unsigned long zx_cpu_highest_speed(void) 
{
	unsigned long rate = 0;
	int i;

	for_each_online_cpu(i) 
		rate = max(rate, target_cpu_speed[i]);

	return rate;
}

extern u32 zDrvTsCtrl_DfsEn(void);
static int zx_target(struct cpufreq_policy *policy,
		       unsigned int target_freq,
		       unsigned int relation)
{
	unsigned int index, old_index;
	int ret = 0;
	unsigned int freq;
    unsigned int cpu;	

	mutex_lock(&cpufreq_lock);

	if((pm_get_mask_info()&PM_NO_CPU_FREQ) /*|| cpu_dfs_is_not_allowed||zDrvTsCtrl_DfsEn()*/)
	{
		ret = -EAGAIN;
		goto out;
	}
	if(freq_change_enabled_by_startup == 0)
        goto out;

	if (frequency_locked && target_freq != locking_frequency) 
	{
		ret = -EAGAIN;
		goto out;
	}

	freqs.old = policy->cur;
	if (cpufreq_frequency_table_target(policy, freq_table,
					   freqs.old, relation, &old_index)) 
	{
		ret = -EINVAL;
		goto out;
	}

	if (cpufreq_frequency_table_target(policy, freq_table,
					   target_freq, relation, &index)) 
	{
		ret = -EINVAL;
		goto out;
	}

	freq = freq_table[index].frequency;
	
	freqs.new = freq_table[index].frequency;
	freqs.cpu = policy->cpu;

	for_each_online_cpu(cpu)
    {
        freqs.cpu = cpu;
        cpufreq_notify_transition(&freqs, CPUFREQ_PRECHANGE);
    }

	if (zx_info->set_freq)
	{
		if(!zx_info->set_freq(old_index, index))
			zx_info->freq_cur_idx = index;
	}

    for_each_online_cpu(cpu)
    {
        freqs.cpu = cpu;
        cpufreq_notify_transition(&freqs, CPUFREQ_POSTCHANGE);
    }
	
	target_cpu_speed[policy->cpu] = freq;

out:
	mutex_unlock(&cpufreq_lock);

	return ret;
}


static int zx_cpufreq_cpu_init(struct cpufreq_policy *policy)
{
	if (policy->cpu >= CONFIG_NR_CPUS)
		return -EINVAL;

	cpufreq_frequency_table_cpuinfo(policy, freq_table);
	cpufreq_frequency_table_get_attr(freq_table, policy->cpu);
	policy->cur = zx_getspeed(policy->cpu);
	target_cpu_speed[policy->cpu] = policy->cur;
	locking_frequency = policy->cur;
	pr_info("[CPUFREQ] %s cpu: %u, target_freq:%u\n", __func__, policy->cpu, policy->cur);

	/* FIXME: what's the actual transition time? */
	policy->cpuinfo.transition_latency = 300 * 1000;

	if (num_online_cpus() == 1) 
	{
		cpumask_copy(policy->related_cpus, cpu_possible_mask);
		cpumask_copy(policy->cpus, cpu_online_mask);
	} 
	else 
	{
		policy->shared_type = CPUFREQ_SHARED_TYPE_ALL;
		cpumask_setall(policy->cpus);
	}

	return 0;
}


static int zx_cpufreq_policy_notifier(
	struct notifier_block *nb, unsigned long event, void *data)
{
	int i, ret;
	struct cpufreq_policy *policy = data;

	if (event == CPUFREQ_NOTIFY) 
	{
		ret = cpufreq_frequency_table_target(policy, freq_table,
			policy->max, CPUFREQ_RELATION_H, &i);
	}
	
	return NOTIFY_OK;
}

static struct notifier_block zx_cpufreq_policy_nb = 
{
	.notifier_call = zx_cpufreq_policy_notifier,
};

/**
 * zx_cpufreq_pm_notifier - block CPUFREQ's activities in suspend-resume
 *			context
 * @notifier
 * @pm_event
 * @v
 *
 * While frequency_locked == true, target() ignores every frequency but
 * locking_frequency. The locking_frequency value is the initial frequency,
 * which is set by the bootloader. In order to eliminate possible
 * inconsistency in clock values, we save and restore frequencies during
 * suspend and resume and block CPUFREQ activities. Note that the standard
 * suspend/resume cannot be used as they are too deep (syscore_ops) for
 * regulator actions.
 */
static int zx_cpufreq_pm_notifier(struct notifier_block *notifier,
				       unsigned long pm_event, void *v)
{
	struct cpufreq_policy *policy = cpufreq_cpu_get(0); /* boot CPU */
	static unsigned int saved_frequency;
	unsigned int temp;

	mutex_lock(&cpufreq_lock);
	if(NULL == policy)
		goto out;	
		
	switch (pm_event) 
	{
	case PM_SUSPEND_PREPARE:
		if (frequency_locked)
			goto out;

		frequency_locked = true;

		if (locking_frequency) 
		{
			saved_frequency = zx_getspeed(0);

			mutex_unlock(&cpufreq_lock);
			zx_target(policy, locking_frequency,
				      CPUFREQ_RELATION_H);
			mutex_lock(&cpufreq_lock);
		}
		break;

	case PM_POST_SUSPEND:
		if (saved_frequency) 
		{
			/*
			 * While frequency_locked, only locking_frequency
			 * is valid for target(). In order to use
			 * saved_frequency while keeping frequency_locked,
			 * we temporarly overwrite locking_frequency.
			 */
			temp = locking_frequency;
			locking_frequency = saved_frequency;

			mutex_unlock(&cpufreq_lock);
			zx_target(policy, locking_frequency,
				      CPUFREQ_RELATION_H);
			mutex_lock(&cpufreq_lock);

			locking_frequency = temp;
		}
		frequency_locked = false;
		break;
	}
out:
	mutex_unlock(&cpufreq_lock);

	return NOTIFY_OK;
}

static struct notifier_block zx_cpufreq_nb = 
{
	.notifier_call = zx_cpufreq_pm_notifier,
};


static struct freq_attr *zx_cpufreq_attr[] = 
{
	&cpufreq_freq_attr_scaling_available_freqs,
	NULL,
};

static struct cpufreq_driver zx_cpufreq_driver = 
{
	.flags		= CPUFREQ_STICKY,
	.verify		= zx_verify_speed,
	.target		= zx_target,
	.get		= zx_getspeed,
	.init		= zx_cpufreq_cpu_init,
	.name		= "zx_cpufreq",
	.attr		= zx_cpufreq_attr,
#ifdef CONFIG_PM
	.suspend	= zx_cpufreq_suspend,
	.resume		= zx_cpufreq_resume,
#endif
};

static int __init zx_cpufreq_init(void)
{
	int ret = -EINVAL;

	/* init dvfs driver */
	zx_info = kzalloc(sizeof(struct zx_dvfs_info), GFP_KERNEL);
	if (!zx_info)
		return -ENOMEM;
	ret = zx29xx_cpufreq_init(zx_info);
	if (ret)
		goto err_mach_init;	
	freq_table = zx_info->freq_table;

	/* cpufreq notify */
	ret = cpufreq_register_notifier(
		&zx_cpufreq_policy_nb, CPUFREQ_POLICY_NOTIFIER);
	if (ret)
		goto err_mach_init;	

	/* pm notify */
	register_pm_notifier(&zx_cpufreq_nb);

	/* register driver */
	if (cpufreq_register_driver(&zx_cpufreq_driver)) 
	{
		pr_info("[CPUFREQ] %s: failed to register cpufreq driver\n", __func__);
		goto err_cpufreq;
	}

	pr_info("[CPUFREQ] register cpufreq driver OK!\n");

	return 0;

err_cpufreq:
	cpufreq_unregister_notifier(&zx_cpufreq_policy_nb, CPUFREQ_POLICY_NOTIFIER);
	unregister_pm_notifier(&zx_cpufreq_nb);
	
err_mach_init:
	kfree(zx_info);
	pr_info("[CPUFREQ] %s: failed initialization\n", __func__);

	return -EINVAL;
}

late_initcall(zx_cpufreq_init);

