/*
 * zx297520v2 CPU idle Routines
 *
 * Copyright (C) 2013 ZTE, Ltd.
 * Shine Yu <yu.xiang5@zte.com.cn>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/sched.h>
#include <linux/export.h>
#include <linux/kernel.h>
#include <linux/cpu.h>
#include <linux/tick.h>
#include <linux/suspend.h>
#include <linux/interrupt.h>
#include <linux/gpio.h>


#include <mach/timex.h>
#include "zx-pm.h"

#define	WHOLE_CHIP_EXIT_LATENCY		(4000) 		/* us */

#define LP2_DEFAULT_EXIT_LATENCY	(500 + WHOLE_CHIP_EXIT_LATENCY) 		/* us */
#define	LP2_MIN_POWER_OFF_TIME		(500)		/* us */

#define LP2_DELTA_EXIT_LATENCY		(100) 		/* us -- for timer setting refresh time, should > 2us.
                                                         Donnot modify this. */

static s64 	zx_idle_sleeptime = 0xffffffff;
static struct cpuidle_driver *cur_idle_drv;

#ifdef CONFIG_ZX_PM_DEBUG
//extern struct zx_idle_stats 	idle_stats;
#endif

/*===================================================================
 *=========  idle states description   ==============================
 *===================================================================
 *========= LP3 -- wfi(target_residency	= 5)            =============
 *========= LP1 -- pwroff(target_residency	= 5000)     =============
 *===================================================================
 */
static struct cpuidle_state zx29_cpuidle_set[] __initdata =
{
	/* LP3 -- wfi */
	[ZX_IDLE_CSTATE_LP3] =
	{
		.enter				= zx_enter_idle,
		.exit_latency		= 2,
		.target_residency	= 5,
		.flags				= CPUIDLE_FLAG_TIME_VALID,
		.name				= "LP3",
		.desc				= "clock gating(WFI)",
	},
	/* LP2 -- POWEROFF  */
	[ZX_IDLE_CSTATE_LP2] =
	{
		.enter				= zx_enter_idle,
		.exit_latency		= LP2_DEFAULT_EXIT_LATENCY,
		.target_residency	= LP2_DEFAULT_EXIT_LATENCY+LP2_MIN_POWER_OFF_TIME,
		.flags				= CPUIDLE_FLAG_TIME_VALID,
		.name				= "LP2",
		.desc				= "POWEROFF",
	},
};

int __init zx_fill_cpuidle_data(struct cpuidle_driver *drv)
{
	int i, max_cpuidle_state;

	max_cpuidle_state = sizeof(zx29_cpuidle_set) / sizeof(struct cpuidle_state);

	for (i = 0; i < max_cpuidle_state; i++)
		memcpy(&drv->states[i], &zx29_cpuidle_set[i],	sizeof(struct cpuidle_state));

	drv->safe_state_index = ZX_IDLE_CSTATE_LP3;

	cur_idle_drv = drv;

	return max_cpuidle_state;
}


/**
 * 	idle_update_sleep_param
 *
 *	when exit from one lp2 level sleep, update the exit_latency/target_residency.
 */
/* our idle exit lattency may */
#if 0
static unsigned int lp2_exit_latencies[MAX_CPU_NUM];
static void idle_update_sleep_param(void)
{

}
#endif
/**
 * 	idle_set_sleeptime
 *
 *	set the wakeup timer
 *
 *  sleep_time  (us)
 */
extern void setup_timer_wakeup(s64 us);
void idle_set_sleeptime(s64 sleep_time)
{
	/* set timer */
	setup_timer_wakeup(sleep_time);

	zx_idle_sleeptime = sleep_time;
}

/**
 * 	idle_get_sleeptime
 *
 *	for PCU sleeptime
 */
s64 idle_get_sleeptime(void)
{
	return zx_idle_sleeptime;
}

/**
 * 	idle_unmask_interrupt
 *
 *
 */
static void idle_unmask_interrupt(void)
{
}

static void idle_unmask_interrupt_restore(void)
{
}

static unsigned int idle_get_exit_latency(int index)
{
	struct cpuidle_state* state = &(cur_idle_drv->states[index]);

	return state->exit_latency;
}

/**
 * 	When enter deep sleep the tick timer maybe stopped for  
 *  26M osc will be closed. So we stop tick before entering 
 *  deep sleep and get sleeping time, then we restart the
 *  tick(minus the sleeping time).
 */
static u64 idle_enter_time = 0;
static void	idle_pre_enter(void)
{
	pm_stop_tick();

	idle_enter_time = read_persistent_us();
}

static void	idle_post_enter(s64 rem_us)
{
	u64 cur_time = read_persistent_us();
	s64 delta;
	u64 max_persist_us;

	if(cur_time >= idle_enter_time)
		delta = cur_time - idle_enter_time;
	else
	{
		max_persist_us = div64_u64((u64)(0x7fffffff)*USEC_PER_SEC, (u64)PERSISTENT_TIMER_CLOCK_RATE);		
		delta = max_persist_us - idle_enter_time + cur_time;
	}

	if(delta > rem_us + LP2_DELTA_EXIT_LATENCY)
		delta -= rem_us;
	else
		delta = LP2_DELTA_EXIT_LATENCY;

	pm_start_tick(delta);
}

#define	PM_IDLE_TRACE	0
#if PM_IDLE_TRACE

#define	TRACE_IDLE_COUNT	1000

typedef struct 
{
  s64	request;
  s64   enter_remainder;
  s64   exit_remainder;  
}pm_idle_trace_t;

volatile pm_idle_trace_t pm_idle_view[TRACE_IDLE_COUNT];
volatile unsigned int 	 pm_idle_index = 0;

void trace_pm_idle_enter(s64 req_t, s64 remainder_t)
{
	pm_idle_view[pm_idle_index].request	= req_t;
	pm_idle_view[pm_idle_index].enter_remainder	= remainder_t;
}

void trace_pm_idle_exit(s64 remainder_t)
{
	pm_idle_view[pm_idle_index].exit_remainder	= remainder_t;

	pm_idle_index++;
	if(pm_idle_index==TRACE_IDLE_COUNT)
	   pm_idle_index=0;
}
#else
void trace_pm_idle_enter(s64 req_t, s64 remainder_t){}
void trace_pm_idle_exit(s64 remainder_t){}
#endif

/**
 * 	zx_enter_deep_idle
 *
 *	enter lp2 mode
 */
 s64 sleep_time = 0;
 s64 request = 0;
 s64 remainder_timer = 0;
 s64 enter_deep_idle_enter_cnt =0;
 s64 enter_deep_idle_exit_cnt =0;
static int zx_enter_deep_idle(int index)
{
	bool sleep_completed = false;
	pm_wake_reason_t wake_reason;

	

#ifdef CONFIG_ZX_PM_DEBUG
	ktime_t entry_time, exit_time;
	s64 idle_time;
	unsigned int cpu = read_cpuid();

	//idle_stats.cpu_ready_count[cpu]++;
 //	idle_stats.tear_down_count[cpu]++;

	entry_time = ktime_get();
#endif
		enter_deep_idle_enter_cnt++;

    	//s64 sleep_time;
	/*s64 */request = ktime_to_us(tick_nohz_get_sleep_length());
	//s64 remainder_timer;

	/*  */
	idle_unmask_interrupt();

	/* set wakeup timer */
	remainder_timer = pm_get_remainder_time();
	sleep_time = request - idle_get_exit_latency(index);
	if ((sleep_time > LP2_DELTA_EXIT_LATENCY) && (sleep_time < remainder_timer))
		idle_set_sleeptime(sleep_time);
	else
		return zx_pm_idle_enter(ZX_IDLE_CSTATE_LP3);

	trace_pm_idle_enter(request, remainder_timer);

	idle_pre_enter();

#ifdef CONFIG_ZX_PM_DEBUG
	//idle_stats.lp2_count[cpu]++;
#endif

	/* sleep */
	zx_enter_sleep(CPU_SLEEP_TYPE_LP3); /*޸ΪOFF/sleepģʽ*/

	enter_deep_idle_exit_cnt++;// tmp
	idle_post_enter(remainder_timer);
    
	remainder_timer = pm_get_remainder_time();
	trace_pm_idle_exit(remainder_timer);

	/* get wakeup cause */
	wake_reason = pm_get_wakeup_reason();
	if (wake_reason != WR_WAKE_SRC_ABNORMAL)
	{
		sleep_completed = true;
	}
	else
	{
#ifdef CONFIG_ZX_PM_DEBUG
		int irq = 0;
		irq = zx29_gic_pending_interrupt();
		//idle_stats.lp2_int_count[irq]++;
#endif
	}

	/*  */
	idle_unmask_interrupt_restore();
/*=================================================================
 *=======end enter deep sleep======================================
 *=================================================================
 */
#ifdef CONFIG_ZX_PM_DEBUG
	exit_time = ktime_get();
	idle_time = ktime_to_us(ktime_sub(exit_time, entry_time));

	//idle_stats.cpu_wants_lp2_time[cpu] += idle_time;
	//idle_stats.in_lp2_time[cpu] += idle_time;

	//if (sleep_completed)
		//idle_stats.lp2_completed_count[cpu]++;
#endif

	return index;
}

/**
 * zx_enter_lowpower - Programs cpu to enter the specified state
 * @dev: cpuidle device
 * @state: The target state to be programmed
 *
 * Called from the CPUidle framework to program the device to the
 * specified low power state selected by the governor.
 * Called with irqs off, returns with irqs on.
 * Returns the amount of time spent in the low power state.
 */
int zx_pm_idle_enter(int index)
{
#ifdef CONFIG_ZX_PM_DEBUG
	//idle_stats.idle_count++;
#endif
	if(ZX_IDLE_CSTATE_LP2 == index)
	{
		return zx_enter_deep_idle(index);
	}
	else
	{
#ifdef CONFIG_ZX_PM_DEBUG
		unsigned cpu = read_cpuid();
	//	idle_stats.lp3_count[cpu]++;
#endif
      //gpio_direction_output(ZX29_GPIO_35, GPIO_HIGH);
		cpu_do_idle();
	  //gpio_direction_output(ZX29_GPIO_35, GPIO_LOW);
		return ZX_IDLE_CSTATE_LP3;
	}
}
