/*
 * zx297510 CPU idle Routines
 *
 * Copyright (C) 2013 ZTE, Ltd.
 * Shine Yu <yu.xiang5@zte.com.cn>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/sched.h>
#include <linux/export.h>
#include <linux/kernel.h>
#include <linux/cpu.h>
#include <linux/tick.h>
#include <linux/suspend.h>
#include <linux/interrupt.h>
#include <linux/soc/zte/pm/drv_idle.h>
#include <linux/module.h>
#include "zx-pm.h"

#include <linux/cp_types.h>  
#include "drvs_lpm.h"

#ifdef CONFIG_ZX_PM_DEBUG
//struct zx_idle_stats 	idle_stats;
#endif
#if 0
struct cpuidle_driver zx_idle_driver = {
	.name = 	"zx_idle",
	.owner =	THIS_MODULE,
	.en_core_tk_irqen	= 0,					/* no use cpuidle time keeping */
};
#endif

static unsigned int deep_idle_disabled_by_startup = 0;
static unsigned int deep_idle_disabled_by_suspend = 0;
static unsigned int deep_idle_disabled_by_debug   = 0;
static unsigned int print_enabled_by_debug = 0;
DEFINE_PER_CPU(struct cpuidle_device, zx_idle_dev);

static struct delayed_work 	pm_idle_work;
#define PM_IDLE_DELAY 		msecs_to_jiffies(30000)

static unsigned int sleep_mode_flag = 0;
static unsigned int drv_cpuidle_flag	= 0;
static DEFINE_SPINLOCK(zx_idle_lock);

void zx_cpuidle_set_busy(drv_idle_flag devId)
{
	unsigned long flags;

	if( devId >= IDLE_FLAG_MAX)
	{
		printk("[zx_cpuidle_set_busy] devId  err, devId = %d\n",devId);
	}
	else
	{
		raw_spin_lock_irqsave(&zx_idle_lock, flags);	
		drv_cpuidle_flag |= (1<<devId);	
		raw_spin_unlock_irqrestore(&zx_idle_lock, flags);
	}

}
EXPORT_SYMBOL(zx_cpuidle_set_busy);

void zx_cpuidle_set_free(drv_idle_flag devId)
{
	unsigned long flags;

	if( devId >= IDLE_FLAG_MAX)
	{
		printk("[zx_cpuidle_set_free] devId  err, devId = %d\n",devId);
	}
	else
	{
		raw_spin_lock_irqsave(&zx_idle_lock, flags);
		drv_cpuidle_flag &= ~(1<<devId);	
		raw_spin_unlock_irqrestore(&zx_idle_lock, flags);
	}
}
EXPORT_SYMBOL(zx_cpuidle_set_free);

static void pm_idle_func(struct work_struct *work)
{
	deep_idle_disabled_by_startup = 1;
}

/**
 * idle_can_enter_deep_sleep - check can enter deep sleep state?
 *
 *
 */
static int idle_can_enter_deep_sleep(void)
{
	/* can not enter deep sleep now */
	if (deep_idle_disabled_by_suspend)
		return false;

	/* This mode only can be entered when other core's are offline */
	if(deep_idle_disabled_by_debug || num_online_cpus() > 1)
		return false;

	/* can not enter deep sleep when kernel startup, we delay 30s now! */
	if(!deep_idle_disabled_by_startup)
		return false;

	if(pm_dma_used())
		return false;

	if(pm_get_mask_info()&PM_IDLE_WFI)
		return false;
	
	if(drv_cpuidle_flag != 0)
		return false;
	

	return true;
}


static int zx_pm_idle_prepare(struct cpuidle_device *dev,
		struct cpuidle_driver *drv, int index)
{
	int new_index = index;

	if(new_index == drv->safe_state_index)
		return new_index;

	if(!idle_can_enter_deep_sleep())
	{
		new_index = drv->safe_state_index;
		return new_index;
	}

	return new_index;
}
/****************************************************/
#define	MAX_TIME_PER_CYCLE 		div64_u64(div64_u64((u64)0x7fffffff*USEC_PER_SEC, (u64)PERSISTENT_TIMER_CLOCK_RATE),(u64)1000)

#define SAVE_RESTORE_TIME 		(9)			//ms
#define SHUTDOWN_SLEEP_TIME 	(2)			//ms
#define	MAX_PM_CB_CNT			(20)

enum {
	DEBUG_IDLE_MSG = 1U << 0,
	DEBUG_WAKE_LOCK = 1U << 1,
};

typedef int (*pm_callback_fn)(void);
typedef struct
{
	pm_callback_fn  cb;
	void 			*is_sucess;
}pm_cb_t;

static int idle_debug_mask = 0; //DEBUG_IDLE_MSG;

/* /sys/module/zx297520_cpuidle/parameters/debug_mask */
module_param(idle_debug_mask, int, 0644);

static u32 sleep_count = 0;
static pm_cb_t  pm_enter_cb[MAX_PM_CB_CNT];
static pm_cb_t  pm_exit_cb[MAX_PM_CB_CNT];
static unsigned int pm_cb_cnt = 0;

extern void pm_idle_sram_start(void);
extern void pm_idle_sram_end(void);
extern void (*arm_pm_idle)(void);
extern  void idle_set_sleeptime(s64 sleep_time);
extern void pm_debug_wakelocks(void);
extern void zDrvInt_MaskIrq( u32 uiLine );
extern void zDrvInt_UnmaskIrq( u32 uiLine );

extern bool zPs_IsTdMasterMode(void);
extern bool zPs_IsLteMasterMode(void);
extern bool zPs_IsFddMasterMode(void);

int zx_idle_get_debug_flag(void)
{
	return idle_debug_mask;
}

int zx_idle_get_idle_flag(void)
{
	return drv_cpuidle_flag;
}
/**
 * zx_pm_register_callback 
 * 
 * register callback for sleep enter and exit,
 * enter_cb: callback for sleep enter, callback for sleep exit
 * 
 * 
 */
int zx_pm_register_callback(pm_callback_fn enter_cb, pm_callback_fn exit_cb)
{
	int i = 0;
	if(pm_cb_cnt >= MAX_PM_CB_CNT)
		return -ENOMEM;

	if(!enter_cb)
		return -EINVAL;

	if(!exit_cb)
		return -EINVAL;

	for(i = 0; i < pm_cb_cnt; i++){
		if(pm_enter_cb[i].cb == enter_cb ||
			pm_exit_cb[i].cb == exit_cb)
			return EINVAL;
	}	
	pm_enter_cb[pm_cb_cnt].cb = enter_cb;
	pm_enter_cb[pm_cb_cnt].is_sucess=0;
	pm_exit_cb[pm_cb_cnt].cb = exit_cb;
	pm_exit_cb[pm_cb_cnt].is_sucess=0;
	
	pm_cb_cnt ++;

	return 0;
}
EXPORT_SYMBOL(zx_pm_register_callback);

/**
 * zx_cpu_dev_idle_enter 
 * 
 * 
 * 
 * 
 * 
 */
int zx_cpu_dev_idle_enter(void)
{
	int i;
	int ret=0;

	for(i=0;i<pm_cb_cnt; i++){
		pm_enter_cb[i].is_sucess = 0;		
	}
	
	for(i=0;i<pm_cb_cnt; i++){
					
		if(pm_enter_cb[i].cb)
			ret = pm_enter_cb[i].cb();

		if(ret)
			return ret;

		pm_enter_cb[i].is_sucess = 1;

	}

	return ret;
}

/**
 * zx_cpu_dev_idle_exit 
 * 
 * only when dev is is_sucess when enter sleep,the exit callback is visited
 * 
 * 
 * 
 */
int zx_cpu_dev_idle_exit(void)
{
	int i;
	int ret=0;

	for(i=0;i<pm_cb_cnt; i++){
		if(pm_exit_cb[i].cb && pm_enter_cb[i].is_sucess )
			ret = pm_exit_cb[i].cb();

		if(ret)
			return ret;	
	}

	return ret;
}
/**
 * zx_debug_check 
 * 
 * check whether debug allowed sleep.
 *
 * when pm_get_mask_info()&PM_IDLE_WFI is 1, return "false", 
 * else return "ture"
 */
static bool zx_debug_check(void)
{
	if(pm_get_mask_info()&PM_IDLE_WFI)
		return false;
	else
		return true;
}
/**
 * zx_sleep_wakelock_check 
 * 
 * check whether linux allowed sleep.
 *
 * when active wakelock count is 0, return "true", 
 * else return "false"
 */
static bool zx_sleep_wakelock_check(void)
{
	unsigned int temp_count;

	if(pm_get_mask_info()&PM_SLEEP_FLAG_PRINT) //if (idle_debug_mask & DEBUG_WAKE_LOCK) 
		pm_debug_wakelocks();
	
	if(pm_get_wakeup_count(&temp_count, false))
	{
		if (pm_save_wakeup_count(temp_count)) 
		{
			return true; 
		}
		else
		{
			pr_info("[SLP]: error save wakeup_count: %d ", temp_count);
		}
	}

	return false;
}

/**
 * zx_sleep_idle_check 
 * 
 * check whether linux allowed idlesleep.
 *
 * when idle flag is 0, return "true", 
 * else return "false"
 */
static bool zx_sleep_idle_check(void)
{
	if(pm_dma_used())
		return false;
	
	if(drv_cpuidle_flag != 0)
		return false;

	return true;

}

/**
 * zx_cpu_kernel_sleep 
 * 
 * 
 *
 */
void zx_cpu_kernel_sleep(void)
{
	cpu_do_idle();
}

/**
 * zx_sleep_set_timer
 * 
 * input: expect_time_ms ---the expect_time can sleep, unit:ms
 *
 * return: elapsed time when enter idle
 */
volatile u32 tick_cur_cnt =0;

u64 zx_sleep_set_timer(u32 expect_time_ms)
{
    u64 tmptime = 0;

	idle_set_sleeptime((s64)expect_time_ms*(s64)1000);
	pm_stop_tick();
	tick_cur_cnt=pm_read_tick();
    tmptime =div64_u64(read_persistent_us(),(u64)1000);// read_persistent_us()/1000; //ms
    return tmptime;	
}

/**
 * zx_sleep_read_timer
 * 
 * return:  elapsed time when exit idle
 *
 * 
 */
u64 zx_sleep_read_timer(void)
{
    u64 tmptime = 0;
    zx29_stop_wake_timer();

    tmptime =div64_u64(read_persistent_us(),(u64)1000);// read_persistent_us()/1000; //ms
    pm_restart_tick(tick_cur_cnt);

    return tmptime;
}

/**
 * zx_sleep_mask_int
 * 
 *  mask int needed
 *
 * 
 */
void zx_sleep_mask_int(void)
{
    /*******************?????D??***********************/
#ifndef CONFIG_ARCH_ZX297520V3_CAP	
    CPPS_FUNC(cpps_callbacks, zDrvLpm_IrqDisable)(LPM_RAT_TD);
    CPPS_FUNC(cpps_callbacks, zDrvLpm_IrqDisable)(LPM_RAT_LTE);
    CPPS_FUNC(cpps_callbacks, zDrvLpm_IrqDisable)(LPM_RAT_W);
#endif	
    /*********************************************************/
  //  zDrvInt_MaskIrq(TD_FRM_INT);
   // zDrvInt_MaskIrq(LTE_LPM5_INT);
   // zDrvInt_MaskIrq(WD_FRM_INT);
  //  CPPS_FUNC(cpps_callbacks, zDrvInt_MaskIrq)(PS_TIMER0_INT);
	pm_save_gic_wake_enable();
}

/**
 * zx_sleep_unmaskInt
 * 
 * unmask int needed
 *
 * 
 */
static void zx_sleep_unmaskInt(void)
{
#ifndef CONFIG_ARCH_ZX297520V3_CAP

   if (CPPS_FUNC(cpps_callbacks, zPs_IsTdMasterMode)())
	   CPPS_FUNC(cpps_callbacks, zDrvLpm_IrqEnable)(LPM_RAT_TD); 
   if (CPPS_FUNC(cpps_callbacks, zPs_IsLteMasterMode)())
	   CPPS_FUNC(cpps_callbacks, zDrvLpm_IrqEnable)(LPM_RAT_LTE); 
   if (CPPS_FUNC(cpps_callbacks, zPs_IsFddMasterMode)())
	   CPPS_FUNC(cpps_callbacks, zDrvLpm_IrqEnable)(LPM_RAT_W); 
 #endif  
 //  zDrvInt_UnmaskIrq(TD_FRM_INT);
//	 zDrvInt_UnmaskIrq(LTE_LPM5_INT);
//	 zDrvInt_UnmaskIrq(WD_FRM_INT);
   //CPPS_FUNC(cpps_callbacks, zDrvInt_UnmaskIrq)(PS_TIMER0_INT); 
 	pm_restore_gic_wake_enable();
}


/**
 * zx_cpu_idle
 * the deep sleep function, enter and exit WFI, dormant or shutdown sleep
 * 
 *
 * 
 */
extern unsigned int zx_getspeed(unsigned int cpu);
extern void clock_event_handler(void);
extern  void pm_uart_mod_timer(void);
extern  void pm_uart_del_timer(void);


#ifdef CONFIG_ARCH_ZX297520V3_CAP
#define  CAPCORE_SLEEP_TIME		(18*60*60*1000)  //ms

void  zx_cpu_idle(void)
{
	u32 expect_time = 0;
	u64 elapsed_time_enter = 0;
	u64 elapsed_time_exit = 0;	
	u64 idle_time = 0;
	
	#ifdef CONFIG_ZX_PM_DEBUG
	pm_write_reg(AP_IDLE_SLEEP_STATUS_FLAG,0xff01);
	#endif
	
	if(!zx_sleep_wakelock_check())
		goto IRQ_LOCK_EXIT;

	if(!zx_debug_check())
		goto IRQ_LOCK_EXIT;

	expect_time =CAPCORE_SLEEP_TIME;
	
	elapsed_time_enter=zx_sleep_set_timer(expect_time );
	
	if(expect_time <= SHUTDOWN_SLEEP_TIME)
		goto IRQ_LOCK_EXIT;
	
	sleep_count++;

	zx_sleep_mask_int();

	#ifdef CONFIG_ZX_PM_DEBUG
	pm_write_reg(AP_IDLE_SLEEP_STATUS_FLAG,0xff03);
	#endif

	if(expect_time >= SAVE_RESTORE_TIME)
      		zx_enter_sleep(CPU_SLEEP_TYPE_LP1);
	else
   		zx_enter_sleep(CPU_SLEEP_TYPE_LP3);

	#ifdef CONFIG_ZX_PM_DEBUG
	pm_write_reg(AP_IDLE_SLEEP_STATUS_FLAG,0xfffe);
	#endif
	
	zx_sleep_unmaskInt();
	
	elapsed_time_exit = zx_sleep_read_timer();
	if(elapsed_time_exit>=elapsed_time_enter)
		idle_time=elapsed_time_exit - elapsed_time_enter;
	else
		idle_time=(elapsed_time_exit - elapsed_time_enter)+MAX_TIME_PER_CYCLE;
	
	clock_event_handler();
	
	pm_ram_log(" @@sleep exit:sleep_count=%u,real_idle_time=%lld,jiffies:%u\n",sleep_count, idle_time, jiffies);
	
IRQ_LOCK_EXIT:
	#ifdef CONFIG_ZX_PM_DEBUG
	pm_write_reg(AP_IDLE_SLEEP_STATUS_FLAG,0xffff);
	#endif
	zx_cpu_kernel_sleep();
	
}
#else
void  zx_cpu_idle(void)
{

	u32 expect_time = 0;
	u64 elapsed_time_enter = 0;
	u64 elapsed_time_exit = 0;	
	u64 idle_time = 0;
 	//s64 request = 0;
 	s64 remainder_timer = 0;

	#ifdef CONFIG_ZX_PM_DEBUG
	pm_write_reg(AP_IDLE_SLEEP_STATUS_FLAG,0xff01);
	pm_write_reg(IRAM_PS_SLEEP_FLAG_ADDR,1);
	#endif

	#ifdef USE_CPPS_KO
	if(cpps_callbacks.psm_ModemSleepCheck) {
		if(!CPPS_FUNC(cpps_callbacks,psm_ModemSleepCheck)())
			goto IRQ_LOCK_EXIT;
	} else{
		goto IRQ_LOCK_EXIT;
	}
	
	#else
	if(!psm_ModemSleepCheck())	
		goto IRQ_LOCK_EXIT;
	#endif	
	if(!zx_sleep_idle_check())
		goto IRQ_LOCK_EXIT;
	if(!zx_sleep_wakelock_check())
		//goto IRQ_LOCK_EXIT;
		sleep_mode_flag=1;
	if(!zx_debug_check())
		goto IRQ_LOCK_EXIT;

	#ifdef USE_CPPS_KO
    if(!sleep_mode_flag)
	expect_time=CPPS_FUNC(cpps_callbacks,psm_ModemSleepTimeGet)();	
    else
     {
		remainder_timer = pm_get_remainder_time();
		//request = ktime_to_us(tick_nohz_get_sleep_length());			
            expect_time=CPPS_FUNC(cpps_callbacks,psm_ModemSleepTimeGet)();  
		if(div64_long(remainder_timer, 1000) < expect_time)
			expect_time =div64_long(remainder_timer, 1000);		
             
             //if(div64_long(request, 1000) < expect_time)
                 //expect_time =div64_long(request, 1000);     

     }
	#else
    if(!sleep_mode_flag)
	expect_time = psm_ModemSleepTimeGet();
    else
    {
		remainder_timer = pm_get_remainder_time();
		//request = ktime_to_us(tick_nohz_get_sleep_length());			
            expect_time=psm_ModemSleepTimeGet();  
		if(div64_long(remainder_timer, 1000) < expect_time)
			expect_time =div64_long(remainder_timer, 1000);		
             
             //if(div64_long(request, 1000) < expect_time)
                 //expect_time =div64_long(request, 1000);     

     }
    
	#endif
	if(expect_time <= SHUTDOWN_SLEEP_TIME)
		goto IRQ_LOCK_EXIT;
	
	elapsed_time_enter=zx_sleep_set_timer(expect_time );
	
	#ifdef CONFIG_ZX_RAM_CONSOLE 
	pm_uart_del_timer();
	pm_idle_sram_start();
	#endif

#ifdef USE_CPPS_KO
	if(cpps_callbacks.zDrvEdcp_IsBusy) {
		if(CPPS_FUNC(cpps_callbacks,zDrvEdcp_IsBusy)(0) ||
		   CPPS_FUNC(cpps_callbacks,zDrvEdcp_IsBusy)(1) ||
		   CPPS_FUNC(cpps_callbacks,zDrvEdcp_IsBusy)(2))
		   BUG_ON(1);
	}
#endif
	
	sleep_count++;
	pm_ram_log(" @@sleep enter,expect_time:%u,jiffies:%u\n", expect_time, jiffies);

	//pm_ram_log(" freq_sel=0x%x \n",  pm_read_reg(ZX_MATRIX_CRM_BASE+0x158) );
	
	#ifdef CONFIG_ZX_PM_DEBUG
	pm_write_reg(AP_IDLE_SLEEP_STATUS_FLAG,0xff02);
	#endif
 		
	
 	if(zx_cpu_dev_idle_enter()) {
		printk(KERN_WARNING"dev idle enter error\n");
		if(zx_cpu_dev_idle_exit())
			printk(KERN_WARNING"dev idle exit error\n");
		goto IRQ_LOCK_EXIT;
	}

	zx_sleep_mask_int();

	#ifdef CONFIG_ZX_PM_DEBUG
	pm_write_reg(AP_IDLE_SLEEP_STATUS_FLAG,0xff03);
	#endif

	if((expect_time >= SAVE_RESTORE_TIME) && (!sleep_mode_flag) )
      	zx_enter_sleep(CPU_SLEEP_TYPE_LP1);
	else
   		zx_enter_sleep(CPU_SLEEP_TYPE_LP3);

	#ifdef CONFIG_ZX_PM_DEBUG
	pm_write_reg(AP_IDLE_SLEEP_STATUS_FLAG,0xfffe);
	#endif
	
	zx_sleep_unmaskInt();

	if(zx_cpu_dev_idle_exit()) {
		printk(KERN_WARNING"dev idle exit error\n");
		goto IRQ_LOCK_EXIT;
	}



	elapsed_time_exit = zx_sleep_read_timer();
	if(elapsed_time_exit>=elapsed_time_enter)
		idle_time=elapsed_time_exit - elapsed_time_enter;
	else
		idle_time=(elapsed_time_exit - elapsed_time_enter)+MAX_TIME_PER_CYCLE;
	
	clock_event_handler();
	#ifdef USE_CPPS_KO
	CPPS_FUNC(cpps_callbacks,psm_TimeCompensate)(idle_time);	
	#else
	psm_TimeCompensate(idle_time);
	#endif

	pm_ram_log(" @@sleep exit:sleep_count=%u,real_idle_time=%lld,jiffies:%u\n",sleep_count, idle_time, jiffies);

	#ifdef CONFIG_ZX_RAM_CONSOLE 
	pm_uart_mod_timer();
	pm_idle_sram_end();	
	#endif


IRQ_LOCK_EXIT:
	#ifdef CONFIG_ZX_PM_DEBUG
	pm_write_reg(IRAM_PS_SLEEP_FLAG_ADDR,0);
	pm_write_reg(AP_IDLE_SLEEP_STATUS_FLAG,0xffff);
	#endif
	sleep_mode_flag=0;
	zx_cpu_kernel_sleep();

}
#endif
/****************************************************/

/**
 * zx_enter_idle
 * @dev: cpuidle device
 * @state: The target state to be programmed
 *
 * Idle function for C1 state, WFI on a single CPU.
 * Called with irqs off, returns with irqs on.
 * Returns the amount of time spent in the low power state.
 */
int zx_enter_idle(struct cpuidle_device *dev,
				struct cpuidle_driver *drv,
				int index)
{
	ktime_t entry_time, exit_time;
	s64 idle_time;
	int new_index = index;

	local_irq_disable();

#ifdef CONFIG_ARCH_ZX297520V2
#else
	local_fiq_disable();
#endif
	entry_time = ktime_get();

/*=================================================================
 *=======begin enter idle sleep====================================
 *=================================================================
 */
	new_index = zx_pm_idle_prepare(dev, drv, index);

	index = zx_pm_idle_enter(new_index);

/*=================================================================
 *=======end enter idle sleep======================================
 *=================================================================
 */
	exit_time = ktime_get();
#ifdef CONFIG_ARCH_ZX297520V2
#else
	local_fiq_enable();
#endif
	local_irq_enable();

	idle_time = ktime_to_us(ktime_sub(exit_time, entry_time));

	dev->last_residency = (int)idle_time;

	if(print_enabled_by_debug != 0)
	{
		printk(KERN_INFO "[CPUIDLE] exit idle: idle time= %d , enter level= %d !\n", (u32)idle_time, index);
	}

	return index;
}



static int idle_pm_notify(struct notifier_block *nb,
	unsigned long event, void *dummy)
{
#ifdef CONFIG_PM_SLEEP
	if (event == PM_SUSPEND_PREPARE)
		deep_idle_disabled_by_suspend = true;
	else if (event == PM_POST_SUSPEND)
		deep_idle_disabled_by_suspend = false;
#endif

	return NOTIFY_OK;
}

static struct notifier_block idle_pm_notifier =
{
	.notifier_call = idle_pm_notify,
};


void zx_apmgclken_set(unsigned en)
{
	unsigned tmp;
	if(en){
		//set ps_clk_switch=1
		tmp = pm_read_reg(CORE_SWITCH_CONFIG_REG);
		#ifdef CONFIG_ARCH_ZX297520V3_CAP
		tmp |= (0x1<<2);		
		#else
		tmp |= (0x1<<0);
		#endif
		pm_write_reg(CORE_SWITCH_CONFIG_REG, tmp);	
	} else{
		//set ps_clk_switch=0
		tmp = pm_read_reg(CORE_SWITCH_CONFIG_REG);
		#ifdef CONFIG_ARCH_ZX297520V3_CAP
		tmp &= ~(0x1<<2);		
		#else		
		tmp &= ~(0x1<<0);
		#endif
		pm_write_reg(CORE_SWITCH_CONFIG_REG, tmp);	
	}
}
/**
 * zx_cpuidle_init - Init routine for zx29xx idle
 *
 * Registers the cpuidle driver with the cpuidle
 * framework with the valid set of states.
 */
int zx_cpuidle_init(void)
{

#if 1
	arm_pm_idle = zx_cpu_idle;
	zx_apmgclken_set(0);
	printk(KERN_INFO "[CPUIDLE] zx_cpu_idle init OK\n,");
#else	
	int cpu_id;
	struct cpuidle_device *device;
	struct cpuidle_driver *drv = &zx_idle_driver;

	/* Setup cpuidle driver */
	drv->state_count = zx_fill_cpuidle_data(drv);
	cpuidle_register_driver(drv);

	/* Setup cpuidle device for each cpu */
	for_each_cpu(cpu_id, cpu_online_mask)
	{
		device = &per_cpu(zx_idle_dev, cpu_id);
		device->cpu = cpu_id;

		if (cpu_id == 0)
			device->state_count = drv->state_count;
		else
			device->state_count = 1;	/* None boot cpu Support IDLE only now ! */

		if (cpuidle_register_device(device))
		{
			printk(KERN_ERR "[CPUIDLE] register device failed\n,");
			return -EIO;
		}
	}

	register_pm_notifier(&idle_pm_notifier);

	INIT_DELAYED_WORK_DEFERRABLE(&pm_idle_work, pm_idle_func);
	schedule_delayed_work(&pm_idle_work, PM_IDLE_DELAY);

	printk(KERN_INFO "[CPUIDLE] register device OK\n,");
#endif

	return 0;
}

#if 0
static void __exit zx_cpuidle_exit(void)
{
	unregister_pm_notifier(&idle_pm_notifier);
	cpuidle_unregister_driver(&zx_idle_driver);
}

late_initcall(zx_cpuidle_init);
module_exit(zx_cpuidle_exit);
#endif

#ifdef CONFIG_ZX_PM_DEBUG
static char*  lp2_debug_show(char *s)
{
	#if 0
	int i;

	s += sprintf(s, "%-30s%8s %8s %8s %8s\n", " ", "cpu0","cpu1","cpu2","cpu3");
	s += sprintf(s, "%s\n", "---------------------------------------------------------------");
	s += sprintf(s, "%-30s%8u %8u %8u %8u\n", "lp3 in count:",
		idle_stats.lp3_count[0],
		idle_stats.lp3_count[1],
		idle_stats.lp3_count[2],
		idle_stats.lp3_count[3]);

	s += sprintf(s, "%-30s%8u %8u %8u %8u\n", "lp2 in count:",
		idle_stats.lp2_count[0],
		idle_stats.lp2_count[1],
		idle_stats.lp2_count[2],
		idle_stats.lp2_count[3]);

	s += sprintf(s, "%-30s%8u %8u %8u %8u\n", "lp2 completed:",
		idle_stats.lp2_completed_count[0],
		idle_stats.lp2_completed_count[1],
		idle_stats.lp2_completed_count[2],
		idle_stats.lp2_completed_count[3]);

	s += sprintf(s, "%-30s%7u%% %7u%% %7u%% %7u%%\n", "lp2 completed%:",
		idle_stats.lp2_completed_count [0]* 100 / (idle_stats.lp2_count[0] ?: 1),
		idle_stats.lp2_completed_count [1]* 100 / (idle_stats.lp2_count[1] ?: 1),
		idle_stats.lp2_completed_count [2]* 100 / (idle_stats.lp2_count[2] ?: 1),
		idle_stats.lp2_completed_count [3]* 100 / (idle_stats.lp2_count[3] ?: 1));
	s += sprintf(s, "%-30s%8u\n", "all idle count:", idle_stats.idle_count);

	s += sprintf(s, "\n%-30s%8llu %8llu %8llu %8llu ms\n", "cpu ready time:",
		div64_u64(idle_stats.cpu_wants_lp2_time[0], 1000),
		div64_u64(idle_stats.cpu_wants_lp2_time[1], 1000),
		div64_u64(idle_stats.cpu_wants_lp2_time[2], 1000),
		div64_u64(idle_stats.cpu_wants_lp2_time[3], 1000));

	s += sprintf(s, "%-30s%8llu %8llu %8llu %8llu ms\n", "lp2 in time:",
		div64_u64(idle_stats.in_lp2_time[0], 1000),
		div64_u64(idle_stats.in_lp2_time[1], 1000),
		div64_u64(idle_stats.in_lp2_time[2], 1000),
		div64_u64(idle_stats.in_lp2_time[3], 1000));

	s += sprintf(s, "%-30s%7d%% %7d%% %7d%% %7d%%\n", "lp2 in time%:",
		(int)(idle_stats.cpu_wants_lp2_time[0] ?
			div64_u64(idle_stats.in_lp2_time[0] * 100,
			idle_stats.cpu_wants_lp2_time[0]) : 0),
		(int)(idle_stats.cpu_wants_lp2_time[1] ?
			div64_u64(idle_stats.in_lp2_time[1] * 100,
			idle_stats.cpu_wants_lp2_time[1]) : 0),
		(int)(idle_stats.cpu_wants_lp2_time[2] ?
			div64_u64(idle_stats.in_lp2_time[2] * 100,
			idle_stats.cpu_wants_lp2_time[2]) : 0),
		(int)(idle_stats.cpu_wants_lp2_time[3] ?
			div64_u64(idle_stats.in_lp2_time[3] * 100,
			idle_stats.cpu_wants_lp2_time[3]) : 0));

	s += sprintf(s, "\n\n%3s %20s %6s %10s\n",
		"int", "name", "count", "last count");
	s += sprintf(s, "%s", "--------------------------------------------\n");
	for (i = 0; i < NR_IRQS; i++) {
		if (idle_stats.lp2_int_count[i] == 0)
			continue;
		s += sprintf(s, "%3d %20s %6d %10d\n",
			i - GIC_SPI_START,
			irq_to_desc(i)->action ? irq_to_desc(i)->action->name ?: "???" : "???",
			idle_stats.lp2_int_count[i],
			idle_stats.lp2_int_count[i] - idle_stats.last_lp2_int_count[i]);

		idle_stats.last_lp2_int_count[i] = idle_stats.lp2_int_count[i];
	};
	#endif
	
	return s;
}

/******************************************************
 *** 1 -- lp2 ******************************
 ******************************************************
 */
static ssize_t lp2_show(struct kobject *kobj, struct kobj_attribute *attr,
			  char *buf)
{
	char *s = buf;

	s = lp2_debug_show(s);


	return (s - buf);
}

static ssize_t lp2_store(struct kobject *kobj, struct kobj_attribute *attr,

			   const char *buf, size_t n)
{

	int error = 0;


	return error ;
}

zte_pm_attr(lp2);


/*=============================================================================
 *========  /sys/zte_pm/cpuidle/disable_lp2  ==================================
 *=============================================================================
 */
static ssize_t disable_lp2_show(struct kobject *kobj, struct kobj_attribute *attr,
			  char *buf)
{
	char *s = buf;

	s += sprintf(s, "%s %d\n", "[CPUIDLE] deep_idle_disabled_by_debug:", deep_idle_disabled_by_debug);

	return (s - buf);
}

/* usage: "echo 1 > disable_lp2" */
static ssize_t disable_lp2_store(struct kobject *kobj, struct kobj_attribute *attr,
			   const char *buf, size_t n)
{
	int error = 0;
	long temp;

	if(strict_strtol(buf, 0, &temp))
		error = -EINVAL;

	deep_idle_disabled_by_debug = temp;

	return error ? error : n;
}
zte_pm_attr(disable_lp2);


/*=============================================================================
 *========  /sys/zte_pm/cpuidle/enable_print  ==================================
 *=============================================================================
 */
static ssize_t enable_print_show(struct kobject *kobj, struct kobj_attribute *attr,
			  char *buf)
{
	char *s = buf;

	s += sprintf(s, "%s %d\n", "[CPUIDLE] print_enabled_by_debug:", print_enabled_by_debug);

	return (s - buf);
}

/* usage: "echo 1 > enable_print" */
static ssize_t enable_print_store(struct kobject *kobj, struct kobj_attribute *attr,
			   const char *buf, size_t n)
{
	int error = 0;
	long temp;

	if(strict_strtol(buf, 0, &temp))
		error = -EINVAL;

	print_enabled_by_debug = temp;

	return error ? error : n;
}

zte_pm_attr(enable_print);


/*=============================================================================
 *========  /sys/zte_pm/cpuidle/drv_cpuidle_flag  ==================================
 *=============================================================================
 */
static ssize_t drv_cpuidle_flag_show(struct kobject *kobj, struct kobj_attribute *attr,
			  char *buf)
{
	char *s = buf;

	s += sprintf(s, "%s %x %x\n", "[CPUIDLE] drv_cpuidle_flag, pm_dma_used(): ", drv_cpuidle_flag, pm_dma_used());

	return (s - buf);
}

/* usage: "echo * > drv_cpuidle_flag" */
static ssize_t drv_cpuidle_flag_store(struct kobject *kobj, struct kobj_attribute *attr,
			   const char *buf, size_t n)
{
	int error = 0;
	long temp;

	if(strict_strtol(buf, 0, &temp))
		error = -EINVAL;

	drv_cpuidle_flag = temp;

	return error ? error : n;
}

zte_pm_attr(drv_cpuidle_flag);


/*=============================================================================
 *========  /sys/zte_pm/cpuidle/idle_debug_mask  ==================================
 *=============================================================================
 */
static ssize_t enable_idle_debug_show(struct kobject *kobj, struct kobj_attribute *attr,
			  char *buf)
{
	char *s = buf;

	s += sprintf(s, "%s %d\n", "[CPUIDLE] idle_print_enabled_by_debug:", idle_debug_mask);

	return (s - buf);
}

/* usage: "echo 1 > idle_debug_mask" */
static ssize_t enable_idle_debug_store(struct kobject *kobj, struct kobj_attribute *attr,
			   const char *buf, size_t n)
{
	int error = 0;
	long temp;

	if(strict_strtol(buf, 0, &temp))
		error = -EINVAL;

	idle_debug_mask = temp;

	return error ? error : n;
}
zte_pm_attr(enable_idle_debug);


static struct attribute * g[] =
{
	&lp2_attr.attr,
	&disable_lp2_attr.attr,
	&enable_print_attr.attr,
	&drv_cpuidle_flag_attr.attr,	
	&enable_idle_debug_attr.attr,	
	NULL,
};


static struct attribute_group idle_attr_group =
{
	.attrs = g,
};
/**
 * idle_debug_init
 * create cpuidle sysfs, we can use cat /sys/zte_pm/cpuidle/lp2 command view debug info
 *
 */
static struct kobject *idle_kobj;

int __init idle_debug_init(void)
{
	int ret;

	idle_kobj = kobject_create_and_add("cpuidle", pm_debug_kobj);
	if (!idle_kobj)
		return -ENOMEM;

    ret = sysfs_create_group(idle_kobj, &idle_attr_group);
    if (ret)
    {
        pr_info("[CPUIDLE] sysfs_create_group ret %d\n", ret);
		return ret;
    }

	return 0;
}

#endif

