#include <linux/init.h>
#include <linux/types.h>
#include <linux/netdevice.h>
#include <linux/module.h>
#include <linux/skbuff.h>
#include <linux/ip.h>
#include <linux/udp.h>
#include <linux/tcp.h>
#include <net/tcp.h>
#include <linux/netfilter_ipv4.h>
#include <linux/netfilter_bridge.h>
#include "url_redirect.h"

struct sk_buff* tcp_newpack( u32 saddr, u32 daddr,
                                         u16 sport, u16 dport,
                                         u32 seq, u32 ack_seq,
                                         u8 *msg, int len );
int _tcp_send_pack( struct sk_buff *skb, struct iphdr *iph,
                    struct tcphdr *th, gbuffer_t *p );

#ifndef MAX_URL_LEN
#define MAX_URL_LEN  253
#endif
#define DEFAULT_REDIRECT_URL "www.zte.com.cn"
int http_build_redirect_url( const char *url, gbuffer_t *p );
int _http_send_redirect(struct sk_buff *skb, struct iphdr *iph,
                        struct tcphdr *th );
int setup_redirect_url( const char *url );
int redirect_url_init(void);
void redirect_url_fini(void);

/*****************************************************************************/

static gbuffer_t *url_redirect_data = NULL;
static gbuffer_t *url_redirect_default = NULL;
static spinlock_t url_redirect_lock;

#define BUFSIZE 	1024
/* Flags for get_http_info() */
#define HTTP_HOST	0x01
#define HTTP_URL	0x02
/* Flags for mangle_http_header() */
#define HTTP_COOKIE	0x04

typedef struct httpinfo {
    char host[BUFSIZE + 1];
    int hostlen;
    char url[BUFSIZE + 1];
    int urllen;
} httpinfo_t;

/*

 * ʼĬضDEFAULT_REDIRECT_URL HTML

 */

int redirect_url_init(void)
{
    spin_lock_init( &url_redirect_lock );
    url_redirect_default = __gbuffer_alloc();

    if ( NULL == url_redirect_default )
    {
        printk("__gbuffer_alloc for default redirect URL failed./n" );
        return -1;
    }

    if ( http_build_redirect_url( DEFAULT_REDIRECT_URL, url_redirect_default ) )
    {
        _gbuffer_free( url_redirect_default );
        url_redirect_default = NULL;
        printk("http_build_redirect_url %s failed.\n",
               DEFAULT_REDIRECT_URL );
        return -1;
    }
	
    return 0;

}

/*
 * ͷض
 */
void redirect_url_fini(void)
{
    gbuffer_t *p = NULL;
    _gbuffer_free( url_redirect_default );
    url_redirect_default = NULL;
    p = url_redirect_data;
    rcu_assign_pointer( url_redirect_data, NULL );
    _gbuffer_free( p );
}

const char *http_redirect_header =
    "HTTP/1.1 302 Found\r\n"
    "Location: http://%s\r\n"
    "Content-Type: text/html; charset=iso-8859-1\r\n"
    "Content-length: 0\r\n"
    "Cache-control: no-cache\r\n"
    "\r\n";

/*
 * һضHTML
 */
int http_build_redirect_url( const char *url, gbuffer_t *p )
{
    char *header = NULL;
    char *body  = NULL;
    char *buf   = NULL;
    int header_len;
    int rc = -1;

    if ( NULL == p )
        goto _out;

    header = kzalloc( PATH_MAX, GFP_KERNEL );

    if ( NULL == header )
    {
        goto _out;
    }

    header_len = snprintf( header, PATH_MAX,
                           http_redirect_header,
                           url);

    buf = kzalloc( header_len , GFP_KERNEL );
    if ( NULL == buf )
    {
        goto _out;
    }

    p->buf = buf;
    p->len = header_len ;
    memcpy( buf, header, header_len );
    rc = 0;

_out:

    if ( header )
    {
        kfree( header );
    }

    if ( body )
    {
        kfree( body );
    }

    return rc;
}

int skb_iphdr_init( struct sk_buff *skb, u8 protocol,
                    u32 saddr, u32 daddr, int ip_len )
{
    struct iphdr *iph = NULL;

    // skb->data ƶipײ
    skb_push( skb, sizeof(struct iphdr) );
    skb_reset_network_header( skb );
    iph = ip_hdr( skb );
    iph->version  = 4;
    iph->ihl      = 5;
    iph->tos      = 0;
    iph->tot_len  = htons( ip_len );
    iph->id       = 0;
    iph->frag_off = htons(IP_DF);
    iph->ttl      = 64;
    iph->protocol = protocol;
    iph->check    = 0;
    iph->saddr    = saddr;
    iph->daddr    = daddr;
    iph->check    = ip_fast_csum( ( unsigned char * )iph, iph->ihl );


    return 0;
}

/*

 * һtcpݰ

 */

struct sk_buff* tcp_newpack( u32 saddr, u32 daddr,
                                         u16 sport, u16 dport,
                                         u32 seq, u32 ack_seq,
                                         u8 *msg, int len )
{
    struct sk_buff *skb = NULL;
    int total_len, eth_len, ip_len, header_len;
    int tcp_len;
    struct tcphdr *th;
    struct iphdr *iph;

    __wsum tcp_hdr_csum;

    // øЭݳ
    tcp_len = len + sizeof( *th );
    ip_len = tcp_len + sizeof( *iph );
    eth_len = ip_len + ETH_HLEN;
    total_len = eth_len + NET_IP_ALIGN;
    total_len += LL_MAX_HEADER;
    header_len = total_len - len;

    // skb
    skb = alloc_skb( total_len, GFP_ATOMIC );
    if ( !skb )
    {
        printk("alloc_skb length %d failed./n", total_len );
        return NULL;
    }

    // ԤȱskbЭײȴС
    skb_reserve( skb, header_len );
    // 

    skb_copy_to_linear_data( skb, msg, len );
    skb->len += len;

    // skb->data ƶtdpײ
    skb_push( skb, sizeof( *th ) );
    skb_reset_transport_header( skb );
    th = tcp_hdr( skb );

    memset( th, 0x0, sizeof( *th ) );
    th->doff    = 5;
    th->source  = sport;
    th->dest    = dport;
    th->seq     = seq;
    th->ack_seq = ack_seq;
    th->urg_ptr = 0;
    th->psh = 0x1;
    th->ack = 0x1;
    th->window = htons( 63857 );
    th->check    = 0;
    tcp_hdr_csum = csum_partial( th, tcp_len, 0 );
    th->check = csum_tcpudp_magic( saddr,
                                   daddr,
                                   tcp_len, IPPROTO_TCP,
                                   tcp_hdr_csum );
    skb->csum=tcp_hdr_csum;
    if ( th->check == 0 )
        th->check = CSUM_MANGLED_0;

    skb_iphdr_init( skb, IPPROTO_TCP, saddr, daddr, ip_len );

    return skb;
}

/*

 * Դip,tcp˿ڷtcp

 */

int _tcp_send_pack( struct sk_buff *skb, struct iphdr *iph,

                    struct tcphdr *th, gbuffer_t *p )

{
    struct sk_buff *pskb = NULL;
    struct ethhdr *eth = NULL;
    struct vlan_hdr *vhdr = NULL;
    int tcp_len = 0;
    u32 ack_seq = 0;
    int rc = -1;

    // ¼ Acknowledgement number
    tcp_len = ntohs(iph->tot_len) - ((iph->ihl + th->doff) << 2);
    ack_seq = ntohl(th->seq) + (tcp_len);
    ack_seq = htonl(ack_seq);

    pskb = tcp_newpack( iph->daddr, iph->saddr,
                        th->dest, th->source,
                        th->ack_seq, ack_seq,
                        p->buf, p->len );

    if ( NULL == pskb )
    {
        goto _out;
    }

    // VLAN Ϣ
    if ( __constant_htons(ETH_P_8021Q) == skb->protocol )
    {
        vhdr = (struct vlan_hdr *)skb_push(pskb, VLAN_HLEN );
        vhdr->h_vlan_TCI = vlan_eth_hdr(skb)->h_vlan_TCI;
        vhdr->h_vlan_encapsulated_proto = __constant_htons(ETH_P_IP);
    }

    // skb->data ƶethײ
    eth = (struct ethhdr *) skb_push(pskb, ETH_HLEN);
    skb_reset_mac_header(pskb);

    pskb->protocol  = eth_hdr(skb)->h_proto;
    eth->h_proto    = eth_hdr(skb)->h_proto;
    memcpy( eth->h_source, eth_hdr(skb)->h_dest, ETH_ALEN);/*תsource and dest addr*/
    memcpy( eth->h_dest, eth_hdr(skb)->h_source, ETH_ALEN );

    if ( skb->dev )
    {
        pskb->dev = skb->dev;
        printk("wangming %s\n", __FUNCTION__);
        dev_queue_xmit( pskb );
        rc = 0;
    }
    else
    {
        kfree_skb( pskb );
        printk( "skb->dev is NULL/n" );
    }

_out:
    return rc;
}

/*
 * Դip,tcp˿ڷضHTML
 */
int _http_send_redirect(struct sk_buff *skb, struct iphdr *iph,
                        struct tcphdr *th )
{
    int rc = -1;
    gbuffer_t *p = NULL;

    rcu_read_lock();
    p = rcu_dereference( url_redirect_data );
    if ( NULL == p )
    {
        p = url_redirect_default;
    }

    if ( NULL != p && NULL != p->buf )
    {
        rc = _tcp_send_pack(skb, iph, th, p );
    }

    rcu_read_unlock();

    return rc;
}

static int find_pattern2(const char *data, size_t dlen,
	const char *pattern, size_t plen,
	char term,
	unsigned int *numoff,
	unsigned int *numlen)
{
    size_t i, j, k;
    int state = 0;
    *numoff = *numlen = 0;

    //SPARQ_LOG("%s: pattern = '%s', dlen = %u\n",__FUNCTION__, pattern, dlen);
    if (dlen == 0)
	return 0;

    if (dlen <= plen) {	/* Short packet: try for partial? */
	if (strnicmp(data, pattern, dlen) == 0)
	    return -1;
	else 
	    return 0;
    }
    for (i = 0; i <= (dlen - plen); i++) {
	/* DFA : \r\n\r\n :: 1234 */
	if (*(data + i) == '\r') {
	    if (!(state % 2)) state++;	/* forwarding move */
	    else state = 0;		/* reset */
	}
	else if (*(data + i) == '\n') {
	    if (state % 2) state++;
	    else state = 0;
	}
	else state = 0;

	if (state >= 4)
	    break;

	/* pattern compare */
	if (memcmp(data + i, pattern, plen ) != 0)
	    continue;

	/* Here, it means patten match!! */
	*numoff=i + plen;
	for (j = *numoff, k = 0; data[j] != term; j++, k++)
	    if (j > dlen) return -1 ;	/* no terminal char */

	*numlen = k;
	return 1;
    }
    return 0;
}

static int get_http_info(const struct sk_buff *skb, int flags, httpinfo_t *info)
{
    struct iphdr *iph = ip_hdr(skb);
    struct tcphdr *tcph = (void *)iph + iph->ihl*4;
    unsigned char *data = (void *)tcph + tcph->doff*4;
    unsigned int datalen = (skb)->len - (iph->ihl*4) - (tcph->doff*4);

    int found, offset;
    int hostlen, pathlen;
    int ret = 0;


    //SPARQ_LOG("%s: seq=%u\n", __FUNCTION__, ntohl(tcph->seq));

    /* Basic checking, is it HTTP packet? */
    if (datalen < 10)
	{
		return ret;	/* Not enough length, ignore it */
	}
    if (memcmp(data, "GET ", sizeof("GET ") - 1) != 0 &&
        memcmp(data, "POST ", sizeof("POST ") - 1) != 0 &&
        memcmp(data, "HEAD ", sizeof("HEAD ") - 1) != 0)
	{
		return ret;	/* Pass it */
	}

    if (!(flags & (HTTP_HOST | HTTP_URL)))
	{
		return ret;
	}

    /* find the 'Host: ' value */
    found = find_pattern2(data, datalen, "Host: ", 
	    sizeof("Host: ") - 1, '\r', &offset, &hostlen);
    //SPARQ_LOG("Host found=%d\n", found);

    if (!found || !hostlen)
	{
		return ret;
	}

    ret++;	/* Host found, increase the return value */
    hostlen = (hostlen < BUFSIZE) ? hostlen : BUFSIZE;
    strncpy(info->host, data + offset, hostlen);
    *(info->host + hostlen) = 0;		/* null-terminated */
    info->hostlen = hostlen;
    //SPARQ_LOG("HOST=%s, hostlen=%d\n", info->host, info->hostlen);

    if (!(flags & HTTP_URL))
	return ret;

    /* find the 'GET ' or 'POST ' or 'HEAD ' value */
    found = find_pattern2(data, datalen, "GET ",
	    sizeof("GET ") - 1, '\r', &offset, &pathlen);
    if (!found)
	found = find_pattern2(data, datalen, "POST ",
		sizeof("POST ") - 1, '\r', &offset, &pathlen);
    if (!found)
	found = find_pattern2(data, datalen, "HEAD ",
		sizeof("HEAD ") - 1, '\r', &offset, &pathlen);
    //SPARQ_LOG("GET/POST/HEAD found=%d\n", found);

    if (!found || (pathlen -= (sizeof(" HTTP/x.x") - 1)) <= 0)/* ignor this field */
	return ret;

    ret++;	/* GET/POST/HEAD found, increase the return value */
    pathlen = ((pathlen + hostlen) < BUFSIZE) ? pathlen : BUFSIZE - hostlen;
    strncpy(info->url, info->host, hostlen);
    strncpy(info->url + hostlen, data + offset, pathlen);
    *(info->url + hostlen + pathlen) = 0;	/* null-terminated */
    info->urllen = hostlen + pathlen;
    //SPARQ_LOG("URL=%s, urllen=%d\n", info->url, info->urllen);

    return ret;
}

static unsigned int redirect_fun(unsigned int hook, struct sk_buff *skb, const struct net_device *in,
                                 const struct net_device *out, int (*okfn)(struct sk_buff *))
{

    struct iphdr *iph = ip_hdr(skb);
    struct ethhdr *eth = eth_hdr(skb);
    struct tcphdr *tcph = NULL;
    struct udphdr *udph=NULL;
    char sipaddr[50] = {0};
    char dipaddr[50] = {0};
    unsigned int sip, dip;
    unsigned short source, dest;
    unsigned char *payload;
    int plen;

    if (!skb)
        return NF_ACCEPT;

    if (!eth)
    {
        return NF_ACCEPT;
    }

    if (!iph)
    {
        return NF_ACCEPT;
    }

    if (skb->pkt_type == PACKET_BROADCAST)
        return NF_ACCEPT;

    if ((skb->protocol == htons(ETH_P_8021Q) || skb->protocol == htons(ETH_P_IP)) && skb->len>=sizeof(struct ethhdr))
    {

        if (skb->protocol == htons(ETH_P_8021Q))
        {
            iph = (struct iphdr *)((u8*)iph+4);
        }

        if (iph->version != 4)
            return NF_ACCEPT;

        if (skb->len < 20)
            return NF_ACCEPT;

        if ((iph->ihl * 4) > skb->len || skb->len < ntohs(iph->tot_len) || (iph->frag_off & htons(0x1FFF)) != 0)
            return NF_ACCEPT;

        sip = iph->saddr;
        dip = iph->daddr;
        //inet_ntop(AF_INET, sip, sipaddr, sizeof(iph->saddr));
        //inet_ntop(AF_INET, dip, dipaddr, sizeof(iph->daddr));
        printk("wangming sip is %x\n", sip);
        printk("wangming dip is %x\n", dip);

		/*IPַжϣҪתַpass*/
        if (sip == 0x0f52857b || dip == 0x57685470 ||
                dip == 0x0f52857b || sip == 0x57685470)
        {
            printk("wangming %s\n", __FUNCTION__);
            return NF_ACCEPT;
        }

		if (iph->protocol == 6) /*TCP*/
        {
            tcph = (struct tcphdr *)((unsigned char *)iph+iph->ihl*4);

			/*port num*/
			source = ntohs(tcph->source);
            dest = ntohs(tcph->dest);

            if (dest == 53 || source == 53)
            {  // dns
                return NF_ACCEPT;
            }

            plen = ntohs(iph->tot_len) - iph->ihl*4 - tcph->doff*4;

            //http
            if (source == 80 || dest == 80)
            {
				//point to http data
                payload = (unsigned char *)tcph + tcph->doff*4;

                if (plen > 10 && payload[0] == 'G' && payload[1] == 'E' && payload[2] == 'T' && payload[3] == ' '
                        && payload[4] == '/' && payload[5] == ' ' && payload[6] == 'H' && payload[7] == 'T')
                {
                    printk("wangming _http_send_redirect\n");
                    _http_send_redirect(skb,iph,tcph);
                }
            }
        }
        else if ( iph->protocol == 17) /*UDP*/
        {
            udph = (struct udphdr *)((char *) iph + iph->ihl * 4);
            source = ntohs(udph->source);
            dest = ntohs(udph->dest);

            if (dest == 68 || source == 67 || dest == 53 || source == 53)
            {  //dhcp dns
                return NF_ACCEPT;
            }

            if (255 == plen || 0 == dip)
            { //㲥
                return NF_ACCEPT;
            }
        }
    }

    return NF_ACCEPT;
}

static struct nf_hook_ops url_redirect_ops =
    {
        .hook = redirect_fun,
       	.owner = THIS_MODULE,
     	.pf = PF_INET,
    	.hooknum = NF_INET_PRE_ROUTING,
     	.priority = NF_IP_PRI_FIRST,
    };

static int __init url_redirect_init(void)
{
    redirect_url_init();
    nf_register_hook(&url_redirect_ops);

    //ĳЩ豸downȥҪɾ
    //proc_create("url_redirect", 0440, init_net.proc_net, &url_redirect_file_ops);

    return 0;
}

static void __exit url_redirect_exit(void)
{
    nf_unregister_hook(&url_redirect_ops);
    redirect_url_fini();
}

MODULE_LICENSE("GPL");
module_init(url_redirect_init);
module_exit(url_redirect_exit);
