/**
 * Copyright (C) 2017 Sanechips Technology Co., Ltd.
 * @author Yanan Liu <liu.yanan@sanechips.com.cn>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

/*******************************************************************************
 *                         Wrapper of GS Download LIB                          *
 ******************************************************************************/

#define _GNU_SOURCE


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <limits.h>


#include "fota_dm_dl.h"
#include "dmgr_api.h"
#include "fota_dm_utils.h"
#include "fota_dm_macros.h"
#include "fota_common.h"
#include "softap_api.h"
#include "fota_dm_netdog.h"

static int __has_new_version = FALSE;
static int __mandatory = FALSE;
//static struct version_info __cloned_version;



#define DOWNLOAD_RETRY_TIMES (5)

int fota_dm_vendor_is_zx()
{
	char status[FOTA_CFG_GET_ITEMLEN] = {0};
   	fota_dm_get_nv_string(NV_FOTA_DM_VENDOR, status, sizeof(status));

	if (0 == strcmp("zx",status))
		return TRUE;
	else
		return FALSE;
}

int fota_dm_vendor_is_rs()//corem0628
{
	char status[FOTA_CFG_GET_ITEMLEN] = {0};
   	fota_dm_get_nv_string(NV_FOTA_DM_VENDOR, status, sizeof(status));

	if (0 == strcmp("rs",status))
		return TRUE;
	else
		return FALSE;
}

static void init_policy_info(struct policy_info * policy)
{
	if(fota_dm_vendor_is_zx()){
		LOGD("fota dm vendor is zx!\n");
		policy->so_transfer_timeout = 60;
		policy->so_connect_timeout = 10;
	}else{
		policy->so_transfer_timeout = 60;
		policy->so_connect_timeout = 10;
	}
	
	policy->so_recv_buffer_size = 8192;
	policy->download_retry = 3;

	if(fota_dm_vendor_is_zx()|| fota_dm_vendor_is_rs()){//corem0628
		LOGD("fota dm vendor is zx or rs!\n");
		policy->server_transfer_opt = SERVER_TRANSOPT_FOTA | SERVER_TRANSOPT_COMPACT_HTTP;
	}else{
		policy->server_transfer_opt = SERVER_TRANSOPT_IOT | SERVER_TRANSOPT_HTTP;
	}	
	
	policy->sstate_cached_path = "/cache/zte_fota";
	policy->freespace_for_file_frac = 90;
}

static char * notifier_action_to_string(unsigned long action)
{
	switch (action) {
		CASE_RETURN_STR(NOTIFY_DOWNLOAD_PREPARE);
		CASE_RETURN_STR(NOTIFY_DOWNLOAD_ABORTED);
		CASE_RETURN_STR(NOTIFY_DOWNLOAD_FINISHED);
		CASE_RETURN_STR(NOTIFY_CHECKVERSION_PREPARE);
		CASE_RETURN_STR(NOTIFY_CHECKVERSION_ABORTED);
		CASE_RETURN_STR(NOTIFY_CHECKVERSION_FINISHED);
		CASE_RETURN_STR(NOTIFY_DOWNLOAD_TRANSLATING);
	default:
		return "NOFIFY_ACTION_UNKNOWN";
	}
}

static const char *dmgr_error_to_string(unsigned int error)
{
	switch (error) {
		CASE_RETURN_STR(E_OK);
		CASE_RETURN_STR(E_BEGIN);
		CASE_RETURN_STR(E_NOMEM);
		CASE_RETURN_STR(E_INVAL_PARAM);
		CASE_RETURN_STR(E_INVAL_OPERATION);
		CASE_RETURN_STR(E_IO_FAILED);
		CASE_RETURN_STR(E_IO_TIMEDOUT);
		CASE_RETURN_STR(E_VERIFIED_FAILED);
		CASE_RETURN_STR(E_CANCELLED);
		CASE_RETURN_STR(E_NESTING_CALLED);
		CASE_RETURN_STR(E_RESERVE1);
		CASE_RETURN_STR(E_INVAL_TOKEN);
		CASE_RETURN_STR(E_INVAL_PLATFORM);
		CASE_RETURN_STR(E_PARAM_MISSING);
		CASE_RETURN_STR(E_UNCONFIGURED_VERSION);
		CASE_RETURN_STR(E_INTERNAL_ERROR);
		CASE_RETURN_STR(E_RESERVE2);
		CASE_RETURN_STR(E_URL_MALFORMAT);
		CASE_RETURN_STR(E_COULDNT_RESOLVE_HOST);
		CASE_RETURN_STR(E_COULDNT_CONNECT);
		CASE_RETURN_STR(E_HTTP_RETURNED_ERROR);
		CASE_RETURN_STR(E_RANGE_ERROR);
		CASE_RETURN_STR(E_HTTP_POST_ERROR);
		CASE_RETURN_STR(E_BAD_DOWNLOAD_RESUME);
		CASE_RETURN_STR(E_ABORTED);
		CASE_RETURN_STR(E_AGAIN);
		CASE_RETURN_STR(E_TOO_MANY_REDIRECTS);
		CASE_RETURN_STR(E_GOT_NOTHING);
		CASE_RETURN_STR(E_SEND_ERROR);
		CASE_RETURN_STR(E_RECV_ERROR);
		CASE_RETURN_STR(E_NO_CONNECTION_AVAILABLE);
		CASE_RETURN_STR(E_END);
	default:
		return "DMGR_ERROR_UNKNOWN";
	}
}

static void clone_version(struct version_info *pdest, struct version_info *psrc)
{
	assert(pdest && psrc);
	pdest->file_size = psrc->file_size;
	pdest->flags = psrc->flags;
	if (psrc->version_name)
		pdest->version_name = strdup(psrc->version_name);
	if (psrc->delta_id)
		pdest->delta_id = strdup(psrc->delta_id);
	if (psrc->md5sum)
		pdest->md5sum = strdup(psrc->md5sum);
	if (psrc->delta_url)
		pdest->delta_url = strdup(psrc->delta_url);
	if (psrc->release_note)
		pdest->release_note = strdup(psrc->release_note);
	if (psrc->meta_data)
		pdest->meta_data = strdup(psrc->meta_data);

}


static void free_version_info(struct version_info *pinfo)
{
	FREE(pinfo->version_name);
	FREE(pinfo->delta_id);
	FREE(pinfo->md5sum);
	FREE(pinfo->delta_url);
	FREE(pinfo->release_note);
	FREE(pinfo->meta_data);
	
}

static void dump_version_info(struct version_info *pversion)
{
	if (pversion == NULL)
		return;
	LOGD(" version->version_name=[%s]\n", pversion->version_name);
	LOGD(" version->delta_id=[%s]\n", pversion->delta_id);
	LOGD(" version->md5sum=[%s]\n", pversion->md5sum);
	LOGD(" version->delta_url=[%s]\n", pversion->delta_url);
	LOGD(" version->file_size=[%d]\n", pversion->file_size);
	LOGD(" version->meta_data=[%s]\n", (pversion->meta_data != NULL) ? pversion->meta_data: "NULL");
}

static void save_meta_data_to_file(const char* pmeta_info)
{
	int result = 0;
	int length = 0;
	if(NULL == pmeta_info)
		return;
    if(fota_dm_is_file_exist(FOTA_DM_META_FILE) ){
        fota_dm_remove(FOTA_DM_META_FILE);
    }
	length = strlen(pmeta_info);
	result = fota_dm_write_file(FOTA_DM_META_FILE, pmeta_info, length);
	if(result != 0){
		LOGE("write meta info to file failed");
	}
}
static char* load_meta_data_from_file(void)
{
	int file_size = 0;
	char *pbuf = NULL;
	int result = 0;
	if(fota_dm_is_file_exist(FOTA_DM_META_FILE) != TRUE)
		return NULL;
	file_size = fota_dm_get_file_size(FOTA_DM_META_FILE);
	if(file_size == 0)
		return NULL;
	pbuf = malloc(file_size + 1);
	if(pbuf == NULL){
		LOGE("failed to malloc memory for meta data");
		return NULL;
	}
	memset(pbuf, 0x00, file_size + 1);
	result = fota_dm_read_file(FOTA_DM_META_FILE, pbuf, file_size+1);
	if(result <= 0){
		FREE(pbuf);
		return NULL;
	}
    LOGD("meta data:size=%d, %s\n",file_size,  pbuf);
	return pbuf;
	
}


static struct  version_info * load_version_info(void)
{
	char buf[FOTA_CFG_GET_ITEMLEN] = {0};
	struct version_info *pinfo = NULL;
	pinfo = malloc(sizeof(struct version_info));
	if (pinfo == NULL)
		return pinfo;
	memset(pinfo, 0x00, sizeof(struct version_info));
	
	sc_cfg_get(NV_FOTA_VERSION_NAME, buf, sizeof(buf));
	pinfo->version_name = strdup(buf);
	sc_cfg_get(NV_FOTA_VERSION_DELTA_ID, buf, sizeof(buf));
	pinfo->delta_id = strdup(buf);
	sc_cfg_get(NV_FOTA_VERSION_MD5SUM, buf, sizeof(buf));
	pinfo->md5sum = strdup(buf);
	sc_cfg_get(NV_FOTA_VERSION_DELTA_URL, buf, sizeof(buf));
	pinfo->delta_url = strdup(buf);
	sc_cfg_get(NV_FOTA_PKG_TOTAL_SIZE, buf, sizeof(buf));
	pinfo->file_size = atoi(buf);
    if(pinfo->file_size < 0 || pinfo->file_size > INT_MAX-1){
        pinfo->file_size = 0;
	}
	
	if(fota_dm_vendor_is_zx()){
		pinfo->meta_data = load_meta_data_from_file();
	}
	dump_version_info(pinfo);

	return pinfo;
}


static void save_version_info(struct version_info *info)
{
	char data[32] = {0};
	if (NULL == info)
		return;
	snprintf(data, 32, "%d", info->file_size);
	sc_cfg_set(NV_FOTA_VERSION_NAME, info->version_name);
	sc_cfg_set(NV_FOTA_VERSION_DELTA_ID, info->delta_id);
	sc_cfg_set(NV_FOTA_VERSION_DELTA_URL, info->delta_url);
	sc_cfg_set(NV_FOTA_VERSION_MD5SUM, info->md5sum);
	sc_cfg_set(NV_FOTA_PKG_TOTAL_SIZE, data);
	if(fota_dm_vendor_is_zx() || fota_dm_vendor_is_rs()){//corem0628
		save_meta_data_to_file(info->meta_data);
	}
	//version info need save for  abnormal power off
	sc_cfg_save();
}


static int handle_checkversion_finished(void *data)
{
	LOGD("Enter\n");
	struct version_info *pversion = NULL;
	int mandatory = 0;

	if (data == NULL) {
		__has_new_version = FALSE;
		LOGD("no new version\n");
		return 0;
	} else {
		pversion = (struct version_info *) data;

		__has_new_version = TRUE;
		mandatory = pversion->flags & VERINFO_FLAGS_FORCE_PACKAGE;
		if (mandatory) {
			/* store mandatory state to nv*/
			fota_dm_set_force_package();
			__mandatory = TRUE;
		}
//      clone_version (&__cloned_version, pversion);

		save_version_info(pversion);
		dump_version_info(pversion);
	}
	return 0;
}

static void handle_checkversion_prepare(void)
{
	LOGD("Enter\n");
	if (fota_dm_is_file_exist("/cache/pid-core-dumps"))
		fota_dm_remove("/cache/pid-core-dumps");
	if (fota_dm_is_file_exist(FOTA_PACKAGE_FILE_FAILED))
		fota_dm_remove(FOTA_PACKAGE_FILE_FAILED);
	if (fota_dm_is_file_exist(FOTA_DM_UPGRADE_TEMP))
		fota_dm_remove(FOTA_DM_UPGRADE_TEMP);
}
static void handle_download_finished(void)
{
	LOGD("rename %s->%s\n", DL_TMP_FILENAME, FOTA_PACKAGE_FILE);

	if(0 != fota_dm_sync_file(DL_TMP_FILENAME)){
		LOGE("Sync delta download file (%s) error!\n", DL_TMP_FILENAME); 
	}

	fota_dm_rename_file(DL_TMP_FILENAME, FOTA_PACKAGE_FILE);
}



static void handle_download_translating(void *data)
{
	struct download_info *pdinfo = NULL;
	int download_size = 0;
	if (data) {
		pdinfo = (struct download_info*) data;
		download_size = pdinfo->now_bytes + pdinfo->total_bytes - pdinfo->break_bytes;
		//LOGD("download_size=%d\n",  download_size);
		fota_dm_set_download_progress(download_size);

	}
}
static int download_notifier(struct notifier_block *nb, unsigned long action,
                             void *data)
{
	//LOGD("action:%ld:%s\n", action, notifier_action_to_string(action));
	switch (action) {
	case NOTIFY_DOWNLOAD_PREPARE:
		break;
	case NOTIFY_DOWNLOAD_ABORTED:
		break;
	case NOTIFY_DOWNLOAD_FINISHED:
		handle_download_finished();
		break;
	case NOTIFY_CHECKVERSION_PREPARE:
		handle_checkversion_prepare();
		break;
	case NOTIFY_CHECKVERSION_ABORTED:
		break;
	case NOTIFY_CHECKVERSION_FINISHED:
		handle_checkversion_finished(data);
		break;
	case NOTIFY_DOWNLOAD_TRANSLATING:
		handle_download_translating(data);
		break;
	default:
		break;
	}
	return 0;
}

static int init_dmgr_default_param(dmgr_t dm)
{
	char fota_oem[32] = {0};
	char fota_token[128] = {0};
	char fota_device_type[32] = {0};
	char fota_platform[32] = {0};
	char fota_models[32] = {0};
	char fota_imei[128] = {0};
	char fota_version[128] = {0};
	char fota_url[256] = {0};

	char fota_productid[32] = {0};
	char fota_productsecret[64] = {0};
	char fota_appversion[8] = {0};
	char fota_networktype[16] = {0};
	
    char *suffix = NULL;
	if(fota_dm_vendor_is_rs()){//corem0628
		suffix = "_rs";
	}else if(fota_dm_vendor_is_zx() == TRUE){
        suffix = "_zx";
    }else{
        suffix = "_gs";
    }
	fota_dm_get_fota_oem(fota_oem, sizeof(fota_oem));
	fota_dm_get_fota_token(fota_token , sizeof(fota_token), suffix);
	fota_dm_get_fota_device_type(fota_device_type, sizeof(fota_device_type));
	fota_dm_get_platform(fota_platform, sizeof(fota_platform));
	fota_dm_get_models(fota_models, sizeof(fota_models));
	fota_dm_get_inner_version(fota_version, sizeof(fota_version));
	fota_dm_get_imei(fota_imei, sizeof(fota_imei));

	fota_dm_get_product_id(fota_productid, sizeof(fota_productid));
	fota_dm_get_product_secret(fota_productsecret, sizeof(fota_productsecret));
	fota_dm_get_app_version(fota_appversion, sizeof(fota_appversion));
	fota_dm_get_network_type(fota_networktype, sizeof(fota_networktype));

	

    if( 0 == strcmp(fota_imei, "")){
        LOGE("failed to get imei\n");
        fota_dm_netdog_set_imei_exception(1);
        goto out_err;
    }else{
    	if(fota_dm_vendor_is_rs()){//corem0628
			char imei_tmp[20] = {0};
			memcpy(imei_tmp, fota_imei, sizeof(imei_tmp)-1);//klocwork
			memset(fota_imei, 0, sizeof(fota_imei));
			snprintf(fota_imei, sizeof(fota_imei), "IMEI:%s", imei_tmp);
		}
        fota_dm_netdog_set_imei_exception(0);
    }
    if( 0 == strcmp(fota_version, "")){
        LOGE("failed to get cr_inner_version\n");
        fota_dm_netdog_set_cr_inner_version_exception(1);
        goto out_err;
    }else{
        fota_dm_netdog_set_cr_inner_version_exception(0);
    }

    if(fota_dm_vendor_is_zx() == TRUE ||fota_dm_vendor_is_rs()){//corem0628
		GOTO_OUT_ERR_IF(dmgr_register_deviceinfo(dm, "oem", fota_oem)<0);
		GOTO_OUT_ERR_IF(dmgr_register_deviceinfo(dm, "token", fota_token)<0);
		GOTO_OUT_ERR_IF(dmgr_register_deviceinfo(dm, "deviceType", fota_device_type)<0);
		GOTO_OUT_ERR_IF(dmgr_register_deviceinfo(dm, "platform", fota_platform)<0);
		GOTO_OUT_ERR_IF(dmgr_register_deviceinfo(dm, "models", fota_models)<0);
		GOTO_OUT_ERR_IF(dmgr_register_deviceinfo(dm, "mid", fota_imei)<0);
		GOTO_OUT_ERR_IF(dmgr_register_deviceinfo(dm, "version", fota_version)<0);

    }else{
		GOTO_OUT_ERR_IF(dmgr_register_deviceinfo(dm, "oem", fota_oem)<0);
		GOTO_OUT_ERR_IF(dmgr_register_deviceinfo(dm, "deviceType", fota_device_type)<0);
		GOTO_OUT_ERR_IF(dmgr_register_deviceinfo(dm, "platform", fota_platform)<0);
		GOTO_OUT_ERR_IF(dmgr_register_deviceinfo(dm, "models", fota_models)<0);
		GOTO_OUT_ERR_IF(dmgr_register_deviceinfo(dm, "mid", fota_imei)<0);
		GOTO_OUT_ERR_IF(dmgr_register_deviceinfo(dm, "version", fota_version)<0);

		// add by new libdmgr
		GOTO_OUT_ERR_IF(dmgr_register_deviceinfo(dm, "productId", fota_productid)<0);
		GOTO_OUT_ERR_IF(dmgr_register_deviceinfo(dm, "productSecret", fota_productsecret)<0);
		GOTO_OUT_ERR_IF(dmgr_register_deviceinfo(dm, "appversion", fota_appversion)<0);
		GOTO_OUT_ERR_IF(dmgr_register_deviceinfo(dm, "networkType", fota_networktype)<0); 

    }


	

	
	fota_dm_get_reg_url(fota_url, sizeof(fota_url), suffix);
	if (strlen(fota_url) > 0){
		GOTO_OUT_ERR_IF(dmgr_register_serverinfo(dm, SERI_REG_URL, fota_url)<0);
	}
	fota_dm_get_chk_url(fota_url, sizeof(fota_url), suffix);
	if (strlen(fota_url) > 0){
		GOTO_OUT_ERR_IF(dmgr_register_serverinfo(dm, SERI_CHK_URL, fota_url)<0);
	}

	fota_dm_get_dl_url(fota_url, sizeof(fota_url), suffix);
	if (strlen(fota_url) > 0){
		GOTO_OUT_ERR_IF(dmgr_register_serverinfo(dm, SERI_DL_URL, fota_url)<0);
	}

	fota_dm_get_report_dlr_url(fota_url, sizeof(fota_url), suffix);
	if (strlen(fota_url) > 0){
		GOTO_OUT_ERR_IF(dmgr_register_serverinfo(dm, SERI_REPORT_DLR_URL, fota_url)<0);
	}
	fota_dm_get_report_upgr_url(fota_url, sizeof(fota_url), suffix);
	if (strlen(fota_url) > 0){
		GOTO_OUT_ERR_IF(dmgr_register_serverinfo(dm, SERI_REPORT_UPGR_URL, fota_url)<0);
	}
	fota_dm_get_report_sales_url(fota_url, sizeof(fota_url), suffix);
	if (strlen(fota_url) > 0){
		GOTO_OUT_ERR_IF(dmgr_register_serverinfo(dm, SERI_REPORT_SALES_URL, fota_url)<0);
	}
	GOTO_OUT_ERR_IF(dmgr_register_device_to_server(dm)<0);
	GOTO_OUT_ERR_IF(dmgr_alloc_notifier(dm, NOTIFY_ID_DOWNLOAD, download_notifier, 0, 0)<0);
	GOTO_OUT_ERR_IF(dmgr_alloc_notifier(dm, NOTIFY_ID_CHECK_VERSION, download_notifier, 0, 0)<0);
	return 0;

out_err:
	return -1;
}

static BOOL fota_dm_need_report_update_result(void)
{
	int result = -1;
	char buffer[128] = {0};
	LOGD("fota_dm_need_report_update_result begin\n");
	if(fota_dm_vendor_is_zx() != TRUE && !fota_dm_vendor_is_rs())//corem0628
		return FALSE;
	result = fota_dm_get_nv_int(NV_FOTA_PKG_DOWNLOADED);
	LOGD("fota_dm_need_report_update_result NV_FOTA_PKG_DOWNLOADED is %d\n", result);
	if (result != 1)
		return FALSE;
	fota_dm_get_nv_string(NV_FOTA_UPGRADE_RESULT_INTERNAL, buffer, sizeof(buffer));
	LOGD("fota_dm_need_report_update_result NV_FOTA_UPGRADE_RESULT_INTERNAL is %s\n", buffer);
	if((strcmp(buffer, UPDATE_SUCCESS) !=0) && (strcmp(buffer, UPDATE_FAIL) !=0) ){
		return  FALSE;
	}
	return TRUE;
}
static int fota_dm_report_update_result_to_server(dmgr_t dm_id)
{
	char update_result[128] = {0};
	char  upgraded_desc[32] = {0};
	struct version_info info;
	int ret = -1;
	memset(&info, 0x00, sizeof(struct version_info));
	LOGD("report update_result\n");
	fota_dm_get_nv_string(NV_FOTA_UPGRADE_RESULT_INTERNAL, update_result, sizeof(update_result));
	if(strcmp(update_result, UPDATE_SUCCESS)== 0 ){
		if(fota_dm_vendor_is_rs()){//corem0628
			snprintf(upgraded_desc, sizeof(upgraded_desc), "%s", "OK");
		}else{
			snprintf(upgraded_desc, sizeof(upgraded_desc), "%d",1);
		}
	}else{
		if(fota_dm_vendor_is_rs()){
			snprintf(upgraded_desc, sizeof(upgraded_desc), "%s", "FAIL");
		}else{
			snprintf(upgraded_desc, sizeof(upgraded_desc), "%d",0);
		}
	}

	info.meta_data =load_meta_data_from_file();
	ret = dmgr_report_upgraded_version(dm_id, &info, upgraded_desc);

	out:
	free_version_info(&info);
	return ret;
}
static void fota_dm_set_pkg_downloaded(int value)
{
	fota_dm_set_nv_int(NV_FOTA_PKG_DOWNLOADED, value);
    sc_cfg_save();
}
static int fota_dm_should_retry(int dl_ret)
{
	/*絼µʧܣҪ*/
	if( (dl_ret==E_IO_TIMEDOUT) || (dl_ret==E_COULDNT_RESOLVE_HOST) || 
		(dl_ret == E_COULDNT_CONNECT) || (dl_ret==E_HTTP_POST_ERROR) ||
		(dl_ret==E_AGAIN) || (dl_ret==E_SEND_ERROR) ||
		(dl_ret == E_RECV_ERROR) || (dl_ret==E_NO_CONNECTION_AVAILABLE)|| 
		(dl_ret==E_INTERNAL_ERROR) )
	{
		return 1;
	}
	return 0;
}

int fota_dm_check_new_version(void)
{

	int check_ret = -1;
	int ret = -1;
	dmgr_t dm = -1;
	struct policy_info policy = {0};
	int i = 0;

	LOGD("Check Network status\n");
	check_ret = fota_dm_wait_network_ready(3, 60);
	if (check_ret != TRUE) {
		LOGE("Network not ready\n");
		ret = -1;
		fota_dm_netdog_set_network_exception(1);
		goto out_err;
	}
	LOGD("Network is ready\n");
	fota_dm_netdog_set_network_exception(0);

	fota_dm_wake_lock(FOTA_DM_WAKE_LOCK);
	init_policy_info(&policy);
	GOTO_OUT_ERR_IF((dm = dmgr_alloc(&policy, 1, LOGGER_STDIO, stdout))<0);
	GOTO_OUT_ERR_IF(init_dmgr_default_param(dm)<0);
	__has_new_version = FALSE;
	
	if(fota_dm_need_report_update_result() == TRUE){
		LOGD("fota_dm_need_report_update_result is true\n");
		if(fota_dm_report_update_result_to_server(dm) == 0){
			fota_dm_set_pkg_downloaded(0);
			fota_dm_set_nv_string(NV_FOTA_UPGRADE_RESULT_INTERNAL, "");
		}
	}
	for (i = 0; i < DOWNLOAD_RETRY_TIMES; i++) {
		LOGD("dmgr_check_version, retry=%d\n", i);
		check_ret = DMGR_ERRNO(dmgr_check_version(dm));
		LOGD("dmgr_check_version, retry=%d,  ret=%d(%s)\n", i, check_ret,
		     dmgr_error_to_string(check_ret));
		if (!fota_dm_should_retry(check_ret)){
			break;
			}
	}

	fota_dm_netdog_set_check_version_result(check_ret);
	ret = (((check_ret == E_OK)
	        || (check_ret == E_UNCONFIGURED_VERSION)) ?  0 : -1);
	
out_err:
	fota_dm_wake_unlock(FOTA_DM_WAKE_LOCK);
	if (dm != -1)
		dmgr_free(dm);
	return ret;
}

#ifndef DOWNLOAD_FILE_SEEK_END//corem0628
#define DOWNLOAD_FILE_SEEK_END ((long)-1)
#endif

int  fota_dm_download_version(void)
{
	char save_path[256] = {0};
	int file_seek = 0;
	int ret = -1;
	int dl_ret = 0;
	int i = 0;
	struct version_info *pversion = NULL;
	dmgr_t dm = -1;
	struct policy_info policy = {0};
	strncpy(save_path, DL_TMP_FILENAME, strlen(DL_TMP_FILENAME));

	LOGD("Check Network status\n");
	dl_ret = fota_dm_wait_network_ready(3, 60);
	if (dl_ret != TRUE) {
		ret = -1;
		fota_dm_netdog_set_network_exception(1);
		LOGE("Network Not Ready\n");
		goto out_err;
	}
	LOGD("Network is ready\n");
	fota_dm_netdog_set_network_exception(0);


	fota_dm_wake_lock(FOTA_DM_WAKE_LOCK);
	init_policy_info(&policy);
	GOTO_OUT_ERR_IF((dm = dmgr_alloc(&policy, 1, LOGGER_STDIO, stdout))<0);
	GOTO_OUT_ERR_IF(init_dmgr_default_param(dm)<0);

	pversion = load_version_info();
	if (pversion == NULL) {
		ret = -1;
		goto out_err;
	}

	for (i = 0; i < DOWNLOAD_RETRY_TIMES; i++) {
        if(fota_dm_vendor_is_zx() == TRUE ||fota_dm_vendor_is_rs()){//corem0628
			if (fota_dm_is_file_exist(save_path))
				file_seek = fota_dm_get_file_size(save_path) - 8192;
			if (file_seek < 0)
				file_seek = 0;
			LOGD("dmgr_download_version, retry=%d, file_size=%d\n",  i, file_seek);
			
			
			dl_ret = DMGR_ERRNO(dmgr_download_version(dm, pversion, save_path,
								file_seek, 0, -1));

        }else{
			dl_ret = DMGR_ERRNO(dmgr_download_version(dm, pversion, save_path,
								DOWNLOAD_FILE_SEEK_END, 0, -1));

        }		

		LOGD("dmgr_download_version, retry=%d, ret=%d(%s)\n",  i, dl_ret,
		     dmgr_error_to_string(dl_ret));
		
		if (fota_dm_should_retry(dl_ret)){
			continue;
		}else if((dl_ret==E_VERIFIED_FAILED) || (dl_ret==E_RANGE_ERROR) ||
			(dl_ret==E_BAD_DOWNLOAD_RESUME) ){
			/*У󣬶ϵҪɾصʱļ*/
			fota_dm_remove(DL_TMP_FILENAME);
			fota_dm_set_download_progress(0);
			break;
		}else{
			/**/
			break;
		}
	}
	
	fota_dm_netdog_set_download_version_result(dl_ret);
	ret = ((dl_ret == E_OK) ?  0 : -1);
	if(dl_ret == E_OK){
		fota_dm_set_pkg_downloaded(1);
	}
	
out_err:
	fota_dm_wake_unlock(FOTA_DM_WAKE_LOCK);
	if (dm != -1)
		dmgr_free(dm);
	if (pversion) {
		free_version_info(pversion);
		FREE(pversion);
	}
	return ret;
}

int fota_dm_has_new_version(void)
{
	return __has_new_version;
}

int fota_dm_has_mandatory_version(void)
{
	return __has_new_version && __mandatory;
}

int fota_dm_get_pkg_total_size(void)
{
	if (!__has_new_version)
		return -1;
	return fota_dm_get_nv_int(NV_FOTA_PKG_TOTAL_SIZE);
}
