/* ===========================================================================
** Project:              Mobile Station (MS)
** Block:                Global SDL definitions
** Process:              -
** ===========================================================================
** This file decribes the datatypes used between Layer 1 and the RR Sublayer
** Processes which are also used internally by the RR Sublayer Processes. 
** Furthermore all function prototypes used between these processses are 
** defined in this file, too.
** ===========================================================================
** History:
**
** Date      Author    Comment
** ------------------------------------------------------------------------
** 22-03-2006 RCI  SMS00215146 : Removal compiler and lint warnings
** 23.08.2005 jst  SMS00098515: moved RLCD_SOFT_BLOCK_SIZE from RLC into 
**                 interface file
** 27.08.2004 mes  SMS00037198: SGOLD2: Change address range of IR memory
** 23.06.2004 jst  CN2-12788: Added new parameter T_PFI to function
**                 rlc_check_tbf_est_posibility ()
** 03.11.2004 tor  CN200009854: added grr_si_in_tbf
** 04.12.2002 mes  added element padding to T_EDATA_SOFT_UNIT.
** 31.10.2002 pma  added #ifndef RLC_ONLY to exclude mac specific interface.
** 25.10.2002 pli  Created.
** ========================================================================
*/

#if !defined (_MS_L1_RR_SUBLAYER_H)
#define _MS_L1_RR_SUBLAYER_H
/*
** =========================================================================
** Includes
** =========================================================================
*/

#include "ms_config.h"

/*
** =========================================================================
** Public Definitions
** =========================================================================
*/

/*
** -------------------------------------------------------------------------
** Hardware Capabilities:
** -------------------------------------------------------------------------
** Max. number of Soft Value Blocks fit into one Soft Value Unit
*/
#define RLCD_MAX_NOF_SOFT_VALUE_BLOCKS 4
#if (RLCD_MAX_NOF_SOFT_VALUE_BLOCKS <= 1)
#error "This makes not really sence!"
#endif
/*
** Addressrange for storing Soft Value Blocks
** The address range 0 ...3327 (2*208*4 = 1664 words) are
** used internally.
*/
#define RLCD_LOW_BYTE_ADDR   3328
#if defined (SGOLD2)
#define RLCD_HIGH_BYTE_ADDR  73727
#else /* SGOLD */
#define RLCD_HIGH_BYTE_ADDR  65311
#endif

/*
** Size of one Soft Block
*/
#define RLCD_SOFT_BLOCK_SIZE       416

/*
** Memory size for storing Soft Value Blocks
*/
#define RLCD_SOFT_MEM_SIZE ((RLCD_HIGH_BYTE_ADDR - RLCD_LOW_BYTE_ADDR) + 1)

/*
** Memory organisation for storing Soft Value Blocks
** Number of bytes for the smallest storage unit.
*/
#define RLCD_SOFT_MEM_UNIT_SIZE 2

/*
** Copying data within the IR Memory.
** The performance needed to copy 1 RLC Data Block (e.g. MCS-5)
** within the IR Memory is defined as:
**
** Perf = RLCD_STATIC_SOFT_COPY + 416 * RLCD_SOFT_COPY_PERF
**
** Example:
** RLCD_STATIC_SOFT_COPY = 10, RLCD_SOFT_COPY_PERF = 1
**
** => Perf = 10 + 416 * 1 = 426.
**
** The summ of the performance needed for all data blocks
** copied shall not exceed the value of RLCD_MAX_SOFT_COPY.
*/

/*
** Performance Index for copying Soft Blocks within the
** IR Memory
*/
#define RLCD_MAX_SOFT_COPY 10000

/*
** Static performance overhead for copying Soft Blocks
** within the IR Memory.
*/
#define RLCD_STATIC_SOFT_COPY_PERF 10

/*
** Performance needed to copy 1 Memory Unit within the
** IR Memory
*/
#define RLCD_SOFT_COPY_PERF 1


/*
** =========================================================================
** Public Data Types
** =========================================================================
*/


/*
** Address handle to the EDATA Soft Block storage area.
** This area is normally located within the HWAL or the DSP.
*/
typedef DWORD T_EDATA_SOFT_BLOCK_ADDR;

/*
** Description of an EDATA Soft Block, normally defined in file ms.h. 
*/
typedef struct
{
  T_BLOCK_FORMAT          dec_type;   /* RLC Decoder Type */
  T_PUNCTURING_SCHEME     punc_scheme;/* Puncturing Scheme */
  /* 
  ** Storage address of the Soft Values, for MCS-5 ...6 blocks, this
  ** address points to the first 208 Soft Values of the block.
  */
  T_EDATA_SOFT_BLOCK_ADDR addr_1;
  /* 
  ** Storage address of the second 208 Soft Values for MCS-5 ...6
  ** blocks. The value of this element is "Don't Care" for all
  ** other blocks (MCS-1 ...4, MCS-7 ...9).
  */
  T_EDATA_SOFT_BLOCK_ADDR addr_2;
} T_EDATA_SOFT_BLOCK;

/*
** Description of a EDATA Soft Unit, holds references to all
** reveived Soft Blocks belonging to one RLC Block.
** Normally defined in file ms.h.
*/
typedef struct
{
  /* 
  ** Block Sequence Number, not needed by the HWAL but part of the RLC
  ** data structure.
  */
  WORD              block_seq_no;
  /* 
  ** Pointer to the  description of the new received
  ** EDATA Soft Block. The parameter 'addr' point to
  ** the area where the Soft Values are to storage if the
  ** decoding fails.
  */
  T_EDATA_SOFT_BLOCK* p_new_block;
  /*
  ** Reference to the location where the Soft Values of the new received
  ** RLC Data Block/EDATA Soft Block are stored.
  */
  T_EDATA_SOFT_BLOCK_ADDR new_block_addr;
  /* 
  ** Array of pointers to up to 3 earlier received Soft Blocks
  ** The parameter 'addr' points to the area where the Soft
  ** Values are stored.
  */
  T_EDATA_SOFT_BLOCK* p_old_blocks[RLCD_MAX_NOF_SOFT_VALUE_BLOCKS-1];
  /*
  ** Indication if padding is used (1) or not (0).
  ** Padding is used if MCS-8 RLC Data Blocks are retransmitted by using
  ** MCS-6 or MCS-3.
  */
  BYTE padding;
} T_EDATA_SOFT_UNIT;

/*
** Description of the Soft Unit List returned from
** the MAC function mac_receive_edata_blocks().
** Normally defined in file mac_receive_edata_blocks.h.
*/
typedef struct
{
  T_EDATA_SOFT_UNIT* p_soft_units[(MAX_RX_TS*2)];
} T_EDATA_SOFT_UNIT_LIST;


/*
** RLC Header:
** Contains all informations about a received MCS-1 to MCS-9 Radio Block
** needed by MAC and RLC.
*/
typedef struct
{
  T_RLC_HEADER_TYPE       type;
  T_TIMESLOT              ts;
  T_RXQUAL                rxqual;   /* obsolete */
  DWORD                   mean_bep;
  SHORT                  cv_bep;
  T_HEADER_PAYLOAD        *p_payload;
  /*
  ** Reference the Soft Values of the received RLC Data Block.
  ** If HEADER_TYPE_1 detected, this element contains the reference to the
  ** Soft Values of the first RLC Data Block in the received Radio Block.
  */
  T_EDATA_SOFT_BLOCK_ADDR addr_1;
  /*
  ** Reference to the Soft Values of the second RLC Data Block in the
  ** received Radio Block. The value of this element is undefined if
  ** HEADER_TYPE_1 or HEADER_TYPE_2 is detected.
  */
  T_EDATA_SOFT_BLOCK_ADDR addr_2;
} T_RLC_HEADER;


/*
** RLC Header List:
** List of MCS-1 to MCS-9 RLC headers.
**
** Elements:
** - nof:
**     Number of valid RLC Header in the list. The first nof number of
**     T_RLC_HEADER elements contain valid entries. The other elements
**     shall be ignored.
**     Range of value: 0 ... MAX_RX_TS.
**
** - header:
**     Array of MAX_RX_TS T_RLC_HEADER structures.
*/
typedef struct
{
  BYTE        nof;
  T_RLC_HEADER header[MAX_RX_TS];
} T_RLC_HEADER_LIST;


/*
** =========================================================================
** Public Function Interface
** =========================================================================
*/

#if !defined (TOOLS_TRACE_TOOL_PARSER)

/*
** FUNCTION: grr_si_in_tbf
**
** FUNCTION DESCRIPTION:
**   this function is called by l1 after AT+XL1SET="si_in_tbf ......"
** ARGUMENTS:
**   string after "si_in_tbf ......"
**
*/
VOID grr_si_in_tbf(const CHAR* str_ptr);


/*
** FUNCTION: rlc_fbi_in_unack_mode_received()
**
** FUNCTION DESCRIPTION:
**   This function returns True (!= 0), if a data block with FBI equal to 1
**   is received in RLC unacknowledged mode.
**   Otherwise, False is returned.
**
** ARGUMENTS:
** none.
**
*/
BYTE rlc_fbi_in_unack_mode_received( BYTE bStackIndex );/*0:stack_1,  1:stack_2*/


/*
** FUNCTION: rlc_process_downlink_edata_blocks()
**
** FUNCTION DESCRIPTION:
** This function is called from the MAC on interrupt level.
** RLC decodes first the header list and determine the block
** sequence number, coding scheme and puncturing scheme. 
** Afterwards rlc provide all information in p_soft_unit_list
** which it is needed by L1 to decode the blocks. Moreover in this
** task IR Memory handling will be considered. RLC will tell    
** the decoder which blocks can be decoded jointly.
** 
** PARAMETER
** - p_header_list (T_RLC_HEADER_LIST):
**     Pointer to up to 8 Header IE (T_RLC_HEADER).
**     The T_RLC_HEADER elements are filled by the Layer 1 in an ordered but
**     not timeslot based way. When an element is not used, the Layer 1 shall
**     set the type to HEADER_UNDEFINED. Upon detection of an element with
**     type HEADER_UNDEFINED, MAC and RLC shall not perform any activity for
**     this and any of the following elements (higher index).
**
** - p_soft_unit_list (T_EDATA_SOFT_UNIT_LIST):
**     Pointer to up to 16 Soft Unit IE (T_EDATA_SOFT_UNIT_LIST).
**
** RETURN VALUE:
*+ none
** RESULT:
** EDATA Soft Unit List, pointed by the parameter p_soft_unit_list, filled
** with pointers to EDATA Soft Units, each containing all earlier received
** EDATA Soft Blocks of the RLC Data Blocks identified by the Header List
** (parameter p_header_list).
**
*/
VOID rlc_process_downlink_edata_blocks
(BYTE bStackIndex,  /*0:stack_1,  1:stack_2*/
 /* Header List */
 T_RLC_HEADER_LIST *p_header_list,

 /* EDATA Soft Unit List */
 T_EDATA_SOFT_UNIT_LIST *p_soft_unit_list
);



/*
** FUNCTION: rlc_check_tbf_est_possibility()
**
** FUNCTION DESCRIPTION:
** This function is called from the MAC on interrupt level.
** Purpose is to check if RLC has some data blocks to transmit.
** If yes then MAC cat send indication to the network that ms want to 
** establish uplink TBF.
** 
** PARAMETER
*+ none
**
** RETURN VALUE:
** - req_peak_throughput_ptr
**     Pointer to peak thougput class value of remaining data blocks.
**
** - req_radio_prio_ptr
**     Pointer to radio priority of remaining data blocks.
**
** - req_rlc_mode_ptr
**     Pointer to rlc_mode value.
**
** - req_llc_type_ptr
**     Pointer to llc type value.
**
** - req_rlc_octet_count_ptr
**     Pointer to value representing number of octets to send by RLC.
**
** - req_pfi_ptr
**     Pointer to current used PFI value.
**
** RESULT:
** SDL_True when RLC has some data blocks to send.
** SDL_False when RLC doe's not have any data blocks to send.
**
*/
GLOBAL SDL_Boolean rlc_check_tbf_est_possibility
(  BYTE              bStackIndex,  /*0:stack_1,  1:stack_2*/
   BYTE              * req_peak_throughput_ptr,
   T_RADIO_PRIORITY  * req_radio_prio_ptr,
   T_RLC_MODE        * req_rlc_mode_ptr,
   T_LLC_TYPE        * req_llc_type_ptr,
   T_RLC_OCTET_COUNT * req_rlc_octet_count_ptr,
   T_PFI             * req_pfi_ptr
);

/*
** FUNCTION: rlc_tbf_est_ind()
**
** FUNCTION DESCRIPTION:
** This function is called from the MAC on interrupt level.
** Purpose is to indicate RLC that information about new TBF
** establishment of uplink TBF was successfully sent by L1.
** 
** PARAMETER
** none
**
** RETURN VALUE:
*+ none
**
*/
GLOBAL VOID rlc_tbf_est_ind
(BYTE bStackIndex);/*0:stack_1,  1:stack_2*/

#endif /*#if !defined (TOOLS_TRACE_TOOL_PARSER)*/
#endif /*#if !defined (_MS_L1_RR_SUBLAYER_H)*/

/* === end of file ms_l1_rr_sublayer.h === */
