/**
 *       Copyright (C) Danish Wireless Design A/S. All rights reserved.
 *
 * This document contains proprietary information belonging to Danish Wireless
 * Design A/S. Passing on and copying of this document, use and communication
 * of its contents is not permitted without prior written authorisation.
 *
 * Description:
 *   State machine for the voice memo service (vms). Everything concerning the vms is
 *              controlled from this code.
 *
 * Revision Information:
 *   File name: \dwddrv\AUD\src\aud_vms_rec.c
 *   Version: \main\62
 *   Date: 2007-08-09 14:14:02
 */
#if defined (VMS_PRESENT)
//#include "string.h"
//#include "icu_funct.h"
//#include "scttypes.h"
#include "bastypes.h"
//#include "dsp.h"                /* for DSP command types */
#include "aud_drv.h"
#include "aud_data.h"           /* data describing the paths */
#include "aud_com.h"
//#include "dsp.h"

/*changed by wangjun for NV change.[20090214]*/
#include "aud_nv.h"  

//#include "atctstif.h"
#include "aud_path_control.h"
#include "aud_vms_rec.h"
//#include "trap.h"
#include "aud_ext_intf.h"
#include "drvs_teak.h"

/*---------------------------------------------*/
/* Global data.                                */
/*---------------------------------------------*/
aud_glob_vms_rec_type aud_glob_vms_rec;
T_Voice_RecordCallBack aud_glob_recordCallBack;
T_Voice_RateCallBack aud_glob_byteCallBack;

/*---------------------------------------------*/
/* Internal data.                              */
/*---------------------------------------------*/
static vms_rec_data_type vms_rec_data[2];
static vms_rec_ctrl_type vms_rec;

/*---------------------------------------------*/
/* Internal functions.                         */
/*---------------------------------------------*/
static void vms_rec_start(aud_dsp_format_enum format);
static void stop_voice_memo_record(void);
static void copy_memo_buffer(void);
static void vms_rec_init(void);
static UINT32 vms_set_fileposition(void);
static void DspStartRecordVm(aud_dsp_format_enum codec_type, UINT16 amr_sample_rate, UINT16 dtx);
static void DspStopRecordVm(void);
static void vm_rec_send_response_signal( SINT8 rc);
static void vms_write_ssal_callback(SINT16 result
#ifdef FFS_EXTENDED_CALLBACK
                             ,void *input
#endif
                            );
static void enable_vms_rec_audio_path(void);
static void disable_vms_rec_audio_path(void);
static void find_nof_bytes_per_frame_record(UINT8 sample_rate);
static void AUD_set_internal_vms_rec_state(UINT8 new_state);


/****************************************************************************************
* Function:... AudVmsRecSM
* Parameters:. event: The reason why the function was called
* Description: State machine for the voice memo service control. 4 states are defined: idle,
****************************************************************************************/
BOOL AudVmsRecSM(aud_event_enum event, void* ptrData)
{
    const aud_resource_enum me = aud_resource_record_vm;
    disp_to_SM_data_type *SM_data = (disp_to_SM_data_type*)ptrData; /* contains the structure passed from the dispatcher */
    BOOL func_ret = TRUE;

#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_sm_vm_recording, 11, 0,               // LLT params 
                       'm', 0x13,                                               // LIS params 
                       aud_log_lis_raw_data, aud_log_lis_display_data, aud_log_lis_llt_trace, // LIS params 
                       vms_rec.aud_vms_rec_state,
                       (UINT32) aud_get_SM_state(me),
                       (UINT32) event,
                       (UINT32) SM_data->parm1,
                       (UINT32) SM_data->parm2,
                       (UINT32) SM_data->parm3,
                       (UINT32) SM_data->parm4,
                       (UINT32) SM_data->parm5,
                       (UINT32) SM_data->parm6,
                       (UINT32) SM_data->parm7,
                       (UINT32) E_audio_llt_type_id_0 );
#endif

    switch (aud_get_SM_state(me)) {
        /*-------------------------------------------------------------------*/
        /*                           Idle state                              */
        /*-------------------------------------------------------------------*/
        case S_idle:
            switch (event) {
                case E_vms_start_voice_recording:
                    aud_disable_standby(me);
                    aud_glob_vms_rec.sender          = (*SM_data).sender;
                    aud_glob_vms_rec.vm_mode         = (aud_vm_mode_enum) SM_data->parm1;
                    aud_glob_vms_rec.media_type      = (aud_media_enum) ((*SM_data).parm2);
                    aud_glob_vms_rec.format          = (aud_dsp_format_enum)((*SM_data).parm3);
                    aud_glob_vms_rec.rate            = (UINT8) ((*SM_data).parm4);
                   /*[begin] add call back function by wangjun*/
                    aud_glob_recordCallBack = (T_Voice_RecordCallBack)((*SM_data).parm8);
                    aud_glob_byteCallBack = (T_Voice_RateCallBack)((*SM_data).parm5);
                   /*[end]*/
                    aud_glob_vms_rec.ssal_media_type = aud_media_type_to_ssal_type(aud_glob_vms_rec.media_type);
                    aud_glob_vms_rec.start_offset    = (UINT32) ((*SM_data).parm6);
                    switch (aud_glob_vms_rec.media_type) {
                        case aud_media_ram:
                            aud_glob_vms_rec.fp_voice_memo_putpacket = &voice_memo_putpacket_in_ram;
                            break;
                        case aud_media_mmf:
                        case aud_media_mmf_test:
                            aud_glob_vms_rec.fp_voice_memo_putpacket = &voice_memo_putpacket_in_mmf;
                            break;
                        case aud_media_ffs:
                        case aud_media_mmc:
#ifdef  AUD_NANDDEVICE_SUPPORT                        	
                        case aud_media_fs_cdrive:
                        case aud_media_fs_fdrive:			
#endif                        	
                            aud_glob_vms_rec.fp_voice_memo_putpacket = &voice_memo_putpacket_in_ffs;
                            break;
                    }
                    vms_rec_init();
                    break;

                case E_vms_rec_set_putbuffer_function:
                    aud_glob_vms_rec.put_buffer_func      = (ptr_GetAudioBufferFunction )(disp_to_SM_data_type*) SM_data->parm7;
                    aud_glob_vms_rec.mmf_user_data        = (UINT16 *)(disp_to_SM_data_type*) SM_data->parm8;
                    vm_rec_send_response_signal( aud_rc_ok);
                    break;
                case E_vms_release_stop_voice_recording:
                case E_vms_stop_voice_recording:
                	  break;
                case E_timer_vms_rec_timeout:
                    AUD_ERROR_REPORTING(aud_error_grade2, TRAP_AUD_ERROR, "Unhandled event: %d in VM Recording SM in Idle state ", event)
                    break;
                default:
                    func_ret = FALSE;
                    break;
            }
            break; /*S_idle */

        case S_active:
            switch (vms_rec.aud_vms_rec_state) {
                /*-------------------------------------------------------------------*/
                /*                       delay before recording.                     */
                /*-------------------------------------------------------------------*/
                case S_aud_vms_delay_before_recording:
                    switch (event) {
                        case E_vms_start_voice_recording:
                            aud_send_response_signal(aud_glob_vms_rec.sender, aud_resource_record_vm, aud_rc_request_error,0, 0, 0, 0, NULL, NULL);
                            break;
                        case E_vms_release_stop_voice_recording:
                        case E_vms_stop_voice_recording:
                            vms_rec.aud_vms_rec_state =   S_aud_vms_rec_idle;
                            stop_voice_memo_record();
                            vm_rec_send_response_signal( aud_rc_recording_finish);
                            InitVmsRec();
                            /* Allow general power down */
                            aud_enable_standby(me);
                            break;

                        case E_timer_vms_rec_timeout:
                            find_nof_bytes_per_frame_record(aud_glob_vms_rec.rate<<3);
                            AUD_set_internal_vms_rec_state(S_aud_vms_wait_for_rec);
                            aud_send_mail(E_timer_vms_rec_timeout, 0, 0, 0, 0, 0, 0, 0, 0, 0,NULL, NULL);
                            break;

                        case E_vms_rec_set_putbuffer_function:
                            AUD_ERROR_REPORTING(aud_error_grade2, TRAP_AUD_ERROR, "Unhandled event: %d in VM Recording SM in %d state ", event,S_aud_vms_delay_before_recording)
                            break;
                        default:
                            func_ret = FALSE;
                            break;
                    }
                    break;/*S_aud_vms_delay_before_recording*/

                /*-------------------------------------------------------------------*/
                /*                           wait for rec.                           */
                /*-------------------------------------------------------------------*/
                case S_aud_vms_wait_for_rec:
                    switch (event) {
                        case E_vms_release_stop_voice_recording:
                        case E_vms_stop_voice_recording:
                            stop_voice_memo_record();
                            vm_rec_send_response_signal(aud_rc_recording_finish);
                            vms_rec.record_to_buffer = FALSE;
                            InitVmsRec();
                            aud_enable_standby(me);
                            break;

                        case E_timer_vms_rec_timeout:
                            vms_rec_start(aud_glob_vms_rec.format);
                            break;

                        case E_vms_start_voice_recording:
                            aud_send_response_signal(aud_glob_vms_rec.sender, aud_resource_record_vm, aud_rc_request_error,0, 0, 0, 0, NULL, NULL);
                            break;

                        case E_vms_rec_set_putbuffer_function:
                            AUD_ERROR_REPORTING(aud_error_grade2, TRAP_AUD_ERROR, "Unhandled event: %d in VM Recording SM in %d state ", event,S_aud_vms_wait_for_rec)
                            break;
                        default:
                            func_ret = FALSE;
                            break;
                    }
                    break; /*S_aud_vms_wait_for_rec*/

                /*-------------------------------------------------------------------*/
                /*                           Recording state                         */
                /*-------------------------------------------------------------------*/
                case S_aud_vms_recording:
                    switch (event) {
                        case E_vms_start_voice_recording:
                            aud_send_response_signal(aud_glob_vms_rec.sender, aud_resource_record_vm, aud_rc_request_error,0, 0, 0, 0, NULL, NULL);
                            break;
                        case E_vms_release_stop_voice_recording:
                        case E_vms_stop_voice_recording:
                            switch (aud_glob_vms_rec.media_type) {
                                case aud_media_ram:
                                case aud_media_mmf:
                                    vms_rec.aud_vms_rec_state =   S_aud_vms_rec_idle;
                                    stop_voice_memo_record();
                                    vm_rec_send_response_signal(aud_rc_recording_finish);
                                    InitVmsRec();
                                    aud_enable_standby(me);
                                    break;

                                case aud_media_ffs:
                                case aud_media_mmc:
#ifdef  AUD_NANDDEVICE_SUPPORT                                	
                                case aud_media_fs_cdrive:
                                case aud_media_fs_fdrive:		                                	
#endif                                 	
                                    AUD_set_internal_vms_rec_state(S_aud_vms_finish_recording);
                                    vms_rec.data_outstanding = TRUE;
                                    //FFS replacement
                                    vms_rec.ffs_result =Aud_SSAL_write((aud_ssal_media_type)aud_glob_vms_rec.ssal_media_type,
                                                                       (aud_ssal_user_type) AUD_SSAL_VR_WRITE,
                                                                       (void*)aud_glob_vms_rec.file_handle,
                                                                       (UINT8*) vms_rec_data[vms_rec.array_number].data,
                                                                       vms_rec.recording_frame,
                                                                       &vms_write_ssal_callback
#ifdef FFS_EXTENDED_CALLBACK
                                                                       ,NULL
#endif
                                                                      );
                                    if (vms_rec.ffs_result != AUD_FFS_SUCCESS && vms_rec.ffs_result != AUD_FS_SUCCESS) {
                                        stop_voice_memo_record();
                                        vm_rec_send_response_signal( aud_rc_storage_problems);
                                        InitVmsRec();
                                        aud_enable_standby(me);
                                    }
                                    break;
                            }
                            break; /*E_vms_stop_voice_recording*/
                        case E_vms_rec_set_putbuffer_function:
                        case E_timer_vms_rec_timeout:
                           	AUD_ERROR_REPORTING(aud_error_grade2, TRAP_AUD_ERROR, "Unhandled event: %d in VM Recording SM in %d state ", event,S_aud_vms_recording)
                            break;

                        default:
                            func_ret = FALSE;
                            break;
                    }
                    break;/*S_aud_vms_recording*/
                /*-------------------------------------------------------------------*/
                /*                       Finish recording state                      */
                /*-------------------------------------------------------------------*/
                case S_aud_vms_finish_recording:
                    switch (event) {
                        case E_vms_start_voice_recording:
                            aud_send_response_signal(aud_glob_vms_rec.sender, aud_resource_record_vm, aud_rc_request_error,0, 0, 0, 0, NULL, NULL);
                            break;
                        case E_timer_vms_rec_timeout:
                            if (!vms_rec.data_outstanding) {
                                stop_voice_memo_record();
                                vm_rec_send_response_signal( aud_rc_recording_finish);
                                InitVmsRec();
                                aud_enable_standby(me);
                            } else {
                                if (vms_rec.aud_vms_rec_error) {
                                    stop_voice_memo_record();
                                    vm_rec_send_response_signal( aud_rc_storage_problems);
                                    InitVmsRec();
                                    aud_enable_standby(me);
                                }
                            }
                            break;
                        case E_vms_release_stop_voice_recording:
                        case E_vms_stop_voice_recording:
                        case E_vms_rec_set_putbuffer_function:
                            AUD_ERROR_REPORTING(aud_error_grade2, TRAP_AUD_ERROR, "Unhandled event: %d in VM Recording SM in %d state", event,S_aud_vms_finish_recording)
                            break;

                        default:
                            func_ret = FALSE;
                            break;
                    }
                    break;
            }/*switch(vms_rec.aud_vms_rec_state)*/
            break;
    }/*switch(aud_get_SM_state(me))*/
    return(func_ret);
}/*AudVmsRecSM */

/****************************************************************************************
* Function:... stop_voice_memo_record
* Description: This function stop recording
****************************************************************************************/
static void stop_voice_memo_record(void)
{
    DspStopRecordVm();
    disable_vms_rec_audio_path();
    memset(vms_rec_data, 0,sizeof(vms_rec_data));
}

/****************************************************************************************
* Function:... vms_rec_init
* Description: -
****************************************************************************************/
static void vms_rec_init(void)
{
    InitVmsRec();

    switch (aud_glob_vms_rec.media_type) {
        case aud_media_ram:
            vms_rec.recording_frame = aud_glob_vms_rec.start_offset;
            break;

        case aud_media_ffs:
        case aud_media_mmc:
#ifdef  AUD_NANDDEVICE_SUPPORT        	
        case aud_media_fs_cdrive:
        case aud_media_fs_fdrive:		    
#endif         	
            vms_rec.data_outstanding = FALSE;
            if (aud_glob_vms_rec.start_offset)
                vms_set_fileposition();
            break;

        case aud_media_mmf:
     	    aud_glob_vms_rec.buffer_size = 0x0000; /*Explicity making this zero */
     	    aud_glob_vms_rec.file_handle = NULL;
         (*aud_glob_vms_rec.put_buffer_func)((UINT8 **)&aud_glob_vms_rec.file_handle,
                                                (UINT16 *)&aud_glob_vms_rec.buffer_size,
                                                aud_glob_vms_rec.mmf_user_data);
            break;
    }
    AUD_set_internal_vms_rec_state(S_aud_vms_delay_before_recording);
    enable_vms_rec_audio_path();
    aud_send_mail(E_timer_vms_rec_timeout, 0, 0, 0, 0, 0, 0, 0, 0, 0,NULL, NULL);

    return;
}/* vms_rec_init */

/****************************************************************************************
* Function:... enable_vms_audio_path
* Parameters:. audio_mode: The actual audio mode for which the gain cells should be set.
*              volume: The actual volume step.
* Description: Enable the earpiece, i.e. all voiceband parameters in the DSP are set, and
*              GAIM is programmed.
****************************************************************************************/
static void enable_vms_rec_audio_path(void)
{
    aud_enable_path_and_resource(aud_resource_record_vm,S_active,FALSE);
}/* End of enable_vms_audio_path */

/****************************************************************************************
* Function:... disable_vms_rec_audio_path
* Description: -
****************************************************************************************/
static void disable_vms_rec_audio_path(void)
{
    aud_disable_path_and_resource(aud_resource_record_vm,S_idle,FALSE); 
} /* End of function disable_vms_rec_audio_path */

/****************************************************************************************
* Function:... DspStartRecordVm
* Description: -
****************************************************************************************/
static void DspStartRecordVm(aud_dsp_format_enum codec_type, UINT16 amr_sample_rate, UINT16 dtx)
{
    unsigned int control_word =0x0;

    DISABLE_CLOCK;
    if (codec_type == aud_dsp_format_fr) {
	 aud_dsp_vm_parms.vm_mode &= (VM_MODE_DTX_BIT |VM_MODE_INIT_COEFF |VM_MODE_AMR_PLAY_BIT |VM_MODE_LOOP_BIT |VM_MODE_AMR_IF2_PLAY_BIT |VM_MODE_PCM_PLAY_BIT |VM_MODE_FR_PLAY_BIT); //0xFEBA;	
        aud_dsp_vm_parms.vm_mode =  aud_dsp_vm_parms.vm_mode | VM_MODE_FR_REC_BIT;
    } 
    if (codec_type == aud_dsp_format_amr) { /* IF1 */
        aud_dsp_vm_parms.vm_mode &= (VM_MODE_INIT_COEFF |VM_MODE_AMR_PLAY_BIT |VM_MODE_LOOP_BIT |VM_MODE_AMR_IF2_PLAY_BIT |VM_MODE_PCM_PLAY_BIT |VM_MODE_FR_PLAY_BIT); //0x0EBA;  
        control_word = ((( (UINT16)((dtx << 3) | amr_sample_rate)  ) << 4 ) | 0x1); /*Lower byte has DTX and data rate info */
        aud_dsp_vm_parms.vm_mode |= (control_word << 8);
    }
    if (codec_type == aud_dsp_format_amr_if2) { /* IF2 */
        aud_dsp_vm_parms.vm_mode &= (VM_MODE_INIT_COEFF |VM_MODE_AMR_PLAY_BIT |VM_MODE_LOOP_BIT |VM_MODE_AMR_IF2_PLAY_BIT |VM_MODE_PCM_PLAY_BIT |VM_MODE_FR_PLAY_BIT); //0x0EBA;  
        control_word = ( (UINT16)((dtx << 3) | amr_sample_rate)  ); /*Lower byte has DTX and data rate info */
        aud_dsp_vm_parms.vm_mode |= ((control_word << 12) | VM_MODE_AMR_IF2_REC_BIT); //change! to support IF2
    }
/*lint -save -e740 */
    DSP_ASYNC_VM_CMD(&aud_dsp_vm_parms); 
/*lint -restore */
    ENABLE_CLOCK;

#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_sm_vm_recording, 1, 0,               // LLT params 
                       'd', 0x02,                                               // LIS params 
                       aud_log_lis_display, aud_log_lis_display_data, aud_log_lis_trace, // LIS params 
                       aud_dsp_vm_parms.vm_mode );  
#endif

} /* End of function DspStartRecordVm */

/****************************************************************************************
* Function:... DspStopRecordVm
* Description: -
****************************************************************************************/
static void DspStopRecordVm(void)
{
#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_sm_vm_recording, 0, 0,               // LLT params 
                       'd', 0x11,                                               // LIS params 
                       aud_log_lis_display, aud_log_lis_display_nodata, aud_log_lis_trace ); // LIS params 
#endif

    DISABLE_CLOCK;
    /* To take care if playback is in parallel */
    aud_dsp_vm_parms.vm_mode = aud_dsp_vm_parms.vm_mode & (VM_MODE_DTX_BIT |VM_MODE_CR2_BIT | VM_MODE_CR1_BIT | VM_MODE_CR0_BIT | VM_MODE_INIT_COEFF |VM_MODE_AMR_PLAY_BIT |VM_MODE_LOOP_BIT |VM_MODE_AMR_IF2_PLAY_BIT |VM_MODE_PCM_PLAY_BIT |VM_MODE_FR_PLAY_BIT);    //0xFEBA;
    DSP_ASYNC_VM_CMD(&aud_dsp_vm_parms);
    ENABLE_CLOCK;
} /* End of function DspStopRecordVm */

/****************************************************************************************
* Function:... AUD_set_internal_vms_rec_state
* Description: -
****************************************************************************************/
static void AUD_set_internal_vms_rec_state(UINT8 new_state)
{
    vms_rec.aud_vms_rec_state = new_state;

#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_sm_vm_recording, 2, 0,     // LLT params 
                       'm', 0x012,                                                  // LIS params 
                       aud_log_lis_display, aud_log_lis_display_data, aud_log_lis_llt_trace,        // LIS params 
                       (UINT32) vms_rec.aud_vms_rec_state,
                       (UINT32) E_audio_llt_type_id_1 );
#endif
}

/****************************************************************************************
* Function:... find_nof_bytes_per_frame_record
* Description: -
****************************************************************************************/
static void find_nof_bytes_per_frame_record(UINT8 header)
{
    UINT8 sample_rate;

    if (aud_glob_vms_rec.format == aud_dsp_format_amr) {
        sample_rate   = header;
        sample_rate  &= 0x78;
        sample_rate >>= 3;

        switch (sample_rate) {
            case 0:    /* 4,75 kbit/sec */
                vms_rec.nof_bytes_per_frame = 13;
                break;
            case 1:    /* 5,15 kbit/sec */
                vms_rec.nof_bytes_per_frame = 14;
                break;
            case 2:    /* 5,90 kbit/sec */
                vms_rec.nof_bytes_per_frame = 16;
                break;
            case 3:    /* 6,70 kbit/sec */
                vms_rec.nof_bytes_per_frame = 18;
                break;
            case 4:    /* 7,40 kbit/sec */
                  vms_rec.nof_bytes_per_frame = 20;
                break;
            case 5:    /* 7,95 kbit/sec */
                vms_rec.nof_bytes_per_frame = 21;
                break;
            case 6:    /*10,2 kbit/sec  */
                  vms_rec.nof_bytes_per_frame = 27;
                break;
            case 7:    /*12,2 kbit/sec  */
                  vms_rec.nof_bytes_per_frame = 32;
                break;
            case 15:   /* NO-DATA frame */
                vms_rec.nof_bytes_per_frame = 1;
                break;
            default:   /* SID frame */
                vms_rec.nof_bytes_per_frame = 6;
                break;
        }
    }
    else if (aud_glob_vms_rec.format == aud_dsp_format_amr_if2) {
        sample_rate   = header;
        sample_rate  &= 0x78;
        sample_rate >>= 3;

        switch (sample_rate) {
            case 0:    /* 4,75 kbit/sec */
                vms_rec.nof_bytes_per_frame = 13;
                break;
            case 1:    /* 5,15 kbit/sec */
                vms_rec.nof_bytes_per_frame = 14;
                break;
            case 2:    /* 5,90 kbit/sec */
                vms_rec.nof_bytes_per_frame = 16;
                break;
            case 3:    /* 6,70 kbit/sec */
                vms_rec.nof_bytes_per_frame = 18;
                break;
            case 4:    /* 7,40 kbit/sec */
                  vms_rec.nof_bytes_per_frame = 19;
                break;
            case 5:    /* 7,95 kbit/sec */
                vms_rec.nof_bytes_per_frame = 21;
                break;
            case 6:    /*10,2 kbit/sec  */
                  vms_rec.nof_bytes_per_frame = 26;
                break;
            case 7:    /*12,2 kbit/sec  */
                  vms_rec.nof_bytes_per_frame = 31;
                break;
            case 15:   /* NO-DATA frame */
                vms_rec.nof_bytes_per_frame = 1;
                break;
            default:   /* SID frame */
                vms_rec.nof_bytes_per_frame = 6;
                break;
        }
    }
    else{
        vms_rec.nof_bytes_per_frame = 34;
      }

    /*[begin] add by wangjun 200907**/
    (*aud_glob_byteCallBack)(vms_rec.nof_bytes_per_frame);
    /*[end]*/
	
}

/****************************************************************************************
* Function:... vms_rec_start
* Description: -
****************************************************************************************/
static void vms_rec_start(aud_dsp_format_enum format)
{
    switch (format) {
        case aud_dsp_format_fr:
            DspStartRecordVm(format, 0, 0);
            break;
        case aud_dsp_format_amr:
        case aud_dsp_format_amr_if2:
            DspStartRecordVm(format, aud_glob_vms_rec.rate, 0);
            break;
    }
    if (! vms_rec.rc_check_record_flag) {
        vm_rec_send_response_signal(aud_rc_recording_started);
        vms_rec.rc_check_record_flag = TRUE;
    }
    AUD_set_internal_vms_rec_state(S_aud_vms_recording);
}

/****************************************************************************************
* Function:... copy_memo_buffer
* Description: This function copy the coded speech buffer from DSP to the application
****************************************************************************************/
static void copy_memo_buffer(void)
{
    aud_glob_vms_rec.fp_voice_memo_putpacket((UINT16 DWD_HUGE *)(p_l1d_vm_data->vm_buffer_0));
}

/****************************************************************************************
* Function   :  AudVmsRecL1EventHandle
* Description:  Voice Memo Record Interrrupt Service Routine
****************************************************************************************/
void AudVmsRecL1EventHandle(aud_event_enum event)
{
// Should be removed after the reference to this function from base file is removed.
}

/****************************************************************************************
* Function   :  AudVmsRecInterruptHandle
* Description:  Voice Memo Record Interrrupt Service Routine
****************************************************************************************/
void AudVmsRecInterruptHandle(void)
{
    SINT8 rc;
#ifdef AUD_HISR_TIMER_MEASUREMENT
    UINT32 time_start;
    UINT32 time_end;
    time_start=RTT_stm_get_time_10us_resolution(); //Get start time
    #endif
    /* Send string : at+xl1set="l1 23l" To enable this LLT trace */

#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_hisr_vm_recording, 3, 0,               // LLT params 
                       'm', 0x15,                                               // LIS params 
                       aud_log_lis_display, aud_log_lis_display_data, aud_log_lis_llt_trace, // LIS params 
                       (UINT32) vms_rec.aud_vms_rec_state,
                       (UINT32) vms_rec.aud_vms_rec_error,
                       (UINT32) vms_rec.vm_result );
#endif
    switch (vms_rec.aud_vms_rec_state) {
        case S_aud_vms_rec_idle:
            break;

        case S_aud_vms_recording:
            (*aud_glob_recordCallBack)();         

            break;
    }
#ifdef AUD_HISR_TIMER_MEASUREMENT
    time_end= RTT_stm_get_time_10us_resolution();

    //check for wrap around of STM timer - happen app. each 5.8 hours
    if (time_start>time_end) {
        time_end=0xFFFFFFFF-(time_start-time_end); //store result in start_end - 32bit counter
        time_start=0;   // do not mess up reading out of result later
    }

    aud_common_trace( llt_group_audio2, llt_type_audio_timing_measurement, 3, 0,               // LLT params 
                       'm', 0x15,                                               // LIS params 
                       aud_log_lis_display, aud_log_lis_display_data, aud_log_llt_trace, // LIS params 
                       (UINT32) vms_rec.aud_vms_rec_state,
                       (UINT32) time_end-time_start,
                       (UINT32) E_audio_llt_type_id_3 );
#endif
}

/****************************************************************************************
* Function:... AudVmsRecActive
* Returns:.... 1: if the voice memo service is active,
*              0: if the voice memo service isn't active, i.e. power down is allowed.
****************************************************************************************/
UINT8 AudVmsRecActive(void)
{
    if (vms_rec.aud_vms_rec_state == S_aud_vms_rec_idle)
        return 0;
    else
        return 1;
} /* End of function AudVmsRecActive */

/****************************************************************************************
* Function:... vms_set_fileposition
* Parameters:. -
* Returns:.... -
* Description: -
* Created:.... -
****************************************************************************************/
static UINT32 vms_set_fileposition(void)
{
    UINT8 dst[VM_REC_BUFFER_SIZE];
    aud_ffs_error_code_type rc;
    UINT16 ltemp =0;
    UINT16 lOffset;

    memset(dst, 0,sizeof(dst));
    vms_rec.data_outstanding = FALSE;

    do {
        if ((aud_glob_vms_rec.start_offset - ltemp)> VM_REC_BUFFER_SIZE)
            lOffset = VM_REC_BUFFER_SIZE;
        else
            lOffset =  (aud_glob_vms_rec.start_offset - ltemp);
        ltemp +=lOffset;

        if (!vms_rec.data_outstanding) {
            vms_rec.data_outstanding = TRUE;
            //FFS replacement
            rc = (aud_ffs_error_code_type)Aud_SSAL_write((aud_ssal_media_type)aud_glob_vms_rec.ssal_media_type,
                                                         (aud_ssal_user_type)AUD_SSAL_VR_WRITE,
                                                         (void*)aud_glob_vms_rec.file_handle,
                                                         (UINT8*) dst,
                                                         lOffset,
                                                         &vms_write_ssal_callback
#ifdef FFS_EXTENDED_CALLBACK
                                                         ,NULL
#endif
                                                        );
            if (rc != AUD_FFS_SUCCESS && rc != AUD_FS_SUCCESS)
                return(0);
        }
    }while (lOffset > VM_REC_BUFFER_SIZE);
    return(TRUE);
}

/****************************************************************************************
* Function:... voice_memo_putpacket_in_ffs
* Description: -
****************************************************************************************/
SINT16 voice_memo_putpacket_in_ffs(UINT16 DWD_HUGE *buf)
{
#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_sm_vm_recording, 0, 0,  // LLT params 
                       'm', 0x0f,                                                // LIS params 
                       aud_log_lis_display, aud_log_lis_display_nodata, aud_log_lis_trace ); // LIS params 
#endif

    if ((vms_rec.recording_frame + vms_rec.nof_bytes_per_frame) >VM_REC_BUFFER_SIZE) {
        if (vms_rec.data_outstanding) {
            vms_rec.vm_result = AUD_VM_PERFORMANCE_PROBLEMS;
            vms_rec.aud_vms_rec_error= TRUE;
        } else {
            vms_rec.data_outstanding = TRUE;
            //FFS replacement
            vms_rec.ffs_result = Aud_SSAL_write((aud_ssal_media_type)aud_glob_vms_rec.ssal_media_type,
                                                (aud_ssal_user_type)AUD_SSAL_VR_WRITE,
                                                (void*)aud_glob_vms_rec.file_handle,
                                                (UINT8*)vms_rec_data[vms_rec.array_number].data,
                                                vms_rec.recording_frame,
                                                &vms_write_ssal_callback
#ifdef FFS_EXTENDED_CALLBACK
                                                ,NULL
#endif
                                               );

             if (vms_rec.ffs_result != AUD_FFS_SUCCESS && vms_rec.ffs_result != AUD_FS_SUCCESS) {
                vms_rec.aud_vms_rec_error= TRUE;
                vms_rec.vm_result = AUD_VM_PERFORMANCE_PROBLEMS;
            } else {
                vms_rec.bytes_stored +=vms_rec.recording_frame;
                vms_rec.recording_frame = 0;
                vms_rec.array_number =! vms_rec.array_number;
                vms_rec.data_outstanding = TRUE;
            }
        }
    }
    memcpy(&vms_rec_data[vms_rec.array_number].data[vms_rec.recording_frame],buf, vms_rec.nof_bytes_per_frame);
    vms_rec.recording_frame += vms_rec.nof_bytes_per_frame;

    return TRUE;
}/* End of function voice_memo_putpacket */

/****************************************************************************************
* Function:... vms_write_ssal_callback
* Description: -
****************************************************************************************/
static void vms_write_ssal_callback(SINT16 result
#ifdef FFS_EXTENDED_CALLBACK
                             ,void *input
#endif
                            )
{
    UINT16 rc_check ;
    vms_rec.ffs_result = result;

    rc_check  = Aud_SSAL_result_ok(aud_glob_vms_rec.ssal_media_type, vms_rec.ffs_result);
    if (rc_check ) {
        vms_rec.data_outstanding = FALSE;
        if (vms_rec.aud_vms_rec_state ==S_aud_vms_finish_recording)
            aud_send_mail(E_timer_vms_rec_timeout, 0, 0, 0, 0, 0, 0, 0, 0, 0,NULL, NULL);
    } else {
        vms_rec.vm_result = aud_rc_performance_problems;
        vms_rec.aud_vms_rec_error= AUD_VM_PERFORMANCE_PROBLEMS;
    }
}

/****************************************************************************************
* Function:... voice_memo_putpacket_in_ram
* Description: -
****************************************************************************************/
SINT16 voice_memo_putpacket_in_ram(UINT16 DWD_HUGE *buf)
{
#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_sm_vm_recording, 0, 0,               // LLT params 
                       'm', 0x10,                                               // LIS params 
                       aud_log_lis_display, aud_log_lis_display_nodata, aud_log_lis_trace ); // LIS params 
#endif

    if ((vms_rec.recording_frame + vms_rec.nof_bytes_per_frame - 1) > aud_glob_vms_rec.buffer_size) {
        vms_rec.vm_result = AUD_VM_RAM_BUFFER_USED;
        vms_rec.aud_vms_rec_error= TRUE;
    } else {
        memcpy(&((UINT8 DWD_HUGE *)aud_glob_vms_rec.file_handle)[vms_rec.recording_frame],buf, vms_rec.nof_bytes_per_frame);
        vms_rec.bytes_stored += vms_rec.nof_bytes_per_frame;
        vms_rec.recording_frame += vms_rec.nof_bytes_per_frame;
    }
    return TRUE;
}/* End of function voice_memo_putpacket */

/****************************************************************************************
* Function:... voice_memo_putpacket_in_mmf
* Parameters:. -
* Returns:.... -
* Description: -
* Created:.... -
****************************************************************************************/
SINT16 voice_memo_putpacket_in_mmf(UINT16 DWD_HUGE *buf)
{
    UINT16 remaining,i;
    static BOOL vms_remainingBuffer_flag = FALSE;

#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_sm_vm_recording, 0, 0,               // LLT params 
                       'm', 0x11,                                               // LIS params 
                       aud_log_lis_display, aud_log_lis_display_nodata, aud_log_lis_trace); // LIS params 
#endif

    if ( (aud_glob_vms_rec.buffer_size == 0) || (aud_glob_vms_rec.file_handle == NULL) )
    	{
  		 	   aud_glob_vms_rec.buffer_size = 0x0000; /*Explicity making this zero */
          aud_glob_vms_rec.file_handle = NULL;
         /* Currently no buffer avalable, check for new */	 	
			    (*aud_glob_vms_rec.put_buffer_func)((UINT8 **)&aud_glob_vms_rec.file_handle,
			                                       (UINT16 *)&aud_glob_vms_rec.buffer_size,
			                                        aud_glob_vms_rec.mmf_user_data);
			    if (aud_glob_vms_rec.file_handle == NULL) 
			    	{
			        /* Still no buffers - skip the data */
			        if (aud_glob_vms_rec.buffer_size == 0) 
			        	{
			           /* Size is 0, i.e. stop recording */
			            stop_voice_memo_record();
			            vm_rec_send_response_signal(aud_rc_recording_finish);
			            InitVmsRec();
			         }
			    } 
			    else 
			    	{
			         if (aud_glob_vms_rec.buffer_size == 0) 
			         	{
				           /* Size is 0, i.e. stop recording */
				            stop_voice_memo_record();
				            vm_rec_send_response_signal(aud_rc_recording_finish);
				            InitVmsRec();
			         }
			         else
			         {
			          /* Buffer are now available */
			  	        vms_rec.recording_frame = 0; 		   
			            if(vms_remainingBuffer_flag == TRUE)
			            	{
			                if(vms_rec.vms_mmf_remaining_buffersize  > aud_glob_vms_rec.buffer_size)
			                	{ /*The Buffer size is less than the remaining data to be copied */
			                   memcpy(&((UINT8 DWD_HUGE *)aud_glob_vms_rec.file_handle)[vms_rec.recording_frame], vms_rec.vms_temp_streaming_buffer,aud_glob_vms_rec.buffer_size);
			                   remaining = vms_rec.vms_mmf_remaining_buffersize - aud_glob_vms_rec.buffer_size;
			                   for(i = 0; i < remaining; i++) 
			                   	{
			                      vms_rec.vms_temp_streaming_buffer[i] = vms_rec.vms_temp_streaming_buffer[ (i + aud_glob_vms_rec.buffer_size)];
			                   }
			                   vms_remainingBuffer_flag     = TRUE;
			                   vms_rec.recording_frame      = aud_glob_vms_rec.buffer_size;
			                   vms_rec.vms_mmf_remaining_buffersize = remaining;            
			                	}
			                else
			                	{
			                	    /*The Buffer size is greater than the remaining previous frame recorded data to be copied */                 
			                   memcpy(&((UINT8 DWD_HUGE *)aud_glob_vms_rec.file_handle)[vms_rec.recording_frame], vms_rec.vms_temp_streaming_buffer, vms_rec.vms_mmf_remaining_buffersize);
			                   vms_remainingBuffer_flag = FALSE;
			                   vms_rec.recording_frame = vms_rec.vms_mmf_remaining_buffersize; 
			                   vms_rec.vms_mmf_remaining_buffersize = 0;           
			                   /* Check whether any space left out to store the current recorded data in the callback buffer */
                 
			                   if ((vms_rec.recording_frame + vms_rec.nof_bytes_per_frame) < aud_glob_vms_rec.buffer_size ) 
			                   	{
						                   	memcpy(&((UINT8 DWD_HUGE *)aud_glob_vms_rec.file_handle)[vms_rec.recording_frame], buf, vms_rec.nof_bytes_per_frame);
						                   vms_rec.recording_frame += vms_rec.nof_bytes_per_frame;
			                   	}
			                   else
			                   	{
																	/* No Enough buffers to store the current recorded data. Discard the frame */
			                   	}
			                	}
			            	}
			            else
			            	{
												if( (vms_rec.recording_frame + vms_rec.nof_bytes_per_frame) > aud_glob_vms_rec.buffer_size )
													{
													      /* Store the recorded data in the callback buffer */
														    remaining =  aud_glob_vms_rec.buffer_size - vms_rec.recording_frame;    
											  		    memcpy(&((UINT8 DWD_HUGE *)aud_glob_vms_rec.file_handle)[vms_rec.recording_frame], buf, remaining);   
											  		    vms_rec.recording_frame += remaining;

											  		     /* Store the remaining part of the recorded data in the 
                      					  temporary buffer so that it could be stored in the next callback function */
												          buf = (UINT16 DWD_HUGE *)((UINT8*)buf + remaining);
												          vms_rec.vms_mmf_remaining_buffersize = vms_rec.nof_bytes_per_frame - remaining;
												          memcpy(&vms_rec.vms_temp_streaming_buffer, buf, vms_rec.vms_mmf_remaining_buffersize);   
												          vms_remainingBuffer_flag = TRUE;
													}
												else 
													{
													    /* Everything is OK - copy data to buffer */
													    memcpy(&((UINT8 DWD_HUGE *)aud_glob_vms_rec.file_handle)[vms_rec.recording_frame], buf,vms_rec.nof_bytes_per_frame);
												      vms_rec.recording_frame += vms_rec.nof_bytes_per_frame; 
													}												
			            	}
			         	}
			    	}
    	}
    else if ( (vms_rec.recording_frame + vms_rec.nof_bytes_per_frame) > aud_glob_vms_rec.buffer_size )
    {
		    remaining =  aud_glob_vms_rec.buffer_size - vms_rec.recording_frame;    
		    memcpy(&((UINT8 DWD_HUGE *)aud_glob_vms_rec.file_handle)[vms_rec.recording_frame], buf, remaining);   
 		 	   aud_glob_vms_rec.buffer_size = 0x0000; /*Explicity making this zero */
        aud_glob_vms_rec.file_handle = NULL;
        /* Request for new buffer */
		    (*aud_glob_vms_rec.put_buffer_func)((UINT8 **)&aud_glob_vms_rec.file_handle, (UINT16 *)&aud_glob_vms_rec.buffer_size,                                               
																																								                                                 aud_glob_vms_rec.mmf_user_data);
		     if (aud_glob_vms_rec.file_handle == NULL)
		     	{
		          /*Save the remaining frame data to be stored in the next Callback function to maintain
		            the synchronisation of frame */
		          if(aud_glob_vms_rec.buffer_size == 0)
		          	{
			           /* Size is 0, i.e. stop recording */
				           stop_voice_memo_record();
		  		         vm_rec_send_response_signal( aud_rc_recording_finish);
		     	  	      InitVmsRec();	
		          	}		          
		          else 
		          	{
		         	  vms_rec.recording_frame = 0;		         
                             buf = (UINT16 DWD_HUGE *)((UINT8*)buf + remaining);
		               vms_rec.vms_mmf_remaining_buffersize = vms_rec.nof_bytes_per_frame - remaining;
		               memcpy(&vms_rec.vms_temp_streaming_buffer, buf, vms_rec.vms_mmf_remaining_buffersize);   
		               vms_remainingBuffer_flag = TRUE;
		          	}
		     }
		     else 
		     	{
		        if(aud_glob_vms_rec.buffer_size == 0)
		        	{
		           /* Size is 0, i.e. stop recording */
		           stop_voice_memo_record();
		           vm_rec_send_response_signal( aud_rc_recording_finish);
		           InitVmsRec();	
		        } 
		        else 
		        	{
		        	   /* Check whether enough callback buffer is there to store remaining recorded data */
				        	if ( remaining < aud_glob_vms_rec.buffer_size )
				        		{
				        		        /*Copy the remaining recorded data */
                                                        buf = (UINT16 DWD_HUGE *)((UINT8*)buf + remaining);	
                                                        vms_rec.recording_frame = 0;
                                                        remaining = vms_rec.nof_bytes_per_frame - remaining;
                                                        memcpy(&((UINT8 DWD_HUGE *)aud_glob_vms_rec.file_handle)[vms_rec.recording_frame], buf, remaining);   
                                                        vms_rec.recording_frame		= remaining;
                                      
				        		}
				        	else 
				        		{
                                                        vms_rec.recording_frame = 0;
                                                        buf = (UINT16 DWD_HUGE *)((UINT8*)buf + remaining);		           
                                                        remaining = vms_rec.nof_bytes_per_frame - remaining;
                                                        memcpy(&((UINT8 DWD_HUGE *)aud_glob_vms_rec.file_handle)[vms_rec.recording_frame], buf, aud_glob_vms_rec.buffer_size);   
                                                        vms_rec.recording_frame = aud_glob_vms_rec.buffer_size;
                                                        buf = (UINT16 DWD_HUGE *)((UINT8*)buf + aud_glob_vms_rec.buffer_size);		     
                                                        vms_rec.vms_mmf_remaining_buffersize =  remaining - aud_glob_vms_rec.buffer_size;    
                                                        memcpy(&vms_rec.vms_temp_streaming_buffer, buf, vms_rec.vms_mmf_remaining_buffersize);   
                                                        vms_remainingBuffer_flag = TRUE;
				        		}
		        	}
		     	}
    	}
     else
     	{
    			/* Everything is OK - copy data to buffer */
		   	 memcpy(&((UINT8 DWD_HUGE *)aud_glob_vms_rec.file_handle)[vms_rec.recording_frame], buf,vms_rec.nof_bytes_per_frame);
		    	vms_rec.recording_frame += vms_rec.nof_bytes_per_frame; 
     	}
    return TRUE;
}/* End of function voice_memo_putpacket */

/***************************************************************************************
* Function:... InitVmsRec
* Description: -
****************************************************************************************/
void InitVmsRec(void)
{
    vms_rec.aud_vms_rec_state= S_aud_vms_rec_idle;
    vms_rec.aud_vms_rec_error = 0;
    vms_rec.wait_for_recording= 0;
    vms_rec.recording_frame = 0;
    vms_rec.record_to_buffer = FALSE;
    vms_rec.vm_result = 0;
    vms_rec.ffs_result = AUD_FFS_SUCCESS;
    vms_rec.array_number = 0;
    vms_rec.data_outstanding = TRUE;
    vms_rec.nof_bytes_per_frame = 0;
    vms_rec.bytes_recorded = 0;
    vms_rec.bytes_stored =0;
    vms_rec.nof_words_per_frame =0;
    vms_rec.rc_check_record_flag = FALSE;
}

/***********************************************************************
* NAME   :  vm_rec_send_response_signal
* INPUT  :  the state for voicememo
* DESCR. :  handles internal voice memo
************************************************************************/
static void vm_rec_send_response_signal(  SINT8 rc )
{
    switch (rc) {
        case aud_rc_ok:
        case aud_rc_recording_started:
        case aud_rc_performance_problems:
            aud_send_response_signal(aud_glob_vms_rec.sender, aud_resource_record_vm, rc,0, 0, 0, 0, NULL, NULL);
            break;

        case aud_rc_storage_problems:
            aud_send_response_signal(aud_glob_vms_rec.sender, aud_resource_record_vm, rc, vms_rec.ffs_result, 0, 0, 0, NULL, NULL);
            break;

        case aud_rc_ram_buffer_used:
            aud_send_response_signal(aud_glob_vms_rec.sender, aud_resource_record_vm, rc,vms_rec.bytes_stored, 0, 0, 0, NULL, NULL);
            break;

        case aud_rc_recording_finish:
            if (aud_glob_vms_rec.media_type == aud_media_mmf)
                aud_send_response_signal(aud_glob_vms_rec.sender, aud_resource_record_vm, rc,vms_rec.recording_frame, 0, 0, 0, NULL, NULL);
            else
                aud_send_response_signal(aud_glob_vms_rec.sender, aud_resource_record_vm, rc,vms_rec.bytes_stored, 0, 0, 0, NULL, NULL);
            break;

        case aud_rc_missing_dsp_resources:
            aud_send_response_signal(aud_glob_vms_rec.sender, aud_resource_record_vm, rc,0, 0, 0, 0, NULL, NULL);
            break;
    }
}
#endif

