/*
 * NAND Flash Controller Device Driver
 * Copyright © 2009-2010, Intel Corporation and its suppliers.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
#include <malloc.h> 
#include <asm/io.h> 
#include <linux/mtd/mtd.h>
#include <asm-generic/ioctl.h>
#include <config.h>
#include <common.h>
#include <command.h>
#include <asm/arch/nand.h> 
#include <linux/mtd/nand.h>
#include "denali.h"
#include <asm/arch/lsp_crpm.h>
#include <boot_mode.h>

 
/* DEBUG */
#if	DENALI_DEBUG
#define denali_debug(fmt,args...)	printf (fmt ,##args)
#define denali_debugX(level,fmt,args...) if (DEBUG>=level) printf(fmt,##args);
#else
#define denali_debug(fmt,args...)
#define denali_debugX(level,fmt,args...)
#endif	/* DEBUG */



/* We define a module parameter that allows the user to override
 * the hardware and decide what timing mode should be used.
 */
#define NAND_DEFAULT_TIMINGS	-1

//static int onfi_timing_mode = NAND_DEFAULT_TIMINGS;

/* We define a macro here that combines all interrupts this driver uses into
 * a single constant value, for convenience. */
#define DENALI_IRQ_ALL	(INTR_STATUS__DMA_CMD_COMP | \
			INTR_STATUS__ECC_ERR | \
			INTR_STATUS__PROGRAM_FAIL | \
			INTR_STATUS__LOAD_COMP | \
			INTR_STATUS__PROGRAM_COMP | \
			INTR_STATUS__TIME_OUT | \
			INTR_STATUS__ERASE_FAIL | \
			INTR_STATUS__RST_COMP | \
			INTR_STATUS__ERASE_COMP)

/* indicates whether or not the internal value for the flash bank is
 * valid or not */
#define CHIP_SELECT_INVALID	-1

#define SUPPORT_8BITECC		1

/* This macro divides two integers and rounds fractional values up
 * to the nearest integer value. */
#define CEIL_DIV(X, Y) (((X)%(Y)) ? ((X)/(Y)+1) : ((X)/(Y)))

/* this macro allows us to convert from an MTD structure to our own
 * device context (denali) structure.
 */
//#define mtd_to_denali(m) container_of(m, struct denali_nand_info, mtd)

/* These constants are defined by the driver to enable common driver
 * configuration options. */
#define SPARE_ACCESS		0x41
#define MAIN_ACCESS		0x42
#define MAIN_SPARE_ACCESS	0x43

#define DENALI_READ	0
#define DENALI_WRITE	0x100

/* types of device accesses. We can issue commands and get status */
#define COMMAND_CYCLE	0
#define ADDR_CYCLE	1
#define STATUS_CYCLE	2

/* this is a helper macro that allows us to
 * format the bank into the proper bits for the controller */
#define BANK(x) ((x) << 24)

#define true		    1
#define false		    0

extern struct nand_flash_device_para nand_flash_para[];
extern struct mtd_info nand_info[];
extern struct nand_chip nand_chip[];
extern int flash_dmabuf_disable_flag; 


struct denali_nand_info denali_info = {0};
struct denali_nand_info *g_denali = &denali_info;
struct nand_flash_device_para *g_nand_dev_info = NULL;

/* forward declarations */
//static void clear_interrupts(struct denali_nand_info *denali);
//static void denali_irq_enable(struct denali_nand_info *denali,
//							uint32_t int_mask);
//static uint32_t read_interrupt_status(struct denali_nand_info *denali);

/* Certain operations for the denali NAND controller use
 * an indexed mode to read/write data. The operation is
 * performed by writing the address value of the command
 * to the device memory followed by the data. This function
 * abstracts this common operation.
*/
static void index_addr(struct denali_nand_info *denali,
				uint32_t address, uint32_t data)
{
	writel(address, denali->flash_mem);
	writel(data, denali->flash_mem + 0x10);
}

/* Perform an indexed read of the device */
static void index_addr_read_data(struct denali_nand_info *denali,
				 uint32_t address, uint32_t *pdata)
{
	writel(address, denali->flash_mem);
	*pdata = readl(denali->flash_mem + 0x10);
}

/* We need to buffer some data for some of the NAND core routines.
 * The operations manage buffering that data. */
static void reset_buf(struct denali_nand_info *denali)
{
	denali->buf.head = denali->buf.tail = 0;
}

static void write_byte_to_buf(struct denali_nand_info *denali, uint8_t byte)
{
	BUG_ON(denali->buf.tail >= sizeof(denali->buf.buf));
	denali->buf.buf[denali->buf.tail++] = byte;
}

/* reads the status of the device */
static void read_status(struct denali_nand_info *denali)
{
	uint32_t status, addr;     
    addr = (uint32_t)MODE_11 | BANK(denali->flash_bank);
	index_addr(denali, (uint32_t)addr | 0, 0x70);
	index_addr_read_data(denali,(uint32_t)addr | 2,	&status);
	write_byte_to_buf(denali, status);
}

/* resets a specific device connected to the core */
static void reset_bank(struct denali_nand_info *denali)
{
//	uint32_t status_type = 0;
//	uint32_t status_mask = INTR_STATUS__RST_COMP |
//			    INTR_STATUS__TIME_OUT;

//	clear_interrupts(denali);

	writel(1 << denali->flash_bank, denali->flash_reg + DEVICE_RESET);

    while (!(readl(denali->flash_reg +
				INTR_STATUS(denali->flash_bank)) &
			(INTR_STATUS__RST_COMP | INTR_STATUS__TIME_OUT)))
    {

		if (readl(denali->flash_reg + INTR_STATUS(denali->flash_bank)) &
			INTR_STATUS__TIME_OUT)
		{
			debug("NAND Reset operation timed out on bank %d\n", denali->flash_bank);
		}
    }
}

/* Reset the flash controller */
static uint16_t denali_nand_reset(struct denali_nand_info *denali)
{
	uint32_t i;

	for (i = 0 ; i < denali->max_banks; i++)
		writel(INTR_STATUS__RST_COMP | INTR_STATUS__TIME_OUT,
		denali->flash_reg + INTR_STATUS(i));

	for (i = 0 ; i < denali->max_banks; i++) 
    {
		writel(1 << i, denali->flash_reg + DEVICE_RESET);
		while (!(readl(denali->flash_reg +
				INTR_STATUS(i)) &
			(INTR_STATUS__RST_COMP | INTR_STATUS__TIME_OUT))); //zhouqi fpga 4.27

		if (readl(denali->flash_reg + INTR_STATUS(i)) &
			INTR_STATUS__TIME_OUT)
			debug("NAND Reset operation timed out on bank %d\n", i);
	}

	for (i = 0; i < denali->max_banks; i++)
		writel(INTR_STATUS__RST_COMP | INTR_STATUS__TIME_OUT,
			denali->flash_reg + INTR_STATUS(i));

	return PASS;
}


/* determines how many NAND chips are connected to the controller. Note for
 * Intel CE4100 devices we don't support more than one device.
 */
 #if 0 
static void find_valid_banks(struct denali_nand_info *denali)
{
	uint32_t id[denali->max_banks];
	int i;

	denali->total_used_banks = 1;
	for (i = 0; i < denali->max_banks; i++) 
    {
		index_addr(denali, (uint32_t)(MODE_11 | (i << 24) | 0), 0x90);
		index_addr(denali, (uint32_t)(MODE_11 | (i << 24) | 1), 0);
		index_addr_read_data(denali,
				(uint32_t)(MODE_11 | (i << 24) | 2), &id[i]);

		debug("Return 1st ID for bank[%d]: %x\n", i, id[i]);

		if (i == 0) 
        {
			if (!(id[i] & 0x0ff))
				break; /* WTF? */
		} 
        else 
        {
			if ((id[i] & 0x0ff) == (id[0] & 0x0ff))
				denali->total_used_banks++;
			else
				break;
		}
	}
	debug(	"denali->total_used_banks: %d\n", denali->total_used_banks);
}
#endif
/*
 * Use the configuration feature register to determine the maximum number of
 * banks that the hardware supports.
 */
static void detect_max_banks(struct denali_nand_info *denali)
{
	//uint32_t features = readl(denali->flash_reg + FEATURES);

	//denali->max_banks = 2 << (features & FEATURES__N_BANKS);
	denali->max_banks = 1; //zhouqi for fpge 4.27 and for evb 7.19
}


static uint32_t detect_nand_bus_freq(void)
{
    uint32_t clk_reg = 0;

	clk_reg = readl(0x01306050);
	clk_reg &= 0xffffcfff;        /*MOD_CLK_SEL[13:12]=00,7520v2 NAND 104MHz*/
    writel(clk_reg, 0x01306050);  
    if((((readl(0x01306050))>>12) & 0x3) == 0)        
        return 104;
    else
        return 26;      
}

#if 0
static void detect_partition_feature(struct denali_nand_info *denali)
{
	/* For MRST platform, denali->fwblks represent the
	 * number of blocks firmware is taken,
	 * FW is in protect partition and MTD driver has no
	 * permission to access it. So let driver know how many
	 * blocks it can't touch.
	 * */
	if (readl(denali->flash_reg + FEATURES) & FEATURES__PARTITION) {
		if ((readl(denali->flash_reg + PERM_SRC_ID(1)) &
			PERM_SRC_ID__SRCID) == SPECTRA_PARTITION_ID) {
			denali->fwblks =
			    ((readl(denali->flash_reg + MIN_MAX_BANK(1)) &
			      MIN_MAX_BANK__MIN_VALUE) *
			     denali->blksperchip)
			    +
			    (readl(denali->flash_reg + MIN_BLK_ADDR(1)) &
			    MIN_BLK_ADDR__VALUE);
		} else
			denali->fwblks = SPECTRA_START_BLOCK;
	} else
		denali->fwblks = SPECTRA_START_BLOCK;
}
#endif

static void denali_nand_register_set(struct denali_nand_info *denali,
                                                struct nand_flash_device_para * table)
{
	writel(table->pages_per_block, denali->flash_reg + PAGES_PER_BLOCK);
    writel(table->bus_num, denali->flash_reg + DEVICE_WIDTH);
    writel(table->page_size, denali->flash_reg + DEVICE_MAIN_AREA_SIZE);
    writel(table->oob_size, denali->flash_reg + DEVICE_SPARE_AREA_SIZE);
    
    if(table->row_addr_num == 2)       
        writel(1, denali->flash_reg + TWO_ROW_ADDR_CYCLES);
    else
        writel(0, denali->flash_reg + TWO_ROW_ADDR_CYCLES);

    writel(table->page_size, denali->flash_reg + LOGICAL_PAGE_DATA_SIZE);
    writel(table->oob_size, denali->flash_reg + LOGICAL_PAGE_SPARE_SIZE);
    writel(1, denali->flash_reg + ECC_ENABLE);

    
}


static void denali_nand_timing_set(struct denali_nand_info *denali,
                                                struct nand_flash_device_para * table)
{
	uint32_t bus_freq;
    struct nand_flash_timing * timing = NULL;

    timing = &(table->nand_timeing);
    bus_freq = detect_nand_bus_freq();
	writel(((timing->Twhr * bus_freq)/1000+2) | (((timing->Trr1 * bus_freq)/1000+2)<<8), 
                denali->flash_reg + WE_2_RE);
    writel(((timing->Tadl * bus_freq)/1000+2) | (((timing->Trr2 * bus_freq)/1000+2)<<8), 
                denali->flash_reg + ADDR_2_DATA);
    writel((timing->Trhw * bus_freq)/1000+2, denali->flash_reg + RE_2_WE);
    writel((timing->Trp * bus_freq)/1000+2, denali->flash_reg + RDWR_EN_LO_CNT);
    writel((timing->Treh * bus_freq)/1000+2, denali->flash_reg + RDWR_EN_HI_CNT);
    writel((timing->Tcs * bus_freq)/1000+2, denali->flash_reg + CS_SETUP_CNT);
    writel((timing->Trhz * bus_freq)/1000+2, denali->flash_reg + RE_2_RE);   
}

static void denali_set_intr_modes(struct denali_nand_info *denali,
					uint16_t INT_ENABLE)
{
	if (INT_ENABLE)
		writel(1, denali->flash_reg + GLOBAL_INT_ENABLE);
	else
		writel(0, denali->flash_reg + GLOBAL_INT_ENABLE);
}

/* validation function to verify that the controlling software is making
 * a valid request
 */
static inline uint32_t is_flash_bank_valid(int flash_bank)
{
	return (flash_bank >= 0 && flash_bank < 4);
}

static void denali_irq_init(struct denali_nand_info *denali)
{
	uint32_t int_mask = 0;
	int i;

	/* Disable global interrupts */
	denali_set_intr_modes(denali, false);

	int_mask = DENALI_IRQ_ALL;

	/* Clear all status bits */
	for (i = 0; i < denali->max_banks; ++i)
		writel(0xFFFF, denali->flash_reg + INTR_STATUS(i));
}

#define BANK(x) ((x) << 24)

static uint32_t wait_for_ready(struct denali_nand_info *denali, uint32_t status_type)
{
    uint32_t status = 0;

    while (!(readl(denali->flash_reg +
				INTR_STATUS(denali->flash_bank)) & status_type));
    status = readl(denali->flash_reg + INTR_STATUS(denali->flash_bank));

    #if 0
    while (!(readl(0x1207410) & status_type));
    status = readl(0x1207410);
    #endif
    
    if (status & INTR_STATUS__ECC_ERR )
    {
        printf ("    Deanli Nand Failed: ECC Error\n");
    }
    if (status & INTR_STATUS__TIME_OUT)
    {
        printf ("    Deanli Nand Failed: Time out\n");
    }
    if (status & INTR_STATUS__PROGRAM_FAIL)
    {
        printf ("    Deanli Nand Failed: Program Fail\n");
    } 
    
    writew(0xffff, denali->flash_reg + INTR_STATUS(denali->flash_bank));
	return status;
}

/* This helper function setups the registers for ECC and whether or not
 * the spare area will be transferred. */
static void setup_ecc_for_xfer(struct denali_nand_info *denali, uint32_t ecc_en,uint32_t transfer_spare)
{
	int ecc_en_flag = 0, transfer_spare_flag = 0;
   #if ECC_TEST_VER
   ecc_en_flag = 0;
   #else
	/* set ECC, transfer spare bits if needed */
	ecc_en_flag = ecc_en ? ECC_ENABLE__FLAG : 0;
   #endif
	transfer_spare_flag = transfer_spare ? TRANSFER_SPARE_REG__FLAG : 0;

	/* Enable spare area/ECC per user's request. */
	writel(ecc_en_flag, denali->flash_reg + ECC_ENABLE);
	writel(transfer_spare_flag,
			denali->flash_reg + TRANSFER_SPARE_REG);
}


static void setup_ecc_the_end(struct denali_nand_info *denali, uint32_t ecc_en)
{
	int ecc_en_flag = 0;

	/* set ECC, transfer spare bits if needed */
	ecc_en_flag = ecc_en ? ECC_ENABLE__FLAG : 0;

	/* Enable spare area/ECC per user's request. */
	writel(ecc_en_flag, denali->flash_reg + ECC_ENABLE);
}


/* sends a pipeline command operation to the controller. See the Denali NAND
 * controller's user guide for more information (section 4.2.3.6).
 */
static int denali_send_pipeline_cmd(struct denali_nand_info *denali,
                            uint32_t ecc_en,
							uint32_t transfer_spare,
							int access_type,
							int op)
{
	int status = PASS;
	uint32_t addr = 0x0, cmd = 0x0, page_count = 1, status_type = 0,
		 status_mask = 0;

	if (op == DENALI_READ)
		status_mask = INTR_STATUS__LOAD_COMP;
	else if (op == DENALI_WRITE)
		status_mask = 0;
	else
		BUG();

	setup_ecc_for_xfer(denali, ecc_en, transfer_spare);

	addr = BANK(denali->flash_bank) | denali->page;

	if (op == DENALI_WRITE && access_type != SPARE_ACCESS) 
    {
		cmd = MODE_01 | addr;
		writel(cmd, denali->flash_mem);
	} 
    else if (op == DENALI_WRITE && access_type == SPARE_ACCESS) 
    {
		/* read spare area */
		cmd = MODE_10 | addr;
		index_addr(denali, (uint32_t)cmd, access_type);

		cmd = MODE_01 | addr;
		writel(cmd, denali->flash_mem);
	} 
    else if (op == DENALI_READ) 
    {
		/* setup page read request for access type */
		cmd = MODE_10 | addr;
		index_addr(denali, (uint32_t)cmd, access_type);

		/* page 33 of the NAND controller spec indicates we should not
		   use the pipeline commands in Spare area only mode. So we
		   don't.
		 */
		if (access_type == SPARE_ACCESS) 
        {
			cmd = MODE_01 | addr;
			writel(cmd, denali->flash_mem);
		} 
        else 
        {
			index_addr(denali, (uint32_t)cmd,
					0x2000 | op | page_count);

			/* wait for command to be accepted
			 * can always use status0 bit as the
			 * mask is identical for each
			 * bank. */
			status_type = wait_for_ready(denali, status_mask);

			if (status_type == 0) 
            {
				debug("cmd, page, addr on timeout "
						"(0x%x, 0x%x, 0x%x)\n",
						cmd, denali->page, addr);
				status = FAIL;
			} 
            else
            {
				cmd = MODE_01 | addr;
				writel(cmd, denali->flash_mem);
			}
		}
	}
	return status;
}

/* helper function that simply writes a buffer to the flash */
static int write_data_to_flash_mem(struct denali_nand_info *denali,
							const uint8_t *buf,
							int len)
{
	uint32_t i = 0, *buf32;

	/* verify that the len is a multiple of 4. see comment in
	 * read_data_from_flash_mem() */
	BUG_ON((len % 4) != 0);

	/* write the data to the flash memory */
	buf32 = (uint32_t *)buf;
	for (i = 0; i < len / 4; i++)
		writel(*buf32++, denali->flash_mem + 0x10);
	return i*4; /* intent is to return the number of bytes read */
}

/* helper function that simply reads a buffer from the flash */
static int read_data_from_flash_mem(struct denali_nand_info *denali,
								uint8_t *buf,
								int len)
{
	uint32_t i = 0, *buf32;

	/* we assume that len will be a multiple of 4, if not
	 * it would be nice to know about it ASAP rather than
	 * have random failures...
	 * This assumption is based on the fact that this
	 * function is designed to be used to read flash pages,
	 * which are typically multiples of 4...
	 */

	BUG_ON((len % 4) != 0);

	/* transfer the data from the flash */
	buf32 = (uint32_t *)buf;
	for (i = 0; i < len / 4; i++)
		*buf32++ = readl(denali->flash_mem + 0x10);
	return i*4; /* intent is to return the number of bytes read */
}

/* writes OOB data to the device */
static int write_oob_data(struct mtd_info *mtd, uint8_t *buf, int page)
{
	struct denali_nand_info *denali = g_denali;
	uint32_t status_type = 0;
	uint32_t status_mask = INTR_STATUS__PROGRAM_COMP |
						INTR_STATUS__PROGRAM_FAIL;
	int status = 0, addr = 0x0, cmd = 0x0;

	denali->page = page;

    /* Modified by zhouqi for xxx, 2013/09/03 */
	if (denali_send_pipeline_cmd(denali, false, true, SPARE_ACCESS,
							DENALI_WRITE) == PASS) 
    {
		write_data_to_flash_mem(denali, buf, mtd->oobsize);

		/* wait for operation to complete */
		status_type = wait_for_ready(denali, status_mask);

		if (status_type & INTR_STATUS__PROGRAM_FAIL)    //zhouqi
        {
			debug("OOB write failed\n");
			status = -1;
		}
	} 
    else 
    {
		debug("unable to send pipeline command\n");
		status = -1;
	}

    /* Added by zhouqi for xxx, 2013/09/03 */
    /* We set the device back to MAIN_ACCESS here as I observed
	* instability with the controller if you do a block erase
	* and the last transaction was a SPARE_ACCESS. Block erase
	* is reliable (according to the MTD test infrastructure)
	* if you are in MAIN_ACCESS.
	*/
	addr = BANK(denali->flash_bank) | denali->page;
	cmd = MODE_10 | addr;
	index_addr(denali, (uint32_t)cmd, MAIN_ACCESS);
    /* End added. zhouqi, 2013/09/03 */
        
	return status;
}

/* reads OOB data from the device */
static void read_oob_data(struct mtd_info *mtd, uint8_t *buf, int page)
{
	struct denali_nand_info *denali = g_denali;
	uint32_t status_mask = INTR_STATUS__LOAD_COMP | INTR_STATUS__TIME_OUT,  //zhouqi
			 status_type = 0, addr = 0x0, cmd = 0x0;

	denali->page = page;

	if (denali_send_pipeline_cmd(denali, false, true, SPARE_ACCESS,
							DENALI_READ) == PASS) 
    {
		read_data_from_flash_mem(denali, buf, mtd->oobsize);

		/* wait for command to be accepted
		 * can always use status0 bit as the mask is identical for each
		 * bank. */
		status_type = wait_for_ready(denali, status_mask);

		if (status_type & INTR_STATUS__TIME_OUT)
			debug("page on OOB timeout %d\n",denali->page);

		/* We set the device back to MAIN_ACCESS here as I observed
		 * instability with the controller if you do a block erase
		 * and the last transaction was a SPARE_ACCESS. Block erase
		 * is reliable (according to the MTD test infrastructure)
		 * if you are in MAIN_ACCESS.
		 */
		addr = BANK(denali->flash_bank) | denali->page;
		cmd = MODE_10 | addr;
		index_addr(denali, (uint32_t)cmd, MAIN_ACCESS);
	}
}

/* this function examines buffers to see if they contain data that
 * indicate that the buffer is part of an erased region of flash.
 */
uint32_t is_erased(uint8_t *buf, int len)
{
	int i = 0;
	for (i = 0; i < len; i++)
		if (buf[i] != 0xFF)
			return false;
	return true;
}
#define ECC_SECTOR_SIZE 512

#define ECC_SECTOR(x)	(((x) & ECC_ERROR_ADDRESS__SECTOR_NR) >> 12)
#define ECC_BYTE(x)	(((x) & ECC_ERROR_ADDRESS__OFFSET))
#define ECC_CORRECTION_VALUE(x) ((x) & ERR_CORRECTION_INFO__BYTEMASK)
#define ECC_ERROR_CORRECTABLE(x) (!((x) & ERR_CORRECTION_INFO__ERROR_TYPE))
#define ECC_ERR_DEVICE(x)	(((x) & ERR_CORRECTION_INFO__DEVICE_NR) >> 8)
#define ECC_LAST_ERR(x)		((x) & ERR_CORRECTION_INFO__LAST_ERR_INFO)

static uint32_t handle_ecc(struct denali_nand_info *denali, uint8_t *buf,
					uint32_t status_type)
{
	int check_erased_page = 0;
	uint32_t err_correction_value = 0;
	uint32_t err_correction_info = 0;
	
#if 1
    /*zx297520 use*/
	if (status_type & INTR_STATUS__ECC_ERR) 
	{
        check_erased_page = 1;
		
	}
	else
	{
	    
		switch(denali->flash_bank)
		{
		    case 0:
			err_correction_info = readl(denali->flash_reg +ERR_CORRECTION_INFO_B01);
			err_correction_value = err_correction_info & ERR_CORRECTION_INFO_B01__MAX_ERRORS_B0;
			break;

			case 1:
			err_correction_info = readl(denali->flash_reg +ERR_CORRECTION_INFO_B01);
			err_correction_value = (err_correction_info & ERR_CORRECTION_INFO_B01__MAX_ERRORS_B1)>>8;
			break;

			case 2:
			err_correction_info = readl(denali->flash_reg +ERR_CORRECTION_INFO_B23);
			err_correction_value = err_correction_info & ERR_CORRECTION_INFO_B01__MAX_ERRORS_B2;
			break;

			case 3:
			err_correction_info = readl(denali->flash_reg +ERR_CORRECTION_INFO_B23);
			err_correction_value = (err_correction_info & ERR_CORRECTION_INFO_B01__MAX_ERRORS_B3)>>8;
			break;

			default:
			break;
				
		}
		if(err_correction_value)
		    printk("correct %d bit errors on page %x.\n",err_correction_value,denali->page);
		 
				
	}
	return check_erased_page;
#else
		/* read the ECC errors. we'll ignore them for now */
		uint32_t err_address = 0, err_correction_info = 0;
		uint32_t err_byte = 0, err_sector = 0, err_device = 0;
		uint32_t err_correction_value = 0;
		denali_set_intr_modes(denali, false);

		do 
        {
			err_address = readl(denali->flash_reg +
						ECC_ERROR_ADDRESS);
			err_sector = ECC_SECTOR(err_address);
			err_byte = ECC_BYTE(err_address);

			err_correction_info = readl(denali->flash_reg +
						ERR_CORRECTION_INFO);
			err_correction_value =
				ECC_CORRECTION_VALUE(err_correction_info);
			err_device = ECC_ERR_DEVICE(err_correction_info);

			if (ECC_ERROR_CORRECTABLE(err_correction_info)) 
            {
				/* If err_byte is larger than ECC_SECTOR_SIZE,
				 * means error happened in OOB, so we ignore
				 * it. It's no need for us to correct it
				 * err_device is represented the NAND error
				 * bits are happened in if there are more
				 * than one NAND connected.
				 * */
				if (err_byte < ECC_SECTOR_SIZE) 
                {
					int offset;
					offset = (err_sector *
							ECC_SECTOR_SIZE +
							err_byte) *
							denali->devnum +
							err_device;
					/* correct the ECC error */
					buf[offset] ^= err_correction_value;
					denali->mtd->ecc_stats.corrected++;
				}
			} else {
				/* if the error is not correctable, need to
				 * look at the page to see if it is an erased
				 * page. if so, then it's not a real ECC error
				 * */
				check_erased_page = true;
			}
		} while (!ECC_LAST_ERR(err_correction_info));
		/* Once handle all ecc errors, controller will triger
		 * a ECC_TRANSACTION_DONE interrupt, so here just wait
		 * for a while for this interrupt
		 * */
		while (!(read_interrupt_status(denali) &
				INTR_STATUS__ECC_ERR))
		clear_interrupts(denali);
		denali_set_intr_modes(denali, false);		
    #endif
	}
    



/* programs the controller to either enable/disable DMA transfers */
static void denali_enable_dma(struct denali_nand_info *denali, uint32_t en)
{
	uint32_t reg_val = 0x0;

	if (en)
		reg_val = DMA_ENABLE__FLAG;

	writel(reg_val, denali->flash_reg + DMA_ENABLE);
	readl(denali->flash_reg + DMA_ENABLE);
}

/* setups the HW to perform the data DMA */
static void denali_setup_dma(struct denali_nand_info *denali, int op)
                                        
{
	uint32_t mode = 0x0;
	const int page_count = 1;
	dma_addr_t addr = denali->buf.dma_buf; 

	mode = MODE_10 | BANK(denali->flash_bank);

	/* DMA is a four step process */

	/* 1. setup transfer type and # of pages */
	index_addr(denali, mode | denali->page, 0x2000 | op | page_count);

	/* 2. set memory high address bits 23:8 */
	index_addr(denali, mode | ((uint16_t)(addr >> 16) << 8), 0x2200);

	/* 3. set memory low address bits 23:8 */
	index_addr(denali, mode | ((uint16_t)addr << 8), 0x2300);

	/* 4.  interrupt when complete, burst len = 64 bytes*/
    //writel(NAND_BASE, MODE_10|0x10000|(4 << 8));//BurstLength =4
	index_addr(denali, mode | 0x14000, 0x2400);     //zhouqi not interrupt 0X40
}
static void denali_setup_dma_buffer(struct denali_nand_info *denali, int op,char* buffer)
                                        
{
	uint32_t mode = 0x0;
	const int page_count = 1;
	dma_addr_t addr = buffer; 

	mode = MODE_10 | BANK(denali->flash_bank);

	/* DMA is a four step process */

	/* 1. setup transfer type and # of pages */
	index_addr(denali, mode | denali->page, 0x2000 | op | page_count);

	/* 2. set memory high address bits 23:8 */
	index_addr(denali, mode | ((uint16_t)(addr >> 16) << 8), 0x2200);

	/* 3. set memory low address bits 23:8 */
	index_addr(denali, mode | ((uint16_t)addr << 8), 0x2300);

	/* 4.  interrupt when complete, burst len = 64 bytes*/
    //writel(NAND_BASE, MODE_10|0x10000|(4 << 8));//BurstLength =4
	index_addr(denali, mode | 0x14000, 0x2400);     //zhouqi not interrupt 0X40
}



/* add by zhouqi */
#if 0
static void denali_setup_dma_derect(struct denali_nand_info *denali, int op,
                                        dma_addr_t addr)    /* add by zhouqi */
{
	uint32_t mode = 0x0;
	const int page_count = 1;
//	dma_addr_t addr = denali->buf.dma_buf; 

	mode = MODE_10 | BANK(denali->flash_bank);

	/* DMA is a four step process */

	/* 1. setup transfer type and # of pages */
	index_addr(denali, mode | denali->page, 0x2000 | op | page_count);

	/* 2. set memory high address bits 23:8 */
	index_addr(denali, mode | ((uint16_t)(addr >> 16) << 8), 0x2200);

	/* 3. set memory low address bits 23:8 */
	index_addr(denali, mode | ((uint16_t)addr << 8), 0x2300);

	/* 4.  interrupt when complete, burst len = 64 bytes*/
    //writel(NAND_BASE, MODE_10|0x10000|(4 << 8));//BurstLength =4
	index_addr(denali, mode | 0x14000, 0x2400);     //zhouqi not interrupt 0X40
}
#endif

/* writes a page. user specifies type, and this function handles the
 * configuration details. */
static void write_page(struct mtd_info *mtd, struct nand_chip *chip,
			const uint8_t *buf, uint32_t raw_xfer)
{
	struct denali_nand_info *denali = g_denali;

	uint32_t status_type = 0;
	uint32_t status_mask = INTR_STATUS__DMA_CMD_COMP |
						INTR_STATUS__PROGRAM_FAIL;

	/* if it is a raw xfer, we want to disable ecc, and send
	 * the spare area.
	 * !raw_xfer - enable ecc
	 * raw_xfer - transfer spare
	 */
	setup_ecc_for_xfer(denali, !raw_xfer, raw_xfer);

	/* copy buffer into DMA buffer */
	memcpy((void*)denali->buf.dma_buf, (void*)buf, mtd->writesize);

	if (raw_xfer) 
    	{
		/* transfer the data to the spare area */
		memcpy((void*)(denali->buf.dma_buf + mtd->writesize),
			(void*)chip->oob_poi,mtd->oobsize);
	}
      
	denali_enable_dma(denali, true);

	denali_setup_dma(denali, DENALI_WRITE); //zhouqi p779-p789

	/* wait for operation to complete */
	status_type = wait_for_ready(denali, status_mask);

	if (status_type & INTR_STATUS__TIME_OUT) 
    {
		debug("timeout on write_page (type = %d)\n",
				raw_xfer);
		denali->status =
			(status_type & INTR_STATUS__PROGRAM_FAIL) ?
			NAND_STATUS_FAIL : PASS;
	}

	denali_enable_dma(denali, false);
    
    setup_ecc_the_end(denali, false);   //zhouqi
    
}

//add by zhouqi
static void write_page_ops(struct mtd_info *mtd, struct nand_chip *chip,
			const uint8_t *buf)
{
	int N,len,sector_size,ecc_bytes,i;
	struct denali_nand_info *denali = g_denali;

	uint32_t status_type = 0;
	uint32_t status_mask = INTR_STATUS__DMA_CMD_COMP |
						INTR_STATUS__PROGRAM_FAIL;

	/* if it is a raw xfer, we want to disable ecc, and send
	 * the spare area.
	 * !raw_xfer - enable ecc
	 * raw_xfer - transfer spare
	 */
	setup_ecc_for_xfer(denali, 1, 1);
	memset((void *)(denali->buf.dma_buf),0xff,denali->mtd->writesize+denali->mtd->oobsize);

	sector_size = denali->nand->ecc.size;
    ecc_bytes = denali->nand->ecc.bytes;
    N = denali->mtd->writesize/(sector_size+ecc_bytes) + 1;
	len = sector_size;

    for(i=0;i < N;i++)
	{
		if(i==N-1)
		{
			len = denali->mtd->writesize - (sector_size+ecc_bytes)*i;
		}
	
	   memcpy((void *)(denali->buf.dma_buf+(sector_size+ecc_bytes)*i), (void *)(buf+sector_size*i), len);  ;
		
	}
	
	len = sector_size - len;
	
    memcpy((void *)(denali->buf.dma_buf + denali->mtd->writesize+2), (void *)(buf + sector_size*i -len), len); 
	memcpy((void *)(denali->buf.dma_buf+denali->mtd->writesize+2+len+ecc_bytes), (void *)(chip->oob_poi + 2+len+ecc_bytes), \
	 		denali->mtd->oobsize-2-len-ecc_bytes);  

	denali_enable_dma(denali, true);

	denali_setup_dma(denali, DENALI_WRITE);

	/* wait for operation to complete */
	status_type = wait_for_ready(denali, status_mask);

	if (status_type & INTR_STATUS__TIME_OUT) 
    {
		debug("timeout on write_page (type = )\n");
		denali->status =
			(status_type & INTR_STATUS__PROGRAM_FAIL) ?
			NAND_STATUS_FAIL : PASS;
	}

	denali_enable_dma(denali, false);
	setup_ecc_the_end(denali, false);   //zhouqi
}

/* NAND core entry points */

/* this is the callback that the NAND core calls to write a page. Since
 * writing a page with ECC or without is similar, all the work is done
 * by write_page above.
 * */
static void denali_write_page(struct mtd_info *mtd, struct nand_chip *chip,
				const uint8_t *buf, struct mtd_oob_ops *ops)//zhouqi add ops
{
	struct denali_nand_info *denali = g_denali;
	/* for regular page writes, we let HW handle all the ECC
	 * data written to the device. */
	int ecc_bits = readl(denali->flash_reg + ECC_CORRECTION);
	if((ops->oobbuf != NULL) && (ops->ooblen != 0))
	{
        write_page_ops(mtd, chip, buf);
	}
    else
    {
		if(denali->page < 64) 
		{
			writel(0x8, denali->flash_reg + ECC_CORRECTION); 
		}
	    write_page(mtd, chip, buf, false);
		writel(ecc_bits, denali->flash_reg + ECC_CORRECTION); 
    }
	update_led_twinkle();
}

/* This is the callback that the NAND core calls to write a page without ECC.
 * raw access is similar to ECC page writes, so all the work is done in the
 * write_page() function above.
 */
static void denali_write_page_raw(struct mtd_info *mtd, struct nand_chip *chip,
					const uint8_t *buf)
{
	/* for raw page writes, we want to disable ECC and simply write
	   whatever data is in the buffer. */
	write_page(mtd, chip, buf, true);
}

static int denali_write_oob(struct mtd_info *mtd, struct nand_chip *chip,
			    int page)
{
	//printk(KERN_EMERG "[denali.c]  denali_write_oob: page = 0x%0x\n",page); //zhouqi
	return write_oob_data(mtd, chip->oob_poi, page);
}

static int denali_read_oob(struct mtd_info *mtd, struct nand_chip *chip,
			   int page, int sndcmd)
{
    //printk(KERN_EMERG "[denali.c]  denali_read_oob: page = 0x%0x\n",page); //zhouqi
	read_oob_data(mtd, chip->oob_poi, page);

	return 0; /* notify NAND core to send command to
			   NAND device. */
}


static int read_page(struct mtd_info *mtd, struct nand_chip *chip,
			    uint8_t *buf, int page)
{
	struct denali_nand_info *denali = g_denali;

	uint32_t status_type = 0;
//	uint32_t status_mask = INTR_STATUS__ECC_ERR |
//			    INTR_STATUS__ECC_ERR;
    uint32_t status_mask = INTR_STATUS__DMA_CMD_COMP;
	uint32_t check_erased_page = false;

	if (page != denali->page) 
    {
		debug("IN %s: page %d is not"
				" equal to denali->page %d, investigate!!",
				__func__, page, denali->page);
		BUG();
	}

	setup_ecc_for_xfer(denali, true, false); 

	denali_enable_dma(denali, true);
	
	if(flash_dmabuf_disable_flag == 1)
	{
		denali_setup_dma(denali, DENALI_READ);
		/* wait for operation to complete */
		status_type = wait_for_ready(denali, status_mask);
		memcpy(buf, (void *)denali->buf.dma_buf, mtd->writesize);   //zhouqi -p939
	}
	else
	{
		denali_setup_dma_buffer(denali, DENALI_READ,buf); //zhouqi
		/* wait for operation to complete */
		status_type = wait_for_ready(denali, status_mask);
	}

	check_erased_page = handle_ecc(denali, buf, status_type);
	//check_erased_page = 0;
	denali_enable_dma(denali, false);
    setup_ecc_the_end(denali, false);   //zhouqi

	if (check_erased_page) 
    {
		read_oob_data(denali->mtd, chip->oob_poi, denali->page);
		
		if (!is_erased(buf, denali->mtd->writesize))
			denali->mtd->ecc_stats.failed++;
		if (!is_erased(chip->oob_poi, denali->mtd->oobsize))
			denali->mtd->ecc_stats.failed++;
	}
	return 0;
}


//add by zhouqi
static int read_page_ops(struct mtd_info *mtd, struct nand_chip *chip,
			    uint8_t *buf, int page)
{
	int N,len,sector_size,ecc_bytes,i;
    struct denali_nand_info *denali = g_denali;

	uint32_t status_type = 0;
    uint32_t status_mask = INTR_STATUS__DMA_CMD_COMP;
	uint32_t check_erased_page = false;
    
	if (page != denali->page) 
    {
		debug("IN %s: page %d is not"
				" equal to denali->page %d, investigate!!",
				__func__, page, denali->page);
		BUG();
	}

	setup_ecc_for_xfer(denali, 1, 1);

	denali_enable_dma(denali, true);

	denali_setup_dma(denali, DENALI_READ);

	/* wait for operation to complete */
	status_type = wait_for_ready(denali, status_mask);

    sector_size = denali->nand->ecc.size;
    ecc_bytes = denali->nand->ecc.bytes;
    N = denali->mtd->writesize/(sector_size+ecc_bytes) + 1;
	len = sector_size;

    for(i=0;i < N;i++)
	{
		if(i==N-1)
		{
			len = denali->mtd->writesize - (sector_size+ecc_bytes)*i;
		}
	
	   memcpy((void *)(buf+sector_size*i),(void *)( denali->buf.dma_buf + (sector_size+ecc_bytes)*i),len);
		
	}
	
	len = sector_size - len;
	memcpy((void *)(buf + sector_size*(N-1)+len), (void *)(denali->buf.dma_buf + denali->mtd->writesize +2), len);   
   
	memset((void *)(chip->oob_poi), 0xFF, len +ecc_bytes+2);
	memcpy((void *)(chip->oob_poi + len+ecc_bytes+2), (void *)(denali->buf.dma_buf +  denali->mtd->writesize+len+ecc_bytes+2),\
		   denali->mtd->oobsize-len -ecc_bytes-2);

	check_erased_page = handle_ecc(denali, buf, status_type);
	//check_erased_page = 0;
	denali_enable_dma(denali, false);
    setup_ecc_the_end(denali, false);   //zhouqi

	if (check_erased_page) 
    {
		read_oob_data(denali->mtd, chip->oob_poi, denali->page);

		/* check ECC failures that may have occurred on erased pages */
		if (!is_erased(buf, denali->mtd->writesize))
			denali->mtd->ecc_stats.failed++;
		if (!is_erased(chip->oob_poi, denali->mtd->oobsize))
			denali->mtd->ecc_stats.failed++;
	}
	return 0;
}

//add by zhouqi
static int denali_read_page(struct mtd_info *mtd, struct nand_chip *chip,
			    uint8_t *buf, int page, struct mtd_oob_ops *ops)//add by zhouqi
{

	struct denali_nand_info *denali = g_denali;

	int ecc_bits = readl(denali->flash_reg + ECC_CORRECTION);
		
	if((ops->oobbuf != NULL) && ops->ooblen != 0)
	{
        //denali_debug("[denali.c]:  read_page_ops: page = 0x%0x\n", page);//zhouqi
        read_page_ops(mtd, chip, buf, page);
	}
    else
    {
    	if(denali->page < 64) 
		{
			writel(0x8, denali->flash_reg + ECC_CORRECTION); 
		}
        //denali_debug("[denali.c]:  read_page: page = 0x%0x\n", page);//zhouqi
        read_page(mtd, chip, buf, page);
		writel(ecc_bits, denali->flash_reg + ECC_CORRECTION); 
    }
	update_led_twinkle();
    return 0;
}

static int denali_read_page_raw(struct mtd_info *mtd, struct nand_chip *chip,
				uint8_t *buf, int page)
{
	struct denali_nand_info *denali = g_denali;

	uint32_t status_type = 0;
	uint32_t status_mask = INTR_STATUS__DMA_CMD_COMP;

	if (page != denali->page) 
    {
		debug("IN %s: page %d is not"
				" equal to denali->page %d, investigate!!",
				__func__, page, denali->page);
		BUG();
	}

	setup_ecc_for_xfer(denali, false, true);
	denali_enable_dma(denali, true);

	denali_setup_dma(denali, DENALI_READ);

	/* wait for operation to complete */
	status_type = wait_for_ready(denali, status_mask);

	denali_enable_dma(denali, false);

	memcpy(buf, (void*)denali->buf.dma_buf, mtd->writesize);
	memcpy(chip->oob_poi, (void*)(denali->buf.dma_buf + mtd->writesize), mtd->oobsize);

	return 0;
}

static uint8_t denali_read_byte(struct mtd_info *mtd)
{
	struct denali_nand_info *denali = g_denali;
	uint8_t result = 0xff;

	if (denali->buf.head < denali->buf.tail)
		result = denali->buf.buf[denali->buf.head++];

	return result;
}

static uint16_t denali_read_word(struct mtd_info *mtd)
{
    struct nand_chip *chip = mtd->priv;
	uint16_t result = 0x0;

    result = (uint16_t)(*(chip->oob_poi));
    result = result << 8;
    result |= (uint16_t)(*(chip->oob_poi + 1));
    
	return result;
}

static void denali_select_chip(struct mtd_info *mtd, int chip)
{
	struct denali_nand_info *denali = g_denali;

	denali->flash_bank = chip;
}

static int denali_waitfunc(struct mtd_info *mtd, struct nand_chip *chip)
{
	struct denali_nand_info *denali = g_denali;
	int status = denali->status;
	denali->status = 0;

	return status;
}

static void denali_erase(struct mtd_info *mtd, int page)
{
    #if ECC_TEST_VER
	return;
	#endif
	struct denali_nand_info *denali = g_denali;

	uint32_t cmd = 0x0, status_type = 0;

	/* setup page read request for access type */
	cmd = MODE_10 | BANK(denali->flash_bank) | page;
	index_addr(denali, (uint32_t)cmd, 0x1);

	/* wait for erase to complete or failure to occur */
	status_type = wait_for_ready(denali, INTR_STATUS__ERASE_COMP |
					INTR_STATUS__ERASE_FAIL);

	denali->status = (status_type & INTR_STATUS__ERASE_FAIL) ?
						NAND_STATUS_FAIL : PASS;
	update_led_twinkle();
}

static void denali_cmdfunc(struct mtd_info *mtd, unsigned int cmd, int col,
			   int page)
{
	struct denali_nand_info *denali = g_denali;
	uint32_t addr, id;
	int i;
    
	switch (cmd) {
	case NAND_CMD_PAGEPROG:
		break;
	case NAND_CMD_STATUS:
        reset_buf(denali);
		read_status(denali);
		break;
	case NAND_CMD_READID:
	case NAND_CMD_PARAM:
		reset_buf(denali);
		/*sometimes ManufactureId read from register is not right
		 * e.g. some of Micron MT29F32G08QAA MLC NAND chips
		 * So here we send READID cmd to NAND insteand
		 * */
		addr = (uint32_t)MODE_11 | BANK(denali->flash_bank);
		index_addr(denali, (uint32_t)addr | 0, 0x90);
		index_addr(denali, (uint32_t)addr | 1, 0);
		for (i = 0; i < 5; i++) 
        {
			index_addr_read_data(denali,(uint32_t)addr | 2,	&id);
			write_byte_to_buf(denali, id);
		}
		break;
	case NAND_CMD_READ0:
	case NAND_CMD_SEQIN:
		denali->page = page;
		break;
	case NAND_CMD_RESET:
		reset_bank(denali);
		break;
	case NAND_CMD_READOOB:
        reset_buf(denali);      //zhouqi
        denali_read_oob(mtd, mtd->priv, page, 0);
		/* TODO: Read OOB data */
		break;
	default:
		debug(": unsupported command"
				" received 0x%x\n", cmd);
		break;
	}
}

/* stubs for ECC functions not used by the NAND core */
static int denali_ecc_calculate(struct mtd_info *mtd, const uint8_t *data,
				uint8_t *ecc_code)
{
	debug("denali_ecc_calculate called unexpectedly\n");
	BUG();
	return -1;
}

static int denali_ecc_correct(struct mtd_info *mtd, uint8_t *data,
				uint8_t *read_ecc, uint8_t *calc_ecc)
{
	debug("denali_ecc_correct called unexpectedly\n");
	BUG();
	return -1;
}

static void denali_ecc_hwctl(struct mtd_info *mtd, int mode)
{
	debug("denali_ecc_hwctl called unexpectedly\n");
	BUG();
}
/* end NAND core entry points */


/* Initialization code to bring the device up to a known good state */
static void denali_hw_init(struct denali_nand_info *denali)
{
    uint32_t id_bytes[5], addr;
	uint8_t i, maf_id, device_id, res_id;
    struct nand_flash_device_para * table = nand_flash_para;

    denali->flash_reg = (void __iomem *)NAND_FLASH_REG;
	denali->flash_mem = (void __iomem *)NAND_FLASH_MEM;

    detect_max_banks(denali);       /* test the max banks support*/
    denali_nand_reset(denali);
    writel(0, denali->flash_reg + DMA_ENABLE);      /* dma disable */
    writel(0, denali->flash_reg + ECC_ENABLE);      /* ecc disable */
	writel(2, denali->flash_reg + SPARE_AREA_SKIP_BYTES);   
	denali->bbtskipbytes = readl(denali->flash_reg +
						SPARE_AREA_SKIP_BYTES);	
	writel(0x0F, denali->flash_reg + RB_PIN_ENABLED);
	writel(0, denali->flash_reg + CHIP_ENABLE_DONT_CARE);
	writel(1, denali->flash_reg + DEVICES_CONNECTED);
	

	writel(0xffff, denali->flash_reg + SPARE_AREA_MARKER);


    /* Use read id method to get device ID and other */
	addr = (uint32_t)MODE_11 | BANK(denali->flash_bank);
	index_addr(denali, (uint32_t)addr | 0, 0x90);
	index_addr(denali, (uint32_t)addr | 1, 0);
	for (i = 0; i < 5; i++)
		index_addr_read_data(denali, addr | 2, &id_bytes[i]);
	maf_id = id_bytes[0];
	device_id = id_bytes[1];
    res_id = id_bytes[2];

    for (; table->manuf_id != 0; table++)
    {
        if ((maf_id == table->manuf_id) && (device_id == table->device_id) 
                                        && (res_id == table->res_id) )
        {
            break;
        }		
    }
    g_nand_dev_info = table;
    printf("maf_id=%x,dev_id=%x,res_id=%x\n",maf_id,device_id,res_id);   
	/* Should set value for these registers when init */ 
    denali_nand_register_set(denali, table);
	denali_nand_timing_set(denali, table);
    //find_valid_banks(denali); //zhouqi for 7520 fpga
	//detect_partition_feature(denali);
	denali_irq_init(denali);

}

/* Althogh controller spec said SLC ECC is forceb to be 4bit,
 * but denali controller in MRST only support 15bit and 8bit ECC
 * correction
 * */

static struct nand_ecclayout g_nand_oob;

static uint8_t bbt_pattern[] = {'B', 'b', 't', '0' };
static uint8_t mirror_pattern[] = {'1', 't', 'b', 'B' };

static struct nand_bbt_descr bbt_main_descr = {
	.options = NAND_BBT_LASTBLOCK | NAND_BBT_CREATE | NAND_BBT_WRITE
		| NAND_BBT_2BIT | NAND_BBT_VERSION | NAND_BBT_PERCHIP,
	.offs =	8,
	.len = 4,
	.veroffs = 12,
	.maxblocks = 4,
	.pattern = bbt_pattern,
};

static struct nand_bbt_descr bbt_mirror_descr = {
	.options = NAND_BBT_LASTBLOCK | NAND_BBT_CREATE | NAND_BBT_WRITE
		| NAND_BBT_2BIT | NAND_BBT_VERSION | NAND_BBT_PERCHIP,
	.offs =	8,
	.len = 4,
	.veroffs = 12,
	.maxblocks = 4,
	.pattern = mirror_pattern,
};

/* initialize driver data structures */
void denali_drv_init(struct denali_nand_info *denali)
{
	/* indicate that MTD has not selected a valid bank yet */
	denali->flash_bank = CHIP_SELECT_INVALID;

	/* initialize our status_type variable to indicate no interrupts */
}

static int denali_nand_ecc_init(struct denali_nand_info  *denali)
{
    int i,eccpos_start;
    denali->nand->ecc.mode = NAND_ECC_HW_SYNDROME;
	denali->nand->ecc.size = g_nand_dev_info->ecc.sector_size;
    denali->nand->ecc.steps = g_nand_dev_info->page_size/g_nand_dev_info->ecc.sector_size;
    denali->nand->ecc.strength = g_nand_dev_info->ecc.strength;
	
	switch (denali->nand->ecc.size) {
	case 512:
		denali->nand->ecc.bytes =
			( denali->nand->ecc.strength  * 13 + 15) / 16 * 2;
		break;
	case 1024:
		denali->nand->ecc.bytes =
			( denali->nand->ecc.strength  * 14 + 15) / 16 * 2;
		break;
	default:
		printk("Unsupported ECC sector size\n");

		BUG_ON(1);
		return -1;
	}

	denali->nand->ecc.total = denali->nand->ecc.bytes* denali->nand->ecc.steps;
	if(g_nand_dev_info->oob_size >= (denali->nand->ecc.total+denali->bbtskipbytes + 8))
	{
		
		writel(g_nand_dev_info->ecc.strength, denali->flash_reg + ECC_CORRECTION);
        g_nand_oob.eccbytes = denali->nand->ecc.total;
	
		eccpos_start = denali->bbtskipbytes;

		for (i = 0; i < g_nand_oob.eccbytes; i++)
		{
			g_nand_oob.eccpos[i] = eccpos_start + i;
		}

		g_nand_oob.oobfree[0].offset = g_nand_oob.eccbytes+denali->bbtskipbytes;
		g_nand_oob.oobfree[0].length = g_nand_dev_info->oob_size -(g_nand_oob.eccbytes+denali->bbtskipbytes);
		denali->nand->ecc.layout = &g_nand_oob;
	}
	else
	{
		printk("Unsupported ECC strength,please check the id table\n");
		BUG();
	}

	
	return 0;
}


/* driver entry point */
int board_nand_init_denali(struct nand_chip *nand)
{
	int ret = -1; 
	struct denali_nand_info *denali = g_denali;

	denali_hw_init(denali);
	denali_drv_init(denali);

	denali->mtd = (struct mtd_info *)&nand_info;
	denali->nand = (struct nand_chip *)&nand_chip;
	denali->buf.dma_buf = (dma_addr_t)CONFIG_NAND_DMA_BUF_ADDR;

	/* register the driver with the NAND core subsystem */
	denali->nand->select_chip = denali_select_chip;
	denali->nand->cmdfunc = denali_cmdfunc;
	denali->nand->read_byte = denali_read_byte;
	denali->nand->read_word = denali_read_word;
	denali->nand->waitfunc = denali_waitfunc;

	/* scan for NAND devices attached to the controller
	 * this is the first stage in a two step process to register
	 * with the nand subsystem */
	
	if (nand_scan_ident(denali->mtd, denali->total_used_banks, NULL)) 
	{
		debug("nand ident: cant ident this nand device");
		return -1;
	}

	/* MTD supported page sizes vary by kernel. We validate our
	 * kernel supports the device here.
	 */
	if (denali->mtd->writesize > NAND_MAX_PAGESIZE + NAND_MAX_OOBSIZE) 
	{
		ret = -1;
		debug("Spectra: device size not supported by this version of MTD.");
		return ret;
	}

	/* support for multi nand
	 * MTD known nothing about multi nand,
	 * so we should tell it the real pagesize
	 * and anything necessery
	 */
	denali->devnum = readl(denali->flash_reg + DEVICES_CONNECTED);
	denali->nand->chipsize <<= (denali->devnum - 1);
	denali->nand->page_shift += (denali->devnum - 1);
	denali->nand->pagemask = (denali->nand->chipsize >> denali->nand->page_shift) - 1;
	denali->nand->bbt_erase_shift += (denali->devnum - 1);
	denali->nand->phys_erase_shift = denali->nand->bbt_erase_shift;
	denali->nand->chip_shift += (denali->devnum - 1);
	denali->mtd->writesize <<= (denali->devnum - 1);
	denali->mtd->oobsize <<= (denali->devnum - 1);
	denali->mtd->erasesize <<= (denali->devnum - 1);
	denali->mtd->size = denali->nand->numchips * denali->nand->chipsize;
	denali->bbtskipbytes *= denali->devnum;

	/* second stage of the NAND scan
	 * this stage requires information regarding ECC and
	 * bad block management. */

	/* Bad block management */
	denali->nand->bbt_td = &bbt_main_descr;
	denali->nand->bbt_md = &bbt_mirror_descr;

	/* skip the scan for now until we have OOB read and write support */
	denali->nand->options |= NAND_USE_FLASH_BBT;//NAND_SKIP_BBTSCAN
	
	// init ecc
	 denali_nand_ecc_init(denali);
    
	/* Let driver know the total blocks number and
	 * how many blocks contained by each nand chip.
	 * blksperchip will help driver to know how many
	 * blocks is taken by FW.
	 * */
	denali->totalblks = denali->mtd->size >> denali->nand->phys_erase_shift;
	denali->blksperchip = denali->totalblks / denali->nand->numchips;

	/* These functions are required by the NAND core framework, otherwise,
	 * the NAND core will assert. However, we don't need them, so we'll stub
	 * them out. */
	denali->nand->ecc.calculate = denali_ecc_calculate;
	denali->nand->ecc.correct = denali_ecc_correct;
	denali->nand->ecc.hwctl = denali_ecc_hwctl;

	/* override the default read operations */
	denali->nand->ecc.read_page = denali_read_page;
	denali->nand->ecc.read_page_raw = denali_read_page_raw;
	denali->nand->ecc.write_page = denali_write_page;
	denali->nand->ecc.write_page_raw = denali_write_page_raw;
	denali->nand->ecc.read_oob = denali_read_oob;
	denali->nand->ecc.write_oob = denali_write_oob;
	denali->nand->erase_cmd = denali_erase;

	if (nand_scan_tail(denali->mtd)) 
	{
		ret = -1;
	}
	
	return 0;
}

