/*******************************************************************************
 * Copyright (C) 2007, ZTE Corporation.
 *
 * File Name:   uicc_api.h 
 * File Mark:    
 * Description:  Provide UICC module Function prototype declaration
*                and type declaration.
 * Others:        
 * Version:       1.0
 * Author:        wangxia
 * Date:          2008-4-18
 * History 1:      
 *     Date: 
 *     Version:
 *     Author: 
 *     Modification:  
 * History 2: 
  ********************************************************************************/

#ifndef _DRVS_UICC_H
#define _DRVS_UICC_H

/****************************************************************************
* 	                                        Include files
****************************************************************************/
#include "pub.h"

/****************************************************************************
* 	                                        Macros
****************************************************************************/
//#define SOC_USIM

#define DRV_UICC_NOF_CLASSES              5 /*!< The number of specified card classes (A,B,C,D & E) */
/****************************************************************************
* 	                                        Types
****************************************************************************/
typedef enum
{
  DRV_UICC_TRANSFER_SUCCEEDED,
  DRV_UICC_TRANSFER_FAILED
} T_ZDrvUicc_ApduReselt;

/**
 * \enum T_ZDrvUicc_ResetResult
 * Response cause for function \ref UICC_reset.
 */
typedef enum
{
  DRV_UICC_ACTIVATION_SUCCEEDED,
  DRV_UICC_ACTIVATION_FAILED,
  DRV_UICC_REJECT_CARD
} T_ZDrvUicc_ResetResult;

/**
 * \enum T_ZDrvUicc_CloseResult
 * Response cause for function \ref UICC_close.
 */
typedef enum
{
  DRV_UICC_DEACTIVATION_SUCCEEDED,
  DRV_UICC_DEACTIVATION_FAILED,
  DRV_UICC_ALREADY_DEACTIVATED
} T_ZDrvUicc_CloseResult;

/**
 * \enum T_ZDrvUicc_SetInOutCallBack
 * Response cause for callback function \ref UICC_set_inout_callback.
 */
typedef enum
{
  DRV_CALLBACK_FUNCTION_SET, /*!< Callback function registered. */
  DRV_CALLBACK_NOT_AVAILABLE /*!< Callback function NOT registered (or not supported). */
} T_ZDrvUicc_SetInOutCallBack;

/**
 * \enum T_ZDrvUicc_CardSelector
 * Card selector.
 */
typedef enum
{
  DRV_UICC_PRIMARY_CARD,
  DRV_UICC_SECONDARY_CARD_1,
  DRV_UICC_SECONDARY_CARD_2
} T_ZDrvUicc_CardSelector;

/**
 * \enum T_ZDrvUicc_CommandCase
 * Command case indicator.
 */
typedef enum
{
  DRV_UICC_CMD_CASE_1,    /*!< No Tx nor Tx command. */
  DRV_UICC_CMD_CASE_2,    /*!< Rx command.           */ 
  DRV_UICC_CMD_CASE_3,    /*!< Tx command.           */
  DRV_UICC_CMD_CASE_4     /*!< Tx and Rx command.    */
} T_ZDrvUicc_CommandCase;

/**
 * \enum T_ZDrvUicc_ResetMode
 * Reset indicator.
 */
typedef enum
{
  DRV_UICC_COLD_RESET,   /*!< Cold Reset. */
  DRV_UICC_WARM_RESET    /*!< Warm Reset (reset with RST line only). */
} T_ZDrvUicc_ResetMode;

/**
 * \enum T_ZDrvUicc_VoltageClass
 * Voltage CLASS selector.
 */
typedef enum
{
  DRV_UICC_CLASS_A,          /*!< 5v   */
  DRV_UICC_CLASS_B,          /*!< 3v   */
  DRV_UICC_CLASS_C,          /*!< 1.8v */
  DRV_UICC_CLASS_D,          /*!< RFU  */
  DRV_UICC_CLASS_E           /*!< RFU  */
}T_ZDrvUicc_VoltageClass;

/**
 * \enum T_ZDrvUicc_ClockStopMode
 * Clock stop level indicator.
 */
typedef enum
{
  DRV_UICC_CLOCK_STOP_ALLOWED,    /*!< Clock Stop NOT allowed. */
  DRV_UICC_NO_PREFERRED_LEVEL,    /*!< Clock allowed, no preferred level requiered. */
  DRV_UICC_HIGH_LEVEL_PREFERRED,  /*!< Clock allowed, High level preferred. */
  DRV_UICC_LOW_LEVEL_PREFERRED,   /*!< Clock allowed, Low level preferred. */
  DRV_UICC_CLOCK_STOP_NOT_ALLOWED,/*!< Clock Stop NOT allowed. */
  DRV_UICC_CLOCK_STOP_ONLY_HIGH,  /*!< Clock Stop NOT allowed, unless at High level. */
  DRV_UICC_CLOCK_STOP_ONLY_LOW    /*!< Clock Stop NOT allowed, unless at Low level. */
}T_ZDrvUicc_ClockStopMode;

/**
 * \enum T_ZDrvUicc_Protocol
 * Protocol selector.
 */
typedef enum
{
  DRV_UICC_T_0,           /*!< T=0 Protocol. */
  DRV_UICC_T_1            /*!< T=1 Protocol. */
}T_ZDrvUicc_Protocol;

/**
 * \enum T_ZDrvUicc_ComConvention
 * Communication convention indicator.
 */
typedef enum
{
  DRV_UICC_DIRECT,        /*!< Direct communication convention.  */
  DRV_UICC_INVERSE        /*!< Inverse communication convention. */
}T_ZDrvUicc_ComConvention;

/**
 * \enum T_ZDrvUicc_CardAction
 * Card In/Out indicator.
 */
typedef enum
{
  DRV_CARD_INSERTED,
  DRV_CARD_REMOVED
} T_ZDrvUicc_CardAction;
/**
 * \}
 */


 /**
 * \struct T_ZDrvUicc_ApduHeader
 * Structure holding the APDU command header. 
 */
typedef struct
{
  UINT8  cla;  /*!< Command CLAss.       */
  UINT8  ins;  /*!< Command INStruction. */
  UINT8  p1;   /*!< Command Parameter.   */
  UINT8  p2;   /*!< Command Parameter.   */
  UINT16 lc;   /*!< Tx-size. */
  UINT16 le;   /*!< Rx-Size. */
} T_ZDrvUicc_ApduHeader;


/**
 * \struct T_ZDrvUicc_ApduFooter
 * Structure holding the APDU command response 'header'. 
 */
typedef struct
{
  UINT16 luicc; /*!< Actual Rx-size recived. */
  UINT8  sw1;   /*!< Received Status Word SW1. */
  UINT8  sw2;   /*!< Received Status Word SW2. */
} T_ZDrvUicc_ApduFooter;


/**
 * \enum T_ZDrvUicc_CardType
 * card type. 
 */
typedef enum 
{
 DRV_UICC_CARD_TYPE_2G,
 DRV_UICC_CARD_TYPE_3G,

 UICC_MAX
 }T_ZDrvUicc_CardType;



/**
 * \struct T_ZDrvUicc_VoltageControl
 * Voltage control structure. 
 */
typedef struct
{
  T_ZDrvUicc_VoltageClass used_voltage;    /*!< Actual used voltage class/level. */
  BOOL                           voltage_settled; /*!< Indicates if the voltages level has 
                                             been settled. This can only happen 
                                             if the card did indicate the voltage 
                                             capabilities in the ATR string. */
} T_ZDrvUicc_VoltageControl;

/**
 * \struct T_ZDrvUicc_Characteristics
 * Protocol characteristics structure. 
 */
typedef struct
{
  T_ZDrvUicc_Protocol  protocol;       /*!< Protocol supported and used. */
  UINT8            f_used;         /*!< The used clock rate convention factor (F). */
  UINT8            f_offered;      /*!< The offered clock rate convention factor (F). */
  UINT8            d_used;         /*!< The used baud rate adjustment factor (D). */ 
  UINT8            d_offered;      /*!< The offered baud rate adjustment factor (D). */ 
  T_ZDrvUicc_ComConvention com_convention; /*!< The supported and used communication convention. */ 
} T_ZDrvUicc_Characteristics;

/**
 * \struct T_ZDrvUicc_ElectricalProfile
 * Electrical Baseband characteristics structure. 
 */
typedef struct
{
  BOOL    class_supported; /*!< Indicates if the class is supported or not. */
  UINT16  voltage_level;   /*!< Holds the voltage level given in millivolts [mV]. E.g. 1800 is 
                              equivalent to 1.8v). */
  UINT16  max_current;     /*!< The maximum allowed current consumption at this voltage_level 
                              given in micro Ampere (uA). E.g. 15500 is equivalent to 15.5mA. */
} T_ZDrvUicc_ElectricalProfile;

/**
 * \struct T_ZDrvUicc_ClockProfile
 * Baseband Clock characteristics. 
 */
typedef struct
{
  UINT8  min_clock_freq;   /*!< The minimum clock frequency supported by the HW. */
  UINT8  max_clock_freq;   /*!< The maximum clock frequency supported by the HW. If only 
                              one frequency is supported, the min. and max. values will 
                              be identical. The frequency resolution is 0.1 MHz i.e. 21h
                              is equivalent to 3.3MHz. */
}T_ZDrvUicc_ClockProfile;

/**
 * \struct T_ZDrvUicc_HwProfile
 * Baseband Clock characteristics. 
 */
typedef struct
{
  UINT8                     nof_card_slots_supported;  
                            /*!< Number of card reader slots supported by HW.\n
                             * 1: Only primary slot (uicc_primary_card)\n   
                             * 2: Primary and ONE additional slot (uicc_secondary_card_1)\n
                             * 3: Primary and TWO additional slots (uicc_secondary_card_1 & uicc_secondary_card_2) */ 

  UINT8                     protocol_supported;        
                            /*!< Protocols supported by the DRV_UICC Handler.\n
                             * b1: T=0 supported\n
                             * b2: T=1 supported\n 
                             * b3: T=x supported\n  
                             * E.g. 0x03 means that both T=0 and T=1 are supported */

  T_ZDrvUicc_ElectricalProfile electrical_profile[DRV_UICC_NOF_CLASSES]; 
                            /*!< Holds the electrical profile of the specified classes - where 
                             *   the max. allowed current consumption is indicated for each class. */

  T_ZDrvUicc_ClockProfile      clock_profile;
                            /*!< Holds the minimum and maximum clock frequencies supported by the 
                             * hardware. */  

  BOOL                      extended_length;           
                            /*!< Indicates the driver support of the use of extended Lc/Le.\n
                             * TRUE:  extended Lc/Le is supported by the driver. The max. Tx and 
                             *        Rx data sizes are 65,535 and 65,536 bytes respectively.\n
                             * FALSE: extended Lc/Le is NOT supported by the driver. The max. Tx 
                             *        and Rx data sizes are 255 and 256 bytes respectively.\n
                             * NOTE: this feature is currently NOT supported! */
} T_ZDrvUicc_HwProfile;

typedef enum {
  CARD_INSERTED,
  CARD_REMOVED
} T_UICC_CARD_ACTION;

typedef enum {
  UICC_PRIMARY_CARD,
  UICC_SECONDARY_CARD_1,
 // UICC_SECONDARY_CARD_2
} T_UICC_CARD_SELECTOR;

typedef enum {
  UICC_CARD,   //0
  VSIM_CARD,  //3
  UICC_VSIM_AUTO,
  UICC_ESIM_CARD,
  CLOUD_CARD, //1//1
  SOFTSIM_CARD, //2//2
  UICC_VSIM_MAX
} T_UICC_CARD_TYPE;
/****************************************************************************
* 	                                        Constants
****************************************************************************/

/****************************************************************************
* 	                                        Global  Variables
****************************************************************************/

/****************************************************************************
* 	                                        Function Prototypes
****************************************************************************/

/*!
 * Function used to exchange APDU with a card. The function will not return 
 * before the response APDU has beed received from the card or until the 
 * appropriate Working Timer has expired.
 *
 * \param   card_selector   (I)   Indicates which card the requested command is meant for.
 * \param   command_case    (I)   Indicates which of the four command cases the current 
 *                          command belongs to.
 * \param   extended_length (I)   Indicates the card support of the use of extended Lc/Le. 
 *                          The caller of this function has the responsibility of 
 *                          allocating the apdu_data store correcesponding to the 
 *                          larger of Lc and Le.\n
 *                          TRUE:  extended Lc/Le is supported. The max. Tx and Rx  
 *                                 data sizes are 65,535 and 65,536 bytes respectively.\n
 *                          FALSE: extended Lc/Le is NOT supported. The max. Tx and Rx  
 *                                 data sizes are 255 and 256 bytes respectively.
 * \param   c_apdu          (I)   The Command APDU containing: CLA, INS, P1, P2, Lc & Le.
 * \param   r_apdu          (O)   The Response APDU containing: Luicc, SW1 & SW2.
 * \param   apdu_data       (I/O) I: Holds Lc Tx-data to be sent to the card. 
 *                                O: Holds Luicc Rx-data received from the card.
 * \return The result of the APDU command execution.
 */ 

extern T_ZDrvUicc_ApduReselt zDrvUicc_TransportApdu(T_ZDrvUicc_CardSelector card_selector,
                  T_ZDrvUicc_CommandCase command_case,
                  BOOL                              extended_length,
                  T_ZDrvUicc_ApduHeader    c_apdu,
                  T_ZDrvUicc_ApduFooter     *r_apdu_ptr,
                  UINT8                             *apdu_data_ptr);


/*!
 * Function used to reset the card. The function will not return until the reset
 * has been successfully carried out i.e. ATR string has been received or if no 
 * response to the reset is received i.e. IWT times out.
 * \param      card_selector       (I) Indicates which card needs to be activated/reset.
 * \return     The outcome of the activation handling.
 */ 
 extern T_ZDrvUicc_ResetResult zDrvUicc_ResetCard(T_ZDrvUicc_CardSelector card_selector);


/*!
 * Function used to deactivat a given card.
 * \param    card_selector       (I) Indicates which card needs to be deactivated.
 * \return   The deactivation result.
 */
 extern T_ZDrvUicc_CloseResult zDrvUicc_Close(T_ZDrvUicc_CardSelector card_selector);

/*!
 * Function for Informing the DRV_UICC Handler of the card characteristics of the current used card.
 * \param        card_selector       (I) Indicates which card the characteristics are valid for.
 * \param        clock_stop_mode     (I) Holds the Clock Stop capabilities indicated by the card.
 * \param        min_clock_freq      (I) Holds the minimum acceptable clock frequency for the
 *                                   card/application. The frequency resolution is 0.1 MHz 
 *                                   i.e. 21h is equivalent to 3.3MHz. 
 * \return       None.
 */
 extern VOID zDrvUicc_CardCharacteristics(T_ZDrvUicc_CardSelector card_selector,
                                T_ZDrvUicc_ClockStopMode clock_stop_mode,
                                UINT8                               min_clock_freq);
/*!
 * Returns an overview of the static profile of the HW related interface.
 * \return        The static characteristics (e.g. number of card slots, voltage 
 *                levels, max. current cunsumption, supported transmission protocols 
 *                etc.
 */
 extern T_ZDrvUicc_HwProfile zDrvUicc_HwProfile(VOID);

/*!
 * Used for initializing the callback function used when the in/out state of the 
 * card changes.
 * \param         inout_call_back (I) Pointer to the function to be invoked when the 
 *                                in/out state changes. The parameter to the call 
 *                                back function shall indicated the related card 
 *                                reader and the IN or OUT action.
 * \return        The outcome of the requested action.
 */
 extern T_ZDrvUicc_SetInOutCallBack  zDrvUicc_SetInOutCallBack( VOID (*inout_call_back)(T_ZDrvUicc_CardAction card_action, T_ZDrvUicc_CardSelector card_selector));
 /*!
  * Used for initializing the callback function used when the in/out state of the 
  * card changes.
  * \param
  * \return 	   The outcome of the requested action.
  */
 extern T_ZDrvUicc_SetInOutCallBack  zDrvUicc_UninstallInOutCallBack(VOID);

/*!
 * Function used to obtain the ATR string of the current operated card. If no card 
 * is active (or if data is not available) the number of characters returned will 
 * be set to zero.
 * \param  atr    Pointer to the store in which the ATR string should be located.
 * \return        Number of ATR characters.
 */
 extern UINT8 zDrvUicc_GetAtr(T_ZDrvUicc_CardSelector card_selector, UINT8 *atr);


extern VOID zDrvUicc_Highisr(VOID);

extern void invoke_callback(T_UICC_CARD_ACTION card_action, T_UICC_CARD_SELECTOR card_selector);

typedef void (*recoverCallback_func)(void);
void uicc_setRecover_callback(recoverCallback_func func);

typedef UINT8 (*simStatuCallback_func)(void);
void uicc_getSimStatu_callback(simStatuCallback_func func);
#define UICC_SIM_PLUGIN			0x11
#define UICC_SIM_PLUGOUT		0x10

typedef void (*hotplugCallback_func)(UINT32);

void uicc_set_hotplugCallback(hotplugCallback_func func);

typedef void (*simSwitchCallback_fun)(void);
void uicc_setUsimSwitch_callback(simSwitchCallback_fun func);

SINT32 uiccramlog_Printf(const VOID *pFormat, ...);

extern SINT32 ramlog_Printf(UINT16 nModNo, const VOID *pFormat, ...);

#ifdef UICC_ENABLE_REALTIME_PRINT
#define zDrvUiccRamlog_PRINTF(nModNo, s...)  uiccramlog_Printf(s)
#else
#ifdef _USE_RAMLOG
#define zDrvUiccRamlog_PRINTF(nModNo, s...)  ramlog_Printf(nModNo, s)
#endif
#endif

#endif /* _DRVSUICC_H */
/** \} */
                                                 /* End of file.              */

