/*******************************************************************************
 * Copyright (C) 2007, ZTE Corporation.
 *
 * File Name:    
 * File Mark:    
 * Description:  This source file contains the T=1 protocol instruction statemachine engine.
 * Others:        
 * Version:       v0.1
 * Author:        mtr
 * Date:          2007-03-28
 * History 1:      
 *     Date: 
 *     Version:
 *     Author: 
 *     Modification:  
 * History 2: 
  ********************************************************************************/


/****************************************************************************
*                                              Include files
****************************************************************************/
#include "uicc_stub.h"
#include "uicc.h"
#include "uicc_ctrl.h"
#include "uicc_access.h"
#include "uicc_drv.h"
#include "uicc_trc.h"
    
#include "USIM_reg.h"


/****************************************************************************
* 	                                           Local Macros
****************************************************************************/
#define T_CHAR     200000 /* [us] The character transmission guard time. */
#define T_CLK_PU   2000   /* [us] The time waiting before initiating an instruction after
                                      enabling the clock. */
#define T_CLK_PD   1000   /* [us] The time waiting before disabling the clock after having
                                      executed an initiating. */
#define T_ERROR    20000  /* [us] The timeout period after an error has been detected. */
#define TX_SPC     1      /* [ETU] Distance between Tx block characters. */
    
#define T_FIFO_REQ 200000  /* [us] The time waiting before FIFO TX Request*/


/****************************************************************************
* 	                                           Local Types
****************************************************************************/
enum
{
  RX_PARITY_ERROR_NAD_POS = 0,
  RX_PARITY_ERROR_PCB_POS = 1,
  RX_PARITY_ERROR_LEN_POS = 2,
  RX_PARITY_ERROR_INF_POS = 3,
  RX_PARITY_ERROR_EDC_POS = 4,
  RX_PARITY_ERROR_NAD_OR_PCB_POS = 5
};


/****************************************************************************
* 	                                           Local Constants
****************************************************************************/

/****************************************************************************
* 	                                           Local Variables
****************************************************************************/
#ifdef UICC_FIFO_ONLY
  static ushort transfer_cnt = 0;
#endif

#ifdef UICC_HW_CONTROLLED_T1
  #if !defined (UICC_ALFEEN_ENABLE)
    static ushort par_ovr_dma_cnt;  /* WS2552 workaround. */
    static ushort cwt_timeout_dma_cnt; /* WS2550 workaround. */     
    static ushort inf_epilogue_cnt = 0;
    static ushort inf_epilogue_cnt_max = 0;
  #endif

  #if !defined (SGOLD2) && !defined (SGOLD3) && !defined (SGOLDRADIO)
      static ubyte  mixed_mode = FALSE;
  #endif
  static ubyte length_error;
#else
  static ubyte  tx_data_cnt;
  static ubyte  rx_data_cnt;
  static ushort tx_spc;
#endif

static ubyte parity_error;
static ulong bwt;
static ulong cwt;

#ifndef UICC_NO_DEBUG
  static T_UICC_T1_BLOCK_CTRL_STATE debug = UICC_BLOCK_NONE;
#endif

/****************************************************************************
* 	                                          Global Constants
****************************************************************************/

/****************************************************************************
* 	                                          Global Variables
****************************************************************************/
T_UICC_T1_BLOCK_CTRL_STATE UICC_t1_block_ctrl_state = UICC_BLOCK_READY_CLK_OFF;


/****************************************************************************
* 	                                          Global Function Prototypes
****************************************************************************/

/****************************************************************************
* 	                                          Function Definitions
****************************************************************************/

/*******************************************************************************
* Function:... instruction_failed
* Description: This function is informing the controlling driver statemachine
*              that the instruction execution has failed. Note that this
*              function must only be called from the timeout interrupt.
* Created:.... 05.06.00 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static void instruction_failed(T_UICC_T1_LINK_EVENTS error_cause)
{
  #ifndef UICC_FIFO_ONLY
    UICC_dma_disable_control();
  #endif
	
  UICC_switch_uart_clock_off();
  UICC_disable_card_interface(); /* To reset the HW for a potential new command. */
  UICC_t1_data_link_handler(error_cause);
}

#ifdef UICC_HW_CONTROLLED_T1
/*******************************************************************************
* Function:... UICC_t1_block_handler
* Parameters:. event: The event triggering the statemachine.
* Returns:.... -
* Description: This statemachine/engine is controlling the execution of the T=1
*              protocol block instructions.
* Created:.... 20.09.02 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
void UICC_reset_t1_block_handler(void)
{
  UICC_stop_timer();
  UICC_t1_block_ctrl_state = UICC_BLOCK_READY_CLK_OFF;
}

#ifdef UICC_FIFO_ONLY 
/*This code is used with FIFO stand alone for SGOLD 3 ES 2.. with ALFEEN*/
void UICC_t1_block_handler(T_UICC_T1_BLOCK_EVENTS event)
{
  ushort i;
  
  switch(UICC_t1_block_ctrl_state)
  {
    /*============================================*/
    /* Ready - to execute block transfer          */
    /*============================================*/
    case UICC_BLOCK_READY_CLK_OFF:
      switch(event)
      {
        /*----------------------------*/
        case UICC_INITIATE_BLOCK:
        /*----------------------------*/
          UICC_init_fifo();

          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;

          UICC_set_rxtx_space(UICC_profile_parameter.bgt);      /* Min. spacing with respect to the last Rx character of 
                                           the previous received Rx block (22 ETU is the minimum
                                           specified block spacing spacing). */
          UICC_set_txtx_space(0);       /* Spacing between TX characters in ETU. Must until a workaround 
                                           has been found in HW be set to 0. */
          
          UICC_enable_card_interface(); /* Switch on the internal UART clock - also in line with UTP: WS00004795 */
          UICC_enable_overrun_error_interrupt();
          UICC_enable_t1_end_interrupt();
          UICC_enable_t1_bwt_interrupt();
          UICC_disable_character_retransmission();
          UICC_select_high_clock_freq();
          UICC_activate_clk(UICC_current_reader);
          UICC_start_timer(T_CLK_PU);
          break;

        /*----------------------------*/
        case UICC_STOP_BLOCK_CLOCK:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_AWAIT_CLOCK_OFF;
          UICC_start_timer(T_CLK_PD); 
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;
       }
       break;

    /*============================================*/
    /* The clock is running - ready for a block.  */
    /*============================================*/
    case UICC_BLOCK_READY:
      switch(event)
      {
        /*----------------------------*/
        case UICC_INITIATE_BLOCK:
        /*----------------------------*/
        UICC_enable_card_interface(); /* Switch on the internal UART clock - also in line with UTP: WS00004795 */
        /* Break left out on purpose. */          

        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/

          parity_error = 0; 
          length_error = 0;
          
          /* Calculate Block Waiting Timer value. */ 
          /* BWT = 11 + 2^BWI * 960 [ETU] */
          bwt = 11 + UICC_usec2etu((1 << (UICC_card_profile[UICC_current_reader].bwi))*960*115); /*ETU*/

          bwt += UICC_profile_parameter.bwt_extension;
          
          if(UICC_t1_tpdu_ctrl.bwt_multiplier > 1)
          {
            bwt = bwt * UICC_t1_tpdu_ctrl.bwt_multiplier;
            /* BWT multiplier has only one shot functionality and is therefore reset. */
            UICC_t1_tpdu_ctrl.bwt_multiplier = 1;  
          }

          /* Calculate Character Waiting Timer value. */           
          /* CWT = 11 + 2^CWI [ETU] */
          cwt = (11 + (1 << UICC_card_profile[UICC_current_reader].cwi) + 20/*safety margin*/); /* [ETU] */

          cwt += UICC_profile_parameter.cwt_extension;

          UICC_enable_char_timeout_interrupt();
          UICC_disable_character_interrupt(); 
          UICC_disable_usim_ok_isr();
          UICC_enable_parity_error_interrupt();

          /*if cwt < bgt then it can occur a CH timeout, it look like when start_T1_mode the USIM start the CWT time while BGT from 
          the previous RX block not timeout */ 
          UICC_set_character_timer(9600); /* 9600 = reset value */            

          UICC_set_bwt_timer(bwt);
          UICC_enable_bgt_timer();        
          transfer_cnt = 0;        

          /* Set the HW controller in Tx mode and start the Tx transmission. */
          UICC_select_T1_tx_instruction_mode();
          UICC_t1_block_ctrl_state = UICC_TX_AWAIT_FIFO_REQ;
          UICC_start_timer(T_FIFO_REQ); /*Timeout await FIFO request*/ 

          /*Set FIFO transmit packed size*/
          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
              memcpy(&UICC_hw_tx_store[0], &UICC_prologue[0], UICC_PROLOGUE_SIZE + UICC_prologue[UICC_LEN_POS] + UICC_EPILOGUE_SIZE);
            else  
            {
              SIM_TPS_CTRL = UICC_PROLOGUE_SIZE + UICC_prologue[UICC_LEN_POS] + UICC_EPILOGUE_SIZE; 
            }
          #else
            SIM_TPS_CTRL = UICC_PROLOGUE_SIZE + UICC_prologue[UICC_LEN_POS] + UICC_EPILOGUE_SIZE; 
          #endif
  		      			
          break;

        /*----------------------------*/
        case UICC_STOP_BLOCK_CLOCK:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_AWAIT_CLOCK_OFF;
          UICC_start_timer(T_CLK_PD); 
          break;
                  
        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;
      }
      break;

    /*============================================*/
    /* Waiting for FIFO request  */
    /*============================================*/
    case UICC_TX_AWAIT_FIFO_REQ:
		/*Wait for FIFO make TX request, Start T1 when receivei TX_BREQ(16bytes), TX_LBREQ(13-16 bytes) or combination: TX_SREQ(+TX_SREQ)+LSREQ = 4+(+4)+1-4=9-12 bytes*/
      switch(event)
      {
        /*----------------------------*/
        case UICC_TX_SREQ:
        /*----------------------------*/
         /*TX single request 4 bytes */

          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
              memcpy(&UICC_simulated_hw_reg._SIM_FIFO_TXD[0],&UICC_prologue[transfer_cnt], 4);
            else
              SIM_FIFO_TXD = (ulong) *((ulong *)(&UICC_prologue[transfer_cnt]));  
          #else
            SIM_FIFO_TXD = (ulong) *((ulong *)(&UICC_prologue[transfer_cnt]));  
          #endif
          
          transfer_cnt += 4;
          UICC_start_timer(T_FIFO_REQ); /*Timeout await FIFO request*/    
          break;

        /*----------------------------*/
        case UICC_TX_BREQ:
        /*----------------------------*/
          /*Burst request 16 bytes = 4 * 4 bytes*/
          UICC_t1_block_ctrl_state = UICC_TX_TRANS;
        
          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
              memcpy(&UICC_simulated_hw_reg._SIM_FIFO_TXD[0],&UICC_prologue[transfer_cnt], 4);
            else
              SIM_FIFO_TXD = (ulong) *((ulong *)(&UICC_prologue[transfer_cnt]));  
          #else
            SIM_FIFO_TXD = (ulong) *((ulong *)(&UICC_prologue[transfer_cnt]));  
          #endif
          
          transfer_cnt += 4;
          UICC_start_t1_mode(UICC_prologue[UICC_LEN_POS]);		  
          /*The remain 12 bytes (16-4)*/ 
          for (i=1;i<=3;i++)
          {
            #ifdef UICC_MODULE_TEST
              if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
                memcpy(&UICC_simulated_hw_reg._SIM_FIFO_TXD[i*4],&UICC_prologue[transfer_cnt], 4);
              else
                SIM_FIFO_TXD = (ulong) *((ulong *)(&UICC_prologue[transfer_cnt]));  
            #else
              SIM_FIFO_TXD = (ulong) *((ulong *)(&UICC_prologue[transfer_cnt]));  
            #endif

            transfer_cnt += 4;
          }
          UICC_start_timer(T_FIFO_REQ); /*Timeout await FIFO request*/ 
          break;

        /*----------------------------*/
        case UICC_TX_LSREQ:
        /*----------------------------*/
          /*TX Last single request 1-4 bytes */
          UICC_t1_block_ctrl_state = UICC_TX_FRAME;	
          UICC_stop_timer();	
          
          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
              memcpy(&UICC_simulated_hw_reg._SIM_FIFO_TXD[0],&UICC_prologue[transfer_cnt], 4);
            else
              SIM_FIFO_TXD = (ulong) *((ulong *)(&UICC_prologue[transfer_cnt]));  
          #else
            SIM_FIFO_TXD = (ulong) *((ulong *)(&UICC_prologue[transfer_cnt]));  
          #endif
          
          UICC_start_t1_mode(UICC_prologue[UICC_LEN_POS]);
          break;

        /*----------------------------*/
        case UICC_TX_LBREQ:
        /*----------------------------*/
          /*Burst request 13-16 bytes = 4 * 4 bytes*/
          UICC_t1_block_ctrl_state = UICC_TX_FRAME;
          UICC_stop_timer();

          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
              memcpy(&UICC_simulated_hw_reg._SIM_FIFO_TXD[0],&UICC_prologue[transfer_cnt], 4);
            else
              SIM_FIFO_TXD = (ulong) *((ulong *)(&UICC_prologue[transfer_cnt]));  
          #else
            SIM_FIFO_TXD = (ulong) *((ulong *)(&UICC_prologue[transfer_cnt]));  
          #endif

          transfer_cnt += 4;
          UICC_start_t1_mode(UICC_prologue[UICC_LEN_POS]);
		
          /*The remain 12 bytes (16-4)*/ 
          for (i=1;i<=3;i++)
          {
            #ifdef UICC_MODULE_TEST
              if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
                memcpy(&UICC_simulated_hw_reg._SIM_FIFO_TXD[i*4],&UICC_prologue[transfer_cnt], 4);
              else
                SIM_FIFO_TXD = (ulong) *((ulong *)(&UICC_prologue[transfer_cnt]));  
            #else
              SIM_FIFO_TXD = (ulong) *((ulong *)(&UICC_prologue[transfer_cnt]));  
            #endif

            transfer_cnt += 4;
          }
          break;

        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
        /*FIFO request does not occur*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_TX_ERROR;
          UICC_start_timer(T_CLK_PD);		
          break;
                  
        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;
      }
      break;

    /*============================================*/
    /* The clock is running - ready for a block.  */
    /*============================================*/
    case UICC_TX_TRANS:
      switch(event)
      {
        /*----------------------------*/
        case UICC_TX_LSREQ:
        /*----------------------------*/
         /*TX Last single request 1-4 bytes */
          UICC_t1_block_ctrl_state = UICC_TX_FRAME;                  
          UICC_stop_timer();		
          
          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
              memcpy(&UICC_simulated_hw_reg._SIM_FIFO_TXD[0],&UICC_prologue[transfer_cnt], 4);
            else
              SIM_FIFO_TXD = (ulong) *((ulong *)(&UICC_prologue[transfer_cnt]));  
          #else
            SIM_FIFO_TXD = (ulong) *((ulong *)(&UICC_prologue[transfer_cnt]));  
          #endif
          
          break;

        /*----------------------------*/
        case UICC_TX_SREQ:
        /*----------------------------*/
          /*TX single request 4 bytes */
          UICC_start_timer(T_FIFO_REQ);
          
          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
              memcpy(&UICC_simulated_hw_reg._SIM_FIFO_TXD[0],&UICC_prologue[transfer_cnt], 4);
            else
              SIM_FIFO_TXD = (ulong) *((ulong *)(&UICC_prologue[transfer_cnt]));  
          #else
            SIM_FIFO_TXD = (ulong) *((ulong *)(&UICC_prologue[transfer_cnt]));  
          #endif
          
          transfer_cnt += 4;
          break;

        /*----------------------------*/
        case UICC_TX_LBREQ:
        /*----------------------------*/
          /*Burst request 13-16 bytes = 4 * 4 bytes*/
          UICC_t1_block_ctrl_state = UICC_TX_FRAME;
          UICC_stop_timer();
          for (i=1;i<=4;i++)
          {          
            #ifdef UICC_MODULE_TEST
              if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
                memcpy(&UICC_simulated_hw_reg._SIM_FIFO_TXD[(i-1)*4],&UICC_prologue[transfer_cnt], 4);
              else
                SIM_FIFO_TXD = (ulong) *((ulong *)(&UICC_prologue[transfer_cnt]));  
            #else
              SIM_FIFO_TXD = (ulong) *((ulong *)(&UICC_prologue[transfer_cnt]));  
              transfer_cnt += 4;/*add 080109*/
            #endif            
          }
          break;

        /*----------------------------*/
        case UICC_TX_BREQ:
        /*----------------------------*/
          /*Burst request 16 bytes = 4 * 4 bytes*/
          UICC_start_timer(T_FIFO_REQ);
          for (i=1;i<=4;i++)
          {
            #ifdef UICC_MODULE_TEST
              if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
                memcpy(&UICC_simulated_hw_reg._SIM_FIFO_TXD[(i-1)*4],&UICC_prologue[transfer_cnt], 4);
              else
                SIM_FIFO_TXD = (ulong) *((ulong *)(&UICC_prologue[transfer_cnt]));  
            #else
              SIM_FIFO_TXD = (ulong) *((ulong *)(&UICC_prologue[transfer_cnt]));  
            #endif            

            transfer_cnt += 4;
          }
          break;

        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
        /*FIFO request does not occur*/

          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_TX_ERROR;
          UICC_start_timer(T_CLK_PD);		
          break;
          
        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;
      }
      break;

    /*============================================*/
    /* Transferring the entire Tx frame.          */
    /*============================================*/
    case UICC_TX_FRAME:
      switch(event)
      {
        /*----------------------------*/
         case UICC_T1_END:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_RX_PROLOGUE_ALFEEN;
          /* Set the HW controller in Rx mode and start the Rx transmission. */
          UICC_select_T1_rx_instruction_mode();                        
          UICC_start_bwt_timer();
          UICC_start_t1_mode(UICC_PROLOGUE_SIZE);
          break;

        /*----------------------------*/
        case UICC_CHTIMEOUT:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_TX_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;

        /*----------------------------*/
        case UICC_OVERRUN_ERROR:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_OVR_ERROR;
          UICC_disable_char_timeout_interrupt();
          UICC_start_timer(T_CLK_PD);
          break;        
     
        case UICC_CHARACTER_OK:
        case UICC_T1_BWT_EXPIRED:
        case UICC_T1_PARITY_ERROR:
        case UICC_PARITY_ERROR:
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;
      }    
      break;

    /*============================================*/
    /* Transfer the Prologue part of the Rx Frame.*/
    /*============================================*/
    case UICC_RX_PROLOGUE_ALFEEN:
      switch(event)
      {         
         /*----------------------------*/
        case UICC_UNKNOWN_INT:
        /*----------------------------*/        
          UICC_t1_block_ctrl_state = UICC_RX_TRANS;
          transfer_cnt = UICC_NAD_POS;
        
          /*Check LEN error (SIMT1LFPAR) this bit is set when end of receive prolog*/
          if(UICC_get_simstatus() &  0x0200) /*Check if it is parity error in LEN  (SIMT1LFPAR) */ 
          {
            UICC_SETBIT(parity_error, RX_PARITY_ERROR_LEN_POS); 
            length_error = TRUE;
          }
          break;

        /*----------------------------*/
        case UICC_T1_BWT_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BWT_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;

        /*----------------------------*/
        case UICC_CHTIMEOUT:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_CWT_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;

        /*----------------------------*/
        case UICC_OVERRUN_ERROR:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_OVR_ERROR;
          UICC_disable_char_timeout_interrupt();
          UICC_start_timer(2*UICC_etu2usec(cwt));
          break;        

        /*----------------------------*/
        case UICC_T1_PARITY_ERROR:
        case UICC_PARITY_ERROR:        
        /*----------------------------*/
          /*Read SIM status to check if the parity occur in LEN or not, cannot use the DMA count because the HW control the DMA*/
          if(UICC_get_simstatus() &  0x0200) /*Check if it is parity error in LEN */ 
          {
            UICC_SETBIT(parity_error, RX_PARITY_ERROR_LEN_POS); 
            length_error = TRUE;
          }
          else  /*NOT parity in LEN then in NAD or PCB*/             
            UICC_SETBIT(parity_error, RX_PARITY_ERROR_NAD_OR_PCB_POS);           
          break;

        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;
      }
      break;

    /*============================================*/
    /* Has received RX prologue , */
    /*============================================*/
    case UICC_RX_TRANS:
      switch(event)
      {
        /*----------------------------*/
        case UICC_RX_LSREQ:
        /*----------------------------*/
          /*Last single request 1-4 bytes*/
          UICC_t1_block_ctrl_state = UICC_RX_FRAME;

          /*Break left out on purpose*/

        /*----------------------------*/
        case UICC_RX_SREQ:
        /*----------------------------*/
          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
              memcpy(&UICC_prologue[transfer_cnt], &UICC_simulated_hw_reg._SIM_FIFO_RXD[0], 4);
            else
              memcpy(&UICC_prologue[transfer_cnt], &SIM_FIFO_RXD, 4);
          #else
            memcpy(&UICC_prologue[transfer_cnt], (const void *)(&SIM_FIFO_RXD), 4);
          #endif

          transfer_cnt += 4;
          break;

        /*----------------------------*/
        case UICC_RX_LBREQ:
        /*----------------------------*/
          /*Last burst request 13-16 bytes = 4 * 4 bytes*/
          UICC_t1_block_ctrl_state = UICC_RX_FRAME;

           /* Break left out on purpose. */

        /*----------------------------*/
        case UICC_RX_BREQ:
        /*----------------------------*/
          /*Burst request 16 bytes = 4 * 4 bytes*/
          for (i=1;i<=4;i++)
          {
            #ifdef UICC_MODULE_TEST
              if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
                memcpy(&UICC_prologue[transfer_cnt], &UICC_simulated_hw_reg._SIM_FIFO_RXD[(i-1)*4], 4);
              else
                memcpy(&UICC_prologue[transfer_cnt], &SIM_FIFO_RXD, 4);
            #else
              memcpy(&UICC_prologue[transfer_cnt], (const void *)(&SIM_FIFO_RXD), 4);
            #endif

            transfer_cnt += 4;
          }
          
          break;
          
         /*----------------------------*/
	  	 case UICC_T1_END:	
         case UICC_UNKNOWN_INT:
        /*----------------------------*/        
        /* When LEN=0  and Last byte EDC has parity error (Test case 7.3.8(parity)) then the RX_LSREG can be absent
           here is a work around for this problem*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_SUCCEEDED;
          //Clear FIFO
          SIM_RUN_CTRL = 0; //RUN = 0
          SIM_FIFO_ICR = 0x3FF; //Clear FIFO interrupt register
          SIM_RUN_CTRL = 1; //RUN = 1 				
        
          /*Check LEN error (SIMT1LFPAR) this bit is set when end of receive prolog*/
          if(UICC_get_simstatus() &  0x0200) /*Check if it is parity error in LEN  (SIMT1LFPAR) */ 
          {
            UICC_SETBIT(parity_error, RX_PARITY_ERROR_LEN_POS); 
            length_error = TRUE;
          }

          UICC_force_timeout();					
          break;
					
        /*----------------------------*/
        case UICC_T1_BWT_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BWT_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;

        /*----------------------------*/
        case UICC_CHTIMEOUT:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_CWT_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;

        /*----------------------------*/
        case UICC_OVERRUN_ERROR:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_OVR_ERROR;
          UICC_disable_char_timeout_interrupt();
          UICC_start_timer(2*UICC_etu2usec(cwt));
          break;        

        /*----------------------------*/
        case UICC_T1_PARITY_ERROR:
        case UICC_PARITY_ERROR:        
        /*----------------------------*/
          /*Read SIM status to check if the parity occur in LEN or not, cannot use the DMA count because the HW control the DMA*/
          if(UICC_get_simstatus() &  0x0200) /*Check if it is parity error in LEN */ 
          {
            UICC_SETBIT(parity_error, RX_PARITY_ERROR_LEN_POS); 
            length_error = TRUE;
          }
          else  /*NOT parity in LEN then in NAD or PCB*/             
            UICC_SETBIT(parity_error, RX_PARITY_ERROR_NAD_OR_PCB_POS); 
          break;

        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;
      }
      break;

    /*============================================*/
    /* Receiving the Prolog, INF and Epilogue field.      */
    /*============================================*/
    case UICC_RX_FRAME:
      switch(event)
      {
        /*----------------------------*/
         /*Sometime  when LEN = 0 and EDC 's PARITY error then UNKNOWN interrupt occur instead of T1_END !!!*/
        case UICC_UNKNOWN_INT:
        case UICC_T1_END:
        /*----------------------------*/
        /* Mask the SIMT1PAR bit from the status register to see if a parity error did occur during 
           block reception. */
          if(UICC_get_simstatus() &  0x0082)
          {
            UICC_SETBIT(parity_error, RX_PARITY_ERROR_INF_POS);           
          }    

          UICC_t1_block_ctrl_state = UICC_BLOCK_SUCCEEDED;
          UICC_force_timeout();
          break;

        /*----------------------------*/
        case UICC_CHTIMEOUT:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_CWT_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;

        /*----------------------------*/
        case UICC_OVERRUN_ERROR:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_OVR_ERROR;
          UICC_disable_char_timeout_interrupt();
          UICC_start_timer(2*UICC_etu2usec(cwt));
          break;        

        /*----------------------------*/
        case UICC_T1_PARITY_ERROR:
        case UICC_PARITY_ERROR:        
        /*----------------------------*/       
          UICC_SETBIT(parity_error, RX_PARITY_ERROR_INF_POS); 
          UICC_start_timer(10000); /* To ensure timeout if no T1END for some reason does not occur. */
          break;

        /*----------------------------*/       
        case UICC_TIMER_EXPIRED:        
        /*----------------------------*/       
          UICC_t1_block_ctrl_state = UICC_BLOCK_SUCCEEDED;
          UICC_force_timeout();
          break;

        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;
      }    
      break;
	
    /*============================================*/
    /* The block transfer/reception succeeded.    */
    /*============================================*/
    case UICC_BLOCK_SUCCEEDED:
      switch(event)
      {
        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;  
          /* Inform main statemachine of the execution result, that the
             clock now is stopped and that power down can be accepted. */


        #ifndef UICC_ALFEEN_ENABLE
          #if defined (UICC_FIFO_MODE)
             // For alignment
            for(cnt=0;cnt<(UICC_prologue[UICC_LEN_POS] + UICC_EPILOGUE_SIZE);cnt++)
                UICC_prologue[UICC_INF_POS+cnt] = UICC_prologue[UICC_INF_POS+cnt+1];
          #endif 
        #endif

        
          /* Check if an illegal length was used. */
          if(length_error)
          {
            UICC_t1_data_link_handler(UICC_LEN_VIOLATION);
          }
          else
          {        
            /* Check to see if a parity error did occur during block reception. */
            if(parity_error)
            {
              UICC_t1_data_link_handler(UICC_CHAR_PARITY_ERROR);          
            }
            else
            {
             UICC_t1_data_link_handler(UICC_BLOCK_RECEIVED);
            }        
          }
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;         
      }    
      break;

    /*============================================*/
    /* Block error (unknown event) encountered.   */
    /*============================================*/
    case UICC_BLOCK_ERROR:
      switch(event)
      {
        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;  
          instruction_failed(UICC_UNKNOWN_EVENT);
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_force_timeout();
          break;
      }    
      break;

    /*============================================*/
    /* Overrun error encountered.                 */
    /*============================================*/
    case UICC_OVR_ERROR:
      switch(event)
      {
        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;  
          instruction_failed(UICC_CHAR_OVERRUN_ERROR);
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_force_timeout();
          break;
      }    
      break;

    /*============================================*/
    /* TX error encountered.                      */
    /*============================================*/
    case UICC_TX_ERROR:
      switch(event)
      {
        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;  
          instruction_failed(UICC_CHAR_TIMEOUT);
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_force_timeout();
          break;
      }    
      break;

    /*============================================*/
    /* BlockWaitingTimer expiration detected.     */
    /*============================================*/
    case UICC_BWT_ERROR:
      switch(event)
      {
        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;  
          instruction_failed(UICC_BWT_TIMEOUT);
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_force_timeout();
          break;
      }    
      break;

    /*============================================*/
    /* CharacterWaitingTimer expiration detected. */
    /*============================================*/
    case UICC_CWT_ERROR:
      switch(event)
      {
        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;  
          /* Check if the timeout was generated as a result of parity error or an 
             illegal length. */
          if(parity_error)
          {
            instruction_failed(UICC_CHAR_PARITY_ERROR);          
          }
          else if(length_error)
          {
            instruction_failed(UICC_LEN_VIOLATION);
          }
          else
          {
            instruction_failed(UICC_CWT_TIMEOUT);
          }
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_force_timeout();
          break;
      }    
      break;

    /*============================================*/
    /* Wait until it is allowed to stop the clock.*/
    /*============================================*/
    case UICC_BLOCK_AWAIT_CLOCK_OFF:
      switch(event)
      {
        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY_CLK_OFF;  
          UICC_stop_clock_conditional(UICC_card_profile[UICC_current_reader].used_clock_stop_mode, UICC_current_reader);
          UICC_switch_uart_clock_off(); /* Switch off the internal UART clock - also in line with UTP: WS00004795 */
          UICC_t1_data_link_handler(UICC_BLOCK_CLOCK_STOPPED);
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_force_timeout();
          break;
      }    
      break;

    /*==========================================*/
    /* Default.                                 */
    /*==========================================*/
    default:
      #ifndef UICC_NO_DEBUG
        debug = UICC_t1_block_ctrl_state;
      #endif
      break;
  }

  #ifndef UICC_NO_DEBUG
    if(debug != UICC_BLOCK_NONE)
    {
      UICC_log_data_ptr[0] = debug;
      UICC_log_data_ptr[1] = event;
      UICC_log_data_ptr[2] = UICC_current_reader;
      UICC_log_data_ptr[3]  = (ubyte) (SIM_CTRL>>8);
      UICC_log_data_ptr[4]  = SIM_CTRL;
      UICC_log_data_ptr[5]  = ((ubyte) (SIM_STATUS>>8)) | ((ubyte)SIM_DMA<<7);
      UICC_log_data_ptr[6]  = SIM_STATUS;
      UICC_log_data_ptr[7]  = SIM_BRF;
      UICC_log_data_ptr[8]  = SIM_IRQEN;
      UICC_log_data_ptr[9]  = SIM_RXSPC;
      UICC_log_data_ptr[10] = SIM_TXSPC;
      UICC_log_data_ptr[11] = SIM_CTRL2;
      UICC_log_data_ptr[12] = SIM_T1CTRL;
      UICC_log_data_ptr[13] = (ubyte) ((ubyte)(SIM_RIS<<4) | SIM_MIS);
      UICC_log_data_ptr[14] = (ubyte) ((ubyte)(SIM_ICR<<4) | SIM_ISR);
      memcpy(&UICC_log_data_ptr[17], &UICC_prologue[0], 3);
      UICC_log_data_ptr[20] = UICC_inf[UICC_EPILOGUE_POS];
      memcpy(&UICC_log_data_ptr[21], &UICC_inf[0], 6);      
      UICC_raise_exception(TRAP_UICC_T1_BLOCK, UICC_T1_BLOCK_ERROR_1, 25); 

      debug = UICC_BLOCK_NONE;
    }
  #endif
}

#else /*End UICC_FIFO_ONLY*/

void UICC_t1_block_handler(T_UICC_T1_BLOCK_EVENTS event)
{

#ifndef UICC_ALFEEN_ENABLE
  #if defined (UICC_FIFO_MODE)
    ushort cnt;
  #endif
#endif
  
  switch(UICC_t1_block_ctrl_state)
  {
    /*============================================*/
    /* Ready - to execute block transfer          */
    /*============================================*/
    case UICC_BLOCK_READY_CLK_OFF:
      switch(event)
      {
        /*----------------------------*/
        case UICC_INITIATE_BLOCK:
        /*----------------------------*/
          #if defined (UICC_FIFO_MODE)
            UICC_init_fifo();
          #endif

          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;

          UICC_set_rxtx_space(UICC_profile_parameter.bgt);      /* Min. spacing with respect to the last Rx character of 
                                           the previous received Rx block (22 ETU is the minimum
                                           specified block spacing spacing). */
          UICC_set_txtx_space(0);       /* Spacing between TX characters in ETU. Must until a workaround 
                                           has been found in HW be set to 0. */

          #if !defined (SGOLD2) && !defined (SGOLD3) && !defined (SGOLDRADIO)
            /* This check actually belongs to the state UICC_RX_PROLOGUE, event=T1END but the check is placed
               here in order to minimise the processing upon this interrupt. */
            if( ((UICC_baseband_profile.version == SYS_BB_VER_SGOLD) && 
                 (UICC_baseband_profile.revision < SYS_BB_REV_12))    && 
                ((UICC_clock_rate_table[UICC_card_profile[UICC_current_reader].f_used] == 372) && 
                 (UICC_baud_rate_table[UICC_card_profile[UICC_current_reader].d_used] == 1)))
            { /* Normal speed. */
              mixed_mode = TRUE;
            }
            else
            {
              mixed_mode = FALSE;
            }
          #endif
          
          UICC_enable_card_interface(); /* Switch on the internal UART clock - also in line with UTP: WS00004795 */
          UICC_enable_overrun_error_interrupt();
          UICC_enable_t1_end_interrupt();
          UICC_enable_t1_bwt_interrupt();
          UICC_disable_character_retransmission();
          UICC_select_high_clock_freq();
          UICC_activate_clk(UICC_current_reader);
          UICC_start_timer(T_CLK_PU);
          break;

        /*----------------------------*/
        case UICC_STOP_BLOCK_CLOCK:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_AWAIT_CLOCK_OFF;
          UICC_start_timer(T_CLK_PD); 
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;
       }
       break;

    /*============================================*/
    /* The clock is running - ready for a block.  */
    /*============================================*/
    case UICC_BLOCK_READY:
      switch(event)
      {
        /*----------------------------*/
        case UICC_INITIATE_BLOCK:
        /*----------------------------*/
          UICC_enable_card_interface(); /* Switch on the internal UART clock - also in line with UTP: WS00004795 */
          /* Break left out on purpose. */          

        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_TX_FRAME;

          parity_error = 0; 
          length_error = 0;
          
          #if !defined (UICC_ALFEEN_ENABLE)
            inf_epilogue_cnt = 0;
          #endif

          /* Calculate Block Waiting Timer value. */ 
          /* BWT = 11 + 2^BWI * 960 [ETU] */
          bwt = 11 + UICC_usec2etu((1 << (UICC_card_profile[UICC_current_reader].bwi))*960*115); /*ETU*/

          bwt += UICC_profile_parameter.bwt_extension;
          
          if(UICC_t1_tpdu_ctrl.bwt_multiplier)
          {
            bwt = bwt * UICC_t1_tpdu_ctrl.bwt_multiplier;
            /* BWT multiplier has only one shot functionality and is therefore reset. */
            UICC_t1_tpdu_ctrl.bwt_multiplier = 1;  
          }

          /* Calculate Character Waiting Timer value. */           
          /* CWT = 11 + 2^CWI [ETU] */
          cwt = (11 + (1 << UICC_card_profile[UICC_current_reader].cwi) + 20/*safety margin*/); /* [ETU] */

          cwt += UICC_profile_parameter.cwt_extension;

          UICC_enable_char_timeout_interrupt();
          UICC_disable_character_interrupt(); 
          UICC_disable_usim_ok_isr();
          UICC_enable_parity_error_interrupt();

          #if defined (SGOLD3) 
            /*if cwt < bgt then it can occur a CH timeout, it look like when start_T1_mode the USIM start the CWT time while BGT from 
            the previous RX block not timeout */ 
            UICC_set_character_timer(9600); /* 9600 = reset value */            
          #else
            #if defined (SGOLD2) || defined (SGOLDRADIO)
              /* Do NOT utilise the CWT timer */
            #else
              if((UICC_baseband_profile.version == SYS_BB_VER_SGOLD) && 
                 (UICC_baseband_profile.revision == SYS_BB_REV_12))
              {
                /* Do NOT utilise the CWT timer */
              }
              else
              {
                UICC_set_character_timer(cwt);
              }
            #endif
          #endif


          
          UICC_set_bwt_timer(bwt);
          UICC_enable_bgt_timer();        

          /* Set the HW controller in Tx mode and start the Tx transmission. */
          UICC_select_T1_tx_instruction_mode();
          UICC_enable_dma_channel();

          /* Setup the DMA to be ready to transfer the entire TX block (Prologue, Information and 
             Epilogue fields. */
          UICC_dma_tx_control(UICC_PROLOGUE_SIZE + UICC_prologue[UICC_LEN_POS] + UICC_EPILOGUE_SIZE,
                              &UICC_prologue[UICC_NAD_POS], 
                              UICC_USIM_FLOW_CONTROL);

          UICC_start_t1_mode(UICC_prologue[UICC_LEN_POS]);
          break;

        /*----------------------------*/
        case UICC_STOP_BLOCK_CLOCK:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_AWAIT_CLOCK_OFF;
          UICC_start_timer(T_CLK_PD); 
          break;
                   
        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;
      }
      break;

    /*============================================*/
    /* Transferring the entire Tx frame.          */
    /*============================================*/
    case UICC_TX_FRAME:
      switch(event)
      {
        /*----------------------------*/
         case UICC_T1_END:
        /*----------------------------*/

        #ifdef UICC_ALFEEN_ENABLE 
          UICC_t1_block_ctrl_state = UICC_RX_PROLOGUE_ALFEEN;
        #else
          UICC_t1_block_ctrl_state = UICC_RX_PROLOGUE;
        #endif

        /* Setup the DMA to be ready to receive the Prologue part of the Rx frame. */
       
        UICC_dma_rx_control(UICC_PROLOGUE_SIZE, 
                            &UICC_prologue[UICC_NAD_POS], 
                            UICC_USIM_FLOW_CONTROL);
        /* Set the HW controller in Rx mode and start the Rx transmission. */
        UICC_select_T1_rx_instruction_mode();
        
        #ifndef SGOLD3 /*CWT in SGOLD3 is working*/
          #if defined (SGOLD2) || defined (SGOLDRADIO)
            cwt_timeout_dma_cnt = UICC_PROLOGUE_SIZE;
            UICC_start_timer(UICC_etu2usec(bwt) + 3*(UICC_etu2usec(cwt)+1));          
          #else
            if(UICC_baseband_profile.version == SYS_BB_VER_SGOLD)
            { /* UTP: WS00002550: RX character timeout does not work - use a core timer to monitor the violation 
                 of a character distance. */
              cwt_timeout_dma_cnt = UICC_PROLOGUE_SIZE;
              UICC_start_timer(UICC_etu2usec(bwt) + 3*(UICC_etu2usec(cwt)+1));          
            }   
            else
            { /* UTP: WS00002550 corrected. RX character timeout is now working again. */
             
            }
          #endif
        #endif
                
        UICC_start_bwt_timer();
        UICC_start_t1_mode(UICC_PROLOGUE_SIZE);

        break;
        /*----------------------------*/
        case UICC_CHTIMEOUT:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_TX_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;

        /*----------------------------*/
        case UICC_OVERRUN_ERROR:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_dma_disable_control();
          UICC_t1_block_ctrl_state = UICC_OVR_ERROR;
          UICC_disable_char_timeout_interrupt();
          UICC_start_timer(T_CLK_PD);
          break;        
     
        case UICC_DMA_END:               

        case UICC_CHARACTER_OK:
        case UICC_T1_BWT_EXPIRED:
        case UICC_T1_PARITY_ERROR:
        case UICC_PARITY_ERROR:
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;
      }    
      break;

   #ifndef UICC_ALFEEN_ENABLE
    /*============================================*/
    /* Transfer the Prologue part of the Rx Frame.*/
    /*============================================*/
    case UICC_RX_PROLOGUE:
      switch(event)
      {  
        ubyte inf_len;

        /*----------------------------*/
        case UICC_UNKNOWN_INT:
        /*----------------------------*/
           #if defined (SGOLD2) || defined (SGOLD3) || defined (SGOLDRADIO)
            /* On SG2, SG3, SGOLDRADIO is the interrupt cause (i.e. T1END) not set when the Prologue 
               field has been received. Thefore treat an unknown even in ES3 as if T1END was indicated. */
            /* Intentionally fall through -> UICC_T1_END will be executed hereafter. */
          #else
            if((UICC_baseband_profile.version == SYS_BB_VER_SGOLD) && 
               (UICC_baseband_profile.revision < SYS_BB_REV_12))
            { 
              #ifndef UICC_NO_DEBUG
                debug = UICC_t1_block_ctrl_state;
              #endif
              UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
              UICC_start_timer(T_CLK_PD);
              break;
            }
            else
            { /* On S-GOLD ES3, and SGOLD2 and.... is the interrupt cause (i.e. T1END) not set when the Prologue 
                 field has been received. Thefore treat an unknown even in ES3 as if T1END was indicated. */

              /* Break removed by purpose -> UICC_T1_END will be executed hereafter. */
            }
          #endif
  
          /*----------------------------*/
          case UICC_T1_END:
          /*----------------------------*/              
          inf_len = UICC_prologue[UICC_LEN_POS];

          /* The Epilogue has now been received successfully received from the card and the 
             remaining part of the block (including the epilogue filed) is now expected. */

          /* But first check if the LEN field holds the invalid length indicatof of 255. */
          if(inf_len == 0xFF)
          { /* Mark that the length is no correct. */           
            length_error = TRUE;
          }
          UICC_t1_block_ctrl_state = UICC_RX_INF_EPILOGUE;

          #if defined(SGOLD2) || defined(SGOLD3) || defined(SGOLDRADIO)
            /* Now check if the parity error has been spotted during the reception of the Prologue field. */
            if(UICC_TESTBIT(parity_error, RX_PARITY_ERROR_LEN_POS))
            { /* The length has been corrupted - the length needs to be set to maximum (to be on the safe side). */
              inf_len = 254;              
            }

            UICC_dma_rx_control(inf_len + UICC_EPILOGUE_SIZE, 
                               &UICC_prologue[UICC_INF_POS], 
                               UICC_USIM_FLOW_CONTROL);

            UICC_start_t1_mode(inf_len);
            
            #if !defined(SGOLD3) 
              /* The CWT timer does not work in T=1 INVERSE mode - use a core timer to monitor the violation 
                 of a character distance. */
              cwt_timeout_dma_cnt = inf_len;
              UICC_start_timer(10*UICC_etu2usec(cwt));
            #endif
            
            break;
          #else  /*Not sgold 2 or sgold3*/
            /* Now check if the parity error has been spotted during the reception of the Prologue field. */
            if(UICC_get_simstatus() & 0x0082)
            { /* One of the prologue bytes has been parity error corrupted but since there is no 
                 way of telling which byte has been inflicted the length needs to be set to 
                 maximum (to be on the safe side). */
              inf_len = 254;
              UICC_SETBIT(parity_error, RX_PARITY_ERROR_LEN_POS); 
            }

            if(mixed_mode)
            { /* To bypass the fact that a parity error cannot be detected by the SGOLD ES2.5 HW block, 
                 switch to Character Mode for the reception of the Information field and Epilogue reception. 
                 This is only valid for SG ES2.5 and only if a Normal Speed UICC is operated. */
              UICC_t1_block_ctrl_state = UICC_CHAR_RX_INF_EPILOGUE;
              SIM_CTRL &= 0x7FFD; /* Enable Character mode (SIMT0=0 & SIMT1=0). */
              SIM_T1CTRL = 0;
              UICC_disable_char_timeout_interrupt();
              UICC_enable_character_interrupt(); 
              UICC_enable_usim_ok_isr();
              inf_epilogue_cnt_max = inf_len+1;
              UICC_start_timer(UICC_etu2usec(cwt+100));
            }
            else
            {
              UICC_dma_rx_control(inf_len + UICC_EPILOGUE_SIZE, 
                                &UICC_prologue[UICC_INF_POS], 
                                UICC_USIM_FLOW_CONTROL);

              UICC_start_t1_mode(inf_len);
     
              if(UICC_baseband_profile.version == SYS_BB_VER_SGOLD)
              { /* UTP: WS00002550: RX character timeout does not work - use a core timer to monitor the violation 
                   of a character distance. */
                cwt_timeout_dma_cnt = inf_len;
                UICC_start_timer(10*UICC_etu2usec(cwt));
              }
              else
              { /* UTP: WS00002550 corrected. RX character timeout is now working again. */
         
              }
            }
            break;
          #endif
        
        /*----------------------------*/
        case UICC_T1_BWT_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BWT_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;

        /*----------------------------*/
        case UICC_CHTIMEOUT:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_CWT_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;

        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          #if defined (SGOLD2) || defined (SGOLDRADIO)
            { 
              ushort cnt = UICC_dma_get_transfer_cnt();
              if(cnt < cwt_timeout_dma_cnt)
              { /* Something has been transferred - let's stay in this state and see if this also 
                   is the case after yet another round of CWT. */
                cwt_timeout_dma_cnt = UICC_dma_get_transfer_cnt();
                UICC_start_timer(2*UICC_etu2usec(cwt));                       
              }
              else 
              {
                if(cnt == UICC_PROLOGUE_SIZE)
                { /* If the counter size is identical to the prologue size, this indicates that no 
                     data has been received at all -> BWT has been violated. */
                  UICC_t1_block_ctrl_state = UICC_BWT_ERROR;          
                }
                else
                {
                  UICC_t1_block_ctrl_state = UICC_CWT_ERROR;          
                }
                UICC_force_timeout();
              }
            }  
          #else
            if(UICC_baseband_profile.version == SYS_BB_VER_SGOLD)
            { /* UTP: WS00002550: RX character timeout does not work - use a core timer to monitor the violation 
                 of a character distance. */
              ushort cnt = UICC_dma_get_transfer_cnt();
              if(cnt < cwt_timeout_dma_cnt)
              { /* Something has been transferred - let's stay in this state and see if this also 
                   is the case after yet another round of CWT. */
                cwt_timeout_dma_cnt = UICC_dma_get_transfer_cnt();
                UICC_start_timer(2*UICC_etu2usec(cwt));                       
              }
              else if(cnt == UICC_PROLOGUE_SIZE)
              { /* If the counter size is identical to the prologue size, this indicates that no 
                   data has been received at all -> BWT has been violated. */
                UICC_t1_block_ctrl_state = UICC_BWT_ERROR;          
                UICC_force_timeout();
              }
              else
              {
                UICC_t1_block_ctrl_state = UICC_CWT_ERROR;          
                UICC_force_timeout();
              }
            }  
            else
            { /* UTP: WS00002550 corrected. RX character timeout is now working again. */
              UICC_t1_block_ctrl_state = UICC_CWT_ERROR;
              UICC_start_timer(T_CLK_PD);
            }
          #endif
          break;

        /*----------------------------*/
        case UICC_OVERRUN_ERROR:
        /*----------------------------*/
          #if defined (SGOLD2) || defined (SGOLD3) || defined (SGOLDRADIO)
            #ifndef UICC_NO_DEBUG
              debug = UICC_t1_block_ctrl_state;
            #endif
            UICC_dma_disable_control();
            UICC_t1_block_ctrl_state = UICC_OVR_ERROR;
            UICC_disable_char_timeout_interrupt();
            UICC_start_timer(2*UICC_etu2usec(cwt));          
          #else
            if((UICC_baseband_profile.version == SYS_BB_VER_SGOLD) && 
               (UICC_baseband_profile.revision < SYS_BB_REV_12))
            {
              UICC_t1_block_ctrl_state = UICC_RX_PAR_BLOCK_ERROR;
              par_ovr_dma_cnt = 256;
              UICC_start_timer(2*UICC_etu2usec(cwt));          
            }
            else
            { /* UTP: WS00002552 corrected. Overrun error is no longer generated after a Rx parity error. */
              #ifndef UICC_NO_DEBUG
                debug = UICC_t1_block_ctrl_state;
              #endif
              UICC_dma_disable_control();
              UICC_t1_block_ctrl_state = UICC_OVR_ERROR;
              UICC_disable_char_timeout_interrupt();
              UICC_start_timer(2*UICC_etu2usec(cwt));          
            }
          #endif
          break;        

        /*----------------------------*/
        case UICC_T1_PARITY_ERROR:
        case UICC_PARITY_ERROR:        
        /*----------------------------*/       
          #if defined(SGOLD2) || defined(SGOLD3) || defined(SGOLDRADIO)
            switch(UICC_dma_get_transfer_cnt())
            {
              case 0: UICC_SETBIT(parity_error, RX_PARITY_ERROR_LEN_POS); break;
              case 1: UICC_SETBIT(parity_error, RX_PARITY_ERROR_PCB_POS); break;
              case 2: UICC_SETBIT(parity_error, RX_PARITY_ERROR_NAD_POS); break;
           }
           break;
          #else /* NOT SG2 or SG3 */
            if((UICC_baseband_profile.version == SYS_BB_VER_SGOLD) && 
               (UICC_baseband_profile.revision < SYS_BB_REV_12))
            {
              UICC_t1_block_ctrl_state = UICC_RX_PAR_BLOCK_ERROR;
              par_ovr_dma_cnt = 256;
              UICC_start_timer(2*UICC_etu2usec(cwt));          
            }
            else
            { /* UTP: WS00002552 corrected. Overrun error is no longer generated after a Rx parity error. */

              UICC_t1_block_ctrl_state = UICC_RX_INF_EPILOGUE;
              UICC_SETBIT(parity_error, RX_PARITY_ERROR_LEN_POS); 
              UICC_dma_rx_control(254 + UICC_EPILOGUE_SIZE, 
                                  &UICC_prologue[UICC_INF_POS], 
                                  UICC_USIM_FLOW_CONTROL);

              UICC_start_t1_mode(254);
            }
            break;        
          #endif
          
        /*----------------------------*/
        case UICC_CHARACTER_OK:
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;
      }    
      break;

    /*============================================*/
    /*                                            */
    /*============================================*/
    case UICC_CHAR_RX_INF_EPILOGUE:
      switch(event)
      {
        /*----------------------------*/
        case UICC_PARITY_ERROR:
        /*----------------------------*/
          UICC_SETBIT(parity_error, RX_PARITY_ERROR_INF_POS);
          /* break; removed by purpose */

        /*----------------------------*/
        case UICC_CHARACTER_OK:
        /*----------------------------*/
          UICC_prologue[UICC_INF_POS + inf_epilogue_cnt] = UICC_get_character();
          inf_epilogue_cnt++;
          if(inf_epilogue_cnt >= inf_epilogue_cnt_max)
          { /* End of INF/EPILOGUE reception. */
            UICC_t1_block_ctrl_state = UICC_BLOCK_SUCCEEDED;
            UICC_start_timer(UICC_etu2usec(20));          
          }
          else
          {
            UICC_start_timer(UICC_etu2usec(cwt+100));          
          }
          break;

        /*----------------------------*/
        case UICC_TIMER_EXPIRED: 
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_CWT_ERROR;          
          UICC_force_timeout();
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;
       }
       break;

    /*============================================*/
    /* Receiving the INF and Epilogue field.      */
    /*============================================*/
    case UICC_RX_INF_EPILOGUE:
      switch(event)
      {
        #if defined(SGOLD2) ||defined(SGOLD3) || defined(SGOLDRADIO)
        /*----------------------------*/
        case UICC_UNKNOWN_INT:
        /*----------------------------*/
        if(parity_error)
        {
          /* If parity error occured on the EDC character, the T1_END indication 
             could be cleared in SIM_STATUS (as a result of the parity error ISR
             processing. */

          //intentionally fallthrough.
        }
        else
        {
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
        
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;
        }
        #endif

        /*----------------------------*/
        case UICC_T1_END:
        /*----------------------------*/
          UICC_dma_disable_control();

          /* Mask the SIMT1PAR bit from the status register to see if a parity error did occur during 
             block reception. */
          if(UICC_get_simstatus() &  0x0082)
          {
            UICC_SETBIT(parity_error, RX_PARITY_ERROR_INF_POS);           
          }    
      
          UICC_t1_block_ctrl_state = UICC_BLOCK_SUCCEEDED;
          UICC_force_timeout();
          break;

        /*----------------------------*/
        case UICC_CHTIMEOUT:
        /*----------------------------*/
          /* See if the timeout could be the result of a parity error corrupted LEN field. */
          if(UICC_TESTBIT(parity_error, RX_PARITY_ERROR_LEN_POS))
          { /* The length has been corrupted - pretend that the frame was successful (but with 
               parity error). */
            UICC_dma_disable_control();
            UICC_t1_block_ctrl_state = UICC_BLOCK_SUCCEEDED;  
            UICC_force_timeout();
          }
          else
          { 
            UICC_t1_block_ctrl_state = UICC_CWT_ERROR;
            UICC_start_timer(T_CLK_PD);
          }
          break;

        /*----------------------------*/
        case UICC_TIMER_EXPIRED: /* WS:2550 workaround. */
        /*----------------------------*/
          #if defined (SGOLD2) || defined (SGOLDRADIO)
            if(UICC_dma_get_transfer_cnt() < cwt_timeout_dma_cnt)
            { /* Something has been transferred - let's stau in this state and see if this also 
                 is the case after yet another round of CWT. */
              cwt_timeout_dma_cnt = UICC_dma_get_transfer_cnt();
              UICC_start_timer(2*UICC_etu2usec(cwt));                       
            }
            else
            {
              UICC_t1_block_ctrl_state = UICC_CWT_ERROR;          
              UICC_force_timeout();
            }
          #else
            if(UICC_baseband_profile.version == SYS_BB_VER_SGOLD)
            { /* UTP: WS00002550: RX character timeout does not work - use a core timer to monitor the violation 
                 of a character distance. */
              if(UICC_dma_get_transfer_cnt() < cwt_timeout_dma_cnt)
              { /* Something has been transferred - let's stau in this state and see if this also 
                   is the case after yet another round of CWT. */
                cwt_timeout_dma_cnt = UICC_dma_get_transfer_cnt();
                UICC_start_timer(2*UICC_etu2usec(cwt));                       
              }
              else
              {
                UICC_t1_block_ctrl_state = UICC_CWT_ERROR;          
                UICC_force_timeout();
              }
            }
            else
            { /* UTP: WS00002550 corrected. RX character timeout is now working again. */
              UICC_t1_block_ctrl_state = UICC_CWT_ERROR;          
              UICC_force_timeout();
            }
          #endif
          break;

        /*----------------------------*/
        case UICC_OVERRUN_ERROR:
        /*----------------------------*/
          #if defined (SGOLD2) || defined (SGOLD3) || defined (SGOLDRADIO)
            #ifndef UICC_NO_DEBUG
              debug = UICC_t1_block_ctrl_state;
            #endif
            UICC_dma_disable_control();
            UICC_t1_block_ctrl_state = UICC_OVR_ERROR;
            UICC_disable_char_timeout_interrupt();
            UICC_start_timer(2*UICC_etu2usec(cwt));
          #else
            if((UICC_baseband_profile.version == SYS_BB_VER_SGOLD) && 
               (UICC_baseband_profile.revision < SYS_BB_REV_12))
            {
              par_ovr_dma_cnt = UICC_dma_get_transfer_cnt();
            
              if (par_ovr_dma_cnt > 2)
              { /* We are not at the end of a block reception because the DMA counter still holds 
                   a valid value. */
                par_ovr_dma_cnt -= 2;
                UICC_t1_block_ctrl_state = UICC_RX_PAR_BLOCK_ERROR;
                UICC_start_timer(2*UICC_etu2usec(cwt));          
              }
              else
              { /* We are at the end of a block reception because the DMA counter is . */
                UICC_t1_block_ctrl_state = UICC_RX_PAR_BLOCK_FAILED;          
                UICC_force_timeout();          
              } 
            }      
            else
            { /* UTP: WS00002552 corrected. Overrun error is no longer generated after a Rx parity error. */
              #ifndef UICC_NO_DEBUG
                debug = UICC_t1_block_ctrl_state;
              #endif
              UICC_dma_disable_control();
              UICC_t1_block_ctrl_state = UICC_OVR_ERROR;
              UICC_disable_char_timeout_interrupt();
              UICC_start_timer(2*UICC_etu2usec(cwt));
            }
          #endif
          break;        

        /*----------------------------*/
        case UICC_T1_PARITY_ERROR:
        case UICC_PARITY_ERROR:
        /*----------------------------*/
          /* Mark that a parity error has been detected during the INF reception. */
          #if defined(SGOLD2) || defined(SGOLD3) || defined(SGOLDRADIO)
             UICC_SETBIT(parity_error, RX_PARITY_ERROR_INF_POS);           
          #else /*NOT SGOLD2 or SGOLD3*/
            UICC_SETBIT(parity_error, RX_PARITY_ERROR_INF_POS);           
            UICC_t1_block_ctrl_state = UICC_BLOCK_SUCCEEDED;
            UICC_force_timeout();
          #endif

          break;

        /*----------------------------*/
        case UICC_CHARACTER_OK:

        case UICC_DMA_END:               

        case UICC_T1_BWT_EXPIRED:
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;
      }    
      break;

    /*============================================*/
    /* UTP SW 2552 workaround: If Rx parity error */
    /* is detected an overrun interrupt is raised */
    /* on the next Rx character. This provlem is  */
    /* handled via this state.                    */
    /*============================================*/
    case UICC_RX_PAR_BLOCK_ERROR:
      switch(event)
      {
        /*----------------------------*/
        case UICC_CHTIMEOUT:
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
        UICC_t1_block_ctrl_state = UICC_RX_PAR_BLOCK_FAILED;          
        UICC_force_timeout();          
        break;

        /*----------------------------*/
        case UICC_OVERRUN_ERROR:
        /*----------------------------*/
        if(par_ovr_dma_cnt)
        { /* We are not at the end of a block reception because the DMA variable still holds a value. */
          /* Stay in this state. */
          par_ovr_dma_cnt--;
          UICC_start_timer(2*UICC_etu2usec(cwt));
        }
        else
        { /* We are at the end of a block reception because the DMA variable is zero. */
          UICC_t1_block_ctrl_state = UICC_RX_PAR_BLOCK_FAILED;          
          UICC_force_timeout();          
        }
        break; 

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_start_timer(2*UICC_etu2usec(cwt));
          break;
      }
      break;
        
    /*============================================*/
    /* UTP SW 2552 workaround: If Rx parity error */
    /* is detected an overrun interrupt is raised */
    /* on the next Rx character. This provlem is  */
    /* handled via this state.                    */
    /*============================================*/
    case UICC_RX_PAR_BLOCK_FAILED:
      switch(event)
      {
        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
        UICC_t1_block_ctrl_state = UICC_BLOCK_READY;  
        instruction_failed(UICC_CHAR_PARITY_ERROR);
        break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_force_timeout();
          break;
      }
      break;


  #else /* defined (UICC_ALFEEN_ENABLE) */

    /*============================================*/
    /* Transfer the Prologue part of the Rx Frame.*/
    /*============================================*/
    case UICC_RX_PROLOGUE_ALFEEN:
      switch(event)
      {
         /*----------------------------*/
        case UICC_UNKNOWN_INT:
        /*----------------------------*/        
          UICC_t1_block_ctrl_state = UICC_RX_FRAME; 
        
          /*Check LEN error (SIMT1LFPAR) this bit is set when end of receive prolog*/
          if(UICC_get_simstatus() &  0x0200) /*Check if it is parity error in LEN  (SIMT1LFPAR) */ 
          {
            UICC_SETBIT(parity_error, RX_PARITY_ERROR_LEN_POS); 
            length_error = TRUE;
          }

          break;

        /*----------------------------*/
        case UICC_T1_BWT_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BWT_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;

        /*----------------------------*/
        case UICC_CHTIMEOUT:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_CWT_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;

        /*----------------------------*/
        case UICC_OVERRUN_ERROR:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_dma_disable_control();
          UICC_t1_block_ctrl_state = UICC_OVR_ERROR;
          UICC_disable_char_timeout_interrupt();
          UICC_start_timer(2*UICC_etu2usec(cwt));
          break;        

        /*----------------------------*/
        case UICC_T1_PARITY_ERROR:
        case UICC_PARITY_ERROR:        
        /*----------------------------*/
          /*Read SIM status to check if the parity occur in LEN or not, cannot use the DMA count because the HW control the DMA*/
          if(UICC_get_simstatus() &  0x0200) /*Check if it is parity error in LEN */ 
          {
            UICC_SETBIT(parity_error, RX_PARITY_ERROR_LEN_POS); 
            length_error = TRUE;
          }
          else  /*NOT parity in LEN then in NAD or PCB*/             
            UICC_SETBIT(parity_error, RX_PARITY_ERROR_NAD_OR_PCB_POS); 
          
          break;

        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;
      }

      break;
   
    /*============================================*/
    /* Receiving the Prolog, INF and Epilogue field.      */
    /*============================================*/
    case UICC_RX_FRAME:
      switch(event)
      {
        /*----------------------------*/
         /*Sometime  when LEN = 0 and EDC 's PARITY error then UNKNOWN interrupt occur instead of T1_END !!!*/
        case UICC_UNKNOWN_INT:
        case UICC_T1_END:
        /*----------------------------*/
          UICC_dma_disable_control();

          /* Mask the SIMT1PAR bit from the status register to see if a parity error did occur during 
             block reception. */
          if(UICC_get_simstatus() &  0x0082)
          {
            UICC_SETBIT(parity_error, RX_PARITY_ERROR_INF_POS);           
          }    
      
          UICC_t1_block_ctrl_state = UICC_BLOCK_SUCCEEDED;
          UICC_force_timeout();
          break;

        /*----------------------------*/
        case UICC_CHTIMEOUT:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_CWT_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;

        /*----------------------------*/
        case UICC_OVERRUN_ERROR:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_dma_disable_control();
          UICC_t1_block_ctrl_state = UICC_OVR_ERROR;
          UICC_disable_char_timeout_interrupt();
          UICC_start_timer(2*UICC_etu2usec(cwt));
          break;        

        /*----------------------------*/
        case UICC_T1_PARITY_ERROR:
        case UICC_PARITY_ERROR:        
        /*----------------------------*/       
          UICC_SETBIT(parity_error, RX_PARITY_ERROR_INF_POS); 
          break;
        
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;
      }    
      break;
    #endif 
	
    /*============================================*/
    /* The block transfer/reception succeeded.    */
    /*============================================*/
    case UICC_BLOCK_SUCCEEDED:
      switch(event)
      {
        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;  
          /* Inform main statemachine of the execution result, that the
             clock now is stopped and that power down can be accepted. */


        #ifndef UICC_ALFEEN_ENABLE
          #if defined (UICC_FIFO_MODE)
             // For alignment
            for(cnt=0;cnt<(UICC_prologue[UICC_LEN_POS] + UICC_EPILOGUE_SIZE);cnt++)
                UICC_prologue[UICC_INF_POS+cnt] = UICC_prologue[UICC_INF_POS+cnt+1];
          #endif 
        #endif

        
          /* Check if an illegal length was used. */
          if(length_error)
          {
            UICC_t1_data_link_handler(UICC_LEN_VIOLATION);
          }
          else
          {        
            /* Check to see if a parity error did occur during block reception. */
            if(parity_error)
            {
              UICC_t1_data_link_handler(UICC_CHAR_PARITY_ERROR);          
            }
            else
            {
             UICC_t1_data_link_handler(UICC_BLOCK_RECEIVED);
            }        
          }
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_CLK_PD);
          break;         
      }    
      break;

    /*============================================*/
    /* Block error (unknown event) encountered.   */
    /*============================================*/
    case UICC_BLOCK_ERROR:
      switch(event)
      {
        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;  
          instruction_failed(UICC_UNKNOWN_EVENT);
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_force_timeout();
          break;
      }    
      break;

    /*============================================*/
    /* Overrun error encountered.                 */
    /*============================================*/
    case UICC_OVR_ERROR:
      switch(event)
      {
        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;  
          instruction_failed(UICC_CHAR_OVERRUN_ERROR);
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_force_timeout();
          break;
      }    
      break;

    /*============================================*/
    /* TX error encountered.                      */
    /*============================================*/
    case UICC_TX_ERROR:
      switch(event)
      {
        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;  
          instruction_failed(UICC_CHAR_TIMEOUT);
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_force_timeout();
          break;
      }    
      break;

    /*============================================*/
    /* BlockWaitingTimer expiration detected.     */
    /*============================================*/
    case UICC_BWT_ERROR:
      switch(event)
      {
        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;  
          instruction_failed(UICC_BWT_TIMEOUT);
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_force_timeout();
          break;
      }    
      break;

    /*============================================*/
    /* CharacterWaitingTimer expiration detected. */
    /*============================================*/
    case UICC_CWT_ERROR:
      switch(event)
      {
        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;  
          /* Check if the timeout was generated as a result of parity error or an 
             illegal length. */
          if(parity_error)
          {
            instruction_failed(UICC_CHAR_PARITY_ERROR);          
          }
          else if(length_error)
          {
            instruction_failed(UICC_LEN_VIOLATION);
          }
          else
          {
            instruction_failed(UICC_CWT_TIMEOUT);
          }
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_force_timeout();
          break;
      }    
      break;

    /*============================================*/
    /* Wait until it is allowed to stop the clock.*/
    /*============================================*/
    case UICC_BLOCK_AWAIT_CLOCK_OFF:
      switch(event)
      {
        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY_CLK_OFF;  
          UICC_stop_clock_conditional(UICC_card_profile[UICC_current_reader].used_clock_stop_mode, UICC_current_reader);
          UICC_switch_uart_clock_off(); /* Switch off the internal UART clock - also in line with UTP: WS00004795 */
          UICC_t1_data_link_handler(UICC_BLOCK_CLOCK_STOPPED);
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_force_timeout();
          break;
      }    
      break;

    /*==========================================*/
    /* Default.                                 */
    /*==========================================*/
    default:
      #ifndef UICC_NO_DEBUG
        debug = UICC_t1_block_ctrl_state;
      #endif
      break;
  }

  #ifndef UICC_NO_DEBUG
    if(debug != UICC_BLOCK_NONE)
    {
      UICC_log_data_ptr[0] = debug;
      UICC_log_data_ptr[1] = event;
      UICC_log_data_ptr[2] = UICC_current_reader;
      UICC_log_data_ptr[3]  = (ubyte) (SIM_CTRL>>8);
      UICC_log_data_ptr[4]  = SIM_CTRL;
      UICC_log_data_ptr[5]  = ((ubyte) (SIM_STATUS>>8)) | ((ubyte)SIM_DMA<<7);
      UICC_log_data_ptr[6]  = SIM_STATUS;
      UICC_log_data_ptr[7]  = SIM_BRF;
      UICC_log_data_ptr[8]  = SIM_IRQEN;
      UICC_log_data_ptr[9]  = SIM_RXSPC;
      UICC_log_data_ptr[10] = SIM_TXSPC;
      UICC_log_data_ptr[11] = SIM_CTRL2;
      UICC_log_data_ptr[12] = SIM_T1CTRL;
      UICC_log_data_ptr[13] = (ubyte) ((ubyte)(SIM_RIS<<4) | SIM_MIS);
      UICC_log_data_ptr[14] = (ubyte) ((ubyte)(SIM_ICR<<4) | SIM_ISR);
      UICC_log_data_ptr[15]  = (ubyte)(UICC_dma_get_transfer_cnt()>>8);
      UICC_log_data_ptr[16]  = (ubyte)(UICC_dma_get_transfer_cnt());	  
      memcpy(&UICC_log_data_ptr[17], &UICC_prologue[0], 3);
      UICC_log_data_ptr[20] = UICC_inf[UICC_EPILOGUE_POS];
      memcpy(&UICC_log_data_ptr[21], &UICC_inf[0], 6);      
      UICC_raise_exception(TRAP_UICC_T1_BLOCK, UICC_T1_BLOCK_ERROR_1, 27); 

      debug = UICC_BLOCK_NONE;
    }
  #endif
}

#endif

#else /* Not UICC_HW_CONTROLLED_T1 -> SW controlled T1 block handling. */

/*******************************************************************************
* Function:... UICC_t1_block_handler
* Parameters:. event: The event triggering the statemachine.
* Returns:.... -
* Description: This statemachine/engine is controlling the execution of the T=1
*              protocol block instructions.
* Created:.... 20.09.02 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
void UICC_reset_t1_block_handler(void)
{
  UICC_stop_timer();
  UICC_t1_block_ctrl_state = UICC_BLOCK_READY_CLK_OFF;
}


void UICC_t1_block_handler(T_UICC_T1_BLOCK_EVENTS event)
{
  switch(UICC_t1_block_ctrl_state)
  {
    /*============================================*/
    /* Ready - to execute block transfer          */
    /*============================================*/
    case UICC_BLOCK_READY_CLK_OFF:
      switch(event)
      {
        /*----------------------------*/
        case UICC_INITIATE_BLOCK:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;

          UICC_enable_card_interface();

          UICC_set_rxtx_space(UICC_profile_parameter.bgt);      /* Min. spacing with respect to the last Rx character of 
                                           the previous received Rx block (22 ETU is the minimum
                                           specified block spacing spacing). */

          #if defined (SGOLD2) || defined (SGOLD3) || defined (SGOLDRADIO)
            UICC_set_txtx_space(0x01);     /* Tx Spacing. */
          #else
            if((UICC_baseband_profile.version == SYS_BB_VER_SGOLD) && 
               (UICC_baseband_profile.revision < SYS_BB_REV_12))
            {
              UICC_set_txtx_space(0x03);     /* Spacing between TX characters. Must be larger than 2 
                                                to avoid 'ghost' characters in after parity error 
                                                retransmission. */
            }
            else
            { /* UTP: WS00002548 corrected. Tx space does not need to hold the TX space value of minimum 3. */
              UICC_set_txtx_space(0x01);     /* Tx Spacing. */
            }
          #endif

          UICC_select_character_mode();
          UICC_enable_usim_ok_isr();
          UICC_enable_character_interrupt();
          UICC_disable_t0_end_interrupt();
          UICC_enable_parity_error_interrupt();
          UICC_enable_overrun_error_interrupt();
          UICC_disable_character_retransmission();
          UICC_enable_parity_error_detection(); 

          UICC_select_high_clock_freq();
          UICC_activate_clk(UICC_current_reader);
          UICC_start_timer(T_CLK_PU);
          break;

        /*----------------------------*/
        case UICC_CHARACTER_OK:
        /*----------------------------*/
          /* Receiving a character at this stage is of no use, but since an overrun otherwise would 
             occur, will thew character be read now. */
          UICC_get_character();        
          break;
        
        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_ERROR);
          break;
       }
       break;

    /*=========================================*/
    /* Wait until the clock is up and running. */
    /*=========================================*/
    case UICC_BLOCK_READY:
      switch(event)
      {
        /*----------------------------*/
        case UICC_INITIATE_BLOCK:
        /*----------------------------*/
          UICC_enable_card_interface();
          /* Break left out on purpose. */          

        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_TX_NAD;
          parity_error = 0; 
          tx_data_cnt = 0;
          rx_data_cnt = 0;

          /* Calculate Block Waiting Timer value. */ 
          /* BWT = 11 [ETU] + 2^BWI * 960 * 372/3.25 [uS] */
          bwt = UICC_etu2usec(11+UICC_profile_parameter.bwt_extension) + (1 << (UICC_card_profile[UICC_current_reader].bwi))*960*115; /* [us] */ 
          tx_spc = (ushort)UICC_etu2usec(TX_SPC);
          
          if(UICC_t1_tpdu_ctrl.bwt_multiplier)
          {
            bwt *= UICC_t1_tpdu_ctrl.bwt_multiplier;
            /* BWT multiplier has only one shot functionality and is therefore reset. */
            UICC_t1_tpdu_ctrl.bwt_multiplier = 1;  
          }
        
          /* Calculate Character Waiting Timer value. */           
          /* CWT = 11 + 2^CWI [ETU] */
          cwt = UICC_etu2usec(11 + UICC_profile_parameter.cwt_extension + (1 << UICC_card_profile[UICC_current_reader].cwi) + 20 /*safety margin*/); /* [us] */

          UICC_start_timer(T_CHAR);
          UICC_send_character(UICC_prologue[UICC_NAD_POS]);
          break;

        /*----------------------------*/
        case UICC_STOP_BLOCK_CLOCK:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_AWAIT_CLOCK_OFF;
          UICC_start_timer(T_CLK_PD); 
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_ERROR);
          break;
       }
       break;

    /*=========================*/
    /* Send Node ADdress byte. */
    /*=========================*/
    case UICC_TX_NAD:
      switch(event)
      {
        /*----------------------------*/
        case UICC_CHARACTER_OK:
        /*----------------------------*/
          UICC_start_guard_timer(tx_spc); /* Create distance to the next Tx-character. */
          break;
            
        /*----------------------------*/
        case UICC_GUARD_TIMER_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_TX_PCB;
          UICC_start_timer(T_CHAR);
          UICC_send_character(UICC_prologue[UICC_PCB_POS]);
          break;

        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;
          instruction_failed(UICC_CHAR_TIMEOUT);
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(bwt);
          break;
       }
       break;

    /*=============================*/
    /* Send Protocol Control Byte. */
    /*=============================*/
    case UICC_TX_PCB:
      switch(event)
      {
        /*----------------------------*/
        case UICC_CHARACTER_OK:
        /*----------------------------*/
          UICC_start_guard_timer(tx_spc); /* Create distance to the next Tx-character. */
          break;
            
        /*----------------------------*/
        case UICC_GUARD_TIMER_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_TX_LEN;
          UICC_start_timer(T_CHAR);
          UICC_send_character(UICC_prologue[UICC_LEN_POS]);
          break;

        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;
          instruction_failed(UICC_CHAR_TIMEOUT);
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(bwt);
          break;
       }
       break;

    /*==========================================================*/
    /* Send the LENgth byte (the length of the INF field size). */
    /*==========================================================*/
    case UICC_TX_LEN:
      switch(event)
      {
        /*----------------------------*/
        case UICC_CHARACTER_OK:
        /*----------------------------*/
          UICC_start_guard_timer(tx_spc); /* Create distance to the next Tx-character. */
          break;
            
        /*----------------------------*/
        case UICC_GUARD_TIMER_EXPIRED:
        /*----------------------------*/
          if(tx_data_cnt < UICC_prologue[UICC_LEN_POS])
          {
            UICC_t1_block_ctrl_state = UICC_TX_INF;
            UICC_start_timer(T_CHAR);
            UICC_send_character(UICC_inf[tx_data_cnt++]);
          }
          else
          {
            UICC_t1_block_ctrl_state = UICC_TX_EDC;
            UICC_start_timer(T_CHAR);
            UICC_send_character(UICC_inf[UICC_EPILOGUE_POS]);
          }
          break;

        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;
          instruction_failed(UICC_CHAR_TIMEOUT);
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(bwt);
          break;
       }
       break;

    /*========================================*/
    /* Send the INformation Field characters. */
    /*========================================*/
    case UICC_TX_INF:
      switch(event)
      {
        /*----------------------------*/
        case UICC_CHARACTER_OK:
        /*----------------------------*/
          UICC_start_guard_timer(tx_spc); /* Create distance to the next Tx-character. */
          break;
            
        /*----------------------------*/
        case UICC_GUARD_TIMER_EXPIRED:
        /*----------------------------*/
          if(tx_data_cnt < UICC_prologue[UICC_LEN_POS]) /* More data */
          {
            UICC_start_timer(T_CHAR);
            UICC_send_character(UICC_inf[tx_data_cnt++]);
          }
          else
          {
            UICC_t1_block_ctrl_state = UICC_TX_EDC;
            UICC_start_timer(T_CHAR);
            UICC_send_character(UICC_inf[UICC_EPILOGUE_POS]);
          }
          break;

        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;
          instruction_failed(UICC_CHAR_TIMEOUT);
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(bwt);
          break;
       }
       break;

    /*====================================*/
    /* Send the Error Detection Code byte */
    /*====================================*/
    case UICC_TX_EDC:
      switch(event)
      {
        /*----------------------------*/
        case UICC_CHARACTER_OK:
        /*----------------------------*/
          /* The Tx frame has now been transmitted - await the Rx frame. */ 
          UICC_t1_block_ctrl_state = UICC_RX_NAD;  
          UICC_start_timer(bwt);
          break;

        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;
          instruction_failed(UICC_CHAR_TIMEOUT);
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(bwt);
          break;
       }
       break;

    /*============================*/
    /* Receive Node ADdress byte. */
    /*============================*/
    case UICC_RX_NAD:
      switch(event)
      {
        /*----------------------------*/
        case UICC_PARITY_ERROR:
        /*----------------------------*/
          UICC_SETBIT(parity_error, RX_PARITY_ERROR_NAD_POS);
          /* 'break' removed by purpose. */

        /*----------------------------*/
        case UICC_CHARACTER_OK:
        /*----------------------------*/
          UICC_prologue[UICC_NAD_POS] = UICC_get_character();          
          UICC_t1_block_ctrl_state = UICC_RX_PCB;  
          UICC_start_timer(cwt);
          break;
         
        /*----------------------------*/                
        case UICC_OVERRUN_ERROR:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_ERROR);
          break;
          
        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;
          instruction_failed(UICC_BWT_TIMEOUT);
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_ERROR);
          break;
       }
       break;

    /*================================*/
    /* Receive Protocol Control Byte. */
    /*================================*/
    case UICC_RX_PCB:
      switch(event)
      {
        /*----------------------------*/
        case UICC_PARITY_ERROR:
        /*----------------------------*/
          UICC_SETBIT(parity_error, RX_PARITY_ERROR_PCB_POS);
          /* 'break' removed by purpose. */

        /*----------------------------*/
        case UICC_CHARACTER_OK:
        /*----------------------------*/
          UICC_prologue[UICC_PCB_POS] = UICC_get_character();          
          UICC_t1_block_ctrl_state = UICC_RX_LEN;  
          UICC_start_timer(cwt);
          break;

        /*----------------------------*/                
        case UICC_OVERRUN_ERROR:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_ERROR);
          break;

        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;
          instruction_failed(UICC_CWT_TIMEOUT);
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_ERROR);
          break;
       }
       break;

    /*=============================================================*/
    /* Receive the LENgth byte (the length of the INF field size). */
    /*=============================================================*/
    case UICC_RX_LEN:
      switch(event)
      {
        /*----------------------------*/
        case UICC_PARITY_ERROR:
        /*----------------------------*/
          UICC_SETBIT(parity_error, RX_PARITY_ERROR_LEN_POS);
          /* 'break' removed by purpose. */

        /*----------------------------*/
        case UICC_CHARACTER_OK:
        /*----------------------------*/
          UICC_prologue[UICC_LEN_POS] = UICC_get_character();          
          if(UICC_prologue[UICC_LEN_POS])
          {
            UICC_t1_block_ctrl_state = UICC_RX_INF;  
          }
          else
          {
            UICC_t1_block_ctrl_state = UICC_RX_EDC;  
          }
          UICC_start_timer(cwt);
          break;

        /*----------------------------*/                
        case UICC_OVERRUN_ERROR:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_ERROR);
          break;

        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;
          instruction_failed(UICC_CWT_TIMEOUT);
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_ERROR);
          break;
       }
       break;

    /*===========================================*/
    /* Receive the INformation Field characters. */
    /*===========================================*/
    case UICC_RX_INF:
      switch(event)
      {
        /*----------------------------*/
        case UICC_PARITY_ERROR:
        /*----------------------------*/
          UICC_SETBIT(parity_error, RX_PARITY_ERROR_INF_POS);
          /* 'break' removed by purpose. */

        /*----------------------------*/
        case UICC_CHARACTER_OK:
        /*----------------------------*/
          UICC_inf[rx_data_cnt] = UICC_get_character();          
          rx_data_cnt++;
          if(rx_data_cnt == UICC_prologue[UICC_LEN_POS])
          {
            UICC_t1_block_ctrl_state = UICC_RX_EDC;
          }
          UICC_start_timer(cwt);
          break;

        /*----------------------------*/                
        case UICC_OVERRUN_ERROR:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_ERROR);
          break;

        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;
          instruction_failed(UICC_CWT_TIMEOUT);
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_ERROR);
          break;
       }
       break;

    /*=======================================*/
    /* Receive the Error Detection Code byte */
    /*=======================================*/
    case UICC_RX_EDC:
      switch(event)
      {
        /*----------------------------*/
        case UICC_PARITY_ERROR:
        /*----------------------------*/
          UICC_SETBIT(parity_error, RX_PARITY_ERROR_EDC_POS);
          /* 'break' removed by purpose. */

        /*----------------------------*/
        case UICC_CHARACTER_OK:
        /*----------------------------*/
          UICC_inf[UICC_EPILOGUE_POS] = UICC_get_character();          
          UICC_t1_block_ctrl_state = UICC_BLOCK_SUCCEEDED;  
          UICC_start_timer(T_CLK_PD); /* Make the appropriate distance before stopping the clock. */ 
          break;

        /*----------------------------*/                
        case UICC_OVERRUN_ERROR:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_ERROR);
          break;

        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;
          instruction_failed(UICC_CWT_TIMEOUT);
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_ERROR);
          break;
       }
       break;

    /*==================================*/
    /* The block reception succeeded.   */
    /*==================================*/
    case UICC_BLOCK_SUCCEEDED:
      switch(event)
      {
        /*----------------------------*/
        case UICC_CHARACTER_OK:
        /*----------------------------*/
          /* Receiving a character at this stage is of no use, but since an overrun otherwise would 
             occur, will thew character be read now. */
          UICC_get_character();        
          break;

        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;  
          /* Inform main statemachine of the execution result, that the
             clock now is stopped and that power down can be accepted. */
          if(parity_error)
          {
            UICC_t1_data_link_handler(UICC_CHAR_PARITY_ERROR);          
          }
          else
          {
            UICC_t1_data_link_handler(UICC_BLOCK_RECEIVED);
          }        
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_ERROR);
          break;
       }
       break;

    /*===============================*/
    /* The block reception failed.   */
    /*===============================*/
    case UICC_BLOCK_ERROR:
      switch(event)
      {
         /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY;
          instruction_failed(UICC_CHAR_OVERRUN_ERROR);
          break;
          
        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_start_timer(cwt);
          break;
       }
       break;

    /*============================================*/
    /* Wait until it is allowed to stop the clock.*/
    /*============================================*/
    case UICC_BLOCK_AWAIT_CLOCK_OFF:
      switch(event)
      {
        /*----------------------------*/
        case UICC_TIMER_EXPIRED:
        /*----------------------------*/
          UICC_t1_block_ctrl_state = UICC_BLOCK_READY_CLK_OFF;  
          UICC_stop_clock_conditional(UICC_card_profile[UICC_current_reader].used_clock_stop_mode, UICC_current_reader);
          UICC_switch_uart_clock_off();
          UICC_t1_data_link_handler(UICC_BLOCK_CLOCK_STOPPED);
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_block_ctrl_state;
          #endif
          UICC_t1_block_ctrl_state = UICC_BLOCK_ERROR;
          UICC_start_timer(T_ERROR);
          break;
      }    
      break;

    /*==========================================*/
    /* Default                                  */
    /*==========================================*/
    default:
      #ifndef UICC_NO_DEBUG
        debug = UICC_t1_block_ctrl_state;
      #endif
      break;
  }

  #ifndef UICC_NO_DEBUG
    if(debug != UICC_BLOCK_NONE)
    {
      UICC_log_data_ptr[0] = debug;
      UICC_log_data_ptr[1] = event;
      UICC_log_data_ptr[2] = UICC_current_reader;
      memcpy(&UICC_log_data_ptr[3], &UICC_prologue[0], UICC_PROLOGUE_SIZE);
      UICC_log_data_ptr[6] = UICC_inf[UICC_EPILOGUE_POS];
      memcpy(&UICC_log_data_ptr[7], &UICC_inf[0], 12);      
      UICC_raise_exception(TRAP_UICC_T1_BLOCK, UICC_T1_BLOCK_ERROR_1, 19); 

      UICC_log_data_ptr[0]  = (ubyte) (SIM_CTRL>>8);
      UICC_log_data_ptr[1]  = SIM_CTRL;
      UICC_log_data_ptr[2]  = ((ubyte) (SIM_STATUS>>8)) | ((ubyte)SIM_DMA<<7);
      UICC_log_data_ptr[3]  = SIM_STATUS;
      UICC_log_data_ptr[4]  = SIM_BRF;
      UICC_log_data_ptr[5]  = SIM_IRQEN;
      UICC_log_data_ptr[6]  = SIM_RXSPC;
      UICC_log_data_ptr[7]  = SIM_TXSPC;
      UICC_log_data_ptr[8]  = SIM_CTRL2;
      UICC_log_data_ptr[9]  = SIM_T1CTRL;
      UICC_log_data_ptr[10] = (ubyte) (SIM_INS>>8);
      UICC_log_data_ptr[11] = SIM_INS;
      UICC_log_data_ptr[12] = SIM_TX;
      UICC_log_data_ptr[13] = SIM_RX;
      UICC_log_data_ptr[14] = SIM_P3;
      UICC_log_data_ptr[15] = SIM_SW1;
      UICC_log_data_ptr[16] = SIM_SW2;
      UICC_log_data_ptr[17] = (ubyte) ((ubyte)(SIM_RIS<<4) | SIM_MIS);
      UICC_log_data_ptr[18] = (ubyte) ((ubyte)(SIM_ICR<<4) | SIM_ISR);
      UICC_raise_exception(TRAP_UICC_T1_BLOCK, UICC_T1_BLOCK_ERROR_2, 19); 

      debug = UICC_BLOCK_NONE;
    }
  #endif
}
#endif 
                                                 /* End of file.              */
