/**
* @file main.c
* @brief flags
*
* Copyright (C) 2023 Sanechips Technology Co., Ltd.
* @author
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License version 2 as
* published by the Free Software Foundation. ѡGPLv2 Licence
*
*/


/*******************************************************************************
 *                           Include header files                              *
 ******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <errno.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <dirent.h>
#include <getopt.h>
#include <unistd.h>

#include "pub_flags.h"
#include "flags_api.h"


/*******************************************************************************
 *                             Macro definitions                               *
 ******************************************************************************/
#define FLAGS_UTILS_GET	_IOWR('b', 1, T_FLAGS_INFO)
#define FLAGS_UTILS_SET	_IOWR('b', 2, T_FLAGS_INFO)


/*******************************************************************************
 *                             Type definitions                                *
 ******************************************************************************/
typedef struct
{
    int	option_value;
    int (*func)(char *);
} option_handle_t;



/*******************************************************************************
 *                       Static function declarations                          *
 ******************************************************************************/
static int excute_command_help(char *option_para);
static int excute_command_init(char *option_para);
static int excute_command_get(char *option_para);
static int excute_command_set(char *option_para);
static int excute_command_switch(char *option_para);
static int excute_command_get_ubifs_status(char *option_para);
static int excute_command_set_ubifs_status(char *option_para);

static int excute_command_kernel_get(char *option_para);
static int excute_command_kernel_set(char *option_para);

static int excute_command_get_nocrc(char *option_para);
static int excute_command_set_nocrc(char *option_para);


/*******************************************************************************
 *                       Global variable declarations                          *
 ******************************************************************************/
static char * g_short_string = "higswubtqac:";

static struct option g_long_options[] =
{
    {"help",				no_argument,	NULL,	'h'},
    {"init",				no_argument,	NULL,	'i'},
    {"get",					no_argument,	NULL,	'g'},
    {"set",					no_argument,	NULL,	's'},
    {"switch",				no_argument,	NULL,	'w'},
    {"get-ubifs-status",	no_argument,	NULL,	'u'},
    {"set-ubifs-status",	no_argument,	NULL,	'b'},
    {"kernel-get",			no_argument,	NULL,	't'},
    {"kernel-set",			no_argument,	NULL,	'q'},
    {"get-nocrc",			no_argument,	NULL,	'a'},
    {"set-nocrc",			required_argument,	NULL,	'c'},
    {0, 0, 0, 0}
};

static option_handle_t g_option_handle[] =
{
    {'h',		excute_command_help},
    {'i',		excute_command_init},
    {'g',		excute_command_get},
    {'s',		excute_command_set},
    {'w',		excute_command_switch},
    {'u',		excute_command_get_ubifs_status},
    {'b',		excute_command_set_ubifs_status},
    {'t',		excute_command_kernel_get},
    {'q',		excute_command_kernel_set},
    {'a',		excute_command_get_nocrc},
    {'c',		excute_command_set_nocrc}
};


/*******************************************************************************
 *                       Static function define                          *
 ******************************************************************************/
static void usage(void)
{
    printf("flags_tool [-higswub] \n "
		" -h,	--help                  show usage \n"
		"  -i,	--init                  init flags \n"
		"  -g,	--get                   get flags data \n"
		"  -s,	--set                   set flags data \n"
		"  -w,	--switch                system A/B switch \n"
		"  -u,	--get-ubifs-status      get ubifs status \n"
		"  -b,	--set-ubifs-status      set ubifs status \n"
		"  -t,	--kernel-get            kernel-get flags data \n"
		"  -q,	--kernel-set            kernel-set flags data \n"
		"  -a,	--get-nocrc             get-nocrc flags data \n"
		"  -c,	--set-nocrc             set-nocrc flags data \n");

	return;
}


static int excute_command_help(char *option_para)
{
	usage();

	return 0;
}


static int excute_command_init(char *option_para)
{
	if (flags_init() != 0)
	{
		printf("Err: flags init fail \n");
		
		return -1;
	}
	else
	{
		printf("Log: flags init success \n");
		
	    return 0;
	}
}


static int excute_command_get(char *option_para)
{
	int i = 0;
	T_FLAGS_INFO flags_info = {0};

	if (flags_get(&flags_info) != 0)
	{
		printf("Err: flags get fail \n");
		
		return -1;
	}
	else
	{
		printf("magic_start                     = 0x%x \n",	flags_info.magic_start);
		
		printf("boot_fota_flag.boot_to          = 0x%x \n",	flags_info.boot_fota_flag.boot_to);
		printf("boot_fota_flag.fota_status      = %u \n",	flags_info.boot_fota_flag.fota_status);
		printf("boot_fota_flag.system.status    = 0x%x \n",	flags_info.boot_fota_flag.system.status);
		printf("boot_fota_flag.system.try_cnt	= %d \n",	flags_info.boot_fota_flag.system.try_cnt);
		printf("boot_fota_flag.system2.status	= 0x%x \n",	flags_info.boot_fota_flag.system2.status);
		printf("boot_fota_flag.system2.try_cnt	= %d \n",	flags_info.boot_fota_flag.system2.try_cnt);
		
		printf("boot_env.dualsys_type           = 0x%x \n",	flags_info.boot_env.dualsys_type);
		printf("boot_env.system_boot_env        = %s \n",	flags_info.boot_env.system_boot_env);
		printf("boot_env.system2_boot_env       = %s \n",	flags_info.boot_env.system2_boot_env);
		
		printf("ubifs_status.fs_status          = %d \n",	flags_info.ubifs_status.fs_status);
		printf("ubifs_status.fs_mtd_name        = %s \n",	flags_info.ubifs_status.fs_mtd_name);
		printf("ubifs_status.fs_ubi_vol_name    = %s \n",	flags_info.ubifs_status.fs_ubi_vol_name);
		
		printf("nvro_flag                       = %u \n",	flags_info.nvro_flag);

		printf("ota_system                      = %d \n",   flags_info.ota_system);

		for (i = 0; i < OTA_PARTITION_NUM_MAX; i++)
		{
			printf("index=%d, mtdnum=%d, len=%u          \n", i, flags_info.ota_partiton_info[i].mtdnum, flags_info.ota_partiton_info[i].len);
		}
		
		printf("magic_end                       = 0x%x \n",	flags_info.magic_end);
		
	    return 0;
	}
}


static int excute_command_set(char *option_para)
{
	T_FLAGS_INFO flags_info = {0};
	
    char system_boot_env[128] = "bootEnv1";
    char system2_boot_env[128] = "2bootEnv";

	char fs_mtd_name[16] = "nameMTD";
	char fs_ubi_vol_name[16] = "nameVOL";
	
    flags_info.magic_start = FLAGS_MAGIC;
	
    flags_info.boot_fota_flag.boot_to = DUAL_SYSTEM;
    flags_info.boot_fota_flag.fota_status = 0;
    flags_info.boot_fota_flag.system.status = DUALSYSTEM_STATUS_BOOTABLE;
    flags_info.boot_fota_flag.system.try_cnt = 5;
    flags_info.boot_fota_flag.system2.status = DUALSYSTEM_STATUS_BOOTABLE;
    flags_info.boot_fota_flag.system2.try_cnt = 13;

    flags_info.boot_env.dualsys_type = DUALSYSTEM_AB;
	strncpy(flags_info.boot_env.system_boot_env, system_boot_env, sizeof(flags_info.boot_env.system_boot_env));
	strncpy(flags_info.boot_env.system2_boot_env, system2_boot_env, sizeof(flags_info.boot_env.system2_boot_env));

	flags_info.ubifs_status.fs_status = 0;
	strncpy(flags_info.ubifs_status.fs_mtd_name, fs_mtd_name, sizeof(flags_info.ubifs_status.fs_mtd_name));
	strncpy(flags_info.ubifs_status.fs_ubi_vol_name, fs_ubi_vol_name, sizeof(flags_info.ubifs_status.fs_ubi_vol_name));
	
    flags_info.magic_end = FLAGS_MAGIC;

	if (flags_set(&flags_info) != 0)
	{
		printf("Err: set flags fail \n");
		
		return -1;
	}
	else
	{
		printf("Log: set flags success \n");
		
		return 0;
	}
}

static int excute_command_switch(char *option_para)
{
	T_FLAGS_INFO flags_info = {0};
	int ret = -1;

	ret = flags_get(&flags_info);
	if (ret < 0)
	{
		printf("flags switch flags_get fail\n");
		return ret;
	}
	if (flags_info.boot_fota_flag.boot_to == DUAL_SYSTEM)
	{
		flags_info.boot_fota_flag.boot_to = DUAL_SYSTEM2;  //A->B
		printf("flags switch system A to system B\n");
	}
	else
	{
		flags_info.boot_fota_flag.boot_to = DUAL_SYSTEM;   //B->A
		printf("flags switch system B to system A\n");
	}
	ret = flags_set(&flags_info);
	if (ret < 0)
	{
		printf("flags switch flags_set fail\n");
		return ret;
	}
	printf("flags AB system switch sucess\n");

	return 0;
}

static int excute_command_get_ubifs_status(char *option_para)
{
	T_UBIFS_STATUS ubifs_status = {0};

	if (flags_get_ubifs_status(&ubifs_status) != 0)
	{
		printf("Err: get ubifs status fail \n");
		
		return -1;
	}
	else
	{
		printf("ubifs_status.fs_status          = %d \n",	ubifs_status.fs_status);
		printf("ubifs_status.fs_mtd_name        = %s \n",	ubifs_status.fs_mtd_name);
		printf("ubifs_status.fs_ubi_vol_name    = %s \n",	ubifs_status.fs_ubi_vol_name);

		return 0;
	}
}


static int excute_command_set_ubifs_status(char *option_para)
{
	T_UBIFS_STATUS ubifs_status = {0};

	char fs_mtd_name[16] = "mtdName";
	char fs_ubi_vol_name[16] = "volName";

	ubifs_status.fs_status = 2;
	strncpy(ubifs_status.fs_mtd_name, fs_mtd_name, sizeof(ubifs_status.fs_mtd_name));
	strncpy(ubifs_status.fs_ubi_vol_name, fs_ubi_vol_name, sizeof(ubifs_status.fs_ubi_vol_name));

	if (flags_set_ubifs_status(&ubifs_status) != 0)
	{
		printf("Err: set ubifs status fail \n");
		
		return -1;
	}
	else
	{
		printf("Log: set ubifs status success \n");
		
		return 0;
	}
}


static int excute_command_kernel_get(char *option_para)
{
	int i = 0;
	int fd = -1;
	T_FLAGS_INFO flags_info = {0};

	fd = open("/dev/chardevnode0", O_RDWR);
	if (-1 == open)
	{
		printf("kernel get open chardevnode fail, errno=%d \n", errno);
		return -1;
	}

	if (ioctl(fd, FLAGS_UTILS_GET, &flags_info) < 0)
	{
		printf("Err: ioctl cmd(0x%x) fail, errno=%d \n", FLAGS_UTILS_GET, errno);
		
		close(fd);
		return -1;
	}

	printf("magic_start                     = 0x%x \n",	flags_info.magic_start);
	
	printf("boot_fota_flag.boot_to          = 0x%x \n",	flags_info.boot_fota_flag.boot_to);
	printf("boot_fota_flag.fota_status      = %u \n",	flags_info.boot_fota_flag.fota_status);
	printf("boot_fota_flag.system.status    = 0x%x \n",	flags_info.boot_fota_flag.system.status);
	printf("boot_fota_flag.system.try_cnt	= %d \n",	flags_info.boot_fota_flag.system.try_cnt);
	printf("boot_fota_flag.system2.status	= 0x%x \n",	flags_info.boot_fota_flag.system2.status);
	printf("boot_fota_flag.system2.try_cnt	= %d \n",	flags_info.boot_fota_flag.system2.try_cnt);
	
	printf("boot_env.dualsys_type           = 0x%x \n",	flags_info.boot_env.dualsys_type);
	printf("boot_env.system_boot_env        = %s \n",	flags_info.boot_env.system_boot_env);
	printf("boot_env.system2_boot_env       = %s \n",	flags_info.boot_env.system2_boot_env);
	
	printf("ubifs_status.fs_status          = %d \n",	flags_info.ubifs_status.fs_status);
	printf("ubifs_status.fs_mtd_name        = %s \n",	flags_info.ubifs_status.fs_mtd_name);
	printf("ubifs_status.fs_ubi_vol_name    = %s \n",	flags_info.ubifs_status.fs_ubi_vol_name);
	
	printf("nvro_flag                       = %u \n",	flags_info.nvro_flag);

	printf("ota_system                      = %d \n",   flags_info.ota_system);

	for (i = 0; i < OTA_PARTITION_NUM_MAX; i++)
	{
		printf("index=%d, mtdnum=%d, len=%u          \n", i, flags_info.ota_partiton_info[i].mtdnum, flags_info.ota_partiton_info[i].len);
	}

	printf("magic_end                       = 0x%x \n",	flags_info.magic_end);
	
	close(fd);
    return 0;
}


static int excute_command_kernel_set(char *option_para)
{
	int fd = -1;
	T_FLAGS_INFO flags_info = {0};
	
    char system_boot_env[128] = "bootEnv1";
    char system2_boot_env[128] = "2bootEnv";

	char fs_mtd_name[16] = "nameMTD";
	char fs_ubi_vol_name[16] = "nameVOL";
	
	fd = open("/dev/chardevnode0", O_RDWR);
	if (-1 == open)
	{
		printf("kernel set open chardevnode fail, errno=%d \n", errno);
		return -1;
	}
	
    flags_info.magic_start = FLAGS_MAGIC;
	
    flags_info.boot_fota_flag.boot_to = DUAL_SYSTEM;
    flags_info.boot_fota_flag.fota_status = 0;
    flags_info.boot_fota_flag.system.status = DUALSYSTEM_STATUS_BOOTABLE;
    flags_info.boot_fota_flag.system.try_cnt = 5;
    flags_info.boot_fota_flag.system2.status = DUALSYSTEM_STATUS_BOOTABLE;
    flags_info.boot_fota_flag.system2.try_cnt = 13;

    flags_info.boot_env.dualsys_type = DUALSYSTEM_AB;
	strncpy(flags_info.boot_env.system_boot_env, system_boot_env, sizeof(flags_info.boot_env.system_boot_env));
	strncpy(flags_info.boot_env.system2_boot_env, system2_boot_env, sizeof(flags_info.boot_env.system2_boot_env));

	flags_info.ubifs_status.fs_status = 0;
	strncpy(flags_info.ubifs_status.fs_mtd_name, fs_mtd_name, sizeof(flags_info.ubifs_status.fs_mtd_name));
	strncpy(flags_info.ubifs_status.fs_ubi_vol_name, fs_ubi_vol_name, sizeof(flags_info.ubifs_status.fs_ubi_vol_name));
	
    flags_info.magic_end = FLAGS_MAGIC;

	if (ioctl(fd, FLAGS_UTILS_SET, &flags_info) < 0)
	{
		printf("Err: ioctl cmd(0x%x) fail, errno=%d \n", FLAGS_UTILS_SET, errno);
		
		close(fd);
		return -1;
	}

	printf("Log: kernel set flags success \n");
	
	close(fd);
	return 0;
}


static int excute_command_get_nocrc(char *option_para)
{
	int i = 0;
	T_FLAGS_INFO flags_info = {0};

	if (flags_get_nocrc(&flags_info) != 0)
	{
		printf("Err: flags get fail \n");
		return -1;
	}
	else
	{
		printf("magic_start                     = 0x%x \n",	flags_info.magic_start);
		
		printf("boot_fota_flag.boot_to          = 0x%x \n",	flags_info.boot_fota_flag.boot_to);
		printf("boot_fota_flag.fota_status      = %u \n",	flags_info.boot_fota_flag.fota_status);
		printf("boot_fota_flag.system.status    = 0x%x \n",	flags_info.boot_fota_flag.system.status);
		printf("boot_fota_flag.system.try_cnt	= %d \n",	flags_info.boot_fota_flag.system.try_cnt);
		printf("boot_fota_flag.system2.status	= 0x%x \n",	flags_info.boot_fota_flag.system2.status);
		printf("boot_fota_flag.system2.try_cnt	= %d \n",	flags_info.boot_fota_flag.system2.try_cnt);
		
		printf("boot_env.dualsys_type           = 0x%x \n",	flags_info.boot_env.dualsys_type);
		printf("boot_env.system_boot_env        = %s \n",	flags_info.boot_env.system_boot_env);
		printf("boot_env.system2_boot_env       = %s \n",	flags_info.boot_env.system2_boot_env);
		
		printf("ubifs_status.fs_status          = %d \n",	flags_info.ubifs_status.fs_status);
		printf("ubifs_status.fs_mtd_name        = %s \n",	flags_info.ubifs_status.fs_mtd_name);
		printf("ubifs_status.fs_ubi_vol_name    = %s \n",	flags_info.ubifs_status.fs_ubi_vol_name);
		
		printf("nvro_flag                       = %u \n",	flags_info.nvro_flag);
		printf("crc32                           = %u \n",	flags_info.crc32);
		
		printf("ota_system                      = %d \n",   flags_info.ota_system);

		for (i = 0; i < OTA_PARTITION_NUM_MAX; i++)
		{
			printf("index=%d, mtdnum=%d, len=%u          \n", i, flags_info.ota_partiton_info[i].mtdnum, flags_info.ota_partiton_info[i].len);
		}
	
		printf("magic_end                       = 0x%x \n",	flags_info.magic_end);
		
	    return 0;
	}
}


static int excute_command_set_nocrc(char *option_para)
{
	T_FLAGS_INFO flags_info = {0};
	
	const char delim[2] = ",";
	char *token_1 = NULL;
	char *token_2 = NULL;
	char *token_3 = NULL;
	char *token_4 = NULL;
	char *token_5 = NULL;
	char *token_6 = NULL;
	
    if (NULL == option_para)
    {
        usage();
        printf("Command input invalid value! null option parameters! \n");
        return -1;
    }
	
	if (flags_get_nocrc(&flags_info) != 0)
	{
		printf("Err: flags get fail \n");
		return -1;
	}

	token_1 = strtok(option_para, delim);
	if (NULL == token_1)
	{
		printf("token_1 delim fail: NULL \n");
		return -1;
	}
	
	token_2 = strtok(NULL, delim);
	if (NULL == token_2)
	{
		printf("token_2 delim fail: NULL \n");
		return -1;
	}
	
	token_3 = strtok(NULL, delim);
	if (NULL == token_3)
	{
		printf("token_3 delim fail: NULL \n");
		return -1;
	}
	
	token_4 = strtok(NULL, delim);
	if (NULL == token_4)
	{
		printf("token_4 delim fail: NULL \n");
		return -1;
	}
	
	token_5 = strtok(NULL, delim);
	if (NULL == token_5)
	{
		printf("token_5 delim fail: NULL \n");
		return -1;
	}

	token_6 = strtok(NULL, delim);
	if (NULL != token_6)
	{
		printf("too many command input, fail \n");
		return -1;
	}

    flags_info.boot_fota_flag.system2.try_cnt = atoi(token_1);

	strncpy(flags_info.boot_env.system_boot_env, token_2, sizeof(flags_info.boot_env.system_boot_env));

	flags_info.ubifs_status.fs_status = atoi(token_3);
	strncpy(flags_info.ubifs_status.fs_mtd_name, token_4, sizeof(flags_info.ubifs_status.fs_mtd_name));
	
    flags_info.magic_end = atoi(token_5);

	if (flags_set_nocrc(&flags_info) != 0)
	{
		printf("Err: set flags fail \n");
		return -1;
	}
	else
	{
		printf("Log: set flags success \n");
		return 0;
	}
}


/*******************************************************************************
 *                       Global function declarations                          *
 ******************************************************************************/
int main(int argc, char *argv[])
{
    int i				= 0;
    int option_index	= 0;
    int cmd_num		= 0;

    int ret			= -1;
    int ch			= -1;

    printf("Log: build date: %s %s \n", __DATE__, __TIME__);

    while ((ch = getopt_long(argc, argv, g_short_string, g_long_options, &option_index)) != -1)
    {
        for (i = 0; i < sizeof(g_option_handle) / sizeof(option_handle_t); i++)
        {
            if (ch != g_option_handle[i].option_value)
            {
                continue;
            }

            cmd_num++;

            if (NULL == g_option_handle[i].func)
            {
                printf("Err: command short string is: %c, but option handle func is NULL \n", ch);
                break;
            }

            ret = g_option_handle[i].func(optarg);
            if (ret < 0)
            {
                ret = -1;
                goto end;
            }
        }
    }

    if (0 == cmd_num)
    {
        printf("Err: can not find valid command \n");
        usage();
        ret = -1;
        goto end;
    }

    ret = 0;

end:
    return ret ;
}

