/*
 * f_diag_acm.c -- USB CDC serial (diag_acm) function driver
 *
 * Copyright (C) 2003 Al Borchers (alborchers@steinerpoint.com)
 * Copyright (C) 2008 by David Brownell
 * Copyright (C) 2008 by Nokia Corporation
 * Copyright (C) 2009 by Samsung Electronics
 * Author: Michal Nazarewicz (mina86@mina86.com)
 *
 * This software is distributed under the terms of the GNU General
 * Public License ("GPL") as published by the Free Software Foundation,
 * either version 2 of that License or (at your option) any later version.
 */

/* #define VERBOSE_DEBUG */

#include <linux/slab.h>
#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/android_notify.h>

#include "u_diag.h"
#include "gadget_chips.h"

#ifndef CONFIG_SYSTEM_RECOVERY
extern int zDrvNand_WriteBootflag( int flag );
#endif
/*
 * This CDC ACM function support just wraps control functions and
 * notifications around the generic serial-over-usb code.
 *
 * Because CDC ACM is standardized by the USB-IF, many host operating
 * systems have drivers for it.  Accordingly, ACM is the preferred
 * interop solution for serial-port type connections.  The control
 * models are often not necessary, and in any case don't do much in
 * this bare-bones implementation.
 *
 * Note that even MS-Windows has some support for ACM.  However, that
 * support is somewhat broken because when you use ACM in a composite
 * device, having multiple interfaces confuses the poor OS.  It doesn't
 * seem to understand CDC Union descriptors.  The new "association"
 * descriptors (roughly equivalent to CDC Unions) may sometimes help.
 */

struct f_diag_acm {
	struct u_diag			port;
	u8				ctrl_id, data_id;
	u8				port_num;

	u8				pending;

	/* lock is mostly for pending and notify_req ... they get accessed
	 * by callbacks both from tty (open/close/break) under its spinlock,
	 * and notify_req.complete() which can't use that lock.
	 */
	spinlock_t			lock;

	struct usb_ep			*notify;
	struct usb_request		*notify_req;

	struct usb_cdc_line_coding	port_line_coding;	/* 8-N-1 etc */

	/* SetControlLineState request -- CDC 1.1 section 6.2.14 (INPUT) */
	u16				port_handshake_bits;
#define ACM_CTRL_RTS	(1 << 1)	/* unused with full duplex */
#define ACM_CTRL_DTR	(1 << 0)	/* host is ready for data r/w */

	/* SerialState notification -- CDC 1.1 section 6.3.5 (OUTPUT) */
	u16				serial_state;
#define ACM_CTRL_OVERRUN	(1 << 6)
#define ACM_CTRL_PARITY		(1 << 5)
#define ACM_CTRL_FRAMING	(1 << 4)
#define ACM_CTRL_RI		(1 << 3)
#define ACM_CTRL_BRK		(1 << 2)
#define ACM_CTRL_DSR		(1 << 1)
#define ACM_CTRL_DCD		(1 << 0)
	int switch_falg; 		
};

static inline struct f_diag_acm *func_to_diag_acm(struct usb_function *f)
{
	return container_of(f, struct f_diag_acm, port.func);
}

static inline struct f_diag_acm *port_to_diag_acm(struct u_diag *p)
{
	return container_of(p, struct f_diag_acm, port);
}

/*-------------------------------------------------------------------------*/

/* notification endpoint uses smallish and infrequent fixed-size messages */

#define GS_LOG2_NOTIFY_INTERVAL		5	/* 1 << 5 == 32 msec */
#define GS_NOTIFY_MAXPACKET		10	/* notification + 2 bytes */

/* interface and class descriptors: */

static struct usb_interface_assoc_descriptor
diag_iad_descriptor = {
	.bLength =		sizeof diag_iad_descriptor,
	.bDescriptorType =	USB_DT_INTERFACE_ASSOCIATION,

	/* .bFirstInterface =	DYNAMIC, */
	.bInterfaceCount = 	2,	// control + data
	.bFunctionClass =	USB_CLASS_COMM,
	.bFunctionSubClass =	USB_CDC_SUBCLASS_ACM,
	.bFunctionProtocol =	USB_CDC_ACM_PROTO_AT_V25TER,
	/* .iFunction =		DYNAMIC */
};


static struct usb_interface_descriptor diag_control_interface_desc = {
	.bLength =		USB_DT_INTERFACE_SIZE,
	.bDescriptorType =	USB_DT_INTERFACE,
	/* .bInterfaceNumber = DYNAMIC */
	.bNumEndpoints =	1,
	.bInterfaceClass =	USB_CLASS_COMM,
	.bInterfaceSubClass =	USB_CDC_SUBCLASS_ACM,
	.bInterfaceProtocol =	USB_CDC_ACM_PROTO_AT_V25TER,
	/* .iInterface = DYNAMIC */
};

static struct usb_interface_descriptor diag_data_interface_desc = {
	.bLength =		USB_DT_INTERFACE_SIZE,
	.bDescriptorType =	USB_DT_INTERFACE,
	/* .bInterfaceNumber = DYNAMIC */
	.bNumEndpoints =	2,
	.bInterfaceClass =	USB_CLASS_CDC_DATA,
	.bInterfaceSubClass =	0,
	.bInterfaceProtocol =	0,
	/* .iInterface = DYNAMIC */
};

static struct usb_cdc_header_desc diag_header_desc = {
	.bLength =		sizeof(diag_header_desc),
	.bDescriptorType =	USB_DT_CS_INTERFACE,
	.bDescriptorSubType =	USB_CDC_HEADER_TYPE,
	.bcdCDC =		cpu_to_le16(0x0110),
};

static struct usb_cdc_call_mgmt_descriptor
diag_call_mgmt_descriptor = {
	.bLength =		sizeof(diag_call_mgmt_descriptor),
	.bDescriptorType =	USB_DT_CS_INTERFACE,
	.bDescriptorSubType =	USB_CDC_CALL_MANAGEMENT_TYPE,
	.bmCapabilities =	0,
	/* .bDataInterface = DYNAMIC */
};

static struct usb_cdc_acm_descriptor diag_descriptor = {
	.bLength =		sizeof(diag_descriptor),
	.bDescriptorType =	USB_DT_CS_INTERFACE,
	.bDescriptorSubType =	USB_CDC_ACM_TYPE,
	.bmCapabilities =	USB_CDC_CAP_LINE,
};

static struct usb_cdc_union_desc diag_union_desc = {
	.bLength =		sizeof(diag_union_desc),
	.bDescriptorType =	USB_DT_CS_INTERFACE,
	.bDescriptorSubType =	USB_CDC_UNION_TYPE,
	/* .bMasterInterface0 =	DYNAMIC */
	/* .bSlaveInterface0 =	DYNAMIC */
};

/* full speed support: */

static struct usb_endpoint_descriptor diag_fs_notify_desc = {
	.bLength =		USB_DT_ENDPOINT_SIZE,
	.bDescriptorType =	USB_DT_ENDPOINT,
	.bEndpointAddress =	USB_DIR_IN,
	.bmAttributes =		USB_ENDPOINT_XFER_INT,
	.wMaxPacketSize =	cpu_to_le16(GS_NOTIFY_MAXPACKET),
	.bInterval =		1 << GS_LOG2_NOTIFY_INTERVAL,
};

static struct usb_endpoint_descriptor diag_fs_in_desc = {
	.bLength =		USB_DT_ENDPOINT_SIZE,
	.bDescriptorType =	USB_DT_ENDPOINT,
	.bEndpointAddress =	USB_DIR_IN,
	.bmAttributes =		USB_ENDPOINT_XFER_BULK,
};

static struct usb_endpoint_descriptor diag_fs_out_desc = {
	.bLength =		USB_DT_ENDPOINT_SIZE,
	.bDescriptorType =	USB_DT_ENDPOINT,
	.bEndpointAddress =	USB_DIR_OUT,
	.bmAttributes =		USB_ENDPOINT_XFER_BULK,
};

static struct usb_descriptor_header *diag_fs_function[] = {
	(struct usb_descriptor_header *) &diag_iad_descriptor,
	(struct usb_descriptor_header *) &diag_control_interface_desc,
	(struct usb_descriptor_header *) &diag_header_desc,
	(struct usb_descriptor_header *) &diag_call_mgmt_descriptor,
	(struct usb_descriptor_header *) &diag_descriptor,
	(struct usb_descriptor_header *) &diag_union_desc,
	(struct usb_descriptor_header *) &diag_fs_notify_desc,
	(struct usb_descriptor_header *) &diag_data_interface_desc,
	(struct usb_descriptor_header *) &diag_fs_in_desc,
	(struct usb_descriptor_header *) &diag_fs_out_desc,
	NULL,
};

/* high speed support: */
static struct usb_endpoint_descriptor diag_hs_notify_desc = {
	.bLength =		USB_DT_ENDPOINT_SIZE,
	.bDescriptorType =	USB_DT_ENDPOINT,
	.bEndpointAddress =	USB_DIR_IN,
	.bmAttributes =		USB_ENDPOINT_XFER_INT,
	.wMaxPacketSize =	cpu_to_le16(GS_NOTIFY_MAXPACKET),
	.bInterval =		GS_LOG2_NOTIFY_INTERVAL+4,
};

static struct usb_endpoint_descriptor diag_hs_in_desc = {
	.bLength =		USB_DT_ENDPOINT_SIZE,
	.bDescriptorType =	USB_DT_ENDPOINT,
	.bmAttributes =		USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize =	cpu_to_le16(512),
};

static struct usb_endpoint_descriptor diag_hs_out_desc = {
	.bLength =		USB_DT_ENDPOINT_SIZE,
	.bDescriptorType =	USB_DT_ENDPOINT,
	.bmAttributes =		USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize =	cpu_to_le16(512),
};

static struct usb_descriptor_header *diag_hs_function[] = {
	(struct usb_descriptor_header *) &diag_iad_descriptor,
	(struct usb_descriptor_header *) &diag_control_interface_desc,
	(struct usb_descriptor_header *) &diag_header_desc,
	(struct usb_descriptor_header *) &diag_call_mgmt_descriptor,
	(struct usb_descriptor_header *) &diag_descriptor,
	(struct usb_descriptor_header *) &diag_union_desc,
	(struct usb_descriptor_header *) &diag_hs_notify_desc,
	(struct usb_descriptor_header *) &diag_data_interface_desc,
	(struct usb_descriptor_header *) &diag_hs_in_desc,
	(struct usb_descriptor_header *) &diag_hs_out_desc,
	NULL,
};

static struct usb_endpoint_descriptor diag_ss_in_desc = {
	.bLength =		USB_DT_ENDPOINT_SIZE,
	.bDescriptorType =	USB_DT_ENDPOINT,
	.bmAttributes =		USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize =	cpu_to_le16(1024),
};

static struct usb_endpoint_descriptor diag_ss_out_desc = {
	.bLength =		USB_DT_ENDPOINT_SIZE,
	.bDescriptorType =	USB_DT_ENDPOINT,
	.bmAttributes =		USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize =	cpu_to_le16(1024),
};

static struct usb_ss_ep_comp_descriptor diag_ss_bulk_comp_desc = {
	.bLength =              sizeof diag_ss_bulk_comp_desc,
	.bDescriptorType =      USB_DT_SS_ENDPOINT_COMP,
};

static struct usb_descriptor_header *diag_ss_function[] = {
	(struct usb_descriptor_header *) &diag_iad_descriptor,
	(struct usb_descriptor_header *) &diag_control_interface_desc,
	(struct usb_descriptor_header *) &diag_header_desc,
	(struct usb_descriptor_header *) &diag_call_mgmt_descriptor,
	(struct usb_descriptor_header *) &diag_descriptor,
	(struct usb_descriptor_header *) &diag_union_desc,
	(struct usb_descriptor_header *) &diag_hs_notify_desc,
	(struct usb_descriptor_header *) &diag_ss_bulk_comp_desc,
	(struct usb_descriptor_header *) &diag_data_interface_desc,
	(struct usb_descriptor_header *) &diag_ss_in_desc,
	(struct usb_descriptor_header *) &diag_ss_bulk_comp_desc,
	(struct usb_descriptor_header *) &diag_ss_out_desc,
	(struct usb_descriptor_header *) &diag_ss_bulk_comp_desc,
	NULL,
};

/* string descriptors: */

#define DIAG_CTRL_IDX	0
#define DIAG_DATA_IDX	1
#define DIAG_IAD_IDX	2

/* static strings, in UTF-8 */
static struct usb_string diag_string_defs[] = {
	[DIAG_CTRL_IDX].s = "CDC Abstract Control Model (ACM)",
	[DIAG_DATA_IDX].s = "CDC ACM Data",
	[DIAG_IAD_IDX ].s = "CDC Serial",
	{  }, /* end of list */
};

static struct usb_gadget_strings diag_string_table = {
	.language =		0x0409,	/* en-us */
	.strings =		diag_string_defs,
};

static struct usb_gadget_strings *diag_strings[] = {
	&diag_string_table,
	NULL,
};

/*-------------------------------------------------------------------------*/

/* ACM control ... data handling is delegated to tty library code.
 * The main task of this function is to activate and deactivate
 * that code based on device state; track parameters like line
 * speed, handshake state, and so on; and issue notifications.
 */

static void diag_complete_set_line_coding(struct usb_ep *ep,
		struct usb_request *req)
{
	struct f_diag_acm	*diag_acm = ep->driver_data;
	struct usb_composite_dev *cdev = diag_acm->port.func.config->cdev;

	if (req->status != 0) {
		usb_printk( "diag_acm ttyGS%d completion, err %d\n",
				diag_acm->port_num, req->status);
		return;
	}

	/* normal completion */
	if (req->actual != sizeof(diag_acm->port_line_coding)) {
		usb_printk( "diag_acm ttyGS%d short resp, len %d\n",
				diag_acm->port_num, req->actual);
		usb_ep_set_halt(ep);
	} else {
		struct usb_cdc_line_coding	*value = req->buf;

		/* REVISIT:  we currently just remember this data.
		 * If we change that, (a) validate it first, then
		 * (b) update whatever hardware needs updating,
		 * (c) worry about locking.  This is information on
		 * the order of 9600-8-N-1 ... most of which means
		 * nothing unless we control a real RS232 line.
		 */
		diag_acm->port_line_coding = *value;
		printk("diag_acm_complete_set_line_coding, datarate:%d, char:%d, parity:%d, databit:%d\n", value->dwDTERate,  value->bCharFormat, value->bParityType, value->bDataBits);
		u32 baud = get_unaligned_le32(&value->dwDTERate);
		if(baud == 4000000){
			if(diag_acm->switch_falg >= 1){
				return;
			}
			diag_acm->switch_falg = 1;
			printk("close DL & switch to user mode\n");
#ifndef CONFIG_SYSTEM_RECOVERY
			zDrvNand_WriteBootflag(1);
#endif
			usb_notify_up(USB_SWITCH_USER, NULL);	
			//usb_notify_up(USB_DEVICE_EXCEPT_RESET, NULL);		
		}
	}
}

static int diag_acm_setup(struct usb_function *f, const struct usb_ctrlrequest *ctrl)
{
	struct f_diag_acm		*diag_acm = func_to_diag_acm(f);
	struct usb_composite_dev *cdev = f->config->cdev;
	struct usb_request	*req = cdev->req;
	int			value = -EOPNOTSUPP;
	u16			w_index = le16_to_cpu(ctrl->wIndex);
	u16			w_value = le16_to_cpu(ctrl->wValue);
	u16			w_length = le16_to_cpu(ctrl->wLength);

	/* composite driver infrastructure handles everything except
	 * CDC class messages; interface activation uses set_alt().
	 *
	 * Note CDC spec table 4 lists the ACM request profile.  It requires
	 * encapsulated command support ... we don't handle any, and respond
	 * to them by stalling.  Options include get/set/clear comm features
	 * (not that useful) and SEND_BREAK.
	 */
	switch ((ctrl->bRequestType << 8) | ctrl->bRequest) {

	/* SET_LINE_CODING ... just read and save what the host sends */
	case ((USB_DIR_OUT | USB_TYPE_CLASS | USB_RECIP_INTERFACE) << 8)
			| USB_CDC_REQ_SET_LINE_CODING:
		if (w_length != sizeof(struct usb_cdc_line_coding)
				|| w_index != diag_acm->ctrl_id)
			goto invalid;

		value = w_length;
		cdev->gadget->ep0->driver_data = diag_acm;
		req->complete = diag_complete_set_line_coding;
		break;

	/* GET_LINE_CODING ... return what host sent, or initial value */
	case ((USB_DIR_IN | USB_TYPE_CLASS | USB_RECIP_INTERFACE) << 8)
			| USB_CDC_REQ_GET_LINE_CODING:
		if (w_index != diag_acm->ctrl_id)
			goto invalid;

		value = min_t(unsigned, w_length,
				sizeof(struct usb_cdc_line_coding));
		memcpy(req->buf, &diag_acm->port_line_coding, value);
		break;

	/* SET_CONTROL_LINE_STATE ... save what the host sent */
	case ((USB_DIR_OUT | USB_TYPE_CLASS | USB_RECIP_INTERFACE) << 8)
			| USB_CDC_REQ_SET_CONTROL_LINE_STATE:
		if (w_index != diag_acm->ctrl_id){
			printk("---SET_CONTROL_LINE_STATE, w_index:%02x, w_value:%02x\n", w_index, w_value);
			goto invalid;
		}
		value = 0;

		/* FIXME we should not allow data to flow until the
		 * host sets the ACM_CTRL_DTR bit; and when it clears
		 * that bit, we should return to that no-flow state.
		 */
		diag_acm->port_handshake_bits = w_value;
		break;

	default:
invalid:
		usb_printk( "invalid control req%02x.%02x v%04x i%04x l%d\n",
			ctrl->bRequestType, ctrl->bRequest,
			w_value, w_index, w_length);
	}

	/* respond with data transfer or status phase? */
	if (value >= 0) {
		usb_printk( "diag_acm ttyGS%d req%02x.%02x v%04x i%04x l%d\n",
			diag_acm->port_num, ctrl->bRequestType, ctrl->bRequest,
			w_value, w_index, w_length);
		req->zero = 0;
		req->length = value;
		value = usb_ep_queue(cdev->gadget->ep0, req, GFP_ATOMIC);
		if (value < 0)
			usb_printk( "diag_acm response on ttyGS%d, err %d\n",
					diag_acm->port_num, value);
	}

	/* device either stalls (value < 0) or reports success */
	return value;
}

static int diag_acm_set_alt(struct usb_function *f, unsigned intf, unsigned alt)
{
	struct f_diag_acm		*diag_acm = func_to_diag_acm(f);
	struct usb_composite_dev *cdev = f->config->cdev;
	int ret = 0;
	/* we know alt == 0, so this is an activation or a reset */

	if (intf == diag_acm->ctrl_id) {
		if (diag_acm->notify->driver_data) {
			usb_printk( "reset diag_acm control interface %d\n", intf);
			usb_ep_disable(diag_acm->notify);
		} else {
			usb_printk( "init diag_acm ctrl interface %d\n", intf);
			if (config_ep_by_speed(cdev->gadget, f, diag_acm->notify))
				return -EINVAL;
		}
		ret = usb_ep_enable(diag_acm->notify);
		diag_acm->notify->driver_data = diag_acm;

	} else if (intf == diag_acm->data_id) {
		if (diag_acm->port.in->driver_data) {
			usb_printk( "reset diag_acm ttyGS%d\n", diag_acm->port_num);
			diag_disconnect(&diag_acm->port);
		}
		if (!diag_acm->port.in->desc || !diag_acm->port.out->desc) {
			usb_printk("activate diag_acm ttyGS%d\n", diag_acm->port_num);
			if (config_ep_by_speed(cdev->gadget, f,
					       diag_acm->port.in) ||
			    config_ep_by_speed(cdev->gadget, f,
					       diag_acm->port.out)) {
				diag_acm->port.in->desc = NULL;
				diag_acm->port.out->desc = NULL;
				return -EINVAL;
			}
		}
		diag_connect(&diag_acm->port, diag_acm->port_num);

	} else
		return -EINVAL;

	return 0;
}

static void diag_acm_disable(struct usb_function *f)
{
	struct f_diag_acm	*diag_acm = func_to_diag_acm(f);
	struct usb_composite_dev *cdev = f->config->cdev;

	usb_printk( "diag_acm ttyGS%d deactivated\n", diag_acm->port_num);
	diag_disconnect(&diag_acm->port);
	usb_ep_disable(diag_acm->notify);
	diag_acm->switch_falg = 0;
	diag_acm->notify->driver_data = NULL;
}
#ifdef CONFIG_PM
unsigned int g_diag_acm_suspend_cnt = 0;
unsigned int g_diag_acm_reume_cnt = 0;

static void diag_acm_suspend(struct usb_function *f)
{
	struct f_diag_acm	*diag_acm = func_to_diag_acm(f);
	struct usb_composite_dev *cdev = f->config->cdev;
    g_diag_acm_suspend_cnt++;
	usb_printk("%s, %u, suspendcnt:%d\n", __func__, __LINE__, g_diag_acm_suspend_cnt);
	USBSTACK_DBG("%s, %u suspendcnt:%d", __func__, __LINE__, g_diag_acm_suspend_cnt);

    diag_acm->port.suspend_state = 1;
	diag_disconnect_ext(&diag_acm->port);
	
    usb_ep_disable((&diag_acm->port)->in);

	
}
static void diag_acm_resume(struct usb_function *f)
{
	struct f_diag_acm	*diag_acm = func_to_diag_acm(f);
	struct usb_composite_dev *cdev = f->config->cdev;
    g_diag_acm_reume_cnt++;
	usb_printk("%s, %u, resumecnt:%d\n", __func__, __LINE__, g_diag_acm_reume_cnt);
	USBSTACK_DBG("%s, %u resumecnt:%d", __func__, __LINE__, g_diag_acm_reume_cnt);

	usb_ep_resume_enable((&diag_acm->port)->in);
//    usb_ep_enable((&diag->port)->out);
 	//diagial_connect_ext(&diag->port);
	diag_acm->port.suspend_state = 0;
	diag_connect_ext(&diag_acm->port);
	
}
#endif
/*-------------------------------------------------------------------------*/

/**
 * diag_acm_cdc_notify - issue CDC notification to host
 * @diag_acm: wraps host to be notified
 * @type: notification type
 * @value: Refer to cdc specs, wValue field.
 * @data: data to be sent
 * @length: size of data
 * Context: irqs blocked, diag_acm->lock held, diag_acm_notify_req non-null
 *
 * Returns zero on success or a negative errno.
 *
 * See section 6.3.5 of the CDC 1.1 specification for information
 * about the only notification we issue:  SerialState change.
 */
static int diag_acm_cdc_notify(struct f_diag_acm *diag_acm, u8 type, u16 value,
		void *data, unsigned length)
{
	struct usb_ep			*ep = diag_acm->notify;
	struct usb_request		*req;
	struct usb_cdc_notification	*notify;
	const unsigned			len = sizeof(*notify) + length;
	void				*buf;
	int				status;

	req = diag_acm->notify_req;
	diag_acm->notify_req = NULL;
	diag_acm->pending = false;

	req->length = len;
	notify = req->buf;
	buf = notify + 1;

	notify->bmRequestType = USB_DIR_IN | USB_TYPE_CLASS
			| USB_RECIP_INTERFACE;
	notify->bNotificationType = type;
	notify->wValue = cpu_to_le16(value);
	notify->wIndex = cpu_to_le16(diag_acm->ctrl_id);
	notify->wLength = cpu_to_le16(length);
	memcpy(buf, data, length);

	/* ep_queue() can complete immediately if it fills the fifo... */
	spin_unlock(&diag_acm->lock);
	status = usb_ep_queue(ep, req, GFP_ATOMIC);
	spin_lock(&diag_acm->lock);

	if (status < 0) {
		usb_printk("diag_acm ttyGS%d can't notify serial state, %d\n",
				diag_acm->port_num, status);
		diag_acm->notify_req = req;
	}

	return status;
}

static int diag_acm_notify_serial_state(struct f_diag_acm *diag_acm)
{
	struct usb_composite_dev *cdev = diag_acm->port.func.config->cdev;
	int			status;

	spin_lock(&diag_acm->lock);
	if (diag_acm->notify_req) {
		usb_printk("diag_acm ttyGS%d serial state %04x\n",
				diag_acm->port_num, diag_acm->serial_state);
		status = diag_acm_cdc_notify(diag_acm, USB_CDC_NOTIFY_SERIAL_STATE,
				0, &diag_acm->serial_state, sizeof(diag_acm->serial_state));
	} else {
		diag_acm->pending = true;
		status = 0;
	}
	spin_unlock(&diag_acm->lock);
	return status;
}

static void diag_acm_cdc_notify_complete(struct usb_ep *ep, struct usb_request *req)
{
	struct f_diag_acm		*diag_acm = req->context;
	u8			doit = false;

	/* on this call path we do NOT hold the port spinlock,
	 * which is why ACM needs its own spinlock
	 */
	spin_lock(&diag_acm->lock);
	if (req->status != -ESHUTDOWN)
		doit = diag_acm->pending;
	diag_acm->notify_req = req;
	spin_unlock(&diag_acm->lock);

	if (doit)
		diag_acm_notify_serial_state(diag_acm);
}

/* connect == the TTY link is open */

static void diag_acm_connect(struct u_diag *port)
{
	struct f_diag_acm		*diag_acm = port_to_diag_acm(port);

	diag_acm->serial_state |= ACM_CTRL_DSR | ACM_CTRL_DCD;
	diag_acm_notify_serial_state(diag_acm);
}

static void diag_acm_disconnect(struct u_diag *port)
{
	struct f_diag_acm		*diag_acm = port_to_diag_acm(port);

	diag_acm->serial_state &= ~(ACM_CTRL_DSR | ACM_CTRL_DCD);
	diag_acm_notify_serial_state(diag_acm);
}

static int diag_acm_send_break(struct u_diag *port, int duration)
{
	struct f_diag_acm		*diag_acm = port_to_diag_acm(port);
	u16			state;

	state = diag_acm->serial_state;
	state &= ~ACM_CTRL_BRK;
	if (duration)
		state |= ACM_CTRL_BRK;

	diag_acm->serial_state = state;
	return diag_acm_notify_serial_state(diag_acm);
}

/*-------------------------------------------------------------------------*/

/* diag_acm function driver setup/binding */
static int
diag_acm_bind(struct usb_configuration *c, struct usb_function *f)
{
	struct usb_composite_dev *cdev = c->cdev;
	struct f_diag_acm		*diag_acm = func_to_diag_acm(f);
	int			status;
	struct usb_ep		*ep;

	/* allocate instance-specific interface IDs, and patch descriptors */
	status = usb_interface_id(c, f);
	if (status < 0)
		goto fail;
	diag_acm->ctrl_id = status;
	diag_iad_descriptor.bFirstInterface = status;

	diag_control_interface_desc.bInterfaceNumber = status;
	diag_union_desc .bMasterInterface0 = status;

	status = usb_interface_id(c, f);
	if (status < 0)
		goto fail;
	diag_acm->data_id = status;

	diag_data_interface_desc.bInterfaceNumber = status;
	diag_union_desc.bSlaveInterface0 = status;
	diag_call_mgmt_descriptor.bDataInterface = status;

	status = -ENODEV;

	/* allocate instance-specific endpoints */
	ep = usb_ep_autoconfig(cdev->gadget, &diag_fs_in_desc);
	if (!ep)
		goto fail;
	diag_acm->port.in = ep;
	ep->driver_data = cdev;	/* claim */

	ep = usb_ep_autoconfig(cdev->gadget, &diag_fs_out_desc);
	if (!ep)
		goto fail;
	diag_acm->port.out = ep;
	ep->driver_data = cdev;	/* claim */

	ep = usb_ep_autoconfig(cdev->gadget, &diag_fs_notify_desc);
	if (!ep)
		goto fail;
	diag_acm->notify = ep;
	ep->driver_data = cdev;	/* claim */

	/* allocate notification */
	diag_acm->notify_req = gdiag_alloc_req(ep,
			sizeof(struct usb_cdc_notification) + 2,
			GFP_KERNEL);
	if (!diag_acm->notify_req)
		goto fail;

	diag_acm->notify_req->complete = diag_acm_cdc_notify_complete;
	diag_acm->notify_req->context = diag_acm;

	/* copy descriptors */
	f->descriptors = usb_copy_descriptors(diag_fs_function);
	if (!f->descriptors)
		goto fail;

	/* support all relevant hardware speeds... we expect that when
	 * hardware is dual speed, all bulk-capable endpoints work at
	 * both speeds
	 */
	if (gadget_is_dualspeed(c->cdev->gadget)) {
		diag_hs_in_desc.bEndpointAddress =
				diag_fs_in_desc.bEndpointAddress;
		diag_hs_out_desc.bEndpointAddress =
				diag_fs_out_desc.bEndpointAddress;
		diag_hs_notify_desc.bEndpointAddress =
				diag_fs_notify_desc.bEndpointAddress;

		/* copy descriptors */
		f->hs_descriptors = usb_copy_descriptors(diag_hs_function);
	}
	if (gadget_is_superspeed(c->cdev->gadget)) {
		diag_ss_in_desc.bEndpointAddress =
			diag_fs_in_desc.bEndpointAddress;
		diag_ss_out_desc.bEndpointAddress =
			diag_fs_out_desc.bEndpointAddress;

		/* copy descriptors, and track endpoint copies */
		f->ss_descriptors = usb_copy_descriptors(diag_ss_function);
		if (!f->ss_descriptors)
			goto fail;
	}

	usb_printk( "diag_acm ttyGS%d: %s speed IN/%s OUT/%s NOTIFY/%s\n",
			diag_acm->port_num,
			gadget_is_superspeed(c->cdev->gadget) ? "super" :
			gadget_is_dualspeed(c->cdev->gadget) ? "dual" : "full",
			diag_acm->port.in->name, diag_acm->port.out->name,
			diag_acm->notify->name);
	return 0;

fail:
	if (diag_acm->notify_req)
		gdiag_free_req(diag_acm->notify, diag_acm->notify_req);

	/* we might as well release our claims on endpoints */
	if (diag_acm->notify)
		diag_acm->notify->driver_data = NULL;
	if (diag_acm->port.out)
		diag_acm->port.out->driver_data = NULL;
	if (diag_acm->port.in)
		diag_acm->port.in->driver_data = NULL;

	usb_printk( "%s/%p: can't bind, err %d\n", f->name, f, status);

	return status;
}

static void
diag_acm_unbind(struct usb_configuration *c, struct usb_function *f)
{
	struct f_diag_acm		*diag_acm = func_to_diag_acm(f);

	if (gadget_is_dualspeed(c->cdev->gadget))
		usb_free_descriptors(f->hs_descriptors);
	if (gadget_is_superspeed(c->cdev->gadget))
		usb_free_descriptors(f->ss_descriptors);
	usb_free_descriptors(f->descriptors);
	gdiag_free_req(diag_acm->notify, diag_acm->notify_req);
	kfree(diag_acm);
}

/* Some controllers can't support CDC ACM ... */
static inline bool diag_can_support_cdc(struct usb_configuration *c)
{
	/* everything else is *probably* fine ... */
	return true;
}

/**
 * diag_acm_bind_config - add a CDC ACM function to a configuration
 * @c: the configuration to support the CDC ACM instance
 * @port_num: /dev/ttyGS* port this interface will use
 * Context: single threaded during gadget setup
 *
 * Returns zero on success, else negative errno.
 *
 * Caller must have called @gserial_setup() with enough ports to
 * handle all the ones it binds.  Caller is also responsible
 * for calling @gserial_cleanup() before module unload.
 */
int diag_acm_bind_config(struct usb_configuration *c, u8 port_num)
{
	struct f_diag_acm	*diag_acm;
	int		status;

	if (!diag_can_support_cdc(c))
		return -EINVAL;

	/* REVISIT might want instance-specific strings to help
	 * distinguish instances ...
	 */

	/* maybe allocate device-global string IDs, and patch descriptors */
	if (diag_string_defs[ACM_CTRL_IDX].id == 0) {
		status = usb_string_id(c->cdev);
		if (status < 0)
			return status;
		diag_string_defs[ACM_CTRL_IDX].id = status;

		diag_control_interface_desc.iInterface = status;

		status = usb_string_id(c->cdev);
		if (status < 0)
			return status;
		diag_string_defs[ACM_DATA_IDX].id = status;

		diag_data_interface_desc.iInterface = status;

		status = usb_string_id(c->cdev);
		if (status < 0)
			return status;
		diag_string_defs[ACM_IAD_IDX].id = status;

		diag_iad_descriptor.iFunction = status;
	}

	/* allocate and initialize one new instance */
	diag_acm = kzalloc(sizeof *diag_acm, GFP_KERNEL);
	if (!diag_acm)
		return -ENOMEM;

	spin_lock_init(&diag_acm->lock);

	diag_acm->port_num = port_num;

	diag_acm->port.connect = diag_acm_connect;
	diag_acm->port.disconnect = diag_acm_disconnect;
	diag_acm->port.send_break = diag_acm_send_break;

	diag_acm->port.func.name = "diag_acm";
	diag_acm->port.func.strings = diag_strings;
	/* descriptors are per-instance copies */
	diag_acm->port.func.bind = diag_acm_bind;
	diag_acm->port.func.unbind = diag_acm_unbind;
	diag_acm->port.func.set_alt = diag_acm_set_alt;
	diag_acm->port.func.setup = diag_acm_setup;
	diag_acm->port.func.disable = diag_acm_disable;
#if 0 //#ifdef CONFIG_PM
	diag_acm->port.func.suspend = diag_acm_suspend;
	diag_acm->port.func.resume = diag_acm_resume;
#endif
	status = usb_add_function(c, &diag_acm->port.func);
	if (status)
		kfree(diag_acm);
	return status;
}
